/*!
 *******************************************************************************
 * \file         spi_tclWiFi.cpp
 * \brief        Wifi Manager class that provides interface to delegate
 the execution of command and handle response
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Wifi Manager class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 10.02.2017  |  Unmukt Jain (RBEI/ECP2)          | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclFactory.h"
#include "spi_tclBluetooth.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdWiFi.h"
#include "RespRegister.h"
#include "spi_tclMediator.h"
#include "spi_tclDeviceIDDataIntf.h"
#include "spi_tclWiFiDevBase.h"
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
#include "spi_tclDiPoWiFi.h"
#endif

#include "spi_tclWiFi.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_WIFI
#include "trcGenProj/Header/spi_tclWiFi.cpp.trc.h"
#endif
#endif

//used to set the user context to Default i.e, 0 so that InitiateWirelessDiscovery Method result is not posted to HMI
static const t_U8 DEFAULT_USER_CONTEXT = 0 ;
static const t_U8 u8InvalidDeviceHandle = 0;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclWiFi::spi_tclWiFi();
 ***************************************************************************/
spi_tclWiFi::spi_tclWiFi(spi_tclWifiRespIntf *poWifiRespIntf) :
                  spi_tclSelectionIntf(e32COMPID_WIFI),
                  m_poWifiRespIntf(poWifiRespIntf),
                  m_bSelectDeviceRespPending(false),
                  m_bInitiateWirelessDiscoveryRespPending(false),
                  m_bIsAPPoweredOn(false),
                  m_poWiFiSettings(NULL),
                  m_bInitiateWirelessDiscoveryRequestPending(false),
                  m_bExchangeCredentials(false),
                  m_enPreferredFrequency(e8_Unknown)
{
   ETG_TRACE_USR1(("spi_tclWiFi() entered "));

   for (t_U8 u8Index = 0; u8Index < e8DEV_TYPE_MAX; u8Index++)
   {
      m_poWiFiDevBase[u8Index] = NULL;
   } //for(t_U8 u8Index=0; u8In

} //!end of spi_tclWiFi()

/***************************************************************************
 ** FUNCTION:  spi_tclWiFi::~spi_tclWiFi();
 ***************************************************************************/
spi_tclWiFi::~spi_tclWiFi()
{
   ETG_TRACE_USR1(("~spi_tclWiFi() entered "));
   m_poWifiRespIntf = NULL;
   m_poWiFiSettings = NULL;
   m_enPreferredFrequency = e8_Unknown;

   for (t_U8 u8Index = 0; u8Index < e8DEV_TYPE_MAX; u8Index++)
   {
      RELEASE_MEM(m_poWiFiDevBase[u8Index]);
      m_poWiFiDevBase[u8Index] = NULL;
   } //for(t_U8 u8Index=0; u8In

} //!end of ~spi_tclWiFi()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclWiFi::bInitialize()
 ***************************************************************************/
t_Bool spi_tclWiFi::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclWiFi::bInitialize() entered "));

   RespRegister *poRespRegister = RespRegister::getInstance();
   if (NULL != poRespRegister)
   {
      poRespRegister->bRegisterObject((spi_tclExtRespWiFi*) this);
      //Registering for OOBT Response
      poRespRegister->bRegisterObject((spi_tclExtRespOOBT*) this);
   }

#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
   m_poWiFiDevBase[e8DEV_TYPE_DIPO] = new spi_tclDiPoWiFi();
   SPI_NORMAL_ASSERT(NULL == m_poWiFiDevBase[e8DEV_TYPE_DIPO]);
#endif

   for (t_U8 u8Index = 0; u8Index < e8DEV_TYPE_MAX; u8Index++)
   {
      if (NULL != m_poWiFiDevBase[u8Index])
      {
         m_poWiFiDevBase[u8Index]->bInitialize();
      } //if((NULL
   } //for(t_U8 u8Index=0;u8Index < 

   vRegisterCallbacks();

   return true;
} //!end of bInitialize()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclWiFi::bUnInitialize()
 ***************************************************************************/
t_Bool spi_tclWiFi::bUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclWiFi::bUnInitialize() entered "));

   t_Bool bUninitSuccess = true;

   for (t_U8 u8Index = 0; u8Index < e8DEV_TYPE_MAX; u8Index++)
   {
      if (NULL != m_poWiFiDevBase[u8Index])
      {
         m_poWiFiDevBase[u8Index]->vUninitialize();
      } //if((NULL !=
      RELEASE_MEM(m_poWiFiDevBase[u8Index]);
   }

   ETG_TRACE_USR2(("spi_tclWiFi::bUnInitialize() status = %d ", ETG_ENUM(BOOL, bUninitSuccess)));
   return bUninitSuccess;
} //!end of bUnInitialize()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vLoadSettings()
 ***************************************************************************/
t_Void spi_tclWiFi::vLoadSettings()
{
   ETG_TRACE_USR1(("spi_tclWiFi::vLoadSettings() entered "));
   for (t_U8 u8Index = 0; u8Index < e8DEV_TYPE_MAX; u8Index++)
   {
      if (NULL != m_poWiFiDevBase[u8Index])
      {
         m_poWiFiDevBase[u8Index]->vSetWiFiSettingsInstance(m_poWiFiSettings);
      } //if((NULL != m_poWiFiDevBase[)
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vSaveSettings()
 ***************************************************************************/
t_Void spi_tclWiFi::vSaveSettings()
{
   ETG_TRACE_USR1(("spi_tclWiFi::vSaveSettings() entered "));
   //Add code
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vInitiateWirelessDiscovery()
 ***************************************************************************/
t_Void spi_tclWiFi::vInitiateWirelessDiscovery(tenDeviceCategory enDevCategory, const t_String& coszBTMACAddress)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vInitiateWirelessDiscovery() entered "));

   t_Bool bProceedWithInitiateWirelessDiscovery = false;
   bProceedWithInitiateWirelessDiscovery = bHandleInitiateWirelessDiscovery(enDevCategory, coszBTMACAddress);

   if (true == bProceedWithInitiateWirelessDiscovery)
   {
      ETG_TRACE_USR1(("spi_tclWiFi::vInitiateWirelessDiscovery(): Initiate Wireless Request being processed now. "));
      spi_tclExtCmdWiFiIntf *poExtCmdWiFiIntf = poFetchExtCmdWiFiIntf();
      if (NULL != poExtCmdWiFiIntf)
      {
         if ((true == bIsCarPlayWirelessAPSetUp()) && (true == bIsPreferredFrequency()))
         {
            vSendInitiateWirelessDiscoveryResult(true);
         }
         else
         {
            m_bInitiateWirelessDiscoveryRespPending = true;
            poExtCmdWiFiIntf->vInitiateWirelessDiscovery(enDevCategory);
         }
      }
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclWiFi::vInitiateWirelessDiscovery(): Initiate Wireless Request can't be processed now. "));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vPostConfigureWiFiResult()
 ***************************************************************************/
t_Void spi_tclWiFi::vPostConfigureWiFiResult(t_Bool bConfigureWiFiResult)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vPostConfigureWiFiResult() entered with bConfigureWiFiResult = %d", ETG_ENUM(BOOL,
            bConfigureWiFiResult)));

   if (true == m_bSelectDeviceRespPending)
   {
      // This update is the response for PrepareSetUp request sent on select device. Notify Device Selector
      m_bSelectDeviceRespPending = false;
      tenErrorCode enErrorCode = (false == bConfigureWiFiResult) ? e8AP_SETUP_ERROR : e8NO_ERRORS;
      vSendSelectDeviceResult(enErrorCode);
   }
   if (true == m_bInitiateWirelessDiscoveryRespPending)
   {
      //There was also a Initiate Wireless discovery request from HMI. so notfy WiFiSetUpHandler to be able to continue with other actions.
      m_bInitiateWirelessDiscoveryRespPending = false;
      vSendInitiateWirelessDiscoveryResult(bConfigureWiFiResult);
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vPostWiFiConfig()
 ***************************************************************************/
t_Void spi_tclWiFi::vPostWiFiConfig(const trWiFiConfig &rfcorWiFiConfig,
         const std::vector<trStationInfo>& corvecStationsInfo)
{
   t_U32 u32DeviceHandle = 0;
   ETG_TRACE_USR1(("spi_tclWiFi::vPostWiFiConfig(): No.Of Stations:%d",corvecStationsInfo.size()));

   m_rWiFiConfig = rfcorWiFiConfig;

   //only after the Powered State has changed to - WIFI_STATE_POWERED_ON
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig szSSID:%s", m_rWiFiConfig.szSSID.c_str()));
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig szPassphrase:%s", m_rWiFiConfig.szPassphrase.c_str()));
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig szInterface:%s", m_rWiFiConfig.szInterface.c_str()));
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig szMode:%s", m_rWiFiConfig.szMode.c_str()));
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig szType:%s", m_rWiFiConfig.szType.c_str()));
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig CurrentOperatingChannel:%d", m_rWiFiConfig.u32CurrentOperatingChannel));
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig enFrequency:%d", ETG_ENUM(WIFI_FREQUENCY, m_rWiFiConfig.enFrequency)));
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig szSecurity:%d", ETG_ENUM(WIFI_SECURITY, m_rWiFiConfig.enSecurity)));
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig bVisible:%d", ETG_ENUM(BOOL, m_rWiFiConfig.bVisible)));
   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig bPowered:%d", ETG_ENUM(BOOL, m_rWiFiConfig.bPowered)));

   ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig vector size:%d", corvecStationsInfo.size()));
   for (t_U8 u8Index = 0; u8Index < corvecStationsInfo.size(); u8Index++)
   {
      ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig(): Station:%d,HostName:%s", u8Index, corvecStationsInfo[u8Index].szDeviceName.c_str()));
      ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig(): IPAddress:%s", corvecStationsInfo[u8Index].szIPAddress.c_str()));
      ETG_TRACE_USR4(("spi_tclWiFi::vPostWiFiConfig(): MacAddress:%s", corvecStationsInfo[u8Index].szMacAddress.c_str()));
   }

   //Check whether there is a change in WiFi AP power state and notify DeviceSelector.
   //It takes care of terminating the active session, if required.
   if(m_bIsAPPoweredOn != m_rWiFiConfig.bPowered)
   {
      m_bIsAPPoweredOn = m_rWiFiConfig.bPowered;

      ETG_TRACE_USR2(("spi_tclWiFi::vPostWiFiConfig:Change in AP Power state. Check active sessions. IsAPOn:%d",
               ETG_ENUM(BOOL,m_bIsAPPoweredOn)));

      spi_tclMediator* poMediator = spi_tclMediator::getInstance();
      if (NULL != poMediator)
      {
         poMediator->vOnHUWiFiAPStateChange(m_bIsAPPoweredOn);
      } //if(NULL != poMediator)
   }

   //Dispatch the update to DipoWifi if the AP type is CarplayWireless
   if(true == m_bIsAPPoweredOn)
   {
      for(t_U8 u8Index = 0; u8Index < e8DEV_TYPE_MAX; u8Index++)
      {
         if (true == bValidateClient(u8Index))
         {
            m_poWiFiDevBase[u8Index]->vPostWiFiConfig(rfcorWiFiConfig,corvecStationsInfo);
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vGetWiFiConfig()
 ***************************************************************************/
t_Void spi_tclWiFi::vGetWiFiConfig(trWiFiConfig &rfrWiFiConfig)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vGetWiFiConfig() entered "));
   rfrWiFiConfig = m_rWiFiConfig;
}

/***************************************************************************
 ** FUNCTION:  spi_tclWiFi::vSelectDevice
 ***************************************************************************/
t_Void spi_tclWiFi::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vSelectDevice:DevID:0x%x,DeviceConnectionType:%d,SelectionType:%d,DeviceCategory:%d,DeviceType:%d,SelectReason:%d",
            corfrSelectReq.m_u32DeviceHandle,ETG_ENUM(CONNECTION_TYPE,corfrSelectReq.m_enDevConnType),ETG_ENUM(CONNECTION_REQ,corfrSelectReq.m_enDevConnReq),
            ETG_ENUM(DEVICE_CATEGORY,corfrSelectReq.m_enDevCategory),ETG_ENUM(DEVICE_TYPE,corfrSelectReq.m_enDeviceType),
            ETG_ENUM(SELECT_REASON,corfrSelectReq.m_enSelectionReason)));

   t_Bool bSendSelectDeviceResult = true;

   if(e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq)
   {
      if(e8DEV_TYPE_DIPO == corfrSelectReq.m_enDevCategory)
      {
         //Dispatching the message to DipoWifi
         if(NULL != m_poWiFiDevBase[e8DEV_TYPE_DIPO])
         {
            m_poWiFiDevBase[e8DEV_TYPE_DIPO]-> vSelectDevice(corfrSelectReq.m_u32DeviceHandle, corfrSelectReq.m_enDevConnReq);
         }

         if(e8USB_CONNECTED != corfrSelectReq.m_enDevConnType)
         {
            if ((true == bIsCarPlayWirelessAPSetUp()) && (true == bIsPreferredFrequency()))
            {
               //Request is to setup CarPlay Wirelss AP and it is already setup.
               //Send the response as success without sending the request
            }
            else
            {
               spi_tclExtCmdWiFiIntf *poExtCmdWiFiIntf = poFetchExtCmdWiFiIntf();
               if(NULL != poExtCmdWiFiIntf)
               {
                  m_bSelectDeviceRespPending = true;
                  bSendSelectDeviceResult = false;
                  poExtCmdWiFiIntf->vInitiateWirelessDiscovery(corfrSelectReq.m_enDevCategory);
               }
            }
         }
      }
   }
   else
   {
      vHandleDeviceDeselection(corfrSelectReq.m_u32DeviceHandle,corfrSelectReq.m_enDevCategory,corfrSelectReq.m_enDevConnType);
   }

   if(true == bSendSelectDeviceResult)
   {
      vSendSelectDeviceResult(e8NO_ERRORS);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWiFi::vSelectDeviceResult
 ***************************************************************************/
t_Void spi_tclWiFi::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vSelectDeviceResult:DevID:0x%x,DeviceConnectionType:%d,SelectionType:%d,DeviceCategory:%d,DeviceType:%d,SelectReason:%d,ErrorCode:%d",
            corfrSelectReq.m_u32DeviceHandle,ETG_ENUM(CONNECTION_TYPE,corfrSelectReq.m_enDevConnType),ETG_ENUM(CONNECTION_REQ,corfrSelectReq.m_enDevConnReq),
            ETG_ENUM(DEVICE_CATEGORY,corfrSelectReq.m_enDevCategory),ETG_ENUM(DEVICE_TYPE,corfrSelectReq.m_enDeviceType),
            ETG_ENUM(SELECT_REASON_UNKNOWN,corfrSelectReq.m_enSelectionReason),ETG_ENUM(ERROR_CODE,enErrorCode)));

   //Response Code
   tenResponseCode enResponseCode = (e8NO_ERRORS == enErrorCode)? e8SUCCESS : e8FAILURE;

   //Dispatching the SelectDeviceResult message to DipoWifi
   if(((e8DEV_TYPE_DIPO == corfrSelectReq.m_enDevCategory)) && (NULL != m_poWiFiDevBase[e8DEV_TYPE_DIPO]))
   {
      m_poWiFiDevBase[e8DEV_TYPE_DIPO]-> vOnSelectDeviceResult(corfrSelectReq.m_u32DeviceHandle, corfrSelectReq.m_enDevConnReq, enResponseCode );
   }

   if(e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq)
   {
      if(e8DEV_TYPE_DIPO == corfrSelectReq.m_enDevCategory)
      {
         if(e8USB_CONNECTED != corfrSelectReq.m_enDevConnType)
         {
            if(e8NO_ERRORS != enErrorCode)
            {
               //Send the error response for the pending GetWiFiCredentials request from MP in case of selection failure.
               if(false == m_szWiFiCredentialExchangePendingForDevice.empty())
               {
                  ETG_TRACE_USR4(("spi_tclWiFi:vSelectDeviceResult: Response to GetWiFiCredentials is pending but not share credentials"));
                  trWiFiAPConfig rWiFiAPConfig;
                  rWiFiAPConfig.szSSID = "";
                  rWiFiAPConfig.szPassphrase = "";
                  rWiFiAPConfig.u16OperatingChannel = 0;
                  rWiFiAPConfig.enSecurity = tenWifiSecurityType::e8_NONE;
                  m_poWifiRespIntf->vSendWiFiCredentials(m_szWiFiCredentialExchangePendingForDevice,
                           e8OPERATION_CANCELLED_BY_USER,
                           rWiFiAPConfig,
                           m_GetWiFiCredentialsContext);
                  m_szWiFiCredentialExchangePendingForDevice.clear();//Soved Issue RTC-750919
               }
               else
               {
                  ETG_TRACE_USR4(("spi_tclWiFi:vSelectDeviceResult: Response to GetWiFiCredentials is not pending"));
               }
            }
            else
            {
               m_szWiFiCredentialExchangePendingForDevice.clear();
            }
         }
      }
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclWiFi::vSelectDeviceResult:Initiate Wireless Request Pending :%d ", ETG_ENUM(BOOL, m_bInitiateWirelessDiscoveryRequestPending)));
      //! Process the New Pending request only after Select Device Result to all the other sub components
      if(true == m_bInitiateWirelessDiscoveryRequestPending)
      {
         ETG_TRACE_USR4(("spi_tclWiFi::vSelectDeviceResult: Processing the pending request"));
         //! this is currently being handled only for DIPO, make changes for Other technologies if Required
         //! set the request pending to false as it is processed now
         m_bInitiateWirelessDiscoveryRequestPending = false;
         vInitiateWirelessDiscovery(e8DEV_TYPE_DIPO);
      }

      //Clearing Pending requests on DESELECTION selectDeviceResult
      m_szWiFiCredentialExchangePendingForDevice.clear();

   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclWiFi::bIsPreferredFrequency()
 ***************************************************************************/
t_Bool spi_tclWiFi::bIsPreferredFrequency()
{
   spi_tclExtCmdWiFiIntf *poExtCmdWiFiIntf = poFetchExtCmdWiFiIntf();
   if (NULL != poExtCmdWiFiIntf)
   {
      m_enPreferredFrequency = poExtCmdWiFiIntf->enGetPreferredFrequency(e8DEV_TYPE_DIPO);
   }
   ETG_TRACE_USR4(("bIsPreferredFrequency  PreferredFrequency=%d APFrequency=%d", ETG_CENUM(tenWifiFrequency,
            m_enPreferredFrequency), ETG_CENUM(tenWifiFrequency, m_rWiFiConfig.enFrequency)));

   return ((m_enPreferredFrequency == m_rWiFiConfig.enFrequency));
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclWiFi::bIsCarPlayWirelessAPSetUp()
 ***************************************************************************/
t_Bool spi_tclWiFi::bIsCarPlayWirelessAPSetUp()
{
   ETG_TRACE_USR1(("spi_tclWiFi::bIsCarPlayWirelessAPSetUp():Power state:%d, AccessPoint:%s", ETG_ENUM(BOOL,
            m_rWiFiConfig.bPowered), m_rWiFiConfig.szType.c_str()));

   return ((scoszTechnology_CPW == m_rWiFiConfig.szType) && (true == m_rWiFiConfig.bPowered));
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclWiFi::vSendSelectDeviceResult(tenErrorCode enErrorCode)
 ***************************************************************************/
t_Void spi_tclWiFi::vSendSelectDeviceResult(tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vSendSelectDeviceResult:ErrorCode:%d", enErrorCode));

   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vPostSelectDeviceRes(e32COMPID_WIFI, enErrorCode);
   } //if(NULL != poMediator)
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclWiFi::vSendInitiateWirelessDiscoveryResult(t_Bool bSuccess)
 ***************************************************************************/
t_Void spi_tclWiFi::vSendInitiateWirelessDiscoveryResult(t_Bool bSuccess)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vSendInitiateWirelessDiscoveryResult:Succeeded:%d", bSuccess));

   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vPostInitiateWirelessDiscoveryResult(bSuccess);
   } //if(NULL != poMediator)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vGetWiFiCredentials(t_String szBTMACAddress,
 ***************************************************************************/
t_Void spi_tclWiFi::vGetWiFiCredentials(t_String szBTMACAddress, const trUserContext& corfrUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vGetWiFiCredentials:Device BTMACAddress:%s", szBTMACAddress.c_str()));
   std::transform(szBTMACAddress.begin(), szBTMACAddress.end(), szBTMACAddress.begin(), ::toupper);

   t_Bool bCredentialsAlreadyShared = false;

   m_CredentialsExchangeLock.s16Lock();
   auto itSet = m_setAccessoryWifiInfoExchangedDevices.find(szBTMACAddress);
   if(itSet != m_setAccessoryWifiInfoExchangedDevices.end())
   {
      bCredentialsAlreadyShared = true;
   }
   m_CredentialsExchangeLock.vUnlock();

   if((true == bCredentialsAlreadyShared) && (true == m_bExchangeCredentials))
   {
      ETG_TRACE_USR1(("spi_tclWiFi::vGetWiFiCredentials: WiFi credentials are exchanged once with device already"));
      trWiFiAPConfig rWiFiAPConfig;

      if ((NULL != m_poWifiRespIntf) && (true == bIsCarPlayWirelessAPSetUp()))
      {
         rWiFiAPConfig.szSSID = m_rWiFiConfig.szSSID;
         rWiFiAPConfig.szPassphrase = m_rWiFiConfig.szPassphrase;
         rWiFiAPConfig.u16OperatingChannel = static_cast<t_U16>(m_rWiFiConfig.u32CurrentOperatingChannel);
         rWiFiAPConfig.enSecurity = m_rWiFiConfig.enSecurity;
         m_poWifiRespIntf->vSendWiFiCredentials(szBTMACAddress,
                  e8NO_ERRORS,
                  rWiFiAPConfig,
                  m_GetWiFiCredentialsContext);
      }
      //!  SPI is supposed to send the credentials always if the request is placed, irrespective of whether SPI has already shared or not.
      else if (false == bIsCarPlayWirelessAPSetUp())
      {
         ETG_TRACE_USR1(("spi_tclWiFi::vGetWiFiCredentials: WiFi credentials will be shared again, after AP setup is done"));
         m_szWiFiCredentialExchangePendingForDevice = szBTMACAddress;
         m_GetWiFiCredentialsContext = corfrUsrCntxt;
      }
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclWiFi::vGetWiFiCredentials: WiFi credentials request is received for the first time are credentials are not exchanged with the device yet"));
      m_szWiFiCredentialExchangePendingForDevice = szBTMACAddress;
      m_GetWiFiCredentialsContext = corfrUsrCntxt;
   }

   //@TODO:if the device is already authorized by the time this request is received, WiFi credentials should be sent back immediately.
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vSendWiFiCredentails(t_String szBTMACAddress
 ***************************************************************************/
t_Void spi_tclWiFi::vSendWiFiCredentails(t_String szBTMACAddress)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vSendWiFiCredentails:Device BTMACAddress:%s",szBTMACAddress.c_str()));

   m_CredentialsExchangeLock.s16Lock();
   auto itSet = m_setAccessoryWifiInfoExchangedDevices.find(szBTMACAddress);
   if(itSet == m_setAccessoryWifiInfoExchangedDevices.end())
   {
      auto itSet = m_setAccessoryWifiInfoExchangedDevices.insert(szBTMACAddress);
   }
   m_CredentialsExchangeLock.vUnlock();

   if(m_szWiFiCredentialExchangePendingForDevice == szBTMACAddress)
   {
      m_szWiFiCredentialExchangePendingForDevice.clear();

      if (NULL != m_poWifiRespIntf)
      {
         trWiFiAPConfig rWiFiAPConfig;
         if (true == bIsCarPlayWirelessAPSetUp())
         {
            rWiFiAPConfig.szSSID = m_rWiFiConfig.szSSID;
            rWiFiAPConfig.szPassphrase = m_rWiFiConfig.szPassphrase;
            rWiFiAPConfig.u16OperatingChannel = static_cast<t_U16>(m_rWiFiConfig.u32CurrentOperatingChannel);
            rWiFiAPConfig.enSecurity = m_rWiFiConfig.enSecurity;

            m_poWifiRespIntf->vSendWiFiCredentials(szBTMACAddress,
                     e8NO_ERRORS,
                     rWiFiAPConfig,
                     m_GetWiFiCredentialsContext);
         }
         else
         {
            ETG_TRACE_USR4(("spi_tclWiFi::vSendWiFiCredentails:CarPlay AP is not setup. do not share credentials"));
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("spi_tclWiFi::vSendWiFiCredentails:There is no pending GetWiFiCredentials request"));
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclExtCmdWiFiIntf* spi_tclWiFi::poFetchExtCmdWiFiIntf()
 ***************************************************************************/
spi_tclExtCmdWiFiIntf* spi_tclWiFi::poFetchExtCmdWiFiIntf()
{
   spi_tclExtCmdWiFiIntf* poExtCmdWiFiIntf = NULL;
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   if (NULL != poExtCompMgr)
   {
      poExtCmdWiFiIntf = poExtCompMgr->poGetCmdWiFiIntfInst();
   }
   return poExtCmdWiFiIntf;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclWiFi::vHandleDeviceDeselection()
 ***************************************************************************/
t_Void spi_tclWiFi::vHandleDeviceDeselection(t_U32 u32DevID, tenDeviceCategory enDevCat,
         tenDeviceConnectionType enDeviceConnType)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vHandleDeviceDeselection:DeviceID:0x%x,DeviceCategory:%d,COnnectionType:%d", u32DevID, ETG_ENUM(DEVICE_CATEGORY,
            enDevCat), enDeviceConnType));

   if (e8USB_CONNECTED != enDeviceConnType)
   {
      //On CPW de-selection, clear the Wi-Fi credentials and disable WiFi Limitation mode
      if (e8DEV_TYPE_DIPO == enDevCat)
      {
         //Clear the WiFi Credentials Pending flag, since BT gets disconnected during de-selection.
         m_szWiFiCredentialExchangePendingForDevice.clear();
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclWiFi::vHandleOOBTPairing()
 ***************************************************************************/
t_Void spi_tclWiFi::vHandleOOBTComplete()
{
   ETG_TRACE_USR1(("spi_tclWiFi::vHandleOOBTComplete Entered"));

   //OOBT Conflict Trigger is received from BT Client Handler, on this Invoke PrepareSetup Of the WBL
   spi_tclExtCmdWiFiIntf *poExtCmdWiFiIntf = poFetchExtCmdWiFiIntf();
   if((false == bIsCarPlayWirelessAPSetUp()) || (false == bIsPreferredFrequency()))
   {
      if(NULL != poExtCmdWiFiIntf)
      {
         poExtCmdWiFiIntf->vInitiateWirelessDiscovery(e8DEV_TYPE_DIPO);
      }
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclWiFi::vHandleOOBTComplete is already setup"));
   }

}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclWiFi::vPostOOBTComplete()
 ***************************************************************************/
t_Void spi_tclWiFi::vPostOOBTComplete()
{
   ETG_TRACE_USR1(("spi_tclWiFi::vPostOOBTComplete()"));
   t_Bool bOOBTEnabled = false;
   //preparesetup of WBL -based on the ConfigReader value
   if (NULL != m_poWiFiSettings)
   {
      bOOBTEnabled = m_poWiFiSettings->bSetUpAPonOOBT();
   }
   if (bOOBTEnabled)
   {
      ETG_TRACE_USR1(("spi_tclWiFi::vPostOOBTComplete(): OOBT is Enabled"));

      vHandleOOBTComplete();

   }
   else
   {
      ETG_TRACE_USR1(("spi_tclWiFi::vPostOOBTComplete(): OOBT is Not Enabled"));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vSetWiFiSettingsInstance()
 ***************************************************************************/
t_Void spi_tclWiFi::vSetWiFiSettingsInstance(spi_tclWiFiSettingsIntf* poWiFiSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vSetWiFiSettingsInstance entered"));
   if (NULL != poWiFiSettingsIntf)
   {
      ETG_TRACE_USR1(("spi_tclWiFi::vSetWiFiSettingsInstance: Setting m_poWiFiSettings val"));
      m_poWiFiSettings = poWiFiSettingsIntf;
   }

}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclWiFi::vSetDevSelectorBusyStatus()
 ***************************************************************************/
t_Void spi_tclWiFi::vSetDevSelectorBusyStatus(const t_U32 cou32DeviceHandle, tenDeviceSelectionState enSelState)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vSetDevSelectorBusyStatus: Device ID = %d, Selection state = %d ", cou32DeviceHandle, ETG_ENUM(SELECTION_STATE,
            enSelState)));
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::bHandleInitiateWirelessDiscovery()
 ***************************************************************************/
t_Bool spi_tclWiFi::bHandleInitiateWirelessDiscovery(tenDeviceCategory enDevCategory, const t_String& coszBTMACAddress)
{

   /*
    *  Handling InitiateWireless Discovery:
    *  1. When BTMACAddress is empty - Then it's internal trigger for InitaiteWirelessDiscovery after the STA gets attached to CarplayWirelessAP
    *  2. Request for the Fresh Device
    *    a. When Selection is in Progress - Ignore the Request - Need to be Handled, Technically Not possible as "ADD NEW DEVICE IS DISABLED". (TODO - Check for PTT ??)
    *    b. When Selection Not in Progress - Already taken care - PREPARE is already triggered in WifiSetupHandle::vInitiateWirelessDiscovery (IsKnownDevice == FALSE)
    *  3. Handle Multiple Initiate Wireless Discovery Call for the same device when already one is triggered -
    *    a. Selection in progress
    *    b. Selection not in progress.
    *  4. Handle InitiateWireless Discovery Call for different device
    *    a. when Session(any technology) is already active
    *    b. when selection is not in progress for the other device.
    * */

   t_U32 u32DeviceHandle = 0;
   t_Bool bProceedWithWirelessDiscovery = false;
   trDeviceSelectionStateDetails rDeviceSelectionDetails;

   ETG_TRACE_USR1(("spi_tclWiFi::bHandleInitiateWirelessDiscovery() Entered with Device Category %d with BTMac Address as %s", ETG_ENUM(DEVICE_CATEGORY,
            enDevCategory), coszBTMACAddress.c_str()));

   //! Get the Selection State from the DeviceSelector
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vGetDeviceSelectorCurrentSelectionState(rDeviceSelectionDetails);
   }

   if (true == coszBTMACAddress.empty())
   {
      bProceedWithWirelessDiscovery = true;
   }
   else
   {
      //Get the Device ID from the BT MAC address - Since it's a Known device we should be able to get the Proper Device ID from Device list handler
      spi_tclDeviceIDDataIntf oDataIntf;
      u32DeviceHandle = oDataIntf.u32GetDeviceIDFromBTAddress(coszBTMACAddress);

      ETG_TRACE_USR1(("spi_tclWiFi::bHandleInitiateWirelessDiscovery() Device handle %d has BTMac Address as %s", u32DeviceHandle, coszBTMACAddress.c_str()));

      switch (rDeviceSelectionDetails.enDeviceSelectionState)
      {
         case e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS:
            //! when selection is progress for any device, just ignore the InitiateWireless Discovery Methodstart and
            m_bInitiateWirelessDiscoveryRequestPending = false;
            bProceedWithWirelessDiscovery = false;
            vSendInitiateWirelessDiscoveryResult(false);
            break;

         case e8_SELECTION_STATE_DEVICE_NOT_SELECTED:
         case e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR:
         case e8_SELECTION_STATE_DEVICE_USER_DESELECTED:
         case e8_SELECTION_STATE_DEVICE_REENUMERATION_INPROGRESS:
         case e8_SELECTION_STATE_DEVICE_SELECTION_ERROR:
            //! for all these cases - Its valid to Proceed with the Wireless Discovery for the Same device or might be different device in the list
            m_bInitiateWirelessDiscoveryRequestPending = false;
            bProceedWithWirelessDiscovery = true;
            break;

         case e8_SELECTION_STATE_DEVICE_SELECTED:
            /*
             * trigger DESELECT (With deselect reason - AUTOMATIC_DESELECT_SELECT) for the currently selected device if it is different device and store the PendingRequest, process this request in the SelectDeviceResult
             * Otherwise, Ignore the request as the same device is already selected
             *
             * */
            if (u32DeviceHandle != rDeviceSelectionDetails.u32DeviceHandle)
            {
               ETG_TRACE_USR1((" spi_tclWiFi::bHandleInitiateWirelessDiscovery() Device handle %d triggered with Automatic deselection", rDeviceSelectionDetails.u32DeviceHandle));
               //! set the requet pending flag as true and process it the SelectDeviceResult
               m_bInitiateWirelessDiscoveryRequestPending = true;
               bProceedWithWirelessDiscovery = false;

               spi_tclMediator* poMediator = spi_tclMediator::getInstance();
               if (NULL != poMediator)
               {
                  poMediator->vPostAutoDeviceSelection(rDeviceSelectionDetails.u32DeviceHandle,
                           e8DEVCONNREQ_DESELECT,
                           e8_REASON_HMI_SELECT_SPI_AUTO_DESELECT_TRIGGER);
               }
            }
            else
            {
               //! Ignore the Requets from HMI as the Same device is already Selected
               //! send the response back to HMI as false for the MethodStart InitiateWirelessDiscover
               ETG_TRACE_USR1((" spi_tclWiFi::bHandleInitiateWirelessDiscovery() Ignoring the Request as the same device with device handle %d is already selected ", rDeviceSelectionDetails.u32DeviceHandle));
               m_bInitiateWirelessDiscoveryRequestPending = false;
               bProceedWithWirelessDiscovery = false;
               vSendInitiateWirelessDiscoveryResult(false);
            }
            break;

         case e8_SELECTION_STATE_DEVICE_DESELECTION_INPROGRESS:
            //! for this case, store the PendingRequest, process this request in the SelectDeviceResult
            ETG_TRACE_USR1((" spi_tclWiFi::bHandleInitiateWirelessDiscovery() Processing this request as DESELECTION is in PROGRESS for the Device %d", rDeviceSelectionDetails.u32DeviceHandle));
            m_bInitiateWirelessDiscoveryRequestPending = true;
            bProceedWithWirelessDiscovery = false;

            break;
      }
   }

   ETG_TRACE_USR1((" spi_tclWiFi::bHandleInitiateWirelessDiscovery()left with m_bInitiateWirelessDiscoveryRequestPending %d and bProceedWithWirelessDiscovery %d  ", ETG_ENUM(BOOL,
            m_bInitiateWirelessDiscoveryRequestPending), ETG_ENUM(BOOL, bProceedWithWirelessDiscovery)));
   return bProceedWithWirelessDiscovery;
}

/***************************************************************************
 ** FUNCTION:  spi_tclWiFi::vRestoreSettings
 ***************************************************************************/
t_Void spi_tclWiFi::vRestoreSettings()
{
   ETG_TRACE_USR1(("spi_tclWiFi::vRestoreSettings Entered"));

   if(false == m_szWiFiCredentialExchangePendingForDevice.empty())
   {
      m_szWiFiCredentialExchangePendingForDevice.clear();
   }

   m_CredentialsExchangeLock.s16Lock();
   m_setAccessoryWifiInfoExchangedDevices.clear();
   m_CredentialsExchangeLock.vUnlock();

   //TODO - For Now, its only done DipoWifi, please extend this for AAP wifi class too
   if (NULL != m_poWiFiDevBase[e8DEV_TYPE_DIPO])
   {
      m_poWiFiDevBase[e8DEV_TYPE_DIPO]->vRestoreSettings();
   }

   m_bExchangeCredentials = false;
}

/***************************************************************************
 ** FUNCTION:  spi_tclWiFi::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclWiFi::vRegisterCallbacks()
{
   ETG_TRACE_USR1(("spi_tclWiFi::vRegisterCallbacks() entered "));

   trWiFiCallbacks rWifiCallback;
   rWifiCallback.fvDeviceDeselectionOnDisconnection =
            std::bind(&spi_tclWiFi::vDeselectionOnDisconnection,
                     this,
                     SPI_FUNC_PLACEHOLDERS_2);

   rWifiCallback.fvDeviceSelectionOnConnection =
            std::bind(&spi_tclWiFi::vHandleDeviceSelection,
                     this,
                     SPI_FUNC_PLACEHOLDERS_1);


   for(t_U8 u8Index = 0; u8Index < e8DEV_TYPE_MAX; u8Index++)
   {
      if (true == bValidateClient(u8Index))
      {
         ETG_TRACE_USR4(("spi_tclWiFi::vRegisterCallbacks %d", u8Index));
         m_poWiFiDevBase[u8Index]->vRegisterCallbacks(rWifiCallback);
      }
   }
}


/***************************************************************************
 ** FUNCTION:  spi_tclWiFi::vDeselectionOnDisconnection(...)
 ***************************************************************************/
t_Void spi_tclWiFi::vDeselectionOnDisconnection(const t_U32 cou32DeviceHandle, tenSelectReason enSelectionReason)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vDeviceDeselectionOnDisconnection() entered with Device handle %d and having selection reason as %d",cou32DeviceHandle,
            enSelectionReason ));

   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if(NULL != poMediator)
   {
      poMediator->vPostAutoDeviceSelection(cou32DeviceHandle, e8DEVCONNREQ_DESELECT,enSelectionReason);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWiFi::bValidateClient(...)
 ***************************************************************************/
t_Bool spi_tclWiFi::bValidateClient(t_U8 u8Index)
{
   //Assert if the Index is greater than the array size
   SPI_NORMAL_ASSERT( u8Index > NUM_WIFI_CLIENTS);
   t_Bool bRet = (u8Index < NUM_WIFI_CLIENTS)
              && (NULL != m_poWiFiDevBase[u8Index]);
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vHandleDeviceSelection()
 ***************************************************************************/
t_Void spi_tclWiFi::vHandleDeviceSelection(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclWiFi::vHandleDeviceSelection(): Entered with STA Information-Device Handle %d", cou32DeviceHandle));
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   spi_tclDeviceIDDataIntf oDataIntf;

   //validate first whether it is allowed to start automatic selection
   t_Bool bisPreselectionValid = false;
   if(NULL != poMediator)
   {
      poMediator->vOnDeviceConnectstoWirelessAP(cou32DeviceHandle, bisPreselectionValid);
      if(true == bisPreselectionValid)
      {
         ETG_TRACE_USR4(("spi_tclWiFi::vHandleDeviceSelection(): Device Handle : %d connected to wireless AP and Can be selected automatically", cou32DeviceHandle));
         //triggering automatic selection for the Device as it connected to Carplay wireless AP
         t_String szBTMacAddress;
         tenDeviceCategory enDevCategory = e8DEV_TYPE_UNKNOWN;
         enDevCategory = oDataIntf.enGetDeviceCategory(cou32DeviceHandle);

         oDataIntf.vGetBTAddress(szBTMacAddress,cou32DeviceHandle);

         //Before triggering Initiate wireless discovery , set the context to 0 as InitiateWirelessDiscoveryResult should not reach HMI in this case.
         trUserContext rfcorUsrCntxt;
         rfcorUsrCntxt.u32SrcAppID = DEFAULT_USER_CONTEXT;
         rfcorUsrCntxt.u32DestAppID= DEFAULT_USER_CONTEXT;
         poMediator->vPostInitiateWirelessDiscovery(rfcorUsrCntxt, szBTMacAddress, enDevCategory);
      }
      else
      {
         ETG_TRACE_USR4(("spi_tclWiFi::vHandleDeviceSelection(): Device Handle : %d connected to wireless AP, but can not be selected automatically", cou32DeviceHandle));
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vSetUserDeselect()
 ***************************************************************************/
t_Void spi_tclWiFi::vSetUserDeselect(const t_String& corszBTMACAddress, t_Bool bUserDeslect)
{
   spi_tclDeviceIDDataIntf oDataIntf;
   t_U32 u32DeviceHandle = oDataIntf.u32GetDeviceIDFromBTAddress(corszBTMACAddress);
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if((NULL != poMediator) && (0 != u32DeviceHandle))
   {
      poMediator->vPostSetUserDeselect(u32DeviceHandle,bUserDeslect);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWiFi::vSetExchangeCredentails()
 ***************************************************************************/
t_Void spi_tclWiFi::vSetExchangeCredentails(t_Bool bSetExchangeCredentails)
{
   m_bExchangeCredentials = bSetExchangeCredentails;
}
//lint -restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
