/***********************************************************************/
/*!
* \file  spi_tclWiFiDevBase.h
* \brief Base class for WiFi specific impl
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Base class for WiFi specific impl
AUTHOR:         Shiva Kumar Gurija
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
13.02.2018  | Shiva Kumar Gurija    | Initial Version

\endverbatim
*************************************************************************/

#ifndef _SPI_tclWIFIDEVBASE_H_
#define _SPI_tclWIFIDEVBASE_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "BaseTypes.h"
#include "spi_WifiTypedefs.h"
#include "spi_tclWiFiSettingsIntf.h"

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/


/****************************************************************************/
/*!
* \class spi_tclWiFiDevBase
* \brief Base class for WiFi specific impl
****************************************************************************/
class spi_tclWiFiDevBase
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclWiFiDevBase::spi_tclWiFiDevBase()
   ***************************************************************************/
   /*!
   * \fn      spi_tclWiFiDevBase()
   * \brief   Default Constructor
   * \sa      ~spi_tclWiFiDevBase()
   **************************************************************************/
   explicit spi_tclWiFiDevBase():m_poWiFiSettings(NULL)
   {
      //Add code
   }

   /***************************************************************************
   ** FUNCTION:  spi_tclWiFiDevBase::~spi_tclWiFiDevBase()
   ***************************************************************************/
   /*!
   * \fn      virtual ~spi_tclWiFiDevBase()
   * \brief   Destructor
   * \sa      spi_tclWiFiDevBase()
   **************************************************************************/
   virtual ~spi_tclWiFiDevBase()
   {
      m_poWiFiSettings = NULL;
   }
  
   /***************************************************************************
   ** FUNCTION:  spi_tclWiFiDevBase::spi_tclWiFiDevBase(const spi_tclWiFiDevBase&)
   ***************************************************************************/
   /*!
   * \fn      spi_tclWiFiDevBase(const spi_tclWiFiDevBase&) = delete;
   * \brief   deletes default copy Constructor
   * \sa      ~spi_tclWiFiDevBase()
   **************************************************************************/
   spi_tclWiFiDevBase(const spi_tclWiFiDevBase&) = delete;

   /***************************************************************************
   ** FUNCTION:  spi_tclWiFiDevBase& soperator=(const spi_tclWiFiDevBase&)
   ***************************************************************************/
   /*!
   * \fn      spi_tclWiFiDevBase& operator=(const spi_tclWiFiDevBase&) = delete;
   * \brief   deletes default assignment operator for the class
   **************************************************************************/
   spi_tclWiFiDevBase& operator=(const spi_tclWiFiDevBase&) = delete;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclWiFiDevBase::bInitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bInitialize()
   * \brief   To Initialize all the WiFi related classes
   * \retval  t_Bool
   * \sa      vUninitialize()
   **************************************************************************/
   virtual t_Bool bInitialize()=0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFiDevBase::vUninitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Void vUninitialize()
   * \brief   To Uninitialize all the WiFi related classes
   * \retval  t_Void
   * \sa      bInitialize()
   **************************************************************************/
   virtual t_Void vUninitialize()=0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFiDevBase::vSelectDevice()
   ***************************************************************************/
   /*!
   * \fn      virtual t_Void vSelectDevice(const t_U32 cou32DevId,
   *          const tenDeviceConnectionReq coenConnReq)
   * \brief   To Initialize/UnInitialize WiFi for thr currently selected device
   * \pram    cou32DevId : [IN] Unique Device Id
   * \param   coenConnReq : [IN] connected/disconnected
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vSelectDevice(const t_U32 /*cou32DevId*/,
      const tenDeviceConnectionReq /*coenConnReq*/){}

  /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFiDevBase::vOnSelectDeviceResult()
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSelectDeviceResult(const t_U32 cou32DevId,
   *                 const tenDeviceConnectionReq coenConnReq,
   *                 const tenResponseCode coenRespCode)
   * \brief   To perform the actions that are required, after the select device is
   *           successful/failed
   * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
   * \pram    coenConnReq : [IN] Identifies the Connection Request.
   * \pram    coenRespCode: [IN] Response code. Success/Failure
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vOnSelectDeviceResult(const t_U32 /*cou32DevId*/,
      const tenDeviceConnectionReq /*coenConnReq*/,
      const tenResponseCode /*coenRespCode*/){}

  /***************************************************************************
   ** FUNCTION:  t_U32 spi_tclWiFiDevBase::u32GetDisconnectedStation()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetDisconnectedStation(
   *                 const std::vector<trStationInfo>& corfrStationsInfo)
   * \brief   Method to check whether station which with SPI session is active got disconnected.
   * \param   [IN] corvecStationsInfo: List of connected stations.
   **************************************************************************/ 
   virtual t_U32 u32GetDisconnectedStation(const std::vector<trStationInfo>& /*corfrStationsInfo*/)
   {
	   return 0;
   }
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFiDevBase::vSetWiFiSettingsInstance(spi_tclWiFiSettingsIntf* poWiFiSettingsIntf)
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetWiFiSettingsInstance()
   * \brief   Method to set Wifi Settings instance.
   * \param   poWiFiSettingsIntf : pointer to WiFiSettingsIntf.
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetWiFiSettingsInstance(spi_tclWiFiSettingsIntf* poWiFiSettingsIntf)
   {
      if( NULL != poWiFiSettingsIntf)
      {
         m_poWiFiSettings = poWiFiSettingsIntf;
      }
   }
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFiDevBase::vPostWiFiConfig()
   ***************************************************************************/
   /*!
   * \fn      t_Void vPostWiFiConfig(const trWiFiConfig &rfcorWiFiConfig,const std::vector<trStationInfo>& corvecStationsInfo)
   * \brief   Update from WBL
   * \param   [IN] rfcorWiFiConfig : Wifi AP configuration
   * \param   [IN] corvecStationsInfo: List of connected stations.
   **************************************************************************/ 
   virtual t_Void vPostWiFiConfig(const trWiFiConfig &/*rfcorWiFiConfig*/,const std::vector<trStationInfo>& /*corvecStationsInfo*/)
   {
   }
   
   /***************************************************************************
    ** FUNCTION:  spi_tclWiFiDevBase::vRestoreSettings
    ***************************************************************************/
   /*!
    * \fn     t_Void vRestoreSettings()
    * \brief  vRestoreSettings Method. Invoked on factory reset to clear the Private Data
    * \param  None
    **************************************************************************/
   virtual t_Void vRestoreSettings(){}

   /***************************************************************************
    ** FUNCTION:  spi_tclWiFiDevBase::vRegisterCallbacks
    ***************************************************************************/
   /*!
    * \fn     vRegisterCallbacks()
    * \brief  interface for the creator class to register for the required
    *        callbacks.
    * \param rfrWifiCallbacks : reference to the callback structure
    *        populated by the caller
    **************************************************************************/
   virtual t_Void vRegisterCallbacks(trWiFiCallbacks &/*rfrWifiCallbacks*/)
   {

   }



   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:
   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/
   
   spi_tclWiFiSettingsIntf* m_poWiFiSettings;

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/


   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/

}; //spi_tclWiFiDevBase

#endif //_SPI_tclWIFIDEVBASE_H_


///////////////////////////////////////////////////////////////////////////////
// <EOF>

