/***********************************************************************/
/*!
* \file  spi_tclResourceMngr.cpp
* \brief Main Resource Manager class that provides interface to delegate 
*        the Resource management mainly of the screen and Audio
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         Priju K Padiyath
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
05.04.2014  | Priju K Padiyath      | Initial Version
25.04.2014  | Shihabudheen P M      | Modified for audio context handling
25.06.2014  | Shihabudheen P M      | Adapted to the CarPlay design changes
20.03.2015  | Shiva Kumar G         | updated with the elements to Android Auto
20.03.2015  | Ramya Murthy          | Included AAP ResourceManager
25.06.2015  | Tejaswini HB          |Featuring out Android Auto

15.06.2015  | Shihabudheen P M      | added vSetVehicleBTAddress()
10.07.2015  | Shihabudheen P M      | Removed vOnCarPlayPluginLoaded.
29.11.2016  | Noopur R K            | HMI Synchronization
01.03.2017 |  Shiva Kumar G         | Changes to enable/disable CarPlay feature    
\endverbatim
*************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "RespRegister.h"
#include "spi_tclMediator.h"
#include "spi_tclRespInterface.h"
#include "spi_tclAppMngrRespIntf.h"
#include "spi_tclResorceMngrDefines.h"
#include "spi_tclResourceMngrBase.h"
#include "spi_tclMediator.h"
#include "spi_tclResourceMngr.h"
#include "spi_tclAudio.h"
#include "spi_tclFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
#include "spi_tclAAPResourceMngr.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
#include "spi_tclDiPOResourceMngr.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
#include "spi_tclMySPINResourceMngr.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
#include "spi_tclBDCLResourceMngr.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
#include "spi_tclOnCarResourceMngr.h"
#endif

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
#include "trcGenProj/Header/spi_tclResourceMngr.cpp.trc.h"
#endif
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
std::map<tenDeviceCategory,GetResourceMngr_t*> ResourceMngr;
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/


/***************************************************************************
** FUNCTION:  spi_tclResourceMngr::spi_tclResourceMngr()
***************************************************************************/
spi_tclResourceMngr::spi_tclResourceMngr(spi_tclResourceMngrResp* poRespIntf) : spi_tclSelectionIntf(e32COMPID_RESOURCEMNGR),
m_poRsrcMngrRespIntf(poRespIntf),
m_u32SelectedDevice(0),
m_enDevCat(e8DEV_TYPE_UNKNOWN),
m_bAudioSrcIdleRequired(false),
m_poResourceMngrSettings(NULL)

{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   m_rCurrAppState.enSpeechAppState = e8SPI_SPEECH_UNKNOWN;
   m_rCurrAppState.enPhoneAppState = e8SPI_PHONE_UNKNOWN;
   m_rCurrAppState.enNavAppState = e8SPI_NAV_UNKNOWN;

   for (t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
   {
      m_poRsrcMngrBase[u8Index] = NULL;
   }
}

/***************************************************************************
** FUNCTION:  spi_tclAppMngr::~spi_tclResourceMngr()
***************************************************************************/
spi_tclResourceMngr::~spi_tclResourceMngr()
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));

   m_poRsrcMngrRespIntf    = NULL;
   m_u32SelectedDevice     = 0;

   m_rCurrAppState.enSpeechAppState = e8SPI_SPEECH_UNKNOWN;
   m_rCurrAppState.enPhoneAppState = e8SPI_PHONE_UNKNOWN;
   m_rCurrAppState.enNavAppState = e8SPI_NAV_UNKNOWN;
   m_poResourceMngrSettings =NULL;

   for (t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
   {
      m_poRsrcMngrBase[u8Index] = NULL;
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclResourceMngr::bInitialize()
***************************************************************************/
t_Bool spi_tclResourceMngr::bInitialize()
{
   t_Bool bRet = false;

   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));

#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
   m_poRsrcMngrBase[e8DEV_TYPE_DIPO] = new spi_tclDiPOResourceMngr();
   SPI_NORMAL_ASSERT(NULL == m_poRsrcMngrBase[e8DEV_TYPE_DIPO]);
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
   if ((NULL != ResourceMngr[e8DEV_TYPE_MIRRORLINK]))
      {
         ETG_TRACE_USR1(("spi_tclResourceMngr::bInitialize before obj create m_poRsrcMngrBase[e8DEV_TYPE_MIRRORLINK] = %p ", m_poRsrcMngrBase[e8DEV_TYPE_MIRRORLINK]));
         m_poRsrcMngrBase[e8DEV_TYPE_MIRRORLINK] = ResourceMngr[e8DEV_TYPE_MIRRORLINK]();
         SPI_NORMAL_ASSERT(NULL == m_poRsrcMngrBase[e8DEV_TYPE_MIRRORLINK]);
      }
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
   m_poRsrcMngrBase[e8DEV_TYPE_ANDROIDAUTO] = new spi_tclAAPResourceMngr();
   SPI_NORMAL_ASSERT(NULL == m_poRsrcMngrBase[e8DEV_TYPE_ANDROIDAUTO]);
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
   m_poRsrcMngrBase[e8DEV_TYPE_MYSPIN] = new (std::nothrow) spi_tclMySPINResourceMngr();
   SPI_NORMAL_ASSERT(NULL == m_poRsrcMngrBase[e8DEV_TYPE_MYSPIN]);
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
   m_poRsrcMngrBase[e8DEV_TYPE_CARLIFE] = new (std::nothrow) spi_tclBDCLResourceMngr();
   SPI_NORMAL_ASSERT(NULL == m_poRsrcMngrBase[e8DEV_TYPE_CARLIFE]);
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
   m_poRsrcMngrBase[e8DEV_TYPE_ONCAR] = new (std::nothrow) spi_tclOnCarResourceMngr();
   SPI_NORMAL_ASSERT(NULL == m_poRsrcMngrBase[e8DEV_TYPE_ONCAR]);
#endif

   RespRegister *pRespRegister = RespRegister::getInstance();
   if (NULL != pRespRegister)
   {
      pRespRegister->bRegisterObject((spi_tclExtRespTelephone*) this);
   }

   for(t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
   {
      tenDeviceCategory enDevCat = static_cast<tenDeviceCategory>(u8Index);
   if (NULL != m_poRsrcMngrBase[enDevCat])
   {
      bRet = m_poRsrcMngrBase[enDevCat]->bInitialize() || bRet;
   }
   }

   //Register for notifications from Mediator
   trNotifyRsrcMngrCbs rNotifyRsrcMngrCbs;
   rNotifyRsrcMngrCbs.fvOnDisableBluetoothTrigger = std::bind(&spi_tclResourceMngr::vOnDisableBluetoothTrigger,this,
		     SPI_FUNC_PLACEHOLDERS_1);
   rNotifyRsrcMngrCbs.fvEvaluateInitialModes = std::bind(&spi_tclResourceMngr::vOnEvaluateInitalModes,this,
            SPI_FUNC_PLACEHOLDERS_1);

   rNotifyRsrcMngrCbs.fvUpdateSessionStatus =
      std::bind(&spi_tclResourceMngr::vCbUpdateSessionStatus,
      this,
      std::placeholders::_1,
      std::placeholders::_2,
      std::placeholders::_3);

   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vRegisterCallbacks(rNotifyRsrcMngrCbs);
   }
   //Register for call backs
   vRegisterCallbacks();

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclResourceMngr::vUnInitialize()
***************************************************************************/
t_Bool spi_tclResourceMngr::bUnInitialize()
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   t_Bool bRetVal = true;

   for (t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
   {
      if (NULL != m_poRsrcMngrBase[u8Index])
      {
         m_poRsrcMngrBase[u8Index]->vUnInitialize();
      } //if((NULL != m_poVideoDevBase[
      RELEASE_MEM(m_poRsrcMngrBase[u8Index]);
   } //for(t_U8 u8Index=0;u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)

   m_poRsrcMngrRespIntf = NULL;
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclResourceMngr::vLoadSettings()
***************************************************************************/
t_Void spi_tclResourceMngr::vLoadSettings()
{
   ETG_TRACE_USR1(("spi_tclResourceMngr::vLoadSettings entered"));
	ETG_TRACE_USR1(("spi_tclResourceMngr::vLoadSettings - m_bAudioSrcIdleRequired is %d", ETG_ENUM(BOOL, m_bAudioSrcIdleRequired)));

	if(NULL != m_poResourceMngrSettings)
	{
	   for (t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
	   {
	      if (NULL != m_poRsrcMngrBase[u8Index])
	      {
	         m_poRsrcMngrBase[u8Index]->vSetResourceMngrSettingsInstance(m_poResourceMngrSettings);
	      } //if((NULL != m_poVideoDevBase[
	   } //for(t_U8 u8Index=0;u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)

       trRsrcSettings rRsrcSettings;
       rRsrcSettings.u32StartTimeInterval = m_poResourceMngrSettings->u32GetStartUpTimeInterval();

       // After Ignition cycle, read the datapool and get the last mode for audio and display
       t_Bool bIsDisplayLastModeProjection = false;
       t_Bool bIsAudioLastModeProjection = false;

       bIsDisplayLastModeProjection = m_poResourceMngrSettings->bIsDisplayLastModeProjection();
       bIsAudioLastModeProjection = m_poResourceMngrSettings->bIsAudioLastModeProjection();

       m_bAudioSrcIdleRequired = bIsAudioLastModeProjection;

       ETG_TRACE_USR1(("spi_tclResourceMngr::bInitialize: Last display mode is projection - %d",
                ETG_ENUM(BOOL, bIsDisplayLastModeProjection)));
       ETG_TRACE_USR1(("spi_tclResourceMngr::bInitialize: Last audio mode is projection - %d",
                ETG_ENUM(BOOL, bIsAudioLastModeProjection)));

       // If the last mode was projection, send the display and audio contexts as projection.
       // This is to handle last mode over ignition cycle.
       // In case HMI doesn't send any context update, these contexts can be used for Info response message.
       trUserContext rUsrContext;

       for(t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
       {
          tenDeviceCategory enDevCat = static_cast<tenDeviceCategory>(u8Index);
          t_Bool bRet = false;
          if (NULL != m_poRsrcMngrBase[enDevCat])
          {
             bRet = m_poRsrcMngrBase[enDevCat]->bInitialize() || bRet;

             //@Note: ETC is used for DIPO only as of now.
             rRsrcSettings.m_enETCSupport = m_poResourceMngrSettings->enGetCarPlayETCEnabledFlag(enDevCat); //TODO - rename function
             rRsrcSettings.m_bIsLastModeSupported = m_poResourceMngrSettings->bGetLastModeSupport(enDevCat);

             m_poRsrcMngrBase[enDevCat]->vUpdateInitialSettings(rRsrcSettings);
          }//if(NULL != m_poRsrcMngrBase[u8Index])
       }
	}

   if (true == m_bAudioSrcIdleRequired)
   {
      spi_tclFactory* m_poFactory = spi_tclFactory::getInstance();
      spi_tclAudio *poAudio = m_poFactory->poGetAudioInstance();
      if (NULL != poAudio)
      {
         // Set Audio source as playable content unknown with new media as reason.
         // This is done to stop switching to native audio after ignition cycle.
         ETG_TRACE_USR2(("[DESC]::vLoadSettings: setting main audio source available since last mode audio is set"));
         poAudio->bSetAudioSrcAvailability(e8AUD_MAIN_OUT, e8SRC_PLAYABLE, e8REASON_SAMEMEDIA);
      }
      m_bAudioSrcIdleRequired = false;

   }//if (true == m_bAudioSrcIdleRequired)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclResourceMngr::vSelectDeviceResult()
***************************************************************************/
t_Void spi_tclResourceMngr::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq,
         tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));

   if((e8NO_ERRORS == enErrorCode) && (e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq))
   {
      m_u32SelectedDevice = corfrSelectReq.m_u32DeviceHandle;
      m_enDevCat = corfrSelectReq.m_enDevCategory;
   }
   tenResponseCode enRespCode = (e8NO_ERRORS == enErrorCode)? e8SUCCESS : e8FAILURE;

   t_U8 u8Index = static_cast<t_U8>(corfrSelectReq.m_enDevCategory);
   if ((true == bValidateClient(u8Index)) && (NULL != m_poRsrcMngrBase[u8Index]))
   {
      m_poRsrcMngrBase[u8Index]->vOnSPISelectDeviceResult(corfrSelectReq.m_u32DeviceHandle,
               corfrSelectReq.m_enDevConnReq, enRespCode, enErrorCode);
   }//if ((true == bValidateClient(u8Index)) &&...)

   //Set the device id to zero, only after the selectdevice result is processed by sub-components.
   //or else DeviceDisplayContext update would be sent with invalid device handlde (Zero)
   if(e8DEVCONNREQ_DESELECT == corfrSelectReq.m_enDevConnReq)
   {
      m_u32SelectedDevice = 0;
      m_enDevCat = e8DEV_TYPE_UNKNOWN;
   }
}

/***************************************************************************
** FUNCTION:  t_Void  spi_tclResourceMngr::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclResourceMngr::vRegisterCallbacks()
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));

   trRsrcMngrCallback rRsrcMngrCallback;

   /*lint -esym(40,fvSelectDeviceResp)fvSelectDeviceResp Undeclared identifier */
   /*lint -esym(40,_1) _1 identifier */
   /*lint -esym(40,fvPostDeviceDisplayContext)fvPostDeviceDisplayContext Undeclared identifier */
   /*lint -esym(40,_2) _2 Undeclared identifier */
   /*lint -esym(40,_3 ) _3 Undeclared identifier */
   /*lint -esym(40,fvPostDeviceAudioContext ) fvPostDeviceAudioContext Undeclared identifier */
   /*lint -esym(40,fvPostDeviceAppState) fvPostDeviceAppState Undeclared identifier */
   /*lint -esym(40,fvUpdateSessionStatus) fvUpdateSessionStatus Undeclared identifier */
   /*lint -esym(40,fpvDeviceAuthAndAccessCb)fpvDeviceAuthAndAccessCb Undeclared identifier */


   rRsrcMngrCallback.fvPostDeviceDisplayContext =
      std::bind(&spi_tclResourceMngr::vCbPostDeviceDisplayContext,
      this,
      std::placeholders::_1,
      std::placeholders::_2,
      std::placeholders::_3,
      std::placeholders::_4);

   rRsrcMngrCallback.fvPostDeviceAudioContext =
      std::bind(&spi_tclResourceMngr::vCbDeviceAudioContext,
      this,
      std::placeholders::_1,
      std::placeholders::_2,
      std::placeholders::_3,
      std::placeholders::_4);

   rRsrcMngrCallback.fvPostDeviceAppState = 
      std::bind(&spi_tclResourceMngr::vCbDeviceAppState,
      this,
      std::placeholders::_1,
      std::placeholders::_2,
      std::placeholders::_3,
      std::placeholders::_4);

   rRsrcMngrCallback.fvUpdateSessionStatus = 
      std::bind(&spi_tclResourceMngr::vCbUpdateSessionStatus,
      this,
      std::placeholders::_1,
      std::placeholders::_2,
      std::placeholders::_3);

   rRsrcMngrCallback.fvSetDeviceAppState =
      std::bind(&spi_tclResourceMngr::vCbSetDeviceAppState,
      this,
      std::placeholders::_1,
      std::placeholders::_2,
      std::placeholders::_3);

   rRsrcMngrCallback.fpvDeviceAuthAndAccessCb = std::bind(
      &spi_tclResourceMngr::vDevAuthAndAccessInfoCb,this,
      SPI_FUNC_PLACEHOLDERS_2);

   rRsrcMngrCallback.fvSelectDeviceResult = std::bind(
            &spi_tclResourceMngr::vCbSelectDeviceResult,
            this,
            std::placeholders::_1);
			
   rRsrcMngrCallback.fvInitiateCall =
      std::bind(&spi_tclResourceMngr::vCbInitiateCall,
      this,
      std::placeholders::_1,
      std::placeholders::_2,
      std::placeholders::_3);

   rRsrcMngrCallback.fvVoiceSessionReq =
      std::bind(&spi_tclResourceMngr::vCbVoiceSessionReq,
      this,
      std::placeholders::_1,
      std::placeholders::_2);
	  
   rRsrcMngrCallback.fvProjectionVideoRenderingStatus = 
      std::bind(&spi_tclResourceMngr::vCbProjectionVideoRenderingStatus,
      this,
      std::placeholders::_1);

   rRsrcMngrCallback.fvPostDeviceSpmSubState =
      std::bind(&spi_tclResourceMngr::vCbDeviceSpmSubState,
      this,
      std::placeholders::_1,
      std::placeholders::_2);

   for (t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
   {
      if (NULL != m_poRsrcMngrBase[u8Index])
      {
         m_poRsrcMngrBase[u8Index]->vRegRsrcMngrCallBack(rRsrcMngrCallback);
      }//if (NULL != m_poRsrcMngrBase[u8Index])
   }// end of for
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclResourceMngr::vSetAccessoryDisplayContext()
***************************************************************************/
t_Void spi_tclResourceMngr::vSetAccessoryDisplayContext(const t_U32 cou32DevId,
                                                        t_Bool bDisplayFlag, 
                                                        tenDisplayContext enDisplayContext,
                                                        const trUserContext& rfrcUsrCntxt)
{
   ETG_TRACE_USR4(("spi_tclResourceMngr::vSetAccessoryDisplayContext:Device ID-0x%x,Accesssory Takes Focus-%d ,AccDisplayContext = %d\n",
      cou32DevId,bDisplayFlag,enDisplayContext));

   for (t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
   {
      if (NULL != m_poRsrcMngrBase[u8Index])
      {
         m_poRsrcMngrBase[u8Index]->vSetAccessoryDisplayContext(cou32DevId,
                  bDisplayFlag,
                  enDisplayContext,
                  rfrcUsrCntxt);
      }
   }
   
   trDisplayContext rDisplayContext;
   rDisplayContext.bDisplayFlag = bDisplayFlag;
   rDisplayContext.enDisplayContext = enDisplayContext;
   trDisplayConstraint rDummyDisplayConstraint;

   vStoreDisplayLastMode(e8_DISPLAY_CONTEXT, rDisplayContext, rDummyDisplayConstraint);
}

/***************************************************************************
** FUNCTION: t_Void spi_tclResourceMngr::vSetAccessoryDisplayMode(t_U32...
***************************************************************************/
t_Void spi_tclResourceMngr::vSetAccessoryDisplayMode(const t_U32 cu32DeviceHandle,
                                                     const tenDeviceCategory enDevCat,
                                                     const trDisplayContext corDisplayContext,
                                                     const trDisplayConstraint corDisplayConstraint,
                                                     const tenDisplayInfo coenDisplayInfo)
{
   SPI_INTENTIONALLY_UNUSED(enDevCat);
   ETG_TRACE_USR1(("[FUNC]spi_tclResourceMngr::vSetAccessoryDisplayMode() entered \n"));
   ETG_TRACE_USR1(("[PARAM]::vSetAccessoryDisplayMode: Device Handle = %d DisplayContextHanding = %d",
		   		   cu32DeviceHandle, ETG_ENUM(DISPLAY_INFO, coenDisplayInfo)));

   if (e8_DISPLAY_CONSTRAINT == coenDisplayInfo)
   {
      // Print the values only if it is HMI responsibility, otherwise unnecessary print values will confused for SPI responsibility
	   ETG_TRACE_USR1(("[PARAM]::vSetAccessoryDisplayMode: TransferType = %d TransferPriority = %d TakeConstraint = %d BorrowConstraint = %d",
	           ETG_ENUM( DIPO_TRANSFERTYPE, corDisplayConstraint.enTransferType),
	           ETG_ENUM( DIPO_TRANSFERPRIO, corDisplayConstraint.enTransferPriority),
	           ETG_ENUM( DIPO_CONSTRAINT, corDisplayConstraint.enTakeConstraint),
	           ETG_ENUM( DIPO_CONSTRAINT, corDisplayConstraint.enBorrowConstraint)));
   }

   for (t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
   {
      if (NULL != m_poRsrcMngrBase[u8Index])
      {
         m_poRsrcMngrBase[u8Index]->vSetAccessoryDisplayMode(cu32DeviceHandle,
                  corDisplayContext,
                  corDisplayConstraint,
                  coenDisplayInfo);
      }
   }

   vStoreDisplayLastMode(coenDisplayInfo, corDisplayContext, corDisplayConstraint);
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vSetAccessoryAudioContext()
***************************************************************************/
t_Void spi_tclResourceMngr::vSetAccessoryAudioContext(const t_U32 cou32DevId, 
                                                      const tenAudioContext coenAudioCntxt,
                                                      t_Bool bReqFlag, 
                                                      tenDeviceCategory enSelDevCat,
                                                      const trUserContext& rfrcUsrCntxt,
                                                      tenDeviceCategory enRequestedDevCat,
                                                      t_Bool bIsHMITrigger)

{
   SPI_INTENTIONALLY_UNUSED(enSelDevCat);
   ETG_TRACE_USR1(("spi_tclResourceMngr::vSetAccessoryAudioContext entered"));

   t_U8 u8Index = static_cast<t_U8>(enRequestedDevCat);
   // Need to update the audio context updates to CarPlay plugin even if CarPlay is not active.
   // This information needs to keep to send the device upon session startup.
   // @Note: Audio context updates should be sent only to requested DeviceCategory handler,
   // since audio context changes are updated differently for different technologies.
   if (true == bValidateClient(u8Index))
   {
      m_poRsrcMngrBase[u8Index]->vSetAccessoryAudioContext(cou32DevId, coenAudioCntxt, bReqFlag, rfrcUsrCntxt);
   }

   // The last mode is stored only if audio context update was triggered by HMI to ignore internal context update (Eg: OnSrcActivity OFF)
   if (true == bIsHMITrigger)
   {
      vStoreAudioLastMode(bReqFlag, coenAudioCntxt);
   }
}


/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vSetAccessoryAppState()
***************************************************************************/
t_Void spi_tclResourceMngr::vSetAccessoryAppState(tenDeviceCategory enDevCat, 
                                                  const tenSpeechAppState enSpeechAppState, 
                                                  const tenPhoneAppState enPhoneAppState,
                                                  const tenNavAppState enNavAppState, 
                                                  const trUserContext& rfrcUsrCntxt)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   SPI_INTENTIONALLY_UNUSED(enDevCat);

   for (t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
   {
      if(NULL != m_poRsrcMngrBase[u8Index])
      {
         m_poRsrcMngrBase[u8Index]->vSetAccessoryAppState(enSpeechAppState, enPhoneAppState, enNavAppState, rfrcUsrCntxt);
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::bValidateClient()
***************************************************************************/
inline t_Bool spi_tclResourceMngr::bValidateClient(t_U8 u8Index)
{
   //Assert if the Index is greater than the array size
   SPI_NORMAL_ASSERT( u8Index > NUM_RSRCMNGR_CLIENTS);

   t_Bool bRet = (u8Index < NUM_RSRCMNGR_CLIENTS)
      && (NULL != m_poRsrcMngrBase[u8Index]);

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vCbPostDeviceDisplayContext()
***************************************************************************/
t_Void spi_tclResourceMngr::vCbPostDeviceDisplayContext(t_Bool bDisplayFlag,
         tenDisplayContextReason enDisplayContextReason, tenDiPOEntity enPermScreen, const trUserContext rcUsrCntxt)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   ETG_TRACE_USR4(("[DESC]:spi_tclResourceMngr::vCbPostDeviceDisplayContext- Device Display Context Updated with Status =  %d and DisplayContextReason =  %d",
                ETG_ENUM(BOOL, bDisplayFlag), ETG_ENUM(DISPLAY_CONTEXT_REASON,enDisplayContextReason)));
   if(NULL != m_poRsrcMngrRespIntf)
   {
      m_poRsrcMngrRespIntf->vPostDeviceDisplayContext(m_u32SelectedDevice, bDisplayFlag, enDisplayContextReason, enPermScreen, rcUsrCntxt);
   }//if(NULL != m_poRsrcMngrRespIntf)
}


/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vCbDeviceAudioContext()
***************************************************************************/
t_Void spi_tclResourceMngr::vCbDeviceAudioContext(t_Bool bPlayFlag, t_U8 u8AudioCntxt,
                                                  tenDiPOEntity enPermMainAudio,
                                                  const trUserContext rcUsrCntxt)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   //m_bAudioPlayFlag = bPlayFlag;
   //m_u8AudioContext = u8AudioCntxt;

   if(NULL != m_poRsrcMngrRespIntf)
   {
      m_poRsrcMngrRespIntf->vPostDeviceAudioContext(m_u32SelectedDevice, bPlayFlag, u8AudioCntxt, enPermMainAudio, rcUsrCntxt);
   } //if(NULL != m_poRsrcMngrRespIntf)
}


/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vCbDeviceAppState()
***************************************************************************/
t_Void spi_tclResourceMngr::vCbDeviceAppState(tenSpeechAppState enSpeechAppState,
                                              tenPhoneAppState enPhoneAppState,
                                              tenNavAppState enNavAppState,
                                              const trUserContext rUserContext)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));

   //Change the app states only for valid states and not for UNKNOWN states.
   /*m_rCurrAppState.enSpeechAppState =
            ((e8SPI_SPEECH_UNKNOWN != enSpeechAppState)? enSpeechAppState:m_rCurrAppState.enSpeechAppState);
   m_rCurrAppState.enPhoneAppState  =
            ((e8SPI_PHONE_UNKNOWN != enPhoneAppState)?enPhoneAppState:m_rCurrAppState.enPhoneAppState);
   m_rCurrAppState.enNavAppState    =
            ((e8SPI_NAV_UNKNOWN != enNavAppState)?enNavAppState:m_rCurrAppState.enNavAppState);*/

   if(NULL != m_poRsrcMngrRespIntf)
   {
      m_poRsrcMngrRespIntf->vPostDeviceAppState(enSpeechAppState, enPhoneAppState, enNavAppState, rUserContext);
   }//if(NULL != m_poRsrcMngrRespIntf)

   spi_tclMediator *poMediator = spi_tclMediator::getInstance();
   if(NULL != poMediator)
   {
      poMediator->vPostDevAppState(enSpeechAppState, enPhoneAppState,enNavAppState);
   }//if(NULL != poMediator)

}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vCbDeviceSpmSubState()
***************************************************************************/
t_Void spi_tclResourceMngr::vCbDeviceSpmSubState(tenSpmSubState enSpmSubState, t_Bool m_bSubStateData)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   if(NULL != m_poRsrcMngrRespIntf)
   {
      m_poRsrcMngrRespIntf->vPostDeviceSpmSubState(enSpmSubState, m_bSubStateData);
   }//if(NULL != m_poRsrcMngrRespIntf)
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vCbSetDeviceAppState()
***************************************************************************/
t_Void spi_tclResourceMngr::vCbSetDeviceAppState(tenSpeechAppState enSpeechAppState,
                                              tenPhoneAppState enPhoneAppState,
                                              tenNavAppState enNavAppState)
{
   ETG_TRACE_USR1((" Resource Manager Set Device App states - Speech: %d, Phone: %d, Nav: %d , Dev Category: %d\n",
            ETG_ENUM(SPEECH_APP_STATE,enSpeechAppState), ETG_ENUM(PHONE_APP_STATE,enPhoneAppState) ,
            ETG_ENUM(NAV_APP_STATE, enNavAppState), ETG_ENUM(DEVICE_CATEGORY, m_enDevCat)));

   t_U8 u8Index = static_cast<t_U8>(m_enDevCat);

   if ((true == bValidateClient(u8Index)) && (NULL != m_poRsrcMngrBase[u8Index]))
   {
      m_poRsrcMngrBase[u8Index]->vAcquireDevAppStateLock();

      //Change the app states only for valid states and not for UNKNOWN states.
      m_rCurrAppState.enSpeechAppState =
            ((e8SPI_SPEECH_UNKNOWN != enSpeechAppState)? enSpeechAppState:m_rCurrAppState.enSpeechAppState);
      m_rCurrAppState.enPhoneAppState  =
            ((e8SPI_PHONE_UNKNOWN != enPhoneAppState)? enPhoneAppState:m_rCurrAppState.enPhoneAppState);
      m_rCurrAppState.enNavAppState    =
            ((e8SPI_NAV_UNKNOWN != enNavAppState)? enNavAppState:m_rCurrAppState.enNavAppState);

      m_poRsrcMngrBase[u8Index]->vReleaseDevAppStateLock();
   }
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vCbDeviceAppState()
***************************************************************************/
t_Void spi_tclResourceMngr::vCbUpdateSessionStatus(t_U32 u32DeviceHandle,
                              tenDeviceCategory enDevCat,
                              tenSessionStatus enSessionStatus)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   if(NULL != m_poRsrcMngrRespIntf)
   {
      m_poRsrcMngrRespIntf->vUpdateSessionStatusInfo(u32DeviceHandle, enDevCat, enSessionStatus);
   }//if(NULL != m_poRsrcMngrRespIntf)
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vCbIncomingCall()
***************************************************************************/
t_Void spi_tclResourceMngr::vCbInitiateCall(t_U32 u32DeviceHandle,
                                            t_String szTelePhoneNumber,
                                            t_String szDisplayText)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));

   if(NULL != m_poRsrcMngrRespIntf)
   {
      m_poRsrcMngrRespIntf->vPostInitiateCallInfo(u32DeviceHandle,
                                              szTelePhoneNumber,szDisplayText);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vCbVoiceSessionReq()
***************************************************************************/
t_Void spi_tclResourceMngr::vCbVoiceSessionReq(t_U32 u32DeviceHandle,
                                             t_U8 u8MySPINVoiceSessionReq)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   if(NULL != m_poRsrcMngrRespIntf)
   {
      m_poRsrcMngrRespIntf->vPostMySPINVoiceSessionReq(u32DeviceHandle,
                                                       u8MySPINVoiceSessionReq);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclResourceMngr::vGetAppStateInfo()
***************************************************************************/
t_Void spi_tclResourceMngr::vGetAppStateInfo(tenSpeechAppState &enSpeechAppState, 
                                             tenPhoneAppState &enPhoneAppState, 
                                             tenNavAppState &enNavAppState)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));

   t_U8 u8Index = static_cast<t_U8>(m_enDevCat);

   if ((true == bValidateClient(u8Index)) && (NULL != m_poRsrcMngrBase[u8Index]))
   {
      m_poRsrcMngrBase[u8Index]->vAcquireDevAppStateLock();

      enSpeechAppState =  m_rCurrAppState.enSpeechAppState;
      enPhoneAppState = m_rCurrAppState.enPhoneAppState ;
      enNavAppState = m_rCurrAppState.enNavAppState;

      m_poRsrcMngrBase[u8Index]->vReleaseDevAppStateLock();
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclResourceMngr::vSetVehicleBTAddress()
***************************************************************************/
t_Void spi_tclResourceMngr::vSetVehicleBTAddress(t_String szBtAddress)
{
	//@Note : This function
	//! Needs to invoke irrespective of the session status.
	if(NULL != m_poRsrcMngrBase[e8DEV_TYPE_DIPO])
	{
	    m_poRsrcMngrBase[e8DEV_TYPE_DIPO]->vSetVehicleBTAddress(szBtAddress);
	}//if(NULL != m_poRsrcMngrBase[e8DEV_TYPE_DIPO])
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vSetFeatureRestrictions(...)
 ***************************************************************************/
t_Void spi_tclResourceMngr::vSetFeatureRestrictions(tenDeviceCategory enDevCategory,
      const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
{
   //! This info is required by CarPlay resource manager 
   if ((e8DEV_TYPE_DIPO == enDevCategory) && (NULL != m_poRsrcMngrBase[e8DEV_TYPE_DIPO]))
   {
      m_poRsrcMngrBase[e8DEV_TYPE_DIPO]->vSetFeatureRestrictions(cou8ParkModeRestrictionInfo,
            cou8DriveModeRestrictionInfo);
   }//if ((e8DEV_TYPE_DIPO == enDevCategory) && (NULL != m_poRsrcMngrBase[e8DEV_TYPE_DIPO]))
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclResourceMngr::vRequestProjUI()
***************************************************************************/
t_Void spi_tclResourceMngr::vRequestProjUI(t_U32 u32DevId,
                                       t_U32 u32AppId,
                                       tenDeviceCategory enDevCat)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);
   if(true == bValidateClient(u8Index))
   {
      m_poRsrcMngrBase[u8Index]->vRequestProjUI(u32DevId,u32AppId);
   }//if(true == bValidateClient(u8Index))

}

/***************************************************************************
** FUNCTION: t_Void spi_tclResourceMngr::vDevAuthAndAccessInfoCb()
***************************************************************************/
t_Void spi_tclResourceMngr::vDevAuthAndAccessInfoCb(const t_U32 cou32DevId,
              const tenHandsetInteractionStatus coenHandsetInteractionReqd)
{
   ETG_TRACE_USR1(("spi_tclResourceMngr::vDevAuthAndAccessInfoCb:Device ID-0x%x Handset Interaction Required-%d",
      cou32DevId,ETG_ENUM(HANDSET_INTERACTION_STATUS,coenHandsetInteractionReqd)));

   if(NULL != m_poRsrcMngrRespIntf)
   {
      m_poRsrcMngrRespIntf->vUpdateDevAuthAndAccessInfo(cou32DevId, coenHandsetInteractionReqd);
   }//if(NULL != m_poVideoRespIntf)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclResourceMngr::vSelectDevice()
***************************************************************************/
t_Void spi_tclResourceMngr::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
	
   m_enDevCat = corfrSelectReq.m_enDevCategory;
   t_U8 u8Index = static_cast<t_U8>(corfrSelectReq.m_enDevCategory);
   if(true == bValidateClient(u8Index))
   {
      m_poRsrcMngrBase[u8Index]->vSelectDevice(corfrSelectReq.m_u32DeviceHandle,
               corfrSelectReq.m_enDevConnReq, corfrSelectReq.m_enDevCategory);
   }//if(true == bValidateClient(u8Index))
}

/***************************************************************************
** FUNCTION: t_Void spi_tclResourceMngr::vSelectDevice()
***************************************************************************/
t_Void spi_tclResourceMngr::vCbSelectDeviceResult(t_Bool bSelctDeviceResult)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   tenErrorCode enErrorCode = (true==bSelctDeviceResult)? e8NO_ERRORS : e8SELECTION_FAILED;
   spi_tclMediator *poMediator = spi_tclMediator::getInstance();
   if(NULL != poMediator)
   {
      poMediator->vPostSelectDeviceRes(e32COMPID_RESOURCEMNGR, enErrorCode);
   }//if(NULL != poMediator)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclResourceMngr::vSetVehicleBTAddress()
***************************************************************************/
t_Void spi_tclResourceMngr::vUpdateVoiceSessionStatus(tenBTVoiceSessionStatus enBTVoiceSessionStatus)
{
   if(NULL != m_poRsrcMngrBase[e8DEV_TYPE_MYSPIN])
   {
       m_poRsrcMngrBase[e8DEV_TYPE_MYSPIN]->vUpdateVoiceSessionStatus(enBTVoiceSessionStatus);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclResourceMngr::vOnSetNativeVideoRenderingStatus()
***************************************************************************/
t_Void spi_tclResourceMngr::vOnSetNativeVideoRenderingStatus(tenNativeVideoRenderingStatus enNativeVideoRenderingStatus)
{
   ETG_TRACE_USR1(("spi_tclResourceMngr::vOnSetNativeVideoRenderingStatus entered"));
  //posting of the message to the AAP resource manager, Dipo and MIrrorlink

   for(t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
   {
      if(NULL != m_poRsrcMngrBase[u8Index])
      {
         m_poRsrcMngrBase[u8Index]->vOnSetNativeVideoRenderingStatus(enNativeVideoRenderingStatus);
      }
   }
   ETG_TRACE_USR1(("spi_tclResourceMngr::vOnSetNativeVideoRenderingStatus left"));
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclResourceMngr::vCbProjectionVideoRenderingStatus
 ***************************************************************************/
t_Void spi_tclResourceMngr::vCbProjectionVideoRenderingStatus(tenProjVideoRenderingStatus enProjVideoRenderingStatus)
{
   ETG_TRACE_USR1(("spi_tclResourceMngr::vCbProjectionVideoRenderingStatus display status - %d"
            ,ETG_ENUM(PROJECTION_VIDEO_RENDERING_STATUS,enProjVideoRenderingStatus)));
   if(NULL != m_poRsrcMngrRespIntf)
   {
      m_poRsrcMngrRespIntf->vUpdateProjectionVideoRenderingStatus(enProjVideoRenderingStatus);
   }//if(NULL != m_poVideoRespIntf)
   ETG_TRACE_USR1(("spi_tclResourceMngr::vCbProjectionVideoRenderingStatus left"));
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vSetGeneralRestrictions(...)
 ***************************************************************************/
t_Void spi_tclResourceMngr::vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory,t_U16 u16GeneralRestrInfo)
{
	//! This info is required by CarPlay resource manager
	if ((e8DEV_TYPE_DIPO == enDeviceCategory) && (NULL != m_poRsrcMngrBase[e8DEV_TYPE_DIPO]))
	{
		m_poRsrcMngrBase[e8DEV_TYPE_DIPO]->vSetGeneralRestrictions(u16GeneralRestrInfo);
	}//if ((e8DEV_TYPE_DIPO == enDevCategory) && (NULL != m_poRsrcMngrBase[e8DEV_TYPE_DIPO]))
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vSetETCStatus(t_Bool bETCStatus)
 ***************************************************************************/
t_Void spi_tclResourceMngr::vSetETCStatus(t_Bool bETCStatus)
{
	 ETG_TRACE_USR1(("spi_tclResourceMngr::vSetETCStatus entered"));
	//Dispatching to all the technologies - As of now only DIPO has the implementation
	for(t_U8 u8Index = 0; u8Index < NUM_RSRCMNGR_CLIENTS; ++u8Index)
	{
		if(NULL != m_poRsrcMngrBase[u8Index])
		{
			m_poRsrcMngrBase[u8Index]->vSetETCStatus(bETCStatus);
		}
	}
	ETG_TRACE_USR1(("spi_tclResourceMngr::vSetETCStatus left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vStoreDisplayLastMode()
 ***************************************************************************/
t_Void spi_tclResourceMngr::vStoreDisplayLastMode(tenDisplayInfo enDisplayInfo,
         const trDisplayContext& rfcorDisplayContext,
         const trDisplayConstraint& rfcorDisplayConstraint)
{
   tenLastModeSupport enLastModeSupport = (NULL != m_poResourceMngrSettings)?
           (m_poResourceMngrSettings->enGetLastModeSupport()) : (e8LASTMODE_DISABLED);

   if (e8LASTMODE_DISABLED != enLastModeSupport)
   {
      tenLastMode enDisplayLastMode = e8NOT_APPLICABLE;

      if (e8_DISPLAY_CONTEXT == enDisplayInfo)
      {
         switch (rfcorDisplayContext.enDisplayContext)
         {
            case e8DISPLAY_CONTEXT_UNKNOWN:
            case e8DISPLAY_CONTEXT_NAVIGATION:
            case e8DISPLAY_CONTEXT_MEDIA:
            case e8DISPLAY_CONTEXT_NATIVE:
               enDisplayLastMode = (rfcorDisplayContext.bDisplayFlag)? (e8NATIVE) : (enDisplayLastMode);
               break;
            case e8DISPLAY_CONTEXT_PROJECTION:
               enDisplayLastMode = (rfcorDisplayContext.bDisplayFlag)? (e8PROJECTION) : (enDisplayLastMode);;
               break;
            default:
               //Nothing to do
               break;
         }
      }
      else if (e8_DISPLAY_CONSTRAINT == enDisplayInfo)
      {
         if (e8DIPO_TRANSFERTYPE_TAKE == rfcorDisplayConstraint.enTransferType)
         {
            enDisplayLastMode = (e8DIPO_TRANSFERPRIO_NICETOHAVE == rfcorDisplayConstraint.enTransferPriority)?
                                     (e8PROJECTION): (e8NATIVE);
         }
      }

      if ((NULL != m_poResourceMngrSettings) && (e8NOT_APPLICABLE != enDisplayLastMode))
      {
         ETG_TRACE_USR2(("[PARAM]::vStoreDisplayLastMode: Evaluated last display mode = %d", ETG_ENUM(LASTMODE, enDisplayLastMode)));
         t_Bool bIsDisplayLastModeProjection = (e8PROJECTION == enDisplayLastMode);
         m_poResourceMngrSettings->bWriteDisplayLastMode(bIsDisplayLastModeProjection);
      }
      else if (e8NOT_APPLICABLE == enDisplayLastMode)
      {
         ETG_TRACE_USR2(("[PARAM]::vStoreDisplayLastMode: No change in last display mode "));
      }
   }//if (e8LASTMODE_DISABLED != enLastModeSupport)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vStoreAudioLastMode()
 ***************************************************************************/
t_Void spi_tclResourceMngr::vStoreAudioLastMode(t_Bool bAudioFlag,
         tenAudioContext enAudioCntxt)
{
   tenLastModeSupport enLastModeSupport = (NULL != m_poResourceMngrSettings)?
             (m_poResourceMngrSettings->enGetLastModeSupport()) : (e8LASTMODE_DISABLED);

   if (e8LASTMODE_DISABLED != enLastModeSupport)
   {
      tenLastMode enAudioLastMode = e8NOT_APPLICABLE;

      switch (enAudioCntxt)
      {
         case e8SPI_AUDIO_MAIN:
         case e8SPI_AUDIO_INTERNET_APP:
            enAudioLastMode = (bAudioFlag)? (e8NATIVE) : (enAudioLastMode);
            break;
         case e8SPI_AUDIO_MAIN_DEFAULT:
         case e8SPI_AUDIO_PROJECTION:
            enAudioLastMode = (bAudioFlag)? (e8PROJECTION) : (enAudioLastMode);
            break;
         default:
            //Nothing to do
            break;
      }

      if ((NULL != m_poResourceMngrSettings) && (e8NOT_APPLICABLE != enAudioLastMode))
      {
         ETG_TRACE_USR2(("[PARAM]::vStoreAudioLastMode: Evaluated last audio mode = %d", ETG_ENUM(LASTMODE, enAudioLastMode)));
         t_Bool bIsAudioLastModeProjection = (e8PROJECTION == enAudioLastMode);
         m_poResourceMngrSettings->bWriteAudioLastMode(bIsAudioLastModeProjection);
      }
      else if (e8NOT_APPLICABLE == enAudioLastMode)
      {
         ETG_TRACE_USR2(("[PARAM]::vStoreAudioLastMode: No change in last audio mode "));
      }
   }//if (e8LASTMODE_DISABLED != enLastModeSupport)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vPostTelephoneCallStatusMsg()
 ***************************************************************************/
t_Void spi_tclResourceMngr::vPostTelephoneCallStatusMsg(const std::vector<
		trTelCallStatusInfo> &corfTelCallStatusInfoList)
{
	ETG_TRACE_USR4(("spi_tclResourceMngr::vPostTelephoneCallStatusMsg():List size:%d",
			corfTelCallStatusInfoList.size()));

	if (NULL != m_poRsrcMngrBase[e8DEV_TYPE_DIPO])
	{
		m_poRsrcMngrBase[e8DEV_TYPE_DIPO]->vOnTelephoneCallStatus(corfTelCallStatusInfoList);
	}

}
/***************************************************************************
** FUNCTION:  t_Void spi_tclResourceMngr::vOnDisableBluetoothTrigger(
***************************************************************************/
t_Void spi_tclResourceMngr::vOnDisableBluetoothTrigger(t_String szBTAddress)
{
	ETG_TRACE_USR4(("spi_tclResourceMngr::vOnDisableBluetoothTrigger():%s",
			szBTAddress.c_str()));

	if (NULL != m_poRsrcMngrBase[e8DEV_TYPE_DIPO])
	{
		m_poRsrcMngrBase[e8DEV_TYPE_DIPO]->vOnDisableBluetoothTrigger(szBTAddress);
	}
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vSetResourceMngrSettingsInstance()
***************************************************************************/
t_Void spi_tclResourceMngr::vSetResourceMngrSettingsInstance(spi_tclResourceMngrSettingsIntf* poResourceMngrSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclResourceMngr::vSetResourceMngrSettingsInstance entered"));
   if( NULL != poResourceMngrSettingsIntf)
   {
	   ETG_TRACE_USR1(("spi_tclResourceMngr::vSetResourceMngrSettingsInstance Setting m_poResourceMngrSettings val"));
       m_poResourceMngrSettings = poResourceMngrSettingsIntf;
   }

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclResourceMngr::vOnEvaluateInitalModes()
***************************************************************************/
t_Void spi_tclResourceMngr::vOnEvaluateInitalModes(const t_String& corfszBTAddress)
{
   ETG_TRACE_USR4(("spi_tclResourceMngr::vOnEvaluateInitalModes():%s", corfszBTAddress.c_str()));

   if (NULL != m_poRsrcMngrBase[e8DEV_TYPE_DIPO])
   {
      m_poRsrcMngrBase[e8DEV_TYPE_DIPO]->vOnEvaluateInitalModes(corfszBTAddress);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vPostExtVRStateUpdateMsg()
 ***************************************************************************/
t_Void spi_tclResourceMngr::vPostExtVRStateUpdateMsg(t_String szBTAddress,
                                                     tenExtVRState enExtVRState,
                                                     tenVRErrorCode enVRErrorCode)
{
    ETG_TRACE_USR4(("spi_tclResourceMngr::vPostExtVRStateUpdateMsg(): VR State :%d",
        enExtVRState));

    if (NULL != m_poRsrcMngrBase[e8DEV_TYPE_MYSPIN])
    {
        m_poRsrcMngrBase[e8DEV_TYPE_MYSPIN]->vOnExtVRStateUpdate(szBTAddress,enExtVRState,enVRErrorCode);
    }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vGetResourceManagerConfig()
***************************************************************************/

t_Void spi_tclResourceMngr::vGetResourceManagerConfig()
{
   ETG_TRACE_USR1(("spi_tclResourceMngr::vGetResourceManagerConfig entered"));
   if( NULL != m_poResourceMngrSettings)
   {
	   trVehicleBrandInfo rfrVehicleBrandInfo;
	   m_poResourceMngrSettings->szGetHardwareVersion();
	   m_poResourceMngrSettings->vGetOemIconData(rfrVehicleBrandInfo);
	   m_poResourceMngrSettings->bGetNativeNavigationEnabled();
	   m_poResourceMngrSettings->enGetDriveSideInfo();
   }

}

//lint –restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
