/***********************************************************************/
/*!
* \file  spi_tclResourceMngr.h
* \brief Main Resource Manager class that provides interface to delegate 
*        the Resource management mainly of the screen and Audio
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         Priju K Padiyath
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
05.04.2014  | Priju K Padiyath      | Initial Version
17.04.2014  | Shihabudheen P M      | Modofied for display context response
19.06.2014  | Shihabudheen P M      | Updated for App state resource management
25.06.2014  |  Shihabudheen P M     | Adapted to the CarPlay design changes
31.07.2014  |  Ramya Murthy         | SPI feature configuration via LoadSettings()
15.06.2015  | Shihabudheen P M      | added vSetVehicleBTAddress()
29.11.2016  |  Noopur R K           | HMI Synchronization

\endverbatim
*************************************************************************/

#ifndef _SPI_TCLRESOURCEMNGR_
#define _SPI_TCLRESOURCEMNGR_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "DiPOTypes.h"
#include "AAPTypes.h"
#include "spi_tclLifeCycleIntf.h"
#include "spi_tclSelectionIntf.h"
#include "spi_tclExtRespTelephone.h"
#include "spi_tclResourceMngrSettingsIntf.h"
/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
class spi_tclResourceMngrBase;
class spi_tclResourceMngrResp;

/****************************************************************************/
/*!
* \class spi_tclResourceMngr
* \brief Main Resource Manager class that provides interface to delegate 
*        the handling of resources like the screen and the audio
****************************************************************************/
class spi_tclResourceMngr : public spi_tclLifeCycleIntf, public spi_tclSelectionIntf,public spi_tclExtRespTelephone
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclResourceMngr::spi_tclResourceMngr()
   ***************************************************************************/
   /*!
   * \fn      spi_tclResourceMngr(spi_tclRsrcMngrRespIntf* poRespIntf)
   * \brief   Parameterized Constructor
   * \sa      ~spi_tclResourceMngr()
   **************************************************************************/
   spi_tclResourceMngr(spi_tclResourceMngrResp* poRespIntf);

   /***************************************************************************
   ** FUNCTION:  spi_tclResourceMngr::~spi_tclResourceMngr()
   ***************************************************************************/
   /*!
   * \fn      virtual ~spi_tclResourceMngr()
   * \brief   Destructor
   * \sa      spi_tclResourceMngr(spi_tclRsrcMngrRespIntf* poRespIntf)
   **************************************************************************/
   ~spi_tclResourceMngr();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclResourceMngr::bInitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bInitialize()
   * \brief   To Initialize all the resource Manager related classes
   * \retval  t_Bool
   * \sa      vUninitialize()
   **************************************************************************/
   t_Bool bInitialize();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclResourceMngr::vUninitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Void vUninitialize()
   * \brief   To Ininitialize all the resource Manager related classes
   * \retval  t_Void
   * \sa      bInitialize()
   **************************************************************************/
   t_Bool bUnInitialize();

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclResourceMngr::vLoadSettings()
    ***************************************************************************/
   /*!
    * \fn      vLoadSettings()
    * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
    * \sa      vSaveSettings()
    **************************************************************************/
   t_Void vLoadSettings();

   /***************************************************************************
   ** FUNCTION:  spi_tclResourceMngr::vSaveSettings()
   ***************************************************************************/
   /*!
   * \fn      vSaveSettings()
   * \brief   vSaveSettings Method. Invoked during  NORMAL->OFF state transition.
   * \sa      vLoadSettings()
   **************************************************************************/
   t_Void vSaveSettings()
   {
   }

   /***************************************************************************
    ** FUNCTION:  spi_tclResourceMngr::vSelectDeviceResult
    ***************************************************************************/
   /*!
    * \fn     vSelectDeviceResult
    * \brief  device selection result
    * \param  corfrSelectReq : Device selection result details
    * \param  enErrorCode : Error code
    **************************************************************************/
   t_Void vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode);


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclResourceMngr::vSetAccessoryDisplayContext()
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetAccessoryDisplayContext(const t_U32 cou32DevId,
   *       t_Bool bDisplayFlag, tenDisplayContext enDisplayContext,
   *       tenDeviceCategory enDevCat, const trUserContext& rfrcUsrCntxt)
   *
   * \brief   To send accessory display context related info .
   * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
   * \param   bDisplayFlag : [IN] Request flag
   * \pram    enDisplayContext : [IN] Display context
   * \pram    enDevCat    : [IN] Identifies the Connection Request.
   * \pram    rfrcUsrCntxt: [IN] User Context Details.
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetAccessoryDisplayContext(const t_U32 cou32DevId,
      t_Bool bDisplayFlag, 
      tenDisplayContext enDisplayContext,
      const trUserContext& rfrcUsrCntxt);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vSetAccessoryDisplayMode(t_U32...
   ***************************************************************************/
   /*!
   * \fn     vSetAccessoryDisplayMode()
   * \brief  Accessory display mode update request.
   * \param  [IN] cu32DeviceHandle      : Uniquely identifies the target Device.
   * \pram    enDevCat    : [IN] Identifies the Connection Request.
   * \param  [IN] corDisplayContext : Display context info
   * \param  [IN] corDisplayConstraint : DiDisplay constraint info
   * \param  [IN] coenDisplayInfo       : Display info flag
   * \sa
   **************************************************************************/
   t_Void vSetAccessoryDisplayMode(const t_U32 cu32DeviceHandle,
      const tenDeviceCategory enDevCat,
      const trDisplayContext corDisplayContext,
      const trDisplayConstraint corDisplayConstraint,
      const tenDisplayInfo coenDisplayInfo);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclResourceMngr::vSetAccessoryAudioContext()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetAccessoryAudioContext(const t_U32 cou32DevId, const t_U8 cu8AudioCntxt,
   *            t_Bool bReqFlag, tenDeviceCategory enDevCat, const trUserContext& rfrcUsrCntxt
   *            tenDeviceCategory enRequestedDevCat)
   *
   * \brief   To send accessory display context related info .
   * \param   cou32DevId       : [IN] Uniquely identifies the target Device.
   * \param   cu8AudioCntxt    : [IN] Audio Source number/base channel number
   * \param   bReqFlag         : [IN] Request flag, true for request and false for release
   * \param   enDevCat         : [IN] Device category of the current selected device.
   * \param   rfrcUsrCntxt     : [IN] User Context Details.
   * \param   enRequestedDevCat: [IN] Device category for which audio context update is requested.
   * \param   bIsHMITrigger    : [IN] True if HMI sends request and false if SPI internally mocks the request.
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetAccessoryAudioContext(const t_U32 cou32DevId, 
      const tenAudioContext coenAudioCntxt,
      t_Bool bReqFlag, tenDeviceCategory enDevCat, 
      const trUserContext& rfrcUsrCntxt, 
      tenDeviceCategory enRequestedDevCat,
      t_Bool bIsHMITrigger = true);

  /***************************************************************************
   ** FUNCTION:  t_Void spi_tclResourceMngr::vSetAccessoryAppState()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetAccessoryAppState(tenSpeechAppState enSpeechAppState, tenPhoneAppState enPhoneAppState,
   *         tenNavAppState enNavAppState
   * \brief   To set accessory app state realated info.
   * \pram    enSpeechAppState: [IN] Accessory speech state.
   * \param   enPhoneAppState : [IN] Accessory phone state
   * \pram    enNavAppState   : [IN] Accessory navigation app state
   * \pram    rfrcUsrCntxt    : [IN] User Context Details.
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetAccessoryAppState(tenDeviceCategory enDevCat, const tenSpeechAppState enSpeechAppState, 
      const tenPhoneAppState enPhoneAppState, const tenNavAppState enNavAppState, const trUserContext& rfrcUsrCntxt);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbPostDeviceDisplayContext
   **                   (t_U32 u32DeviceHandle, tenDisplayContext enDisplayContext,..)
   ***************************************************************************/
   /*!
   * \fn     vCbPostDeviceDisplayContext(t_U32 u32DeviceHandle, t_Bool bDisplayFlag,
   *         tenDisplayContextReason enDisplayContextReason, const trUserContext rcUsrCntxt);
   * \brief  This function get called when there is a display context update
   * \param  [IN] u32DeviceHandle  : Uniquely identifies the target Device.
   * \param  [IN] bDisplayFlag     : TRUE -Start Display Projection, FALSE - Stop Display Projection.
   * \param  [IN] enDisplayContextReason :  Reason for device display context update.
   * \param  [IN] enPermScreen     : Permanent owner of display resource
   * \param  [IN] rcUsrCntxt       : User Context Details.
   * \sa
   **************************************************************************/
   t_Void vCbPostDeviceDisplayContext(t_Bool bDisplayFlag,
      tenDisplayContextReason enDisplayContextReason,
      tenDiPOEntity enPermScreen,
      const trUserContext rcUsrCntxt);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbPostDeviceDisplayContext
   **                   (t_U32 u32DeviceHandle, tenDisplayContext enDisplayContext,..)
   ***************************************************************************/
   /*!
   * \fn     vCbPostDeviceDisplayContext(t_U32 u32DeviceHandle, t_Bool bDisplayFlag,
   *         tenDisplayContext enDisplayContext, const trUserContext rcUsrCntxt);
   * \brief  This function get called when there is a display context update
   * \param  [IN] u32DeviceHandle  : Uniquely identifies the target Device.
   * \param  [IN] bPlayFlag        : TRUE - Start CarPlay playback, FALSE � native playback
   * \param  [IN] u8AudioCntxt     : Audio vontext
   * \param  [IN] enPermMainAudio  : Permanent owner of main audio resource
   * \param  [IN] rcUsrCntxt       : User Context Details.
   * \sa
   **************************************************************************/
   t_Void vCbDeviceAudioContext(t_Bool bPlayFlag, t_U8 u8AudioCntxt,
            tenDiPOEntity enPermMainAudio, const trUserContext rcUsrCntxt);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbDeviceAppState()
   ***************************************************************************/
   /*!
   * \fn     vCbDeviceAppState(tenSpeechAppState enSpeechAppState,...)
   * \brief  To post the device app states.
   * \param  [IN] enSpeechAppState: Speech state
   * \param  [IN] enPhoneAppState : Phone app state
   * \param  [IN] enNavAppState   : Nav App state
   * \param  [IN] rcUsrCntxt      : User Context Details.
   * \sa
   **************************************************************************/
   t_Void vCbDeviceAppState( tenSpeechAppState enSpeechAppState,
      tenPhoneAppState enPhoneAppState, tenNavAppState enNavAppState,
      const trUserContext rUserContext);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbDeviceSpmSubState()
   ***************************************************************************/
   /*!
   * \fn     vCbDeviceSpmSubState()
   * \brief  This function used to update LCM about the SPM substate changes
   * \param  enSpmSubState  : [IN] Spm SubState type
   * \param  m_bSubStateData  : [IN] Spm SubState data
   **************************************************************************/
   t_Void vCbDeviceSpmSubState(tenSpmSubState enSpmSubState, t_Bool m_bSubStateData);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbUpdateSessionStatus()
   ***************************************************************************/
   /*!
   * \fn     vCbUpdateSessionStatus(t_U32 u32DeviceHandle, tenDeviceCategory enDevCat,...)
   * \brief  To update session info
   * \param  [IN] u32DeviceHandle  : Device handle.
   * \param  [IN] enDevCat         : Device category.
   * \param  [IN] enSessionStatus  : Session status
   * \sa
   **************************************************************************/
   t_Void vCbUpdateSessionStatus(t_U32 u32DeviceHandle,
      tenDeviceCategory enDevCat, tenSessionStatus enSessionStatus);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbSetDeviceAppState()
   ***************************************************************************/
   /*!
   * \fn     vCbSetDeviceAppState(tenSpeechAppState enSpeechAppState,
   *           tenPhoneAppState enPhoneAppState, tenNavAppState enNavAppState)
   * \brief  To set the device app states.
   * \param  [IN] enSpeechAppState: Speech state
   * \param  [IN] enPhoneAppState : Phone app state
   * \param  [IN] enNavAppState   : Nav App state
   * \param  [IN] rcUsrCntxt      : User Context Details.
   * \sa
   **************************************************************************/
   t_Void vCbSetDeviceAppState( tenSpeechAppState enSpeechAppState,
      tenPhoneAppState enPhoneAppState, tenNavAppState enNavAppState);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbInitiateCall()
   ***************************************************************************/
   /*!
   * \fn     vCbInitiateCall(t_U32 u32DeviceHandle, ..)
   * \brief  To get the dipslay context info
   * \param  [IN] u32DeviceHandle  : Device handle.
   * \param  [IN] enDevCat         : Device categiry.
   * \param  [IN] enSessionStatus  : Session status.
   * \param  [IN] szTelePhoneNumber  : Telephone
   * \param  [IN] szDisplayText  : Display text
   * \sa
   **************************************************************************/
   t_Void vCbInitiateCall(t_U32 u32DeviceHandle,
                                 t_String szTelePhoneNumber,
                                 t_String szDisplayText);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbVoiceSessionReq()
   ***************************************************************************/
   /*!
   * \fn     vCbVoiceSessionReq(t_U32 u32DeviceHandle, ..)
   * \brief  To get the dipslay context info
   * \param  [IN] u32DeviceHandle  : Device handle.
   * \param  [IN] enMySPINVoiceSessionReq  : Voice session request.
   * \sa
    **************************************************************************/
   t_Void vCbVoiceSessionReq(t_U32 u32DeviceHandle,
                                   t_U8 u8MySPINVoiceSessionReq);
   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vGetAppStateInfo()
   ***************************************************************************/
   /*!
   * \fn     vGetAppStateInfo()
   * \brief  To get the device app state info.
   * \param  [OUT] enSpeechAppState  : Speech App state
   * \param  [OUT] enPhoneAppState   : Phone App state
   * \param  [OUT] enNavAppState     : Navigation App state
   * \sa
   **************************************************************************/
   t_Void vGetAppStateInfo(tenSpeechAppState &enSpeechAppState, 
      tenPhoneAppState &enPhoneAppState, 
      tenNavAppState &enNavAppState);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclResourceMngr::vSetVehicleBTAddress()
   ***************************************************************************/
   /*!
   * \fn      vSetVehicleBTAddress(t_Bool bLocDataAvailable)
   * \brief   Interface to update the vehicle BT address info update.
   * \param   szBtAddress: [IN] BT address.
   * \retval  None
   **************************************************************************/
   t_Void vSetVehicleBTAddress(t_String szBtAddress);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclResourceMngr::vSetFeatureRestrictions()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetFeatureRestrictions(tenDeviceCategory enDevCategory,
    *          const t_U8 cou8ParkModeRestrictionInfo,
    *          const t_U8 cou8DriveModeRestrictionInfo)
    * \brief   Method to set Vehicle Park/Drive Mode Restriction
    * \param   enDevCategory : Device Category
    *          cou8ParkModeRestrictionInfo : Park mode restriction
    *          cou8DriveModeRestrictionInfo : Drive mode restriction
    * \retval  t_Void
    **************************************************************************/
   t_Void vSetFeatureRestrictions(tenDeviceCategory enDevCategory,
         const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclResourceMngr::vRequestProjUI()
   ***************************************************************************/
   /*!
   * \fn      t_Void vRequestProjUI(t_U32 u32DevId,
   *                                t_U32 u32AppId,
   *                                tenDeviceCategory enDevCat)
   * \brief   To Launch the Video for the requested app 
   * \pram    u32DevId  : [IN] Uniquely identifies the target Device.
   * \pram    u32AppId  : [IN] Application Id
   * \pram    enDevCat  : [IN] Identifies the Connection Request.
   * \retval  t_Void
   **************************************************************************/
   t_Void vRequestProjUI(t_U32 u32DevId,
      t_U32 u32AppId,
      tenDeviceCategory enDevCat);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vDevAuthAndAccessInfoCb()
   ***************************************************************************/
   /*!
   * \fn     t_Void vDevAuthAndAccessInfoCb(const t_U32 cou32DevId,
   *            const tenHandsetInteractionStatus coenHandsetInteractionReqd)
   * \brief  method to update the authorization and access to AAP projection device
   * \param  cou32DevId                 : [IN] Uniquely identifies the target Device.
   * \param  coenHandsetInteractionReqd : [IN] Handset/user interaction requirement status
   * \retval t_Void
   **************************************************************************/
   t_Void vDevAuthAndAccessInfoCb(const t_U32 cou32DevId,
            const tenHandsetInteractionStatus coenHandsetInteractionReqd);

   /***************************************************************************
    ** FUNCTION:  spi_tclResourceMngr::vSelectDevice
    ***************************************************************************/
   /*!
    * \fn     vSelectDevice
    * \brief  Device selection request
    * \param  corfrSelectReq : Device selection request details
    **************************************************************************/
   t_Void vSelectDevice(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbSelectDeviceResult()
   ***************************************************************************/
   /*!
   * \fn     t_Void vCbSelectDeviceResult(t_Bool bSelctDeviceResult)
   * \brief  method to get the select device result from sub components.
   * \param  bSelctDeviceResult : [IN] Select device result.
   * \retval t_Void
   **************************************************************************/
   t_Void vCbSelectDeviceResult(t_Bool bSelctDeviceResult);

   /***************************************************************************/
   /*!
   * \fn     vUpdateVoiceSessionStatus()
   * \brief  Update the current voice session status
   * \param  szTelephoneNumber
   * \param  szDisplayText
   * \sa
   **************************************************************************/
   t_Void vUpdateVoiceSessionStatus(tenBTVoiceSessionStatus enBTVoiceSessionStatus);
   
   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vCbProjectionVideoRenderingStatus()
   ***************************************************************************/
   /*!
   * \fn     t_Void vCbProjectionVideoRenderingStatus(t_Bool bDisplayStatus)
   * \brief  Method to inform the Projection video rendering status update.
   * \param  enProjVideoRenderingStatus : Identifies whether Projection
                                          rendering is ON/OFF.
   * \retval t_Void
   **************************************************************************/
   t_Void vCbProjectionVideoRenderingStatus(tenProjVideoRenderingStatus enProjVideoRenderingStatus);
   
   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vOnNativeVideoRenderingStatus()
   ***************************************************************************/
   /*!
   * \fn     t_Void vOnNativeVideoRenderingStatus(t_Bool bDisplayStatus)
   * \brief  Method to inform the Native video rendering status update.
   * \param  enNativeVideoRenderingStatus : Identifies whether Native
                                            rendering is ON/OFF.
   * \retval t_Void
   **************************************************************************/
   t_Void vOnSetNativeVideoRenderingStatus(tenNativeVideoRenderingStatus enNativeVideoRenderingStatus);


   /***************************************************************************
    ** FUNCTION: t_Void spi_tclResourceMngr::vSetGeneralRestrictions
    ***************************************************************************/
   /*!
    * \fn     vSetGeneralRestriction
    * \brief  Interface to set the general restrictions for various technologies.
    * \param  enDeviceCategory  : Device category .
    * \param  u16GeneralRestrInfo : General Restriction Info for the technology.
    * \param  [OUT] void.
    **************************************************************************/
    t_Void vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory,t_U16 u16GeneralRestrInfo);

    /***************************************************************************
    ** FUNCTION: t_Void spi_tclResourceMngr::vSetETCStatus()
    ***************************************************************************/
    /*!
    * \fn     t_Void vSetETCStatus(t_Bool bETCStatus)
    * \brief  method to Set the ETC Status.
    * \param  bETCStatus : [IN] ETC Status Enabled or disabled.
    * \retval t_Void
    **************************************************************************/
    t_Void vSetETCStatus(t_Bool bETCStatus);

    /***************************************************************************
    ** FUNCTION:  t_Void spi_tclResourceMngr::vOnDisableBluetoothTrigger(
    ***************************************************************************/
    /*!
    * \fn      t_Void vOnDisableBluetoothTrigger(t_String szBTAddress)
    * \brief   Method to notify component that DisableBT request is received
    * \param   [IN] szBTAddress: Unique Devcie ID
    * \retval  t_Void
    **************************************************************************/
    t_Void vOnDisableBluetoothTrigger(t_String szBTAddress);

    /***************************************************************************
     ** FUNCTION:  t_Void spi_tclResourceMngr::vSetResourceMngrSettingsInstance(spi_tclResourceMngrSettingsIntf* poResourceMngrSettingsIntf)
     ***************************************************************************/
     /*!
     * \fn      t_Void vSetResourceMngrSettingsInstance()
     * \brief   Method to set Resource manager settings instance.
     * \param   poResourceMngrSettingsIntf : pointer to InputHandlerSettingsIntf.
     * \retval  t_Void
     **************************************************************************/
    t_Void vSetResourceMngrSettingsInstance(spi_tclResourceMngrSettingsIntf* poResourceMngrSettingsIntf);

    /***************************************************************************
    ** FUNCTION:  t_Void spi_tclResourceMngr::vOnEvaluateInitalModes()
    ***************************************************************************/
    /*!
    * \fn      t_Void vOnEvaluateInitalModes
    * \brief   Method to check and set inital modes before session is started
   * \param   [IN] corfszBTAddress: Unique Device ID
    * \retval  t_Void
    **************************************************************************/
    t_Void vOnEvaluateInitalModes(const t_String& corfszBTAddress);

    /***************************************************************************
     ** FUNCTION:  t_Void spi_tclResourceMngr::vPostExtVRStateUpdateMsg()
     ***************************************************************************/
     /*!
     * \fn      t_Void vPostExtVRStateUpdateMsg(t_String szBTAddress,
     *                         tenExtVRState e8ExtVRState,
     *                         tenVRErrorCode enVRErrorCode)
     * \brief   Method to post external VR state.update
     * \param   enExtVRState : External VR state.
     * \param   szBTAddress : BT address
     * \param   enVRErrorCode : VR error codes     
     * \retval  t_Void
     **************************************************************************/
    t_Void vPostExtVRStateUpdateMsg(t_String szBTAddress,
                              tenExtVRState enExtVRState,
                              tenVRErrorCode enVRErrorCode);
							  
    /***************************************************************************
       ** FUNCTION:  t_Void spi_tclResourceMngr::vGetResourceManagerConfig()
       ***************************************************************************/
       /*!
       * \fn      t_Void vGetResourceManagerConfig()
       * \brief   Method to print the Resource Manager configuration to read KDS
       * \retval  t_Void
       **************************************************************************/
    t_Void vGetResourceManagerConfig();
							  

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
	 *********************************PROTECTED**********************************
	 ***************************************************************************/

	/***************************************************************************
	 ** FUNCTION:  t_Void spi_tclResourceMngr::vPostTelephoneCallStatusMsg()
	 ***************************************************************************/
	/*!
	 * \fn      t_Void vPostTelephoneCallStatusMsg()
	 * \brief   Called when Telephone call status result is sent by Telephone Client Handler.
	 * \sa      None
	 **************************************************************************/
	t_Void vPostTelephoneCallStatusMsg(
			const std::vector<trTelCallStatusInfo> &corfTelCallStatusInfoList) override;

	/***************************************************************************
	 ****************************END OF PROTECTED********************************
	 ***************************************************************************/
private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclResourceMngr::spi_tclResourceMngr()
   ***************************************************************************/
   /*!
   * \fn      spi_tclResourceMngr()
   * \brief   Constructor
   * \sa      ~spi_tclResourceMngr()
   **************************************************************************/
   spi_tclResourceMngr();

   /***************************************************************************
   ** FUNCTION:  spi_tclResourceMngr& spi_tclResourceMngr::operator= (const..
   ***************************************************************************/
   /*!
   * \fn      spi_tclResourceMngr& operator= (const spi_tclResourceMngr &corfrSrc)
   * \brief   Assignment Operator, will not be implemented.
   * \note    This is a technique to disable the assignment operator for this class.
   *          So if an attempt for the assignment is made linker complains.
   * \param   corfrSrc : [IN] Source Object
   **************************************************************************/
   spi_tclResourceMngr& operator=(const spi_tclResourceMngr &corfrSrc);

   /***************************************************************************
   ** FUNCTION:  spi_tclResourceMngr::spi_tclResourceMngr(const spi_tclResourceMngr..
   ***************************************************************************/
   /*!
   * \fn      spi_tclResourceMngr(const spi_tclResourceMngr &corfrSrc)
   * \brief   Copy constructor, will not be implemented.
   * \note    This is a technique to disable the Copy constructor for this class.
   * \param   corfrSrc : [IN] Source Object
   **************************************************************************/
   spi_tclResourceMngr(const spi_tclResourceMngr &corfrSrc);

   /***************************************************************************
   ** FUNCTION:  t_Void  spi_tclResourceMngr::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \fn      t_Void vRegisterCallbacks()
   * \brief   To Register for the asynchronous responses that are required from
   *          ML/DiPo Resource Mngr
   * \retval  t_Void 
   **************************************************************************/
   t_Void vRegisterCallbacks();

   /***************************************************************************
   ** FUNCTION:  t_Bool  spi_tclResourceMngr::bValidateClient()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bValidateClient(t_U8 u8Index)
   * \brief   To validate the client index. check whether it is in the range of
   *          the Resource Manager clients Array
   * \retval  t_Bool
   **************************************************************************/
   inline t_Bool bValidateClient(t_U8 u8Index);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclResourceMngr::vStoreDisplayLastMode()
    ***************************************************************************/
   /*!
   * \fn     t_Void vStoreDisplayLastMode()
   * \brief  Method to store the video last mode into datapool
   * \retval t_Void
   **************************************************************************/  
   t_Void vStoreDisplayLastMode(tenDisplayInfo enDisplayInfo,
            const trDisplayContext& rfcorDisplayContext,
            const trDisplayConstraint& rfcorDisplayConstraint);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclResourceMngr::vStoreAudioLastMode()
    ***************************************************************************/
   /*!
   * \fn     t_Void vStoreAudioLastMode()
   * \brief  Method to store the audio last mode into datapool
   * \retval t_Void
   **************************************************************************/
   t_Void vStoreAudioLastMode(t_Bool bAudioFlag,
            tenAudioContext enAudioCntxt);

   //! Member Variable - pointer to the Base class for ML & DiPo Resource Managers
   spi_tclResourceMngrBase* m_poRsrcMngrBase[NUM_RSRCMNGR_CLIENTS];

   //! AppMngr Response Intf
   spi_tclResourceMngrResp* m_poRsrcMngrRespIntf;

   //! Current App state
   trDiPOAppState m_rCurrAppState;

   //! Selected device handle
   t_U32 m_u32SelectedDevice;

   //! selected device category
   tenDeviceCategory m_enDevCat;
   
   //! Indicates if source to be set as unavailable after ignition cycle if last mode was Projection.
   t_Bool m_bAudioSrcIdleRequired;
   
   spi_tclResourceMngrSettingsIntf* m_poResourceMngrSettings;
   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/
};
//spi_tclResourceMngr

#endif //_SPI_TCLRESOURCEMNGR_
