/*!
 *******************************************************************************
 * \file              spi_tclInputHandler.cpp
 * \brief             Main SPI input handler 
 *******************************************************************************
 \verbatim
 PROJECT:       Gen3
 SW-COMPONENT:  Smart Phone Integration
 DESCRIPTION:   Main input handler that creates ML and DiPo Input handlers
 AUTHOR:        Hari Priya E R (RBEI/ECP2)
 COPYRIGHT:     &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 25.11.2013 |  Hari Priya E R              | Initial Version
 12.03.2014 |  Hari Priya E R              | Included changes for handling ML and DiPO 
                                             Input handlers based on Device Type
 06.04.2014 |  Ramya Murthy                | Initialisation sequence implementation
 03.07.2014 |  Hari Priya E R              | Added changes related to Input Response interface
 31.07.2014 |  Ramya Murthy                | SPI feature configuration via LoadSettings()
 13.03.2015 |  Sameer Chandra              | SelectDevice Implementation for AAP.
 25.06.2015 |  Tejaswini HB (RBEI/ECP2)    | Featuring out Android Auto
 17.07.2015 |  Sameer Chandra              | Added Knob Encoder Handling.
 01.03.2017 |  Shiva Kumar G               | Changes to enable/disable CarPlay feature    
 
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/


#include "spi_tclInputHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
#include "spi_tclDiPoInputHandler.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
#include "spi_tclAAPInputHandler.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
#include "spi_tclMySPINInputHandler.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
#include "spi_tclBDCLInputHandler.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
#include "spi_tclOnCarInputHandler.h"
#endif

#include "spi_tclInputDevBase.h"
#include "spi_tclMediator.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_INPUTHANDLER
#include "trcGenProj/Header/spi_tclInputHandler.cpp.trc.h"
#endif
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
std::map<tenDeviceCategory, GetInputHandler_t*> InputHandler;
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
** FUNCTION:  spi_tclInputHandler::spi_tclInputHandler()
***************************************************************************/
spi_tclInputHandler::spi_tclInputHandler(spi_tclInputRespIntf* poInputRespIntf): spi_tclSelectionIntf(e32COMPID_INPUTHANDLER),
m_poInputRespIntf(poInputRespIntf),m_poInputHandlerSettings(NULL)
{
   ETG_TRACE_USR1(("spi_tclInputHandler::spi_tclInputHandler entered"));

   for(t_U8 u8Index=0; u8Index<NUM_INPUT_CLIENTS;u8Index++)
   {
      m_poInputDevBase[u8Index] = NULL;
   }//for()
}

/***************************************************************************
 ** FUNCTION:  spi_tclInputHandler::~spi_tclInputHandler()
 ***************************************************************************/
spi_tclInputHandler::~spi_tclInputHandler()
{
   ETG_TRACE_USR1(("spi_tclInputHandler::~spi_tclInputHandler entered"));

   for(t_U8 u8Index=0;u8Index < NUM_INPUT_CLIENTS; u8Index++)
   {
      RELEASE_MEM(m_poInputDevBase[u8Index]);
   }
   m_poInputRespIntf = NULL;
   m_poInputHandlerSettings = NULL;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclInputHandler::bInitialize()
***************************************************************************/
t_Bool spi_tclInputHandler::bInitialize()
{
   t_Bool bInit = false;
   
   if(NULL!= m_poInputRespIntf)
   {
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
      ETG_TRACE_USR1(("spi_tclInputHandler::bInitialize InputHandler[e8DEV_TYPE_MIRRORLINK]=%p",InputHandler[e8DEV_TYPE_MIRRORLINK]));
      if ((NULL != InputHandler[e8DEV_TYPE_MIRRORLINK]))
      {
         ETG_TRACE_USR1(("spi_tclAudio::bInitialize before obj create m_poInputDevBase[e8DEV_TYPE_MIRRORLINK] = %p ", m_poInputDevBase[e8DEV_TYPE_MIRRORLINK]));
         m_poInputDevBase[e8DEV_TYPE_MIRRORLINK] = InputHandler[e8DEV_TYPE_MIRRORLINK](m_poInputRespIntf);
         SPI_NORMAL_ASSERT(NULL == m_poInputDevBase[e8DEV_TYPE_MIRRORLINK]);
      }
      ETG_TRACE_USR1(("spi_tclAudio::bInitialize obj success m_poInputDevBase[e8DEV_TYPE_MIRRORLINK] = %p ", m_poInputDevBase[e8DEV_TYPE_MIRRORLINK]));
#endif
   //! Create ML or DiPO Input handlers based on the Device Type
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
   m_poInputDevBase[e8DEV_TYPE_DIPO]  =  new(std::nothrow) spi_tclDiPoInputHandler();
   SPI_NORMAL_ASSERT(NULL == m_poInputDevBase[e8DEV_TYPE_DIPO]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
   m_poInputDevBase[e8DEV_TYPE_ANDROIDAUTO] =  new(std::nothrow) spi_tclAAPInputHandler();
   SPI_NORMAL_ASSERT(NULL == m_poInputDevBase[e8DEV_TYPE_ANDROIDAUTO]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
   m_poInputDevBase[e8DEV_TYPE_MYSPIN] = new (std::nothrow) spi_tclMySPINInputHandler();
   SPI_NORMAL_ASSERT(NULL == m_poInputDevBase[e8DEV_TYPE_MYSPIN]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
   m_poInputDevBase[e8DEV_TYPE_CARLIFE] = new (std::nothrow) spi_tclBDCLInputHandler();
   SPI_NORMAL_ASSERT(NULL == m_poInputDevBase[e8DEV_TYPE_CARLIFE]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
       m_poInputDevBase[e8DEV_TYPE_ONCAR] = new (std::nothrow) spi_tclOnCarInputHandler();
       SPI_NORMAL_ASSERT(NULL == m_poInputDevBase[e8DEV_TYPE_ONCAR]);
#endif

   vRegisterCallbacks();
   
      //! Initialize the Input handlers
      for(t_U8 u8Index = 0; u8Index < NUM_INPUT_CLIENTS; ++u8Index)
      {
          tenDeviceCategory enDevCat = static_cast<tenDeviceCategory>(u8Index);
          if (NULL != m_poInputDevBase[enDevCat])
          {
              bInit = m_poInputDevBase[enDevCat]->bInitialise() || bInit;
          }
       }
   }

   ETG_TRACE_USR1(("spi_tclInputHandler::bInitialize Initialization Status = %d", ETG_ENUM(BOOL, bInit)));
 
   return bInit;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclInputHandler::bUnInitialize()
***************************************************************************/
t_Bool spi_tclInputHandler::bUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclInputHandler::bUnInitialize entered"));

   return true;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vLoadSettings()
***************************************************************************/
t_Void spi_tclInputHandler::vLoadSettings()
{
   ETG_TRACE_USR1(("spi_tclInputHandler::vLoadSettings entered"));
   for(t_U8 u8Index=0; u8Index<NUM_INPUT_CLIENTS;u8Index++)
   {
      if( NULL != m_poInputDevBase[u8Index] )
      {
         m_poInputDevBase[u8Index]->vSetInputHandlerSettingsInstance(m_poInputHandlerSettings);
      }
   }//for()

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vSaveSettings()
***************************************************************************/
t_Void spi_tclInputHandler::vSaveSettings()
{
   ETG_TRACE_USR1(("spi_tclInputHandler::vSaveSettings entered"));
   //Add code
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vProcessTouchEvent()
***************************************************************************/
t_Void spi_tclInputHandler::vProcessTouchEvent(t_U32 u32DeviceHandle,
                                               tenDeviceCategory enDevCat,
                                               trTouchData &rfrTouchData)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   if( true == bValidateClient(u8Index))
   {

      //!Forward the touch data to ML or DIPO Handler based on Device Type
      m_poInputDevBase[u8Index]->vProcessTouchEvent(u32DeviceHandle,rfrTouchData);
   } //if(( u8Index < NUM_INPUT_CLIENTS)
}


/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vProcessKeyEvents()
***************************************************************************/
t_Void spi_tclInputHandler::vProcessKeyEvents(t_U32 u32DeviceHandle,
                                              tenDeviceCategory enDevCat,
                                              tenKeyMode enKeyMode, 
                                              tenKeyCode enKeyCode)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   if( true == bValidateClient(u8Index))
   {
      //!Forward the Key data to ML or DIPO Handler based on Device Type
      m_poInputDevBase[u8Index]->vProcessKeyEvents(u32DeviceHandle,
         enKeyMode,enKeyCode);
   } //if(( u8Index < NUM_INPUT_CLIENTS)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vSelectDevice()
***************************************************************************/
t_Void spi_tclInputHandler::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)

{
   t_U8 u8Index = static_cast<t_U8>(corfrSelectReq.m_enDevCategory);
   ETG_TRACE_USR1(("spi_tclInputHandler::vSelectDevice:Device Id - 0x%x  Device Category - %d"
                    ,corfrSelectReq.m_u32DeviceHandle,ETG_ENUM(DEVICE_CATEGORY,u8Index)));

   if (true == bValidateClient(u8Index))
   {
      m_poInputDevBase[u8Index]->vSelectDevice(corfrSelectReq.m_u32DeviceHandle,
               corfrSelectReq.m_enDevConnReq);
   } //if(true == bValidateClient(u8Index))
   else
   {
      vCbSelectDeviceResp(corfrSelectReq.m_u32DeviceHandle, e8INVALID_DEV_HANDLE);
   }
}

/***************************************************************************
** FUNCTION:  t_Void  spi_tclInputHandler::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclInputHandler::vRegisterCallbacks()
{
   /*lint -esym(40,fvSelectDeviceResp)fvSelectDeviceResp Undeclared identifier */
   /*lint -esym(40,_1) _1 Undeclared identifier */
   /*lint -esym(40,_2) _2 Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclInputHandler::vRegisterCallbacks entered"));

   trInputCallbacks rInputCbs;

   rInputCbs.fvSelectDeviceResp = std::bind(&spi_tclInputHandler::vCbSelectDeviceResp,this, SPI_FUNC_PLACEHOLDERS_2);

   //! bind to the callback function for XDevice Key Icon data
   rInputCbs.fvKeyIconDataResp =  std::bind(&spi_tclInputHandler::vCbKeyIconDataResp, this,SPI_FUNC_PLACEHOLDERS_4);

   //! bind to the callback function for ML Server key capabilities
   rInputCbs.fvSrvKeyCapabilitiesInfo = std::bind(&spi_tclInputHandler::vCbServerKeyCapInfoResp, this, SPI_FUNC_PLACEHOLDERS_4);

   for (t_U8 u8Index = 0; u8Index < NUM_INPUT_CLIENTS; u8Index++)
   {
      if (NULL != m_poInputDevBase[u8Index])
      {
         m_poInputDevBase[u8Index]->vRegisterInputCallbacks(rInputCbs);
      }//if((NULL != m_poInputDevBase[

   }//for(t_U8 u8Index=0;u8Index < NUM_INPUT_CLIENTS; u8Index++)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclInputHandler::vCbSelectDeviceResp()
***************************************************************************/
t_Void spi_tclInputHandler::vCbSelectDeviceResp(const t_U32 cou32DevId,
                                         const tenErrorCode coenErrorCode)
{
   ETG_TRACE_USR1(("spi_tclInputHandler::vCbSelectDeviceResp:Device Id-0x%x ErrorCode-%d",
      cou32DevId,ETG_ENUM(ERROR_CODE,coenErrorCode)));

   spi_tclMediator *poMediator = spi_tclMediator::getInstance();

   if(NULL != poMediator)
   {
      poMediator->vPostSelectDeviceRes(e32COMPID_INPUTHANDLER, coenErrorCode);
   }//if(NULL != poMediator)
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vSelectDeviceResult()
***************************************************************************/
t_Void spi_tclInputHandler::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode)
{
   t_U8 u8Index = static_cast<t_U8>(corfrSelectReq.m_enDevCategory);

   ETG_TRACE_USR1(("spi_tclInputHandler::vOnSelectDeviceResult:Device Id-0x%x,Device Category-%d,Connection Request-%d Response Code-%d",
            corfrSelectReq.m_u32DeviceHandle,ETG_ENUM(DEVICE_CATEGORY,corfrSelectReq.m_enDevCategory),
            ETG_ENUM(CONNECTION_REQ,corfrSelectReq.m_enDevConnReq),ETG_ENUM(ERROR_CODE, enErrorCode)));
   tenResponseCode enRespCode = (e8NO_ERRORS == enErrorCode)? e8SUCCESS:e8FAILURE;
   if (true == bValidateClient(u8Index))
   {
      m_poInputDevBase[u8Index]->vOnSelectDeviceResult(corfrSelectReq.m_u32DeviceHandle, corfrSelectReq.m_enDevConnReq, enRespCode);
   }//if (true == bValidateClient(u8Index))
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclInputHandler::bValidateClient()
***************************************************************************/
inline t_Bool spi_tclInputHandler::bValidateClient(const t_U8 cou8Index)
{

   t_Bool bRet = (cou8Index < NUM_INPUT_CLIENTS) && (NULL != m_poInputDevBase[cou8Index]);

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vProcessKnobKeyEvents
***************************************************************************/
t_Void spi_tclInputHandler::vProcessKnobKeyEvents(t_U32 u32DeviceHandle,tenDeviceCategory enDevCat,
      t_S8 s8EncoderDeltaCnt)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   if( true == bValidateClient(u8Index))
   {
      //!Forward the Key data to ML/DIPO/AAP Handler based on Device Type
      m_poInputDevBase[u8Index]->vProcessKnobKeyEvents(u32DeviceHandle, s8EncoderDeltaCnt);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vGetKeyIconData()
***************************************************************************/
t_Void spi_tclInputHandler::vGetKeyIconData(const t_U32 cou32DevId,
                                       t_String szKeyIconUrl,
                                       tenDeviceCategory enDevCat,
                                       const trUserContext& rfrcUsrCntxt)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclInputHandler::vGetKeyIconData:Device Category-%d IconURL-%s",
           ETG_ENUM(DEVICE_CATEGORY,enDevCat),szKeyIconUrl.c_str()));

   if(true == bValidateClient(u8Index))
   {
      m_poInputDevBase[u8Index]->vGetKeyIconData(cou32DevId,szKeyIconUrl,enDevCat,rfrcUsrCntxt);
   }
   else
   {
      //invalid handle - Post error
      vCbKeyIconDataResp(cou32DevId,NULL,0,rfrcUsrCntxt);
   }
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vCbKeyIconDataResp()
***************************************************************************/
t_Void spi_tclInputHandler::vCbKeyIconDataResp(const t_U32 cou32DevId,
                                            const t_U8* pcu8KeyIconData,
                                            t_U32 u32Len,
                                            const trUserContext& rfrcUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclInputHandler::vCbKeyIconDataResp entered"));

   if(NULL != m_poInputRespIntf)
   {
      //! Post KeyIconData Result
      m_poInputRespIntf->vPostKeyIconDataResult(cou32DevId,pcu8KeyIconData,
         u32Len,rfrcUsrCntxt);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclInputHandler::vCbServerKeyCapInfoResp()
***************************************************************************/
t_Void spi_tclInputHandler::vCbServerKeyCapInfoResp(const t_U32 cou32DevId,
                                                  t_U16 u16NumXDevices,
                                                  trMLSrvKeyCapabilities rSrvKeyCapabilities,
                                                  t_U16 u16ValidityOfData)
{
   ETG_TRACE_USR1(("spi_tclInputHandler::vCbServerKeyCapInfoResp entered"));

   if(NULL != m_poInputRespIntf)
   {
      //! Post XDeviceKeyInfo update
      m_poInputRespIntf->vPostMLServerCapInfo(cou32DevId,u16NumXDevices,rSrvKeyCapabilities,u16ValidityOfData);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclInputHandler::vSetInputHandlerSettingsInstance()
***************************************************************************/
t_Void spi_tclInputHandler::vSetInputHandlerSettingsInstance(spi_tclInputHandlerSettingsIntf* poInputHandlerSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclInputHandler::vSetInputHandlerSettingsInstance entered"));
   if( NULL != poInputHandlerSettingsIntf)
   {
       ETG_TRACE_USR1(("spi_tclInputHandler::vSetInputHandlerSettingsInstance: Setting m_poInputHandlerSettings val"));
       m_poInputHandlerSettings = poInputHandlerSettingsIntf;
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclInputHandler::vGetInputHandlerConfig()
***************************************************************************/

t_Void spi_tclInputHandler:: vGetInputHandlerConfig()
{
   ETG_TRACE_USR1(("spi_tclInputHandler::vGetInputHandlerConfig entered"));
   if( NULL != m_poInputHandlerSettings)
   {
       m_poInputHandlerSettings->bGetRotaryCtrlSupport();
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclInputHandler::vRegisterKeyCodes()
***************************************************************************/
t_Void spi_tclInputHandler::vRegisterKeyCodes(
                       std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechSuppHardkeys,
                       std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechSuppSoftkeys,
                       const trUserContext& corfrUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclInputHandler::vRegisterKeyCodes entered"));
   std::map<tenDeviceCategory,std::set<tenKeyCode>> corfrmapTechUnSuppHardkeys;
   std::map<tenDeviceCategory,std::set<tenKeyCode>> corfrmapTechUnSuppSoftkeys;

   t_Bool isValid = true;
   t_U8 u8Index = 0;

   for(auto const &TechKeys : rfrmapTechSuppHardkeys)
   {
       if(0 == TechKeys.second.size())
       {
          rfrmapTechSuppHardkeys[TechKeys.first].insert(e32INVALID_KEY);
       }
   }

   for(auto const &TechKeys : rfrmapTechSuppSoftkeys)
   {
       if(0 == TechKeys.second.size())
       {
          rfrmapTechSuppSoftkeys[TechKeys.first].insert(e32INVALID_KEY);
       }
   }

   if( NULL != m_poInputHandlerSettings)
   {
     m_poInputHandlerSettings->vRegisterKeyCodes(rfrmapTechSuppHardkeys,rfrmapTechSuppSoftkeys);
     isValid = bCheckKeyValidity(rfrmapTechSuppHardkeys,rfrmapTechSuppSoftkeys,corfrmapTechUnSuppHardkeys,corfrmapTechUnSuppSoftkeys);
   }
   for(auto const &TechKeys : rfrmapTechSuppHardkeys)
   {
      u8Index = static_cast<t_U8>(TechKeys.first);

      if (true == bValidateClient(u8Index))
      {
          m_poInputDevBase[u8Index]->vUpdateKeyInfo();
      }
   }
   for(auto const &TechKeys : rfrmapTechSuppSoftkeys)
   {
      u8Index = static_cast<t_U8>(TechKeys.first);

      if (true == bValidateClient(u8Index))
      {
          m_poInputDevBase[u8Index]->vUpdateSoftKeyInfo();
      }
   }
   if(NULL != m_poInputRespIntf)
   {
      //! Post KeyInfo Result to Clients
      m_poInputRespIntf->vPostKeyInfoResult(isValid,corfrmapTechUnSuppHardkeys,corfrmapTechUnSuppSoftkeys,corfrUsrCntxt);
   }
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclInputHandler::bCheckHardKeyValidity()
***************************************************************************/
t_Bool spi_tclInputHandler::bCheckKeyValidity(
              const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppHardkeys,
              const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppSoftkeys,
              std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechUnSuppHardkeys,
              std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechUnSuppSoftkeys)
{
   ETG_TRACE_USR1(("spi_tclInputHandler::bCheckKeyValidity entered"));
   t_Bool bretVal   = true;
   std::set<tenKeyCode> setUnSuppkeys;
   t_U8 u8Index = 0;
   for(auto const &TechKeys : corfrmapTechSuppHardkeys)
   {
      u8Index = static_cast<t_U8>(TechKeys.first);
      if (true == bValidateClient(u8Index))
      {
         if (false == m_poInputDevBase[u8Index]->bCheckHardKeyValidity(
                                      TechKeys.second,setUnSuppkeys))
         {
           rfrmapTechUnSuppHardkeys[static_cast<tenDeviceCategory>(u8Index)] = setUnSuppkeys;
           setUnSuppkeys.clear();
           bretVal = false;
         }
      }
   }
   for(auto const &TechKeys : corfrmapTechSuppSoftkeys)
   {
      u8Index = static_cast<t_U8>(TechKeys.first);
      if (true == bValidateClient(u8Index))
      {
         if (false == m_poInputDevBase[u8Index]->bCheckSoftKeyValidity(
                                      TechKeys.second,setUnSuppkeys))
         {
           rfrmapTechUnSuppSoftkeys[static_cast<tenDeviceCategory>(u8Index)] = setUnSuppkeys;
           setUnSuppkeys.clear();
           bretVal = false;
         }
      }
   }
   return bretVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclInputHandler::vResetKeyInfo()
***************************************************************************/
t_Void spi_tclInputHandler::vResetKeyInfo()
{
   ETG_TRACE_USR1(("spi_tclInputHandler::vResetKeyInfo entered"));
   for(t_U8 u8Index=0; u8Index<NUM_INPUT_CLIENTS;u8Index++)
   {
      if( NULL != m_poInputDevBase[u8Index] )
      {
         m_poInputDevBase[u8Index]->vUpdateKeyInfo();
         m_poInputDevBase[u8Index]->vUpdateSoftKeyInfo();
      }
   }
}
//lint ?restore
