/*!
 *******************************************************************************
 * \file             spi_tclDataServiceDevBase.h
 * \brief            Abstract class that specifies the Data Service interface which
 *                   must be implemented by device class (Mirror Link/Digital iPod out)
 *                   for communication with SDK
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Base class for ML and DiPO Interfaces
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 27.03.2014 |  Ramya Murthy  (RBEI/ECP2)        | Initial Version
 13.06.2014 |  Ramya Murthy (RBEI/ECP2)         | Interface revised to adapt to MPlay FI
                                                  extn to start/stop loc info, and VDSensor 
                                                  data integration in loc data.
 13.10.2014 |  Hari Priya E R (RBEI/ECP2)       | Added interface for Vehicle Data for PASCD.
 05.02.2015 |  Ramya Murthy (RBEI/ECP2)         | Added interface to set availability of LocationData
 28.03.2015 |  SHITANSHU SHEKHAR (RBEI/ECP2)    | Added interface to set vehicle config
 23.04.2015 |  Ramya Murthy (RBEI/ECP2)         | Added interface to set region code
 16.01.2019 |  Roveena Francy Lobo (RBEI/ECO2)  | Added vSetGeneralRestrictions method 
 16.01.2019 |  Roveena Francy Lobo (RBEI/ECO2)  | Added vSetETCStatus method 
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#ifndef SPI_TCLDATASERVICEDEVBASE_H
#define SPI_TCLDATASERVICEDEVBASE_H

#include "BaseTypes.h"
#include "spi_tclDataServiceTypes.h"
#include "spi_tclDataServiceSettingsIntf.h"

/**
 * Abstract class that specifies the Data Service interface which
 * must be implemented by device class (Mirror Link/Digital iPod out)
 * for communication with SDK
 */

class spi_tclDataServiceDevBase
{
public:

	/***************************************************************************
	*********************************PUBLIC*************************************
	***************************************************************************/

	/***************************************************************************
    ** FUNCTION:  spi_tclDataServiceDevBase::spi_tclDataServiceDevBase();
    ***************************************************************************/
   /*!
    * \fn      spi_tclDataServiceDevBase()
    * \brief   Default Constructor
    ***************************************************************************/
   spi_tclDataServiceDevBase():m_poDataServiceSettings(NULL)
   {
   }

   /***************************************************************************
    ** FUNCTION:  spi_tclDataServiceDevBase::~spi_tclDataServiceDevBase();
    ***************************************************************************/
   /*!
    * \fn      ~spi_tclDataServiceDevBase()
    * \brief   Virtual Destructor
    ***************************************************************************/
   virtual ~spi_tclDataServiceDevBase()
   {
      m_poDataServiceSettings = NULL;
   }

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDataServiceDevBase::bInitialise();
    ***************************************************************************/
   /*!
    * \fn      bInitialise()
    * \brief   Method to initialises the service handler. (Performs initialisations which
    *          are not device specific.)
    *          Mandatory interface to be implemented.
    * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
    * \sa      bUninitialise()
    ***************************************************************************/
   virtual t_Bool bInitialise() = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDataServiceDevBase::bUninitialise();
    ***************************************************************************/
   /*!
    * \fn      bUninitialise()
    * \brief   Method to uninitialise the service handler.
    *          Mandatory interface to be implemented.
    * \retval  t_Bool: TRUE - If ServiceHandler is uninitialised successfully, else FALSE.
    * \sa      bInitialise()
    ***************************************************************************/
   virtual t_Bool bUninitialise() = 0;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vLoadSettings()
    ***************************************************************************/
   /*!
    * \fn      vLoadSettings()
    * \brief   Invoked during OFF->NORMAL state transition
    * \retval  N/A
    * \sa      N/A
    ***************************************************************************/
   virtual t_Void vLoadSettings(){};

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vSelectDevice(t_U32...)
    ***************************************************************************/
   /*!
    * \fn      vSelectDevice
    * \brief   Called when a device is selection request is received
    * \param   [IN] cou32DevId: Unique handle of selected device
    * \param   [IN] enConnReq: Indicated the category of the device
    * \retval  None
    **************************************************************************/

   virtual t_Void vSelectDevice(const t_U32 /*cou32DevId*/,
         tenDeviceConnectionReq /*enConnReq*/){};

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnSelectDeviceResult(t_U32...)
    ***************************************************************************/
   /*!
    * \fn      vOnSelectDeviceResult(t_U32 u32DeviceHandle)
    * \brief   Called when a device is selected.
    *          Mandatory interface to be implemented.
    * \param   [IN] u32DeviceHandle: Unique handle of selected device
    * \param   coenConnReq : [IN] Identifies the Connection Request.
    * \param   coenRespCode: [IN] Response code. Success/Failure
    * \retval  None
    **************************************************************************/
   virtual t_Void vOnSelectDeviceResult(t_U32 u32DeviceHandle,
            tenDeviceConnectionReq coenConnReq,
            tenResponseCode coenRespCode) = 0;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnDeselectDeviceResult(t_U32...)
    ***************************************************************************/
   /*!
    * \fn      vOnDeselectDeviceResult(t_U32 u32DeviceHandle)
    * \brief   Called when currently selected device is de-selected.
    *          Mandatory interface to be implemented.
    * \param   [IN] u32DeviceHandle: Unique handle of selected device
    * \retval  None
    **************************************************************************/
   virtual t_Void vOnDeselectDeviceResult(t_U32 u32DeviceHandle) = 0;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vStartLocationData(...)
    ***************************************************************************/
   /*!
    * \fn      vStartLocationData(
    *             const std::vector<tenNmeaSentenceType>& rfcoNmeaSentencesList)
    * \brief   Called to start sending LocationData to selected device.
    *          Mandatory interface to be implemented.
    * \param   [IN] rfcoNmeaSentencesList: List of NMEA sentences to be sent in
    *             location data.
    * \retval  None
    **************************************************************************/
   virtual t_Void vStartLocationData(
         const std::vector<tenNmeaSentenceType>& /*rfcoNmeaSentencesList*/,t_U8 /*u8RequestedBy = 0*/) {}

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vStopLocationData(...)
    ***************************************************************************/
   /*!
    * \fn      vStopLocationData()
    * \brief   Called to stop sending LocationData to selected device.
    *          Mandatory interface to be implemented.
    * \param   [IN] rfcoNmeaSentencesList: List of NMEA sentences which should
    *             be stopped in location data.
    * \retval  None
    **************************************************************************/
   virtual t_Void vStopLocationData(
         const std::vector<tenNmeaSentenceType>& /*rfcoNmeaSentencesList*/,t_U8 /*u8RequestedBy = 0*/) {}

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnData(const trGPSData& rfcorGpsData)
   ***************************************************************************/
   /*!
   * \fn      vOnData(const trGPSData& rfcorGpsData)
   * \brief   Method to receive GPS data.
   *          Optional interface to be implemented.
   * \param   rfcorGpsData: [IN] GPS data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnData(const trGPSData& /*rfcorGpsData*/){}

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnData(const trSensorData&...)
   ***************************************************************************/
   /*!
   * \fn      vOnData(const trSensorData& rfcorSensorData)
   * \brief   Method to receive Sensor data.
   *          Optional interface to be implemented.
   * \param   rfcorSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnData(const trSensorData& /*rfcorSensorData*/){}

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnTimeFormat(const t_U8 cou8TimeFormat)
   ***************************************************************************/
   /*!
   * \fn      vOnTimeFormat(const t_U8 cou8TimeFormat)
   * \brief   Method to receive TimeFormat data.
   *          Optional interface to be implemented.
   * \param   cou8TimeFormat: [IN] TimeFormat data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnTimeFormat(const t_U8 cou8TimeFormat)
   {
      SPI_INTENTIONALLY_UNUSED(cou8TimeFormat);
   }

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnAccSensorData
   ** (const std::vector<trAccSensorData>& corfvecrAccSensorData)
   ***************************************************************************/
   /*!
   * \fn      vOnAccSensorData(const std::vector<trAccSensorData>& corfvecrAccSensorData)
   * \brief   Method to receive Sensor data.
   *          Optional interface to be implemented.
   * \param   corfvecrAccSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnAccSensorData(const std::vector<trAccSensorData>& corfvecrAccSensorData)
   {
	   SPI_INTENTIONALLY_UNUSED(corfvecrAccSensorData);
   }

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnGyroSensorData
   ** (const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
   ***************************************************************************/
   /*!
   * \fn      vOnGyroSensorData(const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
   * \brief   Method to receive Sensor data.
   *          Optional interface to be implemented.
   * \param   corfvecrGyroSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnGyroSensorData(const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
   {
	   SPI_INTENTIONALLY_UNUSED(corfvecrGyroSensorData);
   }

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnData(
                 const trVehicleData& rfcoVehicleData)
   ***************************************************************************/
   /*!
   * \fn      vOnData(const trVehicleData& rfcoVehicleData)
   * \brief   Method to receive Vehicle data.
   *          Optional interface to be implemented.
   * \param   rfcoVehicleData: [IN] Vehicle data
   * \param   bSolicited: [IN] True if the data update is for changed vehicle data, else False
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnData(const trVehicleData& /*rfcoVehicleData*/, t_Bool /*bSolicited*/){}

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vSetLocDataAvailablility(...)
   ***************************************************************************/
   /*!
   * \fn      vSetLocDataAvailablility(t_Bool bLocDataAvailable)
   * \brief   Interface to set the availability of LocationData
   * \param   rfrDataServiceConfigData
   * \retval  None
   **************************************************************************/
   virtual t_Void vSetSensorDataAvailablility(const trDataServiceConfigData& /*rfrDataServiceConfigData*/){}

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDataServiceDevBase::vSetRegion(...)
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetRegion(tenRegion enRegion)
   * \brief  Interface to set the current region
   * \param  [IN] enRegion : Region enumeration
   * \sa
   **************************************************************************/
   virtual t_Void vSetRegion(tenRegion /*enRegion*/) {}

   /***************************************************************************
   ** FUNCTION:  spi_tclDataServiceDevBase::vOnEnvironmentData()
   ***************************************************************************/
   /*!
   * \fn    t_Void vOnEnvironmentData(t_Bool bValidTempUpdate,t_Double dTemp,
   *                                   t_Bool bValidPressureUpdate, t_Double dPressure)
   * \brief Use this call to report Environment data to Phone
   * \param bValidTempUpdate : [IN] Temp update is valid
   * \param dTemp : [IN] Temp in Celsius
   * \param bValidPressureUpdate: [IN] Pressure update is valid
   * \param dPressure : [IN] Pressure in KPA
   * \retval t_Void
   **************************************************************************/
   virtual t_Void vOnEnvironmentData(t_Bool /*bValidTempUpdate*/,t_Double /*dTemp*/,
      t_Bool /*bValidPressureUpdate*/, t_Double /*dPressure*/){}

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vSetFeatureRestrictions()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
    *          const t_U8 cou8DriveModeRestrictionInfo)
    * \brief   Method to set Vehicle Park/Drive Mode Restriction
    * \param   cou8ParkModeRestrictionInfo : Park mode restriction
    *          cou8DriveModeRestrictionInfo : Drive mode restriction
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vSetFeatureRestrictions(const t_U8 /*cou8ParkModeRestrictionInfo*/,
         const t_U8 /*cou8DriveModeRestrictionInfo*/){}

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vSetGeneralRestrictions()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetGeneralRestrictions(t_Bool bETCSupport)
    * \brief   Method to set Vehicle ETC status
    * \param
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vSetGeneralRestrictions(t_U16 u16GeneralRestrInfo)
   {
	   SPI_INTENTIONALLY_UNUSED(u16GeneralRestrInfo);
   }

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vSetETCStatus()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetETCStatus(t_Bool bETCStatus)
    * \brief   Method to set ETC Status
    * \param   bETCStatus: [IN] ETC Status (Enabled or Disabled)
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vSetETCStatus(t_Bool bETCSupport)
   {
	   SPI_INTENTIONALLY_UNUSED(bETCSupport);
   }

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vSetGeoCoordinateSystem()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetGeoCoordinateSystem()
    * \brief   Testing function to set coordinate system
    * \param   [IN] enGeoCoordinateSystemType : Coordinate System
    * \sa      None
    **************************************************************************/
   virtual t_Void vSetGeoCoordinateSystem(tenGeoCoordinateSystemType /*enGeoCoordinateSystemType*/){}

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vPostGPRMCDataStatusValues()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostGPRMCDataStatusValues()
    * \brief   Called when GPRMC Data Status Values are sent by Nav Data Client Handler.
    *          Mandatory interface to be implemented.
    * \param   [IN]u32DeviceId : Device ID
    * \param   [IN]u8GPRMCDataStatusValues : GPRMC Data Status Values i.e. A, V, X
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostGPRMCDataStatusValues(t_U32 /*u32DeviceId*/,t_U8 /*u8GPRMCDataStatusValues*/){}

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vOnVehiclePowerMode(tenVehiclePowerMode enVehiclePowerModeState)
   ***************************************************************************/
   /*!
   * \fn      vOnVehiclePowerModeState(tenVehiclePowerMode enVehiclePowerModeState)
   * \brief   Method to receive Vehicle Power Mode state
   * \param   enVehiclePowerModeState: [IN] Vehicle Power Mode state
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnVehiclePowerMode(tenVehiclePowerMode enVehiclePowerModeState)
   {
      SPI_INTENTIONALLY_UNUSED(enVehiclePowerModeState);
   }

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vOnAmbientColor(const trAmbientColorData& corfrAmbientColorData)
   ***************************************************************************/
   /*!
   * \fn      vOnAmbientColorData(const trAmbientColorData& corfrAmbientColorData)
   * \brief   Method to receive Ambient color data
   * \param   corfrAmbientColorData: [IN] Ambient color Data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnAmbientColor(const trAmbientColorData& corfrAmbientColorData)
   {
      SPI_INTENTIONALLY_UNUSED(corfrAmbientColorData);
   }

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vSetDataServiceSettingsInstance(spi_tclDataServiceSettingsIntf* poDataServiceSettingsIntf)
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetDataServiceSettingsInstance()
    * \brief   Method to set dataservice settings instance.
    * \param   poDataServiceSettingsIntf  : pointer to DataServiceSettingsIntf.
    * \retval  t_Void
    **************************************************************************/
   t_Void vSetDataServiceSettingsInstance(spi_tclDataServiceSettingsIntf* poDataServiceSettingsIntf)
    {
       if( NULL != poDataServiceSettingsIntf)
       {
          m_poDataServiceSettings = poDataServiceSettingsIntf;
       }
    }
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnSendLanguageSetting(tenLanguageSetting enLanguageSetting)
   ***************************************************************************/
   /*!
   * \fn      vOnSendLanguageSetting(tenLanguageSetting enLanguageSetting)
   * \brief   Method to receive Language data.
   *          Optional interface to be implemented.
   * \param   enLanguageSetting: [IN] Language  data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnSendLanguageSetting(tenLanguageSetting enLanguageSetting)
   {
      SPI_INTENTIONALLY_UNUSED(enLanguageSetting);
   }
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnDistanceToEmptyTank(t_S32& rfrDistToEmptyTank)
   ***************************************************************************/
   /*!
   * \fn      vOnDistanceToEmptyTank(t_S32& rfrDistToEmptyTank)
   * \brief   Method to receive DistanceToEmptyTank data
   * \param   rfrDistToEmptyTank : [IN] DistanceToEmptyTank data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnDistanceToEmptyTank(t_S32& /*rfrDistToEmptyTank*/){}

protected:

    //! Data Service Settings Interface Pointer
    spi_tclDataServiceSettingsIntf* m_poDataServiceSettings;

private:

	/***************************************************************************
	*********************************PRIVATE************************************
	***************************************************************************/

};

typedef spi_tclDataServiceDevBase* GetDataService_t(trDataServiceCb&);
extern std::map<tenDataServicesIndex,GetDataService_t*> DataService;

#endif // SPI_TCLDATASERVICEDEVBASE_H

