  /*!
 *******************************************************************************
 * \file         spi_tclDataService.h
 * \brief        Data Service Manager class that provides interface to delegate
                 the execution of command and handle response
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Data Service Manager class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 27.03.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 14.04.2014 |  Ramya Murthy (RBEI/ECP2)         | Implemented sending GPS data to
                                                  MediaPlayer client.
 06.04.2014 |  Ramya Murthy (RBEI/ECP2)         | Initialisation sequence implementation
 13.06.2014 |  Ramya Murthy (RBEI/ECP2)         | Implementation for: 
                                                  (1) MPlay FI extn to start/stop loc info
                                                  (2) VDSensor data integration
                                                  (3) NMEA-PASCD sentence for DiPO
 31.07.2014 |  Ramya Murthy (RBEI/ECP2)         | SPI feature configuration via LoadSettings()
 13.10.2014 |  Hari Priya E R (RBEI/ECP2)       | Added interface for Vehicle Data for PASCD and
                                                  interface with Data Service Response
 05.02.2015 |  Ramya Murthy (RBEI/ECP2)         | Added interface to set availability of LocationData
 28.03.2015 |  SHITANSHU SHEKHAR (RBEI/ECP2)    | Added index for ANDROIDAUTO and
                                                   declared following methods :
                                                   enGetDataServiceIndex()
23.04.2015  |  Ramya Murthy (RBEI/ECP2)         | Added interface to set region code
30.10.2015  | Shiva Kumar G                     | Implemented ReportEnvironment Data feature
 16.01.2019 |  Roveena Francy Lobo (RBEI/ECO2)  | Added vSetGeneralRestrictions method 
 16.01.2019 |  Roveena Francy Lobo (RBEI/ECO2)  | Added vSetETCStatus method 
 20.03.2019 |  Roveena Francy Lobo              | Added vOnSendLanguageSetting() for private cdb

 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLDATASERVICE_H
#define _SPI_TCLDATASERVICE_H

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "spi_tclDataServiceTypes.h"
#include "spi_tclLifeCycleIntf.h"
#include "spi_tclDataServiceRespIntf.h"
#include "spi_tclSelectionIntf.h"
#include "spi_tclExtRespPositionData.h"
#include "spi_tclExtRespSensorData.h"
#include "spi_tclExtRespNavData.h"
#include "spi_tclExtRespClock.h"
#include "spi_tclDataServiceSettingsIntf.h"

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/* Forward Declarations. */
class spi_tclDataServiceDevBase;

/*!
* \class spi_tclDataService
* \brief This is the main Data Service Manager class that provides interface
*        to delegate the execution of command and handle response.
*/
class spi_tclDataService : public spi_tclLifeCycleIntf, public spi_tclSelectionIntf, public spi_tclExtRespPositionData, public spi_tclExtRespSensorData,public spi_tclExtRespNavData, public spi_tclExtRespClock
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclDataService::spi_tclDataService(ahl_tclBaseOneThreadApp...)
   ***************************************************************************/
   /*!
   * \fn      spi_tclDataService(ahl_tclBaseOneThreadApp* poMainApp,
              spi_tclDataServiceRespIntf* poDataSrvcRespIntf)
   * \brief  Parametrized Constructor
   * \param  poMainApp : [IN] Pointer to main app.
   * \param  poDataSrvcRespIntf : [IN]Pointer to Data Service Response Interface
   ***************************************************************************/
   spi_tclDataService(ahl_tclBaseOneThreadApp* poMainApp,
      spi_tclDataServiceRespIntf* poDataSrvcRespIntf);

   /***************************************************************************
   ** FUNCTION:  spi_tclDataService::~spi_tclDataService();
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclDataService()
   * \brief   Virtual Destructor
   ***************************************************************************/
   virtual ~spi_tclDataService();

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDataService::bInitialize()
    ***************************************************************************/
   /*!
    * \fn      bInitialize()
    * \brief   Method to initialise the DataService manager.
   * \retval   t_Bool: TRUE - If service manager is initialised successfully, else FALSE.
    * \sa      bUnInitialize()
    **************************************************************************/
   virtual t_Bool bInitialize();

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDataService::bUnInitialize()
    ***************************************************************************/
   /*!
    * \fn      bUnInitialize()
   * \brief    Method to uninitialise the DataService manager
   * \retval   t_Bool: TRUE - If service manager is uninitialised successfully, else FALSE.
    * \sa      bInitialize()
    **************************************************************************/
   virtual t_Bool bUnInitialize();

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataService::vLoadSettings()
    ***************************************************************************/
   /*!
    * \fn      vLoadSettings()
    * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
    * \sa      vSaveSettings()
    **************************************************************************/
   virtual t_Void vLoadSettings();

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclDataService::vSaveSettings()
    ***************************************************************************/
   /*!
    * \fn      vSaveSettings()
    * \brief   vSaveSettings Method. Invoked during  NORMAL->OFF state transition.
    * \sa      vLoadSettings()
    **************************************************************************/
   virtual t_Void vSaveSettings();

   /***************************************************************************
    ** FUNCTION:  spi_tclDataService::vSelectDevice
    ***************************************************************************/
   /*!
    * \fn     vSelectDevice
    * \brief  Device selection request
    * \param  corfrSelectReq : Device selection request details
    **************************************************************************/
   virtual t_Void vSelectDevice(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
    ** FUNCTION:  spi_tclDataService::vSelectDeviceResult
    ***************************************************************************/
   /*!
    * \fn     vSelectDeviceResult
    * \brief  device selection result
    * \param  corfrSelectReq : Device selection result details
    * \param  enErrorCode : Error code
    **************************************************************************/
   t_Void vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode);


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vStartStopLocationData(...)
   ***************************************************************************/
   /*!
   * \fn      vStartStopLocationData()
   * \brief   Called when SelectDevice operation is complete & with the result
   *          of the operation.
   * \param   [IN] u32DeviceHandle: Unique handle of selected device
   * \param   [IN] enDeviceConnReq: Connection request type for the device
   * \param   [IN] enRespCode: Response code enumeration
   * \param   [IN] enErrorCode: Error code enumeration
   * \retval  None
   **************************************************************************/
   virtual t_Void vStartStopLocationData(t_Bool bStartLocData,
         const std::vector<tenNmeaSentenceType>& rfcoNmeaSentencesList,t_U8 u8RequestedBy = 0);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vOnGPSData(const trGPSData& rfrcGpsData)
   ***************************************************************************/
   /*!
   * \fn      vOnGPSData(const trGPSData& rfrcGpsData)
   * \brief   Method to receive GPS location data.
   * \param   rfrcGpsData: [IN] GPS data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnGPSData(const trGPSData& rfrcGpsData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vOnSensorData(const trSensorData& rfrcSensorData)
   ***************************************************************************/
   /*!
   * \fn      vOnSensorData(const trSensorData& rfrcSensorData)
   * \brief   Method to receive Sensor data.
   * \param   rfrcSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnSensorData(const trSensorData& rfrcSensorData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vOnAccSensorData
   ** (const std::vector<trAccSensorData>& corfvecrAccSensorData)
   ***************************************************************************/
   /*!
   * \fn      vOnAccSensorData(const std::vector<trAccSensorData>& corfvecrAccSensorData)
   * \brief   Method to receive Sensor data.
   * \param   corfvecrAccSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnAccSensorData(const std::vector<trAccSensorData>& corfvecrAccSensorData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vOnGyroSensorData
   ** (const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
   ***************************************************************************/
   /*!
   * \fn      vOnGyroSensorData(const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
   * \brief   Method to receive Sensor data.
   * \param   corfvecrGyroSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnGyroSensorData(const std::vector<trGyroSensorData>& corfvecrGyroSensorData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vOnVehicleData(const trVehicleData 
                                     rfrcVehicleData)
   ***************************************************************************/
   /*!
   * \fn      vOnVehicleData(const trVehicleData rfrcVehicleData)
   * \brief   Method to receive Vehicle data.
   * \param   rfrcVehicleData: [IN] Vehicle data
   * \param   bSolicited: [IN] True if the data update is for changed vehicle data, else False
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnVehicleData(const trVehicleData rfrcVehicleData, t_Bool bSolicited);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vSetLocDataAvailablility(t_Bool...)
   ***************************************************************************/
   /*!
   * \fn      vSetLocDataAvailablility(t_Bool bLocDataAvailable)
   * \brief   Interface to set the availability of LocationData
   * \param   [IN] bIsDeadReckonedData:
   *              True - if Location data is dead reckoned data, else False
   * \retval  None
   **************************************************************************/
   virtual t_Void vSetLocDataAvailablility(t_Bool bIsDeadReckonedData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDataService::vSimulateGPSDataTransfer(t_U32 u32DataRateInMs)
   ***************************************************************************/
   /*!
   * \fn      vSimulateGPSDataTransfer(t_U32 u32DataRateInMs)
   * \brief   Method to start simulation of GPS data. This triggers vOnGpsData()
   *          at the rate of u32DataRateInMs, until sample GPS data is completely sent.
   * \param   u32DataRateInMs: [IN] Rate at which sample GPS data is required.
   * \retval  None
   ***************************************************************************/
   t_Void vSimulateGPSDataTransfer(t_U32 u32DataRateInMs);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDataService::vSetRegion(...)
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetRegion(tenRegion enRegion)
   * \brief  Interface to set the current region
   * \param  [IN] enRegion : Region enumeration
   * \sa
   **************************************************************************/
   virtual t_Void vSetRegion(tenRegion enRegion);

   /***************************************************************************
   ** FUNCTION:  spi_tclDataService::vOnEnvironmentData()
   ***************************************************************************/
   /*!
   * \fn    t_Void vOnEnvironmentData(t_Bool bValidTempUpdate,t_Double dTemp,
   *                                   t_Bool bValidPressureUpdate, t_Double dPressure)
   * \brief Use this call to report Environment data to Phone
   * \param bValidTempUpdate : [IN] Temp update is valid
   * \param dTemp : [IN] Temp in Celsius
   * \param bValidPressureUpdate: [IN] Pressure update is valid
   * \param dPressure : [IN] Pressure in KPA
   * \retval t_Void
   **************************************************************************/
   t_Void vOnEnvironmentData(t_Bool bValidTempUpdate,t_Double dTemp,
      t_Bool bValidPressureUpdate, t_Double dPressure);

   /***************************************************************************
    ** FUNCTION:  t_Void vSetFeatureRestrictions()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetFeatureRestrictions(tenDeviceCategory enDevCategory,
    *          const t_U8 cou8ParkModeRestrictionInfo,
    *          const t_U8 cou8DriveModeRestrictionInfo)
    * \brief   Method to set Vehicle Park/Drive Mode Restriction
    * \param   enDevCategory : Device Category
    *          cou8ParkModeRestrictionInfo : Park mode restriction
    *          cou8DriveModeRestrictionInfo : Drive mode restriction
    * \retval  t_Void
    **************************************************************************/

   t_Void vSetFeatureRestrictions(tenDeviceCategory enDevCategory,
         const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo);

   /***************************************************************************
       ** FUNCTION:  t_Void vPostPositionDataResult()
       ***************************************************************************/
      /*!
       * \fn      vPostPositionDataResult(trDeadReckoningData rDeadReckoningData)
       * \brief   Method to post position data
       * \param   trDeadReckoningData : Structure for dead reckoning data
       * \retval  t_Void
       **************************************************************************/
   t_Void vPostPositionDataResult(trDeadReckoningData rDeadReckoningData);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataService::vPostSensorDataResult()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostSensorDataResult()
    * \brief   Called when Sensor data result is sent by Sensor Client Handler.
    *          Mandatory interface to be implemented.
    * \param   [IN]trSensorData:structure to hold sensor data
    * \sa      None
    **************************************************************************/
    t_Void vPostSensorDataResult(trSensorData rSensorData) ;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataService::vPostAccSensorDataResult()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostAccSensorDataResult()
    * \brief   Called when ACC Sensor data result is sent by Sensor Client Handler.
    *          Mandatory interface to be implemented.
    * \param   [IN]trAccSensorData:structure to hold ACC sensor data
    * \sa      None
    **************************************************************************/
   t_Void vPostAccSensorDataResult(const std::vector<trAccSensorData> &corfrAccSensorData );

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataService::vPostGyroSensorDataResult()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostGyroSensorDataResult()
    * \brief   Called when Gyro Sensor data result is sent by Sensor Client Handler.
    *          Mandatory interface to be implemented.
    * \param   [IN]trGyroSensorData:structure to hold Gyro sensor data
    * \sa      None
    **************************************************************************/
    t_Void vPostGyroSensorDataResult(const std::vector<trGyroSensorData> &corfrGyroSensorData );

    /***************************************************************************
     ** FUNCTION:  t_Void spi_tclExtRespNavData::vPostNavDataResult()
     ***************************************************************************/
    /*!
     * \fn      t_Void vPostNavDataResult()
     * \brief   Called when Nav Data data result is sent by Nav Data Client Handler.
     *          Mandatory interface to be implemented.
     * \param   [IN]vector<tenNmeaSentenceType>:vector to hold NMEA sentences
     * \sa      None
     **************************************************************************/
     t_Void vPostNavDataResult(t_Bool bStartLocData,const std::vector<tenNmeaSentenceType>& rfcoNmeaSentencesList,t_U8 u8RequestedBy);

     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDataService::vPostGPRMCDataStatusValues()
      ***************************************************************************/
     /*!
      * \fn      t_Void vPostGPRMCDataStatusValues()
      * \brief   Called when GPRMC Data Status Values are sent by Nav Data Client Handler.
      *          Mandatory interface to be implemented.
      * \param   [IN]u32DeviceId : Device ID
      * \param   [IN]u8GPRMCDataStatusValues : GPRMC Data Status Values i.e. A, V, X
      * \sa      None
      **************************************************************************/
      t_Void vPostGPRMCDataStatusValues(t_U32 u32DeviceId,t_U8 u8GPRMCDataStatusValues);

     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDataService::vSetGeoCoordinateSystem()
      ***************************************************************************/
     /*!
      * \fn      t_Void vSetGeoCoordinateSystem()
      * \brief   Testing function to set coordinate system
      * \param   [IN] enGeoCoordinateSystemType : Coordinate System
      * \sa      None
      **************************************************************************/
      t_Void vSetGeoCoordinateSystem(tenGeoCoordinateSystemType enGeoCoordinateSystemType);

     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDataService::vSetDataServiceSettingsInstance(spi_tclDataServiceSettingsIntf* poDataServiceSettingsIntf)
      ***************************************************************************/
     /*!
      * \fn      t_Void vSetDataServiceSettingsInstance()
      * \brief   Method to set dataservice settings instance.
      * \param   poDataServiceSettingsIntf  : pointer to DataServiceSettingsIntf.
      * \retval  t_Void
      **************************************************************************/
      t_Void vSetDataServiceSettingsInstance(spi_tclDataServiceSettingsIntf* poDataServiceSettingsIntf);
      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDataService::vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory,t_U16 u16GeneralRestrInfo)
       ***************************************************************************/
       /*!
        * \fn      t_Void vSetGeneralRestrictions()
        * \brief   Method to set general restrictions
        * \param
        * \retval  t_Void
        **************************************************************************/
       t_Void vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory,t_U16 u16GeneralRestrInfo);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDataService::vSetETCStatus()
        ***************************************************************************/
       /*!
        * \fn      t_Void vSetETCStatus(t_Bool bETCStatus)
        * \brief   Method to set ETC Status
        * \param   bETCStatus: [IN] ETC Status (Enabled or Disabled)
        * \retval  t_Void
        **************************************************************************/
       t_Void vSetETCStatus(t_Bool bETCStatus);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDataService::vOnClockData(const t_U8 cou8TimeFormat)
        ***************************************************************************/
       /*!
        * \fn      vOnClockData(const t_U8 cou8TimeFormat)
        * \brief   Method to receive Clock data.
        * \param   cou8TimeFormat: [IN] Time Format
        * \retval  t_Void
        **************************************************************************/
       t_Void vOnClockData(const t_U8 cou8TimeFormat);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDataService::vPostClockResult()
        ***************************************************************************/
       /*!
        * \fn     t_Void vPostClockResult()
        * \brief  Called when Clock data result is sent by Clock Client Handler.
        *         Mandatory interface to be implemented.
        * \param  [IN]cou8TimeFormat:Time Format of the clock
        * \retval  t_Void
        ***************************************************************************/
       t_Void vPostClockResult(const t_U8 cou8TimeFormat);
       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDataService::vLanguageSetting()
        ***************************************************************************/
       /*!
        * \fn     t_Void vLanguageSetting(tenLanguageSetting enLanguageSetting)
        * \brief  Called when languague setting is sent from HMI
        * \param  tenLanguageSetting enLanguageSetting
        * \retval  t_Void
        ***************************************************************************/
       t_Void vOnSendLanguageSetting(tenLanguageSetting enLanguageSetting);

	   /***************************************************************************
	   ** FUNCTION:  t_Void spi_tclDataService::vGetDataServiceConfig()
	   ***************************************************************************/
	   /*!
	   * \fn      t_Void vGetDataServiceConfig()
	   * \brief   Method to print the Data Service configuration to read KDS
	   * \param   None
	   * \retval  t_Void
	   **************************************************************************/
	   t_Void vGetDataServiceConfig();

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDataService::vOnVehiclePowerModeState(tenVehiclePowerMode enVehiclePowerModeState)
       ***************************************************************************/
       /*!
       * \fn      vOnVehiclePowerModeState(tenVehiclePowerMode enVehiclePowerModeState)
       * \brief   Method to receive Vehicle Power Mode state
       * \param   enVehiclePowerModeState: [IN] Vehicle Power Mode state
       * \retval  None
       ***************************************************************************/
       virtual t_Void vOnVehiclePowerModeState(tenVehiclePowerMode enVehiclePowerModeState);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDataService::vOnAmbientColorData(const trAmbientColorData& corfrAmbientColorData)
       ***************************************************************************/
       /*!
       * \fn      vOnAmbientColorData(const trAmbientColorData& corfrAmbientColorData)
       * \brief   Method to receive Ambient color data
       * \param   corfrAmbientColorData: [IN] Ambient color Data
       * \retval  None
       ***************************************************************************/
       virtual t_Void vOnAmbientColorData(const trAmbientColorData& corfrAmbientColorData);

		/***************************************************************************
        ** FUNCTION:  t_Void spi_tclDataService::vDistanceToEmptyTank()
        ***************************************************************************/
       /*!
        * \fn     t_Void vDistanceToEmptyTank()
        * \brief  Used to indicate distance to empty tank
        * \param  [IN]s32DistToEmptyTank: Distance to empty tank
        * \retval  t_Void
        ***************************************************************************/
       t_Void vDistanceToEmptyTank(t_S32 s32DistToEmptyTank);

private:
   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataService::getDataServiceIndex(...)
    ***************************************************************************/
   /*!
   * \fn      getDataServiceIndex(tenDeviceCategory);
   * \brief   Used to get data service index
   * \param   [IN] enDataCategory : Category of device
   * \retval  tenDataServicesIndex : Data service index returned
   **************************************************************************/
   tenDataServicesIndex enGetDataServiceIndex(tenDeviceCategory enDataCategory);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataService::vOnSubscribeForData(t_Bool...)
    ***************************************************************************/
   /*!
   * \fn      vOnSubscribeForData(t_Bool bSubscribe, tenSensorDataType enDataType)
   * \brief   Subscribes/Unsubscribes for data notifications from
   *              service-provider (such as POS_FI).
   * \param   [IN] bSubscribe:
   *               TRUE - To subscribe for data notifications.
   *               FALSE - To unsubscribe for data notifications.
   * \param   [IN] enDataType: Type of data to be subscribed/unsubscribed
   * \retval  None
   **************************************************************************/
   t_Void vOnSubscribeForData(t_Bool bSubscribe, tenSensorDataType enDataType);

   /***************************************************************************
   ** FUNCTION:  spi_tclDataService::spi_tclDataService()
   ***************************************************************************/
   /*!
   * \fn     spi_tclDataService()
   * \brief  Default Constructor, not implemented.
   ***************************************************************************/
   spi_tclDataService();

   /***************************************************************************
   ** FUNCTION:  spi_tclDataService::operator=(const spi_tclDataService&...)
   ***************************************************************************/
   /*!
   * \fn     operator=(const spi_tclDataService& rfcooDataService)
   * \brief  Assignment operator, not implemented.
   ***************************************************************************/
   spi_tclDataService& operator=(const spi_tclDataService& rfcooDataService);


   /***************************************************************************
   * ! Data members
   ***************************************************************************/

   /***************************************************************************
   ** Main application pointer
   ***************************************************************************/
   ahl_tclBaseOneThreadApp*         m_poMainApp;


   /***************************************************************************
   ** List of pointers to DataService Handlers of different device types
   ***************************************************************************/
   spi_tclDataServiceDevBase*       m_poDataSvcHandlersLst[e8_DATASVC_INDEX_SIZE];

   /***************************************************************************
   ** Category type of selected device
   ***************************************************************************/
   tenDeviceCategory                m_enSelDevCategory;

   /***************************************************************************
   ** Data Service Response Interface Pointer
   ***************************************************************************/
   spi_tclDataServiceRespIntf*      m_poDataSrvcRespIntf;

   //! Data Service Settings Interface Pointer
   spi_tclDataServiceSettingsIntf* m_poDataServiceSettings;
};

#endif // _SPI_TCLDATASERVICE_H

