/*!
 *******************************************************************************
 * \file             spi_tclTechnologySelector.cpp
 * \brief            Handles legal disclaimer requirements for SPI technology
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Handles legal disclaimer requirements for SPI technology
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 28.10.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclTechnologySelector.h"
#include "spi_tclMediatorIntf.h"
#include "spi_tclDeviceListIntf.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclTechnologySelector.cpp.trc.h"
#endif
#endif

static const trDeviceUsageEnabledConfig scouDeviceUsageEnabledConfig[] = {{e8USAGE_DISABLED,   e8USAGE_DISABLED,   e8USAGE_DISABLED},
                                                                          {e8USAGE_DISABLED,   e8USAGE_ENABLED,    e8USAGE_DISABLED},
                                                                          {e8USAGE_DISABLED,   e8USAGE_CONF_REQD,  e8USAGE_DISABLED},
                                                                          {e8USAGE_ENABLED,    e8USAGE_DISABLED,   e8USAGE_DISABLED},
                                                                          {e8USAGE_ENABLED,    e8USAGE_ENABLED,    e8USAGE_ENABLED},
                                                                          {e8USAGE_ENABLED,    e8USAGE_CONF_REQD,  e8USAGE_CONF_REQD},
                                                                          {e8USAGE_CONF_REQD,  e8USAGE_DISABLED,   e8USAGE_DISABLED},
                                                                          {e8USAGE_CONF_REQD,  e8USAGE_ENABLED,    e8USAGE_ENABLED},
                                                                          {e8USAGE_CONF_REQD,  e8USAGE_CONF_REQD,  e8USAGE_CONF_REQD}};
/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclTechnologySelector::spi_tclTechnologySelector
 ***************************************************************************/
spi_tclTechnologySelector::spi_tclTechnologySelector(spi_tclDeviceListIntf* poDeviceList, spi_tclTechnologySelectorResp* pResp, spi_tclMediatorIntf* poMediator) :
         spi_tclSelectionIntf(e32COMPID_TECHNOLOGYSELECTOR), m_pDeviceList(poDeviceList), m_pTechSelectorResp(pResp), m_pMediator(poMediator),
         m_bIsSelectRequestPending(false)
{
   ETG_TRACE_USR1((" spi_tclTechnologySelector::spi_tclTechnologySelector() entered \n"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclTechnologySelector::~spi_tclTechnologySelector
 ***************************************************************************/

spi_tclTechnologySelector::~spi_tclTechnologySelector()
{
   ETG_TRACE_USR1((" spi_tclTechnologySelector::~spi_tclTechnologySelector() entered \n"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclTechnologySelector::vSetTechnologySelection
 ***************************************************************************/

t_Void spi_tclTechnologySelector::vSetTechnologySelection(const t_U32 cou32DeviceHandle, tenDeviceCategory enTechnologyPreference)
{
   ETG_TRACE_USR2(("[DESC] spi_tclTechnologySelector::vSetTechnologySelection:Received request to set the Technology preference status to %d  for DeviceHandle = 0x%x \n",
            ETG_ENUM(DEVICE_CATEGORY,enTechnologyPreference), cou32DeviceHandle));

   if ((true == m_bIsSelectRequestPending) && (NULL != m_pDeviceList))
   {
      //Adding device to the DeviceList irrespective of the TechnologyPreference
      m_pDeviceList->vSetTechnologySelection(cou32DeviceHandle,enTechnologyPreference);
      if (e8DEV_TYPE_DEFAULT == enTechnologyPreference)
      {
         //! Device selection is not required if user has chosen to use the device in default mode
         vSendSelectDeviceResponse(e8OPERATION_CANCELLED_BY_USER);
         vUpdateTechnologySelection(cou32DeviceHandle, enTechnologyPreference);
      }
      else
      {
         trDeviceInfo rDeviceInfo;
         t_Bool bValidDevice = m_pDeviceList->bGetDeviceInfo(cou32DeviceHandle, rDeviceInfo);

         vUpdateTechnologySelection(cou32DeviceHandle, enTechnologyPreference);

         tenEnabledInfo enSPISetting = e8USAGE_UNKNOWN;
         //m_pDeviceList->bGetDevProjUsage(enTechnologyPreference, enSPISetting);
         enSPISetting = enGetEffectiveTechnologyPreference(cou32DeviceHandle);

         if ((true == bValidDevice) && (e8USAGE_CONF_REQD == enSPISetting))
         {
            vSendSelectDeviceResponse(e8NO_ERRORS);
         }
      }
      m_bIsSelectRequestPending = false;
   }
   else
   {
      ETG_TRACE_ERR(("[DESC] spi_tclTechnologySelector::vSetTechnologySelection(): Invalid: Received when selection is not in progress \n"));
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclTechnologySelector::vSelectDevice
 ***************************************************************************/

t_Void spi_tclTechnologySelector::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("[DESC] spi_tclTechnologySelector::vSelectDevice(): Selection request received by Technology Selector \n"));
   //! Proceed with selection if authorization is not required else wait for HMI response
   //this will be the overall setting for the logic
   tenEnabledInfo enEffectiveSetting = enGetEffectiveTechnologyPreference(corfrSelectReq.m_u32DeviceHandle);
   switch (enEffectiveSetting)
   {
      case e8USAGE_CONF_REQD:
      {
         m_bIsSelectRequestPending = true;
         vUpdateTechnologySelection(corfrSelectReq.m_u32DeviceHandle, e8DEV_TYPE_UNKNOWN);
      }
         break;

      case e8USAGE_ENABLED:
      {
         vSendSelectDeviceResponse(e8NO_ERRORS);
      }
         break;

      case e8USAGE_DISABLED:
      {
         vSendSelectDeviceResponse(e8PROJECTION_NOT_ENABLED);
      }
         break;

      default:
         ETG_TRACE_ERR(("[ERR]:: vSelectDevice Invalid Enabled info Type! "));
         vSendSelectDeviceResponse(e8OPERATION_REJECTED);
         break;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclTechnologySelector::vSelectDeviceResult
 ***************************************************************************/

t_Void spi_tclTechnologySelector::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode)
{
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   ETG_TRACE_USR1(("[DESC] spi_tclTechnologySelector::vSelectDeviceResult() : Selection result received by Technology Selector \n"));
   if(e8DEVICE_DISCONNECTED_DURING_SELECTION == enErrorCode)
   {
      vUpdateTechnologySelection(0xFFFFFFFF, e8DEV_TYPE_UNKNOWN);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclTechnologySelector::vSendSelectDeviceResponse
 ***************************************************************************/

t_Void spi_tclTechnologySelector::vSendSelectDeviceResponse(tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("[DESC] spi_tclTechnologySelector::vSendSelectDeviceResponse() : Sending select device response to device selector \n"));
   if(NULL != m_pMediator)
   {
      m_pMediator->vPostSelectDeviceRes(enGetComponentID(), enErrorCode);
   }
}


/***************************************************************************
 ** FUNCTION:  spi_tclTechnologySelector::vUpdateTechnologySelection
 ***************************************************************************/
t_Void spi_tclTechnologySelector::vUpdateTechnologySelection(const t_U32 cou32DeviceHandle, tenDeviceCategory enTechnologyPreference)
{
   ETG_TRACE_USR1(("[DESC] spi_tclTechnologySelector::vUpdateTechnologySelection() Updating technology selection status for Device 0x%x \n", cou32DeviceHandle));

   tenDeviceType enDevType = e8_UNKNOWN_DEVICE;
   if (NULL != m_pDeviceList)
   {
      enDevType = m_pDeviceList->enGetDeviceType(cou32DeviceHandle);
      if((e8DEV_TYPE_UNKNOWN != enTechnologyPreference) && (e8DEV_TYPE_DEFAULT != enTechnologyPreference))
      {
         //setting the updated technology selected by HMI on the pop up
         ETG_TRACE_USR1(("spi_tclTechnologySelector::vUpdateTechnologySelection : updating the Device list with the Technology selected by HMI"));
         m_pDeviceList->vSetDeviceCategory(cou32DeviceHandle, enTechnologyPreference);
      }
   }
   if(NULL != m_pTechSelectorResp)
   {
      m_pTechSelectorResp->vUpdateTechnologySelectionStatus(cou32DeviceHandle,enDevType, enTechnologyPreference);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclTechnologySelector::enGetEffectiveTechnologyPreference
 ***************************************************************************/
tenEnabledInfo spi_tclTechnologySelector::enGetEffectiveTechnologyPreference(const t_U32 u32DeviceHandle)
{
   tenDeviceCategory enDeviceCategory = e8DEV_TYPE_UNKNOWN;
   tenEnabledInfo enOverallSetting = e8USAGE_DISABLED;   
   tenEnabledInfo enDeviceSetting = e8USAGE_UNKNOWN;
   tenEnabledInfo enEffectiveSetting = e8USAGE_UNKNOWN;   
   tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE;

   if(NULL != m_pDeviceList)
   {
      enDeviceCategory = m_pDeviceList->enGetDeviceCategory(u32DeviceHandle);
      enDeviceType = m_pDeviceList->enGetDeviceType(u32DeviceHandle);
      m_pDeviceList->bGetDevProjUsage(enDeviceCategory, enOverallSetting, enDeviceType);
      m_pDeviceList->vGetDeviceUsagePreference(u32DeviceHandle, enDeviceSetting);
   }

   t_U32 u32Size = (sizeof(scouDeviceUsageEnabledConfig)) / (sizeof(trDeviceUsageEnabledConfig));

   for (t_U8 u8Index = 0; u8Index < u32Size; u8Index++)
   {
      if ((enOverallSetting == scouDeviceUsageEnabledConfig[u8Index].enOverallSetting) && (enDeviceSetting
                  == scouDeviceUsageEnabledConfig[u8Index].enDeviceSetting))
      {
         enEffectiveSetting = scouDeviceUsageEnabledConfig[u8Index].enEffectiveSetting;
            ETG_TRACE_USR1(("[DESC] spi_tclTechnologySelector::enGetEffectiveTechnologyPreference(): Overall setting for the device is %d", ETG_ENUM(ENABLED_INFO,
                     enOverallSetting)));
         ETG_TRACE_USR1(("[DESC] spi_tclTechnologySelector::enGetEffectiveTechnologyPreference(): Device setting for the device is %d", ETG_ENUM(ENABLED_INFO,
                     enDeviceSetting)));
         ETG_TRACE_USR1(("[DESC] spi_tclTechnologySelector::enGetEffectiveTechnologyPreference(): Effective setting for the device is %d", ETG_ENUM(ENABLED_INFO,
                     enEffectiveSetting)));
      }
   }
   return enEffectiveSetting;
}

//end
