/*!
 *******************************************************************************
 * \file             spi_tclSelectionValidator.cpp
 * \brief            Handles validating device selection request
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Handles validating device selection request
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 28.10.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclSelectionValidator.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclSelectionValidator.cpp.trc.h"
#endif
#endif

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclSelectionValidator::spi_tclSelectionValidator
 ***************************************************************************/
spi_tclSelectionValidator::spi_tclSelectionValidator(spi_tclDeviceListIntf* poDeviceList,
          const t_U32 cou32SelectedDevice, const t_Bool cobBusyStatus) : m_pDeviceList(poDeviceList),
          m_u32CurrSelectedDevice(cou32SelectedDevice), m_bisDeviceSelectorBusy(cobBusyStatus)
{
   ETG_TRACE_USR1((" spi_tclSelectionValidator::spi_tclSelectionValidator() entered "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclSelectionValidator::~spi_tclSelectionValidator
 ***************************************************************************/

spi_tclSelectionValidator::~spi_tclSelectionValidator()
{
   ETG_TRACE_USR1((" spi_tclSelectionValidator::~spi_tclSelectionValidator() entered "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclSelectionValidator::vValidateSelection
 ***************************************************************************/

t_Void spi_tclSelectionValidator::vValidateSelection(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode &rfenErrorCode)
{
   ETG_TRACE_USR1(("[PARAM] spi_tclDeviceSelector::vValidateSelection cou32DeviceHandle = 0x%x,"
            " enDevConnReq = %d , Device category = %d SelectReason = %d",
            corfrSelectReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ, corfrSelectReq.m_enDevConnReq),
            ETG_ENUM(DEVICE_CATEGORY, corfrSelectReq.m_enDevCategory),
            ETG_ENUM(SELECT_REASON,corfrSelectReq.m_enSelectionReason)));

   //! If device category is not known, validate based on device type
   t_Bool bProjectionEnabled = (e8DEV_TYPE_UNKNOWN == corfrSelectReq.m_enDevCategory) ?
            bIsProjectionEnabled(corfrSelectReq.m_enDeviceType):
            bIsProjectionEnabled(corfrSelectReq.m_enDevCategory, corfrSelectReq.m_enDeviceType);

   //! Reject selection if projection is not enabled for the device
   if(false == bProjectionEnabled)
   {
      rfenErrorCode = e8OPERATION_REJECTED;
   }

   //! Check the validity of the device for selection
   bValidateDeviceID(corfrSelectReq,rfenErrorCode);

   //! Check the validaity of the selection request
   if(e8NO_ERRORS == rfenErrorCode)
   {
      bValidateSelectDeviceRequest(corfrSelectReq,rfenErrorCode);
   }

   ETG_TRACE_USR1(("[DESC] spi_tclSelectionValidator::vValidateSelection() : Error =%d Validation of device selection request "
            " %s",ETG_ENUM(ERROR_CODE,rfenErrorCode), (e8NO_ERRORS == rfenErrorCode)? "Succeeded" : "Failed"));
}


/***************************************************************************
 ** FUNCTION:  spi_tclSelectionValidator::bIsProjectionEnabled
 ***************************************************************************/
t_Bool spi_tclSelectionValidator::bIsProjectionEnabled(tenDeviceType enDeviceType)
{
   t_Bool bIsProjectionEnabled = false;

   if (NULL != m_pDeviceList)
   {
      if (e8_ANDROID_DEVICE == enDeviceType)
      {
         tenEnabledInfo enAndroidAutoEnabled = e8USAGE_ENABLED;
         tenEnabledInfo enMirrorlinkEnabled = e8USAGE_ENABLED;
         tenEnabledInfo enMySPINEnabled = e8USAGE_ENABLED;
         tenEnabledInfo enCarlifeEnabled = e8USAGE_ENABLED;
         tenEnabledInfo enOnCarEnabled = e8USAGE_ENABLED;
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_ANDROIDAUTO, enAndroidAutoEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_MIRRORLINK, enMirrorlinkEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_MYSPIN, enMySPINEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_CARLIFE, enCarlifeEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_ONCAR, enOnCarEnabled, enDeviceType);
         bIsProjectionEnabled = ((e8USAGE_DISABLED != enMirrorlinkEnabled)
                  || (e8USAGE_DISABLED != enAndroidAutoEnabled)
                  || (e8USAGE_DISABLED != enMySPINEnabled)
                  || (e8USAGE_DISABLED != enCarlifeEnabled)
                  || (e8USAGE_DISABLED != enOnCarEnabled));
      }
      else if (e8_APPLE_DEVICE == enDeviceType)
      {
         tenEnabledInfo enCarplayEnabled = e8USAGE_ENABLED;
         tenEnabledInfo enMySPINEnabled = e8USAGE_ENABLED;
         tenEnabledInfo enCarlifeEnabled = e8USAGE_ENABLED;
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_DIPO, enCarplayEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_MYSPIN, enMySPINEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_CARLIFE, enCarlifeEnabled, enDeviceType);
         bIsProjectionEnabled = ((e8USAGE_DISABLED != enCarplayEnabled)
                  || (e8USAGE_DISABLED != enMySPINEnabled)
                  || (e8USAGE_DISABLED != enCarlifeEnabled));
      }
      else
      {
         //! unknown device type Ex: Windows/Symbian phone
         tenEnabledInfo enMirrorlinkEnabled = e8USAGE_ENABLED;
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_MIRRORLINK, enMirrorlinkEnabled, enDeviceType);
         bIsProjectionEnabled = (e8USAGE_DISABLED != enMirrorlinkEnabled);
      }
   }
   ETG_TRACE_USR1(("[DESC] spi_tclSelectionValidator::bIsProjectionEnabled() for DeviceType %d Retval = %d ",
            ETG_ENUM(DEVICE_TYPE, enDeviceType), ETG_ENUM(BOOL, bIsProjectionEnabled)));
   return bIsProjectionEnabled;
}
/***************************************************************************
 ** FUNCTION:  spi_tclSelectionValidator::bIsProjectionEnabled
 ***************************************************************************/
t_Bool spi_tclSelectionValidator::bIsProjectionEnabled(tenDeviceCategory enDevCat, tenDeviceType enDeviceType)
{
   t_Bool bIsProjectionEnabled = false;
   tenEnabledInfo enSPIEnabled = e8USAGE_ENABLED;
   if (NULL != m_pDeviceList)
   {
      m_pDeviceList->bGetDevProjUsage(enDevCat, enSPIEnabled, enDeviceType);
   }

   bIsProjectionEnabled = (e8USAGE_DISABLED != enSPIEnabled);

   ETG_TRACE_USR1(("[DESC] spi_tclSelectionValidator::bIsProjectionEnabled() for DeviceCategory %d Retval = %d ",
            ETG_ENUM(DEVICE_CATEGORY, enDevCat), ETG_ENUM(BOOL, bIsProjectionEnabled)));

   return bIsProjectionEnabled;
}

/***************************************************************************
 ** FUNCTION:  spi_tclSelectionValidator::bValidateSelectDeviceRequest
 ***************************************************************************/
t_Bool spi_tclSelectionValidator::bValidateSelectDeviceRequest(const trSelectDeviceRequest &corfrSelDevReq,
         tenErrorCode &rfenErrorCode)
{
   ETG_TRACE_USR1((" spi_tclSelectionValidator::bValidateSelectDeviceRequest() entered "));
   t_Bool bValidRequest = false;
   trDeviceSelectionInfo rDeviceSelectioninfo;

   if (NULL != m_pDeviceList)
   {
      m_pDeviceList->vGetDeviceSelectionInfo(corfrSelDevReq.m_u32DeviceHandle, rDeviceSelectioninfo);
   }

   //! Check if Device selector is busy with another device selection request
   if (true == bDeviceSelectorBusy())
   {
      ETG_TRACE_ERR(("[ERROR]  spi_tclSelectionValidator::bValidateSelectDeviceRequest(): Device selection or deselection is already in progress  "));
      rfenErrorCode = e8RESOURCE_BUSY;
   }//if (true == bDeviceSelectorBusy())

   //! If the device is re-enumerating, then Select device request will be rejected
   else if((e8DEVCONNREQ_SELECT == corfrSelDevReq.m_enDevConnReq) && (e8_SELECTION_STATE_DEVICE_REENUMERATION_INPROGRESS == rDeviceSelectioninfo.enDeviceSelectionState))
   {
      ETG_TRACE_ERR(("[ERROR]  spi_tclSelectionValidator::bValidateSelectDeviceRequest(): Device is re-enumerating, Hence rejecting the request "));
      rfenErrorCode = e8UNSUPPORTED_OPERATION;
   }

   //! SELECT request from HMI for CPW session establishment is UNSUPPORTED Operation- As it always Automatic Selection
   else if((e8DEVCONNREQ_SELECT == corfrSelDevReq.m_enDevConnReq)
            && ( e8WIRELESS_CONNECTED == corfrSelDevReq.m_enDevConnType) &&
            ((e8_REASON_HMI_TRIGGER == corfrSelDevReq.m_enSelectionReason ) || (e8_REASON_USER_TRIGGER == corfrSelDevReq.m_enSelectionReason )))
   {
      ETG_TRACE_USR2(("[DESC] spi_tclSelectionValidator::bValidateSelectDeviceRequest() :SPI will not process the SELECT request from HMI for CPW session establishment as "
               "for CPW Session establishment is always Automatic "));
      rfenErrorCode = e8UNSUPPORTED_OPERATION;
   }
   //! Check if the device is already selected
   else if ((corfrSelDevReq.m_u32DeviceHandle == u32GetCurrSelectedDevice())
            && (e8DEVCONNREQ_SELECT == corfrSelDevReq.m_enDevConnReq))
   {
      if((NULL != m_pDeviceList) &&
               (corfrSelDevReq.m_enDevCategory != m_pDeviceList->enGetDeviceCategory(corfrSelDevReq.m_u32DeviceHandle)))
     {
         ETG_TRACE_ERR(("[ERROR] spi_tclSelectionValidator::bValidateSelectDeviceRequest() : Invalid request. Device already selected for different category "));
         rfenErrorCode = e8INVALID_DEV_HANDLE;
     }
     else
     {
        ETG_TRACE_USR2(("[DESC] spi_tclSelectionValidator::bValidateSelectDeviceRequest() :Device already selected. No action required "));
         rfenErrorCode = e8UNSUPPORTED_OPERATION;
     }
   }
   else if(e8DEVCONNREQ_DESELECT == corfrSelDevReq.m_enDevConnReq)
   {
	  //! Check if the device is already deselected
      if(corfrSelDevReq.m_u32DeviceHandle != u32GetCurrSelectedDevice())
      {
         ETG_TRACE_USR2(("[DESC] spi_tclSelectionValidator::bValidateSelectDeviceRequest() :Device already deselected. No action required "));
		 rfenErrorCode = e8UNSUPPORTED_OPERATION;
      }
	  //This is for A+T Box Factoy reset - CRQ 324734 - AIVI-70844  : INT CRQ A+T Box shall available after restore factory setting
      else if((corfrSelDevReq.m_enSelectionReason == e8_FACTORY_RESET) && (corfrSelDevReq.m_enDevCategory == e8DEV_TYPE_MIRRORLINK))
      {
         rfenErrorCode = (bIsDeviceWhiteListed(corfrSelDevReq.m_u32DeviceHandle))? e8OPERATION_REJECTED: e8NO_ERRORS;
      }
   }

   else
   {
      bValidRequest = true;
   }

   if (e8NO_ERRORS != rfenErrorCode)
   {
      ETG_TRACE_ERR(("[ERROR] Invalid Device Selection Request  for DeviceHandle = %d Errorcode = %d",
               corfrSelDevReq.m_u32DeviceHandle,ETG_ENUM(ERROR_CODE, rfenErrorCode)));
   } // if (e8NO_ERRORS != enErrorCode)
   return bValidRequest;
}

/***************************************************************************
 ** FUNCTION:  spi_tclSelectionValidator::bValidateDeviceID
 ***************************************************************************/
t_Bool spi_tclSelectionValidator::bValidateDeviceID(const trSelectDeviceRequest &corfrSelDevReq, tenErrorCode &rfenErrorCode)
{
   t_Bool bValidRequest = false;

   tenUSBPortType enUSBPortType = e8_PORT_TYPE_NOT_KNOWN;
   tenSPISupport enCarplaySupport = e8SPI_SUPPORT_UNKNOWN;
   tenSPISupport enCarlifeSupport = e8SPI_SUPPORT_UNKNOWN;
   //! Get the connection status and device validity for cou32DeviceHandle
   tenDeviceConnectionStatus enConnStatus = e8DEV_NOT_CONNECTED;
   t_Bool bValidDevice = false;
   if (NULL != m_pDeviceList)
   {
      enUSBPortType = m_pDeviceList->enGetUSBPortType(corfrSelDevReq.m_u32DeviceHandle);
      enConnStatus = m_pDeviceList->enGetDeviceConnStatus(corfrSelDevReq.m_u32DeviceHandle);
      bValidDevice = m_pDeviceList->bIsDeviceValid(corfrSelDevReq.m_u32DeviceHandle);
      enCarplaySupport = m_pDeviceList->enGetSPISupport(corfrSelDevReq.m_u32DeviceHandle, e8DEV_TYPE_DIPO);
      enCarlifeSupport = m_pDeviceList->enGetSPISupport(corfrSelDevReq.m_u32DeviceHandle, e8DEV_TYPE_CARLIFE);
   } //if (NULL != m_pDeviceList)

   ETG_TRACE_USR2(("[DESC] spi_tclSelectionValidator::bValidateDeviceID() :CarplaySupport = %d and CarlifeSupport = %d Device category = %d ",ETG_ENUM(SPI_SUPPORT,enCarplaySupport),ETG_ENUM(SPI_SUPPORT, enCarlifeSupport), ETG_ENUM(DEVICE_CATEGORY, corfrSelDevReq.m_enDevCategory)));
   //! Check if the device handle is valid
   if ((0 == corfrSelDevReq.m_u32DeviceHandle) || (false == bValidDevice))
   {
      ETG_TRACE_ERR(("[ERROR]  spi_tclSelectionValidator::bValidateDeviceID(): Unknown Device ID (0x%x)",corfrSelDevReq.m_u32DeviceHandle ));
      rfenErrorCode = e8INVALID_DEV_HANDLE;
   } // if (0 == corfrSelDevReq.m_u32DeviceHandle)

   //! Check if apple device is connected to OTG port and it supports carplay (carplay support is advertised only above iOS 9).
   else if((e8_APPLE_DEVICE == corfrSelDevReq.m_enDeviceType) && ((enUSBPortType != e8_PORT_TYPE_OTG) || (((e8SPI_SUPPORTED != enCarplaySupport)&&(corfrSelDevReq.m_enDevCategory == e8DEV_TYPE_DIPO ))
            ||((e8SPI_SUPPORTED != enCarlifeSupport)&&(corfrSelDevReq.m_enDevCategory == e8DEV_TYPE_CARLIFE )))))
   {

      ETG_TRACE_ERR(("[ERROR]  spi_tclSelectionValidator::bValidateDeviceID(): Device does not support SPI (0x%x) ",corfrSelDevReq.m_u32DeviceHandle ));
      rfenErrorCode = e8OPERATION_REJECTED;
   }
   //! Check if the selected device is connected
   else if (((e8DEV_NOT_CONNECTED == enConnStatus)) && (e8DEVCONNREQ_SELECT
            == corfrSelDevReq.m_enDevConnReq))
   {
      ETG_TRACE_ERR(("[ERROR]  spi_tclSelectionValidator::bValidateDeviceID(): Device not connected (0x%x)",corfrSelDevReq.m_u32DeviceHandle ));
      rfenErrorCode = e8DEVICE_NOT_CONNECTED;
   } // if ((e8DEV_NOT_CONNECTED == enConnStatus) && ...

   else
   {
      bValidRequest = true;
   }

   ETG_TRACE_USR2(("[DESC]spi_tclSelectionValidator::bValidateDeviceID: Device selection request validation result = %d  ",
            ETG_ENUM(BOOL,bValidRequest)));
   return bValidRequest;
}

/***************************************************************************
 ** FUNCTION:  spi_tclSelectionValidator::bDeviceValidforAutoSelection
 ***************************************************************************/
t_Bool spi_tclSelectionValidator::bDeviceValidforAutoSelection(const trSelectDeviceRequest &corfrSelDevReq)
{
   tenErrorCode enErrorCode = e8NO_ERRORS;
   vValidateSelection(corfrSelDevReq, enErrorCode);
   t_Bool bValidRequest = (e8NO_ERRORS == enErrorCode);

   if ((NULL != m_pDeviceList) && (true == bValidRequest) )
   {
      t_Bool bUserDeselect = false;
      tenSessionTransport enLastActiveSessionTransport = m_pDeviceList->enGetLastActiveSessionTransport(corfrSelDevReq.m_u32DeviceHandle);
      tenEnabledInfo enEnabledInfo = e8USAGE_ENABLED;
      m_pDeviceList->vGetDeviceUsagePreference(corfrSelDevReq.m_u32DeviceHandle, enEnabledInfo);
      t_Bool bIsSelectError = m_pDeviceList->bIsSelectError(corfrSelDevReq.m_u32DeviceHandle);

      //! Use userDeslect flag for AAP or CPW auto selection. For CP auto-select always.
      t_Bool bIsCPWReconnected = ((e8_APPLE_DEVICE == corfrSelDevReq.m_enDeviceType) &&
                                  (e8WIRELESS_CONNECTED == corfrSelDevReq.m_enDevConnType) &&
                                  (e8_SESSION_TRANSPORT_WIFI == enLastActiveSessionTransport));
      if(bIsCPWReconnected || (e8_ANDROID_DEVICE == corfrSelDevReq.m_enDeviceType))
      {
         bUserDeselect = m_pDeviceList->bGetUserDeselectionFlag(corfrSelDevReq.m_u32DeviceHandle);
      }
      //! For CPW - Error count should not considered for Auto Selection.
      bValidRequest = ((false == bUserDeselect) && (e8USAGE_DISABLED != enEnabledInfo) && ((false == bIsSelectError)||(e8WIRELESS_CONNECTED == corfrSelDevReq.m_enDevConnType)) && (0 == m_u32CurrSelectedDevice));
   }
   ETG_TRACE_USR2(("[DESC] spi_tclSelectionValidator::bDeviceValidforAutoSelection:  result = %d  ",
            ETG_ENUM(BOOL,bValidRequest)));
   return bValidRequest;
}

/***************************************************************************
 ** FUNCTION:  spi_tclSelectionValidator::bIsDeviceWhiteListed()
 ***************************************************************************/
t_Bool spi_tclSelectionValidator::bIsDeviceWhiteListed(const t_U32 cou32DeviceHandle)
{
   t_Bool bIsDeviceWhitelist = false;

   if(NULL != m_pDeviceList)
   {
      bIsDeviceWhitelist = m_pDeviceList->bGetDeviceWhitelistedInfo(cou32DeviceHandle);
   }
   return bIsDeviceWhitelist;
}
