/*!
 *******************************************************************************
 * \file             spi_tclDeviceListHandler.cpp
 * \brief            Handles Device List
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Handles SPI Devices list
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 11.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 25.01.2016 | Rachana L Achar              | Logiscope improvements
 01.01.2017 |  Noopur R K                  | Changed bRemoveDeviceFromList.

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <time.h>
#include <sstream>
#include <algorithm>
#include "spi_tclDeviceListHandler.h"
#include "spi_tclDevHistory.h"
#include "crc.h"
//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclDeviceListHandler.cpp.trc.h"
#endif
#endif

static t_U8 scou8MaxErrorCount = 3;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::spi_tclDeviceListHandler
 ***************************************************************************/
spi_tclDeviceListHandler::spi_tclDeviceListHandler(spi_tclConnSettingsIntf* poConnSettings,
spi_tclDevHistoryIntf* poDevHistory):m_poConnSettings(poConnSettings), m_poDevHistory(poDevHistory),
m_enSelectReason(e8_REASON_UNKNOWN)
{
   ETG_TRACE_USR1((" spi_tclDeviceListHandler::spi_tclDeviceListHandler() entered "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::~spi_tclDeviceListHandler
 ***************************************************************************/
spi_tclDeviceListHandler::~spi_tclDeviceListHandler()
{
   ETG_TRACE_USR1((" spi_tclDeviceListHandler::~spi_tclDeviceListHandler() entered "));
   vClearDeviceList();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bRestoreDeviceList
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bRestoreDeviceList()
{
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   /*lint -esym(40,u32DeviceHandle)u32DeviceHandle Undeclared identifier */
   std::vector<trEntireDeviceInfo> rvecDevHistory;
   t_Bool bRetVal = bGetDeviceHistory(rvecDevHistory);

   //! Initially populate device list with the entries from device history
   if (true == bRetVal)
   {
      m_oDeviceListLock.s16Lock();
      for (auto itDevHist = rvecDevHistory.begin(); itDevHist != rvecDevHistory.end(); ++itDevHist)
      {
         m_mapDeviceInfoList.insert(std::pair<t_U32, trEntireDeviceInfo>(itDevHist->rDeviceInfo.u32DeviceHandle,
                  *itDevHist));

      } //for (auto itDevHist = rvecDevHistory.begin()...
      m_oDeviceListLock.vUnlock();
   } // if (true == bInit)

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::bRestoreDeviceList: Restoring Device list from history"
            "Device History size = %d Return Value = %d ", rvecDevHistory.size(), ETG_ENUM(BOOL, bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bSaveDeviceList
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::bSaveDeviceList()
{
   ETG_TRACE_USR1((" spi_tclDeviceListHandler::bSaveDeviceList() : Saving device list persistently "));

   //! Save Device list persistently
   //! Write the data to local variable and then write to db as the process
   //! of writing data to database might be time consuming
   std::map<t_U32, trEntireDeviceInfo> mapDeviceInfoList;
   m_oDeviceListLock.s16Lock();
   mapDeviceInfoList = m_mapDeviceInfoList;
   m_oDeviceListLock.vUnlock();

   if (NULL != m_poDevHistory)
   {
      m_poDevHistory->vSaveDeviceList(mapDeviceInfoList);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vClearDeviceList
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vClearDeviceList()
{
   ETG_TRACE_USR1((" spi_tclDeviceListHandler::vClearDeviceList() entered "));
   m_oDeviceListLock.s16Lock();
   m_mapDeviceInfoList.clear();
   m_oDeviceListLock.vUnlock();
}


/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enAddDeviceToList
 ***************************************************************************/
tenDeviceStatusInfo spi_tclDeviceListHandler::enAddDeviceToList(const t_U32 cou32DeviceHandle,
         const trDeviceInfo &rfrDeviceInfo)
{
   /*lint -esym(40,enDeviceUsageEnabled)enDeviceUsageEnabled Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,enDeviceConnectionStatus)enDeviceConnectionStatus Undeclared identifier */
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,szDeviceName)szDeviceName Undeclared identifier */
   /*lint -esym(40,bSelectedDevice)bSelectedDevice Undeclared identifier */
   /*lint -esym(40,bIsUserDeselected)bIsUserDeselected Undeclared identifier */
   /*lint -esym(40,u32AccessIndex)u32AccessIndex Undeclared identifier */
   /*lint -esym(40,enUserPreference)enUserPreference Undeclared identifier */
   /*lint -esym(40,bDeviceValidated)bDeviceValidated Undeclared identifier */
   /*lint -esym(40,enDeviceConnectionStatus)enDeviceConnectionStatus Undeclared identifier */

   tenDeviceStatusInfo enDeviceListChange = e8DEVICE_ADDED;
   t_Bool bDeviceAddedForWireless = false;

   if (0 != cou32DeviceHandle)
   {
      trEntireDeviceInfo rEntireDeviceInfo;
      rEntireDeviceInfo.bDeviceValidated = false;
      rEntireDeviceInfo.bIsDeviceUsed = false;
      rEntireDeviceInfo.enIsDeviceWhiteListed = e8_DEVICE_NOT_WHITELISTED;
      rEntireDeviceInfo.rDeviceInfo = rfrDeviceInfo;

      //! Store the device in the device list
      m_oDeviceListLock.s16Lock();

      //! Check if the device already exists. If the device exists
      //! update the device with new info. Restore the device
      //! usage enabled flag and other persistent information
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         //! Check if the device name or connection status of existing device has changed
         //! No update is required if the device name and connection status remains same
         //! If connection status changes to connected, device update will be sent with e8DEVICE_ADDED

         if((rEntireDeviceInfo.rDeviceInfo.enDeviceConnectionStatus == (itMapDevList->second).rDeviceInfo.enDeviceConnectionStatus))
         {
            //Get the Device Status Info
            enDeviceListChange = enGetDeviceStatusInfo(rEntireDeviceInfo.rDeviceInfo, cou32DeviceHandle);
         }
         else
         {
            if(e8WIRELESS_CONNECTED == rEntireDeviceInfo.rDeviceInfo.enDeviceConnectionType)
            {
               bDeviceAddedForWireless = bIsDeviceAddedForWireless(rEntireDeviceInfo.rDeviceInfo);
               if(true == bDeviceAddedForWireless)
               {
                  //! Handling Taping on HFP icon, SPI not supposed to start Selection
                  enDeviceListChange = e8DEVICE_ADDED;
               }
               else
               {
                  enDeviceListChange = e8DEVICE_CHANGED;
               }
            }
         }
         //! Update the device name only if its not empty
         if (true == (rEntireDeviceInfo.rDeviceInfo.szDeviceName.empty()))
         {
            rEntireDeviceInfo.rDeviceInfo.szDeviceName = (itMapDevList->second).rDeviceInfo.szDeviceName;
         }

         //! Update the manufacturer name only if its not empty
         if (true == (rEntireDeviceInfo.rDeviceInfo.szDeviceManufacturerName.empty()))
         {
            rEntireDeviceInfo.rDeviceInfo.szDeviceManufacturerName =
                     (itMapDevList->second).rDeviceInfo.szDeviceManufacturerName;
         }

         //! Update the BT Address only if its not empty
         if (true == (rEntireDeviceInfo.rDeviceInfo.szBTAddress.empty()))
         {
            rEntireDeviceInfo.rDeviceInfo.szBTAddress = (itMapDevList->second).rDeviceInfo.szBTAddress;
         }
         //! Update serial number only if its not empty
         if (true == (rEntireDeviceInfo.rDeviceInfo.szSerialNumber.empty()))
         {
            rEntireDeviceInfo.rDeviceInfo.szSerialNumber = (itMapDevList->second).rDeviceInfo.szSerialNumber;
         }

         //! Update session transport only if its valid
         if (e8_SESSION_TRANSPORT_INVALID != (rEntireDeviceInfo.rDeviceInfo.enSessionTransport))
         {
            rEntireDeviceInfo.rDeviceInfo.enSessionTransport = (itMapDevList->second).rDeviceInfo.enSessionTransport;
         }

         //! Update last active session transport only if its valid
         if(e8_SESSION_TRANSPORT_INVALID != (rEntireDeviceInfo.enLastActiveSessionTransport))
         {
            rEntireDeviceInfo.enLastActiveSessionTransport =
                     (itMapDevList->second).enLastActiveSessionTransport;
         }

         rEntireDeviceInfo.rDeviceInfo.enDeviceUsageEnabled =
                  (itMapDevList->second).rDeviceInfo.enDeviceUsageEnabled;
         rEntireDeviceInfo.rDeviceInfo.bSelectedDevice =
                  (itMapDevList->second).rDeviceInfo.bSelectedDevice;
         rEntireDeviceInfo.rDeviceInfo.enUserAuthorizationStatus =
                  (itMapDevList->second).rDeviceInfo.enUserAuthorizationStatus;
         rEntireDeviceInfo.bIsUserDeselected = (itMapDevList->second).bIsUserDeselected;
         rEntireDeviceInfo.u8SelectError = (itMapDevList->second).u8SelectError;
         rEntireDeviceInfo.u32AccessIndex = (itMapDevList->second).u32AccessIndex;
         rEntireDeviceInfo.enUserPreference = (itMapDevList->second).enUserPreference;
         rEntireDeviceInfo.rDeviceInfo.rDeviceSelectionInfo = (itMapDevList->second).rDeviceInfo.rDeviceSelectionInfo;
         rEntireDeviceInfo.rDeviceInfo.u32VendorID = (itMapDevList->second).rDeviceInfo.u32VendorID;
         rEntireDeviceInfo.rDeviceInfo.u32ProductID = (itMapDevList->second).rDeviceInfo.u32ProductID;
         rEntireDeviceInfo.enDeviceSwitchState = (itMapDevList->second).enDeviceSwitchState;


         //! Update the szUSBSession IP Address only if its not empty
         if(true == (rEntireDeviceInfo.szUSBSessionIP.empty()))
         {
            rEntireDeviceInfo.szUSBSessionIP =
                     (itMapDevList->second).szUSBSessionIP;
         }

         //! Update the szWifiSession IP Address only if its not empty
         if(true == (rEntireDeviceInfo.szWifiSessionIP.empty()))
         {
            rEntireDeviceInfo.szWifiSessionIP =
                     (itMapDevList->second).szWifiSessionIP;
         }
         //! If the reported device's USB port type is not known, use the old USB port type.
         //! To avoid overwriting of the actual USB port type with e8_PORT_TYPE_NOT_KNOWN.
         if (e8_PORT_TYPE_NOT_KNOWN == rEntireDeviceInfo.rDeviceInfo.rProjectionCapability.enUSBPortType)
         {
            rEntireDeviceInfo.rDeviceInfo.rProjectionCapability.enUSBPortType =
                     (itMapDevList->second).rDeviceInfo.rProjectionCapability.enUSBPortType;
         }
         rEntireDeviceInfo.rDeviceConnectionCountInfo = (itMapDevList->second).rDeviceConnectionCountInfo;

      } // if (m_mapDeviceInfoList.end() != itMapDevList)
      else
      {

         if(e8WIRELESS_CONNECTED == rEntireDeviceInfo.rDeviceInfo.enDeviceConnectionType)
         {
            bDeviceAddedForWireless = bIsDeviceAddedForWireless(rEntireDeviceInfo.rDeviceInfo);
            if(true == bDeviceAddedForWireless)
            {
               //! Handling Taping on HFP icon, SPI not supposed to start Selection
               enDeviceListChange = e8DEVICE_ADDED;
            }
            else
            {
               enDeviceListChange = e8DEVICE_CHANGED;
            }
         }
      }


      if (e8DEVICE_ADDED == enDeviceListChange)
      {
         t_U32 u32NextAccessIndex = 0;
         u32NextAccessIndex = (u32GetMaxIndex() + 1);
         rEntireDeviceInfo.u32AccessIndex = u32NextAccessIndex;

         ETG_TRACE_USR1(("spi_tclDeviceListHandler::enAddDeviceToList -Device Handle - %d and It's Access Index - %d", cou32DeviceHandle, rEntireDeviceInfo.u32AccessIndex));
      }

      m_mapDeviceInfoList[cou32DeviceHandle] = rEntireDeviceInfo;
      m_oDeviceListLock.vUnlock();
   } // if( 0!= cou32DeviceHandle)

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enAddDeviceToList: Adding Device 0x%x to list due to change in DeviceStatus = %d ", cou32DeviceHandle, ETG_ENUM(DEVICE_STATUS_INFO,
            enDeviceListChange)));
   return enDeviceListChange;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bRemoveDeviceFromList
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bRemoveDeviceFromList(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::bRemoveDeviceFromList: "
            "Removing Device from list  = %x  \n", cou32DeviceHandle));
   t_Bool bResult = false;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::bRemoveDeviceFromList device found\n"));
         /*now we will be checking whether the device is selected or is in the state of ongoing
          selection or deselection.*/
         tenDeviceSelectionState enDeviceSelectionState =
                  itMapDevList->second.rDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionState;
         if (enDeviceSelectionState == e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS
                  || enDeviceSelectionState == e8_SELECTION_STATE_DEVICE_SELECTED
                  || enDeviceSelectionState == e8_SELECTION_STATE_DEVICE_DESELECTION_INPROGRESS)
         {
            ETG_TRACE_USR4(("spi_tclDeviceListHandler::bRemoveDeviceFromList device not "
                     "deleted due to the one of the following reasons"
                     "Selection state : %d of device %d\n", enDeviceSelectionState, cou32DeviceHandle));
            bResult = false;
         } //if(check selections status)
         else
         {
            ETG_TRACE_USR4(("spi_tclDeviceListHandler::bRemoveDeviceFromList device safe to delete\n"));
            m_mapDeviceInfoList.erase(itMapDevList);
            bResult = true;
         } //else
      } //if (m_mapDeviceInfoList.end() != itMapDevList)
      else
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::bRemoveDeviceFromList device NOT found\n"));
         bResult = false; //  if (m_mapDeviceInfoList.end() != itMapDevList)
      } //else
      m_oDeviceListLock.vUnlock();
   } //(0 != cou32DeviceHandle)
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vGetDeviceList
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vGetDeviceList(const t_U32 cou32DeviceHandle,
         std::vector<trDeviceInfo>& rfvecDeviceInfoList)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vGetDeviceList entered with device id 0x%x", cou32DeviceHandle));
   if ((cou32MAX_DEVICEHANDLE == cou32DeviceHandle) || (0 == cou32DeviceHandle))
   {
      vGetDeviceList(rfvecDeviceInfoList, false, false);
   }
   //Updates HMi only for the requested device handle.
   else if (true == bIsDeviceValid(cou32DeviceHandle))
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         //! If the device friendly name is not available, update HMi with manufacturer name
         if ((true == itMapDevList->second.rDeviceInfo.szDeviceName.empty())
                  && (false == itMapDevList->second.rDeviceInfo.szDeviceManufacturerName.empty()))
         {
            itMapDevList->second.rDeviceInfo.szDeviceName = itMapDevList->second.rDeviceInfo.szDeviceManufacturerName;
         }
         rfvecDeviceInfoList.push_back(itMapDevList->second.rDeviceInfo);
      }
      m_oDeviceListLock.vUnlock();
      vDisplayDeviceList(rfvecDeviceInfoList);
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclDeviceListHandler::vGetDeviceList: invalid device id %d", cou32DeviceHandle));
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vGetConnectedDeviceList
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vGetConnectedDeviceList(std::vector<trDeviceInfo>& rfvecDeviceInfoList)
{
   vGetDeviceList(rfvecDeviceInfoList, false, true);
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vDisplayDeviceList(const std::vec...
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vDisplayDeviceList(const std::vector<trDeviceInfo>& corfvecDeviceInfoList) const
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vDisplayDeviceList entered"));
   // Iterate through all the devices in the given device list
   for (t_U32 u32DevCount = 0; u32DevCount < corfvecDeviceInfoList.size(); ++u32DevCount)
   {
      const trDeviceInfo& corfrDeviceInfo = corfvecDeviceInfoList[u32DevCount];
      //Display device information of each device
      ETG_TRACE_USR4((" \n Device information List  Index = %d", u32DevCount));
      ETG_TRACE_USR4((" DeviceHandle= %u (0x%x)", corfrDeviceInfo.u32DeviceHandle, corfrDeviceInfo.u32DeviceHandle));
      ETG_TRACE_USR4((" Device Name = %s", corfrDeviceInfo.szDeviceName.c_str()));
      ETG_TRACE_USR4((" Device Model Name = %s", corfrDeviceInfo.szDeviceModelName.c_str()));
      ETG_TRACE_USR4((" Device Manufacturer Name = %s", corfrDeviceInfo.szDeviceManufacturerName.c_str()));
      ETG_TRACE_USR4((" Bluetooth Address = %s", corfrDeviceInfo.szBTAddress.c_str()));
      ETG_TRACE_USR4((" Device Category = %d", ETG_ENUM(DEVICE_CATEGORY, corfrDeviceInfo.enDeviceCategory)));
      ETG_TRACE_USR4((" Device Connection Status = %d", ETG_ENUM(CONNECTION_STATUS,
               corfrDeviceInfo.enDeviceConnectionStatus)));
      ETG_TRACE_USR4((" Device Connection Type = %d", ETG_ENUM(CONNECTION_TYPE, corfrDeviceInfo.enDeviceConnectionType)));
      ETG_TRACE_USR4((" Mirrorlink Version : %d.%d", corfrDeviceInfo.rVersionInfo.u32MajorVersion, corfrDeviceInfo.rVersionInfo.u32MinorVersion));
      ETG_TRACE_USR4((" Device Usage Enabled: %d", ETG_ENUM(ENABLED_INFO, corfrDeviceInfo.enDeviceUsageEnabled)));
      ETG_TRACE_USR4((" Device Selection Flag: %d", ETG_ENUM(BOOL, corfrDeviceInfo.bSelectedDevice)));
      ETG_TRACE_USR4((" DAP Support: %d", ETG_ENUM(BOOL, corfrDeviceInfo.bDAPSupport)));
      ETG_TRACE_USR4((" Device Type: %d", ETG_ENUM(DEVICE_TYPE, corfrDeviceInfo.rProjectionCapability.enDeviceType)));
      ETG_TRACE_USR4((" USBPortType: %d", ETG_ENUM(USB_PORT_TYPE, corfrDeviceInfo.rProjectionCapability.enUSBPortType)));
      ETG_TRACE_USR4((" Mirrorlink Support: %d", ETG_ENUM(SPI_SUPPORT,
               corfrDeviceInfo.rProjectionCapability.enMirrorlinkSupport)));
      ETG_TRACE_USR4((" Android Auto support: %d", ETG_ENUM(SPI_SUPPORT,
               corfrDeviceInfo.rProjectionCapability.enAndroidAutoSupport)));
      ETG_TRACE_USR4((" Carplay Support : %d", ETG_ENUM(SPI_SUPPORT,
               corfrDeviceInfo.rProjectionCapability.enCarplaySupport)));
      ETG_TRACE_USR4((" Carlife Support : %d", ETG_ENUM(SPI_SUPPORT,
               corfrDeviceInfo.rProjectionCapability.enCarlifeSupport)));
      ETG_TRACE_USR4((" User authorization status : %d ", ETG_ENUM(AUTHORIZATION_STATUS,
               corfrDeviceInfo.enUserAuthorizationStatus)));
      ETG_TRACE_USR4((" Device Selection State : %d ", ETG_ENUM(SELECTION_STATE,
               corfrDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionState)));
      ETG_TRACE_USR4((" Device selection Error Type : %d ", ETG_ENUM(SELECTION_ERROR,
               corfrDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionErrorType)));
      ETG_TRACE_USR4((" Device selection Progress state : %d ", ETG_ENUM(SEL_PROGRESS_STATE,
               corfrDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionProgressState)));
      ETG_TRACE_USR4((" Device profile : %d ", ETG_ENUM(USB_PROFILE, corfrDeviceInfo.enDeviceProfile)));
      ETG_TRACE_USR4((" Device serial number : %s ", corfrDeviceInfo.szSerialNumber.c_str()));
      ETG_TRACE_USR4((" Session Transport : %d \n", ETG_ENUM(SESSION_TRANSPORT, corfrDeviceInfo.enSessionTransport)));
      ETG_TRACE_USR4((" Vendor id : %d and product id : %d \n", corfrDeviceInfo.u32VendorID, corfrDeviceInfo.u32ProductID));
   } //for (t_U32 u32DevCount = 0; u32DevCount < corfvecDeviceInfoList.size(); u32DevCount++)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vGetEntireDeviceList
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vGetEntireDeviceList(std::map<t_U32, trEntireDeviceInfo> &rfrmapDeviceInfoList)
{
   m_oDeviceListLock.s16Lock();
   rfrmapDeviceInfoList = m_mapDeviceInfoList;
   m_oDeviceListLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclDeviceListHandler:: vGetEntireDeviceList: DeviceList size = %d ", rfrmapDeviceInfoList.size()));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetUserDeselectionFlag
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetUserDeselectionFlag(const t_U32 cou32DeviceHandle, t_Bool bState)
{
   /*lint -esym(40,bIsUserDeselected)bIsUserDeselected Undeclared identifier */
   /*lint -esym(40,second)second Undeclared identifier */

   ETG_TRACE_USR1(("spi_tclDeviceListHandler:: vSetUserDeselectionFlag cou32DeviceHandle =0x%x,  bState = %d ", cou32DeviceHandle, ETG_ENUM(BOOL,
            bState)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         (itMapDevList->second).bIsUserDeselected = bState;
      }
      m_oDeviceListLock.vUnlock();

      if ((e8_FACTORY_RESET != m_enSelectReason) && (NULL != m_poDevHistory))
      {
         m_poDevHistory->vSetUserDeselectionFlag(cou32DeviceHandle, bState);
      }

   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bGetUserDeselectionFlag
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bGetUserDeselectionFlag(const t_U32 cou32DeviceHandle)
{
   t_Bool bState = false;
   if (0 != cou32DeviceHandle)
   {
      /*lint -esym(40,bIsUserDeselected)bIsUserDeselected Undeclared identifier */
      /*lint -esym(40,second)second Undeclared identifier */
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         bState = (itMapDevList->second).bIsUserDeselected;
      }
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)      
   ETG_TRACE_USR1(("spi_tclDeviceListHandler:: bGetUserDeselectionFlag cou32DeviceHandle =0x%x,  bState = %d ", cou32DeviceHandle, ETG_ENUM(BOOL,
            bState)));
   return bState;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::u32GetNoofConnectedDevices
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetNoofConnectedDevices()
{
   t_U32 u32NoofConnectedDevices = 0;
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   /*lint -esym(40,enDeviceConnectionStatus)bSelectedDevice Undeclared identifier */
   m_oDeviceListLock.s16Lock();
   for (auto itMapDevList = m_mapDeviceInfoList.begin(); itMapDevList != m_mapDeviceInfoList.end(); ++itMapDevList)
   {
      if (e8DEV_CONNECTED == (itMapDevList->second).rDeviceInfo.enDeviceConnectionStatus)
      {
         u32NoofConnectedDevices++;
      } // if (e8DEV_CONNECTED == (itMapDevList->second).rDeviceInfo.enDeviceConnectionStatus)
   }
   m_oDeviceListLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetNoofConnectedDevices = %d ", u32NoofConnectedDevices));
   return u32NoofConnectedDevices;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceValidity
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceValidity(const t_U32 cou32DeviceHandle, t_Bool bDeviceValid)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,bDeviceValidated) bDeviceValidated Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceValidity: cou32DeviceHandle = 0x%x "
            "bValidDevice = %d ", cou32DeviceHandle, ETG_ENUM(BOOL, bDeviceValid)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         (itMapDevList->second).bDeviceValidated = bDeviceValid;
      }
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bIsDeviceValid
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bIsDeviceValid(const t_U32 cou32DeviceHandle)
{
   t_Bool bValidDevice = false;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         bValidDevice = true;
      }
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)

   ETG_TRACE_USR2(("spi_tclDeviceListHandler::bIsDeviceValid  = %d ", ETG_ENUM(BOOL, bValidDevice)));
   return bValidDevice;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bGetDeviceInfo
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bGetDeviceInfo(const t_U32 cou32DeviceHandle, trDeviceInfo &rfrDeviceInfo)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   t_Bool bDevFound = false;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         bDevFound = true;
         rfrDeviceInfo = itMapDevList->second.rDeviceInfo;
      }
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)
   ETG_TRACE_USR4(("spi_tclDeviceListHandler::bGetDeviceInfo: u32DeviceHandle = 0x%x  bDevFound = %d", cou32DeviceHandle, ETG_ENUM(BOOL,
            bDevFound)));
   return bDevFound;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceName
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceName(const t_U32 cou32DeviceHandle, t_String &rfrszDeviceName)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceName: cou32DeviceHandle = 0x%x  rfrszDeviceName = %s", cou32DeviceHandle, rfrszDeviceName.c_str()));
   if ((0 != cou32DeviceHandle) && (false == rfrszDeviceName.empty()))
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.szDeviceName = rfrszDeviceName;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)

      m_oDeviceListLock.vUnlock();

      if (NULL != m_poDevHistory)
      {
         m_poDevHistory->vSetDeviceName(cou32DeviceHandle, rfrszDeviceName);
      }

   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vGetDeviceName
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vGetDeviceName(const t_U32 cou32DeviceHandle, t_String &rfrszDeviceName)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vGetDeviceName Entered"));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         rfrszDeviceName = itMapDevList->second.rDeviceInfo.szDeviceName;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)

      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vGetDeviceName: cou32DeviceHandle = 0x%x  rfrszDeviceName = %s",
            cou32DeviceHandle, rfrszDeviceName.c_str()));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vAddDeviceToHistory
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vAddDeviceToHistory(const t_U32 cou32DeviceHandle)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,u32AccessIndex)u32AccessIndex Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vAddDeviceToHistory: Adding Device = 0x%x to history ", cou32DeviceHandle));

   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      //! Fetch the device info for requested handle and write to history
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         trEntireDeviceInfo rfrEntireDevInfo = itMapDevList->second;
         m_oDeviceListLock.vUnlock();
         //! Set the device connection status to disconnected before writing to history
         rfrEntireDevInfo.rDeviceInfo.enDeviceConnectionStatus = e8DEV_NOT_CONNECTED;
         //! Write the device history to a database
         if (NULL != m_poDevHistory)
         {
            m_poDevHistory->bAddtoHistorydb(rfrEntireDevInfo);
            m_poDevHistory->vDisplayDevHistorydb();

         }

      }
      else
      {
         m_oDeviceListLock.vUnlock();
         ETG_TRACE_ERR(("Unable to Add to History: Device 0x%x not found in the list ", cou32DeviceHandle));
      } // if (m_mapDeviceInfoList.end() != itMapDevList)
   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vRemoveDeviceFromHistory
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vRemoveDeviceFromHistory(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vRemoveDeviceFromHistory: "
            "Removing Device from history = 0x%x  ", cou32DeviceHandle));
   if (0 != cou32DeviceHandle)
   {
      if (NULL != m_poDevHistory)
      {
         m_poDevHistory->bDeleteFromHistorydb(cou32DeviceHandle);
      }

   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bGetDeviceHistory
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bGetDeviceHistory(std::vector<trEntireDeviceInfo>& rfvecDevHistory) const
{
   t_Bool bRetDevHistory = false;
   if (NULL != m_poDevHistory)
   {
      bRetDevHistory = m_poDevHistory->bGetDeviceHistoryFromdb(rfvecDevHistory);
      m_poDevHistory->vDisplayDevHistorydb();

      ETG_TRACE_USR4(("spi_tclDeviceListHandler::bGetDeviceHistory: Device hsistory size = %d ", rfvecDevHistory.size()));
   }
   return bRetDevHistory;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceCategory
 ***************************************************************************/
tenDeviceCategory spi_tclDeviceListHandler::enGetDeviceCategory(const t_U32 cou32DeviceHandle)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,enDeviceCategory)enDeviceCategory Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   tenDeviceCategory enDevCat = e8DEV_TYPE_UNKNOWN;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         enDevCat = itMapDevList->second.rDeviceInfo.enDeviceCategory;
      } //if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)

   ETG_TRACE_USR4(("spi_tclDeviceListHandler::enGetDeviceCategory: Device Handle = 0x%x  "
            "has Category = %d ", cou32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY, enDevCat)));
   return enDevCat;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceCategory
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceCategory(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCategory)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceCategory: "
            "Device 0x%x's category set to %d  ", cou32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY, enDeviceCategory)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.enDeviceCategory = enDeviceCategory;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();

      if ((NULL != m_poDevHistory) && (e8_FACTORY_RESET != m_enSelectReason))
      {
         m_poDevHistory->vSetDeviceCategory(cou32DeviceHandle, enDeviceCategory);
      }
   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceConnType
 ***************************************************************************/
tenDeviceConnectionType spi_tclDeviceListHandler::enGetDeviceConnType(const t_U32 cou32DeviceHandle)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,enDeviceConnectionType)enDeviceConnectionType Undeclared identifier */

   tenDeviceConnectionType enDevConnType = e8UNKNOWN_CONNECTION;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         enDevConnType = itMapDevList->second.rDeviceInfo.enDeviceConnectionType;
      } //if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle)

   ETG_TRACE_USR4(("spi_tclDeviceListHandler::enGetDeviceConnType Device Handle = 0x%x  "
            "has ConnType = %d ", cou32DeviceHandle, ETG_ENUM(CONNECTION_TYPE, enDevConnType)));
   return enDevConnType;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceConnStatus
 ***************************************************************************/
tenDeviceConnectionStatus spi_tclDeviceListHandler::enGetDeviceConnStatus(const t_U32 cou32DeviceHandle)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   tenDeviceConnectionStatus enDevConnStatus = e8DEV_NOT_CONNECTED;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         enDevConnStatus = itMapDevList->second.rDeviceInfo.enDeviceConnectionStatus;
      } //if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)

   ETG_TRACE_USR2(("spi_tclDeviceListHandler::enGetDeviceConnStatus Device Handle = 0x%x  "
            "ConnStatus = %d ", cou32DeviceHandle, ETG_ENUM(CONNECTION_STATUS, enDevConnStatus)));
   return enDevConnStatus;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetBTAddress
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetBTAddress(const t_U32 cou32DeviceHandle, const t_String& corfrszBTAddress)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetBTAddress: cou32DeviceHandle = 0x%x, BTAddress = %s", cou32DeviceHandle, corfrszBTAddress.c_str()));
   if ((0 != cou32DeviceHandle) && (false == corfrszBTAddress.empty()))
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.szBTAddress = corfrszBTAddress;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();

      if (NULL != m_poDevHistory)
      {
         m_poDevHistory->vSetBTAddress(cou32DeviceHandle, corfrszBTAddress);
      }

   } //if ((0 != cou32DeviceHandle) && (false == corfrszBTAddress.empty()))
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vGetBTAddress
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vGetBTAddress(const t_U32 cou32DeviceHandle, t_String &rfszBTAddress)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,szBTAddress)szBTAddress Undeclared identifier */

   m_oDeviceListLock.s16Lock();
   auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
   if (m_mapDeviceInfoList.end() != itMapDevList)
   {
      rfszBTAddress = itMapDevList->second.rDeviceInfo.szBTAddress;
   } //if (m_mapDeviceInfoList.end() != itMapDevList)
   m_oDeviceListLock.vUnlock();

   ETG_TRACE_USR4(("spi_tclDeviceListHandler::vGetBTAddress Device Handle = 0x%x  "
            "szBTAddress = %s ", cou32DeviceHandle, rfszBTAddress.c_str()));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceSelection
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceSelection(const t_U32 cou32DeviceHandle, t_Bool bSelectDev)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   /*lint -esym(40,bIsDeviceUsed)bIsDeviceUsed  Undeclared identifier */
   /*lint -esym(40,bSelectedDevice)bSelectedDevice Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceSelection: "
            "cou32DeviceHandle = 0x%x, bSelectDev = %d ", cou32DeviceHandle, ETG_ENUM(BOOL, bSelectDev)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.bSelectedDevice = bSelectDev;
         if (true == bSelectDev)
         {
            itMapDevList->second.bIsDeviceUsed = true;
         }
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)

      m_oDeviceListLock.vUnlock();

      if ((e8_FACTORY_RESET != m_enSelectReason) && (NULL != m_poDevHistory))
      {
         m_poDevHistory->vSetSelectedDevice(cou32DeviceHandle, bSelectDev);
      }

   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::u32GetSelectedDevice
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetSelectedDevice()
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   /*lint -esym(40,bSelectedDevice)bSelectedDevice Undeclared identifier */
   /*lint -esym(40,first)first Undeclared identifier */
   t_U32 u32DeviceHandle = 0;
   m_oDeviceListLock.s16Lock();
   for (auto itMapDevList = m_mapDeviceInfoList.begin(); itMapDevList != m_mapDeviceInfoList.end(); ++itMapDevList)
   {
      if (true == (itMapDevList->second).rDeviceInfo.bSelectedDevice)
      {
         u32DeviceHandle = itMapDevList->first;
         break;
      } // if (true == (itMapDevList->second).rDeviceInfo.bSelectedDevice)
   }
   m_oDeviceListLock.vUnlock();
   ETG_TRACE_USR4(("spi_tclDeviceListHandler::u32GetSelectedDevice SelectedDevice  = 0x%x  ", u32DeviceHandle));
   return u32DeviceHandle;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bWasDeviceSelected
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bWasDeviceSelected(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::bWasDeviceSelected: "));
   //! Device was previously selected(used) if it was written to History
   if (NULL == m_poDevHistory)
      return false;

   t_Bool bFindDev = false;

   if (NULL != m_poDevHistory)
   {
      bFindDev = m_poDevHistory->bFindDevice(cou32DeviceHandle);
   }

   return bFindDev;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceConnectionStatus
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceConnectionStatus(const t_U32 cou32DeviceHandle,
         tenDeviceConnectionStatus enConnStatus)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,enDeviceConnectionStatus)enDeviceConnectionStatus Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceConnectionStatus: "
            "cou32DeviceHandle = 0x%x, enConnStatus = %d ", cou32DeviceHandle, ETG_ENUM(CONNECTION_STATUS, enConnStatus)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.enDeviceConnectionStatus = enConnStatus;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bIsDeviceConnected
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bIsDeviceConnected(const t_U32 cou32DeviceHandle)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   t_Bool bDevConnected = false;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         bDevConnected = (e8DEV_CONNECTED == itMapDevList->second.rDeviceInfo.enDeviceConnectionStatus);
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)
   ETG_TRACE_USR2(("spi_tclDeviceListHandler::bIsDeviceConnected: cou32DeviceHandlee = 0x%x bDevConnected? %d ", cou32DeviceHandle, ETG_ENUM(BOOL,
            bDevConnected)));
   return bDevConnected;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDAPSupport
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDAPSupport(const t_U32 cou32DeviceHandle, t_Bool bDAPSupported)
{
   /*lint -esym(40,bDAPSupport)bDAPSupport Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDAPSupport: cou32DeviceHandle = 0x%x bIsDAPSupported = %d", cou32DeviceHandle, ETG_ENUM(BOOL,
            bDAPSupported)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.bDAPSupport = bDAPSupported;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bIsDAPSupported
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bIsDAPSupported(const t_U32 cou32DeviceHandle)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   /*lint -esym(40,bDAPSupport)bDAPSupport Undeclared identifier */
   t_Bool bDAPSupport = false;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         bDAPSupport = itMapDevList->second.rDeviceInfo.bDAPSupport;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)
   ETG_TRACE_USR2((" spi_tclDeviceListHandler::bIsDAPSupported: cou32DeviceHandle = 0x%x bDAPSupport = %d", cou32DeviceHandle, ETG_ENUM(BOOL,
            bDAPSupport)));
   return bDAPSupport;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDeviceListHandler::vSetDeviceUsagePreference
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceUsagePreference(const t_U32 cou32DeviceHandle, tenEnabledInfo enEnabledInfo)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   /*lint -esym(40,enDeviceUsageEnabled)enDeviceUsageEnabled Undeclared identifier */
   /*lint -esym(40,enDeviceConnectionStatus)enDeviceConnectionStatus Undeclared identifier */
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.enDeviceUsageEnabled = enEnabledInfo;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();

      if ((e8_FACTORY_RESET != m_enSelectReason) && (NULL != m_poDevHistory))
      {
         m_poDevHistory->vSetDeviceUsagePreference(cou32DeviceHandle, enEnabledInfo);
      }

   } //  if (0 != cou32DeviceHandle)

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceUsagePreference for device 0x%x is set to %d ", cou32DeviceHandle, ETG_ENUM(ENABLED_INFO,
            enEnabledInfo)));
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDeviceListHandler::vGetDeviceUsagePreference
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vGetDeviceUsagePreference(const t_U32 cou32DeviceHandle,
         tenEnabledInfo& rfenEnabledInfo)
{
   /*lint -esym(40,second)second Undeclared identifier */
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         rfenEnabledInfo = itMapDevList->second.rDeviceInfo.enDeviceUsageEnabled;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle)

   ETG_TRACE_USR4(("spi_tclDeviceListHandler::bGetDeviceUsagePreference: cou32DeviceHandle = 0x%x"
            "enUsagePref = %d ", cou32DeviceHandle, ETG_ENUM(ENABLED_INFO, rfenEnabledInfo)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bSetDevProjUsage
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bSetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo enSPIState,
         tenDeviceType enDeviceType)
{
   /*lint -esym(40,second)second Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler:: bSetDevProjUsage enSPIType =%d,  bSPIState = %d ", ETG_ENUM(DEVICE_CATEGORY,
            enSPIType), ETG_ENUM(BOOL, enSPIState)));

   //If the preferred setting is different then the overall setting for the technology then change
   //the enabled info value for the device category of the devices has to be changed.
   tenEnabledInfo enCurOverallSetting = e8USAGE_DISABLED;
   bGetDevProjUsage(enSPIType, enCurOverallSetting, enDeviceType);
   if ((enSPIState != enCurOverallSetting) && (e8USAGE_DISABLED != enSPIState))
   {
      vChangeDeviceUsageSetting(enSPIType, enSPIState);
   }
   t_Bool bRetVal = false;
   if (NULL != m_poConnSettings)
   {
      //Store setting in datapool.
      if (e8DEV_TYPE_DIPO == enSPIType)
      {
         bRetVal = m_poConnSettings->bSetDipoEnableSetVal(enSPIState);
      }
      else if (e8DEV_TYPE_MIRRORLINK == enSPIType)
      {
         bRetVal = m_poConnSettings->bSetMLLinkEnableSetVal(enSPIState);
      }
      else if (e8DEV_TYPE_ANDROIDAUTO == enSPIType)
      {
         bRetVal = m_poConnSettings->bSetAAPEnableSetVal(enSPIState);
      }
      else if (e8DEV_TYPE_MYSPIN == enSPIType)
      {
         bRetVal = m_poConnSettings->bSetMySPINEnableSetVal(enSPIState);
      }
      else if (e8DEV_TYPE_CARLIFE == enSPIType)
      {
         bRetVal = m_poConnSettings->bSetCarlifeEnableSetVal(enSPIState);
      }
      else if (e8DEV_TYPE_ONCAR == enSPIType)
      {
         bRetVal = m_poConnSettings->bWriteOnCarEnableSetting(enSPIState);
      }
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vEvaluateEnableInfoForCarlife
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vEvaluateEnableInfoForCarlife(tenEnabledInfo &rfrEnabledInfo,
         tenDeviceType enDeviceType)
{

   t_Bool bIsCarlifeFeatureSupported = false;

   switch (enDeviceType)
   {
      case e8_APPLE_DEVICE:
      {
         bIsCarlifeFeatureSupported = m_rSPIFeatureSupport.bCarlifeSupportedForIOS();
      }
         break;
      case e8_ANDROID_DEVICE:
      {
         bIsCarlifeFeatureSupported = m_rSPIFeatureSupport.bCarlifeSupportedForAndroid();
      }
         break;
      default:
         bIsCarlifeFeatureSupported = (m_rSPIFeatureSupport.bCarlifeSupportedForIOS()
                  || m_rSPIFeatureSupport.bCarlifeSupportedForAndroid());
   }

   if (true == bIsCarlifeFeatureSupported)
   {
      if (NULL != m_poConnSettings)
      {
         rfrEnabledInfo = m_poConnSettings->enReadCarlifeEnableSettingVal();
      }
   }

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vEvaluateEnableInfoForCarlife - rfrEnabledInfo = %d  ", ETG_ENUM(ENABLED_INFO,
            rfrEnabledInfo)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bGetDevProjUsage
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bGetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo &rfrEnabledInfo,
         tenDeviceType enDeviceType)
{
   t_Bool bRetVal = true;

   //Read feature support from calibration
   if (false == m_rSPIFeatureSupport.bGetFeatureSupport(enSPIType, enDeviceType))
   {
      rfrEnabledInfo = e8USAGE_DISABLED;
      ETG_TRACE_USR2(("[DESC]: spi_tclDeviceListHandler::bGetDevProjUsage - %d technology is disabled in KDS", ETG_ENUM(DEVICE_CATEGORY,
               enSPIType)));
   }

   else
   {
      //Read setting from datapool.
      switch (enSPIType)
      {
         case e8DEV_TYPE_DIPO:
         {
            if (NULL != m_poConnSettings)
            {
               rfrEnabledInfo = m_poConnSettings->enReadDipoEnableSettingVal();
            }
         }
            break;
         case e8DEV_TYPE_MIRRORLINK:
         {
            if (NULL != m_poConnSettings)
            {
               rfrEnabledInfo = m_poConnSettings->enReadMLEnableSettingVal();
            }
         }
            break;
         case e8DEV_TYPE_ANDROIDAUTO:
         {
            if (NULL != m_poConnSettings)
            {
               rfrEnabledInfo = m_poConnSettings->enReadAAPEnableSettingVal();
            }
         }
            break;
         case e8DEV_TYPE_MYSPIN:
         {
            if (NULL != m_poConnSettings)
            {
               rfrEnabledInfo = m_poConnSettings->enReadmySPINEnableSettingVal();
            }
         }
            break;
         case e8DEV_TYPE_CARLIFE:
         {

            vEvaluateEnableInfoForCarlife(rfrEnabledInfo, enDeviceType);

         }
            break;
         case e8DEV_TYPE_ONCAR:
         {
            if (NULL != m_poConnSettings)
            {
               rfrEnabledInfo = m_poConnSettings->enReadOnCarEnableSetting();
            }
         }
            break;
         case e8DEV_TYPE_UNKNOWN:
         {
            //! if the device category is not known then further processing is required to evaluate category
            //! Hence by default detection of unknown devices will be always enabled
            rfrEnabledInfo = e8USAGE_ENABLED;
         }
            break;
         default:
         {
            rfrEnabledInfo = e8USAGE_DISABLED;
         }
            break;
      }
   }

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::bGetDevProjUsage  enDevCat = %d rfrEnabledInfo = %d  ", ETG_ENUM(DEVICE_CATEGORY,
            enSPIType), ETG_ENUM(ENABLED_INFO, rfrEnabledInfo)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetSelectError
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetSelectError(const t_U32 cou32DeviceHandle, t_Bool bIsError)
{
   /*lint -esym(40,second)second Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetSelectError cou32DeviceHandle  = 0x%x,  bIsError - %d ", cou32DeviceHandle, ETG_ENUM(BOOL,
            bIsError)));
   if (0 != cou32DeviceHandle)
   {
      t_Bool bErrorCountReached = false;
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         t_U8 u8ErrorCount = itMapDevList->second.u8SelectError;
         //! Currently retry mechanism is  for carplay, android auto and mirrorlink. For other technologies error flag will be set
         //! when the selection fails
         if ((e8DEV_TYPE_DIPO == itMapDevList->second.rDeviceInfo.enDeviceCategory)
                  || (e8DEV_TYPE_MIRRORLINK == itMapDevList->second.rDeviceInfo.enDeviceCategory)
                  || (e8DEV_TYPE_ANDROIDAUTO == itMapDevList->second.rDeviceInfo.enDeviceCategory)
                  || (e8DEV_TYPE_CARLIFE == itMapDevList->second.rDeviceInfo.enDeviceCategory)
                  || (e8DEV_TYPE_MYSPIN == itMapDevList->second.rDeviceInfo.enDeviceCategory)
                  || (e8DEV_TYPE_ONCAR == itMapDevList->second.rDeviceInfo.enDeviceCategory))
         {
            u8ErrorCount = static_cast<t_U8>((true == bIsError) ? (u8ErrorCount + 1) : 0);
         }
         else
         {
            u8ErrorCount = (true == bIsError) ? (scou8MaxErrorCount) : 0;
         }
         itMapDevList->second.u8SelectError = u8ErrorCount;
         bErrorCountReached = (scou8MaxErrorCount <= u8ErrorCount);
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::vSetSelectError cou32DeviceHandle  = 0x%x,  ErrorCount - %d ", cou32DeviceHandle, u8ErrorCount));
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();

      if (true == bErrorCountReached)
      {
         trDeviceSelectionInfo rDeviceSelectionInfo;
         rDeviceSelectionInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_DEVICE_ERROR;
         rDeviceSelectionInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
         vSetDeviceSelectionInfo(cou32DeviceHandle, rDeviceSelectionInfo);
      }
   } // if (0 != cou32DeviceHandle
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bIsSelectError
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bIsSelectError(const t_U32 cou32DeviceHandle)
{
   /*lint -esym(40,second)second Undeclared identifier */
   t_Bool bIsError = false;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         bIsError = (scou8MaxErrorCount <= itMapDevList->second.u8SelectError);

         ETG_TRACE_USR1(("spi_tclDeviceListHandler::bIsSelectError cou32DeviceHandle  = 0x%x,  ErrorCount - %d ", cou32DeviceHandle, itMapDevList->second.u8SelectError));
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle 
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::bIsSelectError cou32DeviceHandle  = 0x%x,  bIsError - %d  ", cou32DeviceHandle, ETG_ENUM(BOOL,
            bIsError)));
   return bIsError;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::u32GetLastSelectedDevice
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetLastSelectedDevice()
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetLastSelectedDevice Entered()"));
   t_U32 u32GetLastSelectedDev = 0;
   if (NULL != m_poDevHistory)
   {
      u32GetLastSelectedDev = m_poDevHistory->u32GetLastSelectedDevice();
   }

   return u32GetLastSelectedDev;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vClearPrivateData
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vClearPrivateData()
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vClearPrivateData: Clearing device list "));

   m_oDeviceListLock.s16Lock();
   //! @Note: Erasing element and then incrementing iterator will result in incrementing invalid iterator
   //! Hence post increment operator is used for erase
   for (auto itMapDevList = m_mapDeviceInfoList.begin(); itMapDevList != m_mapDeviceInfoList.end(); /*No Increment*/)
   {
      t_U32 u32DeviceHandle = (itMapDevList->second).rDeviceInfo.u32DeviceHandle;

      if ((false == (itMapDevList->second).rDeviceInfo.bSelectedDevice)
               && (false == bGetDeviceWhitelistedInfo(u32DeviceHandle)))
      {
         ETG_TRACE_USR1(("spi_tclDeviceListHandler::vClearPrivateData:Deleting the Device %d ", (itMapDevList->second).rDeviceInfo.u32DeviceHandle));

         itMapDevList = m_mapDeviceInfoList.erase(itMapDevList);
      } // if (e8DEV_CONNECTED == (itMapDevList->second).rDeviceInfo.enDeviceConnectionStatus)
      else
      {
         ++itMapDevList; //! Can use pre-increment in this case to make sure you have the efficient version
      }
   }
   m_oDeviceListLock.vUnlock();

   if (NULL != m_poDevHistory)
   {
      m_poDevHistory->vClearPrivateData();
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bGetUSBConnectedFlag
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bGetUSBConnectedFlag(const t_U32 cou32DeviceHandle)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,bIsUSBConnected)bIsUSBConnected Undeclared identifier */
   t_Bool bRetval = false;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         bRetval = itMapDevList->second.bIsUSBConnected;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
     //ETG_TRACE_USR1(("spi_tclDeviceListHandler::bGetUSBConnectedFlag cou32DeviceHandle = 0x%x USBConnected = %d", cou32DeviceHandle, ETG_ENUM(BOOL, bRetval)));
   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetUSBConnectedFlag
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetUSBConnectedFlag(const t_U32 cou32DeviceHandle, t_Bool bUSBConnected)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,bIsUSBConnected)bIsUSBConnected Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetUSBConnectedFlag cou32DeviceHandle = 0x%x USBConnected = %d", cou32DeviceHandle, ETG_ENUM(BOOL,
            bUSBConnected)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.bIsUSBConnected = bUSBConnected;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetUserPreference
 ***************************************************************************/
tenUserPreference spi_tclDeviceListHandler::enGetUserPreference(const t_U32 cou32DeviceHandle)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,enUserPreference)enUserPreference Undeclared identifier */
   tenUserPreference enUserPref = e8PREFERENCE_NOTKNOWN;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::enGetUserPreference found"));
         enUserPref = itMapDevList->second.enUserPreference;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetUserPreference cou32DeviceHandle = 0x%x enUserPref = %d", cou32DeviceHandle, ETG_ENUM(USER_PREFERENCE,
            enUserPref)));
   return enUserPref;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetUserPreference
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetUserPreference(const t_U32 cou32DeviceHandle, tenUserPreference enUserPref)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,enUserPreference)enUserPreference Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetUserPreference cou32DeviceHandle = 0x%x enUserPref = %d", cou32DeviceHandle, ETG_ENUM(USER_PREFERENCE,
            enUserPref)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::enGetUserPreference found"));
         itMapDevList->second.enUserPreference = enUserPref;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetSPISupport
 ***************************************************************************/
tenSPISupport spi_tclDeviceListHandler::enGetSPISupport(const t_U32 cou32DeviceHandle, tenDeviceCategory enSPIType)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   /*lint -esym(40,enMirrorlinkSupport)enMirrorlinkSupport Undeclared identifier */
   /*lint -esym(40,enAndroidAutoSupport)enAndroidAutoSupport Undeclared identifier */
   /*lint -esym(40,enCarplaySupport)enCarplaySupport Undeclared identifier */
   /*lint -esym(40,enCarlifeSupport)enCarlifeSupport Undeclared identifier */
   tenSPISupport enSPISupport = e8SPI_SUPPORT_UNKNOWN;
   if (true == bIsDeviceValid(cou32DeviceHandle))
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         switch (enSPIType)
         {
            case e8DEV_TYPE_MIRRORLINK:
            {
               enSPISupport = (itMapDevList->second).rDeviceInfo.rProjectionCapability.enMirrorlinkSupport;
               break;
            }
            case e8DEV_TYPE_ANDROIDAUTO:
            {
               enSPISupport = (itMapDevList->second).rDeviceInfo.rProjectionCapability.enAndroidAutoSupport;
               break;
            }
            case e8DEV_TYPE_DIPO:
            {
               enSPISupport = (itMapDevList->second).rDeviceInfo.rProjectionCapability.enCarplaySupport;
               break;
            }
            case e8DEV_TYPE_CARLIFE:
            {
               enSPISupport = (itMapDevList->second).rDeviceInfo.rProjectionCapability.enCarlifeSupport;
               break;
            }
            case e8DEV_TYPE_ONCAR:
            {
               enSPISupport = (itMapDevList->second).rDeviceInfo.rProjectionCapability.enOnCarSupport;
               break;
            }
            default:
            {
               enSPISupport = e8SPI_SUPPORT_UNKNOWN;
               break;
            }
         }
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   }
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetSPISupport Device ID = %d SPI Type = %d Support =%d ", cou32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY,
            enSPIType), ETG_ENUM(SPI_SUPPORT, enSPISupport)));
   return enSPISupport;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetSPISupport
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetSPISupport(const t_U32 cou32DeviceHandle, tenDeviceCategory enSPIType,
         tenSPISupport enSPISupport)
{
   /*lint -esym(40,second)second Undeclared identifier */
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   /*lint -esym(40,enMirrorlinkSupport)enMirrorlinkSupport Undeclared identifier */
   /*lint -esym(40,enAndroidAutoSupport)enAndroidAutoSupport Undeclared identifier */
   /*lint -esym(40,enCarplaySupport)enCarplaySupport Undeclared identifier */
   /*lint -esym(40,enCarlifeSupport)enCarlifeSupport Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetSPISupport Device ID = %d SPI Type = %d Support =%d ", cou32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY,
            enSPIType), ETG_ENUM(SPI_SUPPORT, enSPISupport)));
   if (true == bIsDeviceValid(cou32DeviceHandle))
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         switch (enSPIType)
         {
            case e8DEV_TYPE_MIRRORLINK:
            {
               (itMapDevList->second).rDeviceInfo.rProjectionCapability.enMirrorlinkSupport = enSPISupport;
               break;
            }
            case e8DEV_TYPE_ANDROIDAUTO:
            {
               (itMapDevList->second).rDeviceInfo.rProjectionCapability.enAndroidAutoSupport = enSPISupport;
               break;
            }
            case e8DEV_TYPE_DIPO:
            {
               (itMapDevList->second).rDeviceInfo.rProjectionCapability.enCarplaySupport = enSPISupport;
               break;
            }
            case e8DEV_TYPE_CARLIFE:
            {
               (itMapDevList->second).rDeviceInfo.rProjectionCapability.enCarlifeSupport = enSPISupport;
               break;
            }
            case e8DEV_TYPE_ONCAR:
            {
               (itMapDevList->second).rDeviceInfo.rProjectionCapability.enOnCarSupport = enSPISupport;
               break;
            }
            default:
            {
               //! TODO Error mSg
               break;
            }
         }
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetUserAuthorizationStatus
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetUserAuthorizationStatus(const t_U32 cou32DeviceHandle,
         tenUserAuthorizationStatus enUserAuthStatus)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetUserAuthorizationStatus: User authorization for device 0x%d is set to %d ", cou32DeviceHandle, ETG_ENUM(AUTHORIZATION_STATUS,
            enUserAuthStatus)));
   if (0 != cou32DeviceHandle)
   {
      trEntireDeviceInfo rEntireDeviceInfo;
      m_oDeviceListLock.s16Lock();
      std::map<t_U32, trEntireDeviceInfo>::iterator itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::vSetUserAuthorizationStatus Device found"));
         itMapDevList->second.rDeviceInfo.enUserAuthorizationStatus = enUserAuthStatus;
         rEntireDeviceInfo = itMapDevList->second;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
      //! Add device to history on accepting/dismissing legal disclaimer

      if (NULL != m_poDevHistory)
      {
         m_poDevHistory->bAddtoHistorydb(rEntireDeviceInfo);
      }

   } // if (0 != cou32DeviceHandle
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetUserAuthorizationStatus
 ***************************************************************************/
tenUserAuthorizationStatus spi_tclDeviceListHandler::enGetUserAuthorizationStatus(const t_U32 cou32DeviceHandle)
{
   tenUserAuthorizationStatus enUserAuth = e8_USER_AUTH_UNKNOWN;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      std::map<t_U32, trEntireDeviceInfo>::iterator itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::enGetUserAuthorizationStatus found"));
         enUserAuth = itMapDevList->second.rDeviceInfo.enUserAuthorizationStatus;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetUserAuthorizationStatus: User Authorization for device %d is %d ", cou32DeviceHandle, ETG_ENUM(AUTHORIZATION_STATUS,
            enUserAuth)));
   return enUserAuth;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceType
 ***************************************************************************/
tenDeviceType spi_tclDeviceListHandler::enGetDeviceType(const t_U32 cou32DeviceHandle)
{
   tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::enGetDeviceType found"));
         enDeviceType = itMapDevList->second.rDeviceInfo.rProjectionCapability.enDeviceType;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetDeviceType: Device type for device %d is %d ", cou32DeviceHandle, ETG_ENUM(DEVICE_TYPE,
            enDeviceType)));
   return enDeviceType;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetUSBPortType
 ***************************************************************************/
tenUSBPortType spi_tclDeviceListHandler::enGetUSBPortType(const t_U32 cou32DeviceHandle)
{
   tenUSBPortType enUSBPortType = e8_PORT_TYPE_NOT_KNOWN;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::enGetUSBPortType found"));
         enUSBPortType = itMapDevList->second.rDeviceInfo.rProjectionCapability.enUSBPortType;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetUSBPortType: Port type for device %d is %d ", cou32DeviceHandle, enUSBPortType));
   return enUSBPortType;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::u32GetDeviceIDFromBTAddress
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetDeviceIDFromBTAddress(const t_String &coszBTAddress)
{
   t_U32 u32DeviceID = 0;
   m_oDeviceListLock.s16Lock();
   for (auto itMapDevList = m_mapDeviceInfoList.begin(); itMapDevList != m_mapDeviceInfoList.end(); ++itMapDevList)
   {
      if (coszBTAddress == (itMapDevList->second).rDeviceInfo.szBTAddress)
      {
         u32DeviceID = itMapDevList->first;
         break;
      } // if (true == (itMapDevList->second).rDeviceInfo.szBTAddress)
   }
   //! Calculate crc of the BT Address if its a unknown device
   if (0 == u32DeviceID)
   {
      ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetDeviceIDFromBTAddress: New Device. Generating Device ID"));
      u32DeviceID = u16CalcCRC16((const t_UChar*) coszBTAddress.c_str(), coszBTAddress.size());
   }

   m_oDeviceListLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetDeviceIDFromBTAddress: Device 0x%x has BT Address %s ", u32DeviceID, coszBTAddress.c_str()));
   return u32DeviceID;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::u32GetDeviceIDFromSerialNo
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetDeviceIDFromSerialNo(const t_String &coszSerialNo)
{
   t_U32 u32DeviceID = 0;
   m_oDeviceListLock.s16Lock();
   for (auto itMapDevList = m_mapDeviceInfoList.begin(); itMapDevList != m_mapDeviceInfoList.end(); ++itMapDevList)
   {
      if (coszSerialNo == (itMapDevList->second).rDeviceInfo.szSerialNumber)
      {
         u32DeviceID = itMapDevList->first;
         break;
      } // if (true == (itMapDevList->second).rDeviceInfo.szBTAddress)
   }
   //! Calculate crc of the serial number if its a unknown device
   if (0 == u32DeviceID)
   {
      ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetDeviceIDFromSerialNo: New Device  Generating Device ID"));
      u32DeviceID = u16CalcCRC16((const t_UChar*) coszSerialNo.c_str(), coszSerialNo.size());
   }
   m_oDeviceListLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetDeviceIDFromSerialNo: Device 0x%x has Serial Number %s ", u32DeviceID, coszSerialNo.c_str()));
   return u32DeviceID;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::u32GetDeviceIDFromName
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetDeviceIDFromName(const t_String &coszDeviceName)
{
   t_U32 u32DeviceID = 0;
   m_oDeviceListLock.s16Lock();
   for (auto itMapDevList = m_mapDeviceInfoList.begin(); itMapDevList != m_mapDeviceInfoList.end(); ++itMapDevList)
   {
      if (coszDeviceName == (itMapDevList->second).rDeviceInfo.szDeviceName)
      {
         u32DeviceID = itMapDevList->first;
         break;
      } // if (true == (itMapDevList->second).rDeviceInfo.szBTAddress)
   }
   m_oDeviceListLock.vUnlock();
   //! Calculate crc of the serial number if its a unknown device
   if (0 == u32DeviceID)
   {
      ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetDeviceIDFromName: New Device  Generating Device ID"));
      u32DeviceID = u16CalcCRC16((const t_UChar*) coszDeviceName.c_str(), coszDeviceName.size());
   }
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetDeviceIDFromName: Device 0x%x has Name %s ", u32DeviceID, coszDeviceName.c_str()));
   return u32DeviceID;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vGetSerialNumber
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vGetSerialNumber(t_String &coszSerialNo, const t_U32 cou32DeviceID)
{
   m_oDeviceListLock.s16Lock();
   auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceID);
   if (m_mapDeviceInfoList.end() != itMapDevList)
   {
      coszSerialNo = itMapDevList->second.rDeviceInfo.szSerialNumber;
   } //if (m_mapDeviceInfoList.end() != itMapDevList)
   m_oDeviceListLock.vUnlock();

   ETG_TRACE_USR4(("spi_tclDeviceListHandler::Serial for device Device Handle = 0x%x  "
            "is = %s ", cou32DeviceID, coszSerialNo.c_str()));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceProfile
 ***************************************************************************/
tenDeviceProfile spi_tclDeviceListHandler::enGetDeviceProfile(const t_U32 cou32DeviceHandle)
{
   tenDeviceProfile enDeviceProfile = e8_PROFILE_UNKNOWN;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         enDeviceProfile = itMapDevList->second.rDeviceInfo.enDeviceProfile;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetDeviceProfile: Device profile for device %d is %d ", cou32DeviceHandle, ETG_ENUM(USB_PROFILE,
            enDeviceProfile)));
   return enDeviceProfile;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceProfile
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceProfile(const t_U32 cou32DeviceID, tenDeviceProfile enDeviceProfile)
{
   if (0 != cou32DeviceID)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceID);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.enDeviceProfile = enDeviceProfile;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceProfile: Device profile for device %d set to %d ", cou32DeviceID, ETG_ENUM(USB_PROFILE,
            enDeviceProfile)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetSessionTransport()
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetSessionTransport(const t_U32 cou32DeviceHandle,
         tenSessionTransport enSessionTransport)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetSessionTransport Setting session transport for DeviceHandle = 0x%x to %d", cou32DeviceHandle, ETG_ENUM(SESSION_TRANSPORT,
            enSessionTransport)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.enSessionTransport =  enSessionTransport;
         //! update only when on Session start message
         if(e8_SESSION_TRANSPORT_UNKNOWN != enSessionTransport)
         {
            itMapDevList->second.enLastActiveSessionTransport = enSessionTransport;
         }
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();


      //! update only when on Session start message
      if((e8_SESSION_TRANSPORT_UNKNOWN != enSessionTransport) && (NULL != m_poDevHistory))
      {
         m_poDevHistory->vSetLastActiveSessionTransport(cou32DeviceHandle, enSessionTransport);
      }
   } // if (0 != cou32DeviceHandle
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetSessionTransport()
 ***************************************************************************/
tenSessionTransport spi_tclDeviceListHandler::enGetSessionTransport(const t_U32 cou32DeviceHandle)
{
   tenSessionTransport enSessionTransport = e8_SESSION_TRANSPORT_UNKNOWN;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         enSessionTransport = itMapDevList->second.rDeviceInfo.enSessionTransport;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetSessionTransport:device %d has established session on %d transport", cou32DeviceHandle, ETG_ENUM(SESSION_TRANSPORT,
            enSessionTransport)));
   return enSessionTransport;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceSelectionInfo
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceSelectionInfo(const t_U32 cou32DeviceHandle,
         const trDeviceSelectionInfo &corfrDeviceSelectioninfo)
{
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::vSetDeviceSelectionInfo found"));
         itMapDevList->second.rDeviceInfo.rDeviceSelectionInfo = corfrDeviceSelectioninfo;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceSelectionInfo: Device selection info for device 0x%x is set"
            " to Selection state = %d and selection error = %d ", cou32DeviceHandle, ETG_ENUM(SELECTION_STATE,
            corfrDeviceSelectioninfo.enDeviceSelectionState), ETG_ENUM(SELECTION_ERROR,
            corfrDeviceSelectioninfo.enDeviceSelectionErrorType)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceSelectionErrorType
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceSelectionErrorType(const t_U32 cou32DeviceHandle,
                        const tenDeviceSelectionErrorType coenDeviceSelectionErrorType)
{
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceSelectionErrorType: Device selection error for device 0x%x is set"
                  " to = %d ", cou32DeviceHandle, ETG_CENUM(tenDeviceSelectionErrorType, coenDeviceSelectionErrorType)));
         itMapDevList->second.rDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionErrorType = coenDeviceSelectionErrorType;
      } 
      m_oDeviceListLock.vUnlock();
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceSelectionErrorType
 ***************************************************************************/
tenDeviceSelectionErrorType spi_tclDeviceListHandler::enGetDeviceSelectionErrorType(const t_U32 cou32DeviceHandle)
{
   tenDeviceSelectionErrorType enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_NONE;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::enGetDeviceSelectionErrorType found"));
         enDeviceSelectionErrorType = itMapDevList->second.rDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionErrorType;
      }
      m_oDeviceListLock.vUnlock();
   }

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetDeviceSelectionErrorType: Device selection error for device 0x%x is set"
            " to = %d ", cou32DeviceHandle, ETG_CENUM(tenDeviceSelectionErrorType, enDeviceSelectionErrorType)));
   return enDeviceSelectionErrorType;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vGetDeviceSelectionInfo
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vGetDeviceSelectionInfo(const t_U32 cou32DeviceHandle,
         trDeviceSelectionInfo &rfrDeviceSelectioninfo)
{
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         rfrDeviceSelectioninfo = itMapDevList->second.rDeviceInfo.rDeviceSelectionInfo;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vGetDeviceSelectionInfo: Device selection info for device %d is "
            "  Selection state = %d and selection error = %d ", cou32DeviceHandle, ETG_ENUM(SELECTION_STATE,
            rfrDeviceSelectioninfo.enDeviceSelectionState), ETG_ENUM(SELECTION_ERROR,
            rfrDeviceSelectioninfo.enDeviceSelectionErrorType)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSortDeviceList
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSortDeviceList(std::vector<trEntireDeviceInfo>& rfvecEntireDeviceInfoList,
         std::vector<trDeviceInfo>& rfvecDeviceInfoList)
{
   /*lint -esym(40,rDeviceInfo)rDeviceInfo Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSortDeviceList : Sorting device list in the order"
            " of last selected and connected devices "));
   //! Sorts device list in the order Connected Devices (arranged as per selection time) -> Not connected
   //! stable_sort is used preserves the relative order of the elements with equivalent values
   //! The last argument trEntireDeviceInfo() is a call for overloaded function call operator
   //! for comparison. Used by std::sort algorithm to compare objects of type trEntireDeviceInfo
   std::stable_sort(rfvecEntireDeviceInfoList.begin(), rfvecEntireDeviceInfoList.end(), trEntireDeviceInfo());

   //! Copy the device info from the sorted device list
   std::vector<trEntireDeviceInfo>::iterator itvecEntireList;
   for (itvecEntireList = rfvecEntireDeviceInfoList.begin(); itvecEntireList != rfvecEntireDeviceInfoList.end();
            itvecEntireList++)
   {
      rfvecDeviceInfoList.push_back(itvecEntireList->rDeviceInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vGetDeviceList
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vGetDeviceList(std::vector<trDeviceInfo>& rfvecDeviceInfoList, t_Bool bCertifiedOnly,
         t_Bool bConnectedOnly)
{
   /*lint -esym(40,bDeviceValidated) bDeviceValidated Undeclared identifier */
   /*lint -esym(40,enDeviceCategory) enDeviceCategory Undeclared identifier */

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vGetDeviceList: Devices reported should be Certified = %d "
            "Connected = %d  DeviceList size = %d", bCertifiedOnly, bConnectedOnly, m_mapDeviceInfoList.size()));

   m_oDeviceListLock.s16Lock();
   //! Populate device list based on bCertifiedOnly and bConnectedOnly
   //! if bCertifiedOnly = true, check if device is certified
   //! if bConnectedOnly = true, check if device is in connected state
   //! if both are false add all devices to device list
   //! ----------------------------------------------------------------------------
   //! !bCertifiedOnly || |   !bConnectedOnly ||                    |     Result
   //!     bDeviceValidated  |enDeviceConnectionStatus == e8DEV_CONNECTED |(add to list)
   //!   true                |    true                                    |   true
   //!   true                |    false                                   |   false
   //!   false               |    true                                    |   false
   //!   false               |    false                                   |   false

   std::vector<trEntireDeviceInfo> vecEntireDeviceInfoList;
   for (auto itMapDevList = m_mapDeviceInfoList.begin(); itMapDevList != m_mapDeviceInfoList.end(); ++itMapDevList)
   {
      trEntireDeviceInfo& rfrDeviceInfoListItem = (itMapDevList->second);
      //! If the device friendly name is not available, update HMi with manufacturer name
      if ((true == rfrDeviceInfoListItem.rDeviceInfo.szDeviceName.empty())
               && (false == rfrDeviceInfoListItem.rDeviceInfo.szDeviceManufacturerName.empty()))
      {
         rfrDeviceInfoListItem.rDeviceInfo.szDeviceName = rfrDeviceInfoListItem.rDeviceInfo.szDeviceManufacturerName;
      }

      t_Bool bDeviceCertified = (rfrDeviceInfoListItem).bDeviceValidated;
      t_Bool bDeviceConnected = (e8DEV_CONNECTED == rfrDeviceInfoListItem.rDeviceInfo.enDeviceConnectionStatus);
      //! Invalidate the device if the device is deleted by user and is not yet connected.
      //! This is required in the following use case: If user disconnects and  deletes the device from the list,
      //! add device will still be shown as HMI doen't have information on device connection status
      t_Bool bDeviceValid = true;
      t_Bool bValidityCondMet = (true == bDeviceValid) && ((false == bCertifiedOnly) || (true == bDeviceCertified));
      t_Bool bConnectionCondMet = (false == bConnectedOnly) || (true == bDeviceConnected);
      if ((true == bValidityCondMet) && (true == bConnectionCondMet))
      {
         vecEntireDeviceInfoList.push_back((rfrDeviceInfoListItem));
      }   //if ((true == bValidityCondMet) && (true == bConnectionCondMet))
   }
   m_oDeviceListLock.vUnlock();

   vSortDeviceList(vecEntireDeviceInfoList, rfvecDeviceInfoList);

   //! Display device list
   vDisplayDeviceList(rfvecDeviceInfoList);
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vChangeDeviceUsageSetting
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vChangeDeviceUsageSetting(tenDeviceCategory enDeviceCategory,
         tenEnabledInfo enNewDeviceSetting)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceListHandler::vChangeDeviceUsageSetting entered"));
   //Change the device usage preference for all the devices in the list
   for (auto itMapDevList = m_mapDeviceInfoList.begin(); itMapDevList != m_mapDeviceInfoList.end(); ++itMapDevList)
   {
      if (enDeviceCategory == (itMapDevList->second).rDeviceInfo.enDeviceCategory)
      {
         (itMapDevList->second).rDeviceInfo.enDeviceUsageEnabled = enNewDeviceSetting;
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void vSetSpiFeatureSupport(const trSpiFeatureSupport& corfrSpiFeatureSupp)
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetSpiFeatureSupport(const trSpiFeatureSupport& corfrSpiFeatureSupp)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetSpiFeatureSupport entered"));
   m_rSPIFeatureSupport = corfrSpiFeatureSupp;
}

/***************************************************************************
 ** FUNCTION: t_Void vSetSelectionProgressState(const trSpiFeatureSupport& corfrSpiFeatureSupp)
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetSelectionProgressState(const t_U32 cou32DeviceHandle,
         tenDeviceSelectionProgressState enDeviceSelectionProgressState)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler:: vSetSelectionProgressState entered"));
   tenDeviceSelectionProgressState enSelectionProgressState = e8_SELECTION_PROGRESS_STATE_NOT_APPLICABLE;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::vSetSelectionProgressState found"));
         tenDeviceSelectionState enDeviceSelectionState =
                  itMapDevList->second.rDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionState;
         if (enDeviceSelectionState == e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS)
         {
            itMapDevList->second.rDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionProgressState =
                     enDeviceSelectionProgressState;
         }
         else
         {
            itMapDevList->second.rDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionProgressState =
                     e8_SELECTION_PROGRESS_STATE_NOT_APPLICABLE;
         }
         enSelectionProgressState =
                  itMapDevList->second.rDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionProgressState;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetSelectionProgressState: Device selection info for device 0x%x is set"
            " to vSetSelectionProgressState = %d", cou32DeviceHandle, ETG_ENUM(SEL_PROGRESS_STATE,
            enSelectionProgressState)));
}

/***************************************************************************
 ** FUNCTION: t_Void vSetTechnologySelection(const t_U32 cou32DeviceHandle, tenDeviceCategory enTechnologyPreference)
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetTechnologySelection(const t_U32 cou32DeviceHandle,
         tenDeviceCategory enTechnologyPreference)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetTechnologySelection Entered()"));
   ETG_TRACE_USR1(("Technology Preference is %d for DeviceHandle %d", ETG_ENUM(DEVICE_CATEGORY, enTechnologyPreference), cou32DeviceHandle));
   if (0 != cou32DeviceHandle)
   {
      //If the User Selects "NO" on AskOnPop , Add the Device to the DeviceHistory
      if (e8DEV_TYPE_DEFAULT == enTechnologyPreference)
      {
         trEntireDeviceInfo rEntireDeviceInfo;
         m_oDeviceListLock.s16Lock();
         std::map<t_U32, trEntireDeviceInfo>::iterator itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
         if (m_mapDeviceInfoList.end() != itMapDevList)
         {
            rEntireDeviceInfo = itMapDevList->second;
         } //  if (m_mapDeviceInfoList.end() != itMapDevList)
         m_oDeviceListLock.vUnlock();
         //Adding Device to the History

         if (NULL != m_poDevHistory)
         {
            m_poDevHistory->bAddtoHistorydb(rEntireDeviceInfo);
         }
      }
      else
      {
         //In this case Device is added to the Device list once the Selection Sequence is completed
         //!Set the technology selected by user / HMI
         vSetDeviceCategory(cou32DeviceHandle, enTechnologyPreference);
      }
   } // if (0 != cou32DeviceHandle

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetTechnologySelection Left()"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::rGetDeviceConnectionCountInfo
 ***************************************************************************/
trDeviceConnectionCountInfo spi_tclDeviceListHandler::rGetDeviceConnectionCountInfo(const t_U32 cou32DeviceHandle)
{

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::rGetDeviceConnectionCountInfo Entered() cou32DeviceHandle = 0x%x ", cou32DeviceHandle));
   trDeviceConnectionCountInfo rRetrievedDeviceConnectionCountInfo;

   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         rRetrievedDeviceConnectionCountInfo = itMapDevList->second.rDeviceConnectionCountInfo;
      } //if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle)

   return rRetrievedDeviceConnectionCountInfo;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceConnectionCountInfo
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceConnectionCountInfo(const t_U32 cou32DeviceHandle,
         trDeviceConnectionCountInfo &rfrNewDeviceConnectionCountInfo)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceConnectionCountInfo: cou32DeviceHandle = 0x%x ", cou32DeviceHandle));
   trEntireDeviceInfo rfrEntireDevInfo;

   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);

      if (NULL != m_poDevHistory)
      {
         t_Bool bDeviceExists = m_poDevHistory->bFindDevice(cou32DeviceHandle);

         if (m_mapDeviceInfoList.end() != itMapDevList)
         {
            itMapDevList->second.rDeviceConnectionCountInfo = rfrNewDeviceConnectionCountInfo;
            rfrEntireDevInfo = itMapDevList->second;
         }

         if (bDeviceExists != true)
         {
            ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceConnectionCountInfo: cou32DeviceHandle = 0x%x NOT found. So adding the deivce to History", cou32DeviceHandle));
            //! Adding values to History Db should be performed after releasing the lock for Device history as
            //! Db also acquires one more lock for DB operations
            m_poDevHistory->bAddtoHistorydb(rfrEntireDevInfo);

         }
         else
         {
            t_Bool bRetDevHistroy = m_poDevHistory->bSetDeviceConnectionCountInfo(cou32DeviceHandle,
                     rfrNewDeviceConnectionCountInfo);
            SPI_INTENTIONALLY_UNUSED(bRetDevHistroy);
         }
      }

      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  tenDeviceCategory spi_tclDeviceListHandler::enGetTechnologyPreference
 ***************************************************************************/
tenDeviceCategory spi_tclDeviceListHandler::enGetTechnologyPreference(const tenDeviceType coenDeviceType)
{
   //Reading From Datapool  if the TechnologyPreference From the Datapool is unknown then read it from XML
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetTechnologyPreference Entered with DeviceType %d", ETG_ENUM(DEVICE_TYPE,
            coenDeviceType)));
   t_Bool bReadStatus = false;
   tenDeviceCategory enTechnologyPreference = e8DEV_TYPE_UNKNOWN;
   switch (coenDeviceType)
   {
      case e8_ANDROID_DEVICE:
         if (NULL != m_poConnSettings)
         {
            bReadStatus = m_poConnSettings->bReadAndroidDeviceTechnologyPreferenceVal(enTechnologyPreference);
         }
         break;
      case e8_APPLE_DEVICE:
         if (NULL != m_poConnSettings)
         {
            bReadStatus = m_poConnSettings->bReadAppleDeviceTechnologyPreferenceVal(enTechnologyPreference);
         }
         break;
      default:
         ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetTechnologyPreference unknown device type"));
   }

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetTechnologyPreference bReadStatus %d", ETG_ENUM(BOOL, bReadStatus)));
   //If the TechnologyPreference is unknown
   if ((false == bReadStatus) || (e8DEV_TYPE_UNKNOWN == enTechnologyPreference))
   {
      //reading from the polixy.xml
      if (NULL != m_poConnSettings)
      {
         enTechnologyPreference = m_poConnSettings->enGetTechnologyPreference(coenDeviceType);
      }
   }
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetTechnologyPreference left with the TechnologyPreference %d", ETG_ENUM(DEVICE_CATEGORY,
            enTechnologyPreference)));
   return enTechnologyPreference;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceListHandler::vSetTechnologyPreference
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetTechnologyPreference(const tenDeviceType coenDeviceType,
         const tenDeviceCategory coenTechnologyPref)
{
   t_Bool bWriteStatus = false;
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetTechnologyPreference Entered with DeviceTyep %d and TechnologyPrefernce %d", ETG_ENUM(DEVICE_TYPE,
            coenDeviceType), ETG_ENUM(DEVICE_CATEGORY, coenTechnologyPref)));

   switch (coenDeviceType)
   {
      case e8_ANDROID_DEVICE:
         if (NULL != m_poConnSettings)
         {
            bWriteStatus = m_poConnSettings->bSetAndroidDeviceTechnologyPreference(coenTechnologyPref);
         }
         break;
      case e8_APPLE_DEVICE:
         if (NULL != m_poConnSettings)
         {
            bWriteStatus = m_poConnSettings->bSetAppleDeviceTechnologyPreference(coenTechnologyPref);
         }
         break;
      default:
         ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetTechnologyPreference unknown device type"));
   }
   SPI_INTENTIONALLY_UNUSED(bWriteStatus);
}

/***************************************************************************
 ** FUNCTION: t_Void vSetVendorProductID(const t_U32 cou32DeviceHandle,...)
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetVendorProductID(const t_U32 cou32DeviceHandle, const t_U32 cou32VendorID,
         const t_U32 cou32ProductID)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetVendorProductID Entered()"));
   if ((0 != cou32DeviceHandle) && (0 != cou32ProductID))
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if ((m_mapDeviceInfoList.end() != itMapDevList)
               && (cou32VendorID == itMapDevList->second.rDeviceInfo.u32VendorID))
      {
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::vSetVendorProductID: VendorID:%d,and Product ID:%d", cou32VendorID, cou32ProductID));
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      else
      {
         m_mapDeviceInfoList[cou32DeviceHandle].rDeviceInfo.u32VendorID = cou32VendorID;
         m_mapDeviceInfoList[cou32DeviceHandle].rDeviceInfo.u32ProductID = cou32ProductID;

         if ((e8_FACTORY_RESET != m_enSelectReason) && (NULL != m_poDevHistory))
         {
            m_poDevHistory->vSetVendorID(cou32DeviceHandle, cou32VendorID);
            m_poDevHistory->vSetProductID(cou32DeviceHandle, cou32ProductID);
         }

      }
      m_oDeviceListLock.vUnlock();
   }
}

/***************************************************************************
 ** FUNCTION: t_U32 u32GetVendorID(const t_U32 cou32DeviceHandle,...)
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetVendorID(const t_U32 cou32DeviceHandle)
{
   t_U32 u32VendorID = 0;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         u32VendorID = itMapDevList->second.rDeviceInfo.u32VendorID;
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::u32GetVendorID %d", u32VendorID));
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   }
   return u32VendorID;
}

/***************************************************************************
 ** FUNCTION: t_U32 u32GetProductID(const t_U32 cou32DeviceHandle,...)
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetProductID(const t_U32 cou32DeviceHandle)
{
   t_U32 u32ProductID = 0;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         u32ProductID = itMapDevList->second.rDeviceInfo.u32ProductID;
         ETG_TRACE_USR4(("spi_tclDeviceListHandler::u32ProductID %d", u32ProductID));
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   }
   return u32ProductID;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceListHandler::u32GetMaxIndex
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetMaxIndex()
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetMaxIndex: Entered()"));

   t_U32 u32MaxAccessIndex = 0;
   m_oDeviceListLock.s16Lock();
   for (auto itMapDeviceList = m_mapDeviceInfoList.begin(); itMapDeviceList != m_mapDeviceInfoList.end();
            itMapDeviceList++)
   {
      if (u32MaxAccessIndex < itMapDeviceList->second.u32AccessIndex)
      {
         u32MaxAccessIndex = itMapDeviceList->second.u32AccessIndex;
      }

      ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetMaxIndex: updated max Index is %d", u32MaxAccessIndex));
   }
   m_oDeviceListLock.vUnlock();

   return u32MaxAccessIndex;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetSessionTransport()
 ***************************************************************************/
tenSessionTransport spi_tclDeviceListHandler::enGetSessionTransportFromSerialNumber(const t_String coszSerialNumber)
{
   tenSessionTransport enSessionTransport = e8_SESSION_TRANSPORT_UNKNOWN;
   if (!coszSerialNumber.empty())
   {
      m_oDeviceListLock.s16Lock();
      for (auto itMapDevList = m_mapDeviceInfoList.begin(); itMapDevList != m_mapDeviceInfoList.end(); itMapDevList++)
      {
         if (coszSerialNumber == itMapDevList->second.rDeviceInfo.szSerialNumber)
         {
            enSessionTransport = itMapDevList->second.rDeviceInfo.enSessionTransport;
            break;
         }
      }
      m_oDeviceListLock.vUnlock();
   } //(!coszSerialNumber.empty())
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetSessionTransportFromSerialNumber:device has established session on %d transport", ETG_ENUM(SESSION_TRANSPORT,
            enSessionTransport)));
   return enSessionTransport;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceWhitelistedInfo()
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceWhitelistedInfo(const t_U32 cou32DeviceHandle,
         const t_Bool bIsDeviceWhiteListed)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vSetDeviceWhitelistedInfo: Device %d,  is %s", cou32DeviceHandle,
            (true == bIsDeviceWhiteListed) ? "whitelisted" : "Not whitelisted"));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         if (true == bIsDeviceWhiteListed)
         {
            itMapDevList->second.enIsDeviceWhiteListed = e8_DEVICE_WHITELISTED;
         }
         else
         {
            itMapDevList->second.enIsDeviceWhiteListed = e8_DEVICE_NOT_WHITELISTED;
         }
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bGetDeviceWhitelistedInfo()
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bGetDeviceWhitelistedInfo(const t_U32 cou32DeviceHandle)
{
   t_Bool bIsWhitelisted = false;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         if (e8_DEVICE_WHITELISTED == itMapDevList->second.enIsDeviceWhiteListed)
         {
            ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::bGetDeviceWhitelistedInfo: Device %d is White listed", cou32DeviceHandle));
            bIsWhitelisted = true;
         }
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle)
   return bIsWhitelisted;
}


/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetSessionIP
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetSessionIP(const t_U32 cou32DeviceHandle, t_String &rfrszSessionIP)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetSessionIP: cou32DeviceHandle = 0x%x  rfrszSessionIP = %s",
            cou32DeviceHandle, rfrszSessionIP.c_str()));
   if ((0 != cou32DeviceHandle) && (false == rfrszSessionIP.empty()))
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         //Update the Session IP in the trEntireDeviceInfo accordingly based on USB(IPv6) or WIFI(IPv4)
         if(e8_SESSION_TRANSPORT_USB == itMapDevList->second.rDeviceInfo.enSessionTransport)
         {
            itMapDevList->second.szUSBSessionIP = rfrszSessionIP;
         }
         else if(e8_SESSION_TRANSPORT_WIFI == itMapDevList->second.rDeviceInfo.enSessionTransport)
         {
            itMapDevList->second.szWifiSessionIP = rfrszSessionIP;
         }
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();

      //Need to update to device histor too
   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION: t_U32 szGetSessionIP(const t_U32 cou32DeviceHandle,...)
 ***************************************************************************/
t_String spi_tclDeviceListHandler::szGetSessionIP(const t_U32 cou32DeviceHandle)
{
   t_String szSessionIP;
   if(0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         if(e8_SESSION_TRANSPORT_USB == itMapDevList->second.rDeviceInfo.enSessionTransport)
         {
            szSessionIP = itMapDevList->second.szUSBSessionIP ;
         }
         else if(e8_SESSION_TRANSPORT_WIFI == itMapDevList->second.rDeviceInfo.enSessionTransport)
         {
            szSessionIP = itMapDevList->second.szWifiSessionIP ;
         }

         ETG_TRACE_USR4(("spi_tclDeviceListHandler::szGetSessionIP  DEviceHandle %d Session IP Address %s",itMapDevList->first, szSessionIP.c_str()));
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   }
   return szSessionIP;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::u32GetDeviceIDFromIPAddress
 ***************************************************************************/
t_U32 spi_tclDeviceListHandler::u32GetDeviceIDFromIPAddress(const t_String &coszIPAddress)
{

   t_U32 u32DeviceID = 0;
   if(false == coszIPAddress.empty())
   {
      m_oDeviceListLock.s16Lock();
      for (auto itMapDevList = m_mapDeviceInfoList.begin();
               itMapDevList != m_mapDeviceInfoList.end(); ++itMapDevList)
      {
         if( (coszIPAddress == itMapDevList->second.szWifiSessionIP) || (coszIPAddress == itMapDevList->second.szUSBSessionIP))
         {
            ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetDeviceIDFromIPAddress: Device found "));
            u32DeviceID = itMapDevList->second.rDeviceInfo.u32DeviceHandle;
            break;
         } // if (true == (itMapDevList->second).rDeviceInfo.szBTAddress)
      }
      m_oDeviceListLock.vUnlock();
   }

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::u32GetDeviceIDFromIPAddress: Device 0x%x has IP Address %s ", u32DeviceID,coszIPAddress.c_str()));
   return u32DeviceID;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceStatusInfo
 ***************************************************************************/
tenDeviceStatusInfo spi_tclDeviceListHandler::enGetDeviceStatusInfo(const trDeviceInfo &rfrDeviceInfo,
         const t_U32 cou32DeviceHandle)
{

   tenDeviceStatusInfo enDeviceStatusInfo = e8DEVICE_STATUS_NOT_KNOWN;
   tenDeviceCategory enDeviceCategory = e8DEV_TYPE_UNKNOWN;
   trDeviceSelectionInfo rDeviceSelectioninfo;
   vGetDeviceSelectionInfo(cou32DeviceHandle, rDeviceSelectioninfo);
   m_oDeviceListLock.s16Lock();

   auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
   if (itMapDevList != m_mapDeviceInfoList.end())
   {
      enDeviceCategory = enGetDeviceCategory(cou32DeviceHandle);

      //! Connection Types are not Equal
      if (rfrDeviceInfo.enDeviceConnectionType != (itMapDevList->second.rDeviceInfo.enDeviceConnectionType))
      {
         switch(enDeviceCategory)
         {
            case e8DEV_TYPE_DIPO:
            {
               if (e8USB_CONNECTED == rfrDeviceInfo.enDeviceConnectionType)
               {
                  enDeviceStatusInfo = enGetDipoDeviceStatusInfo(rfrDeviceInfo.rProjectionCapability.enCarplaySupport,
                           (itMapDevList->second).rDeviceInfo.rProjectionCapability.enCarplaySupport, rDeviceSelectioninfo.enDeviceSelectionState);
               }
               else if (e8WIRELESS_CONNECTED == rfrDeviceInfo.enDeviceConnectionType)
               {
                  enDeviceStatusInfo = enGetDipoDeviceStatusInfo(rfrDeviceInfo.rProjectionCapability.enCarplaySupport,
                           (itMapDevList->second).rDeviceInfo.rProjectionCapability.enCarplaySupport, rDeviceSelectioninfo.enDeviceSelectionState,
                           rfrDeviceInfo.bIsBTFunctionalityLimited);
               }
            }
            break;

            default :
               break;
         }
      }
      else
      {
         if(e8USB_CONNECTED == rfrDeviceInfo.enDeviceConnectionType)
         {
            //! Send DEVICE_CHANGED if the device name has changed or capability has changed
            enDeviceStatusInfo =
                     ((rfrDeviceInfo.szDeviceName != (itMapDevList->second).rDeviceInfo.szDeviceName)
                              || (rfrDeviceInfo.rProjectionCapability.enCarplaySupport
                                       != (itMapDevList->second.rDeviceInfo.rProjectionCapability.enCarplaySupport))) ?
                                                e8DEVICE_CHANGED : e8DEVICE_STATUS_NOT_KNOWN;
         }
         else if(e8WIRELESS_CONNECTED == rfrDeviceInfo.enDeviceConnectionType)
         {
            //! Send DEVICE_CHANGED if the device name has changed or capability has changed
            enDeviceStatusInfo = ((rfrDeviceInfo.szDeviceName != (itMapDevList->second).rDeviceInfo.szDeviceName) ||
                     (rfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport!=
                              (itMapDevList->second.rDeviceInfo.rProjectionCapability.enCarplayWirelessSupport)))
                                                           ? e8DEVICE_CHANGED: e8DEVICE_STATUS_NOT_KNOWN;

            if((e8DEVICE_STATUS_NOT_KNOWN == enDeviceStatusInfo) && (rfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport ==
                     (itMapDevList->second.rDeviceInfo.rProjectionCapability.enCarplayWirelessSupport)))
            {
               t_Bool bDeviceAddedForWireless = bIsDeviceAddedForWireless(rfrDeviceInfo);
               if(true == bDeviceAddedForWireless)
               {
                  if(false == (itMapDevList->second.rDeviceInfo.bIsBTFunctionalityLimited))
                  {
                     enDeviceStatusInfo = e8DEVICE_ADDED;
                  }
                  else
                  {
                     enDeviceStatusInfo = e8DEVICE_STATUS_NOT_KNOWN;
                  }
               }
            }
         }
      }
   }
   m_oDeviceListLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetDeviceStatusInfo left with Device Status Info as %d for the Device Handle %d", ETG_ENUM(DEVICE_STATUS_INFO, enDeviceStatusInfo), cou32DeviceHandle));
   return enDeviceStatusInfo;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceConnectionType()
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceConnectionType(const t_U32 cou32DeviceHandle,
         tenDeviceConnectionType enDeviceConnectionType)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceConnectionType: cou32DeviceHandle = 0x%x  Device Connection Type = %d", cou32DeviceHandle, ETG_ENUM(CONNECTION_TYPE,
            enDeviceConnectionType)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.rDeviceInfo.enDeviceConnectionType = enDeviceConnectionType;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } //  if (0 != cou32DeviceHandle)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceListHandler::bUpdateDeviceName
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bUpdateDeviceName(const t_U32 cou32DeviceHandle, t_String szDeviceName)
{
   t_Bool bUpdatedDeviceName = false;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         if (true == (itMapDevList->second).rDeviceInfo.szDeviceName.empty())
         {
            (itMapDevList->second).rDeviceInfo.szDeviceName = szDeviceName;
            bUpdatedDeviceName = true;
         }

      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   }

   ETG_TRACE_USR1((" spi_tclDeviceListHandler::bUpdateDeviceName Left with Updating Device Name: %d", ETG_ENUM(BOOL,
            bUpdatedDeviceName)));
   return bUpdatedDeviceName;
}


/***************************************************************************
 ** FUNCTION: t_Void spi_tclDeviceListHandler::vSetDeviceSelectionReason
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceSelectionReason(const t_U32 cou32DeviceHandle, tenSelectReason enSelectReason)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceSelectionReason: "
            "cou32DeviceHandle = 0x%x, Selection Reason = %d ",
            cou32DeviceHandle, enSelectReason));
   if (0 != cou32DeviceHandle)
   {
      m_enSelectReason = enSelectReason;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetLastActiveSessionTransport()
 ***************************************************************************/
tenSessionTransport spi_tclDeviceListHandler::enGetLastActiveSessionTransport(const t_U32 cou32DeviceHandle)
{
   tenSessionTransport enLastActiveSessionTransport = e8_SESSION_TRANSPORT_UNKNOWN;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (itMapDevList != m_mapDeviceInfoList.end())
      {
         enLastActiveSessionTransport = itMapDevList->second.enLastActiveSessionTransport;
         ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetLastActiveSessionTransport:device %d had established session on %d transport",
                  cou32DeviceHandle, ETG_ENUM(SESSION_TRANSPORT, itMapDevList->second.enLastActiveSessionTransport)));
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      else
      {
         ETG_TRACE_USR1((" spi_tclDeviceListHandler::enGetLastActiveSessionTransport Device Not Found"));
      }
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetLastActiveSessionTransport:device %d had established session on %d transport",
            cou32DeviceHandle, ETG_ENUM(SESSION_TRANSPORT, enLastActiveSessionTransport)));
   return enLastActiveSessionTransport;
}


/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::bIsDeviceAddedForWireless
 ***************************************************************************/
t_Bool spi_tclDeviceListHandler::bIsDeviceAddedForWireless(const trDeviceInfo& corfrDeviceInfo)
{
   t_Bool bDeviceAddedForWireless = false;

   switch(corfrDeviceInfo.enDeviceCategory)
   {
      case e8DEV_TYPE_DIPO:
         if ((e8SPI_SUPPORTED == corfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport)
                  && (true == corfrDeviceInfo.bIsBTFunctionalityLimited))
         {
            //! Handling Taping on HFP icon, SPI not supposed to start Selection
            bDeviceAddedForWireless = true;
         }
         else
         {
            bDeviceAddedForWireless = false;
         }

         break;
      default :
         break;
   }

   return bDeviceAddedForWireless;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDipoDeviceStatusInfo
 ***************************************************************************/
tenDeviceStatusInfo spi_tclDeviceListHandler::enGetDipoDeviceStatusInfo(tenSPISupport enReportedCarplaySupport,
         tenSPISupport enStoredCarplaySupport, tenDeviceSelectionState enDeviceSelectionState)
{
   tenDeviceStatusInfo enDeviceStatusInfo = e8DEVICE_STATUS_NOT_KNOWN;

   if (e8SPI_SUPPORTED == enStoredCarplaySupport)
   {
      if (e8SPI_SUPPORTED == enReportedCarplaySupport)
      {
         //This change is done for the useecase where MP reports the feasibility and SPI changes the connection state of the device
         // as connected if device is already connected over BT.For the usecase where A2DP/HFP paired/
         if ((e8_SELECTION_STATE_DEVICE_NOT_SELECTED == enDeviceSelectionState) ||
                  (e8_SELECTION_STATE_DEVICE_SELECTION_ERROR == enDeviceSelectionState)||
                  (e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR == enDeviceSelectionState))
         {
            enDeviceStatusInfo = e8DEVICE_ADDED;
         }
         else
         {
            enDeviceStatusInfo = e8DEVICE_STATUS_NOT_KNOWN;
         }
      }
      else if (e8SPI_NOTSUPPORTED == enReportedCarplaySupport)
      {
         enDeviceStatusInfo = e8DEVICE_CHANGED;
      }
   }
   else if ((e8SPI_NOTSUPPORTED == enStoredCarplaySupport) || (e8SPI_SUPPORT_UNKNOWN == enStoredCarplaySupport))
   {
      if (e8SPI_SUPPORTED == enReportedCarplaySupport)
      {
         enDeviceStatusInfo = e8DEVICE_ADDED;
      }
      else if (e8SPI_NOTSUPPORTED == enReportedCarplaySupport)
      {
         enDeviceStatusInfo = e8DEVICE_STATUS_NOT_KNOWN;
      }
   }

return enDeviceStatusInfo;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDipoDeviceStatusInfo
 ***************************************************************************/
tenDeviceStatusInfo spi_tclDeviceListHandler::enGetDipoDeviceStatusInfo(tenSPISupport enReportedCarplaySupport,
         tenSPISupport enStoredCarplaySupport, tenDeviceSelectionState enDeviceSelectionState,
         t_Bool bIsBTFunctionalityLimited)
{
   trDeviceSelectionInfo rDeviceSelectioninfo;
   tenDeviceStatusInfo enDeviceStatusInfo =e8DEVICE_STATUS_NOT_KNOWN;

   if (e8SPI_SUPPORTED == enStoredCarplaySupport)
   {
      if (e8SPI_SUPPORTED == enReportedCarplaySupport)
      {

         if ((true == bIsBTFunctionalityLimited) &&
                  ((e8_SELECTION_STATE_DEVICE_NOT_SELECTED == enDeviceSelectionState) ||
                           (e8_SELECTION_STATE_DEVICE_SELECTION_ERROR == enDeviceSelectionState)||
                           (e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR == enDeviceSelectionState)))
         {
            enDeviceStatusInfo = e8DEVICE_ADDED;
         }
         else
         {
            enDeviceStatusInfo = e8DEVICE_STATUS_NOT_KNOWN;
         }
      }
      else if (e8SPI_NOTSUPPORTED == enReportedCarplaySupport)
      {
         enDeviceStatusInfo = e8DEVICE_CHANGED;
      }
   }
   else if ((e8SPI_NOTSUPPORTED == enStoredCarplaySupport) || (e8SPI_SUPPORT_UNKNOWN == enStoredCarplaySupport))
   {
      if (e8SPI_SUPPORTED == enReportedCarplaySupport)
      {
         if(true == bIsBTFunctionalityLimited)
         {
            enDeviceStatusInfo = e8DEVICE_ADDED;
         }
         else
         {
            enDeviceStatusInfo = e8DEVICE_CHANGED;
         }
      }
      else if (e8SPI_NOTSUPPORTED == enReportedCarplaySupport)
      {
         enDeviceStatusInfo = e8DEVICE_STATUS_NOT_KNOWN;
      }
   }

return enDeviceStatusInfo;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceSwitchState
 ***************************************************************************/
tenUSBSwitchState spi_tclDeviceListHandler::enGetDeviceSwitchState(const t_U32 cou32DeviceHandle)
{
   tenUSBSwitchState enDeviceSwitchState = e8_DEVICESWITCHSTATE_UNKNONW;
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         enDeviceSwitchState = itMapDevList->second.enDeviceSwitchState;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle

   ETG_TRACE_USR1(("spi_tclDeviceListHandler::enGetDeviceSwitchState cou32DeviceHandle = 0x%x enDeviceSwitchState = %d", cou32DeviceHandle, ETG_ENUM(DEVICESWITCHSTATE,
            enDeviceSwitchState)));

   return enDeviceSwitchState;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceSwitchState
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vSetDeviceSwitchState(const t_U32 cou32DeviceHandle, tenUSBSwitchState enDeviceSwitchState)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vSetDeviceSwitchState cou32DeviceHandle = 0x%x enDeviceSwitchState = %d", cou32DeviceHandle, ETG_ENUM(DEVICESWITCHSTATE,
            enDeviceSwitchState)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (m_mapDeviceInfoList.end() != itMapDevList)
      {
         itMapDevList->second.enDeviceSwitchState = enDeviceSwitchState;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   } // if (0 != cou32DeviceHandle
}


/***************************************************************************
 ** FUNCTION:  spi_tclDeviceListHandler::vUpdateBTLimitedFunctionality
 ***************************************************************************/
t_Void spi_tclDeviceListHandler::vUpdateBTLimitedFunctionality(const t_U32 cou32DeviceHandle, t_Bool bIsDeviceBTFunctionalityLimited)
{
   ETG_TRACE_USR1(("spi_tclDeviceListHandler::vUpdateBTLimitedFunctionality entered with Device Handle %d with the BT limited Functionality as %d", cou32DeviceHandle, ETG_ENUM(BOO, bIsDeviceBTFunctionalityLimited)));
   if (0 != cou32DeviceHandle)
   {
      m_oDeviceListLock.s16Lock();
      auto itMapDevList = m_mapDeviceInfoList.find(cou32DeviceHandle);
      if (itMapDevList != m_mapDeviceInfoList.end())
      {
            (itMapDevList->second).rDeviceInfo.bIsBTFunctionalityLimited = bIsDeviceBTFunctionalityLimited;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_oDeviceListLock.vUnlock();
   }
}

//lint �restore
