/*!
 *******************************************************************************
 * \file             spi_tclDevHistory.cpp
 * \brief            Handles Device History
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Handles storage of Device History for SPI Devices
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 25.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 03.04.2019 |  Ashwini Savadi              | modification to open database on loadsettings
 and  close database on savesettings.

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <sstream>
#include "spi_tclDevHistory.h"
#include "DirHandler.h"
#include "FileHandler.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclDevHistory.cpp.trc.h"
#endif
#endif

#define SQLITE_OK           0
#define SQLITE_ROW         100

using namespace shl::io;
static const char* coczSqliteNotAnError = "not an error";
static const t_S8 scos8ReadUntilNullTermination = -1;
//! Columns in the DataBase table
static const t_U8 scou8ColDeviceHandle = 0;
static const t_U8 scou8ColDeviceName = 1;
static const t_U8 scou8ColDeviceCategory = 2;
static const t_U8 scou8ColDeviceModelName = 3;
static const t_U8 scou8ColDeviceManufacturerName = 4;
//static const t_U8 scou8ColDeviceConnectionStatus = 5;
static const t_U8 scou8ColDeviceConnectionType = 6;
static const t_U8 scou8ColMajorVersion = 7;
static const t_U8 scou8ColMinorVersion = 8;
static const t_U8 scou8ColDAPSupport = 9;
//static const t_U8 scou8ColSelectedDevice = 10;
static const t_U8 scou8ColDeviceUsed = 11;
static const t_U8 scou8ColBTAddress = 12;
static const t_U8 scou8ColAccessIndex = 13;
//static const t_U8 scou8ColUserDeslected = 14;
static const t_U8 scou8ColAAPSupport = 15;
static const t_U8 scou8ColMLSupport = 16;
static const t_U8 scou8ColCarplaySupport = 17;
static const t_U8 scou8ColCarlifeSupport = 18;
static const t_U8 scou8ColDevicetype = 19;
static const t_U8 scou8ColUserAuth = 20;
static const t_U8 scou8ColSerialNumber = 21;
static const t_U8 scou8ColrDeviceConnectionCountInfo = 22;
static const t_U8 scou8ColVendorID = 23;
static const t_U8 scou8ColProductID = 24;
static const t_U8 scou8ColLastActiveSessionTransport = 25;
static const t_U8 scou8FirstParameter = 1;

static Lock m_oDBOperationInProgress;
//Device history version needs to be changed when schema is modified.
static const t_U8 scou8Version = 12;


//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::spi_tclDevHistory
 ***************************************************************************/
spi_tclDevHistory::spi_tclDevHistory(spi_tclConnSettingsIntf* poConnSettings, SPIDatabaseMngrIntf* poDevHistorydb) :
         m_bisDBOpen(false), m_poConnSettings(poConnSettings), m_poDevHistorydb(poDevHistorydb)
{
   ETG_TRACE_USR1((" spi_tclDevHistory::spi_tclDevHistory() entered "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::~spi_tclDevHistory
 ***************************************************************************/
spi_tclDevHistory::~spi_tclDevHistory()
{
   ETG_TRACE_USR1((" spi_tclDevHistory::~spi_tclDevHistory() m_bisDBOpen = %d \n", ETG_ENUM(BOOL, m_bisDBOpen)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::bInitialize
 ***************************************************************************/
t_Bool spi_tclDevHistory::bInitialize()
{
   ETG_TRACE_USR1((" spi_tclDevHistory::bInitialize()"));
   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::bUnInitialize
 ***************************************************************************/
t_Bool spi_tclDevHistory::bUnInitialize()
{
   ETG_TRACE_USR1((" spi_tclDevHistory::bUnInitialize()"));
   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vLoadSettings
 ***************************************************************************/
t_Void spi_tclDevHistory::vLoadSettings()
{
   ETG_TRACE_USR1((" spi_tclDevHistory::vLoadSettings()"));
   m_oDBOperationInProgress.s16Lock();
   //! Fetch the project specific storage path
   t_String szStorageDir;
   if (NULL != m_poDevHistorydb)
   {
      m_poDevHistorydb->bInitialize();
   }
   if (NULL != m_poConnSettings)
   {
      m_poConnSettings->vGetPersistentStoragePath(szStorageDir);
   } // if(NULL != poConnSettings)
   ETG_TRACE_USR1(("spi_tclDevHistory()::szStorageDir =  %s  ", szStorageDir.c_str()));
   if (true == bCheckPathValidity(szStorageDir))
   {
      std::stringstream ssDatabasePath;
      ssDatabasePath << szStorageDir << "DeviceHistory.db";
      //!<stringstream>.str() returns a temporary string object that's destroyed at the end of the full expression.
      //!Hence to extend the lifetime of the string stream, a reference string is created instead
      //! of <stringstream>.str().c_str().
      const t_String rfszDatabasePath = ssDatabasePath.str();
      //! Open sqlite DataBase
      m_bisDBOpen = bOpenDatabase(rfszDatabasePath);
      t_Bool bVersionValidity = false;
      if (true == m_bisDBOpen)
      {
         bVersionValidity = bIsVersionCompatible();
      }
      //! Delete the incompatible database
      if (false == bVersionValidity)
      {
         vCloseDatabase();
         vDeleteDeviceHistory(rfszDatabasePath);
         //! Recreate database with new version
         m_bisDBOpen = bOpenDatabase(rfszDatabasePath);
         vSetVersion(scou8Version);
      }
      //! Create table
      if (false == bCreateDevHistoryTable())
      {
         ETG_TRACE_ERR(("Creation of Device History Table failed "));
      } //if(false == bCreateDevHistoryTable())
   }
   m_oDBOperationInProgress.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vSaveSettings
 ***************************************************************************/
t_Void spi_tclDevHistory::vSaveSettings()
{
   ETG_TRACE_USR1((" spi_tclDevHistory::vSaveSettings()"));
   m_oDBOperationInProgress.s16Lock();
   vCloseDatabase();
   m_oDBOperationInProgress.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vClearPrivateData
 ***************************************************************************/
t_Void spi_tclDevHistory::vClearPrivateData()
{
   ETG_TRACE_USR1((" spi_tclDevHistory::vClearPrivateData()"));

   //! Drop table
   if (false == bDropDevHistoryTable())
   {
      ETG_TRACE_ERR(("Drop of Device History Table failed "));
   } //if(false == bCreateDevHistoryTable())

   //! Create table
   if (false == bCreateDevHistoryTable())
   {
      ETG_TRACE_ERR(("Creation of Device History Table failed "));
   } //if(false == bCreateDevHistoryTable())
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::u32GetNumOfDevices
 ***************************************************************************/
t_U32 spi_tclDevHistory::u32GetNumOfDevices()
{
   t_U32 u32DBSize = 0;

   //! construct query to count number of records in database
   t_Char czQueryofRecords[] = "SELECT COUNT(*) FROM DeviceHistory";
   if (NULL != m_poDevHistorydb)
   {
      //! Prepare sqlite statement
      if (m_poDevHistorydb->SPI_DB_prepare_v2(czQueryofRecords, scos8ReadUntilNullTermination) == SQLITE_OK)
      {
         //! Step through the sqlite query
         //! SQLITE_ROW is returned if a row matches the query
         if (SQLITE_ROW == m_poDevHistorydb->SPI_DB_step())
         {
            // Fetch the number of records in database
            u32DBSize = m_poDevHistorydb->SPI_DB_column_int(0);
         }

         //! Finalize the sqlite statement
         m_poDevHistorydb->SPI_DB_finalize();
      } //if (sqlite3_prepare_v2(m_poDevHistorydb,

      //! Check for error returned from sqlite library
      t_String szError = m_poDevHistorydb->SPI_DB_errmsg();
      if (szError != coczSqliteNotAnError)
      {
         ETG_TRACE_ERR((" %s  ", szError.c_str()));
      } // if (szError != coczSqliteNotAnError)
   }

   ETG_TRACE_USR1(("spi_tclDevHistory::u32GetNumOfDevices  u32DbSize = %d", u32DBSize));
   return u32DBSize;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::bAddtoHistorydb
 ***************************************************************************/
t_Bool spi_tclDevHistory::bAddtoHistorydb(trEntireDeviceInfo& rfrDevInfo)
{
   ETG_TRACE_USR1(("spi_tclDevHistory::bAddtoHistorydb  u32DeviceID =0x%x", rfrDevInfo.rDeviceInfo.u32DeviceHandle));

   //! No need to check if Maintain DB results in success or failure
   bMaintainDbSize();
   t_Bool bRetVal = true;
   if (NULL != m_poDevHistorydb)
   {
      //! Use SQL mprintf instead of string stream to avoid conflicts in sql query
      //! while using special characters in strings(Ex Phone Name) such as ' " etc.

      t_Char *pczSQLQuery = m_poDevHistorydb->SPI_DB_mprintf("INSERT OR REPLACE INTO DeviceHistory VALUES(%d,'%q', %d,"
	                        " '%q','%q', %d, %d, %d, %d, %d, %d, %d,'%q', %d, %d, %d, %d,%d,%d,%d, %d,'%q', ?, %d, %d, %d)",
               rfrDevInfo.rDeviceInfo.u32DeviceHandle,
               rfrDevInfo.rDeviceInfo.szDeviceName.c_str(),
               rfrDevInfo.rDeviceInfo.enDeviceCategory,
               rfrDevInfo.rDeviceInfo.szDeviceModelName.c_str(),
               rfrDevInfo.rDeviceInfo.szDeviceManufacturerName.c_str(),
               rfrDevInfo.rDeviceInfo.enDeviceConnectionStatus,
               rfrDevInfo.rDeviceInfo.enDeviceConnectionType,
               rfrDevInfo.rDeviceInfo.rVersionInfo.u32MajorVersion,
               rfrDevInfo.rDeviceInfo.rVersionInfo.u32MinorVersion,
               rfrDevInfo.rDeviceInfo.bDAPSupport,
               rfrDevInfo.rDeviceInfo.bSelectedDevice,
               rfrDevInfo.rDeviceInfo.enDeviceUsageEnabled,
               rfrDevInfo.rDeviceInfo.szBTAddress.c_str(),
               rfrDevInfo.u32AccessIndex,
               rfrDevInfo.bIsUserDeselected,
               rfrDevInfo.rDeviceInfo.rProjectionCapability.enAndroidAutoSupport,
               rfrDevInfo.rDeviceInfo.rProjectionCapability.enMirrorlinkSupport,
	           rfrDevInfo.rDeviceInfo.rProjectionCapability.enCarplaySupport,
               rfrDevInfo.rDeviceInfo.rProjectionCapability.enCarlifeSupport,
               rfrDevInfo.rDeviceInfo.rProjectionCapability.enDeviceType,
               rfrDevInfo.rDeviceInfo.enUserAuthorizationStatus,
               rfrDevInfo.rDeviceInfo.szSerialNumber.c_str(),
               rfrDevInfo.rDeviceInfo.u32VendorID,
	           rfrDevInfo.rDeviceInfo.u32ProductID,
	           rfrDevInfo.enLastActiveSessionTransport);

      //! Prepare sqlite statement
      if (m_poDevHistorydb->SPI_DB_prepare_v2(pczSQLQuery, scos8ReadUntilNullTermination) == SQLITE_OK)
      {
         //! Step through the sqlite query
         //!bind the blob data. rDeviceConnectionCountInfo is simple structure without pointers.
         //!If we are using pointers are present insertion should be handled in different way.
         m_poDevHistorydb->SPI_DB_bind_blob(1,
                  &rfrDevInfo.rDeviceConnectionCountInfo,
                  sizeof(trDeviceConnectionCountInfo));
         m_poDevHistorydb->SPI_DB_step();
         //! Finalize the sqlite statement
         m_poDevHistorydb->SPI_DB_finalize();
      }

      //! Check for error returned from sqlite library
      t_String szError = m_poDevHistorydb->SPI_DB_errmsg();
      if (szError != coczSqliteNotAnError)
      {
         bRetVal = false;
         ETG_TRACE_ERR((" %s  ", szError.c_str()));
      } // if (szError != coczSqliteNotAnError)
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::bDeleteFromHistorydb
 ***************************************************************************/
t_Bool spi_tclDevHistory::bDeleteFromHistorydb(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1((" spi_tclDevHistory::bDeleteFromHistorydb cou32DeviceHandle = 0x%x ", cou32DeviceHandle));

   //! Construct string stream with the device info for adding to database
   std::ostringstream ssDeleteQuery;
   ssDeleteQuery << "DELETE FROM DeviceHistory WHERE u32DeviceHandle =" << cou32DeviceHandle;

   //! Send the query request
   return bExecuteQuery(ssDeleteQuery.str().c_str());
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::bGetDeviceHistoryFromdb
 ***************************************************************************/
t_Bool spi_tclDevHistory::bGetDeviceHistoryFromdb(std::vector<trEntireDeviceInfo>& rfrDeviceInfo,
         tenSelModePriority enPriority, tenDeviceCategory enDevTypePref, tenDeviceConnectionType enConnModePref)
{
   ETG_TRACE_USR1((" spi_tclDevHistory::bGetDeviceHistoryFromdb entered "));

   std::ostringstream ssSelectDevHistory;

   //! Assign the second preference for device selection
   tenDeviceConnectionType enConnSecondPref =
            (e8USB_CONNECTED == enConnModePref) ? e8WIRELESS_CONNECTED : e8USB_CONNECTED;

   tenDeviceCategory enDevSecondPref =
            (e8DEV_TYPE_MIRRORLINK == enDevTypePref) ? e8DEV_TYPE_DIPO : e8DEV_TYPE_MIRRORLINK;

   //! Construct Query based on priority( change the order of the database
   //! based on the priority)
   switch (enPriority)
   {
      case e8PRIORITY_DEVICELIST_HISTORY:
      {
         ETG_TRACE_USR4((" %s ", ssSelectDevHistory.str().c_str()));
         ssSelectDevHistory << "SELECT * FROM DeviceHistory ORDER BY u32AccessIndex,"
                  "case when enDeviceConnectionType=" << enConnModePref << " THEN 0 "
                  "when enDeviceConnectionType=" << enConnSecondPref << " THEN 1 end, "
                  "case when enDeviceCategory=" << enDevTypePref << " THEN 0 "
                  "when enDeviceCategory=" << enDevSecondPref << " THEN 1 end";
         break;
      }
      case e8PRIORITY_CONNMODE_PREFERENCE:
      {
         ETG_TRACE_USR4((" %s ", ssSelectDevHistory.str().c_str()));
         ssSelectDevHistory << "SELECT * FROM DeviceHistory ORDER BY " << "case when enDeviceConnectionType="
                  << enConnModePref << " THEN 0 "
                           "when enDeviceConnectionType=" << enConnSecondPref << " THEN 1 end, "
                           "u32AccessIndex,"
                           "case when enDeviceCategory=" << enDevTypePref << " THEN 0 "
                           "when enDeviceConnectionType=" << enDevSecondPref << " THEN 1 end";
         break;
      }

      case e8PRIORITY_DEVICETYPE_PREFERENCE:
      {
         ETG_TRACE_USR4((" %s ", ssSelectDevHistory.str().c_str()));
         ssSelectDevHistory << "SELECT * FROM DeviceHistory ORDER BY " << "case when enDeviceCategory=" << enDevTypePref
                  << " THEN 0 "
                           "when enDeviceCategory=" << enDevSecondPref << " THEN 1 end, "
                           "u32AccessIndex,"
                           "case when enDeviceConnectionType=" << enConnModePref << " THEN 0 "
                           "when enDeviceConnectionType=" << enConnSecondPref << " THEN 1 end";
         break;
      }

      default:
      {
         ETG_TRACE_ERR((" Unknown Priority for Selection "));
         break;
      }
   }   //switch (enPriority)

   return bPopulateDeviceList(rfrDeviceInfo, ssSelectDevHistory.str());
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::bFindDevice
 ***************************************************************************/
t_Bool spi_tclDevHistory::bFindDevice(t_U32 u32Key)
{
   t_Bool bRetFindkey = false;

   //! construct query to select all devices from database matching the key
   t_Char czFindKey[] = "SELECT * FROM DeviceHistory WHERE U32DeviceHandle = ?";
   if (NULL != m_poDevHistorydb)
   {
      //! Prepare sqlite statement
      if (m_poDevHistorydb->SPI_DB_prepare_v2(czFindKey, /* SQL statement, UTF-8 encoded */
      scos8ReadUntilNullTermination) == SQLITE_OK) /* Maximum length of ssFindKey in bytes. */
      {
         //! bind the integer arguments (prevents injection attack on database)
         m_poDevHistorydb->SPI_DB_bind_int(scou8FirstParameter, u32Key);

         //! Step through the sqlite query
         //! SQLITE_ROW is returned if a row matches the query
         if (SQLITE_ROW == m_poDevHistorydb->SPI_DB_step())
         {
            bRetFindkey = true;
         }

         //! Finalize the sqlite statement
         m_poDevHistorydb->SPI_DB_finalize();
      } //if (sqlite3_prepare_v2(m_poDevHistorydb,

      //! Check for error returned from sqlite library
      t_String szError = m_poDevHistorydb->SPI_DB_errmsg();
      if (szError != coczSqliteNotAnError)
      {
         bRetFindkey = true;
         ETG_TRACE_ERR((" %s  ", szError.c_str()));
      } // if (szError != coczSqliteNotAnError)
   }
   ETG_TRACE_USR1(("spi_tclDevHistory::bFindDevice:  u32Key = %d bRetFindkey = %d", u32Key, ETG_ENUM(BOOL, bRetFindkey)));
   return bRetFindkey;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vSetSelectedDevice
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetSelectedDevice(const t_U32 cou32DeviceHandle, t_Bool bIsDevSelected)
{
   //! Construct query to change bSelectedDevice
   //! column of the requested device
   std::ostringstream ssbIsDeviceSelected;
   ssbIsDeviceSelected << bIsDevSelected;
   vSetValue(cou32DeviceHandle, "bSelectedDevice", ssbIsDeviceSelected.str());

   ETG_TRACE_USR1(("spi_tclDevHistory::vSetSelectedDevice cou32DeviceHandle = 0x%x, "
            " bIsDevSelected = %d", cou32DeviceHandle, ETG_ENUM(BOOL, bIsDevSelected)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vSetDeviceName
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetDeviceName(const t_U32 cou32DeviceHandle, t_String &rfrszDeviceName)
{
   ETG_TRACE_USR1(("spi_tclDevHistory:: vSetDeviceName cou32DeviceHandle =0x%x,  rfrszDeviceName = %s ", cou32DeviceHandle, rfrszDeviceName.c_str()));
   //! Construct query to change DeviceName
   //! column of the requested device
   vSetValue(cou32DeviceHandle, "szDeviceName", rfrszDeviceName);
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vSetUserDeselectionFlag
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetUserDeselectionFlag(const t_U32 cou32DeviceHandle, t_Bool bState)
{
   ETG_TRACE_USR1(("spi_tclDevHistory:: vSetUserDeselectionFlag cou32DeviceHandle =0x%x,  bState = %d ", cou32DeviceHandle, ETG_ENUM(BOOL,
            bState)));
   //! Construct query to change bIsUserDeselected
   //! column of the requested device
   std::ostringstream ssbIsUserDeselected;
   ssbIsUserDeselected << bState;
   vSetValue(cou32DeviceHandle, "bIsUserDeselected", ssbIsUserDeselected.str());
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vSetBTAddress
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetBTAddress(const t_U32 cou32DeviceHandle, const t_String& corfrszBTAddress)
{
   ETG_TRACE_USR1(("spi_tclDevHistory::vSetBTAddress: DeviceHandle =0x%x, BTAddress = %s ", cou32DeviceHandle, corfrszBTAddress.c_str()));
   //! Construct query to change BTAddress column of the requested device
   vSetValue(cou32DeviceHandle, "szBTAddress", corfrszBTAddress);
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::u32GetSelectedDevice
 ***************************************************************************/
t_U32 spi_tclDevHistory::u32GetLastSelectedDevice()
{
   //! construct query to select all devices from database matching the key
   t_Char czGetLastSelectedDevice[] =
            "SELECT U32DeviceHandle, MAX(u32AccessIndex) FROM DeviceHistory WHERE  bIsUserDeselected = ?";
   t_U32 u32DeviceHandle = 0;

   if (NULL != m_poDevHistorydb)
   {
      //! Prepare sqlite statement
      if (m_poDevHistorydb->SPI_DB_prepare_v2(czGetLastSelectedDevice, scos8ReadUntilNullTermination) == SQLITE_OK)
      {
         //! bind the integer arguments (prevents injection attack on database)
         m_poDevHistorydb->SPI_DB_bind_int(scou8FirstParameter, 0);

         //! Step through the sqlite query
         //! SQLITE_ROW is returned if a row matches the query
         if (SQLITE_ROW == m_poDevHistorydb->SPI_DB_step())
         {
            // Fetch the device handle of the selected device
            u32DeviceHandle = m_poDevHistorydb->SPI_DB_column_int(scou8ColDeviceHandle);
         }

         //! Finalize the sqlite statement
         m_poDevHistorydb->SPI_DB_finalize();
      } //if (sqlite3_prepare_v2(m_poDevHistorydb,

      //! Check for error returned from sqlite library
      t_String szError = m_poDevHistorydb->SPI_DB_errmsg();
      if (szError != coczSqliteNotAnError)
      {
         u32DeviceHandle = 0;
         ETG_TRACE_ERR((" %s  ", szError.c_str()));
      } // if (szError != coczSqliteNotAnError)
   }
   ETG_TRACE_USR1((" spi_tclDevHistory::u32GetLastSelectedDevice = 0x%x ", u32DeviceHandle));
   return u32DeviceHandle;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vSaveDeviceList
 ***************************************************************************/
t_Void spi_tclDevHistory::vSaveDeviceList(std::map<t_U32, trEntireDeviceInfo> &m_mapDeviceList)
{
   /*lint -esym(40,bIsDeviceUsed)bIsDeviceUsed Undeclared identifier */
   /*lint -esym(40,second) second Undeclared identifier */

   ETG_TRACE_USR1((" spi_tclDevHistory::vSaveDeviceList entered "));
   std::map<t_U32, trEntireDeviceInfo>::iterator itMapList;
   for (itMapList = m_mapDeviceList.begin(); itMapList != m_mapDeviceList.end(); itMapList++)
   {
      if (true == itMapList->second.bIsDeviceUsed)
      {
         trEntireDeviceInfo trEntireDevInfo = itMapList->second;
         bAddtoHistorydb(trEntireDevInfo);
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::u32GetMaxAccessIndex
 ***************************************************************************/
t_U32 spi_tclDevHistory::u32GetMaxAccessIndex()
{

   //! construct query to select all devices from database matching the key
   t_Char czGetLastSelectedDevice[] = "SELECT U32DeviceHandle, MAX(u32AccessIndex) FROM DeviceHistory";

   t_U32 u32MaxAccessIndex = 0;

   if (NULL != m_poDevHistorydb)
   {
      //! Prepare sqlite statement
      if (m_poDevHistorydb->SPI_DB_prepare_v2(czGetLastSelectedDevice, scos8ReadUntilNullTermination) == SQLITE_OK)
      {
         //! Step through the sqlite query
         //! SQLITE_ROW is returned if a row matches the query
         if (SQLITE_ROW == m_poDevHistorydb->SPI_DB_step())
         {
            // Fetch the AccessIndex of last selected device
            u32MaxAccessIndex = m_poDevHistorydb->SPI_DB_column_int(1);

            ETG_TRACE_USR1((" spi_tclDevHistory::DeviceID =0x%x ", m_poDevHistorydb->SPI_DB_column_int(0)));

            ETG_TRACE_USR1((" spi_tclDevHistory::u32GetMaxAccessIndex = %d ", u32MaxAccessIndex));
         }

         //! Finalize the sqlite statement
         m_poDevHistorydb->SPI_DB_finalize();
      } //if (sqlite3_prepare_v2(m_poDevHistorydb,

      //! Check for error returned from sqlite library
      t_String szError = m_poDevHistorydb->SPI_DB_errmsg();
      if (szError != coczSqliteNotAnError)
      {
         u32MaxAccessIndex = 0;
         ETG_TRACE_ERR((" %s  ", szError.c_str()));
      } // if (szError != coczSqliteNotAnError)
   }
   ETG_TRACE_USR1((" spi_tclDevHistory::u32GetMaxAccessIndex = %d ", u32MaxAccessIndex));
   return u32MaxAccessIndex;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vSetDeviceUsagePreference
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetDeviceUsagePreference(const t_U32 cou32DeviceHandle, tenEnabledInfo enEnabledInfo)
{
   ETG_TRACE_USR1((" spi_tclDevHistory::vSetDeviceUsagePreference = 0x%x enEnabledInfo = %d ", cou32DeviceHandle, ETG_ENUM(ENABLED_INFO,
            enEnabledInfo)));
   //! Construct query to change DeviceUsagePreference
   //! column of the requested device
   std::ostringstream ssbIsDeviceUsageEnabled;
   tenEnabledInfo enDevEnabledInfo = enEnabledInfo;
   ssbIsDeviceUsageEnabled << enDevEnabledInfo;
   vSetValue(cou32DeviceHandle, "enDeviceUsageEnabled", ssbIsDeviceUsageEnabled.str());
}

/***************************************************************************
 *********************************PRIVATE **********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vDisplayDevHistorydb
 ***************************************************************************/
t_Bool spi_tclDevHistory::bCreateDevHistoryTable()
{
   ETG_TRACE_USR1((" bCreateDevHistoryTable entered "));

   //! Construct Query to create table for Device History
   t_String szDeviceHistoryTable =
            " CREATE TABLE IF NOT EXISTS DeviceHistory(                                            \
                        U32DeviceHandle            INT PRIMARY KEY     NOT NULL,                   \
                        szDeviceName               TEXT                NOT NULL,                   \
                        enDeviceCategory           INT,                                            \
                        szDeviceModelName          TEXT                NOT NULL,                   \
                        szDeviceMaunufacturerName  TEXT                NOT NULL,                   \
                        enDeviceConnectionStatus   INT,                                            \
                        enDeviceConnectionType     INT,                                            \
                        trVersionMajor             INT,                                            \
                        trVersionMinor             INT,                                            \
                        bDAPSupport                INT,                                            \
                        bSelectedDevice            INT,                                            \
                        enDeviceUsageEnabled       INT,                                            \
                        szBTAddress                TEXT                NOT NULL,                   \
                        u32AccessIndex             INT,                                            \
                        bIsUserDeselected          INT,                                            \
                        bAAPSupport                INT,                                            \
                        bMLSupport                 INT,                                            \
                        bCarplaySupport            INT,                                            \
                        bCarlifeSupport            INT,                                            \
                        enDeviceType               INT,                                            \
                        enUserAuthorizationStatus  INT,                                            \
                        szSerialNumber             TEXT                NOT NULL,                   \
                        rDeviceConnectionCountInfo BLOB,                                           \
                        u32VendorID                INT,                                            \
                        u32ProductID               INT,                                            \
                        enLastActiveSessionTransport         INT                                   \
                        )";

   //! Send the query request
   return bExecuteQuery(szDeviceHistoryTable.c_str());
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vDisplayDevHistorydb
 ***************************************************************************/
t_Void spi_tclDevHistory::vDisplayDevHistorydb()
{
   ETG_TRACE_USR1((" vDisplayDevHistorydb entered "));

   //! Construct Query to fetch device history table
   t_String szDisplayQuery = "SELECT * FROM DeviceHistory;";

   //! Send the query request
   bExecuteQuery(szDisplayQuery, &spi_tclDevHistory::s32DisplayDevHistoryCb);
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::bExecuteQuery
 ***************************************************************************/
t_Bool spi_tclDevHistory::bExecuteQuery(t_String szQuery, fs32SqliteCallback* pfSqliteCb)
{
   SPI_INTENTIONALLY_UNUSED(pfSqliteCb);
   ETG_TRACE_USR1((" bExecuteQuery entered "));
   t_Bool bRetExecQuery = false;
   t_S32 u32ExecRet = -1;

   if (true == m_bisDBOpen)
   {
      //! Pointer to sqlite error message
      char *pczErrorMsg = NULL;
      if (NULL != m_poDevHistorydb)
      {
         //! Execute the requested query
         u32ExecRet = m_poDevHistorydb->SPI_DB_exec(szQuery, pczErrorMsg);
      }

      //! Check for error returned from sqlite library
      if ((u32ExecRet != SQLITE_OK) && (NULL != pczErrorMsg))
      {
         ETG_TRACE_ERR(("bExecuteQuery::SQL error: %s", pczErrorMsg));
         if (NULL != m_poDevHistorydb)
         {
            //! free memory allocated by sqlite for error message
            m_poDevHistorydb->SPI_DB_free(pczErrorMsg);
         }
      }
      else
      {
         bRetExecQuery = true;
      } //if ((u32ExecRet != SQLITE_OK) && (NULL != pczErrorMsg))
   } // if (true == m_bisDBOpen)
   ETG_TRACE_USR4((" spi_tclDevHistory::bExecuteQuery bRetExecQuery = %d  ", ETG_ENUM(BOOL, bRetExecQuery)));
   return bRetExecQuery;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vSetValue
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetValue(const t_U32 cou32DeviceHandle, t_String szColName, t_String szColValue)
{
   ETG_TRACE_USR2(("spi_tclDevHistory::vSetValue: cou32DeviceHandle =0x%x, szColName = %s ", cou32DeviceHandle, szColName.c_str()));
   ETG_TRACE_USR2(("szColValue =%s ", szColValue.c_str()));

   t_Char *pczSQLQuery = NULL;

   if (NULL != m_poDevHistorydb)
   {
      pczSQLQuery = m_poDevHistorydb->SPI_DB_mprintf("UPDATE DeviceHistory SET %q = '%q' WHERE u32DeviceHandle = %d",
               szColName.c_str(),
               szColValue.c_str(),
               cou32DeviceHandle);
   }

   //! Send the query request
   if (NULL != pczSQLQuery)
   {
      bExecuteQuery(pczSQLQuery);
      if (NULL != m_poDevHistorydb)
      {
         m_poDevHistorydb->SPI_DB_free(pczSQLQuery);
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::bPopulateDeviceList
 ***************************************************************************/
t_Bool spi_tclDevHistory::bPopulateDeviceList(std::vector<trEntireDeviceInfo> &rfrDeviceInfo, t_String szDevListQuery)
{
   ETG_TRACE_USR1((" spi_tclDevHistory::bPopulateDeviceList entered "));
   t_Bool bRetVal = true;
   if (NULL != m_poDevHistorydb)
   {
      //! Prepare sqlite statement
      if (m_poDevHistorydb->SPI_DB_prepare_v2(szDevListQuery.c_str(), scos8ReadUntilNullTermination) == SQLITE_OK)
      {
         //! Step through the sqlite query
         //! SQLITE_ROW is returned if a row matches the query
         while (SQLITE_ROW == m_poDevHistorydb->SPI_DB_step())
         {
            //! construct device info from the database
            trEntireDeviceInfo rDevInfo;
            rDevInfo.rDeviceInfo.u32DeviceHandle = m_poDevHistorydb->SPI_DB_column_int(scou8ColDeviceHandle);
            rDevInfo.rDeviceInfo.szDeviceName = (const char*) m_poDevHistorydb->SPI_DB_column_text(scou8ColDeviceName);
            rDevInfo.rDeviceInfo.enDeviceCategory =
                     (tenDeviceCategory) m_poDevHistorydb->SPI_DB_column_int(scou8ColDeviceCategory);
            rDevInfo.rDeviceInfo.szDeviceModelName =
                     (const char*) m_poDevHistorydb->SPI_DB_column_text(scou8ColDeviceModelName);
            rDevInfo.rDeviceInfo.szDeviceManufacturerName =
                     (const char*) m_poDevHistorydb->SPI_DB_column_text(scou8ColDeviceManufacturerName);
            rDevInfo.rDeviceInfo.enDeviceConnectionStatus = e8DEV_NOT_CONNECTED;
            rDevInfo.rDeviceInfo.enDeviceConnectionType =
                     (tenDeviceConnectionType) m_poDevHistorydb->SPI_DB_column_int(scou8ColDeviceConnectionType);
            rDevInfo.rDeviceInfo.rVersionInfo.u32MajorVersion =
                     m_poDevHistorydb->SPI_DB_column_int(scou8ColMajorVersion);
            rDevInfo.rDeviceInfo.rVersionInfo.u32MinorVersion =
                     m_poDevHistorydb->SPI_DB_column_int(scou8ColMinorVersion);
            rDevInfo.rDeviceInfo.bDAPSupport = m_poDevHistorydb->SPI_DB_column_int(scou8ColDAPSupport);
            rDevInfo.rDeviceInfo.bSelectedDevice = 0;
            rDevInfo.rDeviceInfo.enDeviceUsageEnabled =
                     (tenEnabledInfo) m_poDevHistorydb->SPI_DB_column_int(scou8ColDeviceUsed);
            rDevInfo.rDeviceInfo.szBTAddress = (const char*) m_poDevHistorydb->SPI_DB_column_text(scou8ColBTAddress);
            rDevInfo.u32AccessIndex = m_poDevHistorydb->SPI_DB_column_int(scou8ColAccessIndex);
            //! Donot restore user device deselection info across powercycles
            rDevInfo.bIsUserDeselected = false;
            rDevInfo.rDeviceInfo.rProjectionCapability.enAndroidAutoSupport =
                     (tenSPISupport) m_poDevHistorydb->SPI_DB_column_int(scou8ColAAPSupport);
            rDevInfo.rDeviceInfo.rProjectionCapability.enMirrorlinkSupport =
                     (tenSPISupport) m_poDevHistorydb->SPI_DB_column_int(scou8ColMLSupport);

            rDevInfo.rDeviceInfo.rProjectionCapability.enCarplaySupport =
            		(tenSPISupport) m_poDevHistorydb->SPI_DB_column_int(scou8ColCarplaySupport);
            rDevInfo.rDeviceInfo.rProjectionCapability.enCarlifeSupport =
            		(tenSPISupport) m_poDevHistorydb->SPI_DB_column_int(scou8ColCarlifeSupport);

            rDevInfo.rDeviceInfo.rProjectionCapability.enDeviceType =
                     (tenDeviceType) m_poDevHistorydb->SPI_DB_column_int(scou8ColDevicetype);
            rDevInfo.rDeviceInfo.enUserAuthorizationStatus =
                     (tenUserAuthorizationStatus) m_poDevHistorydb->SPI_DB_column_int(scou8ColUserAuth);
            rDevInfo.rDeviceInfo.szSerialNumber =
                     (const char*) m_poDevHistorydb->SPI_DB_column_text(scou8ColSerialNumber);
            rDevInfo.rDeviceConnectionCountInfo =
                     *((trDeviceConnectionCountInfo *) m_poDevHistorydb->SPI_DB_column_blob(scou8ColrDeviceConnectionCountInfo));
            rDevInfo.rDeviceInfo.u32VendorID = m_poDevHistorydb->SPI_DB_column_int(scou8ColVendorID);
            rDevInfo.rDeviceInfo.u32ProductID = m_poDevHistorydb->SPI_DB_column_int(scou8ColProductID);
            rDevInfo.enLastActiveSessionTransport = (tenSessionTransport) m_poDevHistorydb->SPI_DB_column_int(scou8ColLastActiveSessionTransport);
            rfrDeviceInfo.push_back(rDevInfo);
         }      // while (SQLITE_ROW == sqlite3_step(poStatement))

         //! Finalize the sqlite statement
         m_poDevHistorydb->SPI_DB_finalize();
      }      //if (sqlite3_prepare_v2(m_poDevHistorydb,...

      //! Check for error returned from sqlite library
      t_String szError = m_poDevHistorydb->SPI_DB_errmsg();
      if (szError != coczSqliteNotAnError)
      {
         bRetVal = false;
         ETG_TRACE_ERR(("spi_tclDevHistory::bPopulateDeviceLis: Error: %s  ", szError.c_str()));
      } // if (szError != coczSqliteNotAnError)
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDevHistory::bMaintainDbSize
 ***************************************************************************/
t_Bool spi_tclDevHistory::bMaintainDbSize()
{
   t_Bool bRetVal = false;
   t_U32 u32DBSize = u32GetNumOfDevices();

   //! Fetch the project specific Device History size
   t_U32 u32DevHistSize = 0;
   if (NULL != m_poConnSettings)
   {
      u32DevHistSize = m_poConnSettings->u32GetDeviceHistorySize();
   } // if(NULL != poConnSettings)

   if (u32DBSize >= u32DevHistSize)
   {
      //! construct query to delete the oldest entry
      std::ostringstream ssQueryDelEntry;
      ssQueryDelEntry
               << "DELETE FROM DeviceHistory WHERE u32AccessIndex=(SELECT MIN(u32AccessIndex) FROM DeviceHistory)";
      bRetVal = bExecuteQuery(ssQueryDelEntry.str().c_str());
   }

   ETG_TRACE_USR1(("bMaintainDbSize  u32DBSize = %d u32DevHistSize = %d bRetVal = %d  ", u32DBSize, u32DevHistSize, ETG_ENUM(BOOL,
            bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  static t_S32 spi_tclDevHistory::s32DisplayDevHistoryCb
 ***************************************************************************/
t_S32 spi_tclDevHistory::s32DisplayDevHistoryCb(t_Void *pvNotUsed, t_S32 s32ArgCountMax, t_Char **ppczArgv,
         t_Char **ppczColName)
{
   SPI_INTENTIONALLY_UNUSED(pvNotUsed);
   ETG_TRACE_USR1((" Device History "));
   //! Check for negative argument count as it is a signed value
   if (s32ArgCountMax > 0)
   {
      for (t_S32 s32ArgCount = 0; s32ArgCount < s32ArgCountMax; s32ArgCount++)
      {
         //! Print the values returned by sqlite
         if ((NULL != ppczColName) && (NULL != ppczArgv))
         {
            std::ostringstream ssDispValue;
            ssDispValue << ppczColName[s32ArgCount] << "\t";
            ssDispValue << (ppczArgv[s32ArgCount] ? ppczArgv[s32ArgCount] : "NULL");
            ETG_TRACE_USR4((" %s  ", ssDispValue.str().c_str()));
         } // if ((NULL != ppczColName) && (NULL != ppczArgv))
      } //for (t_S32 s32ArgCount = 0; s32ArgCount < s32ArgCountMax; s32ArgCount++)
      ETG_TRACE_USR4((""));
   } // if (s32ArgCountMax > 0)
   return 0;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDevHistory::bCheckPathValidity
 ***************************************************************************/
t_Bool spi_tclDevHistory::bCheckPathValidity(t_String &rfrszStorageDir)
{
   t_Bool bRetVal = true;

   //! Check if the directory is valid
   DirHandler oDirHandler(rfrszStorageDir.c_str());
   if (false == oDirHandler.bIsValid())
   {
      ETG_TRACE_USR2(("spi_tclDevHistory::bCheckPathValidity: Directory not"
               " present. Creating .. %s", rfrszStorageDir.c_str()));
      //! Try creating directory . (Recursive creation not attempted)
      DirHandler oNewDir("/");
      bRetVal = oNewDir.bMkDir(rfrszStorageDir.c_str());
   }

   ETG_TRACE_USR1((" spi_tclDevHistory::bCheckPathValidity bRetVal=%d ", ETG_ENUM(BOOL, bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDevHistory::bOpenDatabase
 ***************************************************************************/
t_Bool spi_tclDevHistory::bOpenDatabase(const t_String &corfszDatabasePath)
{
   t_Bool bRetval = false;
   t_S32 s32DbOpeRes = -1;
   if (NULL != m_poDevHistorydb)
   {
      //! Open sqlite DataBase
      s32DbOpeRes = m_poDevHistorydb->SPI_DB_open(corfszDatabasePath);
   }
   if (SQLITE_OK == s32DbOpeRes)
   {
      bRetval = true;
   } //if (SQLITE_OK == s32DbOpeRes)
   else
   {
      if (NULL != m_poDevHistorydb)
      {
         ETG_TRACE_ERR(("Opening database failed:m_poDevHistorydb "
                  "Error = %s ", m_poDevHistorydb->SPI_DB_errmsg()));
         m_poDevHistorydb->SPI_DB_close();
      }
      bRetval = false;
   }
   ETG_TRACE_USR1((" spi_tclDevHistory::bOpenDatabase entered bRetval = %d", ETG_ENUM(BOOL, bRetval)));
   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDevHistory::bCloseDatabase
 ***************************************************************************/
t_Void spi_tclDevHistory::vCloseDatabase()
{
   ETG_TRACE_USR1((" spi_tclDevHistory::vCloseDatabase entered \n"));
   //! Close the sqlite database
   if ((true == m_bisDBOpen) && (NULL != m_poDevHistorydb))
   {
      m_poDevHistorydb->bUnInitialize();
   } //if (true == m_bisDBOpen)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDevHistory::vDeleteDeviceHistory
 ***************************************************************************/
t_Void spi_tclDevHistory::vDeleteDeviceHistory(const t_String &corfszDatabasePath)
{
   ETG_TRACE_USR1((" spi_tclDevHistory::vDeleteDeviceHistory at path %s ", corfszDatabasePath.c_str()));
   spi::io::FileHandler oFilehandler(corfszDatabasePath.c_str(), spi::io::SPI_EN_REMOVE);
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDevHistory::bIsVersionCompatible
 ***************************************************************************/
t_Bool spi_tclDevHistory::bIsVersionCompatible()
{
   t_U32 u32ExistingDatabaseVersion = u32GetVersion();
   //! Compare the new database version with the database version read on the target
   t_Bool bRetval = (scou8Version == u32ExistingDatabaseVersion);
   ETG_TRACE_USR1((" spi_tclDevHistory::bIsVersionCompatible? %d : Existing database Version = %d,"
            "New database version =%d \n", ETG_ENUM(BOOL, bRetval), u32ExistingDatabaseVersion, scou8Version));
   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDevHistory::u32GetVersion
 ***************************************************************************/
t_U32 spi_tclDevHistory::u32GetVersion()
{
   t_U32 u32Version = 0;

   //! construct query to count number of records in database
   t_Char csVersionQuery[] = "PRAGMA user_version;";
   if (NULL != m_poDevHistorydb)
   {
      //! Prepare sqlite statement
      if (m_poDevHistorydb->SPI_DB_prepare_v2(csVersionQuery, scos8ReadUntilNullTermination) == SQLITE_OK)
      {
         //! Step through the sqlite query
         //! SQLITE_ROW is returned if a row matches the query
         if (SQLITE_ROW == m_poDevHistorydb->SPI_DB_step())
         {
            // Fetch the version
            u32Version = m_poDevHistorydb->SPI_DB_column_int(0);
         }

         //! Finalize the sqlite statement
         m_poDevHistorydb->SPI_DB_finalize();
      } //if (sqlite3_prepare_v2(m_poDevHistorydb,

      //! Check for error returned from sqlite library
      t_String szError = m_poDevHistorydb->SPI_DB_errmsg();
      if (szError != coczSqliteNotAnError)
      {
         ETG_TRACE_ERR((" %s  ", szError.c_str()));
      } // if (szError != coczSqliteNotAnError)
   }
   ETG_TRACE_USR1((" spi_tclDevHistory::u32GetVersion %d ", u32Version));
   return u32Version;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDevHistory::vSetVersion
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetVersion(t_U32 u32Version)
{
   ETG_TRACE_USR1((" spi_tclDevHistory::vSetVersion %d ", u32Version));
   t_Char *pczSQLQuery = NULL;

   if (NULL != m_poDevHistorydb)
   {
      //! construct query to set the user version for database
      pczSQLQuery = m_poDevHistorydb->SPI_DB_mprintf("PRAGMA user_version=%d;", u32Version);
   }

   //! Send the query request
   if (NULL != pczSQLQuery)
   {
      bExecuteQuery(pczSQLQuery);
      if (NULL != m_poDevHistorydb)
      {
         m_poDevHistorydb->SPI_DB_free(pczSQLQuery);
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDevHistory::vSetDeviceCategory
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetDeviceCategory(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCategory)
{
   ETG_TRACE_USR1(("spi_tclDevHistory::vSetDeviceCategory: DeviceHandle =0x%x, DeviceCategory = %d ", cou32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY,
            enDeviceCategory)));
   std::ostringstream ssenDeviceCategory;
   ssenDeviceCategory << enDeviceCategory;
   //! Construct query to change Device Category column of the requested device
   vSetValue(cou32DeviceHandle, "enDeviceCategory", ssenDeviceCategory.str());
}

/***************************************************************************
 ** FUNCTION:  spi_tclDevHistory::vSetDeviceConnectionCountInfo
 ***************************************************************************/
t_Bool spi_tclDevHistory::bSetDeviceConnectionCountInfo(const t_U32 cou32DeviceHandle,
         trDeviceConnectionCountInfo &rfrDeviceConnectionCountInfo)
{
   ETG_TRACE_USR1((" spi_tclDevHistory::bSetDeviceConnectionCountInfo DeviceHandle = %d ", cou32DeviceHandle));
   t_Bool bRetVal = true;

   if (NULL != m_poDevHistorydb)
   {
      t_Char *pczSQLQuery =
               m_poDevHistorydb->SPI_DB_mprintf("UPDATE DeviceHistory SET rDeviceConnectionCountInfo = ? WHERE U32DeviceHandle = %d",
                        cou32DeviceHandle);

      if (NULL == pczSQLQuery)
         return false;
      //! Prepare sqlite statement
      if (m_poDevHistorydb->SPI_DB_prepare_v2(pczSQLQuery, scos8ReadUntilNullTermination) == SQLITE_OK)
      {
         //! Step through the sqlite query
         //! SQLITE_ROW is returned if a row matches the query

         m_poDevHistorydb->SPI_DB_bind_blob(1, &rfrDeviceConnectionCountInfo, sizeof(trDeviceConnectionCountInfo));
         m_poDevHistorydb->SPI_DB_step();
         //! Finalize the sqlite statement
         m_poDevHistorydb->SPI_DB_finalize();
      }

      //! Check for error returned from sqlite library
      t_String szError = m_poDevHistorydb->SPI_DB_errmsg();
      if (szError != coczSqliteNotAnError)
      {
         bRetVal = false;
         ETG_TRACE_ERR((" %s  ", szError.c_str()));
      } // if (szError != coczSqliteNotAnError)
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDevHistory::vSetVendorID
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetVendorID(const t_U32 cou32DeviceHandle, const t_U32 cou32VendorID)
{
   ETG_TRACE_USR1(("spi_tclDevHistory::vSetVendorID: DeviceHandle =0x%x, VendorID = %d ", cou32DeviceHandle, cou32VendorID));
   std::ostringstream sscou32VendorID;
   sscou32VendorID << cou32VendorID;
   //! Construct query to change Vendor ID column of the requested device
   vSetValue(cou32DeviceHandle, "u32VendorID", sscou32VendorID.str());
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDevHistory::vSetProductID
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetProductID(const t_U32 cou32DeviceHandle, const t_U32 cou32ProductID)
{
   ETG_TRACE_USR1(("spi_tclDevHistory::vSetProductID: DeviceHandle =0x%x, ProductID = %d ", cou32DeviceHandle, cou32ProductID));
   std::ostringstream sscou32ProductID;
   sscou32ProductID << cou32ProductID;
   //! Construct query to change Product ID column of the requested device
   vSetValue(cou32DeviceHandle, "u32ProductID", sscou32ProductID.str());
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDevHistory::bDropDevHistoryTable
 ***************************************************************************/
t_Bool spi_tclDevHistory::bDropDevHistoryTable()
{
   ETG_TRACE_USR1((" bDropDevHistoryTable entered "));

   //! Construct Query to drop table for Device History
   t_String szDeviceHistoryTable = "DROP TABLE IF EXISTS DeviceHistory";

   //! Send the query request
   return bExecuteQuery(szDeviceHistoryTable.c_str());
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDevHistory::vSetLastActiveSessionTransport
 ***************************************************************************/
t_Void spi_tclDevHistory::vSetLastActiveSessionTransport(const t_U32 cou32DeviceHandle, tenSessionTransport enSessionTransport)
{
    ETG_TRACE_USR1(("spi_tclDevHistory::vSetLastActiveSessionTransport: DeviceHandle =0x%x, Session Transport = %d ",
             cou32DeviceHandle, ETG_ENUM(SESSION_TRANSPORT,enSessionTransport)));
    std::ostringstream ssenSessionTransport;
    ssenSessionTransport << enSessionTransport;
    //! Construct query to change Session Transport column of the requested device
    vSetValue(cou32DeviceHandle, "enLastActiveSessionTransport", ssenSessionTransport.str());
}

//lint -restore
