/*!
 *******************************************************************************
 * \file             spi_tclConnMngr.cpp
 * \brief            Core implementation for Connection Management
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Core implementation for Connection Management
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 13.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 06.04.2014 |  Ramya Murthy                | Initialisation sequence implementation
 31.07.2014 |  Ramya Murthy (RBEI/ECP2)    | SPI feature configuration via LoadSettings()
 10.12.2014 |  Shihabudheen P M            | Changed for blocking device usage 
 preference updates during
 select/deselect is in progress.
 05.11.2014 |  Ramya Murthy (RBEI/ECP2)    | Added callback for Application metadata.
 06.05.2015 |  Tejaswini HB                |Lint Fix
 12.05.2015 |  Shiva Kumar G (RBEI/ECP2)   | Changes to continue with Craplay initialization,
 Even if the Mirrorlink initialization is failed.
 25.01.2016 |  Rachana L Achar             | Logiscope improvements
 01.01.2017 |  Noopur R K                  | Added changes for Delete device interface implementation
 18.07.2017 |  Noopur R K                  | Added vSetGeneralRestrictions method
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "sstream"
#include "FileHandler.h"
#include "spi_tclConnMngrResp.h"
#include "spi_tclConnMngr.h"
#include "spi_tclMediator.h"

///**********Hack*/
//#include "spi_tclExtCmdAppleDiscoverer.h"
//#include "spi_tclExtCompManager.h"
///**********Hack*/

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclConnMngr.cpp.trc.h"
#endif
#endif

static const t_U32 scou32InvalidDeviceHandle = 0;

using namespace spi::io;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::spi_tclConnMngr
 ***************************************************************************/
spi_tclConnMngr::spi_tclConnMngr(spi_tclConnMngrResp *poRespInterface, spi_tclDeviceListIntf *poDeviceList,
         std::map<tenDeviceCategory, spi_tclConnection*> &poConnections, spi_tclMediatorIntf* poMediator,
         spi_tclDiscoveryDataIntf *poDiscoverIntf) :
                  spi_tclSelectionIntf(e32COMPID_CONNECTIONMANAGER),
                  m_poDeviceListHandler(poDeviceList),
                  m_poConnMngrResp(poRespInterface),
                  m_mapoConnHandlers(poConnections),
                  m_bIsDevSelectorBusy(false),
                  m_poMediator(poMediator),
                  m_u32DeviceUnderSelection(0),
                  m_poDiscovrerDataIntf(poDiscoverIntf),
                  m_poConnSettings(NULL)
{
   ETG_TRACE_USR1((" spi_tclConnMngr::spi_tclConnMngr() entered "));
   for (t_U8 u8Index = 0; u8Index < e8_CONN_INDEX_SIZE; u8Index++)
   {
      m_bSPIState[u8Index] = true;
   }

   SPI_NORMAL_ASSERT(NULL == m_poConnMngrResp);
   SPI_NORMAL_ASSERT(NULL == m_poDeviceListHandler);
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::~spi_tclConnMngr
 ***************************************************************************/
spi_tclConnMngr::~spi_tclConnMngr()
{
   ETG_TRACE_USR1((" spi_tclConnMngr::~spi_tclConnMngr()  entered "));

   m_poConnMngrResp = NULL;
   m_poDeviceListHandler = NULL;
   m_bIsDevSelectorBusy = false;
   m_poConnSettings = NULL;

   for (t_U8 u8Index = 0; u8Index < e8_CONN_INDEX_SIZE; u8Index++)
   {
      m_bSPIState[u8Index] = false;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConnMngr::bInitialize
 ***************************************************************************/
t_Bool spi_tclConnMngr::bInitialize()
{
   t_Bool bRetConnInit = true;

   bRetConnInit = (NULL != m_poDeviceListHandler);
   m_poLockConnections.s16Lock();
   for (auto itmap = m_mapoConnHandlers.begin(); itmap != m_mapoConnHandlers.end(); itmap++)
   {
      if (NULL != itmap->second)
      {
         bRetConnInit = (itmap->second->bInitializeConnection()) || (bRetConnInit);
      }
   }
   m_poLockConnections.vUnlock();

   vRegisterCallbacks();

   ETG_TRACE_USR4(("spi_tclConnMngr::bInitialize bRetConnInit= %d ", ETG_ENUM(BOOL, bRetConnInit)));
   return bRetConnInit;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConnMngr::bUnInitialize
 ***************************************************************************/
t_Bool spi_tclConnMngr::bUnInitialize()
{
   ETG_TRACE_USR1((" spi_tclConnMngr::bUnInitialize() entered "));
   m_poLockConnections.s16Lock();
   for (auto itmap = m_mapoConnHandlers.begin(); itmap != m_mapoConnHandlers.end(); itmap++)
   {
      if (NULL != itmap->second)
      {
         (itmap->second->vUnInitializeConnection());
      }
   }
   m_poLockConnections.vUnlock();
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnMngr::vLoadSettings()
 ***************************************************************************/
t_Void spi_tclConnMngr::vLoadSettings()
{
   trSpiFeatureSupport rfrSpiFeatureSupport;
   if (NULL != m_poConnSettings)
   {
      m_poConnSettings->vGetSpiFeatureSupport(rfrSpiFeatureSupport);
      ETG_TRACE_USR1(("[DESC] spi_tclConnMngr::vLoadSettings() entered : Feautres supported Android Auto =%d, Mirrorlink = %d, Carplay = %d", ETG_ENUM(BOOL,
               rfrSpiFeatureSupport.bAndroidAutoSupported()), ETG_ENUM(BOOL,
               rfrSpiFeatureSupport.bMirrorLinkSupported()), ETG_ENUM(BOOL, rfrSpiFeatureSupport.bDipoSupported())));
      //! Store the Feature support settings
      m_rSPIFeatureSupport = rfrSpiFeatureSupport;

      //! Set the vehicle information
      trHeadUnitInfo rfrHeadUnitInfo;
      m_poConnSettings->vGetHeadUnitInfo(rfrHeadUnitInfo);

      tenCertificateType enCertificateType = m_poConnSettings->enGetCertificateType();

      m_poLockConnections.s16Lock();
      for (auto itmap = m_mapoConnHandlers.begin(); itmap != m_mapoConnHandlers.end(); itmap++)
      {
         if (NULL != itmap->second)
         {
            itmap->second->vSetConnectionsettingsInstance(m_poConnSettings);
            itmap->second->vOnLoadSettings(rfrHeadUnitInfo, enCertificateType);
            //load the general restrictions info for every technology here.
            ETG_TRACE_USR1(("[DESC] spi_tclConnMngr::vLoadSettings Setting General Restriction info for all technologies"));
            t_U16 u16GeneralRestrictionInfo = m_poConnSettings->u16GetGeneralRestrictionsInfo(itmap->first);
            itmap->second->vSetGeneralRestrictions(u16GeneralRestrictionInfo);
         }
      }
      m_poLockConnections.vUnlock();
   } //if (NULL != poConnSettings)

   if (NULL != m_poDeviceListHandler)
   {
      t_Bool bRetDevListInit = m_poDeviceListHandler->bRestoreDeviceList();
      SPI_INTENTIONALLY_UNUSED(bRetDevListInit);
      m_poDeviceListHandler->vSetSpiFeatureSupport(rfrSpiFeatureSupport);
   } //if (NULL != m_poDeviceListHandler)
   vLoadSPITechnologySettings();

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnMngr::vLoadSettingsCompleted()
 ***************************************************************************/
t_Void spi_tclConnMngr::vLoadSettingsCompleted()
{
   ETG_TRACE_USR1(("[FUNC] spi_tclConnMngr::vLoadSettingsCompleted() entered"));

   m_poLockConnections.s16Lock();
   for (auto itmap = m_mapoConnHandlers.begin(); itmap != m_mapoConnHandlers.end(); itmap++)
   {
      if (NULL != itmap->second)
      {
         itmap->second->vOnLoadSettingsCompleted();
      }
   }
   m_poLockConnections.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnMngr::vSaveSettings()
 ***************************************************************************/
t_Void spi_tclConnMngr::vSaveSettings()
{
   ETG_TRACE_USR1((" spi_tclConnMngr::vSaveSettings() entered "));
   //! Currently device list need not be saved again on save settings as
   //! devices will be dynamically added to list on selection

   m_poLockConnections.s16Lock();
   for (auto itmap = m_mapoConnHandlers.begin(); itmap != m_mapoConnHandlers.end(); itmap++)
   {
      if (NULL != itmap->second)
      {
         itmap->second->vOnSaveSettings();
      }
   }
   m_poLockConnections.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vRestoreSettings
 ***************************************************************************/
t_Void spi_tclConnMngr::vRestoreSettings()
{
   t_String szStoragePath;
   trHeadUnitInfo rfrHeadUnitInfo;
   if (NULL != m_poConnSettings)
   {
      m_poConnSettings->vGetPersistentStoragePath(szStoragePath);

      //! Re read the configuration data. This is required to update fields like vehicle ID
      m_poConnSettings->vClearPrivateData();
      m_poConnSettings->vGetHeadUnitInfo(rfrHeadUnitInfo);
      //! Reload the settings
      tenCertificateType enCertificateType = m_poConnSettings->enGetCertificateType();
      m_poLockConnections.s16Lock();
      for (auto itmap = m_mapoConnHandlers.begin(); itmap != m_mapoConnHandlers.end(); itmap++)
      {
         if (NULL != itmap->second)
         {
            itmap->second->vOnLoadSettings(rfrHeadUnitInfo, enCertificateType);
            //load the general restrictions info for every technology here.
            ETG_TRACE_USR1(("[DESC] spi_tclConnMngr::vLoadSettings Setting General Restriction info for all technologies"));
            t_U16 u16GeneralRestrictionInfo = m_poConnSettings->u16GetGeneralRestrictionsInfo(itmap->first);
            itmap->second->vSetGeneralRestrictions(u16GeneralRestrictionInfo);
         }
      }
      m_poLockConnections.vUnlock();
   }

   if (NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->vClearPrivateData();
   }

   tenEnabledInfo enCurrentProjSetting = e8USAGE_DISABLED;
   tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE;
   //! Reread the datapool and update SPI setting states
   bGetDevProjUsage(e8DEV_TYPE_DIPO, enCurrentProjSetting, enDeviceType);
   bSetDevProjUsage(e8DEV_TYPE_DIPO, enCurrentProjSetting, enDeviceType);
   bGetDevProjUsage(e8DEV_TYPE_ANDROIDAUTO, enCurrentProjSetting, enDeviceType);
   bSetDevProjUsage(e8DEV_TYPE_ANDROIDAUTO, enCurrentProjSetting, enDeviceType);
   bGetDevProjUsage(e8DEV_TYPE_MIRRORLINK, enCurrentProjSetting, enDeviceType);
   bSetDevProjUsage(e8DEV_TYPE_MIRRORLINK, enCurrentProjSetting, enDeviceType);
   bGetDevProjUsage(e8DEV_TYPE_MYSPIN, enCurrentProjSetting, enDeviceType);
   bSetDevProjUsage(e8DEV_TYPE_MYSPIN, enCurrentProjSetting, enDeviceType);
   bGetDevProjUsage(e8DEV_TYPE_CARLIFE, enCurrentProjSetting, enDeviceType);
   bSetDevProjUsage(e8DEV_TYPE_CARLIFE, enCurrentProjSetting, enDeviceType);
   bGetDevProjUsage(e8DEV_TYPE_ONCAR, enCurrentProjSetting, enDeviceType);
   bSetDevProjUsage(e8DEV_TYPE_ONCAR, enCurrentProjSetting, enDeviceType);

   //! Send Device list change
   if (NULL != m_poConnMngrResp)
   {
      m_poConnMngrResp->vPostDeviceStatusInfo(scou32InvalidDeviceHandle, e8UNKNOWN_CONNECTION, e8DEVICE_REMOVED);
   } //if (NULL != m_poConnMngrResp)

}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vGetDeviceList
 ***************************************************************************/
/*t_Void spi_tclConnMngr::vGetDeviceList(
 std::vector<trDeviceInfo>& rfvecDeviceInfoList)
 {
 rfvecDeviceInfoList.clear();

 if (NULL != m_poDeviceListHandler)
 {
 m_poDeviceListHandler->vGetDeviceList(cou32MAX_DEVICEHANDLE, rfvecDeviceInfoList);
 } //if (NULL != m_poDeviceListHandler)
 }*/

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vGetDeviceInfoList
 ***************************************************************************/
t_Void spi_tclConnMngr::vGetDeviceInfoList(const trUserContext &corfrUsrCntxt)
{
   std::vector<trDeviceInfo> vecDeviceInfoList;

   if (NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->vGetDeviceList(0xFFFFFFFF, vecDeviceInfoList);
   } //if (NULL != m_poDeviceListHandler)

   if (NULL != m_poConnMngrResp)
   {
      m_poConnMngrResp->vPostDeviceInfoList(vecDeviceInfoList, corfrUsrCntxt);
   } //if (NULL != m_poConnMngrResp)
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vGetDeviceList
 ***************************************************************************/
t_Void spi_tclConnMngr::vGetDeviceList(const t_U32 cou32DeviceHandle, const trUserContext &corfrUsrCntxt)
{
   std::vector<trDeviceInfo> vecDeviceInfoList;

   if (NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->vGetDeviceList(cou32DeviceHandle, vecDeviceInfoList);
   } //if (NULL != m_poDeviceListHandler)

   if (NULL != m_poConnMngrResp)
   {
      m_poConnMngrResp->vPostDeviceList(vecDeviceInfoList, corfrUsrCntxt);
   } //if (NULL != m_poConnMngrResp)
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vOnSelectDevice
 ***************************************************************************/
t_Void spi_tclConnMngr::vSelectDevice(const trSelectDeviceRequest &rfrSelDeviceRequest)
{
   ETG_TRACE_USR1(("spi_tclConnMngr:: vOnSelectDevice : cou32DeviceHandle =%d enDevSelReq = %d  SelectReason = %d", rfrSelDeviceRequest.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            rfrSelDeviceRequest.m_enDevConnReq), ETG_ENUM(SELECT_REASON, rfrSelDeviceRequest.m_enSelectionReason)));

   m_u32DeviceUnderSelection = rfrSelDeviceRequest.m_u32DeviceHandle;

   if (NULL != m_poDeviceListHandler)
   {
      tenDeviceCategory enDevCat = rfrSelDeviceRequest.m_enDevCategory;
      m_rDeviceSelectionRequest = rfrSelDeviceRequest;
      m_rDeviceSelectionRequest.m_enDevCategory =
               (e8DEVCONNREQ_SELECT == rfrSelDeviceRequest.m_enDevConnReq) ? enDevCat : e8DEV_TYPE_UNKNOWN;

      m_poLockConnections.s16Lock();
      std::map<tenDeviceCategory, spi_tclConnection*>::iterator itmap = m_mapoConnHandlers.find(enDevCat);
      tenDeviceType enDeviceType = m_poDeviceListHandler->enGetDeviceType(rfrSelDeviceRequest.m_u32DeviceHandle);
      trDeviceSwitchInfo rDevSwitchInfo;
      if (NULL != m_poDiscovrerDataIntf)
      {
         m_poDiscovrerDataIntf->vGetDeviceSwitchInfo(rfrSelDeviceRequest.m_u32DeviceHandle, rDevSwitchInfo);
      }

      if ((m_mapoConnHandlers.end() != itmap) && (NULL != itmap->second))
      {
         itmap->second->vSetDeviceSwitchInfo(rDevSwitchInfo);
         //Send the connection information only during selection and not during Deselection
         if (e8DEVCONNREQ_SELECT == rfrSelDeviceRequest.m_enDevConnReq)
         {
            trDeviceConnectionCountInfo rCurrDeviceConnectionCountInfo =
                     m_poDeviceListHandler->rGetDeviceConnectionCountInfo(rfrSelDeviceRequest.m_u32DeviceHandle);
            itmap->second->vUpdateDeviceConnectionCountInfo(rCurrDeviceConnectionCountInfo);
         }
         itmap->second->vOnSelectDevice(rfrSelDeviceRequest.m_u32DeviceHandle,
                  rfrSelDeviceRequest.m_enDevConnType,
                  rfrSelDeviceRequest.m_enDevConnReq,
                  e8USAGE_DISABLED,
                  e8USAGE_DISABLED,
                  rfrSelDeviceRequest.m_enSelectionReason,
                  rfrSelDeviceRequest.m_corUsrCntxt,
                  enDeviceType);
      }
      m_poLockConnections.vUnlock();
      //! on device deselection, result is posted directly to HMI because
      //! device disconnection status might go before the inactive status for
      //! carplay when reverse role switch is requested.
      if (e8DEVCONNREQ_DESELECT == rfrSelDeviceRequest.m_enDevConnReq)
      {
         //! Set Currently selected device to default value if select device
         //! request is received for device deselection
         m_poDeviceListHandler->vSetDeviceSelection(rfrSelDeviceRequest.m_u32DeviceHandle, false);
         //! Send Device list change
         if (NULL != m_poConnMngrResp)
         {
            m_poConnMngrResp->vPostDeviceStatusInfo(rfrSelDeviceRequest.m_u32DeviceHandle,
                     rfrSelDeviceRequest.m_enDevConnType,
                     e8DEVICE_CHANGED);
         } //if (NULL != m_poConnMngrResp)
      }
   } //if (NULL != m_poDeviceListHandler)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnMngr::vSelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclConnMngr::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vSelectDeviceResult : cou32DeviceHandle =%d "
            "enDevSelReq = %d enErrorCode = %d", corfrSelectReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            corfrSelectReq.m_enDevConnReq), ETG_ENUM(ERROR_CODE, enErrorCode)));

   tenDeviceConnectionStatus enConnStatus = e8DEV_NOT_CONNECTED;
   tenResponseCode enRespCode = (e8NO_ERRORS == enErrorCode) ? e8SUCCESS : e8FAILURE;
   if (NULL != m_poDeviceListHandler)
   {
      enConnStatus = m_poDeviceListHandler->enGetDeviceConnStatus(corfrSelectReq.m_u32DeviceHandle);

      if ((e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq) && (e8NO_ERRORS == enErrorCode))
      {
         //! Add selected device to history
         //! Store the currently selected device
         m_poDeviceListHandler->vSetDeviceSelection(corfrSelectReq.m_u32DeviceHandle, true);
         m_poDeviceListHandler->vAddDeviceToHistory(corfrSelectReq.m_u32DeviceHandle);
      } //if (NULL != m_poDeviceListHandler)
      m_poLockConnections.s16Lock();
      std::map<tenDeviceCategory, spi_tclConnection*>::iterator itmap =
               m_mapoConnHandlers.find(corfrSelectReq.m_enDevCategory);
      spi_tclConnection* poConnection = (m_mapoConnHandlers.end() != itmap) ? itmap->second : NULL;
      m_poLockConnections.vUnlock();

      tenDeviceType enDeviceType = m_poDeviceListHandler->enGetDeviceType(corfrSelectReq.m_u32DeviceHandle);

      if (NULL != poConnection)
      {
         poConnection->vOnSelectDeviceResult(corfrSelectReq.m_u32DeviceHandle,
                  corfrSelectReq.m_enDevConnReq,
                  enRespCode,
                  corfrSelectReq.m_enDevCategory,
                  corfrSelectReq.m_enSelectionReason,
                  enDeviceType);
      }

      //! Set the flag if Device selection fails so that further auto selections are not triggered
      t_Bool bIsError = false;
      if (e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq)
      {
         bIsError = (e8FAILURE == enRespCode)
                  && ((e8SELECTION_FAILED == enErrorCode) || (e8UNSUPPORTED_OPERATION == enErrorCode)
                           || (e8UNKNOWN_ERROR == enErrorCode));
         m_poDeviceListHandler->vSetSelectError(corfrSelectReq.m_u32DeviceHandle, bIsError);
      }

      //! Send Device list change
      if (NULL != m_poConnMngrResp)
      {
         m_poConnMngrResp->vPostDeviceStatusInfo(corfrSelectReq.m_u32DeviceHandle,
                  corfrSelectReq.m_enDevConnType,
                  e8DEVICE_CHANGED);
      } //if (NULL != m_poConnMngrResp)
   }
   SPI_INTENTIONALLY_UNUSED(enConnStatus);
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::bSetDevProjUsage
 ***************************************************************************/
t_Bool spi_tclConnMngr::bSetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo enSPIState,
         tenDeviceType enDeviceType)
{
   ETG_TRACE_USR1(("spi_tclConnMngr:: bSetDevProjUsage: Changing the overall setting for %d to State = %d ", ETG_ENUM(DEVICE_CATEGORY,
            enSPIType), ETG_ENUM(ENABLED_INFO, enSPIState)));

   //! To be used in future (If Mirrorlink/DiPo on/off fails)
   t_Bool bSetState = false;
   m_poLockConnections.s16Lock();
   std::map<tenDeviceCategory, spi_tclConnection*>::iterator itmap = m_mapoConnHandlers.find(enSPIType);
   spi_tclConnection* poConnection = (m_mapoConnHandlers.end() != itmap) ? itmap->second : NULL;
   m_poLockConnections.vUnlock();
   if (NULL != poConnection)
   {
      bSetState = poConnection->bSetDevProjUsage(enSPIState);
   }

   if (NULL != m_poDeviceListHandler)
   {
      bSetState = (m_poDeviceListHandler->bSetDevProjUsage(enSPIType, enSPIState, enDeviceType)) && (bSetState);
   }
   //! Apply selection strategy when Mirrorlink/Carplay setting is turned ON
   if (e8USAGE_ENABLED == enSPIState)
   {
      if (NULL != m_poMediator)
      {
         m_poMediator->vApplySelectionStrategy(enSPIType);
      }
   }

   return bSetState;
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vGetBTAddress
 ***************************************************************************/
t_Void spi_tclConnMngr::vGetBTAddress(const t_U32 cou32DeviceHandle, t_String &rfszBTAddress) const
{
   if (NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->vGetBTAddress(cou32DeviceHandle, rfszBTAddress);
   } //if(NULL != m_poDeviceListHandler)
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConnMngr::vSetDeviceUsagePreference
 ***************************************************************************/
t_Void spi_tclConnMngr::vSetDeviceUsagePreference(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCategory,
         tenEnabledInfo enUsagePref, const trUserContext &rfrUsrCntxt)
{
   ETG_TRACE_USR1((" spi_tclConnMngr::vSetDeviceUsagePreference: cou32DeviceHandle =0x%x,"
            " enDeviceCategory =%d, enUsagePref=%d ", cou32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY, enDeviceCategory), ETG_ENUM(ENABLED_INFO,
            enUsagePref)));

   tenErrorCode enErrorCode = e8NO_ERRORS;

   //! if cou32DeviceHandle = 0xFFFFFFFF, Overall setting for the specified
   //! service is changed
   if ((cou32MAX_DEVICEHANDLE == cou32DeviceHandle) && (NULL != m_poDeviceListHandler))
   {
      if (true == m_bIsDevSelectorBusy)
      {
         enErrorCode = e8RESOURCE_BUSY;
      }
      else
      {
         t_U32 u32SelectedDevice = m_poDeviceListHandler->u32GetSelectedDevice();
         tenDeviceCategory enSelectedDeviceCategory = enGetDeviceCategory(u32SelectedDevice);
         //! check if dev projection usage was set to off during a session
         if ((0 != u32SelectedDevice) && (enUsagePref == e8USAGE_DISABLED)
                  && (enDeviceCategory == enSelectedDeviceCategory))
         {
            //! Send Device deselection to SPI components if device projection
            //! setting is disabled during a mirrorlink/carplay session
            if (NULL != m_poMediator)
            {
               /* Deselection due to SetDeviceUsagePref(DISABLED) for wireless devices shouldn't trigger BT Reconnection during de-selection as User disables setting.
                Hence Deselect reason is explicitly set to HMI_TRGGER. if the reason is UNKNOWN, this will be equivalent to WiFI off on Phone or interference cases*/
               m_poMediator->vPostAutoDeviceSelection(u32SelectedDevice, e8DEVCONNREQ_DESELECT, e8_REASON_HMI_TRIGGER);
            } // if (NULL != poMediator)
         }
         bSetDevProjUsage(enDeviceCategory, enUsagePref, e8_UNKNOWN_DEVICE);
      }
   }
   //! To set the device usage preference flag
   else
   {
      if (NULL != m_poDeviceListHandler)
      {
         enDeviceCategory = m_poDeviceListHandler->enGetDeviceCategory(cou32DeviceHandle);
         if ((0 == cou32DeviceHandle) || (false == m_poDeviceListHandler->bIsDeviceValid(cou32DeviceHandle)))
         {
            enErrorCode = e8INVALID_DEV_HANDLE;
         }
         else if (true == m_bIsDevSelectorBusy)
         {
            enErrorCode = e8OPERATION_REJECTED;
         }
         else
         {
            m_poDeviceListHandler->vSetDeviceUsagePreference(cou32DeviceHandle, enUsagePref);

            //go for deselection
            //Check for selected device, whether the technology is disabled or not
            t_U32 u32SelectedDevice = m_poDeviceListHandler->u32GetSelectedDevice();
            if ((cou32DeviceHandle == u32SelectedDevice) && (enUsagePref == e8USAGE_DISABLED))
            {
               /* Deselection due to SetDeviceUsagePref(DISABLED) for wireless devices shouldn't trigger BT Reconnection during de-selection as User disables setting.
                Hence Deselect reason is explicitly set to HMI_TRGGER. if the reason is UNKNOWN, this will be equivalent to WiFI off on Phone or interference cases*/
               if (NULL != m_poMediator)
               {
                  m_poMediator->vPostAutoDeviceSelection(u32SelectedDevice,
                           e8DEVCONNREQ_DESELECT,
                           e8_REASON_HMI_TRIGGER);
               }
            }
            //! Send Device list change
            if (NULL != m_poConnMngrResp)
            {
               m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle, e8USB_CONNECTED, e8DEVICE_CHANGED);
            } //  if (NULL != m_poConnMngrResp)
         } // if ((0 == cou32DeviceHandle) || ...
      } //if (NULL != m_poDeviceListHandler)
   } //if(cou32MAX_DEVICEHANDLE == cou32DeviceHandle)

   if (NULL != m_poConnMngrResp)
   {
      m_poConnMngrResp->vPostDeviceUsagePrefResult(cou32DeviceHandle,
               enErrorCode,
               enDeviceCategory,
               enUsagePref,
               rfrUsrCntxt);
   } //if (NULL != m_poConnMngrResp)
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConnMngr::bGetDeviceUsagePreference
 ***************************************************************************/
t_Bool spi_tclConnMngr::bGetDeviceUsagePreference(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCategory,
         tenEnabledInfo& rfenUsagePref) const
{
   t_Bool bRetVal = false;
   if (cou32MAX_DEVICEHANDLE == cou32DeviceHandle)
   {
      //! Currently no error is posted if fetching Device Projection Usage fails
      bRetVal = bGetDevProjUsage(enDeviceCategory, rfenUsagePref, e8_UNKNOWN_DEVICE);
   }
   else
   {
      if ((NULL != m_poDeviceListHandler) && (0 != cou32DeviceHandle))
      {
         bRetVal = true;
         m_poDeviceListHandler->vGetDeviceUsagePreference(cou32DeviceHandle, rfenUsagePref);
      } // if ((NULL != m_poDeviceListHandler) && (0 != cou32DeviceHandle))
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vSetSelectError
 ***************************************************************************/
t_Void spi_tclConnMngr::vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory, t_U16 u16GeneralRestrInfo)
{
   ETG_TRACE_USR1((" spi_tclConnMngr::vSetGeneralRestriction: enDeviceCategory - %d,"
            "General Restriction info = %d", ETG_ENUM(DEVICE_CATEGORY, enDeviceCategory), u16GeneralRestrInfo));

   m_poLockConnections.s16Lock();
   std::map<tenDeviceCategory, spi_tclConnection*>::iterator itmap = m_mapoConnHandlers.find(enDeviceCategory);
   spi_tclConnection* poConnection = (m_mapoConnHandlers.end() != itmap) ? itmap->second : NULL;
   m_poLockConnections.vUnlock();
   if (NULL != poConnection)
   {
      poConnection->vSetGeneralRestrictions(u16GeneralRestrInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vSetSelectError
 ***************************************************************************/
t_Void spi_tclConnMngr::vSetSelectError(const t_U32 cou32DeviceHandle, t_Bool bIsError)
{
   if (NULL != m_poDeviceListHandler)
   {
      //! Set the Error flag to prevent further automatic selections
      m_poDeviceListHandler->vSetSelectError(cou32DeviceHandle, bIsError);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vUpdateSelectionState
 ***************************************************************************/
t_Void spi_tclConnMngr::vUpdateSelectionState(const t_U32 cou32DeviceHandle, tenDeviceSelectionState enSelState, tenDeviceConnectionType enDeviceConnectionType)
{
    ETG_TRACE_USR1(("spi_tclConnMngr::vUpdateSelectionState: Device ID = %d, Selection state = %d ",
             cou32DeviceHandle, ETG_ENUM( SELECTION_STATE, enSelState)));
    vOnSetSelectionProgressState(enSelState);
    m_bIsDevSelectorBusy = (e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS == enSelState)
             || (e8_SELECTION_STATE_DEVICE_DESELECTION_INPROGRESS == enSelState);
    if(NULL != m_poConnMngrResp)
    {
        m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle, enDeviceConnectionType, e8DEVICE_CHANGED);
    }
}

/**************************************************************************
 ** FUNCTION   : tVoid spi_tclConnMngr:: vSetTechnologyPreference()
 ***************************************************************************/
t_Void spi_tclConnMngr::vSetTechnologyPreference(const t_U32 cou32DeviceHandle, const tenDeviceType coenDeviceType,
         tenDeviceCategory enTechnologyPreference, tenErrorCode &rfenErrorCode) const
{

   if ((cou32MAX_DEVICEHANDLE == cou32DeviceHandle))
   {
      ETG_TRACE_USR1(("spi_tclConnMngr::vSetTechnologyPreference: "
               "per device setting with the DeviceType %d", ETG_ENUM(DEVICE_TYPE, coenDeviceType)));
      if (NULL != m_poDeviceListHandler)
      {
         m_poDeviceListHandler->vSetTechnologyPreference(coenDeviceType, enTechnologyPreference);
         rfenErrorCode = e8NO_ERRORS;
      }

   }
   else if (NULL != m_poDeviceListHandler)
   {
      if (cou32DeviceHandle == m_poDeviceListHandler->u32GetSelectedDevice())
      {
         ETG_TRACE_ERR(("[ERR] spi_tclConnMngr::vSetTechnologyPreference : Ignoring request as device is already selected "));
         rfenErrorCode = e8OPERATION_REJECTED;
      }
      else
      {
         ETG_TRACE_USR1(("[DESC] spi_tclConnMngr::vSetTechnologyPreference : Setting preferred technology to %d  ", ETG_ENUM(DEVICE_CATEGORY,
                  enTechnologyPreference)));
         tenDeviceCategory enDeviceCat = e8DEV_TYPE_UNKNOWN;
         enDeviceCat = m_poDeviceListHandler->enGetDeviceCategory(cou32DeviceHandle);
         if (enDeviceCat != enTechnologyPreference)
         {
            //reset the error count
            m_poDeviceListHandler->vSetSelectError(cou32DeviceHandle, false);
         }
         m_poDeviceListHandler->vSetDeviceCategory(cou32DeviceHandle, enTechnologyPreference);
         rfenErrorCode = e8NO_ERRORS;
      }
   }
}

/**************************************************************************
 ** FUNCTION   : tVoid spi_tclConnMngr:: vGetTechnologyPreference()
 ***************************************************************************/
t_Void spi_tclConnMngr::vGetTechnologyPreference(const t_U32 cou32DeviceHandle, const tenDeviceType coenDeviceType,
         tenDeviceCategory &rfenTechnologyPreference) const
{
   if (cou32MAX_DEVICEHANDLE == cou32DeviceHandle)
   {
      if (NULL != m_poDeviceListHandler)
      {
         rfenTechnologyPreference = m_poDeviceListHandler->enGetTechnologyPreference(coenDeviceType);
      }
   }
   else
   {
      rfenTechnologyPreference = enGetDeviceCategory(cou32DeviceHandle);
   }
   ETG_TRACE_USR1(("spi_tclConnMngr::vGetTechnologyPreference : DeviceHandle %d preferred technology is %d ", cou32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY,
            rfenTechnologyPreference)));
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vSelectDeviceResultCb
 ***************************************************************************/
t_Void spi_tclConnMngr::vSelectDeviceResultCb(tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vSelectDeviceResultCb enErrorCode = %d ", ETG_ENUM(ERROR_CODE, enErrorCode)));

   m_u32DeviceUnderSelection = 0;

   //! Post Select device result
   if (NULL != m_poMediator)
   {
      m_poMediator->vPostSelectDeviceRes(e32COMPID_CONNECTIONMANAGER, enErrorCode);
   } //if (NULL != poMediator)
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vUpdateDeviceNameCb
 ***************************************************************************/
t_Void spi_tclConnMngr::vUpdateDeviceNameCb(t_U32 cou32DeviceHandle, t_String szDeviceName)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vUpdateDeviceNameCb: Updating Device = %d  with new name = %s ", cou32DeviceHandle, szDeviceName.c_str()));
   if (NULL != m_poDeviceListHandler)
   {
      trDeviceInfo rDeviceInfo;
      t_Bool bDevFound = m_poDeviceListHandler->bGetDeviceInfo(cou32DeviceHandle, rDeviceInfo);

      /* Note:
       * 1. vUpdateDeviceNameCb will be called from GAL receiver module
       * 2. Initially before the updates received from modules Structure will hold default Manufacturer name
       * 3. Update only if name stored is equal to ManufacturerName.
       */
      t_Bool bDeviceNameUpdate = ((rDeviceInfo.szDeviceManufacturerName == rDeviceInfo.szDeviceName) && !(szDeviceName.empty()));
      ETG_TRACE_USR4(("spi_tclConnMngr::vUpdateDeviceNameCb: ManufacturerName = %s", rDeviceInfo.szDeviceManufacturerName.c_str()));
      ETG_TRACE_USR4(("spi_tclConnMngr::vUpdateDeviceNameCb: Stored device name = %s", rDeviceInfo.szDeviceName.c_str()));

      //! Send Device list change only if there is a change in device name
      if ((NULL != m_poConnMngrResp) && (true == bDevFound) && (true == bDeviceNameUpdate))
      {
         m_poDeviceListHandler->vSetDeviceName(cou32DeviceHandle, szDeviceName);
         m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle,
                  rDeviceInfo.enDeviceConnectionType,
                  e8DEVICE_CHANGED);
      } //if (NULL != m_poConnMngrResp)
   } //if (NULL != m_poDeviceListHandler)
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vUpdateBTDeviceInfoCb
 ***************************************************************************/
t_Void spi_tclConnMngr::vUpdateBTDeviceInfoCb(t_U32 cou32DeviceHandle, t_String szBTDeviceName, t_String szBTAddress)
{
   ETG_TRACE_USR1(("[PARAM]::vUpdateBTDeviceInfoCb: Updating Device = %d with new name = %s ", cou32DeviceHandle, szBTDeviceName.c_str()));
   ETG_TRACE_USR1(("[PARAM]::vUpdateBTDeviceInfoCb: Received BT address = %s ", szBTAddress.c_str()));

   if (NULL != m_poDeviceListHandler)
   {
      trDeviceInfo rDeviceInfo;
      t_Bool bDevFound = m_poDeviceListHandler->bGetDeviceInfo(cou32DeviceHandle, rDeviceInfo);
      if ((NULL != m_poConnMngrResp) && (true == bDevFound))
      {
         /* Note:
          * AA Device name from BT & GAL receiver can be received in any order.
          * BT Device name is used only if GAL receiver has not yet sent the Device name.
          */

         //! Store the BT Device name
         t_Bool bDeviceNameUpdate = ((false == szBTDeviceName.empty()) && (szBTDeviceName != rDeviceInfo.szDeviceName));
         if (true == bDeviceNameUpdate)
         {
            m_poDeviceListHandler->vSetDeviceName(cou32DeviceHandle, szBTDeviceName);
         }

         //! Store the BT Address
         t_Bool bBTAddressUpdate = ((false == szBTAddress.empty()) && (rDeviceInfo.szBTAddress != szBTAddress));
         if (true == bBTAddressUpdate)
         {
            m_poDeviceListHandler->vSetBTAddress(cou32DeviceHandle, szBTAddress);
         }

         //! Send Device list change if there is a change in Device name and/or BT Address
         if ((true == bDeviceNameUpdate) || (true == bBTAddressUpdate))
         {
            m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle,
                     rDeviceInfo.enDeviceConnectionType,
                     e8DEVICE_CHANGED);
         }
      } //if ((NULL != m_poConnMngrResp) && (true == bDevFound))
   } //if (NULL != m_poDeviceListHandler)
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclConnMngr::vRegisterCallbacks()
{
   /*lint -esym(40,fvUpdateDAPStatus) fvUpdateDAPStatus Undeclared identifier */
   /*lint -esym(40,fvDeviceConnection)fvDeviceConnection Undeclared identifier */
   /*lint -esym(40,fvDeviceDisconnection) fvDeviceDisconnection identifier */
   /*lint -esym(40,fvSelectDeviceResult)fvSelectDeviceResult Undeclared identifier */
   /*lint -esym(40,_1) _1 Undeclared identifier */
   /*lint -esym(40,_2) _2 Undeclared identifier */
   /*lint -esym(40,_3) _3 Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclConnMngr::vRegisterCallbacks() entered "));

   trConnCallbacks rConnMngrCallbacks;

   rConnMngrCallbacks.fvSelectDeviceResult = std::bind(&spi_tclConnMngr::vSelectDeviceResultCb, this,
   SPI_FUNC_PLACEHOLDERS_1);

   rConnMngrCallbacks.fvSelectDeviceError = std::bind(&spi_tclConnMngr::vSetSelectError, this,
   SPI_FUNC_PLACEHOLDERS_2);

   rConnMngrCallbacks.fvSetDeviceName = std::bind(&spi_tclConnMngr::vUpdateDeviceNameCb, this, SPI_FUNC_PLACEHOLDERS_2);

   rConnMngrCallbacks.fvSessionTransport = std::bind(&spi_tclConnMngr::vUpdateSessionTransport,
            this,
            SPI_FUNC_PLACEHOLDERS_3);

   rConnMngrCallbacks.fvSelectionProgressState = std::bind(&spi_tclConnMngr::vSelectionProgressStateCb,
            this,
            SPI_FUNC_PLACEHOLDERS_2);

   rConnMngrCallbacks.fvTriggerDeselection = std::bind(&spi_tclConnMngr::vTriggerDeselection,
            this,
            SPI_FUNC_PLACEHOLDERS_2);

   rConnMngrCallbacks.fvOnSessionMessageStoreIPAddress = std::bind(&spi_tclConnMngr::vUpdateSessionIP,
            this,
            SPI_FUNC_PLACEHOLDERS_2);

   m_poLockConnections.s16Lock();
   //! Register for callbacks with SPI connections
   for (auto itmap = m_mapoConnHandlers.begin(); itmap != m_mapoConnHandlers.end(); itmap++)
   {
      if (NULL != itmap->second)
      {
         itmap->second->vRegisterCallbacks(rConnMngrCallbacks);
      }
   }
   m_poLockConnections.vUnlock();

   if (NULL != m_poMediator)
   {
      trConnMngrCallback rConnMngrCb;
      rConnMngrCb.fvSetBTDeviceInfo = std::bind(&spi_tclConnMngr::vUpdateBTDeviceInfoCb, this, SPI_FUNC_PLACEHOLDERS_3);
      rConnMngrCb.fvSelectionProgressState = std::bind(&spi_tclConnMngr::vSelectionProgressStateCb,
               this,
               SPI_FUNC_PLACEHOLDERS_2);

      m_poMediator->vRegisterCallbacks(rConnMngrCb);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::bGetDevProjUsage
 ***************************************************************************/
t_Bool spi_tclConnMngr::bGetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo &rfenEnabledInfo,
         tenDeviceType enDeviceType) const
{
   t_Bool bRetVal = false;

   //! Feature is disabled if the project calibration does not support specific SPI technology
   if (false == m_rSPIFeatureSupport.bGetFeatureSupport(enSPIType, enDeviceType))
   {
      rfenEnabledInfo = e8USAGE_DISABLED;
   }
   else if (NULL != m_poDeviceListHandler)
   {
      bRetVal = m_poDeviceListHandler->bGetDevProjUsage(enSPIType, rfenEnabledInfo, enDeviceType);
   }
   ETG_TRACE_USR1(("spi_tclConnMngr::bGetDevProjUsage  enDevCat = %d rfenEnabledInfo = %d  ", ETG_ENUM(DEVICE_CATEGORY,
            enSPIType), ETG_ENUM(ENABLED_INFO, rfenEnabledInfo)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::enGetDeviceCategory
 ***************************************************************************/
tenDeviceCategory spi_tclConnMngr::enGetDeviceCategory(const t_U32 cou32DeviceHandle) const
{
   tenDeviceCategory enDevCat = e8DEV_TYPE_UNKNOWN;
   if ((NULL != m_poDeviceListHandler) && (0 != cou32DeviceHandle))
   {
      enDevCat = m_poDeviceListHandler->enGetDeviceCategory(cou32DeviceHandle);
   } //if ((NULL != m_poDeviceListHandler) && (0 != cou32DeviceHandle))
   return enDevCat;
}
/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vLoadSPITechnologySettings
 ***************************************************************************/
t_Void spi_tclConnMngr::vLoadSPITechnologySettings()
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vLoadSPITechnologySettings entered"));
   if (NULL != m_poDeviceListHandler)
   {
      tenEnabledInfo enProjEnabledInfo = e8USAGE_ENABLED;
      tenEnabledInfo enCurrentProjSetting = e8USAGE_UNKNOWN;
      tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE;
      //! Store the default setting to datapool on the first startup after virgin flash
      bGetDevProjUsage(e8DEV_TYPE_DIPO, enCurrentProjSetting, enDeviceType);
      if (e8USAGE_UNKNOWN == enCurrentProjSetting)
      {
         m_poDeviceListHandler->bSetDevProjUsage(e8DEV_TYPE_DIPO, enProjEnabledInfo, enDeviceType);
      }

      bGetDevProjUsage(e8DEV_TYPE_ANDROIDAUTO, enCurrentProjSetting, enDeviceType);
      if (e8USAGE_UNKNOWN == enCurrentProjSetting)
      {
         m_poDeviceListHandler->bSetDevProjUsage(e8DEV_TYPE_ANDROIDAUTO, enProjEnabledInfo, enDeviceType);
      }

      bGetDevProjUsage(e8DEV_TYPE_MIRRORLINK, enCurrentProjSetting, enDeviceType);
      if (e8USAGE_UNKNOWN == enCurrentProjSetting)
      {
         m_poDeviceListHandler->bSetDevProjUsage(e8DEV_TYPE_MIRRORLINK, enProjEnabledInfo, enDeviceType);
      }

      bGetDevProjUsage(e8DEV_TYPE_MYSPIN, enCurrentProjSetting, enDeviceType);
      if (e8USAGE_UNKNOWN == enCurrentProjSetting)
      {
         m_poDeviceListHandler->bSetDevProjUsage(e8DEV_TYPE_MYSPIN, enProjEnabledInfo, enDeviceType);
      }

      bGetDevProjUsage(e8DEV_TYPE_CARLIFE, enCurrentProjSetting, enDeviceType);
      if (e8USAGE_UNKNOWN == enCurrentProjSetting)
      {
         m_poDeviceListHandler->bSetDevProjUsage(e8DEV_TYPE_CARLIFE, enProjEnabledInfo, enDeviceType);
      }

      bGetDevProjUsage(e8DEV_TYPE_ONCAR, enCurrentProjSetting, enDeviceType);
      if (e8USAGE_UNKNOWN == enCurrentProjSetting)
      {
         m_poDeviceListHandler->bSetDevProjUsage(e8DEV_TYPE_ONCAR, enProjEnabledInfo, enDeviceType);
      }
   } //if (NULL != m_poDeviceListHandler)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnMngr::vUpdateSessionTransport
 ***************************************************************************/
t_Void spi_tclConnMngr::vUpdateSessionTransport(const t_U32 cou32DeviceID, tenDeviceCategory enDeviceCategory,
         tenSessionTransport enSessionTransport)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vUpdateSessionTransport Updating Session Transport Info for DeviceID = 0x%x of Category %d as %d", cou32DeviceID, ETG_ENUM(DEVICE_CATEGORY,
            enDeviceCategory), ETG_ENUM(SESSION_TRANSPORT, enSessionTransport)));
   if (NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->vSetSessionTransport(cou32DeviceID, enSessionTransport);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vSelectionProgressStateCb
 ***************************************************************************/
t_Void spi_tclConnMngr::vSelectionProgressStateCb(const t_U32 cou32DeviceHandle,
         tenDeviceSelectionProgressState enDeviceSelectionProgressState)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vSelectionProgressStateCb Device ID = %d, enDeviceSelectionProgressState = %d ", cou32DeviceHandle, ETG_ENUM(SEL_PROGRESS_STATE,
            enDeviceSelectionProgressState)));
   if ((NULL != m_poDeviceListHandler) && (NULL != m_poConnMngrResp))
   {
      m_poDeviceListHandler->vSetSelectionProgressState(cou32DeviceHandle, enDeviceSelectionProgressState);
      m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle, e8USB_CONNECTED, e8DEVICE_CHANGED);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnection::vTriggerDeselection()
 ***************************************************************************/
t_Void spi_tclConnMngr::vTriggerDeselection(t_U32 u32DevID, tenDeviceCategory enDevCat)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vTriggerDeselection:DeviceID:0x%x Category:%d", u32DevID, ETG_ENUM(DEVICE_CATEGORY,
            enDevCat)));

   if (NULL != m_poMediator)
   {
      m_poMediator->vPostAutoDeviceSelection(u32DevID, e8DEVCONNREQ_DESELECT, e8_RETRIAL_FAILED, enDevCat);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnMngr::vSetConnsettingsInstance()
 ***************************************************************************/
t_Void spi_tclConnMngr::vSetConnsettingsInstance(spi_tclConnSettingsIntf* poConnSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vSetConnsettingsInstance entered"));

   if (NULL != poConnSettingsIntf)
   {
      m_poConnSettings = poConnSettingsIntf;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnMngr::vUpdateSessionIP
 ***************************************************************************/
t_Void spi_tclConnMngr::vUpdateSessionIP(t_U32 cou32DeviceHandle, t_String szSessionIP)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vUpdateSessionIP Updating Session IP Info for DeviceID = 0x%x with IP %s", cou32DeviceHandle, szSessionIP.c_str()));
   if (NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->vSetSessionIP(cou32DeviceHandle, szSessionIP);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnMngr::vGetConnMngrConfig()
 ***************************************************************************/

t_Void spi_tclConnMngr::vGetConnMngrConfig()
{
   ETG_TRACE_USR1(("spi_tclConnection::vGetConnMngrConfig entered"));

   if (NULL != m_poConnSettings)
   {
      t_U8 u8RegionCode = 0;
      t_U8 u8VendorCode = 0;
      t_U16 u16VehicleCode = 0;
      t_String rfszPersStoragePath;
      m_poConnSettings->bGetSysVehicleInfo(u8RegionCode, u8VendorCode, u16VehicleCode);
      m_poConnSettings->vGetPersistentStoragePath(rfszPersStoragePath);
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr::vOnSetSelectionProgressState()
 ***************************************************************************/
t_Void spi_tclConnMngr::vOnSetSelectionProgressState(tenDeviceSelectionState enDeviceSelectionState)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vOnSetSelectionProgressState() entered"));
   m_poLockConnections.s16Lock();

   for(auto itmap = m_mapoConnHandlers.begin(); itmap != m_mapoConnHandlers.end(); itmap++)
   {
      if (NULL != itmap->second)
      {
         itmap->second->vOnSetSelectionProgressState(enDeviceSelectionState);
      }
   }
   m_poLockConnections.vUnlock();
}

//lint -restore
