/*!
 *******************************************************************************
 * \file             spi_tclDeviceListHandler.h
 * \brief            Handles Device List
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Handles Device List and Device History for SPI Devices
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 11.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 25.01.2016 |  Rachana L Achar             | Logiscope improvements
 01.01.2017 |  Noopur R K                  | Changed bRemoveDeviceFromList.
 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLDEVICELISTHANDLER_H_
#define SPI_TCLDEVICELISTHANDLER_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <map>
#include "spi_ConnMngrTypeDefines.h"
#include "Lock.h"
#include "spi_tclDeviceListIntf.h"
#include "spi_tclConnSettingsIntf.h"
#include "spi_tclDevHistory.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/*!
 * \class spi_tclDeviceListHandler
 * \brief Handles Device List and Device History for SPI Devices
 */
class spi_tclDeviceListHandler: public spi_tclDeviceListIntf
{

   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::spi_tclDeviceListHandler
       ***************************************************************************/
      /*!
       * \fn     spi_tclDeviceListHandler()
       * \brief  Default Constructor
       * \sa      ~spi_tclDeviceListHandler()
       **************************************************************************/
      spi_tclDeviceListHandler(spi_tclConnSettingsIntf* poConnSettings,
      spi_tclDevHistoryIntf* poDevHistory);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::~spi_tclDeviceListHandler
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclDeviceListHandler()
       * \brief   Destructor
       * \sa     spi_tclDeviceListHandler()
       **************************************************************************/
      ~spi_tclDeviceListHandler();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bRestoreDeviceList
       ***************************************************************************/
      /*!
       * \fn     bRestoreDeviceList
       * \brief  Reads database to restore device history to device list
       * \retval true: if the history was added successfully to device list
       *         false: on Error
       **************************************************************************/
      t_Bool bRestoreDeviceList();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bSaveDeviceList
       ***************************************************************************/
      /*!
       * \fn     bSaveDeviceList
       * \brief  Saves the device list to persistent memory device list
       **************************************************************************/
      t_Void bSaveDeviceList();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vClearDeviceList
       ***************************************************************************/
      /*!
       * \fn     vClearDeviceList
       * \brief  clears Device List
       **************************************************************************/
      t_Void vClearDeviceList();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enAddDeviceToList
       ***************************************************************************/
      /*!
       * \fn     enAddDeviceToList
       * \brief  Adds device to device list
       * \param cou32DeviceHandle: Device Handle to be added to device list
       * \param rfrDeviceInfo: Device Info of the device to be added to list
       **************************************************************************/
      tenDeviceStatusInfo enAddDeviceToList(const t_U32 cou32DeviceHandle,
               const trDeviceInfo &rfrDeviceInfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bRemoveDeviceFromList
       ***************************************************************************/
      /*!
       * \fn     bRemoveDeviceFromList
       * \brief  Removes device to device list
       * \param  coU32DeviceHandle:  Device Handle to be removed from device list
       **************************************************************************/
      t_Bool bRemoveDeviceFromList(const t_U32 coU32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vGetDeviceList
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceList
       * \brief  interface to get SPI devices list
       * \param  rfvecDeviceInfoList: reference to the device list which will
       *         be populated with the detected SPI devices
       * \param  bCertifiedOnly: if true, only certified devices will be reported.
       *         if false, all detected devices will be reported
       * \param  bConnectedOnly: if true only connected devices will be returned
       **************************************************************************/
      t_Void vGetDeviceList(const t_U32 cou32DeviceHandle, std::vector<trDeviceInfo>& rfvecDeviceInfoList);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vGetConnectedDeviceList
       ***************************************************************************/
      /*!
       * \fn     vGetConnectedDeviceList
       * \brief  interface to get SPI devices list with only connected devices
       * \param  rfvecDeviceInfoList: reference to the device list which will
       *         be populated with the connected SPI devices
       **************************************************************************/
      t_Void vGetConnectedDeviceList(std::vector<trDeviceInfo>& rfvecDeviceInfoList);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vGetEntireDeviceList
       ***************************************************************************/
      /*!
       * \fn     vGetEntireDeviceList
       * \brief  interface to get SPI devices list with complete device info
       * \param  rfrmapDeviceInfoList: reference to the device list which will
       *         be populated with the detected SPI devices
       **************************************************************************/
      t_Void vGetEntireDeviceList(std::map<t_U32, trEntireDeviceInfo> &rfrmapDeviceInfoList);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetUserDeselectionFlag
       ***************************************************************************/
      /*!
       * \fn     vSetUserDeselectionFlag
       * \brief  Sets the flag when user deselects the device from HMI
       * \param  cou32DeviceHandle :Device Handle
       * \param  bState : indicates the value of UserDeselectionFlag
       **************************************************************************/
      t_Void vSetUserDeselectionFlag(const t_U32 cou32DeviceHandle, t_Bool bState);

       /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bGetUserDeselectionFlag
       ***************************************************************************/
      /*!
       * \fn     bGetUserDeselectionFlag
       * \brief  returns the flag when user deselects the device from HMI
       * \param  cou32DeviceHandle :Device Handle
       * \retval  bState : indicates the value of UserDeselectionFlag
       **************************************************************************/
      t_Bool bGetUserDeselectionFlag(const t_U32 cou32DeviceHandle);

      /***************************************************************************
      ** FUNCTION:  spi_tclDeviceListHandler::u32GetNoofConnectedDevices
      ***************************************************************************/
     /*!
      * \fn     u32GetNoofConnectedDevices
      * \brief  returns no of connected devices
      * \retval  t_U32 : indicates the no of connected devices
      **************************************************************************/
     t_U32 u32GetNoofConnectedDevices();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceValidity
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceValidity
       * \brief  Sets the validity of the device (based on certification)
       * \param  u32DeviceHandle : Device handle of which the validity has to be set
       * \param  bDeviceValid: true if device is certified else false
       **************************************************************************/
      t_Void vSetDeviceValidity(const t_U32 cou32DeviceHandle,
               t_Bool bDeviceValid);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bIsDeviceValid
       ***************************************************************************/
      /*!
       * \fn     bIsDeviceValid
       * \brief  Interface to get device information for a particular device handle
       * \param  cou32DeviceHandle: Device handle of the  device
       * \retval true: if the device is valid otherwise false
       **************************************************************************/
      t_Bool bIsDeviceValid(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bGetDeviceInfo
       ***************************************************************************/
      /*!
       * \fn     bGetDeviceInfo
       * \brief  Interface to get device information for a particular device handle
       * \param  u32DeviceHandle: Device handle of the  device
       * \param  rfrDeviceInfo: Device information for the requested device
       **************************************************************************/
      t_Bool bGetDeviceInfo(t_U32 u32DeviceHandle,
               trDeviceInfo &rfrDeviceInfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceName
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceName
       * \brief  Sets the device name of the device identified by device handle
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       * \param rfrszDeviceName : Device Name to be set
       **************************************************************************/
      t_Void vSetDeviceName(const t_U32 cou32DeviceHandle, t_String &rfrszDeviceName);
      
      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vGetDeviceName
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceName
       * \brief  Gets the device name of the device identified by device handle
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       * \param rfrszDeviceName : Device Name to be get
       **************************************************************************/      
      t_Void vGetDeviceName(const t_U32 cou32DeviceHandle, t_String &rfrszDeviceName);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vAddDeviceToHistory
       ***************************************************************************/
      /*!
       * \fn     vAddDeviceToHistory
       * \brief  Adds device to history
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       **************************************************************************/
      t_Void vAddDeviceToHistory(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vRemoveDeviceFromHistory
       ***************************************************************************/
      /*!
       * \fn     vRemoveDeviceFromHistory
       * \brief  removes device from history
       * \param  cou32DeviceHandle: Device Handle of the device to be removed from history
       **************************************************************************/
      t_Void vRemoveDeviceFromHistory(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bGetDeviceHistory
       ***************************************************************************/
      /*!
       * \fn     bGetDeviceHistory
       * \brief  Interface to get device history
       * \param  rfvecDevHistory: reference to vector containing device history
       **************************************************************************/
      t_Bool bGetDeviceHistory(std::vector<trEntireDeviceInfo> &rfvecDevHistory) const;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceCategory
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceCategory
       * \brief  Returns Device category for the provided device ID
       * \param  cou32DeviceHandle: Device Handle for which device
       *         category is requested
       **************************************************************************/
      tenDeviceCategory enGetDeviceCategory(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceCategory
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceCategory
       * \brief  Sets the Device category for the device
       * \param  cou32DeviceHandle :Device Handle
       * \param  enDeviceCategory : indicates the device category to be used for selection
       **************************************************************************/
      t_Void vSetDeviceCategory(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCategory);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceConnType
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceConnType
       * \brief  Returns Device connection type for the provided device ID
       * \param  cou32DeviceHandle: Device Handle for which device
       *         category is requested
       **************************************************************************/
      tenDeviceConnectionType enGetDeviceConnType(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceConnStatus
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceConnStatus
       * \brief  Returns Device connection status for the provided device ID
       * \param  cou32DeviceHandle: Device Handle for which device
       *         category is requested
       **************************************************************************/
      tenDeviceConnectionStatus enGetDeviceConnStatus(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetBTAddress
       ***************************************************************************/
      /*!
       * \fn     vSetBTAddress
       * \brief  Sets the BT Address of the device identified by device handle
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       * \param corfrszBTAddress : BT Address to be set
       **************************************************************************/
      t_Void vSetBTAddress(const t_U32 cou32DeviceHandle,
            const t_String& corfrszBTAddress);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vGetBTAddress
       ***************************************************************************/
      /*!
       * \fn     vGetBTAddress
       * \brief  Gets the BT Address of the device : cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle whose connection status is requested
       * \param  rfszBTAddress: [OUT] BT address for the requested device handle
       **************************************************************************/
      t_Void vGetBTAddress(const t_U32 cou32DeviceHandle, t_String &rfszBTAddress);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceSelection
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceSelection
       * \brief  Sets the bDeviceSelection flag if the device was selected during
       *         a power cycle
       * \param  cou32DeviceHandle: Device Handle for which device
       *         selection is requested
       **************************************************************************/
      t_Void vSetDeviceSelection(const t_U32 cou32DeviceHandle, t_Bool bSelectDev);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::u32GetSelectedDevice
       ***************************************************************************/
      /*!
       * \fn     u32GetSelectedDevice
       * \brief  returns the currently selected device
       * \retval Device Handle of the currently selected device. O if not found
       **************************************************************************/
      t_U32 u32GetSelectedDevice();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bWasDeviceSelected
       ***************************************************************************/
      /*!
       * \fn     bWasDeviceSelected
       * \brief  Returns whether a device was selected or not previously.
       * \param  cou32DeviceHandle: Device Handle for which device
       *         selection is requested
       **************************************************************************/
      t_Bool bWasDeviceSelected(const t_U32 cou32DeviceHandle);


      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceConnectionStatus
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceConnectionStatus
       * \brief  Sets the Device Connection status flag for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device Handle for which device
       *         Connection is requested
       **************************************************************************/
      t_Void vSetDeviceConnectionStatus(const t_U32 cou32DeviceHandle,
               tenDeviceConnectionStatus enConnStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bIsDeviceConnected
       ***************************************************************************/
      /*!
       * \fn     bIsDeviceConnected
       * \brief  Returns whether a device was Connected or not previously
       * \param  cou32DeviceHandle: Device Handle for which device
       *         selection is requested
       **************************************************************************/
      t_Bool bIsDeviceConnected(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetDAPSupport
       ***************************************************************************/
      /*!
       * \fn     vSetDAPSupport
       * \brief  Sets the DAP Supported field for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device Handle
       **************************************************************************/
      t_Void vSetDAPSupport(const t_U32 cou32DeviceHandle, t_Bool bDAPSupported);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bIsDAPSupported
       ***************************************************************************/
      /*!
       * \fn     bIsDAPSupported
       * \brief  Returns the DAP Support status of cou32DeviceHandle
       * \param  cou32DeviceHandle: Device Handle
       * \retval true: If device supports DAP. false otherwise
       **************************************************************************/
      t_Bool bIsDAPSupported(const t_U32 cou32DeviceHandle);

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclDeviceListHandler::vSetDeviceUsagePreference
      ***************************************************************************/
      /*!
      * \fn     vSetDeviceUsagePreference
      * \brief  Interface to set the preference for the usage of the connected
      *         Mirror Link/DiPO device. This can be set for individual devices or
      *         for all the connected Mirror Link/DiPO devices.
      * \param  cou32DeviceHandle  : Unique handle which identifies the device.
      *              If the value is 0xFFFFFFFF, then this function sets the overall
      *              preference usage for Mirror Link/DiPO.
      * \param  enEnabledInfo    : Enable Information.
      **************************************************************************/
      t_Void vSetDeviceUsagePreference(const t_U32 cou32DeviceHandle,
               tenEnabledInfo enEnabledInfo) ;

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclDeviceListHandler::vGetDeviceUsagePreference
      ***************************************************************************/
      /*!
      * \fn     vGetDeviceUsagePreference
      * \brief  Interface to get the preference for the usage of the
      *         connected Mirror Link/iPod Out device during startup.
      * \param  cou32DeviceHandle  : Unique handle which identifies the device.
      *              If the value is 0xFFFFFFFF, then this function sets the overall
      *              preference usage for Mirror Link/DiPO.
      * \param  [OUT] rfenEnabledInfo : Enable Information.
      **************************************************************************/
      t_Void vGetDeviceUsagePreference(const t_U32 cou32DeviceHandle,
               tenEnabledInfo& rfenEnabledInfo) ;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bSetDevProjUsage
       ***************************************************************************/
      /*!
       * \fn     bSetDevProjUsage
       * \brief  Called when the SPI feature is turned ON or OFF by the user
       * \param  enSPIType: indicates the SPI Type (Mirrorlink, Dipo ..)
       * \param  enSPIState : indicates the service status of enSPIType
       **************************************************************************/
      t_Bool bSetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo enSPIState, tenDeviceType enDeviceType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bGetDevProjUsage
       ***************************************************************************/
      /*!
       * \fn     bGetDevProjUsage
       * \brief  Returns the present SPI Feature Stata
       * \param  enSPIType: SPI service type for which state is
       *         requested (Mirrorlink, DiPo ..)
       * \param  rfrEnabledInfo: [OUT]Indicates if enSPIType is enabled or not
       * \retval true if enSPIType is enabled.
       *         false if enSPIType is disabled
       **************************************************************************/
      t_Bool bGetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo &rfrEnabledInfo, tenDeviceType enDeviceType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::u32GetLastSelectedDevice
       ***************************************************************************/
      /*!
       * \fn     u32GetLastSelectedDevice
       * \brief  Returns the last selected device
       * \retval Device handle of the selected device
       **************************************************************************/
      t_U32 u32GetLastSelectedDevice();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetSelectError
       ***************************************************************************/
      /*!
       * \fn     vSetSelectError
       * \brief  set to true if the error occurs on device side during selection
        *         Ex: Role switch failure
       * \param cou32DeviceHandle: Device handle of the selected device
       * \parama bIsError value of the device error flag
       **************************************************************************/
      t_Void vSetSelectError(const t_U32 cou32DeviceHandle, t_Bool bIsError);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bIsSelectError
       ***************************************************************************/
      /*!
       * \fn     bIsSelectError
       * \brief  Returns true if the error occurs on device side during selection
       *         Ex: Role switch failure
       * \param cou32DeviceHandle: Device handle of the selected device
       * \retval value of the device error flag
       **************************************************************************/
      t_Bool bIsSelectError(const t_U32 cou32DeviceHandle);
      
      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vClearPrivateData
       ***************************************************************************/
      /*!
       * \fn     vClearPrivateData
       * \brief  Clears previously connected devices from the list
       **************************************************************************/
      t_Void vClearPrivateData();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::bGetUSBConnectedFlag
       ***************************************************************************/
      /*!
       * \fn     bGetUSBConnectedFlag
       * \brief  Returns true if when USB connection for the device is active
       * \param cou32DeviceHandle: Device handle
       * \retval true if when USB connection for the device is active else false
       **************************************************************************/
      t_Bool bGetUSBConnectedFlag(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vClearPrivateData
       ***************************************************************************/
      /*!
       * \fn     vClearPrivateData
       * \brief  Set USB connection for the device (active/inactive)
       * \param cou32DeviceHandle: Device handle
       * \param bUSBConnected: Indicates whether USB connection is active or not
       **************************************************************************/
      t_Void vSetUSBConnectedFlag(const t_U32 cou32DeviceHandle, t_Bool bUSBConnected);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetUserPreference
       ***************************************************************************/
      /*!
       * \fn     enGetUserPreference
       * \brief  Returns user preference SPI technology for the device
       * \param cou32DeviceHandle: Device handle
       **************************************************************************/
      tenUserPreference enGetUserPreference(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetUserPreference
       ***************************************************************************/
      /*!
       * \fn     vSetUserPreference
       * \brief   Sets user preference SPI technology for the device
       * \param cou32DeviceHandle: Device handle
       * \param  enUserPref: user preference SPI technology for the device
       **************************************************************************/
      t_Void vSetUserPreference(const t_U32 cou32DeviceHandle, tenUserPreference enUserPref);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetSPISupport
       ***************************************************************************/
      /*!
       * \fn     enGetSPISupport
       * \brief  Get SPI technology support for indicated device handle and category
       * \param  cou32DeviceHandle: Device handle
       * \param  enSPIType: SPI technology supporet to be checked
       * \retval Indicates if this SPi technology is supported
       **************************************************************************/
      tenSPISupport enGetSPISupport(const t_U32 cou32DeviceHandle, tenDeviceCategory enSPIType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetSPISupport
       ***************************************************************************/
      /*!
       * \fn     vSetSPISupport
       * \brief  Sets SPI technology support for indicated device handle and category
       * \param  cou32DeviceHandle: Device handle
       * \param  enSPIType: SPI technology supporet to be checked
       * \param enSPISupport: indicates SPi technology is supported
       **************************************************************************/
      t_Void vSetSPISupport(const t_U32 cou32DeviceHandle, tenDeviceCategory enSPIType,
               tenSPISupport enSPISupport);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetUserAuthorizationStatus
       ***************************************************************************/
      /*!
       * \fn     vSetUserAuthorizationStatus
       * \brief  Sets device authorization status
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      t_Void vSetUserAuthorizationStatus(const t_U32 cou32DeviceHandle, tenUserAuthorizationStatus enUserAuthStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetUserAuthorizationStatus
       ***************************************************************************/
      /*!
       * \fn     enGetUserAuthorizationStatus
       * \brief  Get device authorization status
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      tenUserAuthorizationStatus enGetUserAuthorizationStatus(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceType
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceType
       * \brief  Gets the device type
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      tenDeviceType enGetDeviceType(const t_U32 cou32DeviceHandle);
      /***************************************************************************
      ** FUNCTION:  spi_tclDeviceListHandler::enGetUSBPortType
      ***************************************************************************/
      /*!
      * \fn     enGetUSBPortType
      * \brief  Gets the USB port type
      * \param  cou32DeviceHandle: Device handle
      **************************************************************************/
      tenUSBPortType enGetUSBPortType(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceSelectionInfo
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceSelectionInfo
       * \brief  Sets the device selection info for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle
       * \param  corfrDeviceSelectioninfo: Device selection info
       **************************************************************************/
      t_Void vSetDeviceSelectionInfo(const t_U32 cou32DeviceHandle, const trDeviceSelectionInfo &corfrDeviceSelectioninfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceSelectionErrorType
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceSelectionErrorType
       * \brief  Sets the device selection error type for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle
       * \param  enDeviceSelectionErrorType: Device selection error type
       **************************************************************************/
      t_Void vSetDeviceSelectionErrorType(const t_U32 cou32DeviceHandle, const tenDeviceSelectionErrorType coenDeviceSelectionErrorType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceSelectionErrorType
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceSelectionErrorType
       * \brief  Gets the device selection error type for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      tenDeviceSelectionErrorType enGetDeviceSelectionErrorType(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vGetDeviceSelectionInfo
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceSelectionInfo
       * \brief  Get the device selection info for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle
       * \param  rfrDeviceSelectioninfo: [OUT]Device selection info
       **************************************************************************/
      t_Void vGetDeviceSelectionInfo(const t_U32 cou32DeviceHandle, trDeviceSelectionInfo &rfrDeviceSelectioninfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetSpiFeatureSupport(const trSpiFeatureSupport&)
       ***************************************************************************/
      /*!
       * \fn     vSetSpiFeatureSupport
       * \brief  Sets SPI feature support for all the technologies (as per KDS)
       * \param  corfrSpiFeatureSupp: [IN] reference for SPI features supported structure
       **************************************************************************/
      t_Void vSetSpiFeatureSupport(const trSpiFeatureSupport& corfrSpiFeatureSupp);


      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::u32GetDeviceIDFromBTAddress
       ***************************************************************************/
      /*!
       * \fn     u32GetDeviceIDFromBTAddress(coszBTAddress)
       * \brief  Returns DeviceID for the Device having BT address coszBTAddress
       * \param  coszBTAddress: [IN] BTAddress of the device
       * \retval  t_U32: Device ID having BTAddress coszBTAddress
       **************************************************************************/
       t_U32 u32GetDeviceIDFromBTAddress(const t_String &coszBTAddress);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::u32GetDeviceIDFromSerialNo
       ***************************************************************************/
      /*!
       * \fn     u32GetDeviceIDFromSerialNo(coszSerialNo)
       * \brief  Returns DeviceID for the Device having Serial number coszSerialNo
       * \param  coszSerialNo: [IN] Serial number of the device
       * \retval  t_U32: Device ID having Serial number coszSerialNo
       **************************************************************************/
       t_U32 u32GetDeviceIDFromSerialNo(const t_String &coszSerialNo);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::u32GetDeviceIDFromName
        ***************************************************************************/
       /*!
       * \fn     u32GetDeviceIDFromName(coszDeviceName)
       * \brief  Returns DeviceID for the Device having Device name coszDeviceName
       * \param  coszDeviceName: [IN]  Device name  of the device
       * \retval  t_U32: Device ID having  Device name  coszDeviceName
       **************************************************************************/
       t_U32 u32GetDeviceIDFromName(const t_String &coszDeviceName);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vGetSerialNumber
       ***************************************************************************/
      /*!
       * \fn     vGetSerialNumber
       * \brief  Returns Serial number for the Device with ID cou32DeviceID
       * \param  cou32DeviceID: [IN] Device ID
       * \param  coszSerialNo: [OUT] Serial number f the device
       **************************************************************************/
       t_Void vGetSerialNumber(t_String &coszSerialNo, const t_U32 cou32DeviceID);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceProfile
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceProfile(const t_U32 cou32DeviceID)
       * \brief  Returns Device PRofile for the Device with ID cou32DeviceID
       * \param  cou32DeviceID: [IN] Device ID
       * \retval  tenDeviceProfile: Device profile
        **************************************************************************/
       tenDeviceProfile enGetDeviceProfile(const t_U32 cou32DeviceID);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceProfile
        ***************************************************************************/
       /*!
        * \fn     vSetDeviceProfile(const t_U32 cou32DeviceID)
        * \brief  Sets Device PRofile for the Device with ID cou32DeviceID
        * \param  cou32DeviceID: [IN] Device ID
        * \retval  enDeviceProfile: Device profile
        **************************************************************************/
        t_Void vSetDeviceProfile(const t_U32 cou32DeviceID, tenDeviceProfile enDeviceProfile);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetSessionTransport()
        ***************************************************************************/
       /*!
       * \fn     vSetSessionTransport
       * \brief  Sets Transport on which session is established
       * \param  cou32DeviceHandle: [IN] Device ID
       * \param  enSessionTransport: [IN] Session Transport
       **************************************************************************/
       t_Void vSetSessionTransport(const t_U32 cou32DeviceHandle, tenSessionTransport enSessionTransport);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::enGetSessionTransport()
        ***************************************************************************/
       /*!
       * \brief  Get Transport on which session is established
       * \param  cou32DeviceHandle: [IN] Device ID
       * \retval  enSessionTransport: Session Transport Type
       **************************************************************************/
       tenSessionTransport enGetSessionTransport(const t_U32 cou32DeviceHandle);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::enGetSessionTransportFromSerialNumber()
        ***************************************************************************/
       /*!
       * \brief  Get Transport on which session is established using SerialNumber
       * \param  coszSerialNo: [IN] Serial Number
       * \retval  enSessionTransport: Session Transport Type
       **************************************************************************/
       tenSessionTransport enGetSessionTransportFromSerialNumber(const t_String coszSerialNo);


       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetSelectionProgressState()
        ***************************************************************************/
       /*!
       * \fn     vSetSelectionProgressState
       * \brief  Updates device selection progress state
       * \param  cou32DeviceHandle: [IN] Device ID
       * \param  enDeviceSelectionProgressState: [IN] Selection Progress state
       **************************************************************************/
       t_Void vSetSelectionProgressState(const t_U32 cou32DeviceHandle,tenDeviceSelectionProgressState enDeviceSelectionProgressState);
       
       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetTechnologySelection
        ***************************************************************************/
       /*!
        * \fn     vSetTechnologySelection
        * \brief  Adding the Device to the Device History
        * \param  u32DeviceHandle : Uniquely identifies the target Device.
        * \param  enTechnologyPreference : Technology Preference
        **************************************************************************/
       t_Void vSetTechnologySelection(const t_U32 cou32DeviceHandle, tenDeviceCategory enTechnologyPreference);
       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::rGetDeviceConnectionCountInfo
        ***************************************************************************/
       /*!
        * \fn     rGetDeviceConnectionCountInfo
        * \brief  Returns device connection count information
        * \retval Structure containing device connection count information
        **************************************************************************/
       trDeviceConnectionCountInfo rGetDeviceConnectionCountInfo(const t_U32 cou32DeviceHandle);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceConnectionCountInfo
        ***************************************************************************/
       /*!
        * \fn     vSetDeviceConnectionCountInfo
        * \brief  Adding Device connection info to device list
        * \param  u32DeviceHandle : Uniquely identifies the target Device.
        * \param  rfrNewDeviceConnectionCountInfo : Structure containing device connection count information
        **************************************************************************/
       t_Void vSetDeviceConnectionCountInfo(const t_U32 cou32DeviceHandle,  trDeviceConnectionCountInfo &rfrNewDeviceConnectionCountInfo);
       
       /***************************************************************************
        ** FUNCTION:  tenDeviceCategory spi_tclDeviceListIntf::enGetTechnologyPreference
        ***************************************************************************/
       /*!
        * \fn     tenDeviceCategory enGetTechnologyPreference()
        * \brief  returns Device type preference
        * \param   coenDeviceType : Device Type
        **************************************************************************/
       tenDeviceCategory enGetTechnologyPreference(const tenDeviceType coenDeviceType);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDeviceListIntf::vSetTechnologyPreference
        ***************************************************************************/
       /*!
        * \fn     t_Void vSetTechnologyPreference()
        * \brief  returns Device type preference
        * \param  coenDeviceType : Device Type
        * \param  coenTechnologyPref : Technology preference
        **************************************************************************/
       t_Void vSetTechnologyPreference(const tenDeviceType coenDeviceType ,const tenDeviceCategory coenTechnologyPref);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetVendorProductID
        ***************************************************************************/
       /*!
        * \fn     vSetVendorProductID
        * \brief  Updates vendor ID
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        * \param  cou32VendorID : Vendor ID
        * \param  cou32ProductID : Product ID
        **************************************************************************/
       t_Void vSetVendorProductID(const t_U32 cou32DeviceHandle,
               const t_U32 cou32VendorID,const t_U32 cou32ProductID);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::u32GetVendorID
        ***************************************************************************/
       /*!
        * \fn     u32GetVendorID
        * \brief  Get vendor ID
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        **************************************************************************/
       t_U32 u32GetVendorID(const t_U32 cou32DeviceHandle);
      
       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::u32GetProductID
        ***************************************************************************/
       /*!
        * \fn     u32GetProductID
        * \brief  Get product ID
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        **************************************************************************/
       t_U32 u32GetProductID(const t_U32 cou32DeviceHandle);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceWhitelistedInfo
        ***************************************************************************/
       /*!
        * \fn     vSetDeviceWhitelistedInfo
        * \brief  Set the Device Whitelisting Info
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        * \param  bIsDeviceWhiteListed : True if the Device is white listed
        **************************************************************************/
       t_Void vSetDeviceWhitelistedInfo(const t_U32 cou32DeviceHandle, const t_Bool bIsDeviceWhiteListed);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceWhitelistedInfo
        ***************************************************************************/
       /*!
        * \fn     bGetDeviceWhitelistedInfo
        * \brief  Get the Device Whitelisting Info
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        * \ret    Return true if the Device is white listed
        **************************************************************************/
       t_Bool bGetDeviceWhitelistedInfo(const t_U32 cou32DeviceHandle);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vUpdateDeviceName
        ***************************************************************************/
       /*!
        * \fn     vUpdateDeviceName
        * \param  [IN] cou32DeviceHandle : Uniquely identifies the target Device.
        * \param  [IN] szDeviceName : Device Name
        * \retval BOOL: Returns True or False
        ***************************************************************************/
       t_Bool bUpdateDeviceName(const t_U32 cou32DeviceHandle,t_String szDeviceName) ;

       /***************************************************************************
        ** FUNCTION: t_Void spi_tclDeviceListHandler::vSetDeviceSelectionReason()
        ***************************************************************************/
       /*!
        * \fn     vSetDeviceSelectionReason
        * \param  [IN] cou32DeviceHandle : Uniquely identifies the target Device.
        * \param  [IN] enSelectReason : Device Selection reason
        ***************************************************************************/
       t_Void vSetDeviceSelectionReason(const t_U32 cou32DeviceHandle, tenSelectReason enSelectReason);
	   
	   /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::enGetLastActiveSessionTransport()
        ***************************************************************************/
       /*!
       * \brief  Get Transport on which session is established in the last trail
       * \param  cou32DeviceHandle: [IN] Device ID
       * \retval  enSessionTransport: Session Transport Type
       **************************************************************************/
       tenSessionTransport enGetLastActiveSessionTransport(const t_U32 cou32DeviceHandle);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceSwitchState
        ***************************************************************************/
       /*!
        * \fn     vSetDeviceSwitchState
        * \brief  Sets the Device Switch State
        * \param  cou32DeviceHandle [IN]:Device Handle
        * \param  enDeviceSwitchState[IN] : Device Switch state - PROJECTION/DEFAULT
        **************************************************************************/
       t_Void vSetDeviceSwitchState(const t_U32 cou32DeviceHandle, tenUSBSwitchState enDeviceSwitchState);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceSwitchState
        ***************************************************************************/
       /*!
        * \fn     enGetDeviceSwitchState
        * \brief  returns the Device Switch State
        * \param  cou32DeviceHandle :Device Handle
        * \retval enDeviceSwitchState : indicates Device Switch state - PROJECTION/DEFAULT
        **************************************************************************/
       tenUSBSwitchState enGetDeviceSwitchState(const t_U32 cou32DeviceHandle);

   private: 

      /***************************************************************************
      ** FUNCTION: spi_tclDeviceListHandler(const spi_tclDeviceListHandler &corfobjRhs)
      ***************************************************************************/
      /*!
      * \fn      spi_tclDeviceListHandler(const spi_tclDeviceListHandler &corfobjRhs)
      * \brief   Copy constructor not implemented hence made protected to prevent
      *          misuse
      **************************************************************************/
      spi_tclDeviceListHandler(const spi_tclDeviceListHandler &corfobjRhs);

      /***************************************************************************
      ** FUNCTION: const spi_tclDeviceListHandler & operator=(
      **           const spi_tclDeviceListHandler &corfobjRhs);
      ***************************************************************************/
      /*!
      * \fn      const spi_tclDeviceListHandler & operator=(
      *          const spi_tclDeviceListHandler &corfobjRhs);
      * \brief   assignment operator not implemented hence made protected to
      *          prevent misuse
      **************************************************************************/
      const spi_tclDeviceListHandler & operator=(
         const spi_tclDeviceListHandler &corfobjRhs);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSortDeviceList
       ***************************************************************************/
      /*!
       * \fn     vSortDeviceList
       * \brief  Sorts SPI devices list in the order Connected, Selected
       * \param  rfvecDeviceInfoList: reference to the device list which will
       *         be populated with the detected SPI devices
       **************************************************************************/
      t_Void vSortDeviceList(std::vector<trEntireDeviceInfo>& rfvecEntireDeviceInfoList,
               std::vector<trDeviceInfo>& rfvecDeviceInfoList);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vDisplayDeviceList(const std::vec...
       ***************************************************************************/
      /*!
       * \fn     vDisplayDeviceList
       *         (const std::vector<trDeviceInfo>& corfvecDeviceInfoList) const
       * \brief  Display device information of all SPI devices
       * \param  corfvecDeviceInfoList: reference to the device list whose device
       *         information has to be displayed
       **************************************************************************/
      t_Void vDisplayDeviceList(const std::vector<trDeviceInfo>& corfvecDeviceInfoList) const;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vGetDeviceList
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceList
       * \brief  interface to get SPI devices list
       * \param  rfvecDeviceInfoList: reference to the device list which will
       *         be populated with the detected SPI devices
       * \param  bCertifiedOnly: if true, only certified devices will be reported.
       *         if false, all detected devices will be reported
       * \param  bConnectedOnly: if true only connected devices will be returned
       **************************************************************************/
      t_Void vGetDeviceList(std::vector<trDeviceInfo>& rfvecDeviceInfoList, t_Bool bCertifiedOnly, t_Bool bConnectedOnly);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vChangeDeviceUsageSetting
       ***************************************************************************/
      /*!
       * \fn     vChangeDeviceUsageSetting
       * \brief  interface to set new enabled info to the device info list
       * \param  New setting that has to be set for the enabled info feild
       **************************************************************************/
      t_Void vChangeDeviceUsageSetting(tenDeviceCategory enDeviceCategory, tenEnabledInfo enNewDeviceSetting);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::u32GetMaxIndex
       ***************************************************************************/
      /*!
       * \fn     u32GetMaxIndex
       * \brief  Method to get the Maximum Access Index in the Device List
       * \retVa1 Returns the Maximum Access Index
       **************************************************************************/
      t_U32 u32GetMaxIndex();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetSessionIP
       ***************************************************************************/
      /*!
       * \fn     vSetSessionIP
       * \brief  Sets the Session IP of the device identified by device handle
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       * \param rfrszDeviceName : Session IP to be set
       **************************************************************************/
      t_Void vSetSessionIP(const t_U32 cou32DeviceHandle, t_String &rfrszSessionIP);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::szGetSessionIP
       ***************************************************************************/
      /*!
       * \fn     szGetSessionIP
       * \brief  Get the Session IP of the device identified by device handle
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       **************************************************************************/
      t_String szGetSessionIP(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::u32GetDeviceIDFromIPAddress
       ***************************************************************************/
      /*!
       * \fn     u32GetDeviceIDFromIPAddress(coszBTAddress)
       * \brief  Returns DeviceID for the Device having IP address coszBTAddress
       * \param  coszBTAddress: [IN] BTAddress of the device
       * \retval  t_U32: Device ID having IPTAddress coszIPAddress
       **************************************************************************/
       t_U32 u32GetDeviceIDFromIPAddress(const t_String &coszIPAddress);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceStatusInfo
        ***************************************************************************/
       /*!
        * \fn     enGetDeviceStatusInfo(coszBTAddress)
        * \brief  Returns Device Status Info
        * \param  rfrDeviceInfo: [IN] Device Info
        * \param  cou32DeviceHandle: [IN] Device Handle
        * \retval  tenDeviceStatusInfo: Device Status Info
        **************************************************************************/
       tenDeviceStatusInfo enGetDeviceStatusInfo(const trDeviceInfo &rfrDeviceInfo, const t_U32 cou32DeviceHandle);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceConnectionType()
        ***************************************************************************/
       /*!
       * \fn     vSetDeviceConnectionType
       * \brief  Sets Device Connection Type
       * \param  cou32DeviceHandle: [IN] Device ID
       * \param  enDeviceConnectionType: [IN] Device Connection Type
       **************************************************************************/
       t_Void vSetDeviceConnectionType(const t_U32 cou32DeviceHandle, tenDeviceConnectionType enDeviceConnectionType);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::vEvaluateEnableInfoForCarlife
        ***************************************************************************/
       /*!
        * \fn     vEvaluateEnableInfoForCarlife
        * \brief  Returns DeviceID for the Device having IP address coszBTAddress
        * \param  rfrEnabledInfo: [OUT]Indicates if enSPIType is enabled or not
        * \param  enDeviceType: Indicates device type - android or apple
        * \retval  t_U32: Device ID having IPTAddress coszIPAddress
        **************************************************************************/
       t_Void vEvaluateEnableInfoForCarlife(tenEnabledInfo &rfrEnabledInfo, tenDeviceType enDeviceType);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::bIsDeviceAddedForWireless
        ***************************************************************************/
       /*!
        * \fn     bIsDeviceAddedForWireless
        * \brief  Returns DeviceID for the Device having IP address coszBTAddress
        * \param  corfrDeviceInfo : Device Info
        * \retval t_Bool : returns the True if device details are valid for wireless Selection
        **************************************************************************/
       t_Bool bIsDeviceAddedForWireless(const trDeviceInfo& corfrDeviceInfo);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::enGetDipoDeviceStatusInfo
        ***************************************************************************/
       /*!
        * \fn     enGetDipoDeviceStatusInfo
        * \brief  Returns DeviceID for the Device having IP address coszBTAddress
        * \param  enReportedCarplaySupport : MP reported Carplay Support
        * \param  enStoredCarplaySupport : Stored Carplay Support
        * \param  enDeviceSelectionState : Device Selection State
        * \retval enGetDipoDeviceStatusInfo : returns Device Status Info
        **************************************************************************/
       tenDeviceStatusInfo enGetDipoDeviceStatusInfo(tenSPISupport enReportedCarplaySupport,
                tenSPISupport enStoredCarplaySupport, tenDeviceSelectionState enDeviceSelectionState);

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListHandler::enGetDipoDeviceStatusInfo
        ***************************************************************************/
       /*!
        * \fn     enGetDipoDeviceStatusInfo
        * \brief  Returns DeviceID for the Device having IP address coszBTAddress
        * \param  enReportedCarplaySupport : MP reported Carplay Support
        * \param  enStoredCarplaySupport : Stored Carplay Support
        * \param  enDeviceSelectionState : Device Selection State
        * \param  bIsBTFunctionalityLimited : BT Limited Functionality
        * \retval enGetDipoDeviceStatusInfo : returns Device Status Info
        **************************************************************************/
       tenDeviceStatusInfo enGetDipoDeviceStatusInfo(tenSPISupport enReportedCarplaySupport,
                tenSPISupport enStoredCarplaySupport, tenDeviceSelectionState enDeviceSelectionState, t_Bool bIsBTFunctionalityLimited );


       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceListHandler::vUpdateBTLimitedFunctionality
       ***************************************************************************/
       /*!
       * \fn       t_Void vUpdateBTLimitedFunctionality
       * \brief   To Post the BT Limitated Functionality
       * \param   cou32DeviceHandle : [IN] Device Handle
       * \param   m_bIsDeviceBTFunctionalityLimited : [IN] BT Limited functionality is set or not
       * \retval  t_Void
       **************************************************************************/
       t_Void vUpdateBTLimitedFunctionality(const t_U32 cou32DeviceHandle, t_Bool bIsDeviceBTFunctionalityLimited);


      //! map containing Device list
      std::map<t_U32, trEntireDeviceInfo> m_mapDeviceInfoList;

      //! Lock to protect Device Info in th list
      Lock m_oDeviceListLock;

      //! connection settings handler
      spi_tclConnSettingsIntf *m_poConnSettings;

      //! Stores the SPI technology supported as per calibration
      trSpiFeatureSupport m_rSPIFeatureSupport;
      
      //!Pointer object of DevHistory 
      spi_tclDevHistoryIntf* m_poDevHistory;

      //! Store Device selection Reason
      tenSelectReason m_enSelectReason;
};
/*! } */
#endif // SPI_TCLDEVICELISTHANDLER_H_
