/*!
 *******************************************************************************
 * \file             spi_tclDiscoveryDataIntf.h
 * \brief            Discoverer data interface class
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Discoverer data interface class
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#include "spi_tclDiscoveryDataIntf.h"
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclDiscoveryDataIntf.cpp.trc.h"
#endif
#endif


/***************************************************************************
 ** FUNCTION:  spi_tclDiscoveryDataIntf::spi_tclDiscoveryDataIntf
 ***************************************************************************/
spi_tclDiscoveryDataIntf::spi_tclDiscoveryDataIntf()
{

}

/***************************************************************************
 ** FUNCTION:  spi_tclDiscoveryDataIntf::~spi_tclDiscoveryDataIntf()
 ***************************************************************************/
spi_tclDiscoveryDataIntf::~spi_tclDiscoveryDataIntf()
{

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiscoveryDataIntf::vSetUSBDeviceDetails
 ***************************************************************************/
t_Void spi_tclDiscoveryDataIntf::vSetUSBDeviceDetails(const t_U32 cou32DeviceHandle, trUSBDeviceInfo &corfrUSBDeviceInfo)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDiscoveryDataIntf::vSetUSBDeviceDetails: 0x%x Device Profile %d", cou32DeviceHandle, ETG_ENUM(USB_PROFILE,corfrUSBDeviceInfo.enUSBProfile)));
   m_oLockDeviceInfo.s16Lock();
   corfrUSBDeviceInfo.rDevSwitchInfo = m_mapDeviceInfo[cou32DeviceHandle].rDevSwitchInfo;
   corfrUSBDeviceInfo.u32SwitchRetrialCount = m_mapDeviceInfo[cou32DeviceHandle].u32SwitchRetrialCount;
   m_mapDeviceInfo[cou32DeviceHandle] = corfrUSBDeviceInfo;
   m_oLockDeviceInfo.vUnlock();

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiscoveryDataIntf::vGetUSBDeviceDetails
 ***************************************************************************/
t_Void spi_tclDiscoveryDataIntf::vGetUSBDeviceDetails(const t_U32 cou32DeviceHandle, trUSBDeviceInfo &rfrUSBDeviceInfo)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDiscoveryDataIntf::vGetUSBDeviceDetails: Fetching USB details for 0x%x", cou32DeviceHandle));
   t_Bool bDevFound = false;
   m_oLockDeviceInfo.s16Lock();
   if (m_mapDeviceInfo.end() != m_mapDeviceInfo.find(cou32DeviceHandle))
   {
      rfrUSBDeviceInfo = m_mapDeviceInfo[cou32DeviceHandle];
      bDevFound = true;
   }
   m_oLockDeviceInfo.vUnlock();
   if(false == bDevFound)
   {
      //! TODO: workaround for carplay: will be removed once apple discoverer changes are included
      rfrUSBDeviceInfo.u32DeviceHandle = cou32DeviceHandle;
      ETG_TRACE_ERR(("[DESC] spi_tclDiscoveryDataIntf::vGetUSBDeviceDetails: Device details not found \n"));
   }
   ETG_TRACE_USR4(("[PARAM] spi_tclDiscoveryDataIntf::vGetUSBDeviceDetails: Device Handle = 0x%x , VendorID =0x%x, ProductID = 0x%x, USBProfile = %d Serial Number = %s\n",
            rfrUSBDeviceInfo.u32DeviceHandle, rfrUSBDeviceInfo.u32VendorID, rfrUSBDeviceInfo.u32ProductID,  ETG_ENUM(USB_PROFILE, rfrUSBDeviceInfo.enUSBProfile), rfrUSBDeviceInfo.szSerialNumber.c_str()));

}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiscoveryDataIntf::u32GetStoredDeviceID
 ***************************************************************************/
t_U32 spi_tclDiscoveryDataIntf::u32GetStoredDeviceID(t_String szManufacturerName, t_String szUUID)
{
   ETG_TRACE_USR4(("[PARAM] spi_tclDiscoveryDataIntf::u32GetStoredDeviceID for manufacturer = %s \n", szManufacturerName.c_str()));
   ETG_TRACE_USR4(("[PARAM] spi_tclDiscoveryDataIntf::u32GetStoredDeviceID for UUID = %s \n", szUUID.c_str()));
   t_U32 u32DeviceID = 0;
   m_oLockDeviceInfo.s16Lock();
   std::map<t_U32, trUSBDeviceInfo>::iterator itDeviceInfo;
   for(itDeviceInfo = m_mapDeviceInfo.begin(); itDeviceInfo != m_mapDeviceInfo. end(); itDeviceInfo++)
   {
      if((szManufacturerName == itDeviceInfo->second.szManufacturer))
      {
         u32DeviceID = itDeviceInfo->first;
         break;
      }
   }
   m_oLockDeviceInfo.vUnlock();
   ETG_TRACE_USR1(("[DESC] spi_tclDiscoveryDataIntf::u32GetStoredDeviceID DeviceID = %d",u32DeviceID));
   return u32DeviceID;
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiscoveryDataIntf::u32GetStoredDeviceID
 ***************************************************************************/
t_Bool spi_tclDiscoveryDataIntf::bIsDeviceInfoExists(const t_U32 cou32DeviceHandle)
{
   m_oLockDeviceInfo.s16Lock();
   t_Bool bDeviceInfoExists = false;
   if (m_mapDeviceInfo.end() != m_mapDeviceInfo.find(cou32DeviceHandle))
   {
      bDeviceInfoExists = (0 != m_mapDeviceInfo[cou32DeviceHandle].u32DeviceHandle);
   }
   m_oLockDeviceInfo.vUnlock();

   ETG_TRACE_USR1(("[DESC] spi_tclDiscoveryDataIntf::bIsDeviceInfoExists DeviceID = 0x%x %s",cou32DeviceHandle,
            (true == bDeviceInfoExists)? "Exists": "Doesn't exist"));
   return bDeviceInfoExists;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiscoveryDataIntf::u32GetStoredDeviceID
 ***************************************************************************/
t_Void spi_tclDiscoveryDataIntf::vSetDeviceSwitchInfo(const trDeviceSwitchInfo &corfrDeviceSwitchInfo)
{
   m_oLockDeviceInfo.s16Lock();
   m_mapDeviceInfo[corfrDeviceSwitchInfo.u32DeviceID].rDevSwitchInfo = corfrDeviceSwitchInfo;
   m_oLockDeviceInfo.vUnlock();

   ETG_TRACE_USR1(("[DESC] spi_tclDiscoveryDataIntf::vSetDeviceSwitchInfo DeviceID =0x%x, Device Type = %d having connection Type as %d", corfrDeviceSwitchInfo.u32DeviceID, ETG_ENUM(DEVICE_TYPE,corfrDeviceSwitchInfo.enDeviceType), ETG_ENUM(CONNECTION_TYPE,corfrDeviceSwitchInfo.enDeviceConnectionType)));
   ETG_TRACE_USR4(("[PARAM] spi_tclDiscoveryDataIntf::vSetDeviceSwitchInfo switch in progress = %d, Early Switch in Progress = %d Switch type =%d\n ",
            ETG_ENUM(BOOL,corfrDeviceSwitchInfo.bSwitchInProgress),ETG_ENUM(BOOL,corfrDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress),
            ETG_ENUM(DEVICE_CATEGORY, corfrDeviceSwitchInfo.enDeviceSwitchType)));
   ETG_TRACE_USR4(("[PARAM] spi_tclDiscoveryDataIntf::vSetDeviceSwitchInfo s32HeadUnitID = %d, s32SwitchedDeviceID =0x%x\n ",
            corfrDeviceSwitchInfo.s32HeadUnitID, corfrDeviceSwitchInfo.s32SwitchedDeviceID));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiscoveryDataIntf::u32GetStoredDeviceID
 ***************************************************************************/
t_Void spi_tclDiscoveryDataIntf::vGetDeviceSwitchInfo(const t_U32 cou32DeviceHandle, trDeviceSwitchInfo &corfrDeviceSwitchInfo)
{
   m_oLockDeviceInfo.s16Lock();
   corfrDeviceSwitchInfo = m_mapDeviceInfo[cou32DeviceHandle].rDevSwitchInfo;
   m_oLockDeviceInfo.vUnlock();

   ETG_TRACE_USR1(("[DESC] spi_tclDiscoveryDataIntf::vGetDeviceSwitchInfo DeviceID =0x%x, Device Type = %d having Device Connection Type %d", cou32DeviceHandle, ETG_ENUM(DEVICE_TYPE,corfrDeviceSwitchInfo.enDeviceType), ETG_ENUM(CONNECTION_TYPE,corfrDeviceSwitchInfo.enDeviceConnectionType)));
   ETG_TRACE_USR4(("[PARAM] spi_tclDiscoveryDataIntf::vGetDeviceSwitchInfo switch in progress = %d, Early Switch in Progress = %d Switch type =%d\n ",
            ETG_ENUM(BOOL,corfrDeviceSwitchInfo.bSwitchInProgress),ETG_ENUM(BOOL,corfrDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress),
            ETG_ENUM(DEVICE_CATEGORY, corfrDeviceSwitchInfo.enDeviceSwitchType)));
   ETG_TRACE_USR4(("[PARAM] spi_tclDiscoveryDataIntf::vGetDeviceSwitchInfo s32HeadUnitID = %d, s32SwitchedDeviceID =0x%x\n ",
            corfrDeviceSwitchInfo.s32HeadUnitID, corfrDeviceSwitchInfo.s32SwitchedDeviceID));
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiscoveryDataIntf::vSetSwitchRetrialFailCount
 ***************************************************************************/
t_Void spi_tclDiscoveryDataIntf::vSetSwitchRetrialFailCount(const t_U32 cou32DeviceHandle, t_Bool bFailed)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDiscoveryDataIntf::vSetSwitchRetrialFailCount for 0x%x to %d", cou32DeviceHandle, ETG_ENUM(BOOL,bFailed)));
   m_oLockDeviceInfo.s16Lock();
   std::map<t_U32, trUSBDeviceInfo>::iterator itDeviceInfo = m_mapDeviceInfo.find(cou32DeviceHandle);
   if(m_mapDeviceInfo.end() != itDeviceInfo)
   {
      itDeviceInfo->second.u32SwitchRetrialCount = (true == bFailed)? (1 + (itDeviceInfo->second.u32SwitchRetrialCount)) : 0 ;
   }
   m_oLockDeviceInfo.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBDiscoveryManager::vGetSwitchRetrialFailCount
 ***************************************************************************/
t_U32 spi_tclDiscoveryDataIntf::u32GetSwitchRetrialFailCount(const t_U32 cou32DeviceHandle)
{
   t_U32 u32RetrialCount = 0;
   m_oLockDeviceInfo.s16Lock();
   std::map<t_U32, trUSBDeviceInfo>::iterator itDeviceInfo = m_mapDeviceInfo.find(cou32DeviceHandle);
   if(m_mapDeviceInfo.end() != itDeviceInfo)
   {
      u32RetrialCount = itDeviceInfo->second.u32SwitchRetrialCount;
   }
   m_oLockDeviceInfo.vUnlock();
   ETG_TRACE_USR1(("[DESC] spi_tclDiscoveryDataIntf::u32GetSwitchRetrialFailCount for 0x%x is %d", cou32DeviceHandle, u32RetrialCount));
   return u32RetrialCount;
}
