/*!
 *******************************************************************************
 * \file             spi_tclDeviceDiscoverer.cpp
 * \brief            Unified Device Discoverer
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Unified Device Discoverer
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#include <sys/time.h>
#include "RespRegister.h"
#include "spi_tclDeviceDiscoverer.h"
#include "spi_tclDiscovererFactory.h"
#include "spi_tclDeviceIDDataIntf.h"
#include "Trace.h"
#include "SPITypes.h"
#include "Timer.h"
#include "crc.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclDeviceDiscoverer.cpp.trc.h"
#endif
#endif

/*lint -save -e1013 */
/*lint -save -e1055 */
/*lint -save -e10 */
/*lint -save -e40 */
/*lint -save -e774 */

static const t_U32 scou32InvalidDeviceHandle = 0;
static const t_U32 scou32ClientAvailabilityMaxAllowedTime_ms = 15000;
/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::spi_tclDeviceDiscoverer
 ***************************************************************************/
spi_tclDeviceDiscoverer::spi_tclDeviceDiscoverer(spi_tclConnMngrResp *poRespInterface,
         spi_tclDeviceListIntf *poDeviceList, std::map<tenDiscovererType, spi_tclDiscovererBase*> &rfrmapoDiscoverers,
         spi_tclDeviceSwitcherIntf *poDeviceSwitcher, spi_tclDiscoveryDataIntf* pDiscoveryData,
         spi_tclDiscovererSettingsIntf* poDiscovSettings, spi_tclMediatorIntf* poMediator) :
                  spi_tclSelectionIntf(e32COMPID_DISCOVERER),
                  m_poConnMngrResp(poRespInterface),
                  m_mapoDiscoverers(rfrmapoDiscoverers),
                  m_poDeviceSwitcher(poDeviceSwitcher),
                  m_pDiscoveryData(pDiscoveryData),
                  m_poMediator(poMediator),
                  m_poDeviceListHandler(poDeviceList),
                  m_poDiscovererSettings(poDiscovSettings),
                  m_u32DeviceUnderSelection(0),
                  m_bIsMasterClientAvailable(false),
                  m_rClientAvailTimerID(NULL)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::spi_tclDeviceDiscoverer entered\n"));
   memset(&m_rConnectionStartTime, 0, sizeof(timeval));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::bUnInitialize()
 ***************************************************************************/
spi_tclDeviceDiscoverer::~spi_tclDeviceDiscoverer()
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::~spi_tclDeviceDiscoverer entered\n"));
   m_poDiscovererSettings = NULL;
}


/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::bInitialize()
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bInitialize()
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::bInitialize(): Creating device discoverers and initializing \n"));
   t_Bool bRetVal = true;
   std::map<tenDiscovererType, spi_tclDiscovererBase*>::iterator itmapDiscoverers;
   m_LockDiscoverers.s16Lock();
   for (itmapDiscoverers = m_mapoDiscoverers.begin(); itmapDiscoverers != m_mapoDiscoverers.end(); itmapDiscoverers++)
   {
      if (NULL != (itmapDiscoverers->second))
      {
         bRetVal = ((itmapDiscoverers->second)->bInitialize()) || bRetVal;
      }
   }
   m_LockDiscoverers.vUnlock();

   ETG_TRACE_USR4(("[PARAM] spi_tclDeviceDiscoverer::bInitialize(): Returned %d \n", ETG_ENUM(BOOL, bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::bUnInitialize()
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bUnInitialize()
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::bUnInitialize() entered\n"));

   std::map<tenDiscovererType, spi_tclDiscovererBase*>::iterator itmapDiscoverers;
   vStopDeviceDetection(e8_DISCOVERER_TYPE_USB);
   m_LockDiscoverers.s16Lock();
   for (itmapDiscoverers = m_mapoDiscoverers.begin(); itmapDiscoverers != m_mapoDiscoverers.end(); itmapDiscoverers++)
   {
      if (NULL != (itmapDiscoverers->second))
      {
         (itmapDiscoverers->second)->vUninitialize();
      }
   }
   m_LockDiscoverers.vUnlock();
   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vLoadSettings
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vLoadSettings()
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vLoadSettings() entered\n"));
   //! Store DiPO feature support status
   trSpiFeatureSupport rfrSpiFeatureSupport;
   if (NULL != m_poDiscovererSettings)
   {
      m_poDiscovererSettings->vGetSpiFeatureSupport(rfrSpiFeatureSupport);
   }
   m_rSpiFeatureSupport = rfrSpiFeatureSupport;

   //setting connection settings pointer to discoverer classes
   std::map<tenDiscovererType, spi_tclDiscovererBase*>::iterator itmapDiscoverers;
   m_LockDiscoverers.s16Lock();
   for (itmapDiscoverers = m_mapoDiscoverers.begin(); itmapDiscoverers != m_mapoDiscoverers.end(); itmapDiscoverers++)
   {
      if (NULL != (itmapDiscoverers->second))
      {
         (itmapDiscoverers->second)->vSetDiscoverersettingsInstance(m_poDiscovererSettings);
      }
   }
   m_LockDiscoverers.vUnlock();
   // ML Discoverer needs to be started on Load Settings to avoid RealVNC USB reset
   std::map<tenDiscovererType, spi_tclDiscovererBase*>::iterator itmapMLDiscoverer;
   tenDiscovererType enMLDiscovererType = e8_DISCOVERER_TYPE_ML;
   t_Bool bRetVal = true;

   m_LockDiscoverers.s16Lock();
   itmapMLDiscoverer = m_mapoDiscoverers.find(enMLDiscovererType);
   if ((itmapMLDiscoverer != m_mapoDiscoverers.end()) && (NULL != (itmapMLDiscoverer->second)))
   {

      bRetVal = ((itmapMLDiscoverer->second)->bLoadSettings()) && bRetVal;
   }
   m_LockDiscoverers.vUnlock();
   ETG_TRACE_USR2(("[DESC] spi_tclDeviceDiscoverer::vLoadSettings() Discoverer Initialization status = %d", ETG_ENUM(BOOL,
            bRetVal)));

   //! Register Callback with DeviceSwitcher
   trDeviceSwitcherCbs rDeviceSwitcherCbs;
   rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb = std::bind(&spi_tclDeviceDiscoverer::vOnDeviceSwitchCompleteCb, this,
   SPI_FUNC_PLACEHOLDERS_2);

   //! Set the vehicle information
   trHeadUnitInfo rfrHeadUnitInfo;
   if (NULL != m_poDiscovererSettings)
   {
      m_poDiscovererSettings->vGetHeadUnitInfo(rfrHeadUnitInfo);
   }
   if (NULL != m_poDeviceSwitcher)
   {
      m_poDeviceSwitcher->vRegisterCallbacks(rDeviceSwitcherCbs);
      m_poDeviceSwitcher->vSetHeadUnitInfo(rfrHeadUnitInfo);
   }

   bRegisterObject(this);
   if (NULL != m_poDiscovererSettings)
   {
      m_poDiscovererSettings->vIntializeSPISettings();
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vStartDeviceDetection
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vStartDeviceDetection()
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vStartDeviceDetection() entered"));
   //! Start device detection for  USB devices
   bStartDeviceDetection(e8_DISCOVERER_TYPE_USB);
   bStartDeviceDetection(e8_DISCOVERER_TYPE_APPLE);
   bStartDeviceDetection(e8_DISCOVERER_TYPE_ML);
   bStartDeviceDetection(e8_DISCOVERER_TYPE_WIFI);
   vStartClientAvailabilityTimer();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vSaveSettings
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vSaveSettings()
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vSaveSettings() entered\n"));
   bUnRegisterObject(this);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::bStartDeviceDetection
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bStartDeviceDetection(tenDiscovererType enDiscovererType)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::bStartDeviceDetection : Starting device detection for discoverer %d", ETG_ENUM(DISCOVERER_TYPE,
            enDiscovererType)));
   t_Bool bRetval = false;
   m_LockDiscoverers.s16Lock();
   std::map<tenDiscovererType, spi_tclDiscovererBase*>::iterator itmapDiscoverers =
            m_mapoDiscoverers.find(enDiscovererType);
   if ((m_mapoDiscoverers.end() != itmapDiscoverers) && (NULL != itmapDiscoverers->second))
   {
      bRetval = itmapDiscoverers->second->bStartDeviceDetection();
   }
   m_LockDiscoverers.vUnlock();

   ETG_TRACE_USR1(("[PARAM] spi_tclDeviceDiscoverer::bStartDeviceDetection : Returned with %d", ETG_ENUM(BOOL, bRetval)));
   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStartDeviceReporting
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vStartDeviceReporting(tenDiscovererType enDiscovererType)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vStartDeviceReporting : Starting Device reporting for discoverer %d", ETG_ENUM(DISCOVERER_TYPE,
            enDiscovererType)));
   m_LockDiscoverers.s16Lock();
   std::map<tenDiscovererType, spi_tclDiscovererBase*>::iterator itmapDiscoverers =
            m_mapoDiscoverers.find(enDiscovererType);
   if ((m_mapoDiscoverers.end() != itmapDiscoverers) && (NULL != itmapDiscoverers->second))
   {
      itmapDiscoverers->second->vStartDeviceReporting();
   }
   m_LockDiscoverers.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStopDeviceDetection
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vStopDeviceDetection(tenDiscovererType enDiscovererType)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vStopDeviceDetection :  Stopped Device detection for discoverer %d", ETG_ENUM(DISCOVERER_TYPE,
            enDiscovererType)));
   m_LockDiscoverers.s16Lock();
   std::map<tenDiscovererType, spi_tclDiscovererBase*>::iterator itmapDiscoverers =
            m_mapoDiscoverers.find(enDiscovererType);
   if ((m_mapoDiscoverers.end() != itmapDiscoverers) && (NULL != itmapDiscoverers->second))
   {
      itmapDiscoverers->second->vStopDeviceDetection();
   }
   m_LockDiscoverers.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStopDeviceReporting
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vStopDeviceReporting(tenDiscovererType enDiscovererType)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vStopDeviceReporting Device reporting stopped for discoverer %d", ETG_ENUM(DISCOVERER_TYPE,
            enDiscovererType)));
   m_LockDiscoverers.s16Lock();
   std::map<tenDiscovererType, spi_tclDiscovererBase*>::iterator itmapDiscoverers =
            m_mapoDiscoverers.find(enDiscovererType);
   if ((m_mapoDiscoverers.end() != itmapDiscoverers) && (NULL != itmapDiscoverers->second))
   {
      itmapDiscoverers->second->vStopDeviceReporting();
   }
   m_LockDiscoverers.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::bIsDiPoRoleSwitchRequired(tU32 ..)
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bIsDiPoRoleSwitchRequired(const t_String coszSerialNo, const trUserContext corUsrCntxt,
         const t_U32 cou32MplayDeviceID, const t_Bool cobDeviceCarplayCapability)
{
   trDeviceInfo rDeviceInfo;
   spi_tclDeviceIDDataIntf oDeviceIDIntf;
   rDeviceInfo.u32DeviceHandle = oDeviceIDIntf.u32GetDeviceIDFromSerialNo(coszSerialNo);

   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::bIsDiPoRoleSwitchRequired: Early role switch required request received from mediaplayer for Device = 0x%x ", rDeviceInfo.u32DeviceHandle));
   vOnFastReConnectionClearDeviceSwitchInfo(rDeviceInfo.u32DeviceHandle);

   trRoleSwitchInfo rRoleSwitchInfo;
   rRoleSwitchInfo.u32MplayDeviceID = cou32MplayDeviceID;
   rRoleSwitchInfo.rUserContext = corUsrCntxt;

   //! Early roleswitch required request and select device needs to be synchronized
   //! Role Switch needs to be performed only if the device selection validation succeeds and
   //! a select device is issued by device selector.
   //! Hence store the request from mediaplayer until Device selector takes action
   //! If device selection validation fails, vUpdateSelectionState will be called to inform that role switch is not required
   //! If device selection is triggered, vSelectDevice will be called and role switch will be requested
   //! If device selection fails then selectdeviceresult will be received on which roleswitch required will be answered as false

   //! Store the roleswitch request and inform device connection to device selector
   m_oRoleSwitchRespQueueLock.s16Lock();
   m_mapRoleSwitchRespQueue[rDeviceInfo.u32DeviceHandle] = rRoleSwitchInfo;
   m_oRoleSwitchRespQueueLock.vUnlock();
   if ((true == m_rSpiFeatureSupport.bDipoSupported())
            || (true == m_rSpiFeatureSupport.bCarlifeSupportedForIOS()
                     || (true == m_rSpiFeatureSupport.bMySPINSupported())))
   {
      //Update all the device information only if roleswitch is required
      rDeviceInfo.rProjectionCapability.enDeviceType = e8_APPLE_DEVICE;
      rDeviceInfo.enDeviceConnectionStatus = e8DEV_CONNECTED;
      rDeviceInfo.enDeviceConnectionType = e8USB_CONNECTED;
      rDeviceInfo.enDeviceCategory = e8DEV_TYPE_UNKNOWN;
      rDeviceInfo.rProjectionCapability.enUSBPortType = e8_PORT_TYPE_OTG;
      rDeviceInfo.rProjectionCapability.enCarlifeSupport = e8SPI_SUPPORTED;
      if (cobDeviceCarplayCapability == true)
      {
         rDeviceInfo.rProjectionCapability.enCarplaySupport = e8SPI_SUPPORTED;
      }
      else
      {
         rDeviceInfo.rProjectionCapability.enCarplaySupport = e8SPI_NOTSUPPORTED;
      }
      rDeviceInfo.szSerialNumber = coszSerialNo;
      //! Do roleswitch if asme device is connected over UBS when carplay wireless session is active
      if (true == bIsCarplayActiveOverWiFi(rDeviceInfo))
      {
         if(NULL != m_poDeviceListHandler)
         {
            ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::bIsDiPoRoleSwitchRequired:setting switch state to %d for device handle = %d",
                     ETG_ENUM(DEVICESWITCHSTATE,e8_DEVICESWITCHSTATE_PROJECTION),rDeviceInfo.u32DeviceHandle));
            m_poDeviceListHandler->vSetDeviceSwitchState(rDeviceInfo.u32DeviceHandle, e8_DEVICESWITCHSTATE_PROJECTION);
         }
         bSendRoleSwitchResponse(rDeviceInfo.u32DeviceHandle, e8DIPO_ROLE_SWITCH_REQUIRED);
      }
      trUSBDeviceInfo rUSBDeviceInfo;
      //solution for issue:281160 
      //serial number is not available during launching of myspin app though it is available here 
      //as it is not stored. Hence setting the serial number using vSetUSBDeviceDetails.
      if (NULL != m_pDiscoveryData)
      {
         rUSBDeviceInfo.u32DeviceHandle = rDeviceInfo.u32DeviceHandle;
         rUSBDeviceInfo.szSerialNumber = rDeviceInfo.szSerialNumber;
         rUSBDeviceInfo.enConnectionStatus = rDeviceInfo.enDeviceConnectionStatus;
         m_pDiscoveryData->vSetUSBDeviceDetails(rDeviceInfo.u32DeviceHandle, rUSBDeviceInfo);
      }
      
      //! Send device connection update to start automatic selection
      vOnDeviceConnectionCb(rDeviceInfo, e8_DISCOVERER_TYPE_APPLE);

   }
   else
   {
      ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::bIsDiPoRoleSwitchRequired sending roleswitch response = e8DIPO_ROLE_SWITCH_NOT_REQUIRED as no SPI feature is supported"));
      bSendRoleSwitchResponse(rDeviceInfo.u32DeviceHandle, e8DIPO_ROLE_SWITCH_NOT_REQUIRED);
   }
   return true;
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::bRegisterObject(RespBase *poRespReg)
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bRegisterObject(RespBase *poRespBase)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::bRegisterObject  Entered \n"));
   RespRegister *pRespRegister = RespRegister::getInstance();
   t_Bool bRetReg = false;
   if (NULL != pRespRegister)
   {
      bRetReg = pRespRegister->bRegisterObject(poRespBase);
   }
   return bRetReg;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::bUnRegisterObject(RespBase *poRespReg)
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bUnRegisterObject(RespBase *poRespBase)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::bUnRegisterObject  Entered \n"));
   RespRegister *pRespRegister = RespRegister::getInstance();
   t_Bool bRetReg = false;
   if (NULL != pRespRegister)
   {
      bRetReg = pRespRegister->bUnregisterObject(poRespBase);
   }
   return bRetReg;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vReportDeviceConnection
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vReportDeviceConnection(const trDeviceInfo& corfrDeviceInfo,
         tenDiscovererType enDiscovererType)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vReportDeviceConnection  Entered having device handle %d with Device category %d\n",
            corfrDeviceInfo.u32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY, corfrDeviceInfo.enDeviceCategory)));

   if((e8_DISCOVERER_TYPE_APPLE == enDiscovererType) && (e8USB_CONNECTED == corfrDeviceInfo.enDeviceConnectionType)
            && (e8_PROFILE_DEFAULT == corfrDeviceInfo.enDeviceProfile) && (true == bIsWirelessSelectionInProgress(corfrDeviceInfo.u32DeviceHandle)))
   {
      ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vReportDeviceConnection - Nothing to update as Selection in Progress for Wireless Transport with the same device"));
   }
   else
   {
      if ((NULL != m_poMediator))
      {
         m_poMediator->vPostDeviceConnectionStatus(corfrDeviceInfo.u32ProductID, corfrDeviceInfo.u32VendorID, corfrDeviceInfo.u32DeviceHandle);
      }

      trDeviceSwitchInfo rDeviceSwitchInfo;
      if (NULL != m_poDeviceSwitcher)
      {
         m_poDeviceSwitcher->vGetDeviceSwitchInfo(corfrDeviceInfo.u32DeviceHandle, rDeviceSwitchInfo);
      }

      //! Called when a new device has appeared on USB bus
      //! Don't report device connection reported by central discoverer after mirrorlink switch.
      //! Wait for the mirrorlink discoverer to report the device
      ETG_TRACE_USR4(("[PARAM] spi_tclDeviceDiscoverer::vReportDeviceConnection Device Switch Progress status for Device 0x%x is %d " "Device switch type =%d, Discoverer Type = %d USB Profile = %d\n", rDeviceSwitchInfo.u32DeviceID, ETG_ENUM(BOOL,
               rDeviceSwitchInfo.bSwitchInProgress), rDeviceSwitchInfo.enDeviceSwitchType, enDiscovererType, ETG_ENUM(USB_PROFILE,
                        corfrDeviceInfo.enDeviceProfile)));
      if ((corfrDeviceInfo.u32DeviceHandle != rDeviceSwitchInfo.u32DeviceID)
               || (false == rDeviceSwitchInfo.bSwitchInProgress)
               || (e8DEV_TYPE_MIRRORLINK != rDeviceSwitchInfo.enDeviceSwitchType)
               || (enDiscovererType == e8_DISCOVERER_TYPE_ML))
      {

         trDeviceInfo rDeviceInfo = corfrDeviceInfo;
         //! Product id changes when the device comes up in CDC-NCM mode. Since HMI would not have the
         //! new product id, the product id obtained when the device is detected in usb mode is retained
         m_poDeviceListHandler->vSetVendorProductID(rDeviceInfo.u32DeviceHandle,
                  rDeviceInfo.u32VendorID,
                  rDeviceInfo.u32ProductID);

         //! Device handle is intentionally modified in this case.
         //! This is because some of the devices like HTC One M8 doesn't return serial number in mirrolrink mode
         if ((0 == corfrDeviceInfo.u32DeviceHandle) && (NULL != m_pDiscoveryData))
         {
            rDeviceInfo.u32DeviceHandle = m_pDiscoveryData->u32GetStoredDeviceID(corfrDeviceInfo.szDeviceManufacturerName,
                     corfrDeviceInfo.szUUID);
         }

         if (NULL != m_pDiscoveryData)
         {
            trUSBDeviceInfo rUSBDeviceInfo;
            if (false == m_pDiscoveryData->bIsDeviceInfoExists(rDeviceInfo.u32DeviceHandle))
            {
               rUSBDeviceInfo.u32DeviceHandle = rDeviceInfo.u32DeviceHandle;
               rUSBDeviceInfo.szSerialNumber = rDeviceInfo.szSerialNumber;
               rUSBDeviceInfo.enConnectionStatus = e8DEV_CONNECTED;
               rUSBDeviceInfo.enUSBProfile = rDeviceInfo.enDeviceProfile;
            }
            else
            {
               m_pDiscoveryData->vGetUSBDeviceDetails(rDeviceInfo.u32DeviceHandle, rUSBDeviceInfo);
               if (((e8_PROFILE_CDCNCM_MIRRORLINK != rUSBDeviceInfo.enUSBProfile)
                        && (e8_PROFILE_CDCNCM_UNKNOWNMODE != rUSBDeviceInfo.enUSBProfile))
                        || ((e8_PROFILE_CDCNCM_UNKNOWNMODE == rUSBDeviceInfo.enUSBProfile)
                                 && (e8_PROFILE_CDCNCM_MIRRORLINK == rDeviceInfo.enDeviceProfile)))
               {
                  rUSBDeviceInfo.enUSBProfile = rDeviceInfo.enDeviceProfile;
               }
            }
            m_pDiscoveryData->vSetUSBDeviceDetails(rDeviceInfo.u32DeviceHandle, rUSBDeviceInfo);
         }

         //! Fetching the intial values for vendor and product id from device list handler
         rDeviceInfo.u32VendorID = m_poDeviceListHandler->u32GetVendorID(rDeviceInfo.u32DeviceHandle);
         rDeviceInfo.u32ProductID = m_poDeviceListHandler->u32GetProductID(rDeviceInfo.u32DeviceHandle);

         vOnDeviceConnectionCb(rDeviceInfo, enDiscovererType);
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vReportDeviceDisconnection
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vReportDeviceDisconnection(const t_U32 cou32DeviceHandle,
         tenDiscovererType enDiscovererType,tenDeviceConnectionType enDeviceConnectionType,
         tenSPISupport enCarplaySupport)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vReportDeviceDisconnection  Entered Device ID 0x%x Connection type %d\n", cou32DeviceHandle, ETG_ENUM(CONNECTION_TYPE,
            enDeviceConnectionType)));
   if ((NULL != m_poMediator))
   {
      m_poMediator->vPostDeviceDisconnectionStatus(cou32DeviceHandle);
   }

   trDeviceSwitchInfo rDeviceSwitchInfo;
   if (NULL != m_poDeviceSwitcher)
   {
      m_poDeviceSwitcher->vGetDeviceSwitchInfo(cou32DeviceHandle, rDeviceSwitchInfo);
   }

   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vReportDeviceDisconnection cou32DeviceHandle = 0x%x\n", cou32DeviceHandle));
     ETG_TRACE_USR4(("[PARAM] spi_tclDeviceDiscoverer::vReportDeviceDisconnection Device Switch Progress status for Device 0x%x is %d \n",
              rDeviceSwitchInfo.u32DeviceID, ETG_ENUM(BOOL, rDeviceSwitchInfo.bSwitchInProgress)));


   //! Update if the device has disappeared before or after aoap switch
   if ((NULL != m_pDiscoveryData) && (true == m_pDiscoveryData->bIsDeviceInfoExists(cou32DeviceHandle)))
   {
      trUSBDeviceInfo rUSBDeviceInfo;
      m_pDiscoveryData->vGetUSBDeviceDetails(cou32DeviceHandle, rUSBDeviceInfo);
      rUSBDeviceInfo.enConnectionStatus = e8DEV_NOT_CONNECTED;
      rUSBDeviceInfo.enUSBProfile = e8_PROFILE_DEFAULT;
      m_pDiscoveryData->vSetUSBDeviceDetails(cou32DeviceHandle, rUSBDeviceInfo);

   }

   //! @TODO TO BE REMOVED
   //! As of now wireless device disconnection cannot be determined. Hence ignore Devce disconnections for wireless device
   tenSessionTransport enSessionTransport = e8_SESSION_TRANSPORT_UNKNOWN;
   tenDeviceConnectionType enStoredConnectionType = e8UNKNOWN_CONNECTION;
   trDeviceSelectionInfo rDeviceSelectioninfo;
   if(NULL != m_poDeviceListHandler)
   {
       enSessionTransport = m_poDeviceListHandler->enGetSessionTransport(cou32DeviceHandle);
       m_poDeviceListHandler->vGetDeviceSelectionInfo(cou32DeviceHandle, rDeviceSelectioninfo);
       enStoredConnectionType = m_poDeviceListHandler->enGetDeviceConnType(cou32DeviceHandle);
       tenDeviceCategory enDeviceCategory = m_poDeviceListHandler->enGetDeviceCategory(cou32DeviceHandle);

       if(e8SPI_NOTSUPPORTED == enCarplaySupport)
       {
          m_poDeviceListHandler->vSetSPISupport(cou32DeviceHandle,enDeviceCategory, enCarplaySupport);

          //! Send Device list change - For the HMI to refresh the Screen
          if (NULL != m_poConnMngrResp)
          {
             m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle, enStoredConnectionType, e8DEVICE_CHANGED);
          } //if (NULL != m_poConnMngrResp)
       }
   }

   ETG_TRACE_USR1((" spi_tclDeviceDiscoverer::vReportDeviceDisconnection m_rDeviceSelectionRequest.m_enDevConnType %d ", ETG_ENUM(CONNECTION_TYPE ,m_rDeviceSelectionRequest.m_enDevConnType)));
   ETG_TRACE_USR1(("enStoredConnectionType %d ", ETG_ENUM(CONNECTION_TYPE ,enStoredConnectionType)));

   t_Bool bValidDisconnection = (
   //! USB Disconnection
            ((e8_SESSION_TRANSPORT_USB == enSessionTransport) && (e8USB_CONNECTED == enDeviceConnectionType))
                     ||
                     //! WIFI disconnection
                     ((e8_SESSION_TRANSPORT_WIFI == enSessionTransport)
                              && (e8WIRELESS_CONNECTED == enDeviceConnectionType))
                     || ((e8_SESSION_TRANSPORT_UNKNOWN == enSessionTransport)
                              && (e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS
                                       != rDeviceSelectioninfo.enDeviceSelectionState)) ||
                     //! Wireless Selection is in Progress, Device is disconnected during Selection
                     ((e8_SESSION_TRANSPORT_UNKNOWN == enSessionTransport)
                              && (e8WIRELESS_CONNECTED == enDeviceConnectionType)
                              && (e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS
                                       == rDeviceSelectioninfo.enDeviceSelectionState)
                              && (enStoredConnectionType != e8USB_CONNECTED)) ||
                     //! USB Selection is in Progress, Device is disconnected during selection
                     ((e8_SESSION_TRANSPORT_UNKNOWN == enSessionTransport)
                              && (e8USB_CONNECTED == enDeviceConnectionType)
                              && (e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS
                                       == rDeviceSelectioninfo.enDeviceSelectionState)
                              && (enStoredConnectionType != e8WIRELESS_CONNECTED)) ||
                     //! Disonnection of the MTP or iPod
                     ((e8_SESSION_TRANSPORT_UNKNOWN == enSessionTransport)
                              && (e8USB_CONNECTED == enStoredConnectionType)
                              && (e8USB_CONNECTED == enDeviceConnectionType)));

   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vDeviceDisappearedCallback bValidDisconnection = %d", ETG_ENUM(BOOL,
            bValidDisconnection)));
   //! Don't report device disconnection if the device switch is in progress or device is connected over WiFi
   if (((cou32DeviceHandle != rDeviceSwitchInfo.u32DeviceID) || (false == rDeviceSwitchInfo.bSwitchInProgress))
            && (true == bValidDisconnection))
   {
      vOnDeviceDisconnectionCb(cou32DeviceHandle, enDiscovererType);
   }

   //! There can be cases where in Selection has been started for USB, but SESSION started over WIFI. This will cause issues on the tap on SK_CP icon to END and START the session.
   //! i.e, HMI might endup sending SELECT request for Carplay Wireless
   if (false == bValidDisconnection)
   {
      //! 0. Proceed further only if Validation Fails
      //! 1. Check for the SELECTION state - Proceed only the State is SELECTED
      //! 2. Get the currently selected Device Handle
      //! 3. Get the Connection type for that device handle
      //! 4. Get the Session transport the Selected device Handle
      //! 5. Compare the Selected Device Handle and Selected Device - If same Proceed further
      //! 6. If the Connection Type and Session transport are different for the Selected device handle. Then Change the Connection type according to Session transport

      if (e8_SESSION_TRANSPORT_UNKNOWN != enSessionTransport)
      {
         t_U32 u32DeviceHandle = m_poDeviceListHandler->u32GetSelectedDevice();
         if (cou32DeviceHandle == u32DeviceHandle)
         {
            tenDeviceConnectionType enDeviceConnectionType =
                     (e8_SESSION_TRANSPORT_USB == enSessionTransport) ? e8USB_CONNECTED : e8WIRELESS_CONNECTED;
            if (enDeviceConnectionType != enStoredConnectionType)
            {
               tenDeviceStatusInfo enDeviceStatusInfo = e8DEVICE_CHANGED;
               ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vReportDeviceDisconnection Device Connection Type is Changed to %d", ETG_ENUM(CONNECTION_TYPE,
                        enDeviceConnectionType)));
               m_poDeviceListHandler->vSetDeviceConnectionType(cou32DeviceHandle, enDeviceConnectionType);
               //! Update the HMI about change in the Device Status
               //! Send Device list change
               if (NULL != m_poConnMngrResp)
               {

                  m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle,
                           enDeviceConnectionType,
                           enDeviceStatusInfo);
               } //if (NULL != m_poConnMngrResp)
            }
         }
      }
   }
}
/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vReportAuthenticationtatus
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vReportAuthenticationtatus(const t_U32 cou32DeviceHandle, tenDAPStatus enDAPStatus)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vReportAuthenticationtatus  Entered \n"));
   vUpdateDAPStatusCb(cou32DeviceHandle, enDAPStatus);
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vSelectDevice
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vSelectDevice(const trSelectDeviceRequest &rfrSelDeviceRequest)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer:: vSelectDevice : cou32DeviceHandle =%d enDevSelReq = %d  SelectReason = %d Device Category %d", rfrSelDeviceRequest.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            rfrSelDeviceRequest.m_enDevConnReq), ETG_ENUM(SELECT_REASON, rfrSelDeviceRequest.m_enSelectionReason), ETG_ENUM(DEVICE_CATEGORY,
            rfrSelDeviceRequest.m_enDevCategory)));

   ETG_TRACE_USR1(("Select Device Connection type :%d", ETG_ENUM(CONNECTION_TYPE, rfrSelDeviceRequest.m_enDevConnType)));

   m_u32DeviceUnderSelection = rfrSelDeviceRequest.m_u32DeviceHandle;

   if (NULL != m_poDeviceListHandler)
   {
      tenDeviceProfile enDevProfile = m_poDeviceListHandler->enGetDeviceProfile(rfrSelDeviceRequest.m_u32DeviceHandle);
      m_rDeviceSelectionRequest = rfrSelDeviceRequest;
      m_rDeviceSelectionRequest.m_enDevCategory =
               (e8DEVCONNREQ_SELECT == rfrSelDeviceRequest.m_enDevConnReq) ?
                        rfrSelDeviceRequest.m_enDevCategory : e8DEV_TYPE_UNKNOWN;
      //! Request for device switch
      if ((e8DEVCONNREQ_SELECT == rfrSelDeviceRequest.m_enDevConnReq))
      {
         //Time when the device connection starts
         gettimeofday(&m_rConnectionStartTime, 0);
         vSetDeviceConnectionCount(rfrSelDeviceRequest.m_u32DeviceHandle, rfrSelDeviceRequest.m_enDevCategory);
         //! Request for device switch only:
         //! if device is in cdc ncm for mirrorlink don't trigger mirrorlink command and
         //! Always perform the role switch, if the its Dipo Device Category
         if (((e8DEV_TYPE_DIPO ==rfrSelDeviceRequest.m_enDevCategory) && (e8USB_CONNECTED == rfrSelDeviceRequest.m_enDevConnType)) ||
                  ((e8_PROFILE_CDCNCM_CARPLAY != enDevProfile) && (e8_PROFILE_CDCNCM_MIRRORLINK != enDevProfile)))
         {
            vOnSelectDevice(rfrSelDeviceRequest);
         }
         else
         {
            //! Post Select device result
            if (NULL != m_poMediator)
            {
               m_poMediator->vPostSelectDeviceRes(e32COMPID_DISCOVERER, e8NO_ERRORS);
            } //if (NULL != poMediator)
         }
      }

      //! on device deselection, result is posted directly to HMI because
      //! device disconnection status might go before the inactive status for
      //! carplay when reverse role switch is requested.
      else if (e8DEVCONNREQ_DESELECT == rfrSelDeviceRequest.m_enDevConnReq)
      {
         //! Set Currently selected device to default value if select device
         //! request is received for device deselection
         m_poDeviceListHandler->vSetDeviceSelection(rfrSelDeviceRequest.m_u32DeviceHandle, false);

         //! Send Device list change
         if (NULL != m_poConnMngrResp)
         {
            m_poConnMngrResp->vPostDeviceStatusInfo(rfrSelDeviceRequest.m_u32DeviceHandle,
                     rfrSelDeviceRequest.m_enDevConnType,
                     e8DEVICE_CHANGED);
         } //if (NULL != m_poConnMngrResp)

         //! Post Select device result
         if (NULL != m_poMediator)
         {
            m_poMediator->vPostSelectDeviceRes(e32COMPID_DISCOVERER, e8NO_ERRORS);
         } //if (NULL != poMediator)
      }
   } //if (NULL != m_poDeviceListHandler)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vSelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq,
         tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vSelectDeviceResult : cou32DeviceHandle =%d "
            "enDevSelReq = %d enErrorCode = %d", corfrSelectReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            corfrSelectReq.m_enDevConnReq), ETG_ENUM(ERROR_CODE, enErrorCode)));

   tenDeviceConnectionStatus enConnStatus = e8DEV_NOT_CONNECTED;
   tenDeviceStatusInfo enDeviceStatusInfo = e8DEVICE_CHANGED;
   if (NULL != m_poDeviceListHandler)
   {
      enConnStatus = m_poDeviceListHandler->enGetDeviceConnStatus(corfrSelectReq.m_u32DeviceHandle);
      //! If device selection fails then vOnSelectDeviceResult will be received on which roleswitch required will be answered as false
      if (e8NO_ERRORS != enErrorCode)
      {
         if ((e8USB_CONNECTED == corfrSelectReq.m_enDevConnType))
         {
            bSendRoleSwitchResponse(corfrSelectReq.m_u32DeviceHandle, e8DIPO_ROLE_SWITCH_NOT_REQUIRED);
         }
      }

      if ((e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq) && (e8NO_ERRORS == enErrorCode))
      {
         vSetDeviceConnectionTime(corfrSelectReq.m_u32DeviceHandle, corfrSelectReq.m_enDevCategory);
         vSetDeviceConnectSuccessCount(corfrSelectReq.m_u32DeviceHandle, corfrSelectReq.m_enDevCategory);
         if (NULL != m_poDeviceSwitcher)
         {
            m_poDeviceSwitcher->vClearSwitchFailureCount(corfrSelectReq.m_u32DeviceHandle);
         }

         /*
          * Send the roleswitch response, if there are any pending requests.
          * This improvement is needed for CPW Selection is in PROGRESS and Same device is connected over USB.
          * If the CPW Selection is in PROGRESS and Different Device is connected over USB, this already handled -i.e, in UpdateSelectionState()
          */
         bSendRoleSwitchResponse(corfrSelectReq.m_u32DeviceHandle, e8DIPO_ROLE_SWITCH_REQUIRED);

      }

      t_Bool bIsDeviceConnectedOverUSB = ((e8DEV_CONNECTED == enConnStatus)
               && (e8USB_CONNECTED == m_poDeviceListHandler->enGetDeviceConnType(corfrSelectReq.m_u32DeviceHandle)));

      tenUSBSwitchState enDeviceSwitchState = m_poDeviceListHandler->enGetDeviceSwitchState(corfrSelectReq.m_u32DeviceHandle);

      //! USB Reset is required only if below conditions are satisfied
      // 1. If deselection is triggered for any technology other than Mirrorlink
      // 2. If selection fails for any technology including Mirrorlink
      t_Bool bIsUSBResetRequired = (((e8DEV_TYPE_MIRRORLINK != corfrSelectReq.m_enDevCategory)
               || ((e8DEV_TYPE_MIRRORLINK == corfrSelectReq.m_enDevCategory) && (e8NO_ERRORS != enErrorCode)))
               && (e8OPERATION_CANCELLED_BY_USER != enErrorCode)
               && (true == bIsDeviceConnectedOverUSB)
               && (e8_DEVICESWITCHSTATE_PROJECTION == enDeviceSwitchState));
      ETG_TRACE_USR4(("spi_tclDeviceDiscoverer::vSelectDeviceResult : USB Reset %s for this device ",
               (true == bIsUSBResetRequired) ? "required" : "not required"));

      //! Reset the device connection if device is still connected.
      if ((e8DEVCONNREQ_DESELECT == corfrSelectReq.m_enDevConnReq) ||
          ((e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq) && (e8NO_ERRORS != enErrorCode))
         )
      {
         //! TODO check if reset is required for Mirrorlink device
         //! Reset is currently required for Android Auto as the device doesn't switches to default USB profile
         //! on receiving ByeBye message

         /* Reverse roleswitch to be performed for CarPlay USB devices during deselection of CarPlay Wireless session,
          if the device is connected over USB. Hence we need to consider the device connection type stored in device list to do reverse roleswitch*/

         //! Wait for switch to complete
         if ((NULL != m_poDeviceSwitcher) && (true == bIsUSBResetRequired))
         {
            m_poDeviceSwitcher->vPrepareDeviceSwitch(e8DEV_TYPE_UNKNOWN, corfrSelectReq.m_u32DeviceHandle, m_rDeviceSelectionRequest.m_enDevConnType);
            bRequestDeviceSwitch(corfrSelectReq.m_u32DeviceHandle, e8DEV_TYPE_UNKNOWN);
         }

         //Trigger disconnection, only if the device is not connected over USB while ending Wireless session.
         if (e8_SESSION_TRANSPORT_WIFI == m_poDeviceListHandler->enGetSessionTransport(corfrSelectReq.m_u32DeviceHandle)
                  && (e8DEV_CONNECTED == enConnStatus)
                  && (e8USB_CONNECTED != m_poDeviceListHandler->enGetDeviceConnType(corfrSelectReq.m_u32DeviceHandle)))
         {
            vOnDeviceDisconnectionCb(corfrSelectReq.m_u32DeviceHandle, e8_DISCOVERER_TYPE_USB);
         }
         m_poDeviceListHandler->vSetSessionTransport(corfrSelectReq.m_u32DeviceHandle, e8_SESSION_TRANSPORT_UNKNOWN);
      }

      //! Send Device list change
      if (NULL != m_poConnMngrResp)
      {
         m_poConnMngrResp->vPostDeviceStatusInfo(corfrSelectReq.m_u32DeviceHandle,
                  corfrSelectReq.m_enDevConnType,
                  enDeviceStatusInfo);
      } //if (NULL != m_poConnMngrResp)
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vUpdateSelectionState
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vUpdateSelectionState(const t_U32 cou32DeviceHandle,
         tenDeviceSelectionState enDevSelectionState, tenDeviceConnectionType enDeviceConnType)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vUpdateSelectionState () Device Handle = 0x%x with selection state =%d having Device connection type is %d",
            cou32DeviceHandle, ETG_ENUM(SELECTION_STATE, enDevSelectionState), ETG_ENUM(CONNECTION_TYPE, enDeviceConnType)));


   std::map<tenDiscovererType, spi_tclDiscovererBase*>::iterator itmapDiscoverers;
   m_LockDiscoverers.s16Lock();
   for(itmapDiscoverers = m_mapoDiscoverers.begin(); itmapDiscoverers != m_mapoDiscoverers.end(); itmapDiscoverers++)
   {
      if (NULL != (itmapDiscoverers->second))
      {
         (itmapDiscoverers->second)->vUpdateSelectionState(cou32DeviceHandle,enDevSelectionState,enDeviceConnType);
      }
   }
   m_LockDiscoverers.vUnlock();

   //! Populate the map
   m_oDeviceSelectionStateLock.s16Lock();
   auto itMap = m_mapDeviceSelectionState.find(cou32DeviceHandle);
   if(m_mapDeviceSelectionState.end() != itMap)
   {
      if(e8WIRELESS_CONNECTED == enDeviceConnType)
      {
         itMap->second.m_enWifiDeviceSelectionState = enDevSelectionState;
      }
      else if(e8USB_CONNECTED == enDeviceConnType)
      {
         itMap->second.m_enUSBDeviceSelectionState = enDevSelectionState;
      }
   }
   else
   {
      if(e8WIRELESS_CONNECTED == enDeviceConnType)
      {
         m_mapDeviceSelectionState[cou32DeviceHandle].m_enWifiDeviceSelectionState = enDevSelectionState;
      }
      else if(e8USB_CONNECTED == enDeviceConnType)
      {
         m_mapDeviceSelectionState[cou32DeviceHandle].m_enUSBDeviceSelectionState = enDevSelectionState;
      }
   }
   m_oDeviceSelectionStateLock.vUnlock();

   m_oRoleSwitchRespQueueLock.s16Lock();
   if ((m_mapRoleSwitchRespQueue.end() != m_mapRoleSwitchRespQueue.find(cou32DeviceHandle))
            && (e8_SELECTION_STATE_DEVICE_NOT_SELECTED == enDevSelectionState) && (e8USB_CONNECTED == enDeviceConnType))
   {
      //! Respond to role switch if pending
      //! Send the response to mediaplayer
      trEAPAppInfo rEAPAppInfo;
      if (NULL != m_poConnMngrResp)
      {
         m_poConnMngrResp->vPostDipoRoleSwitchResponse(e8DIPO_ROLE_SWITCH_NOT_REQUIRED,
                  m_mapRoleSwitchRespQueue[cou32DeviceHandle].u32MplayDeviceID,
                  m_mapRoleSwitchRespQueue[cou32DeviceHandle].rUserContext,
                  rEAPAppInfo);
      }
      m_mapRoleSwitchRespQueue.erase(cou32DeviceHandle);
   }
   m_oRoleSwitchRespQueueLock.vUnlock();
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::bSendRoleSwitchResponse
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bSendRoleSwitchResponse(const t_U32 cou32DeviceHandle,
         tenRoleSwitchResponse enRoleswitchResponse)
{
   //! If device selection fails then vOnSelectDeviceResult will be received on which roleswitch required will be answered as false
   t_Bool bSendStatus = false;
   m_oRoleSwitchRespQueueLock.s16Lock();
   if ((m_mapRoleSwitchRespQueue.end() != m_mapRoleSwitchRespQueue.find(cou32DeviceHandle)))
   {
      //! Respond to role switch if pending
      bSendStatus = true;
      if ((NULL != m_poConnMngrResp) && (NULL != m_poDiscovererSettings))
      {
         trEAPAppInfo rfrEAPAppInfo;
         if (e8DIPO_ROLE_SWITCH_REQUIRED_FOR_CARLIFE == enRoleswitchResponse)
         {
            m_poDiscovererSettings->vGetEAPAppInfo(rfrEAPAppInfo);
         }
         m_poConnMngrResp->vPostDipoRoleSwitchResponse(enRoleswitchResponse,
                  m_mapRoleSwitchRespQueue[cou32DeviceHandle].u32MplayDeviceID,
                  m_mapRoleSwitchRespQueue[cou32DeviceHandle].rUserContext,
                  rfrEAPAppInfo);
      }
      m_mapRoleSwitchRespQueue.erase(cou32DeviceHandle);
   }

   m_oRoleSwitchRespQueueLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclDeviceDiscoverer::bSendRoleSwitchResponse: cou32DeviceHandle = 0x%x "
            "Sending early roleswitch response status = %d ", cou32DeviceHandle, ETG_ENUM(BOOL, bSendStatus)));
   return bSendStatus;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vUpdateDAPStatusCb
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vUpdateDAPStatusCb(const t_U32 cou32DeviceHandle, tenDAPStatus enDAPStatus)
{
   ETG_TRACE_USR2(("spi_tclDeviceDiscoverer::vUpdateDAPStatusCb: cou32DeviceHandle = 0x%x "
            "tenDAPStatus = %d ", cou32DeviceHandle, ETG_ENUM(DAP_STATUS, enDAPStatus)));

   if ((NULL != m_poConnMngrResp) && (0 != cou32DeviceHandle))
   {
      tenDeviceConnectionType enDevConnType = e8UNKNOWN_CONNECTION;
      if (NULL != m_poDeviceListHandler)
      {
         switch (enDAPStatus)
         {
            case e8DAP_IN_PROGRESS:
            {
               m_poDeviceListHandler->vSetDAPSupport(cou32DeviceHandle, true);
               break;
            }
            case e8DAP_SUCCESS:
            {
               ETG_TRACE_USR2(("DAP successfully completed for Device-0x%x", cou32DeviceHandle));
               m_poDeviceListHandler->vSetDeviceValidity(cou32DeviceHandle, true);
               break;
            }
            case e8DAP_FAILED:
            {
               ETG_TRACE_USR2(("DAP failed for Device-0x%x and will not be shown to user", cou32DeviceHandle));
               m_poDeviceListHandler->vSetDeviceValidity(cou32DeviceHandle, false);
               break;
            }
            default:
            {
               ETG_TRACE_ERR((" Unknown DAP status "));
            }
         }

         enDevConnType = m_poDeviceListHandler->enGetDeviceConnType(cou32DeviceHandle);
      } // if (NULL != m_poDeviceListHandler)

      m_poConnMngrResp->vPostDAPStatusInfo(cou32DeviceHandle, enDevConnType, enDAPStatus);
   } // if (NULL != m_poConnMngrResp) && ...

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vResetDeviceSwitchInfo
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vResetDeviceSwitchInfo(const trDeviceInfo& corfrDeviceInfo)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vResetDeviceSwitchInfo: cou32DeviceHandle = 0x%x having Device Connection Type as %d and"
            "having Device Category as %d", corfrDeviceInfo.u32DeviceHandle, ETG_ENUM(CONNECTION_TYPE, corfrDeviceInfo.enDeviceConnectionType), ETG_ENUM(DEVICE_CATEGORY, corfrDeviceInfo.enDeviceCategory)));
   //! Reset switch info once device has reappeared in projection mode
   //t_Bool bCarplayWifiActive = bIsCarplayActiveOverWiFi(corfrDeviceInfo);
   tenDeviceType enDevicetype = e8_UNKNOWN_DEVICE;
   if (NULL != m_poDeviceListHandler)
   {
      enDevicetype = m_poDeviceListHandler->enGetDeviceType(corfrDeviceInfo.u32DeviceHandle);
   }

   if (NULL != m_poDeviceSwitcher)
   {
      trDeviceSwitchInfo rDeviceSwitchInfo;
      m_poDeviceSwitcher->vGetDeviceSwitchInfo(corfrDeviceInfo.u32DeviceHandle, rDeviceSwitchInfo);

      if((true == rDeviceSwitchInfo.bSwitchInProgress) || (true == rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress))
      {
         t_Bool bUSBResetforiPod = ((e8_APPLE_DEVICE == enDevicetype) &&(e8DEV_TYPE_UNKNOWN == rDeviceSwitchInfo.enDeviceSwitchType)   &&
                  ((e8_PROFILE_DEFAULT == corfrDeviceInfo.enDeviceProfile) || (e8_PROFILE_UNKNOWN == corfrDeviceInfo.enDeviceProfile)));
         ETG_TRACE_USR4(("spi_tclDeviceDiscoverer::vResetDeviceSwitchInfo: bUSBResetforiPod = %d", ETG_ENUM(BOOL, bUSBResetforiPod)));

         //! Check if USB reset caused mirrorlink device to reappear in mirrorlink mode
         t_Bool bUSBResetforMirrorlink = ((e8DEV_TYPE_UNKNOWN == rDeviceSwitchInfo.enDeviceSwitchType) &&
                  (e8DEV_TYPE_MIRRORLINK == corfrDeviceInfo.enDeviceCategory) && (true == rDeviceSwitchInfo.bSwitchInProgress));
         ETG_TRACE_USR4(("spi_tclDeviceDiscoverer::vResetDeviceSwitchInfo: bUSBResetforMirrorlink = %d", ETG_ENUM(BOOL, bUSBResetforMirrorlink)));

         //e8DEV_TYPE_UNKNOWN == rDeviceSwitchInfo.enDeviceSwitchType check is added because, device switch type to e8DEV_TYPE_UNKNOWN will be requested  for USB reset. Otherwise continuous animation will be shown on the HMI till the timer expires.
         if (((true == rDeviceSwitchInfo.bSwitchInProgress) ||(true == rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress))
                   && ((((corfrDeviceInfo.enDeviceCategory == rDeviceSwitchInfo.enDeviceSwitchType) && (corfrDeviceInfo.enDeviceConnectionType == rDeviceSwitchInfo.enDeviceConnectionType)) ||
               (((e8_APPLE_DEVICE == enDevicetype) && (true == bUSBResetforiPod)) ||
               ((e8_APPLE_DEVICE != enDevicetype) && (e8DEV_TYPE_UNKNOWN == rDeviceSwitchInfo.enDeviceSwitchType))))
                  || (((e8DEV_TYPE_CARLIFE == rDeviceSwitchInfo.enDeviceSwitchType)||
                  (e8DEV_TYPE_ONCAR == rDeviceSwitchInfo.enDeviceSwitchType) ||
                  (e8DEV_TYPE_MYSPIN == rDeviceSwitchInfo.enDeviceSwitchType))&& (e8_ANDROID_DEVICE == enDevicetype))
               || (true == bUSBResetforMirrorlink)))//TODO - remove when proper category is populated for Android devices
         {
            m_oDeviceSwitchLock.s16Lock();
            m_poDeviceSwitcher->vResetDeviceSwitchInfo(corfrDeviceInfo.u32DeviceHandle);
            m_oDeviceSwitchLock.vUnlock();
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vOnDeviceConnectionCb
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vOnDeviceConnectionCb(const trDeviceInfo& corfrDeviceInfo,
         tenDiscovererType enDiscovererType /*const t_U32 cou32DeviceHandle,
          const trDeviceInfo& corfrDeviceInfo, tenDeviceCategory enReportingDiscoverer*/)
{
   t_U32 u32InvalidDeviceHandle = scou32InvalidDeviceHandle;
   t_String szSerialNumber;
   tenSPISupport enSPISupport = e8SPI_SUPPORT_UNKNOWN;
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vOnDeviceConnectionCb: cou32DeviceHandle = 0x%x Device Category = %d Discoverer type = %d ",
            corfrDeviceInfo.u32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY, corfrDeviceInfo.enDeviceCategory), ETG_ENUM(DISCOVERER_TYPE, enDiscovererType)));

   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vOnDeviceConnectionCb: cou32DeviceHandle = 0x%x ", corfrDeviceInfo.u32DeviceHandle));

   if (true == m_bIsMasterClientAvailable)
   {
   //! check whether the Device with the serial number exsits by finding the crc of BT Mac
   u32InvalidDeviceHandle = u16CalcCRC16((const t_UChar*)corfrDeviceInfo.szBTAddress.c_str(),corfrDeviceInfo.szBTAddress.size());
   if(NULL != m_poDeviceListHandler)
   {
      tenDeviceCategory enDeviceCategory = m_poDeviceListHandler->enGetDeviceCategory(u32InvalidDeviceHandle);
      m_poDeviceListHandler->vGetSerialNumber(szSerialNumber, u32InvalidDeviceHandle);
      enSPISupport = m_poDeviceListHandler->enGetSPISupport(u32InvalidDeviceHandle, enDeviceCategory);
   }

   if((NULL != m_poDeviceListHandler) && ("invalid" == szSerialNumber)&&(e8SPI_NOTSUPPORTED == enSPISupport) && ("invalid" != corfrDeviceInfo.szSerialNumber))
   {
      //! if So Delete that invalid device handle first and then proceed further
      t_Bool bIsDeviceRemoved = m_poDeviceListHandler->bRemoveDeviceFromList(u32InvalidDeviceHandle);
      if(true == bIsDeviceRemoved)
      {
         m_poDeviceListHandler->vRemoveDeviceFromHistory(u32InvalidDeviceHandle);
      }
   }
   vProceedwithConnection(corfrDeviceInfo, enDiscovererType);

   }
   else
   {
      trDiscoveredDevicesInfo pendingDevicesInfo(corfrDeviceInfo,enDiscovererType);
      m_oPendingConnectionsLock.s16Lock();
      m_vecPendingDeviceConnection.push_back(pendingDevicesInfo);
      m_oPendingConnectionsLock.vUnlock();
      ETG_TRACE_USR1(("[DEBUG]spi_tclDeviceDiscoverer::vOnDeviceConnectionCb: push_back to pendingDevicesInfo: cou32DeviceHandle = 0x%x ", corfrDeviceInfo.u32DeviceHandle));
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vOnDeviceDisconnectionCb
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vOnDeviceDisconnectionCb(const t_U32 cou32DeviceHandle,
         tenDiscovererType enDiscovererType)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vOnDeviceDisconnectionCb: DeviceHandle = 0x%x DiscovererType = %d", cou32DeviceHandle, ETG_ENUM(DISCOVERER_TYPE,
            enDiscovererType)));

   tenDeviceConnectionType enDeviceConnectionType = e8UNKNOWN_CONNECTION;

   t_Bool bSwitchinProgress = false;
   if ((NULL != m_poDeviceSwitcher))
   {
      bSwitchinProgress = m_poDeviceSwitcher->bIsSwitchInProgress(cou32DeviceHandle);
   }

   t_Bool isDeviceDisconnectedOnBT = true;
   if ((NULL != m_poDeviceListHandler) && (e8_DISCOVERER_TYPE_BT == enDiscovererType))
   {
      isDeviceDisconnectedOnBT = (e8_SESSION_TRANSPORT_UNKNOWN
               == m_poDeviceListHandler->enGetSessionTransport(cou32DeviceHandle));
   }

   //! Check the discoverer reporting the device disconnection Ex Mirrorlink/Android Auto
   //! This is to prevent false device disconnections reported by Mirrorlink
   //! during Android Auto session

   if ((NULL != m_poDeviceListHandler) && (false == bSwitchinProgress) && (0 != cou32DeviceHandle)
            && (true == isDeviceDisconnectedOnBT))
   {
      ETG_TRACE_USR2(("Device 0x%x  is disconnected from system", cou32DeviceHandle));
      //delete current device from pending connections list
      m_oPendingConnectionsLock.s16Lock();
      std::vector<trDiscoveredDevicesInfo>::iterator itDevList = m_vecPendingDeviceConnection.begin();
      for (;itDevList != m_vecPendingDeviceConnection.end();)
      {
         if(itDevList->m_prDeviceInfo.u32DeviceHandle == cou32DeviceHandle)
         {
            itDevList = m_vecPendingDeviceConnection.erase(itDevList);
         }
         else
         {
            itDevList++;
         }
      }
      m_oPendingConnectionsLock.vUnlock();
      enDeviceConnectionType = m_poDeviceListHandler->enGetDeviceConnType(cou32DeviceHandle);
      m_poDeviceListHandler->vSetDeviceProfile(cou32DeviceHandle, e8_PROFILE_DEFAULT);

      if (true == m_poDeviceListHandler->bIsDeviceConnected(cou32DeviceHandle))
      {
         //! check if selected device was disconnected
         if (cou32DeviceHandle == m_poDeviceListHandler->u32GetSelectedDevice())
         {
            //! Set the device state to  not selected
            m_poDeviceListHandler->vSetDeviceSelection(cou32DeviceHandle, false);
         }
         else
         {
            //! Remove the device from list if its was never selected
            if (false == m_poDeviceListHandler->bWasDeviceSelected(cou32DeviceHandle))
            {
               m_poDeviceListHandler->bRemoveDeviceFromList(cou32DeviceHandle);
            }
         }

         //! Set the connection status to not connected
         m_poDeviceListHandler->vSetDeviceConnectionStatus(cou32DeviceHandle, e8DEV_NOT_CONNECTED);

         if(e8USB_CONNECTED == enDeviceConnectionType)
         {
            //! Set the Device Switch State to UNKNOWN on DISCONNECTION
            m_poDeviceListHandler->vSetDeviceSwitchState(cou32DeviceHandle, e8_DEVICESWITCHSTATE_UNKNONW);
         }

         //! Send Device list change
         if (NULL != m_poConnMngrResp)
         {
            m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle, enDeviceConnectionType, e8DEVICE_REMOVED);
         } //if (NULL != m_poConnMngrResp)

         trDeviceSelectionInfo rSelInfo;
         m_poDeviceListHandler->vGetDeviceSelectionInfo(cou32DeviceHandle, rSelInfo);

         //! Send Device disconnection to SPI components if deselection is not already triggered
         if ((NULL != m_poMediator)
                  && (e8_SELECTION_STATE_DEVICE_DESELECTION_INPROGRESS != rSelInfo.enDeviceSelectionState))
         {
            m_poMediator->vPostDeviceDisconnection(cou32DeviceHandle);
         } // if (NULL != poMediator)
      }
      else
      {
         ETG_TRACE_ERR(("Invalid Device : DeviceHandle = 0x%x is not connected! or connected is on WiFi", cou32DeviceHandle));
      } //if((NULL != m_poDeviceListHandler) && ...
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vOnDeviceSwitchCompleteCb
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vOnDeviceSwitchCompleteCb(const t_U32 cou32DeviceHandle, t_Bool bSwitchStatus)
{
   ETG_TRACE_USR1(("[PARAM] spi_tclDeviceDiscoverer::vOnDeviceSwitchCompleteCb cou32DeviceID =0x%x bSwitchStatus = %d",
            cou32DeviceHandle, ETG_ENUM(BOOL,bSwitchStatus)));

   if((m_rDeviceSelectionRequest.m_u32DeviceHandle == cou32DeviceHandle) &&
            (e8DEV_TYPE_UNKNOWN != m_rDeviceSelectionRequest.m_enDevCategory) &&
            (e8DEVCONNREQ_SELECT == m_rDeviceSelectionRequest.m_enDevConnReq))
   {
      tenErrorCode enErrorCode = (true == bSwitchStatus) ? e8NO_ERRORS : e8DEVICE_SWITCH_FAILED;

      //if no ERRORS, set the Device Switch state to PROJECTION/DEFAULT
      if(NULL != m_poDeviceListHandler)
      {
         if(e8NO_ERRORS == enErrorCode)
         {
            m_poDeviceListHandler->vSetDeviceSwitchState(cou32DeviceHandle, e8_DEVICESWITCHSTATE_PROJECTION);
         }
         else
         {
            m_poDeviceListHandler->vSetDeviceSwitchState(cou32DeviceHandle, e8_DEVICESWITCHSTATE_DEFAULT);
         }
      }

      vSelectDeviceResultCb(cou32DeviceHandle, enErrorCode);
   }

   trDeviceSelectionInfo rSelInfo;
   if(NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->vGetDeviceSelectionInfo(cou32DeviceHandle,rSelInfo);
   }
   if (e8_SELECTION_STATE_DEVICE_REENUMERATION_INPROGRESS == rSelInfo.enDeviceSelectionState)
   {
      ETG_TRACE_USR1(("[PARAM] spi_tclDeviceDiscoverer::vOnDeviceSwitchCompleteCb Device Under re-enumerating = %d",
               cou32DeviceHandle));
      vOnDeviceReenumeration(cou32DeviceHandle);
   }

    if ((NULL != m_pDiscoveryData) && (NULL != m_poDeviceListHandler) && (NULL != m_poConnMngrResp)
            && (NULL != m_poDeviceSwitcher))
   {
      trUSBDeviceInfo rUSBDeviceInfo;
      trDeviceSwitchInfo rDeviceSwitchInfo;
      m_pDiscoveryData->vGetUSBDeviceDetails(cou32DeviceHandle, rUSBDeviceInfo);
      m_poDeviceSwitcher->vGetDeviceSwitchInfo(cou32DeviceHandle, rDeviceSwitchInfo);

      if ((false == bSwitchStatus) && (e8DEV_TYPE_MIRRORLINK == rDeviceSwitchInfo.enDeviceSwitchType)
               && (true == rUSBDeviceInfo.bIsAOAPSupported))
      {
         trDeviceInfo rDeviceInfo;
         rDeviceInfo.u32DeviceHandle = rUSBDeviceInfo.u32DeviceHandle;
         rDeviceInfo.u32VendorID = rUSBDeviceInfo.u32VendorID;
         rDeviceInfo.u32ProductID = rUSBDeviceInfo.u32ProductID;
         rDeviceInfo.szSerialNumber = rUSBDeviceInfo.szSerialNumber;
         rDeviceInfo.enDeviceConnectionStatus = rUSBDeviceInfo.enConnectionStatus;
         rDeviceInfo.szDeviceManufacturerName = rUSBDeviceInfo.szManufacturer;
         rDeviceInfo.szDeviceModelName = rUSBDeviceInfo.szProduct;
         rDeviceInfo.rProjectionCapability.enAndroidAutoSupport = e8SPI_SUPPORTED;
         rDeviceInfo.rProjectionCapability.enMirrorlinkSupport = e8SPI_NOTSUPPORTED;

         m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle,
                  e8USB_CONNECTED,
                  m_poDeviceListHandler->enAddDeviceToList(cou32DeviceHandle, rDeviceInfo));
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vSelectDeviceResultCb
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vSelectDeviceResultCb(t_U32 cou32DeviceHandle, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vSelectDeviceResultCb enErrorCode = %d ", ETG_ENUM(ERROR_CODE, enErrorCode)));

   if(cou32DeviceHandle == m_u32DeviceUnderSelection)
   {
      m_u32DeviceUnderSelection = 0;

      //! Post Select device result
      if (NULL != m_poMediator)
      {
         m_poMediator->vPostSelectDeviceRes(e32COMPID_DISCOVERER, enErrorCode);
      } //if (NULL != poMediator)
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::bRequestDeviceSwitch
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bRequestDeviceSwitch(const t_U32 cou32DeviceHandle, tenDeviceCategory enDevCat)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::bRequestDeviceSwitch() Device Handle = %d  switchtype = %d", cou32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY,
            enDevCat)));
   t_Bool bRetVal = false;
   tenDeviceType enDevicetype = e8_UNKNOWN_DEVICE;
   if (NULL != m_poDeviceListHandler)
   {
      enDevicetype = m_poDeviceListHandler->enGetDeviceType(cou32DeviceHandle);
      if ((e8DEV_TYPE_UNKNOWN == enDevCat))
      {
         trDeviceSelectionInfo rSelInfo;
         rSelInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_REENUMERATION_INPROGRESS;
         rSelInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_NONE;
         m_poDeviceListHandler->vSetDeviceSelectionInfo(cou32DeviceHandle, rSelInfo);
      }
      else if (e8DEV_TYPE_CARLIFE == enDevCat) //TODO device check has to be removed when connection progress state is implemented for all technologies
      {
         m_poDeviceListHandler->vSetSelectionProgressState(cou32DeviceHandle,
                  e8_SELECTION_PROGRESS_STATE_SWITCHING_PROFILE);
         if (NULL != m_poConnMngrResp)
         {
            m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle, e8USB_CONNECTED, e8DEVICE_CHANGED);
         }
      }
   }
   trDeviceSwitchInfo rDeviceSwitchInfo;

   m_oDeviceSwitchLock.s16Lock();
   if (NULL != m_poDeviceSwitcher)
   {
      m_poDeviceSwitcher->vGetDeviceSwitchInfo(cou32DeviceHandle, rDeviceSwitchInfo);
   }
   m_oDeviceSwitchLock.vUnlock();

   m_LockDiscoverers.s16Lock();
   for (auto itmapDiscoverers = m_mapoDiscoverers.begin(); itmapDiscoverers != m_mapoDiscoverers.end();
            itmapDiscoverers++)
   {
      if (NULL != itmapDiscoverers->second)
      {
         itmapDiscoverers->second->vSetDeviceSwitchInfo(cou32DeviceHandle, rDeviceSwitchInfo);
      }
   }
   m_LockDiscoverers.vUnlock();

   m_oDeviceSwitchLock.s16Lock();
   if (NULL != m_poDeviceSwitcher)
   {
      bRetVal = m_poDeviceSwitcher->bSwitchDeviceMode(enDevCat, cou32DeviceHandle, enDevicetype);
   }
   m_oDeviceSwitchLock.vUnlock();

   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vProceedwithConnection
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vProceedwithConnection(const trDeviceInfo& corfrDeviceInfo,
         tenDiscovererType enDiscovererType)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vProceedwithConnection () "));

   ETG_TRACE_USR4(("spi_tclDeviceDiscoverer::vProceedwithConnection Device Handle = 0x%x  "
            "has ConnType = %d ", corfrDeviceInfo.u32DeviceHandle, ETG_ENUM(CONNECTION_TYPE,
            corfrDeviceInfo.enDeviceConnectionType)));

   if ((NULL != m_poDeviceListHandler) && (0 != corfrDeviceInfo.u32DeviceHandle))
   {
      ETG_TRACE_USR2(("DeviceHandle = 0x%x of device category %d is connected to the system with Device type =%d ", corfrDeviceInfo.u32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY,
               corfrDeviceInfo.enDeviceCategory), ETG_ENUM(DEVICE_TYPE,
               corfrDeviceInfo.rProjectionCapability.enDeviceType)));
      trDeviceInfo rDeviceInfo = corfrDeviceInfo;

      tenDeviceStatusInfo enDevStatus = e8DEVICE_STATUS_NOT_KNOWN;
      tenDeviceCategory enChosenDeviceCategory = enChooseDeviceCategory(rDeviceInfo.u32DeviceHandle,
               rDeviceInfo.rProjectionCapability.enDeviceType);
      vEvaluateErrorStates(rDeviceInfo.u32DeviceHandle, enChosenDeviceCategory);
      t_Bool bIsWLEnabledCat = false;
      t_Bool bIsDeviceWhiteListed = false;

      //! Whitelisting is added to check whether a device that is connected to HU can be allowed for a 
      //! particular technology. In case of AIVI project, only A+T box has to be allowed for ML and other
      //! devices must be blocked for ML. This is done based on the list maintained by HMI which contains
      //! the vendor and product ids along with category for which the device is allowed.

      if ((NULL != m_poDiscovererSettings) && (true == m_poDiscovererSettings->bIsDeviceWhitelistingEnabled()))
      {
         m_poDiscovererSettings->vIsDeviceWhitelisted(corfrDeviceInfo.u32ProductID,
                  corfrDeviceInfo.u32VendorID,
                  rDeviceInfo.enDeviceCategory);
         if (rDeviceInfo.enDeviceCategory != e8DEV_TYPE_UNKNOWN)
         {
            enChosenDeviceCategory = rDeviceInfo.enDeviceCategory;
            bIsDeviceWhiteListed = true;
         }
         else
         {
            //Not whitelisted ML Device, SPI is not supposed to select that device
            bIsWLEnabledCat = m_poDiscovererSettings->bIsWhitelistEnabledCat(enChosenDeviceCategory);
         }
      }
      if (false == bIsWLEnabledCat)
      {
         if ((e8_DISCOVERER_TYPE_ML != enDiscovererType) && (e8DEV_TYPE_MIRRORLINK == enChosenDeviceCategory))
         {
            vResetDeviceSwitchInfo(corfrDeviceInfo);
            vHandleMirrorlinkDevices(corfrDeviceInfo);
         }
         else
         {
            //! If the mirrorlink device is already in CDC NCM mode, update the detaisl into USB list
            if ((e8_PROFILE_CDCNCM_MIRRORLINK == rDeviceInfo.enDeviceProfile)
                     || (e8_PROFILE_CDCNCM_UNKNOWNMODE == rDeviceInfo.enDeviceProfile))
            {
               if (NULL != m_pDiscoveryData)
               {
                  trUSBDeviceInfo rUSBDeviceInfo;
                  if (false == m_pDiscoveryData->bIsDeviceInfoExists(rDeviceInfo.u32DeviceHandle))
                  {
                     rUSBDeviceInfo.u32DeviceHandle = rDeviceInfo.u32DeviceHandle;
                     rUSBDeviceInfo.szSerialNumber = rDeviceInfo.szSerialNumber;
                     rUSBDeviceInfo.enConnectionStatus = e8DEV_CONNECTED;
                     rUSBDeviceInfo.enUSBProfile = rDeviceInfo.enDeviceProfile;
                  }
                  else
                  {
                     m_pDiscoveryData->vGetUSBDeviceDetails(rDeviceInfo.u32DeviceHandle, rUSBDeviceInfo);
                     rUSBDeviceInfo.enUSBProfile = rDeviceInfo.enDeviceProfile;
                  }
                  m_pDiscoveryData->vSetUSBDeviceDetails(rDeviceInfo.u32DeviceHandle, rUSBDeviceInfo);
               }
            }

            rDeviceInfo.enDeviceCategory = enChosenDeviceCategory;
            enDevStatus = m_poDeviceListHandler->enAddDeviceToList(rDeviceInfo.u32DeviceHandle, rDeviceInfo);

            m_poDeviceListHandler->vSetDeviceWhitelistedInfo(rDeviceInfo.u32DeviceHandle, bIsDeviceWhiteListed);

            tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
            m_poDeviceListHandler->vGetDeviceUsagePreference(rDeviceInfo.u32DeviceHandle, enEnabledInfo);

            //If a new device is connected, the device usage enabled value is unknown, this value has
            //to be changed to the overall setting for the device
            if (e8USAGE_UNKNOWN == enEnabledInfo)
            {
               ETG_TRACE_USR1((" spi_tclDeviceDiscoverer::vProceedwithConnection Device Usage info= %d ", ETG_ENUM(ENABLED_INFO,
                        rDeviceInfo.enDeviceUsageEnabled)));
               tenEnabledInfo enOverallSetting = e8USAGE_UNKNOWN;
               m_poDeviceListHandler->bGetDevProjUsage(enChosenDeviceCategory,
                        enOverallSetting,
                        rDeviceInfo.rProjectionCapability.enDeviceType);

               m_poDeviceListHandler->vSetDeviceUsagePreference(rDeviceInfo.u32DeviceHandle, enOverallSetting);
            }

            //! Send Device list change only if there is a change in device status
            if ((NULL != m_poConnMngrResp) && (e8DEVICE_STATUS_NOT_KNOWN != enDevStatus))
            {
               //! If Device Status is ADDED - the set the Device Switch State as DEFAUL
               if((e8DEVICE_ADDED == enDevStatus)&& (e8USB_CONNECTED == corfrDeviceInfo.enDeviceConnectionType))
               {
                  m_poDeviceListHandler->vSetDeviceSwitchState(rDeviceInfo.u32DeviceHandle, e8_DEVICESWITCHSTATE_DEFAULT);
               }

               m_poConnMngrResp->vPostDeviceStatusInfo(rDeviceInfo.u32DeviceHandle,
                        rDeviceInfo.enDeviceConnectionType,
                        enDevStatus);
            } //if (NULL != m_poConnMngrResp)
              //! Reset switch info once device has reappeared in projection mode
            vResetDeviceSwitchInfo(corfrDeviceInfo);
            //! inform subcomponents about new device connection
            //! Ex: for automatic selection
            //! this update will trigger only if the device move from DISCONNECTED state to CONNECTED state.
            t_U32 u32SelectedDevice = m_poDeviceListHandler->u32GetSelectedDevice();
            if ((e8DEVICE_ADDED == enDevStatus))
            {
               ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vProceedwithConnection DEVICE_ADDED "));
               if ((NULL != m_poMediator)/* && (0 == u32SelectedDevice)*/)
               {
                  m_poMediator->vPostDeviceConnection(rDeviceInfo.u32DeviceHandle);
               }
            }      // if (0 == u32SelectedDevice)
            else if (e8DEVICE_CHANGED == enDevStatus)
            {
               ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vProceedwithConnection DEVICE_Changed"));
               if (rDeviceInfo.u32DeviceHandle == u32SelectedDevice)
               {
                  ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vProceedwithConnection DEVICE_Changed and selected"));
                  //! Update device history if the selected device changes
                  m_poDeviceListHandler->vAddDeviceToHistory(rDeviceInfo.u32DeviceHandle);
               }
               ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vProceedwithConnection- Device Profile :%d", ETG_ENUM(USB_PROFILE,
                        rDeviceInfo.enDeviceProfile)));

               ETG_TRACE_USR2(("spi_tclDeviceDiscoverer::vProceedwithConnection DeviceHandle = 0x%x of device category %d is connected to the system ", rDeviceInfo.u32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY,
                        rDeviceInfo.enDeviceCategory)));

               ETG_TRACE_USR2(("spi_tclDeviceDiscoverer::vProceedwithConnection(enChosenDeviceCategory) DeviceHandle = 0x%x of device category %d is connected to the system ", rDeviceInfo.u32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY,
                        enChosenDeviceCategory)));

               if (e8_PROFILE_IAP2BT_CPW_NOT_FEASIBLE == rDeviceInfo.enDeviceProfile)
               {
                  ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vProceedwithConnection since the Device profile is IAP2BT_CPW_NOT_FEASIBLE - Triggering DeviceDisconnection"));
                  m_poMediator->vPostDeviceChanged(rDeviceInfo.u32DeviceHandle);
               }
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("Invalid Device: corfrDeviceInfo.u32DeviceHandle = 0x%x ", corfrDeviceInfo.u32DeviceHandle));
   }   //if((NULL != m_poDeviceListHandler) && ...
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::enChooseDeviceCategory
 ***************************************************************************/
tenDeviceCategory spi_tclDeviceDiscoverer::enChooseDeviceCategory(const t_U32 cou32DeviceHandle,
         tenDeviceType enDeviceType)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::enChooseDeviceCategory() Device Handle = %d ", cou32DeviceHandle));
   tenDeviceCategory enCurrentDeviceCategory = e8DEV_TYPE_UNKNOWN;
   tenEnabledInfo enEnabledInfo = e8USAGE_DISABLED;
   tenEnabledInfo enMLEnabled = e8USAGE_DISABLED;
   if (NULL != m_poDeviceListHandler)
   {
      //fetching the Devicecategory that was in the previous session in the same ignition cycle
      enCurrentDeviceCategory = m_poDeviceListHandler->enGetDeviceCategory(cou32DeviceHandle);
      if (e8DEV_TYPE_UNKNOWN != enCurrentDeviceCategory)
      {
         m_poDeviceListHandler->bGetDevProjUsage(enCurrentDeviceCategory, enEnabledInfo, enDeviceType);
      }
      m_poDeviceListHandler->bGetDevProjUsage(e8DEV_TYPE_MIRRORLINK, enMLEnabled, enDeviceType);
   }
   tenDeviceCategory enChosenDeviceCategory = enCurrentDeviceCategory;
   if ((e8DEV_TYPE_UNKNOWN == enCurrentDeviceCategory) || (enEnabledInfo == e8USAGE_DISABLED))
   {
      if (e8_ANDROID_DEVICE == enDeviceType)
      {
         enChosenDeviceCategory = enChooseAndroidDeviceCategory(enCurrentDeviceCategory);
      }
      else if (e8_APPLE_DEVICE == enDeviceType)
      {
         enChosenDeviceCategory = enChooseAppleDeviceCategory(cou32DeviceHandle);
      }
      else if (e8USAGE_DISABLED != enMLEnabled)
      {
         //! if device type is not known
         enChosenDeviceCategory = e8DEV_TYPE_MIRRORLINK;
      }
   }
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::enChooseDeviceCategory() Choosing %d technology for Device Handle = %d ", ETG_ENUM(DEVICE_CATEGORY,
            enChosenDeviceCategory), cou32DeviceHandle));
   return enChosenDeviceCategory;

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::enChooseAndroidDeviceCategory
 ***************************************************************************/
tenDeviceCategory spi_tclDeviceDiscoverer::enChooseAndroidDeviceCategory(tenDeviceCategory enCurrentDeviceCategory)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::enChooseAndroidDeviceCategory entered"));

   tenDeviceCategory enRequiredDeviceSwitch = e8DEV_TYPE_UNKNOWN;

   //! check preferred technology on a newly connected device
   tenDeviceCategory enPrefDeviceCategory = enCurrentDeviceCategory;
   if ((NULL != m_poDeviceListHandler) && (e8DEV_TYPE_UNKNOWN == enCurrentDeviceCategory))
   {
      enPrefDeviceCategory = m_poDeviceListHandler->enGetTechnologyPreference(e8_ANDROID_DEVICE);
   } //if (NULL != m_poDeviceListHandler)

   //! Check if the preferred technology is enabled
   tenEnabledInfo enPrefTechEnabled = e8USAGE_DISABLED;
   if (NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->bGetDevProjUsage(enPrefDeviceCategory, enPrefTechEnabled, e8_ANDROID_DEVICE);
   }
   t_Bool bPrefTechSupported = ((e8USAGE_ENABLED == enPrefTechEnabled) || (e8USAGE_CONF_REQD == enPrefTechEnabled));

   enRequiredDeviceSwitch = enPrefDeviceCategory;
   //! If preferred technology is not enabled, choose one of the other enabled technology for android device
   if ((false == bPrefTechSupported) && (NULL != m_poDeviceListHandler))
   {
      //! Check for the currently enabled SPI technologies
      tenEnabledInfo enMLEnabled = e8USAGE_DISABLED;
      tenEnabledInfo enAAPEnabled = e8USAGE_DISABLED;
      tenEnabledInfo enMySPINEnabled = e8USAGE_DISABLED;
      tenEnabledInfo enBDCLEnabled = e8USAGE_DISABLED;
      tenEnabledInfo enOnCarEnabled = e8USAGE_DISABLED;
      m_poDeviceListHandler->bGetDevProjUsage(e8DEV_TYPE_MIRRORLINK, enMLEnabled, e8_ANDROID_DEVICE);
      m_poDeviceListHandler->bGetDevProjUsage(e8DEV_TYPE_ANDROIDAUTO, enAAPEnabled, e8_ANDROID_DEVICE);
      m_poDeviceListHandler->bGetDevProjUsage(e8DEV_TYPE_MYSPIN, enMySPINEnabled, e8_ANDROID_DEVICE);
      m_poDeviceListHandler->bGetDevProjUsage(e8DEV_TYPE_CARLIFE, enBDCLEnabled, e8_ANDROID_DEVICE);
      m_poDeviceListHandler->bGetDevProjUsage(e8DEV_TYPE_ONCAR, enOnCarEnabled, e8_ANDROID_DEVICE);

      t_Bool bMLSupported = ((e8USAGE_ENABLED == enMLEnabled) || (e8USAGE_CONF_REQD == enMLEnabled));
      t_Bool bAAPSupported = ((e8USAGE_ENABLED == enAAPEnabled) || (e8USAGE_CONF_REQD == enAAPEnabled));
      t_Bool bMySPINSupported = ((e8USAGE_ENABLED == enMySPINEnabled) || (e8USAGE_CONF_REQD == enMySPINEnabled));
      t_Bool bBDCLSupported = ((e8USAGE_ENABLED == enBDCLEnabled) || (e8USAGE_CONF_REQD == enBDCLEnabled));
      t_Bool bOnCarSupported = ((e8USAGE_ENABLED == enOnCarEnabled) || (e8USAGE_CONF_REQD == enOnCarEnabled));

      if (true == bAAPSupported)
      {
         enRequiredDeviceSwitch = e8DEV_TYPE_ANDROIDAUTO;
      }
      else if (true == bMLSupported)
      {
         enRequiredDeviceSwitch = e8DEV_TYPE_MIRRORLINK;
      }
      // else if one of them is supported then choose that
      else if (true == bMySPINSupported)
      {
         enRequiredDeviceSwitch = e8DEV_TYPE_MYSPIN;
      }
      else if (true == bBDCLSupported)
      {
         enRequiredDeviceSwitch = e8DEV_TYPE_CARLIFE;
      }
      else if (true == bOnCarSupported)
      {
         enRequiredDeviceSwitch = e8DEV_TYPE_ONCAR;
      }
      else
      {
         enRequiredDeviceSwitch = enPrefDeviceCategory;
      }
      ETG_TRACE_USR4(("[PARAM] spi_tclDeviceDiscoverer::enChooseAndroidDeviceCategory(): "
               "Android Auto Enabled? =%d Mirrorlink Enabled? =%d "
               "mySPIN Enabled? =%d Baidu Carlife enabled? = %d ", ETG_ENUM(BOOL, bAAPSupported), ETG_ENUM(BOOL,
               bMLSupported), ETG_ENUM(BOOL, bMySPINSupported), ETG_ENUM(BOOL, bBDCLSupported)));
   }

   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::enChooseAndroidDeviceCategory(): "
            "Preferred device category =%d, Required device switch = %d", ETG_ENUM(DEVICE_CATEGORY,
            enPrefDeviceCategory), ETG_ENUM(DEVICE_CATEGORY, enRequiredDeviceSwitch)));
   return enRequiredDeviceSwitch;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::enChooseAppleDeviceCategory
 ***************************************************************************/
tenDeviceCategory spi_tclDeviceDiscoverer::enChooseAppleDeviceCategory(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::enChooseAppleDeviceCategory entered"));

   tenDeviceCategory enRequiredDeviceSwitch = e8DEV_TYPE_UNKNOWN;
   t_Bool bIsTechSupportOnPhone = true;

   //! check preferred technology on a newly connected device
   tenDeviceCategory enPrefDeviceCategory = e8DEV_TYPE_DIPO;
   tenSPISupport enCarplaySupport = e8SPI_SUPPORT_UNKNOWN;
   if (NULL != m_poDeviceListHandler)
   {
      enPrefDeviceCategory = m_poDeviceListHandler->enGetTechnologyPreference(e8_APPLE_DEVICE);
      enCarplaySupport = m_poDeviceListHandler->enGetSPISupport(cou32DeviceHandle, e8DEV_TYPE_DIPO);
   } //if (NULL != m_poDeviceListHandler)

   enRequiredDeviceSwitch = enPrefDeviceCategory;
   //! Check if the preferred technology is enabled
   tenEnabledInfo enPrefTechEnabled = e8USAGE_DISABLED;
   if (NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->bGetDevProjUsage(enPrefDeviceCategory, enPrefTechEnabled, e8_APPLE_DEVICE);
   }

   t_Bool bPrefTechSupported = ((e8USAGE_ENABLED == enPrefTechEnabled) || (e8USAGE_CONF_REQD == enPrefTechEnabled));

   //! Fix for the ticket NCG3D-138961 As Carlay support can be enabled and disabled from phone side
   //! so bool variable bIsTechSupportOnPhone is added to confirm Carplay support in phone.
   //! for other technologies like mySPIN and Carlife, support can't be determined and hence bIsTechSupportOnPhone is true by default
   bIsTechSupportOnPhone = (((e8DEV_TYPE_DIPO == enPrefDeviceCategory) &&(e8SPI_SUPPORTED == enCarplaySupport)) || (e8DEV_TYPE_DIPO != enPrefDeviceCategory));

   //! If preferred technology is not enabled, choose one of the other enabled technology for apple device
   if (((false == bPrefTechSupported) || (false == bIsTechSupportOnPhone)) && (NULL != m_poDeviceListHandler))
   {
      tenEnabledInfo enMySPINEnabled = e8USAGE_DISABLED;
      tenEnabledInfo enCarplayEnabled = e8USAGE_DISABLED;
      tenEnabledInfo enBDCLEnabled = e8USAGE_DISABLED;

      m_poDeviceListHandler->bGetDevProjUsage(e8DEV_TYPE_DIPO, enCarplayEnabled, e8_APPLE_DEVICE);
      m_poDeviceListHandler->bGetDevProjUsage(e8DEV_TYPE_MYSPIN, enMySPINEnabled, e8_APPLE_DEVICE);
      m_poDeviceListHandler->bGetDevProjUsage(e8DEV_TYPE_CARLIFE, enBDCLEnabled, e8_APPLE_DEVICE);

      t_Bool bCarplayEnabledOnHU = ((e8USAGE_ENABLED == enCarplayEnabled) ||(e8USAGE_CONF_REQD == enCarplayEnabled));
      t_Bool bBDCLSupported = ((e8USAGE_ENABLED == enBDCLEnabled) || (e8USAGE_CONF_REQD == enBDCLEnabled));
      t_Bool bMySPINSupported = ((e8USAGE_ENABLED == enMySPINEnabled) || (e8USAGE_CONF_REQD == enMySPINEnabled));

      if ((true == bCarplayEnabledOnHU) && (true == bIsTechSupportOnPhone))
      {
         enRequiredDeviceSwitch = e8DEV_TYPE_DIPO;
      }
      else if (true == bBDCLSupported)
      {
         enRequiredDeviceSwitch = e8DEV_TYPE_CARLIFE;
      }
      else if (true == enMySPINEnabled)
      {
         enRequiredDeviceSwitch = e8DEV_TYPE_MYSPIN;
      }
      else
      {
         enRequiredDeviceSwitch = enPrefDeviceCategory;
      }
      ETG_TRACE_USR4(("[PARAM] spi_tclDeviceDiscoverer::enChooseAppleDeviceCategory(): Carplay Enabled? =%d"
               "  mySPIN Enabled? =%d ", ETG_ENUM(BOOL, bCarplayEnabledOnHU), ETG_ENUM(BOOL, bMySPINSupported)));
   }

   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::enChooseAppleDeviceCategory(): Preferred device category =%d, Required device switch = %d", ETG_ENUM(DEVICE_CATEGORY,
            enPrefDeviceCategory), ETG_ENUM(DEVICE_CATEGORY, enRequiredDeviceSwitch)));
   return enRequiredDeviceSwitch;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vEvaluateErrorStates
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vEvaluateErrorStates(const t_U32 cou32DeviceHandle,
         tenDeviceCategory enChosenDeviceCategory)
{
   ETG_TRACE_USR1(("[DESC] spi_tclConnMngr::vEvaluateErrorStates for 0x%x for category %d", cou32DeviceHandle, ETG_ENUM(DEVICE_CATEGORY,
            enChosenDeviceCategory)));
   tenDeviceCategory enExistingDevCat = e8DEV_TYPE_UNKNOWN;
   t_Bool bIsSelectError = false;
   if (NULL != m_poDeviceListHandler)
   {
      enExistingDevCat = m_poDeviceListHandler->enGetDeviceCategory(cou32DeviceHandle);
      bIsSelectError = m_poDeviceListHandler->bIsSelectError(cou32DeviceHandle);
      if ((enExistingDevCat != enChosenDeviceCategory) && (true == bIsSelectError))
      {
         //! Clear selection error if device is selected for a different technology
         m_poDeviceListHandler->vSetSelectError(cou32DeviceHandle, false);
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::vHandleMirrorlinkDevices
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vHandleMirrorlinkDevices(const trDeviceInfo& corfrDeviceInfo)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vHandleMirrorlinkDevices for 0x%x ", corfrDeviceInfo.u32DeviceHandle));

   t_U32 u32SelectedDevice = 0;
   if (NULL != m_poDeviceListHandler)
   {
      u32SelectedDevice = m_poDeviceListHandler->u32GetSelectedDevice();
   }
   //! Wait for switch to complete
   if ((NULL != m_poDeviceSwitcher) && (0 == u32SelectedDevice))
   {
      m_poDeviceSwitcher->vPrepareDeviceSwitch(e8DEV_TYPE_MIRRORLINK,
               corfrDeviceInfo.u32DeviceHandle,
               corfrDeviceInfo.enDeviceConnectionType);
      if (false == bRequestDeviceSwitch(corfrDeviceInfo.u32DeviceHandle, e8DEV_TYPE_MIRRORLINK))
      {
         if ((NULL != m_pDiscoveryData) && (NULL != m_poDeviceListHandler) && (NULL != m_poConnMngrResp))
         {
            trUSBDeviceInfo rUSBDeviceInfo;
            m_pDiscoveryData->vGetUSBDeviceDetails(corfrDeviceInfo.u32DeviceHandle, rUSBDeviceInfo);

            if (true == rUSBDeviceInfo.bIsAOAPSupported)
            {
               trDeviceInfo rDeviceInfo = corfrDeviceInfo;
               rDeviceInfo.rProjectionCapability.enAndroidAutoSupport = e8SPI_SUPPORTED;
               rDeviceInfo.rProjectionCapability.enMirrorlinkSupport = e8SPI_NOTSUPPORTED;
               m_poConnMngrResp->vPostDeviceStatusInfo(rDeviceInfo.u32DeviceHandle,
                        e8USB_CONNECTED,
                        m_poDeviceListHandler->enAddDeviceToList(rDeviceInfo.u32DeviceHandle, rDeviceInfo));
            }
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::bIsCarplayActiveOverWiFi
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bIsCarplayActiveOverWiFi(const trDeviceInfo& corfrDeviceInfo)
{
   t_Bool bSessionActive = false;
   t_U32 u32SelectedDevice = 0;
   tenSessionTransport enSessionTransport = e8_SESSION_TRANSPORT_UNKNOWN;
   if (NULL != m_poDeviceListHandler)
   {
      u32SelectedDevice = m_poDeviceListHandler->u32GetSelectedDevice();
      enSessionTransport = m_poDeviceListHandler->enGetSessionTransport(u32SelectedDevice);
   }
   bSessionActive = (u32SelectedDevice == corfrDeviceInfo.u32DeviceHandle)
            && (e8_SESSION_TRANSPORT_WIFI == enSessionTransport);
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::bIsCarplayActiveOverWiFi for 0x%x  Result = %d", corfrDeviceInfo.u32DeviceHandle, ETG_ENUM(BOOL,
            bSessionActive)));
   return bSessionActive;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::vSetDeviceConnectionCount
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vSetDeviceConnectionCount(const t_U32 cou32DeviceHandle,
         tenDeviceCategory enDeviceCategory)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceDiscoverer::vSetDeviceConnectionCount entered for device handle 0x%x ", cou32DeviceHandle));

   if (NULL != m_poDeviceListHandler)
   {
      trDeviceConnectionCountInfo rCurrDeviceConnectionCountInfo =
               m_poDeviceListHandler->rGetDeviceConnectionCountInfo(cou32DeviceHandle);

      for (t_U8 u8Index = 1; u8Index < MAX_DEV_CATEGORY; u8Index++)
      {
         ETG_TRACE_USR4(("[PARAM]::spi_tclDeviceDiscoverer::vSetDeviceConnectionCount::Connect count for %d  is %d ", ETG_ENUM(DEVICE_CATEGORY,
                  u8Index), rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectCount));
      }

      t_U8 u8Index = static_cast<t_U8>(enDeviceCategory);

      if (u8Index < MAX_DEV_CATEGORY)
      {
         rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectCount =
                  rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectCount + 1;
         ETG_TRACE_USR4(("[PARAM]::spi_tclDeviceDiscoverer::vSetDeviceConnectionCount::Updated Connected count for %d  is %d ", ETG_ENUM(DEVICE_CATEGORY,
                  enDeviceCategory), rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectCount));
      }
      else
      {
         ETG_TRACE_USR4(("spi_tclDeviceDiscoverer::vSetDeviceConnectionCount:: Invalid Device ID "));
      }

      m_poDeviceListHandler->vSetDeviceConnectionCountInfo(cou32DeviceHandle, rCurrDeviceConnectionCountInfo);
   }

}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::vSetDeviceConnectSuccessCount
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vSetDeviceConnectSuccessCount(const t_U32 cou32DeviceHandle,
         tenDeviceCategory enDeviceCategory)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vSetDeviceConnectSuccessCount entered for Device handle 0x%x ", cou32DeviceHandle));

   if (NULL != m_poDeviceListHandler)
   {
      trDeviceConnectionCountInfo rCurrDeviceConnectionCountInfo =
               m_poDeviceListHandler->rGetDeviceConnectionCountInfo(cou32DeviceHandle);

      for (t_U8 u8Index = 1; u8Index < MAX_DEV_CATEGORY; u8Index++)
      {
         ETG_TRACE_USR4(("[PARAM]::spi_tclDeviceDiscoverer::vSetDeviceConnectSuccessCount::Connected success count for %d  is %d ", ETG_ENUM(DEVICE_CATEGORY,
                  u8Index), rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectSuccessCount));
      }

      t_U8 u8Index = static_cast<t_U8>(enDeviceCategory);

      if (u8Index < MAX_DEV_CATEGORY)
      {
         rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectSuccessCount =
                  rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectSuccessCount + 1;
         ETG_TRACE_USR4(("[PARAM]::spi_tclDeviceDiscoverer::vSetDeviceConnectSuccessCount::Updated Connected success count for %d  is %d ", ETG_ENUM(DEVICE_CATEGORY,
                  enDeviceCategory), rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectSuccessCount));
      }
      else
      {
         ETG_TRACE_USR4(("spi_tclDeviceDiscoverer::vSetDeviceConnectSuccessCount:: Invalid Device ID "));
      }

      m_poDeviceListHandler->vSetDeviceConnectionCountInfo(cou32DeviceHandle, rCurrDeviceConnectionCountInfo);
   }

}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::vSetDeviceConnectionTime
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vSetDeviceConnectionTime(const t_U32 cou32DeviceHandle,
         tenDeviceCategory enDeviceCategory)
{
   ETG_TRACE_USR1((" spi_tclDeviceDiscoverer::vSetDeviceConnectionTime entered for Device handle 0x%x ", cou32DeviceHandle));

   if (NULL != m_poDeviceListHandler)
   {
      trDeviceConnectionCountInfo rCurrDeviceConnectionCountInfo =
               m_poDeviceListHandler->rGetDeviceConnectionCountInfo(cou32DeviceHandle);

      for (t_U8 u8Index = 1; u8Index < MAX_DEV_CATEGORY; u8Index++)
      {
         ETG_TRACE_USR4(("[PARAM]::spi_tclDeviceDiscoverer::vSetDeviceConnectionTime::Connect Time for %d  is %d ", ETG_ENUM(DEVICE_CATEGORY,
                  u8Index), rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectTime));
      }

      struct timeval rConnectionEndTime;
      //Time when the device connection is successful
      gettimeofday(&rConnectionEndTime, 0);
      //Calculate the time taken for session establishment in Miliseconds
      t_U32 u32ConnectionTime = static_cast<t_U32>((rConnectionEndTime.tv_sec - m_rConnectionStartTime.tv_sec) * 1000);

      t_U8 u8Index = static_cast<t_U8>(enDeviceCategory);

      if (u8Index < MAX_DEV_CATEGORY)
      {
         rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectTime = u32ConnectionTime;
         ETG_TRACE_USR4(("[PARAM]::spi_tclDeviceDiscoverer::vSetDeviceConnectionTime::Updated Connected Time for %d  is %d ", ETG_ENUM(DEVICE_CATEGORY,
                  enDeviceCategory), rCurrDeviceConnectionCountInfo.rDeviceStatisticInfo[u8Index].u32ConnectTime));
      }
      else
      {
         ETG_TRACE_USR4(("spi_tclDeviceDiscoverer::vSetDeviceConnectionTime:: Invalid Device ID "));
      }

      m_poDeviceListHandler->vSetDeviceConnectionCountInfo(cou32DeviceHandle, rCurrDeviceConnectionCountInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::vOnSelectDevice
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vOnSelectDevice(const trSelectDeviceRequest &rfrSelDeviceRequest)
{
   tenErrorCode enErrorCode = e8NO_ERRORS;
   tenDeviceSelectionState enUSBDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
   tenDeviceSelectionState enWifiDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;

   tenDeviceCategory enDevCat = rfrSelDeviceRequest.m_enDevCategory;
   t_Bool bSendStatus = false;
   //! Wait for switch to complete
   if (NULL != m_poDeviceSwitcher)
   {
      //! Whitelisting is added to check whether a device that is connected to HU can be allowed for a
      //! particular technology. In case of AIVI project, only A+T box has to be allowed for ML and other
      //! devices must be blocked for ML. This is done based on the list maintained by HMI which contains
      //! the vendor and product ids along with category for which the device is allowed.

      if ((NULL != m_poDiscovererSettings) && (true == m_poDiscovererSettings->bIsDeviceWhitelistingEnabled()))
      {
         t_U32 u32VendorID = m_poDeviceListHandler->u32GetVendorID(rfrSelDeviceRequest.m_u32DeviceHandle);
         t_U32 u32ProductID = m_poDeviceListHandler->u32GetProductID(rfrSelDeviceRequest.m_u32DeviceHandle);
         m_poDiscovererSettings->vIsDeviceWhitelisted(u32ProductID, u32VendorID, enDevCat);
         if (enDevCat == e8DEV_TYPE_UNKNOWN)
         {
            t_Bool bIsWLEnabledCat =
                     m_poDiscovererSettings->bIsWhitelistEnabledCat(rfrSelDeviceRequest.m_enDevCategory);
            if (true == bIsWLEnabledCat)
            {
               //! If device category is whitelisted category, then send selection error if the device is not whitelisted
               enErrorCode = e8SELECTION_FAILED;
            }
            else
            {
               enDevCat = rfrSelDeviceRequest.m_enDevCategory;
               if (NULL != m_poConnMngrResp)
               {
                  m_poConnMngrResp->vPostDeviceStatusInfo(rfrSelDeviceRequest.m_u32DeviceHandle,
                           rfrSelDeviceRequest.m_enDevConnType,
                           e8DEVICE_CHANGED);
               } //if (NULL != m_poConnMngrResp)
            }
         }
      }
      if (enDevCat != e8DEV_TYPE_UNKNOWN)
      {
         m_poDeviceSwitcher->vPrepareDeviceSwitch(enDevCat,
                  rfrSelDeviceRequest.m_u32DeviceHandle,
                  rfrSelDeviceRequest.m_enDevConnType);
      }
   }
   if ((e8NO_ERRORS == enErrorCode) && (enDevCat != e8DEV_TYPE_UNKNOWN))
   {
      if ((enDevCat == e8DEV_TYPE_CARLIFE) && (rfrSelDeviceRequest.m_enDeviceType == e8_APPLE_DEVICE))
      {
         ETG_TRACE_USR2(("[DESC] spi_tclDeviceDiscoverer:: vOnSelectDevice : Sending roleswitch response for Carlife"));
         bSendStatus = bSendRoleSwitchResponse(rfrSelDeviceRequest.m_u32DeviceHandle,
                  e8DIPO_ROLE_SWITCH_REQUIRED_FOR_CARLIFE);
      }
      else if (((enDevCat == e8DEV_TYPE_MYSPIN) && (rfrSelDeviceRequest.m_enDeviceType == e8_APPLE_DEVICE)))
      {
         ETG_TRACE_USR2(("[DESC] spi_tclDeviceDiscoverer:: vOnSelectDevice : Sending roleswitch response for MySPIN"));
         bSendStatus = bSendRoleSwitchResponse(rfrSelDeviceRequest.m_u32DeviceHandle,
                  e8DIPO_ROLE_SWITCH_REQUIRED_FOR_NATIVE_TRANSPORT);
      }
      else
      {
         ETG_TRACE_USR2(("[DESC]spi_tclDeviceDiscoverer:: vOnSelectDevice : Sending roleswitch response for Carplay"));
         //! TODO - Before Sending Role Switch response as REQUIRED, check whether the Selection is Progress For the WIFI Transport
         //! If So, Respond to the role switch after the selection is complete. In the Selection device result.

         m_oDeviceSelectionStateLock.s16Lock();
         auto itMap = m_mapDeviceSelectionState.find(rfrSelDeviceRequest.m_u32DeviceHandle);
         if(itMap != m_mapDeviceSelectionState.end())
         {

            enUSBDeviceSelectionState = itMap->second.m_enUSBDeviceSelectionState;
            enWifiDeviceSelectionState = itMap->second.m_enWifiDeviceSelectionState;
         }
         m_oDeviceSelectionStateLock.vUnlock();

         //! if found check the Selection state for the WIFI transport
         //! if Selection State for Wifi transport is Selection in Progress, delay the Role switch response till SelectDeviceResult
         //! if Selection State for USB transport is Selection in Progress, Send the role switch immidiately
         if(enUSBDeviceSelectionState == e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS)
         {
            bSendStatus = bSendRoleSwitchResponse(rfrSelDeviceRequest.m_u32DeviceHandle, e8DIPO_ROLE_SWITCH_REQUIRED);
         }
         else if(enWifiDeviceSelectionState == e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS)
         {
            ETG_TRACE_USR2(("[DESC]spi_tclDeviceDiscoverer:: vOnSelectDevice : Sending roleswitch response is Delayed as Selection is in Progress for WIFI"));
         }

      }
      if ((false == bSendStatus) && (e8USB_CONNECTED == rfrSelDeviceRequest.m_enDevConnType)
               && (e8_REASON_NO_DEVICE_ERROR != rfrSelDeviceRequest.m_enSelectionReason))
      {
         bRequestDeviceSwitch(rfrSelDeviceRequest.m_u32DeviceHandle, enDevCat);
      }
   }
   else
   {
      //! If selection category is unknown or selection error occured, send selection error to HMI
      if (NULL != m_poMediator)
      {
         m_poMediator->vPostSelectDeviceRes(e32COMPID_DISCOVERER, e8SELECTION_FAILED);
      } //if (NULL != poMediator)
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vRestoreSettings
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vRestoreSettings()
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vRestoreSettings entered"));
   m_LockDiscoverers.s16Lock();
   //calling the respective technologies discoveres to clear the private data
   for (auto itmapDiscoverers = m_mapoDiscoverers.begin(); itmapDiscoverers != m_mapoDiscoverers.end();
            itmapDiscoverers++)
   {
      itmapDiscoverers->second->vRestoreSettings();
   }
   m_LockDiscoverers.vUnlock();

   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vRestoreSettings left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vUpdateDeviceName
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vUpdateDeviceName(const t_U32 cou32DeviceHandle, t_String szDeviceName,
         tenDiscovererType enDiscovererType)
{
   SPI_INTENTIONALLY_UNUSED(enDiscovererType);
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vUpdateDeviceName entered with Device Handle %d having Device name as %s", cou32DeviceHandle, szDeviceName.c_str()));
   t_Bool bUpdatedDeviceName = false;

   if (NULL != m_poDeviceListHandler)
   {
      bUpdatedDeviceName = m_poDeviceListHandler->bUpdateDeviceName(cou32DeviceHandle, szDeviceName);
   }

   //Post the Device status info back to HMI as there is update in the SPI device list
   if (true == bUpdatedDeviceName)
   {
      tenDeviceConnectionType enDeviceConnectionType = m_poDeviceListHandler->enGetDeviceConnType(cou32DeviceHandle);
      if (NULL != m_poConnMngrResp)
      {
         m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle, enDeviceConnectionType, e8DEVICE_CHANGED);
      } //if (NULL != m_poConnMngrResp)
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vGetDeviceDiscovererConfig()
 ***************************************************************************/

t_Void spi_tclDeviceDiscoverer::vGetDeviceDiscovererConfig()
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vGetDeviceDiscovererConfig entered"));
   if (NULL != m_poDiscovererSettings)
   {
      t_Bool bUSBPort1 = true;
      t_Bool bUSBPort2 = true;
      t_Bool bUSBPort3 = true;
      m_poDiscovererSettings->vGetUSBPortsEnabledStatus(bUSBPort1, bUSBPort2, bUSBPort3);
   }
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStartDeviceReporting(...
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vStartDeviceReporting()
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vStartDeviceReporting entered"));
   m_oPendingConnectionsLock.s16Lock();
   std::vector<trDiscoveredDevicesInfo> vecPendingDevoceConn = m_vecPendingDeviceConnection;
   m_oPendingConnectionsLock.vUnlock();
   std::vector<trDiscoveredDevicesInfo>::iterator itDevList;
   for (itDevList = vecPendingDevoceConn.begin();itDevList != vecPendingDevoceConn.end();itDevList++)
   {
      vProceedwithConnection(itDevList->m_prDeviceInfo, itDevList->m_enDiscovererType);
   }
   m_oPendingConnectionsLock.s16Lock();
   m_vecPendingDeviceConnection.clear();
   m_oPendingConnectionsLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vSetClientAvailability(...
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bSetClientState(tenClientState enClientState)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vSetSPIMasterClientAvailability entered Client Availability status is %d", ETG_ENUM(CLIENT_STATE,
            enClientState)));
   t_Bool bRetVal = false;
   if (e8_CLIENT_STATUS_INITIALIZED == enClientState)
   {
      //! If SPI is waiting for the master client availability status, start apple device detection
      //! else the detection is either already started or we SPI has not completed loadsettings.
      //! Hence keep track of the status
      if (0 != m_rClientAvailTimerID)
      {
         vCancelClientAvailabilityTimer();
         vStartDeviceReporting();
      }
      m_bIsMasterClientAvailable = true;
      bRetVal = true;
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStartClientAvailabilityTimer(...
***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vStartClientAvailabilityTimer()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDeviceDiscoverer::vStartClientAvailabilityTimer entered\n"));
   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer))
   {
      timer_t rTimerID;
      poTimer->StartTimer(rTimerID, scou32ClientAvailabilityMaxAllowedTime_ms, 0, this,
               &spi_tclDeviceDiscoverer::bClientAvailabilityTimerCb, NULL);
      m_rClientAvailTimerID = rTimerID;
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vCancelClientAvailabilityTimer(...
***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vCancelClientAvailabilityTimer()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDeviceDiscoverer::vCancelClientAvailabilityTimer entered"));
   if (0 != m_rClientAvailTimerID)
   {
      Timer* poTimer = Timer::getInstance();
      if ((NULL != poTimer))
      {
         poTimer->CancelTimer(m_rClientAvailTimerID);
      }
     m_rClientAvailTimerID = 0;
     m_bIsMasterClientAvailable = true;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::bClientAvailabilityTimerCb
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bClientAvailabilityTimerCb(timer_t rTimerID, t_Void *pvObject,
         const t_Void *pvUserData)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDeviceDiscoverer::bClientAvailabilityTimerCb"));
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   SPI_INTENTIONALLY_UNUSED(pvUserData);

   // Client availability is not yet received. As a recovery mechanism start device detection
   spi_tclDeviceDiscoverer* poDiscoverer= static_cast<spi_tclDeviceDiscoverer*>(pvObject);
   if(NULL != poDiscoverer)
   {
      poDiscoverer->vCancelClientAvailabilityTimer();
      poDiscoverer->vStartDeviceReporting();
   }
   return true;
}


/***************************************************************************
** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vSetDeviceProfile
***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vSetDeviceProfile(const t_U32 cou32DeviceHandle, tenDiscovererType enDiscovererType, tenDeviceProfile enDeviceProfile)
{
   SPI_INTENTIONALLY_UNUSED(enDiscovererType);
   ETG_TRACE_USR4(("spi_tclDeviceDiscoverer::vSetDeviceProfile Entered with Device handle %d having Device Profile %d",cou32DeviceHandle, ETG_ENUM(DEVICE_PROFILE, enDeviceProfile)));

   tenDeviceSelectionState enUSBDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
   tenDeviceSelectionState enWifiDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;

   m_oDeviceSelectionStateLock.s16Lock();
   auto itMap = m_mapDeviceSelectionState.find(cou32DeviceHandle);
   if(itMap != m_mapDeviceSelectionState.end())
   {

       enUSBDeviceSelectionState = itMap->second.m_enUSBDeviceSelectionState;
       enWifiDeviceSelectionState = itMap->second.m_enWifiDeviceSelectionState;
   }
   m_oDeviceSelectionStateLock.vUnlock();

   if(NULL != m_poDeviceListHandler)
   {
      if(e8_SELECTION_STATE_DEVICE_SELECTED == enWifiDeviceSelectionState)
      {
         tenSessionTransport enSessionTransport = m_poDeviceListHandler->enGetSessionTransport(cou32DeviceHandle);
         if(e8_SESSION_TRANSPORT_USB != enSessionTransport)
         {
            m_poDeviceListHandler->vSetDeviceProfile(cou32DeviceHandle,enDeviceProfile);
         }
      }
      else if(e8_SELECTION_STATE_DEVICE_SELECTED != enUSBDeviceSelectionState )
      {
         m_poDeviceListHandler->vSetDeviceProfile(cou32DeviceHandle,enDeviceProfile);
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vOnFastReConnectionClearDeviceSwitchInfo(...)
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vOnFastReConnectionClearDeviceSwitchInfo(const t_U32 cou32DeviceHandle)
{
   trDeviceSwitchInfo rDeviceSwitchInfo;

   //! This robustness mainly required when the MP doesn't update the disconnection of the device, during Role switch/Reverse role switch process
   //! Device switch info should not be reset, If the Switch in Progress is set for Wireless.
   if((NULL != m_poDeviceSwitcher) && (NULL != m_poDeviceListHandler))
   {
      m_poDeviceSwitcher->vGetDeviceSwitchInfo(cou32DeviceHandle,rDeviceSwitchInfo);

      tenDeviceConnectionType enDeviceConnectionType = m_poDeviceListHandler->enGetDeviceConnType(cou32DeviceHandle);
      if(e8WIRELESS_CONNECTED != enDeviceConnectionType)
      {
         ETG_TRACE_USR4(("[DESC] spi_tclDeviceDiscoverer::vOnFastReConnectionClearDeviceSwitchInfo rDeviceSwitchInfo.bSwitchInProgress - %d and"
                  "rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress %d", ETG_ENUM(BOOL, rDeviceSwitchInfo.bSwitchInProgress),
                  ETG_ENUM(BOOL, rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress)));

         m_poDeviceListHandler->vSetDeviceConnectionStatus(cou32DeviceHandle,e8DEV_NOT_CONNECTED);
         if((true == rDeviceSwitchInfo.bSwitchInProgress) || (true == rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress))
         {
            m_poDeviceSwitcher->vOnSwitchStatusFailed(cou32DeviceHandle, false);
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vDeleteDevice
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vDeleteDevice(const std::vector<t_U32>& rfvecrDeleteDeviceList, const trUserContext& rfcorUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vDeleteDevice entered"));
   t_Bool bRetRemoveDevice = false;
   std::vector<t_U32> vecrResultDeleteDeviceList;
   std::vector<t_U32> vecrDeletedDeviceList;
   tenResponseCode enResponsecode = e8SUCCESS;
   for (auto itvec = rfvecrDeleteDeviceList.begin(); itvec != rfvecrDeleteDeviceList.end(); ++itvec)
   {
      /*call the function for deleting the device handle from the list;on the basis of the
          result from the same method, if true then call the function for deletion from the histroy of devices otherwise no*/
      if (NULL != m_poDeviceListHandler)
      {
         ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vDeleteDevice - The device handle to be deleted from list : %d", *itvec));
         bRetRemoveDevice = m_poDeviceListHandler->bRemoveDeviceFromList(*itvec);

         if (bRetRemoveDevice == true)
         {
            ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vDeleteDevice - The device handle to be deleted from history is : %d", *itvec));
            m_poDeviceListHandler->vRemoveDeviceFromHistory(*itvec);
            vecrDeletedDeviceList.push_back(*itvec);
         }
         else
         {
            ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vDeleteDevice - The device handle failed to delete : %d", *itvec));
            vecrResultDeleteDeviceList.push_back(*itvec);
         }
      }
   }
   if (0 != vecrResultDeleteDeviceList.size())
   {
      enResponsecode = e8FAILURE;
   }
   else
   {
      tenDiscovererType enAppleDiscovererType = e8_DISCOVERER_TYPE_APPLE;
      m_LockDiscoverers.s16Lock();
      auto itMap = m_mapoDiscoverers.find(enAppleDiscovererType);
      if(itMap != m_mapoDiscoverers.end())
      {
         (itMap->second)->vClearDeviceDetails(vecrDeletedDeviceList);
      }
      m_LockDiscoverers.vUnlock();
   }


   if (NULL != m_poConnMngrResp)
   {
      ETG_TRACE_USR1(("spi_tclConnMngr::vDeleteDevice - Response code : %d", enResponsecode));

      m_poConnMngrResp->vPostDeleteDevice(vecrResultDeleteDeviceList, enResponsecode, rfcorUsrCntxt);
   }
}


/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vOnDeviceReenumeration
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vOnDeviceReenumeration(const t_U32 cou32DeviceHandle)
{
   trDeviceSelectionInfo rSelInfo;
   if((NULL != m_poDeviceListHandler) && (NULL != m_poConnMngrResp))
   {
      rSelInfo.enDeviceSelectionErrorType = m_poDeviceListHandler->enGetDeviceSelectionErrorType(cou32DeviceHandle);
      m_poDeviceListHandler->vSetDeviceSelectionInfo(cou32DeviceHandle, rSelInfo);
      //! Send Device list change only if there is a change in device status
      m_poConnMngrResp->vPostDeviceStatusInfo(cou32DeviceHandle, e8USB_CONNECTED, e8DEVICE_CHANGED);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vOnDeviceReenumeration
 ***************************************************************************/
t_Bool spi_tclDeviceDiscoverer::bIsWirelessSelectionInProgress(const t_U32 cou32DeviceHandle)
{
   t_Bool bIsSelectionInprogress= false;
   m_oDeviceSelectionStateLock.s16Lock();
   auto itMap = m_mapDeviceSelectionState.find(cou32DeviceHandle);
   if(itMap != m_mapDeviceSelectionState.end())
   {
      if(e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS == itMap->second.m_enWifiDeviceSelectionState)
      {
         bIsSelectionInprogress = true;
      }
   }
   m_oDeviceSelectionStateLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::bIsWirelessSelectionInProgress - Selection is in Progress over Wireless %d for the Device Handle : %d",ETG_ENUM(BOOL,bIsSelectionInprogress), cou32DeviceHandle));
   return bIsSelectionInprogress;
}


/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer::vUpdateBTLimitedFunctionality
 ***************************************************************************/
t_Void spi_tclDeviceDiscoverer::vUpdateBTLimitedFunctionality(const t_U32 cou32DeviceHandle, tenDiscovererType enDiscovererType,
         t_Bool bIsDeviceBTFunctionalityLimited)
{
   SPI_INTENTIONALLY_UNUSED(enDiscovererType);
   ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vUpdateBTLimitedFunctionality entered with Device Handle %d with the BT limited Functionality as %d", cou32DeviceHandle, ETG_ENUM(BOO, bIsDeviceBTFunctionalityLimited)));

   if(NULL != m_poDeviceListHandler)
   {
      m_poDeviceListHandler->vUpdateBTLimitedFunctionality(cou32DeviceHandle, bIsDeviceBTFunctionalityLimited);
   }
}


//lint ?restore
