/*!
 *******************************************************************************
 * \file             spi_tclDiscovererBase.h
 * \brief            Discoverer base class
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Unified Device Discoverer
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLDISCOVERERBASE_H_
#define SPI_TCLDISCOVERERBASE_H_

#include <functional> //! C++11
#include "spi_ConnMngrTypeDefines.h"
#include "spi_tclDiscovererSettingsIntf.h"

/**
 * \class spi_tclDiscovererBase
 * \brief Base class for all discoverers
 */

class spi_tclDiscovererBase
{

   public:
      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::spi_tclDiscovererBase
       ***************************************************************************/
      /*!
       * \brief   Default Constructor
       **************************************************************************/
      spi_tclDiscovererBase():m_poDiscovererSettings(NULL){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::~spi_tclDiscovererBase()
       ***************************************************************************/
      /*!
       * \brief   Destructor
       **************************************************************************/
      virtual ~spi_tclDiscovererBase(){ m_poDiscovererSettings = NULL; }

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::bInitialize
       ***************************************************************************/
      /*!
       * \brief   Initializes the Discoverer
       * \retval  true if initialization succeeds
       **************************************************************************/
      virtual t_Bool bInitialize(){return true;}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vUninitialize
       ***************************************************************************/
      /*!
       * \brief   Uninitializes the Discoverer
       **************************************************************************/
      virtual t_Void vUninitialize(){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vLoadSettings()
       ***************************************************************************/
      /*!
       * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
       *          Used to Initialize ML Discoverer
       * \sa      vSaveSettings()
       **************************************************************************/
       virtual t_Bool bLoadSettings(){return true;}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::bStartDeviceDetection
       ***************************************************************************/
      /*!
       * \brief   Start device detection
       * \retval  true on success
       **************************************************************************/
      virtual t_Bool bStartDeviceDetection(){ return true; }

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vStartDeviceReporting
       ***************************************************************************/
      /*!
       * \brief   Start reporting detected devices
       **************************************************************************/
      virtual t_Void vStartDeviceReporting(){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vStopDeviceDetection
       ***************************************************************************/
      /*!
       * \brief   Stop device detection
       **************************************************************************/
      virtual t_Void vStopDeviceDetection(){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vStartDeviceReporting
       ***************************************************************************/
      /*!
       * \fn      vStartDeviceReporting()
       * \brief   Stop reporting detected devices
       **************************************************************************/
      virtual t_Void vStopDeviceReporting(){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \fn      vRegisterCallbacks()
       * \brief   Register discoverer callbacks
       **************************************************************************/
      virtual t_Void vRegisterCallbacks(const trDiscovererCbs /*corfrDisccbs*/){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vSetDeviceSwitchInfo
       ***************************************************************************/
      /*!
       * \brief   Sets the device switch information for the device specified by the device handle
       * \param   cou32DeviceHandle: Device Handle
       * \param   [IN]reference to the device switch information structure
       **************************************************************************/
      virtual t_Void vSetDeviceSwitchInfo(const t_U32 /*cou32Devicehandle*/, const trDeviceSwitchInfo &/*corfrDeviceSwitchInfo*/){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vRestoreSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vRestoreSettings()
       * \brief  vRestoreSettings Method. Invoked on factory reset to clear the private data when SPI Session is active
       * \param  None
       **************************************************************************/
       virtual t_Void vRestoreSettings(){}

       /***************************************************************************
        ** FUNCTION:  t_Void  spi_tclDiscovererBase::vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
        ***************************************************************************/
        /*!
        * \fn      t_Void vSetDiscoverersettingsInstance()
        * \brief   Method to set discoverer settings instance.
        * \param   poDiscovererSettingsIntf  : pointer to DiscovererSettingsIntf.
        * \retval  t_Void
        **************************************************************************/
       t_Void vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
       {
          if( NULL != poDiscovererSettingsIntf)
          {
             m_poDiscovererSettings = poDiscovererSettingsIntf;
          }
       }

       /***************************************************************************
        ** FUNCTION:  spi_tclDiscovererBase::vClearDeviceDetails
        ***************************************************************************/
       /*!
        * \fn     vClearDeviceDetails
        * \brief  removes device from Apple Discoverer Map
        * \param  cou32DeviceHandle: Device Handle of the device to be removed
        **************************************************************************/
       virtual t_Void vClearDeviceDetails(const std::vector<t_U32>& /*rfvecrDeleteDeviceList*/){}
	   
	   
       /***************************************************************************
        ** FUNCTION:  spi_tclDiscovererBase::vUpdateSelectionState
        ***************************************************************************/
       /*!
        * \fn     vUpdateSelectionState
        * \brief  Informs if the device selection is not initiated for a reported device
        * \param  cou32DeviceHandle: Device handle
        **************************************************************************/
       virtual t_Void vUpdateSelectionState(const t_U32 /*cou32DeviceHandle*/, tenDeviceSelectionState /*enDevSelectionState*/, tenDeviceConnectionType /*enDeviceConnType*/){}

   protected:
       spi_tclDiscovererSettingsIntf* m_poDiscovererSettings;

   private:
      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vDeviceAppearedCallback
       ***************************************************************************/
      /*!
       * \fn      vDeviceAppearedCallback()
       * \brief   Called when a new device is detected.
       * \param   u32DeviceHandle: Device handle
       **************************************************************************/
      static t_Void vDeviceAppearedCallback(t_U32 /*u32DeviceHandle*/){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::vDeviceDisappearedCallback
       ***************************************************************************/
      /*!
       * \brief   Called when a device is removed.
       * \param   u32DeviceHandle: Device handle
       **************************************************************************/
      static t_Void vDeviceDisappearedCallback(t_U32 /*u32DeviceHandle*/){}

};

typedef spi_tclDiscovererBase* GetDiscoverer_t();
extern std::map<tenDiscovererType, GetDiscoverer_t*> Discoverer;

/*! } */
#endif // SPI_TCLDISCOVERERBASE_H_
