/*!
 *******************************************************************************
 * \file             spi_tclDeviceDiscovererIntfIntf.h
 * \brief            Unified Device Discoverer
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Unified Device Discoverer
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLDEVICEDISCOVERERINTF_H_
#define SPI_TCLDEVICEDISCOVERERINTF_H_

#include "RespBase.h"
#include "spi_ConnMngrTypeDefines.h"

class spi_tclDiscovererBase;

/**
 * \class spi_tclDeviceDiscovererIntf
 * \brief Unified discoverer responsible for reporting device connections and disconnections
 *        reported from different connection mediums such as USB, Wi-Fi. All devices detected on
 *        these connection mediums will be reported to the connection manager,
 *  \sa USBDiscoverer, WiFiDiscoverer
 */
class spi_tclDeviceDiscovererIntf
{

   public:
      spi_tclDeviceDiscovererIntf(){}
      virtual ~spi_tclDeviceDiscovererIntf(){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscovererIntf::bInitialize()
       ***************************************************************************/
      /*!
       * \brief   Creates required discoverers and initializes them
       * \sa      bUnInitialize()
       **************************************************************************/
      virtual t_Bool bInitialize() = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscovererIntf::bUnInitialize()
       ***************************************************************************/
      /*!
       * \brief   Uninitialize and Deletes the created discoverers
       * \sa      bInitialize()
       **************************************************************************/
      virtual t_Bool bUnInitialize() = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscovererIntf::vLoadSettings()
       ***************************************************************************/
      /*!
       * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
       * \param   rfcrSpiFeatureSupp: Indicates feature support for SPI technologies
       * \sa      vSaveSettings()
       **************************************************************************/
      virtual t_Void vLoadSettings() = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscovererIntf::vSaveSettings()
       ***************************************************************************/
      /*!
       * \brief   vSaveSettings Method. Invoked during  NORMAL->OFF state transition.
       * \sa      vLoadSettings()
       **************************************************************************/
      virtual t_Void vSaveSettings() = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscovererIntf::bStartDeviceDetection
       ***************************************************************************/
      /*!
       * \brief   Starts Device detection. This will further invoke the discoverers (USB/Wi-Fi)
       *          to start device detection by listening to libusb, connman etc.
       *          Has to be called when device detection for SPI has to be started.
       *          Can be stopped and restarted when system states change toggles between OFF and ON
       * \param   enDiscovererType: Type of the discoverer for which device detection has to be started
       * \retval  true if the discoverer is started successfully false otherwise
       * \sa      vStopDeviceDetection
       **************************************************************************/
      virtual t_Bool bStartDeviceDetection(tenDiscovererType enDiscovererType) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscovererIntf::vStartDeviceReporting
       ***************************************************************************/
      /*!
       * \brief   Starts reporting detected devices. This can be used to start updating the
       *          detected devices to upper layers. Can be used in use cases such as
       *          or ON OFF where devices detected during the OFF period have to be remembered.
       *          Device reporting can be stopped using vStopDeviceReporting although
       *          Device discovery is in progress
       * \param   enDiscovererType: Type of the discoverer for which device reporting has to be started
       * \note    This interface is not implemented and is for future use
       * \warning Calling this interface without starting device discovery will internally start device discovery
       * \sa      vStopDeviceReporting
       **************************************************************************/
      virtual t_Void vStartDeviceReporting(tenDiscovererType enDiscovererType) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscovererIntf::vStopDeviceDetection
       ***************************************************************************/
      /*!
       * \brief   Stops Device detection. This will further invoke the discoverers (USB/Wi-Fi)
       *          to stop device detection by listening to libusb, connman etc.
       *          Has to be called when device detection for SPI has to be stopped.
       *          Can be stopped and restarted when system states change toggles between OFF and ON
       * \param   enDiscovererType: Type of the discoverer for which device detection has to be stopped
       * \sa      bStartDeviceDetection
       **************************************************************************/
      virtual t_Void vStopDeviceDetection(tenDiscovererType enDiscovererType) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscovererIntf::vStopDeviceReporting
       ***************************************************************************/
      /*!
       * \brief   Stops reporting detected devices. This can be used to stop updating the
       *          detected devices to upper layers. Can be used in use cases such as
       *          or ON OFF where devices detected during the OFF period have to be remembered.
       *          Device reporting can be started using vStartDeviceReporting
       * \param   enDiscovererType: Type of the discoverer for which device reporting has to be stopped
       * \note    This interface is not implemented and is for future use
       * \sa      vStopDeviceReporting
       **************************************************************************/
      virtual t_Void vStopDeviceReporting(tenDiscovererType enDiscovererType) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscovererIntf::vSetDeviceSwitchInfo
       ***************************************************************************/
      /*!
       * \brief   Sets the device switch information for the device specified by the device handle
       * \param   cou32DeviceHandle: Device Handle
       * \param   [IN]reference to the device switch information structure
       **************************************************************************/
      virtual t_Void vSetDeviceSwitchInfo(const t_U32 /*cou32Devicehandle*/, const trDeviceSwitchInfo & /*corfrDeviceSwitchInfo*/,
               tenDiscovererType /*enDiscovererType*/) {};

/*! } */
};

#endif /* SPI_TCLDEVICEDISCOVERERINTF_H_ */
