/*!
 *******************************************************************************
 * \file              spi_tclUSBDiscovererImpl.cpp
 * \brief             Device discovery 
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    USB Device discovery implementation
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 26.02.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <unistd.h>
#include <aauto/AautoLogging.h>

#include "crc.h"
#include "spi_tclUSBDiscovererImpl.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclUSBDiscovererImpl.cpp.trc.h"
#endif
#endif

//using namespace adit::aauto;

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/
#define OTG_PORT_STRING_LIMITER  "gadget"

/* USB PORT Information for IMX Board :
 * Below is the information on connecting device to 3 different ports.
 * SystemPath-/sys/devices/soc0/soc.0/2100000.aips-bus/2184200.usb/ci_hdrc.1/usb2/2-1/2-1.2--->Not Available to end user (Internal Hub->Port2 )
   SystemPath-/sys/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1      ---->OTG
   SystemPath-/sys/devices/soc0/soc.0/2100000.aips-bus/2184200.usb/ci_hdrc.1/usb2/2-1/2-1.1---Non-OTG (Internal Hub->Port1)
 *
 */
static const t_String sczUSBPort1Info="usb1";
static const t_String sczUSBPort2Info="usb2/2-1/2-1.1";
static const t_String sczUSBPort3Info="usb2/2-1/2-1.2";

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
static t_U32 scou32AppleVendorID = 0x5ac;

/*!
 * \class spi_tclUSBDiscovererImpl
 * \brief Device discovery implementation using ADIT USB discoverer
 */

/***************************************************************************
static t_U32 scou32AppleVendorID = 0x5ac;
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscovererImpl::spi_tclUSBDiscovererImpl();
 ***************************************************************************/
spi_tclUSBDiscovererImpl::spi_tclUSBDiscovererImpl() :
         m_poDiscoverer(NULL),
         m_poDiscovererSettings(NULL)
{
   ETG_TRACE_USR1(("spi_tclUSBDiscovererImpl::spi_tclUSBDiscovererImpl() \n"));
}

/***************************************************************************
 ** FUNCTION:  virtual spi_tclUSBDiscovererImpl::~spi_tclUSBDiscovererImpl()
 ***************************************************************************/
spi_tclUSBDiscovererImpl::~spi_tclUSBDiscovererImpl()
{
   ETG_TRACE_USR1(("spi_tclUSBDiscovererImpl::~spi_tclUSBDiscovererImpl() \n"));
   m_poDiscoverer = NULL;
   m_poDiscovererSettings = NULL;
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscovererImpl::bInitialize
 ***************************************************************************/
t_Bool spi_tclUSBDiscovererImpl::bInitialize(const trUSBDiscovererCbs &corfrDiscovererCbs)
{
   ETG_TRACE_USR1((" spi_tclUSBDiscovererImpl::bInitialize(): Initializing USB discoverer and registering callbacks \n"));
   //! register  logging
   // aautoRegisterAppWithDLT("AAP", "Android Auto demo");
   //! register all AAUTO DLT context of aauto/platform */
   aautoRegisterCtxtWithDLT();

   m_rDiscCbs = corfrDiscovererCbs;

   SpiUsbDiscovererCallbacks rDiscovererCbs;
   rDiscovererCbs.notifyDeviceFound = &spi_tclUSBDiscovererImpl::vUSBEntityAppearedCb;
   rDiscovererCbs.notifyDeviceRemoved = &spi_tclUSBDiscovererImpl::vUSBEntityDisappearedCb;
   rDiscovererCbs.notifyDeviceFoundAccessoryMode = &spi_tclUSBDiscovererImpl::vAOAPEntityAppearedCb;

   //! Create USB discoverer and register callbacks
   m_poDiscoverer = new AOAP::UsbDiscoverer(this, &rDiscovererCbs);
   SPI_NORMAL_ASSERT(NULL == m_poDiscoverer);
   t_Bool bRetval = (NULL != m_poDiscoverer);
   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscovererImpl::vUnInitialize
 ***************************************************************************/
t_Void spi_tclUSBDiscovererImpl::vUnInitialize()
{
   ETG_TRACE_USR1(("  spi_tclUSBDiscovererImpl::vUnInitializeDiscoverer(): uninitializing USB device discoverer \n"));
   RELEASE_MEM(m_poDiscoverer)
}
/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscovererImpl::s32StartMonitoring
 ***************************************************************************/
t_S32 spi_tclUSBDiscovererImpl::s32StartMonitoring()
{
   t_S32 s32ErrorCode = 0;
   if (NULL != m_poDiscoverer)
   {
      //! Start monitoring USB devices
      s32ErrorCode = m_poDiscoverer->startMonitoring();
   }
   ETG_TRACE_USR1((" spi_tclUSBDiscovererImpl::bStartDeviceDiscovery(): Started USB device discovery: Result = %d \n", s32ErrorCode));
   return s32ErrorCode;
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscovererImpl::s32StopMonitoring
 ***************************************************************************/
t_S32 spi_tclUSBDiscovererImpl::s32StopMonitoring()
{
   t_S32 s32ErrorCode = 0;
   if (NULL != m_poDiscoverer)
   {
      //! Stop monitoring USB devices
      s32ErrorCode = m_poDiscoverer->stopMonitoring();
   }
   ETG_TRACE_USR1((" spi_tclUSBDiscovererImpl::vStopDeviceDiscovery(): Stopping USB device discovery: Result = %d \n", s32ErrorCode));
   return s32ErrorCode;
}


/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  t_U32 spi_tclUSBDiscovererImpl::u32GenerateUniqueDeviceID
 ***************************************************************************/
t_U32 spi_tclUSBDiscovererImpl::u32GenerateUniqueDeviceID(trUSBDeviceInfo & rfrUSBDeviceInfo)
{
   ETG_TRACE_USR1(("spi_tclUSBDiscovererImpl::u32GenerateUniqueDeviceID  ProductID=0x%x VendorID= 0x%x SerialNumber=%s\n",
            rfrUSBDeviceInfo.u32ProductID, rfrUSBDeviceInfo.u32VendorID, rfrUSBDeviceInfo.szSerialNumber.c_str() ));
   t_U32 u32UniqueID = 0;

   //! TODO : Need confirmation from google for finding the unique device ID
   t_String szUniqueID = rfrUSBDeviceInfo.szSerialNumber;
   u32UniqueID = u16CalcCRC16((const t_UChar*)szUniqueID.c_str(),szUniqueID.size());
   ETG_TRACE_USR1(("Generated unique Device ID=0x%x for the unique string =%s\n",
            u32UniqueID, szUniqueID.c_str()));
   return u32UniqueID;
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBDiscovererImpl::vCopyUSBDeviceInfo
 ***************************************************************************/
t_Void spi_tclUSBDiscovererImpl::vCopyUSBDeviceInfo(trUSBDeviceInfo &rfrUSBDeviceInfo,const  t_usbDeviceInformation *prUSBDevInfo)
{
   ETG_TRACE_USR1((" spi_tclUSBDiscovererImpl::vCopyUSBDeviceInfo entered \n"));
   if(NULL != prUSBDevInfo)
   {
      rfrUSBDeviceInfo.szSerialNumber = prUSBDevInfo->serial;
      rfrUSBDeviceInfo.u32ProductID = prUSBDevInfo->productId;
      rfrUSBDeviceInfo.u32VendorID = prUSBDevInfo->vendorId;
      rfrUSBDeviceInfo.szProduct = prUSBDevInfo->product;
      rfrUSBDeviceInfo.szSystemPath = prUSBDevInfo->sysPath;
      rfrUSBDeviceInfo.szManufacturer = prUSBDevInfo->manufacturer;
      rfrUSBDeviceInfo.u32USBDeviceNumber = prUSBDevInfo->devNum;
      rfrUSBDeviceInfo.u32Interface = prUSBDevInfo->interface;
      rfrUSBDeviceInfo.bIsAOAPSupported = prUSBDevInfo->aoapSupported;
      rfrUSBDeviceInfo.u32DeviceHandle = u32GenerateUniqueDeviceID(rfrUSBDeviceInfo);
      rfrUSBDeviceInfo.enDeviceType = (true == prUSBDevInfo->aoapSupported)?e8_ANDROID_DEVICE: e8_UNKNOWN_DEVICE;
   }

   if (false == rfrUSBDeviceInfo.szSystemPath.empty())
   {
      t_Char szGadgetPath[1024] = { 0 };
      snprintf(szGadgetPath,
               sizeof(szGadgetPath),
               "%s/../../%s",
               rfrUSBDeviceInfo.szSystemPath.c_str(),
               OTG_PORT_STRING_LIMITER);
      //! Extract information for device is device connected to OTG port
      rfrUSBDeviceInfo.enUSBPortType =
               (0 == access(szGadgetPath, F_OK)) ? e8_PORT_TYPE_OTG : e8_PORT_TYPE_NON_OTG;

      ETG_TRACE_USR4(("OTG port supported = %d", ETG_ENUM(USB_PORT_TYPE, ( rfrUSBDeviceInfo.enUSBPortType ))));
   }
}



/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBDiscovererImpl::vUSBEntityAppearedCb
 ***************************************************************************/
t_Void spi_tclUSBDiscovererImpl::vUSBEntityAppearedCb(t_Void *pContext, t_usbDeviceInformation *prUSBDevInfo, t_S32 s32Result)
{
   ETG_TRACE_USR1(("[DESC] spi_tclUSBDiscovererImpl::vUSBEntityAppearedCb: New USB device detected pContext = %p, prUSBDevInfo - %p, s32Result = %d \n",
                     pContext, prUSBDevInfo, s32Result));
   spi_tclUSBDiscovererImpl* poCmdDisc = static_cast<spi_tclUSBDiscovererImpl*> (pContext);
   if ((NULL != poCmdDisc) && (NULL != prUSBDevInfo) && (NULL != poCmdDisc->m_poDiscovererSettings)
            && (true == poCmdDisc->m_poDiscovererSettings->bIsDeviceBlacklisted(prUSBDevInfo->productId, prUSBDevInfo->vendorId)))
   {
      ETG_TRACE_USR4(("spi_tclUSBDiscovererImpl::vUSBEntityAppearedCb The following device : %d is blacklisted", prUSBDevInfo->devNum));
   }

   else if ((NULL != poCmdDisc) && (false == poCmdDisc->bIsDeviceSelectableOnUSBPort(prUSBDevInfo)))
   {
      ETG_TRACE_USR4(("spi_tclUSBDiscovererImpl::vUSBEntityAppearedCb : USB PORT is DISABLED"));
   }

   else
   {
      if ((NULL != pContext) && (NULL != prUSBDevInfo))
      {
         ETG_TRACE_USR4(("spi_tclUSBDiscovererImpl::vUSBEntityAppearedCb The following device : %d is  not blacklisted", prUSBDevInfo->devNum));
         ETG_TRACE_USR4(("[PARAM] Detected USB device has: AOAP support =%d, DeviceNo- %d, Interface-%d, productId- 0x%x, vendorId=0x%x, Product-%s ", ETG_ENUM(BOOL,
                  prUSBDevInfo->aoapSupported), prUSBDevInfo->devNum, prUSBDevInfo->interface, prUSBDevInfo->productId, prUSBDevInfo->vendorId, (prUSBDevInfo->product).c_str()));
         ETG_TRACE_USR4(("[PARAM]  Manufacturer-%s, ", (prUSBDevInfo->manufacturer).c_str()));
         ETG_TRACE_USR4(("[PARAM]  SystemPath-%s", (prUSBDevInfo->sysPath).c_str()));

         //! Skip apple devices
         if ((scou32AppleVendorID != prUSBDevInfo->vendorId) && (NULL != poCmdDisc))
         {
            trUSBDeviceInfo rUSBDeviceInfo;
            poCmdDisc->vCopyUSBDeviceInfo(rUSBDeviceInfo, prUSBDevInfo);
            rUSBDeviceInfo.enUSBProfile = e8_PROFILE_DEFAULT;
            rUSBDeviceInfo.enConnectionStatus = e8DEV_CONNECTED;
            rUSBDeviceInfo.u32VendorID = prUSBDevInfo->vendorId;
            rUSBDeviceInfo.u32ProductID = prUSBDevInfo->productId;

            if (NULL != poCmdDisc->m_rDiscCbs.fvUSBDeviceConnectionCb)
            {
               (poCmdDisc->m_rDiscCbs.fvUSBDeviceConnectionCb)(rUSBDeviceInfo);
            }
         }
         else
         {
            ETG_TRACE_USR4(("[DESC] Skipping reporting of Apple devices as this will be reported by mediaplayer\n"));
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBDiscovererImpl::vUSBEntityDisappearedCb
 ***************************************************************************/
t_Void spi_tclUSBDiscovererImpl::vUSBEntityDisappearedCb(t_Void *pContext,
         t_usbDeviceInformation *prUSBDevInfo, t_S32 s32Result)
{
   ETG_TRACE_USR1((" spi_tclUSBDiscovererImpl::vUSBEntityDisappearedCb USB device was removed pContext = %p, prUSBDevInfo - %p, s32Result = %d \n",
            pContext, prUSBDevInfo, s32Result));

   if ((NULL != prUSBDevInfo) && (NULL != pContext))
   {
      spi_tclUSBDiscovererImpl* poCmdDisc = static_cast<spi_tclUSBDiscovererImpl*>(pContext);

      ETG_TRACE_USR4(("Removed device has DeviceNo- %d, Interface-%d, productId- 0x%x, vendorId=0x%x ",
                        prUSBDevInfo->devNum, prUSBDevInfo->interface, prUSBDevInfo->productId, prUSBDevInfo->vendorId));

      //! Skip apple devices
      if (scou32AppleVendorID != prUSBDevInfo->vendorId)
      {
         //! Copy USB device info and calculate the device handle
         trUSBDeviceInfo rUSBDeviceInfo;
         poCmdDisc->vCopyUSBDeviceInfo(rUSBDeviceInfo, prUSBDevInfo);
         rUSBDeviceInfo.enUSBProfile = e8_PROFILE_UNKNOWN;

         if(NULL != poCmdDisc->m_rDiscCbs.fvDeviceDisconnectionCb)
         {
            (poCmdDisc->m_rDiscCbs.fvDeviceDisconnectionCb)(rUSBDeviceInfo);
         }
      }
      else
      {
         ETG_TRACE_USR4(("[DESC] Skipping reporting of Apple devices as this will be reported by mediaplayer\n"));
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBDiscovererImpl::vAOAPEntityAppearedCb
 ***************************************************************************/
t_Void spi_tclUSBDiscovererImpl::vAOAPEntityAppearedCb(t_Void *pContext, t_usbDeviceInformation *prUSBDevInfo,
         t_S32 s32Result)
{
   ETG_TRACE_USR1((" spi_tclUSBDiscovererImpl::vAOAPEntityAppearedCb: AOAP Device is detected pContext = %p, prUSBDevInfo - %p, s32Result = %d \n", pContext, prUSBDevInfo, s32Result));
   if ((NULL != prUSBDevInfo) && (NULL != pContext))
   {
      ETG_TRACE_USR4(("Detected AOAP device has USBDeviceNo- %d, Interface-%d, productId- %d, vendorId=%d and systempath = %s\n", prUSBDevInfo->devNum, prUSBDevInfo->interface, prUSBDevInfo->productId, prUSBDevInfo->vendorId, prUSBDevInfo->sysPath.c_str()));
      spi_tclUSBDiscovererImpl* poCmdDisc = static_cast<spi_tclUSBDiscovererImpl*>(pContext);

      if (NULL != poCmdDisc)
      {
         trUSBDeviceInfo rAOAPDeviceInfo;
         //! Copy AOAP information
         poCmdDisc->vCopyUSBDeviceInfo(rAOAPDeviceInfo, prUSBDevInfo);
         rAOAPDeviceInfo.enUSBProfile = e8_PROFILE_AOAP;
         rAOAPDeviceInfo.enConnectionStatus = e8DEV_CONNECTED;

         ETG_TRACE_USR4(("[PARAM] spi_tclUSBDiscovererImpl::vAOAPEntityAppearedCb: Device Handle = 0x%x , VendorID =%d, ProductID = %d\n",
                  rAOAPDeviceInfo.u32DeviceHandle, rAOAPDeviceInfo.u32VendorID, rAOAPDeviceInfo.u32ProductID));
         if(NULL != poCmdDisc->m_rDiscCbs.fvProjectionDeviceConnectionCb)
         {
            (poCmdDisc->m_rDiscCbs.fvProjectionDeviceConnectionCb)(rAOAPDeviceInfo);
         }
      }
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscovererImpl::bIsDeviceSelectableOnUSBPort
 ***************************************************************************/
t_Bool spi_tclUSBDiscovererImpl::bIsDeviceSelectableOnUSBPort(t_usbDeviceInformation *prUSBDevInfo)
{
   ETG_TRACE_USR1((" spi_tclUSBDiscovererImpl::bIsDeviceSelectableOnUSBPort entered"));
   t_Bool bIsPortEnabled = true;
   t_Bool bUSBPort1Enabled = true;
   t_Bool bUSBPort2Enabled = true;
   t_Bool bUSBPort3Enabled = true;

   if ((NULL != m_poDiscovererSettings) && (NULL != prUSBDevInfo))
   {
      t_String szSystempath =prUSBDevInfo->sysPath;
      m_poDiscovererSettings->vGetUSBPortsEnabledStatus(bUSBPort1Enabled,bUSBPort2Enabled,bUSBPort3Enabled);

      if (szSystempath.find(sczUSBPort1Info, 0) != std::string::npos)
      {
         if (true == bUSBPort1Enabled)
         {
            bIsPortEnabled=true;
         }
         else
         {
            bIsPortEnabled=false;
         }
      }
      else if ((szSystempath.find(sczUSBPort2Info, 0)) != std::string::npos)
      {
         if (true == bUSBPort2Enabled)
         {
            bIsPortEnabled=true;
         }
         else
         {
            bIsPortEnabled=false;
         }

      }
      else if ((szSystempath.find(sczUSBPort3Info, 0)) != std::string::npos)
      {
         if (true == bUSBPort3Enabled)
         {
            bIsPortEnabled=true;
         }
         else
         {
            bIsPortEnabled=false;
         }
      }
   }
   return bIsPortEnabled;
}

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclUSBDiscovererImpl::vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
 ***************************************************************************/
t_Void spi_tclUSBDiscovererImpl::vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclUSBDiscovererImpl::vSetDiscoverersettingsInstance entered"));
   if( NULL != poDiscovererSettingsIntf)
   {
	   ETG_TRACE_USR1(("spi_tclUSBDiscovererImpl::vSetDiscoverersettingsInstance : Setting m_poDiscovererSettings Value"));
       m_poDiscovererSettings = poDiscovererSettingsIntf;
   }
}

