/*!
 *******************************************************************************
 * \file              spi_tclUSBDiscoverer.cpp
 * \brief             Device discovery wrapper
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Device discovery wrapper
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 08.03.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#include "crc.h"
#include "StringHandler.h"
#include "SPITypes.h"
#include "spi_tclUSBDiscoverer.h"
#include "spi_tclDiscoveryMsgQInterface.h"
#include "spi_tclDiscoveryDispatcher.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclUSBDiscoverer.cpp.trc.h"
#endif
#endif

/*lint -save -e1013 */
/*lint -save -e1055 */
/*lint -save -e10 */
/*lint -save -e40 */
/*lint -save -e746 */
/*lint -save -e774 */
/*lint -save -e516 */
/*lint -save -e63 */

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscoverer::spi_tclUSBDiscoverer
 ***************************************************************************/
spi_tclUSBDiscoverer::spi_tclUSBDiscoverer(spi_tclUSBDiscovererIntf* poExtDiscoverer, spi_tclDiscoveryDataIntf* poDiscoveryData) :
         m_poExtDiscoverer(poExtDiscoverer),
         m_poDiscoveryData(poDiscoveryData)
{
   ETG_TRACE_USR1((" spi_tclUSBDiscoverer::spi_tclUSBDiscoverer()\n"));
   SPI_NORMAL_ASSERT(NULL == m_poExtDiscoverer);
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscoverer::~spi_tclUSBDiscoverer
 ***************************************************************************/
spi_tclUSBDiscoverer::~spi_tclUSBDiscoverer()
{
   ETG_TRACE_USR1((" spi_tclUSBDiscoverer::~spi_tclUSBDiscoverer()\n"));

}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscoverer::bInitialize
 ***************************************************************************/
t_Bool spi_tclUSBDiscoverer::bInitialize()
{
   ETG_TRACE_USR1((" spi_tclUSBDiscoverer::bInitializeDiscoverer(): Initializing USB discoverer and registering callbacks \n"));


   trUSBDiscovererCbs rUSBDiscCbs;
   rUSBDiscCbs.fvUSBDeviceConnectionCb = std::bind(&spi_tclUSBDiscoverer::vOnUSBDeviceConnectionCb,
            this,
            SPI_FUNC_PLACEHOLDERS_1);
   rUSBDiscCbs.fvDeviceDisconnectionCb = std::bind(&spi_tclUSBDiscoverer::vOnDeviceDisconnectionCb,
            this,
            SPI_FUNC_PLACEHOLDERS_1);
   rUSBDiscCbs.fvProjectionDeviceConnectionCb = std::bind(&spi_tclUSBDiscoverer::vOnAOAPDeviceConnectionCb,
            this,
            SPI_FUNC_PLACEHOLDERS_1);

   t_Bool bRetval = (NULL != m_poExtDiscoverer);
   if(NULL != m_poExtDiscoverer)
   {
      t_Bool bInit = m_poExtDiscoverer->bInitialize(rUSBDiscCbs);
      bRetval = bRetval && bInit;
   }

   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscoverer::vUninitialize
 ***************************************************************************/
t_Void spi_tclUSBDiscoverer::vUninitialize()
{
   ETG_TRACE_USR1(("  spi_tclUSBDiscoverer::vUnInitializeDiscoverer(): uninitializing USB device discoverer \n"));
   m_poExtDiscoverer->vUnInitialize();
   RELEASE_MEM(m_poExtDiscoverer)
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscoverer::bStartDeviceDetection
 ***************************************************************************/
t_Bool spi_tclUSBDiscoverer::bStartDeviceDetection()
{
   t_Bool bResult = false;
   if(NULL != m_poExtDiscoverer)
   {
      //! Start monitoring USB devices
      t_S32 s32ErrorCode = m_poExtDiscoverer->s32StartMonitoring();
      bResult = (0 == s32ErrorCode);
   }

   ETG_TRACE_USR1((" spi_tclUSBDiscoverer::bStartDeviceDetection(): Started USB device discovery: Result = %d \n",
            ETG_ENUM(BOOL, bResult)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscoverer::vStopDeviceDetection
 ***************************************************************************/
t_Void spi_tclUSBDiscoverer::vStopDeviceDetection()
{
   t_Bool bResult = false;
   if(NULL != m_poExtDiscoverer)
   {
      //! Stop monitoring USB devices
      t_S32 s32ErrorCode = m_poExtDiscoverer->s32StopMonitoring();
      bResult = (0 == s32ErrorCode);
   }
   ETG_TRACE_USR1((" spi_tclUSBDiscoverer::vStopDeviceDiscovery(): Stopping USB device discovery: Result = %d \n", ETG_ENUM(BOOL, bResult)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscoverer::vStartDeviceReporting
 ***************************************************************************/
t_Void spi_tclUSBDiscoverer::vStartDeviceReporting()
{
   //! not implemented.
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscoverer::vStartDeviceReporting
 ***************************************************************************/
t_Void spi_tclUSBDiscoverer::vStopDeviceReporting()
{
   //! not implemented.
}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBDiscoverer::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclUSBDiscoverer::vRegisterCallbacks(const trDiscovererCbs corfrDisccbs)
{
   SPI_INTENTIONALLY_UNUSED(corfrDisccbs);
   ETG_TRACE_USR1(("[DESC] spi_tclUSBDiscoverer::vRegisterCallbacks entered : Not used\n"));
}



/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBDiscoverer::vUSBEntityAppearedCb
 ***************************************************************************/
t_Void spi_tclUSBDiscoverer::vOnUSBDeviceConnectionCb(trUSBDeviceInfo &rfrUSBDeviceInfo)
{

   ETG_TRACE_USR1((" spi_tclUSBDiscoverer::vOnUSBDeviceConnectionCb entered \n"));


   trUSBDeviceInfo rExistingUSBDeviceInfo;
   //! Store the device USB information
   if(NULL != m_poDiscoveryData)
   {
      m_poDiscoveryData->vGetUSBDeviceDetails(rfrUSBDeviceInfo.u32DeviceHandle, rExistingUSBDeviceInfo);
      if(e8_PROFILE_CDCNCM_MIRRORLINK == rExistingUSBDeviceInfo.enUSBProfile)
      {
         rfrUSBDeviceInfo.enUSBProfile = e8_PROFILE_CDCNCM_MIRRORLINK;
      }
      m_poDiscoveryData->vSetUSBDeviceDetails(rfrUSBDeviceInfo.u32DeviceHandle, rfrUSBDeviceInfo);
   }

   trDeviceInfo rDeviceInfo;
   rDeviceInfo.u32DeviceHandle = rfrUSBDeviceInfo.u32DeviceHandle;
   rDeviceInfo.szDeviceManufacturerName = rfrUSBDeviceInfo.szManufacturer;
   //CRQ 430789: [A-IVI][G3G] - Interface to update USB port using system path information
   rDeviceInfo.szSystemPath = rfrUSBDeviceInfo.szSystemPath;
   //! Use the device category as Unknown until aoap switch is confirmed
   rDeviceInfo.enDeviceCategory = e8DEV_TYPE_UNKNOWN;

   rDeviceInfo.enDeviceConnectionStatus = e8DEV_CONNECTED;
   rDeviceInfo.enDeviceConnectionType = e8USB_CONNECTED;
   rDeviceInfo.rProjectionCapability.enUSBPortType = rfrUSBDeviceInfo.enUSBPortType;
   rDeviceInfo.u32VendorID = rfrUSBDeviceInfo.u32VendorID;
   rDeviceInfo.u32ProductID = rfrUSBDeviceInfo.u32ProductID;
   rDeviceInfo.szSerialNumber = rfrUSBDeviceInfo.szSerialNumber;
   if (true == rfrUSBDeviceInfo.bIsAOAPSupported)
   {
      rDeviceInfo.rProjectionCapability.enAndroidAutoSupport = e8SPI_SUPPORTED;
      rDeviceInfo.rProjectionCapability.enCarlifeSupport = e8SPI_SUPPORTED;
      rDeviceInfo.rProjectionCapability.enDeviceType = e8_ANDROID_DEVICE;
   }
   else
   {
      rDeviceInfo.rProjectionCapability.enAndroidAutoSupport = e8SPI_NOTSUPPORTED;
      rDeviceInfo.rProjectionCapability.enCarlifeSupport = e8SPI_NOTSUPPORTED;
      rDeviceInfo.rProjectionCapability.enDeviceType = e8_UNKNOWN_DEVICE;
   }


   DeviceInfoMsg oDevInfoMsg;
   oDevInfoMsg.m_enDiscovererType = e8_DISCOVERER_TYPE_USB;
   *(oDevInfoMsg.m_prDeviceInfo) = rDeviceInfo;
   spi_tclDiscoveryMsgQInterface *poMsgQinterface = spi_tclDiscoveryMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oDevInfoMsg, sizeof(oDevInfoMsg));
   }   //if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBDiscoverer::vUSBEntityDisappearedCb
 ***************************************************************************/
t_Void spi_tclUSBDiscoverer::vOnDeviceDisconnectionCb(trUSBDeviceInfo &rfrDeviceInfo)
{
   ETG_TRACE_USR1((" spi_tclUSBDiscoverer::vOnDeviceDisconnectionCb entered \n"));
   DeviceDisconnectionMsg oDeviceDisconnectionMsg;
   oDeviceDisconnectionMsg.m_u32DeviceHandle = rfrDeviceInfo.u32DeviceHandle;
   oDeviceDisconnectionMsg.m_enDiscovererType = e8_DISCOVERER_TYPE_USB;
   oDeviceDisconnectionMsg.m_enDeviceConnectionType = e8USB_CONNECTED;
   spi_tclDiscoveryMsgQInterface *poMsgQinterface = spi_tclDiscoveryMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oDeviceDisconnectionMsg, sizeof(oDeviceDisconnectionMsg));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBDiscoverer::vAOAPEntityAppearedCb
 ***************************************************************************/
t_Void spi_tclUSBDiscoverer::vOnAOAPDeviceConnectionCb(trUSBDeviceInfo &rfrUSBDeviceInfo)
{
   ETG_TRACE_USR1((" spi_tclUSBDiscoverer::vOnAOAPDeviceConnectionCb entered \n"));

   trDeviceInfo rDeviceInfo;
   rDeviceInfo.u32DeviceHandle = rfrUSBDeviceInfo.u32DeviceHandle;
   rDeviceInfo.enDeviceCategory = e8DEV_TYPE_ANDROIDAUTO;
   rDeviceInfo.enDeviceConnectionStatus = e8DEV_CONNECTED;
   rDeviceInfo.enDeviceConnectionType = e8USB_CONNECTED;
   rDeviceInfo.enDeviceProfile = e8_PROFILE_AOAP;
   rDeviceInfo.rProjectionCapability.enDeviceType = e8_ANDROID_DEVICE;
   rDeviceInfo.rProjectionCapability.enAndroidAutoSupport = e8SPI_SUPPORTED;
   //! Store the device aoap information
   if (NULL != m_poDiscoveryData)
   {
      m_poDiscoveryData->vSetUSBDeviceDetails(rfrUSBDeviceInfo.u32DeviceHandle, rfrUSBDeviceInfo);
   }

   DeviceInfoMsg oDevInfoMsg;
   oDevInfoMsg.m_enDiscovererType = e8_DISCOVERER_TYPE_USB;
   *(oDevInfoMsg.m_prDeviceInfo) = rDeviceInfo;
   spi_tclDiscoveryMsgQInterface *poMsgQinterface = spi_tclDiscoveryMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oDevInfoMsg, sizeof(oDevInfoMsg));
   }   //if (NULL != poMsgQinterface)
}
