/*!
 *******************************************************************************
 * \file             spi_tclUSBDiscovererImpl.h
 * \brief            USB device discoverer implementation using ADIT discoverer
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    USB device discoverer implementation using ADIT discoverer
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLUSBDISCOVERERIMPL_H_
#define SPI_TCLUSBDISCOVERERIMPL_H_



/******************************************************************************
 | includes:
 | 1)AAP - includes
 | 2)Typedefines
 |----------------------------------------------------------------------------*/
#include "spi_tclUSBDiscovererIntf.h"
#include "spi_ConnMngrTypeDefines.h"
#include "spi_usb_discoverer_types.h"
#include "spi_usb_discoverer.h"
#include "spi_tclDiscovererSettingsIntf.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/*!
 * \class spi_tclUSBDiscovererImpl
 * \brief USB device discoverer implementation using ADIT discoverer. This class
 * overrides the USB device discovery interface class, spi_tclUSBDiscovererIntf.
 */

class spi_tclUSBDiscovererImpl: public spi_tclUSBDiscovererIntf
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclUSBDiscovererImpl::spi_tclUSBDiscovererImpl();
       ***************************************************************************/
      /*!
       * \brief  Default Constructor
       * \sa      ~spi_tclUSBDiscovererImpl()
       **************************************************************************/
      spi_tclUSBDiscovererImpl();

      /***************************************************************************
       ** FUNCTION:  virtual spi_tclUSBDiscovererImpl::~spi_tclUSBDiscovererImpl()
       ***************************************************************************/
      /*!
       * \brief   Destructor
       * \sa      spi_tclUSBDiscovererImpl()
       **************************************************************************/
      virtual ~spi_tclUSBDiscovererImpl();

      /***************************************************************************
       ** FUNCTION:  spi_tclUSBDiscovererIntf::bInitialize
       ***************************************************************************/
      /*!
       * \brief  Initializes USB device discovery
       **************************************************************************/
      virtual t_Bool bInitialize(const trUSBDiscovererCbs &corfrDiscovererCbs);

      /***************************************************************************
       ** FUNCTION:  spi_tclUSBDiscovererImpl::vUnInitialize
       ***************************************************************************/
      /*!
       * \brief  Uninitializes USB device discovery
       **************************************************************************/
      virtual t_Void  vUnInitialize() ;

      /***************************************************************************
       ** FUNCTION:  spi_tclUSBDiscovererImpl::s32StartMonitoring
       ***************************************************************************/
      /*!
       * \brief  Start monitoring USB devices. Creates a thread which listens to USB
       *         devices reported (using libraries such as libusb)
       **************************************************************************/
      virtual t_S32  s32StartMonitoring();

      /***************************************************************************
       ** FUNCTION:  spi_tclUSBDiscovererImpl::s32StopMonitoring
       ***************************************************************************/
      /*!
       * \brief  Stop monitoring USB devices
       **************************************************************************/
      virtual t_S32  s32StopMonitoring();
	  
	   /***************************************************************************
       ** FUNCTION:  spi_tclUSBDiscovererImpl::bIsDeviceSelectableOnUSBPort
       ***************************************************************************/
      /*!
       * \brief   Check if USB port where device is connected is enabled/disabled
       **************************************************************************/
      t_Bool bIsDeviceSelectableOnUSBPort(t_usbDeviceInformation *prUSBDevInfo);

      /***************************************************************************
       ** FUNCTION:  t_Void  spi_tclUSBDiscovererImpl::vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
       ***************************************************************************/
       /*!
       * \fn      t_Void vSetDiscoverersettingsInstance()
       * \brief   Method to set discoverer settings instance.
       * \param   poDiscovererSettingsIntf  : pointer to DiscovererSettingsIntf.
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf) override;

   private:

      /***************************************************************************
       *********************************PRIVATE************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclUSBDiscoverer::u32GenerateUniqueDeviceID
       ***************************************************************************/
      /*!
       * \brief   Generates unique device ID by calculating crc with product vendor and serial number
       * \param   rfrUSBDeviceInfo : Contains information of the USB device
       **************************************************************************/
      t_U32 u32GenerateUniqueDeviceID(trUSBDeviceInfo &rfrUSBDeviceInfo);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclUSBDiscoverer::vCopyUSBDeviceInfo
       ***************************************************************************/
      /*!
       * \brief   Copies USB device info to a different structure
       **************************************************************************/
      t_Void vCopyUSBDeviceInfo(trUSBDeviceInfo &rfrUSBDeviceInfo, const t_usbDeviceInformation *prUSBDevInfo);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclUSBDiscoverer::vUSBEntityAppearedCb
       ***************************************************************************/
      /*!
       * \brief   Callback from USB discoverer when a new USb device is detected
       * \param   pContext: context passed during registration
       * \param   prUSBDevInfo: Structure containing USB info
       * \param   s32Result: Result
       **************************************************************************/
      static t_Void vUSBEntityAppearedCb(t_Void *pContext, t_usbDeviceInformation *prUSBDevInfo, t_S32 s32Result);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclUSBDiscoverer::vUSBEntityDisappearedCb
       ***************************************************************************/
      /*!
       * \brief   Callback from USB discoverer when a USB device disappears
       * \param   pContext: context passed during registration
       * \param   prUSBDevInfo: Structure containing USB info
       * \param   s32Result: Result
       **************************************************************************/
      static t_Void vUSBEntityDisappearedCb(t_Void *pContext, t_usbDeviceInformation *prUSBDevInfo, t_S32 s32Result);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclUSBDiscoverer::vAOAPEntityAppearedCb
       ***************************************************************************/
      /*!
       * \brief   Callback from discoverer when a device is detected in AOAP mode
       * \param   pContext: context passed during registration
       * \param   prUSBDevInfo: Structure containing USB info
       * \param   s32Result: Result
       **************************************************************************/
      static t_Void vAOAPEntityAppearedCb(t_Void *pContext, t_usbDeviceInformation *prUSBDevInfo, t_S32 s32Result);

      //! Discoverer pointer
      AOAP::UsbDiscoverer *m_poDiscoverer;

      trUSBDiscovererCbs m_rDiscCbs;
      spi_tclDiscovererSettingsIntf* m_poDiscovererSettings;
};

#endif /* SPI_TCLUSBDISCOVERERIMPL_H_ */
