/*!
 *******************************************************************************
 * \file             spi_tclUSBReset.cpp
 * \brief            ML device switcher
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:     ML device switcher
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.06.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#include <libusb-1.0/libusb.h>
#include "spi_tclUSBReset.h"
#include "spi_tclMPlayClientHandler.h"
#include "spi_tclFactory.h"
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclUSBReset.cpp.trc.h"
#endif
#endif

static const t_U32 scou32BufferSize = 256;
static const t_U32 scou32ResetIntervalinms = 4000;
static const t_U32 scou32MaxNoofUSBResetRetries = 4;

/***************************************************************************
 ** FUNCTION:  spi_tclUSBReset::spi_tclUSBReset
 ***************************************************************************/
spi_tclUSBReset::spi_tclUSBReset(spi_tclDiscoveryDataIntf *pDiscovereryDataIntf): m_pDiscovereryDataIntf(pDiscovereryDataIntf)
{

}

/***************************************************************************
 ** FUNCTION:  spi_tclUSBReset::~spi_tclUSBReset()
 ***************************************************************************/
spi_tclUSBReset::~spi_tclUSBReset()
{
   m_pDiscovereryDataIntf = NULL;
}

/***************************************************************************
 ** FUNCTION:  t_U32 spi_tclUSBReset::bResetUSBDevice
 ***************************************************************************/
t_Bool spi_tclUSBReset::bResetUSBDevice(const trUSBDeviceInfo &rfrDeviceInfo)
{
   ETG_TRACE_USR1(("[DESC] spi_tclUSBReset::bResetUSBDevice entered\n"));
   t_Bool bRetval = false;
   //! @ TODO R1 : Follow coding conventions and comments for below impl
   //! @ TODO R2 : Separate interface for libusb

   libusb_device_handle *pDeviceHandler = NULL;
   libusb_device **ppUSBDevice = NULL;
   //libusb_device *pUSBDeviceFound = NULL;
   libusb_context* ctx = NULL;
   t_S32 s32Result = libusb_init(&ctx);
   ETG_TRACE_USR4(("[PARAM] spi_tclUSBReset::bSwitchToUSBMode: libusbinit = %d\n",s32Result));
   ETG_TRACE_USR4(("[PARAM] spi_tclUSBReset::bSwitchToUSBMode: LIBUSB_SUCCESS = %d\n",LIBUSB_SUCCESS));
   if (LIBUSB_SUCCESS != s32Result)
   {
      return bRetval;
   }
   //! TODO check result before proceeding
   ssize_t count = libusb_get_device_list(ctx, &ppUSBDevice);
   ETG_TRACE_USR4(("[PARAM] spi_tclUSBReset::bSwitchToUSBMode: libusbinit = %d, No of devices = %d\n", s32Result, count));
   for (auto i = 0; i < count; i++)
   {
      //! Get the serial number of the device
      libusb_device_descriptor desc;
      s32Result = libusb_get_device_descriptor(ppUSBDevice[i], &desc);

      t_UChar czSerialNumber[scou32BufferSize];

      if (s32Result >= 0)
      {
         s32Result = libusb_open(ppUSBDevice[i], &pDeviceHandler);
         ETG_TRACE_USR4((" [PARAM]spi_tclUSBReset::bSwitchToUSBMode: libusb_open = %d\n", s32Result));
         if (s32Result >= 0)
         {
            t_S32 s32RetVal = libusb_get_string_descriptor_ascii(pDeviceHandler,
                     desc.iSerialNumber,
                     czSerialNumber,
                     scou32BufferSize);
            t_String szSerialNumberFound = (t_Char*) (czSerialNumber);
            ETG_TRACE_USR4(("[PARAM] spi_tclUSBReset::bSwitchToUSBMode: NO of bytes = %d\n",s32RetVal));
            ETG_TRACE_USR4(("[PARAM] spi_tclUSBReset::bSwitchToUSBMode: Device Serial number = %s\n",szSerialNumberFound.c_str()));
            if (rfrDeviceInfo.szSerialNumber == szSerialNumberFound)
            {
               ETG_TRACE_USR4(("[DESC] Device with required serial number found. Triggering USB reset \n"));
               if ((LIBUSB_SUCCESS == s32Result) && pDeviceHandler != NULL)
               {
                  s32Result = libusb_reset_device(pDeviceHandler);
                  bRetval = ((LIBUSB_SUCCESS == s32Result) || (LIBUSB_ERROR_NOT_FOUND == s32Result));
                  ETG_TRACE_USR4(("[PARAM] spi_tclUSBReset::bSwitchToUSBMode: USB Reset Result (libusb_reset_device) = %d\n", s32Result));
                  //s32Result >= LIBUSB_SUCCESS ?
               }
               libusb_close(pDeviceHandler);
               break;
            }
            libusb_close(pDeviceHandler);
         }
      }
   }
   libusb_exit(ctx);
   vRetryUSBReset(rfrDeviceInfo.u32DeviceHandle);
   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBReset::vRetryUSBReset
 ***************************************************************************/
t_Void spi_tclUSBReset::vRetryUSBReset(const t_U32 cou32DeviceHandle)
{
   //! If USB reset has failed. try to reset the device for scou32NoofUSBResetRetries times
   //! at an interval of scou32ResetIntervalinms seconds
   //! Start the reset timer only for the first trial of USB reset
   //! Check if the device is not already under USB reset retrial.
   //! If the USB reset retrail is already in progress for the device, don't start new timer

   ETG_TRACE_USR1((" spi_tclAAPCmdDiscoverer::vRetryUSBReset : Retrying USB reset for device = 0x%x n",
         cou32DeviceHandle));
   t_Bool bValidRequest = true;
   //! Check if the device is not already under USB reset retrial.
   //! If the USB reset retrail is already in progress for the device, don't start new timer
   m_oLockUSBResetInfo.s16Lock();
   if (m_mapUSBDeviceResetInfo.end() != m_mapUSBDeviceResetInfo.find(cou32DeviceHandle))
   {
      bValidRequest = false;
   }
   m_oLockUSBResetInfo.vUnlock();

   ETG_TRACE_USR1((" spi_tclAAPCmdDiscoverer::vRetryUSBReset : Retrying USB reset for device = 0x%x bValidDevice = %d\n",
         cou32DeviceHandle, ETG_ENUM(BOOL, bValidRequest)));
   if(true == bValidRequest)
   {
      Timer* poTimer = Timer::getInstance();
      if (NULL != poTimer)
      {
         //! Start the USB reset retrial timer. On the expirty of this timer, USB reset will be attempted
         timer_t rTimerID;
         poTimer->StartTimer(rTimerID, scou32ResetIntervalinms, scou32ResetIntervalinms, this,
               &spi_tclUSBReset::bUSBResetTimerCb, NULL);

         //! Store the USB reset retrial details
         trUSBResetInfo rUSBResetInfo;
         rUSBResetInfo.oTimerID = rTimerID;
         rUSBResetInfo.u32RetryCount = 0;
         ETG_TRACE_USR4((" Started USB reset Retrail timer for device = 0x%x with Timer ID = %p", cou32DeviceHandle,rTimerID));
         m_oLockUSBResetInfo.s16Lock();
         m_mapUSBDeviceResetInfo.insert(std::pair<t_U32, trUSBResetInfo>(cou32DeviceHandle, rUSBResetInfo));
         m_oLockUSBResetInfo.vUnlock();
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBReset::vStopUSBResetRetrial
 ***************************************************************************/
t_Void spi_tclUSBReset::vStopUSBResetRetrial(const t_U32 cou32DeviceHandle)
{
   Timer* poTimer = Timer::getInstance();
  //! Cancel the timer if the USB reset is successful and also delete the USB reset details for the device
   t_Bool bUSBResetInProgress = false;
   m_oLockUSBResetInfo.s16Lock();
   ETG_TRACE_USR4(("No of devices under USB Reset = %d \n", m_mapUSBDeviceResetInfo.size()));
   std::map<t_U32, trUSBResetInfo>::iterator itMapUSBDev = m_mapUSBDeviceResetInfo.find(cou32DeviceHandle);
   if (itMapUSBDev != m_mapUSBDeviceResetInfo.end())
   {
      trUSBResetInfo rUSBResetInfo = itMapUSBDev->second;
      if ((0 != rUSBResetInfo.oTimerID) && (NULL != poTimer))
      {
         ETG_TRACE_USR4(("USB reset has succeeded for device 0x%x or the device is disconnected by user."
                  " Cancelling USB reset retrial timer\n ", itMapUSBDev->first));
         poTimer->CancelTimer(rUSBResetInfo.oTimerID);
         //! Clear USB reset device info as the USB reset as succeeded
         m_mapUSBDeviceResetInfo.erase(itMapUSBDev);
         bUSBResetInProgress = true;
      }
   }
   SPI_INTENTIONALLY_UNUSED(bUSBResetInProgress);
   m_oLockUSBResetInfo.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclUSBReset::bUSBResetTimerCb
 ***************************************************************************/
t_Bool spi_tclUSBReset::bUSBResetTimerCb(timer_t rTimerID, t_Void *pvObject,
         const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   ETG_TRACE_USR1(("spi_tclUSBReset::bUSBResetTimerCb : USB reset failed: Trying again .\n"));
   if(NULL != pvObject)
   {
   spi_tclUSBReset* poUSBReset = static_cast<spi_tclUSBReset*> (pvObject);
   Timer* poTimer = Timer::getInstance();
   //! Get the Device info for the corresponding timer ID
   if (NULL != poUSBReset)
   {
      //! Fetch the USB reset info for the timer indicated by rTimerID
      trUSBResetInfo rUSBResetInfo;
      t_U32 u32DeviceHandle = 0;
      poUSBReset->m_oLockUSBResetInfo.s16Lock();
      std::map<t_U32, trUSBResetInfo>::iterator itMapUSBResetInfo;
      ETG_TRACE_USR4(("No of devices under USB Reset = %d, timer ID = %p",
               poUSBReset->m_mapUSBDeviceResetInfo.size(), rTimerID));
      //! Iterate through the map to find the USB reset details corresponding to this timer
      for (itMapUSBResetInfo = poUSBReset->m_mapUSBDeviceResetInfo.begin(); itMapUSBResetInfo
            != poUSBReset->m_mapUSBDeviceResetInfo.end(); itMapUSBResetInfo++)
      {
         if (rTimerID == (itMapUSBResetInfo->second).oTimerID)
         {
            rUSBResetInfo = itMapUSBResetInfo->second;
            u32DeviceHandle = (itMapUSBResetInfo->first);
            break;
         }
      }
      //Fix applied for reset issues - ticket reference: NCG3D-48726
      t_Bool bResetInfoExists = (poUSBReset->m_mapUSBDeviceResetInfo.end() != itMapUSBResetInfo);
      poUSBReset->m_oLockUSBResetInfo.vUnlock();
      //!  Retry USB reset if the USB Reset Details are found in the list
      if ((NULL != poTimer) && (0 != u32DeviceHandle) && (true == bResetInfoExists))
      {
         poUSBReset->m_oLockUSBResetInfo.s16Lock();
         itMapUSBResetInfo = poUSBReset->m_mapUSBDeviceResetInfo.find(u32DeviceHandle);
         if(itMapUSBResetInfo != poUSBReset->m_mapUSBDeviceResetInfo.end())
         {
            ((itMapUSBResetInfo->second).u32RetryCount)++;
            ETG_TRACE_USR4((" Retry count =%d for Device = 0x%x\n", (itMapUSBResetInfo->second).u32RetryCount, u32DeviceHandle));
         }
         else
         {
            u32DeviceHandle = 0;
         }
         poUSBReset->m_oLockUSBResetInfo.vUnlock();
         trUSBDeviceInfo rUSBDeviceInfo;
         if(NULL != poUSBReset->m_pDiscovereryDataIntf)
         {
            poUSBReset->m_pDiscovereryDataIntf->vGetUSBDeviceDetails(u32DeviceHandle, rUSBDeviceInfo);
         }
         poUSBReset->bResetUSBDevice(rUSBDeviceInfo);
         poUSBReset->m_oLockUSBResetInfo.s16Lock();
         itMapUSBResetInfo = poUSBReset->m_mapUSBDeviceResetInfo.find(u32DeviceHandle);
         if(itMapUSBResetInfo != poUSBReset->m_mapUSBDeviceResetInfo.end())
         {
            //! Cancel the reset timer after maximum number of retries
            if ((0 != rTimerID)
                     && (((itMapUSBResetInfo->second).u32RetryCount)
                              == scou32MaxNoofUSBResetRetries))
            {
               ETG_TRACE_ERR(("Maximum no of retrials for USB reset reached. Cancelling timer.\n"));
               poTimer->CancelTimer(rTimerID);
               //! Clear USB Reset device info
               if(itMapUSBResetInfo!= poUSBReset->m_mapUSBDeviceResetInfo.end() )
               {
                  poUSBReset->m_mapUSBDeviceResetInfo.erase(itMapUSBResetInfo);
               }
            }
         }
        poUSBReset->m_oLockUSBResetInfo.vUnlock();
      }
   }
   }
   return true;
}

