/*!
 *******************************************************************************
 * \file             spi_tclDeviceSwitcher.cpp
 * \brief            Device Profile Switcher
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Unified Device Discoverer
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#include "spi_tclDeviceSwitcher.h"
#include "spi_tclDeviceSwitchBase.h"
#include "Timer.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclDeviceSwitcher.cpp.trc.h"
#endif
#endif

/*lint -save -e1013 */
/*lint -save -e1055 */
/*lint -save -e10 */
/*lint -save -e40 */
/*lint -save -e746 */
/*lint -save -e774 */
/*lint -save -e63 */

//Timeout is in milliseconds.
static const t_U32 scou32DeviceSwitchTimeOut = 25000;
static const t_U32 scou32DeviceSwitchTimeOutForWireless = 58000;
static const t_U32 scou32SwitchFailureRetrialLimit = 2;

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::spi_tclDeviceSwitcher
 ***************************************************************************/
spi_tclDeviceSwitcher::spi_tclDeviceSwitcher(spi_tclDiscoveryDataIntf *poDiscoverIntf,
         std::map<tenDeviceCategory, spi_tclDeviceSwitchBase*> poDeviceSwitchers, spi_tclUSBResetIntf *poUSBReset) :
                  m_mapDeviceSwitchers(poDeviceSwitchers),
                  m_pDiscovereryDataIntf(poDiscoverIntf),
                  m_pUSBReset(poUSBReset),
                  m_poDiscovererSettings(NULL)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::spi_tclDeviceSwitcher Entered\n "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::~spi_tclDeviceSwitcher()
 ***************************************************************************/
spi_tclDeviceSwitcher::~spi_tclDeviceSwitcher()
{
   m_poDiscovererSettings = NULL;
}
/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vRegisterCallbacks(const trDeviceSwitcherCbs corfrDisccbs)
{
   m_rDeviceSwitcherCbs = corfrDisccbs;

   //! TODO move it to initialize
   trDeviceSwitcherCbs rDeviceSwitcherCbs;
   rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb = std::bind(&spi_tclDeviceSwitcher::vOnDeviceSwitchCompleteCb, this,
   SPI_FUNC_PLACEHOLDERS_2);

   m_oDeviceSwitchersMapLock.s16Lock();
   for (auto itvecDeviceSwitchers = m_mapDeviceSwitchers.begin(); itvecDeviceSwitchers != m_mapDeviceSwitchers.end();
            itvecDeviceSwitchers++)
   {
      if ((NULL != m_poDiscovererSettings) && (NULL != itvecDeviceSwitchers->second))
      {
         itvecDeviceSwitchers->second->vSetDiscoverersettingsInstance(m_poDiscovererSettings);
      }
      itvecDeviceSwitchers->second->vRegisterCallbacks(rDeviceSwitcherCbs);
   }
   m_oDeviceSwitchersMapLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::vPrepareDeviceSwitch
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vPrepareDeviceSwitch(tenDeviceCategory enDeviceCat, const t_U32 cou32DeviceHandle,
         tenDeviceConnectionType enDeviceConnType)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::vPrepareDeviceSwitch: Preparing switch  for device ID = %d", cou32DeviceHandle));
   //! Set the initial parameters before trigerring the device switch request
   trDeviceSwitchInfo rDeviceSwitchInfo;
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vGetDeviceSwitchInfo(cou32DeviceHandle, rDeviceSwitchInfo);
   }
   if ((false == rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress)
            || (rDeviceSwitchInfo.u32DeviceID != cou32DeviceHandle))
   {
      trUSBDeviceInfo rUSBDeviceInfo;
      if (NULL != m_pDiscovereryDataIntf)
      {
         m_pDiscovereryDataIntf->vGetUSBDeviceDetails(cou32DeviceHandle, rUSBDeviceInfo);
      }
      rDeviceSwitchInfo.u32DeviceID = cou32DeviceHandle;
      rDeviceSwitchInfo.u32VendorID = rUSBDeviceInfo.u32VendorID;
      rDeviceSwitchInfo.u32ProductID = rUSBDeviceInfo.u32ProductID;
      rDeviceSwitchInfo.szSerialNumber = rUSBDeviceInfo.szSerialNumber;
      rDeviceSwitchInfo.enDeviceType = rUSBDeviceInfo.enDeviceType;
      rDeviceSwitchInfo.bSwitchInProgress = true;
      rDeviceSwitchInfo.enDeviceSwitchType = enDeviceCat;
      rDeviceSwitchInfo.enDeviceConnectionType = enDeviceConnType;
   }
   else
   {
      ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::vPrepareDeviceSwitch: Early Device switch is in progress for device ID = 0x%x", cou32DeviceHandle));
   }

   t_U32 u32SwitchTimeout =
            (e8USB_CONNECTED == enDeviceConnType) ? scou32DeviceSwitchTimeOut : scou32DeviceSwitchTimeOutForWireless;

   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      //! Start the USB reset retrial timer. On the expiry of this timer, Device switch failed will be informed
      timer_t rTimerID = 0;
      poTimer->StartTimer(rTimerID, u32SwitchTimeout, 0, this, &spi_tclDeviceSwitcher::bDeviceSwitchTimerCb, NULL);
      m_oDeviceIDForSwitchLock.s16Lock();
      m_mapDeviceIDForSwitch[rTimerID] = cou32DeviceHandle;
      m_oDeviceIDForSwitchLock.vUnlock();
      ETG_TRACE_USR4(("[DESC] spi_tclDeviceSwitcher::vPrepareDeviceSwitch Started Device switch Timer %p", rTimerID));
      rDeviceSwitchInfo.t_TimerID = rTimerID;
   }
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vSetDeviceSwitchInfo(rDeviceSwitchInfo);
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::bSwitchDeviceMode
 ***************************************************************************/
t_Bool spi_tclDeviceSwitcher::bSwitchDeviceMode(tenDeviceCategory enDeviceCat, const t_U32 cou32DeviceHandle,
         tenDeviceType enDeviceType)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::bSwitchDeviceMode: Request to switch device to %d for Device ID = %d, Device Type is = %d", ETG_ENUM(DEVICE_CATEGORY,
            enDeviceCat), cou32DeviceHandle, ETG_ENUM(DEVICE_TYPE, enDeviceType)));
   t_Bool bRetVal = false;
   trUSBDeviceInfo rUSBDeviceInfo;
   trDeviceSwitchInfo rDeviceSwitchInfo;
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vGetUSBDeviceDetails(cou32DeviceHandle, rUSBDeviceInfo);
      m_pDiscovereryDataIntf->vGetDeviceSwitchInfo(cou32DeviceHandle, rDeviceSwitchInfo);
   }
   tenDeviceSwitchResult enDevSwitchResult = e8_DEVICESWITCH_RESULT_UNKNOWN;
   t_Bool bEarlySwitchinProgress = rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress;
   t_U32 u32DeviceIDUnderSwitch = rDeviceSwitchInfo.u32DeviceID;
   //! Donot trigger a device switch if the switch is already in progress
   if ((true == bEarlySwitchinProgress) && (u32DeviceIDUnderSwitch == cou32DeviceHandle))
   {
      //! Nothing to do
      ETG_TRACE_USR4(("[DESC] spi_tclDeviceSwitcher::bSwitchDeviceMode Device switch already in progress for the device 0x%x \n", cou32DeviceHandle));
      enDevSwitchResult = e8_DEVICESWITCH_RESULT_INPROGRESS;
   }
   //! Trigger device switch to projection mode for all devices. For apple devices, switch back to default mode is also triggered
   //! using the same interface
   else if ((e8_APPLE_DEVICE == enDeviceType) || (e8DEV_TYPE_UNKNOWN != enDeviceCat))
   {
      //! TODO combine reverse role switch and USB reset logic
      if (e8DEV_TYPE_UNKNOWN == enDeviceCat)
      {
         enDeviceCat = e8DEV_TYPE_DIPO;
      }


      spi_tclDeviceSwitchBase* poDeviceSwitcher = NULL;

      m_oDeviceSwitchersMapLock.s16Lock();
      if(m_mapDeviceSwitchers.end() != m_mapDeviceSwitchers.find(enDeviceCat))
      {
         poDeviceSwitcher = m_mapDeviceSwitchers[enDeviceCat];
      }
      m_oDeviceSwitchersMapLock.vUnlock();

      if ((NULL != poDeviceSwitcher) && (NULL != m_pDiscovereryDataIntf))
      {
         enDevSwitchResult = poDeviceSwitcher->enSwitchDeviceMode(rUSBDeviceInfo, rDeviceSwitchInfo, enDeviceType);
         m_pDiscovereryDataIntf->vSetDeviceSwitchInfo(rDeviceSwitchInfo);
      }
      else
      {
         enDevSwitchResult = e8_DEVICESWITCH_RESULT_FAILED;
      }
      bRetVal = (enDevSwitchResult != e8_DEVICESWITCH_RESULT_FAILED);
   }
   //! Trigger USB reset if the requested switch type is unknown for all devices except apple devices
   else if ((e8_APPLE_DEVICE != enDeviceType) && (NULL != m_pUSBReset))
   {
      bRetVal = m_pUSBReset->bResetUSBDevice(rUSBDeviceInfo);
   }

   //! If device switch is not required, clear the device switch flags and return
   if (e8_DEVICESWITCH_RESULT_SWITCH_NOTREQUIRED == enDevSwitchResult)
   {
      vOnDeviceSwitchCompleteCb(cou32DeviceHandle, true);
   }

   //! If device switch is not required, clear the device switch flags and return
   if ((e8_DEVICESWITCH_RESULT_DEVICE_RESET_REQUIRED == enDevSwitchResult) && (NULL != m_pUSBReset))
   {
      //! Abort device switch and reset the device
      vOnDeviceSwitchCompleteCb(cou32DeviceHandle, false);
      m_pUSBReset->bResetUSBDevice(rUSBDeviceInfo);
   }
   //! If device switch is not processed, clear the device switch flags and return
   if (e8_DEVICESWITCH_RESULT_FAILED == enDevSwitchResult)
   {
      vOnDeviceSwitchCompleteCb(cou32DeviceHandle, false);
   }
   //! if the early switch in in progress then return from device switch request without resetting the flags
   //! The device switch flags will be reset once the device reappears in projection mode
   if ((e8_DEVICESWITCH_RESULT_INPROGRESS == enDevSwitchResult)
            && (e8DEV_TYPE_UNKNOWN != rDeviceSwitchInfo.enDeviceSwitchType))
   {
      if (NULL != m_rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb)
      {
         (m_rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb)(cou32DeviceHandle, true);
      }
   }
   return bRetVal;
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceSwitcher::vSetHeadUnitInfo
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vSetHeadUnitInfo(const trHeadUnitInfo &corfrHeadUnitInfo)
{
   ETG_TRACE_USR1((" spi_tclDeviceSwitcher::vSetHeadUnitInfo entered \n"));

   m_oDeviceSwitchersMapLock.s16Lock();
   for (auto itvecDeviceSwitchers = m_mapDeviceSwitchers.begin(); itvecDeviceSwitchers != m_mapDeviceSwitchers.end();
            itvecDeviceSwitchers++)
   {
      itvecDeviceSwitchers->second->vSetHeadUnitInfo(corfrHeadUnitInfo);
   }
   m_oDeviceSwitchersMapLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceSwitcher::bIsSwitchInProgress
 ***************************************************************************/
t_Bool spi_tclDeviceSwitcher::bIsSwitchInProgress(const t_U32 cou32DeviceHandle)
{
   t_Bool bSwitchInProgress = false;

   trDeviceSwitchInfo rDeviceSwitchInfo;
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vGetDeviceSwitchInfo(cou32DeviceHandle, rDeviceSwitchInfo);
   }

   if (rDeviceSwitchInfo.u32DeviceID == cou32DeviceHandle)
   {
      bSwitchInProgress = (rDeviceSwitchInfo.bSwitchInProgress) || (rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress);
   }

   ETG_TRACE_USR1((" spi_tclDeviceSwitcher::bIsSwitchInProgress Device switch status is %d for device ID = 0x%x \n", ETG_ENUM(BOOL,
            bSwitchInProgress), cou32DeviceHandle));
   return bSwitchInProgress;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceSwitcher::vResetDeviceSwitchInfo
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vResetDeviceSwitchInfo(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::Resetting device switch information as the switch is complete\n "));

   //! For mySPIN Android devices, the switch call would have already returned.
   //! Hence don't respond for Reset device switch call
   //! Only stop the timer and Reset the flags
   trDeviceSwitchInfo rDeviceSwitchInfo;
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vGetDeviceSwitchInfo(cou32DeviceHandle, rDeviceSwitchInfo);
   }

   //tenDeviceType enDeviceType = rDeviceSwitchInfo.enDeviceType;
   //tenDeviceCategory enDeviceSwitchType = rDeviceSwitchInfo.enDeviceSwitchType;
   // if (e8DEV_TYPE_MYSPIN == enDeviceSwitchType)
   // {
   // Timer* poTimer = Timer::getInstance();
   // if ((NULL != poTimer) && (0 != rDeviceSwitchInfo.t_TimerID))
   // {
   // //! Cancel Device switch timer
   // poTimer->CancelTimer(rDeviceSwitchInfo.t_TimerID);
   // rDeviceSwitchInfo.t_TimerID = 0;
   // }
   // rDeviceSwitchInfo.bSwitchInProgress = false;
   // rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress = false;
   // if(NULL != m_pDiscovereryDataIntf)
   // {
   // m_pDiscovereryDataIntf->vSetDeviceSwitchInfo(rDeviceSwitchInfo);
   // }
   // }
   // else
   // {
   vOnDeviceSwitchCompleteCb(cou32DeviceHandle, true);
   // }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceSwitcher::vGetDeviceSwitchInfo
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vGetDeviceSwitchInfo(const t_U32 cou32Devicehandle,
         trDeviceSwitchInfo &rfrDeviceSwitchInfo)
{

   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vGetDeviceSwitchInfo(cou32Devicehandle, rfrDeviceSwitchInfo);
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::bIsDeviceSwitcherBusy
 ***************************************************************************/
t_Bool spi_tclDeviceSwitcher::bIsDeviceSwitcherBusy(const t_U32 cou32DeviceHandle)
{
   trDeviceSwitchInfo rDeviceSwitchInfo;
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vGetDeviceSwitchInfo(cou32DeviceHandle, rDeviceSwitchInfo);
   }
   t_Bool bDeviceSwitcherBusy = rDeviceSwitchInfo.bSwitchInProgress;
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::bIsDeviceSwitcherBusy =%d \n ", ETG_ENUM(BOOL, bDeviceSwitcherBusy)));
   return bDeviceSwitcherBusy;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::vSetProjectionState
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vSetProjectionState(tenDeviceCategory enProjType, tenEnabledInfo enEnabledInfo)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::vSetProjectionState for %d to %d : currently unused\n ", ETG_ENUM(DEVICE_CATEGORY,
            enProjType), ETG_ENUM(ENABLED_INFO, enEnabledInfo)));
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDeviceSwitcher::vSetEarlyDeviceSwitchInfo
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vSetEarlyDeviceSwitchInfo(const t_U32 cou32DeviceID, t_Bool bEarlySwitchRequired)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::vSetEarlyDeviceSwitchInfo Device ID = 0x%x bEarlySwitchRequired = %d\n", cou32DeviceID, ETG_ENUM(BOOL,
            bEarlySwitchRequired)));
   trDeviceSwitchInfo rDeviceSwitchInfo;
   rDeviceSwitchInfo.u32DeviceID = cou32DeviceID;
   rDeviceSwitchInfo.bIsEarlyDeviceSwitchInProgress = bEarlySwitchRequired;
   rDeviceSwitchInfo.enDeviceSwitchType = e8DEV_TYPE_DIPO;
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vSetDeviceSwitchInfo(rDeviceSwitchInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::vOnDeviceSwitchCompleteCb
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vOnDeviceSwitchCompleteCb(const t_U32 cou32DeviceHandle, t_Bool bSwitchStatus)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::vOnDeviceSwitchCompleteCb entered\n"));
   trDeviceSwitchInfo rDevSwitchinfo;
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vGetDeviceSwitchInfo(cou32DeviceHandle, rDevSwitchinfo);

      t_Bool bIsUSBResetComplete = false;

      Timer* poTimer = Timer::getInstance();
      if ((NULL != poTimer) && (0 != rDevSwitchinfo.t_TimerID))
      {
         //! Cancel Device switch timer
         poTimer->CancelTimer(rDevSwitchinfo.t_TimerID);
         //erase the timer id info in m_mapDeviceIDForSwitch
         m_oDeviceIDForSwitchLock.s16Lock();
         if (SPI_MAP_NOT_EMPTY(m_mapDeviceIDForSwitch))
         {
            if (m_mapDeviceIDForSwitch.end() != m_mapDeviceIDForSwitch.find(rDevSwitchinfo.t_TimerID))
            {
               m_mapDeviceIDForSwitch.erase(rDevSwitchinfo.t_TimerID);
            }
         }
         else
         {
            ETG_TRACE_USR1(("m_mapDeviceIDForSwitch is empty\n"));
         }
         m_oDeviceIDForSwitchLock.vUnlock();
         rDevSwitchinfo.t_TimerID = 0;
      }

      //! if USB reset was in progress, stop the retrial timer
//   m_oLockDeviceSwitchInfo.s16Lock();
      if ((true == rDevSwitchinfo.bSwitchInProgress) && (cou32DeviceHandle == rDevSwitchinfo.u32DeviceID)
               && (e8DEV_TYPE_UNKNOWN == rDevSwitchinfo.enDeviceSwitchType) && (NULL != m_pUSBReset))
      {
         m_pUSBReset->vStopUSBResetRetrial(cou32DeviceHandle);
         bIsUSBResetComplete = true;
      }
      rDevSwitchinfo.bSwitchInProgress = false;
      rDevSwitchinfo.bIsEarlyDeviceSwitchInProgress = false;
      t_U32 u32DeviceID = rDevSwitchinfo.u32DeviceID;

      m_pDiscovereryDataIntf->vSetDeviceSwitchInfo(rDevSwitchinfo);

      if (NULL != m_rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb)
      {
         (m_rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb)(u32DeviceID, bSwitchStatus);
      }

      //! If device switch failed for whitelisted device, retry device switch after USB reset is complete
      //! Currently this handling is required only for mirrorlink
      if ((true == bIsUSBResetComplete))
      {
         if (NULL != m_poDiscovererSettings)
         {
            if (true
                     == m_poDiscovererSettings->bIsDeviceWhitelisted(rDevSwitchinfo.u32ProductID,
                              rDevSwitchinfo.u32VendorID,
                              e8DEV_TYPE_MIRRORLINK))
            {
               vRetryDeviceSwitch(cou32DeviceHandle, e8DEV_TYPE_MIRRORLINK);
            }
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::bIsDeviceWhitelisted
 ***************************************************************************/
t_Bool spi_tclDeviceSwitcher::bIsDeviceWhitelisted(const t_U32 cou32DeviceID)
{
   trDeviceSwitchInfo rDeviceSwitchInfo;
   if (NULL != m_pDiscovereryDataIntf && NULL != m_poDiscovererSettings)
   {
      m_pDiscovereryDataIntf->vGetDeviceSwitchInfo(cou32DeviceID, rDeviceSwitchInfo);
      if (true
               == m_poDiscovererSettings->bIsDeviceWhitelisted(rDeviceSwitchInfo.u32ProductID,
                        rDeviceSwitchInfo.u32VendorID,
                        e8DEV_TYPE_MIRRORLINK))
      {
         return true;
      }
   }
   return false;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::bDeviceSwitchCb
 ***************************************************************************/
t_Bool spi_tclDeviceSwitcher::bDeviceSwitchTimerCb(timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   ETG_TRACE_USR1(("spi_tclDeviceSwitcher::bDeviceSwitchTimerCb : Device switch failed.\n"));
   if (NULL != pvObject)
   {
      spi_tclDeviceSwitcher* poDeviceSwitcher = static_cast<spi_tclDeviceSwitcher*>(pvObject);
      if (poDeviceSwitcher != NULL)
      {
         poDeviceSwitcher->m_oDeviceIDForSwitchLock.s16Lock();
         if (poDeviceSwitcher->m_mapDeviceIDForSwitch.end() != poDeviceSwitcher->m_mapDeviceIDForSwitch.find(rTimerID))
         {
            t_U32 u32DeviceID = poDeviceSwitcher->m_mapDeviceIDForSwitch[rTimerID];
            ETG_TRACE_USR4(("rTimerID: %p,Device ID: %d", rTimerID, u32DeviceID));
            poDeviceSwitcher->vOnDeviceSwitchCompleteCb(u32DeviceID, false);
            //! Trigger USB reset for whitelisted mirrorlink device if device switch fails
            //if (true == poDeviceSwitcher->bIsDeviceWhitelisted(u32DeviceID, e8DEV_TYPE_MIRRORLINK))
            if (true == poDeviceSwitcher->bIsDeviceWhitelisted(u32DeviceID))
            {
               poDeviceSwitcher->vDoUSBResetOnSwitchFailure(u32DeviceID);
            }
         }
         else
         {
            ETG_TRACE_USR1(("Timer Id : %p not found in m_mapDeviceIDForSwitch", rTimerID));
         }
         poDeviceSwitcher->m_oDeviceIDForSwitchLock.vUnlock();
      }
   }
   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::vRetryDeviceSwitch
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vRetryDeviceSwitch(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCat)
{
   ETG_TRACE_USR1(("spi_tclDeviceSwitcher::vRetryDeviceSwitch for device 0x%x \n", cou32DeviceHandle));
   t_U32 u32SwitchRetrialErrorCount = 0;
   if (NULL != m_pDiscovereryDataIntf)
   {
      u32SwitchRetrialErrorCount = m_pDiscovereryDataIntf->u32GetSwitchRetrialFailCount(cou32DeviceHandle);
   }

   if (u32SwitchRetrialErrorCount < scou32SwitchFailureRetrialLimit)
   {
      ETG_TRACE_USR1(("spi_tclDeviceSwitcher::vRetryWhitelistDeviceOnUSBReset : Retrying Device Switch"));
      vPrepareDeviceSwitch(enDeviceCat, cou32DeviceHandle, e8USB_CONNECTED);
      bSwitchDeviceMode(enDeviceCat, cou32DeviceHandle, e8_UNKNOWN_DEVICE);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::vDoUSBResetOnSwitchFailure
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vDoUSBResetOnSwitchFailure(const t_U32 cou32DeviceHandle)
{
   //! Trigger USB reset if device switch fails
   ETG_TRACE_USR1(("spi_tclDeviceSwitcher::vDoUSBResetOnSwitchFailure for device 0x%x: \n", cou32DeviceHandle));
   t_U32 u32SwitchRetrialErrorCount = 0;
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vSetSwitchRetrialFailCount(cou32DeviceHandle, true);
      u32SwitchRetrialErrorCount = m_pDiscovereryDataIntf->u32GetSwitchRetrialFailCount(cou32DeviceHandle);
   }

   if (u32SwitchRetrialErrorCount < scou32SwitchFailureRetrialLimit)
   {
      vPrepareDeviceSwitch(e8DEV_TYPE_UNKNOWN, cou32DeviceHandle, e8USB_CONNECTED);
      bSwitchDeviceMode(e8DEV_TYPE_UNKNOWN, cou32DeviceHandle, e8_UNKNOWN_DEVICE);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::vClearSwitchFailureCount
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vClearSwitchFailureCount(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDeviceSwitcher::vClearSwitchFailureCount for device 0x%x: \n", cou32DeviceHandle));
   if (NULL != m_pDiscovereryDataIntf)
   {
      m_pDiscovereryDataIntf->vSetSwitchRetrialFailCount(cou32DeviceHandle, false);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSwitcher::vSetDiscoverersettingsInstance
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclDeviceSwitcher::vSetDiscoverersettingsInstance entered"));
   if (NULL != poDiscovererSettingsIntf)
   {
      ETG_TRACE_USR1(("spi_tclDeviceSwitcher::vSetDiscoverersettingsInstance : Setting m_poDiscovererSettings Value"));
      m_poDiscovererSettings = poDiscovererSettingsIntf;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceSwitcher::vOnSwitchStatusFailed(...)
 ***************************************************************************/
t_Void spi_tclDeviceSwitcher::vOnSwitchStatusFailed(const t_U32 cou32DeviceHandle, t_Bool bSwitchStatus)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSwitcher::vOnSwitchStatusFailed device switch information as the switch is complete with Role switch status as %d\n ", ETG_ENUM(BOOL, bSwitchStatus )));
   vOnDeviceSwitchCompleteCb(cou32DeviceHandle, bSwitchStatus);
}
