/*!
 *******************************************************************************
 * \file             spi_tclDeviceSwitcher.h
 * \brief            Device Profile Switcher
 * \addtogroup       Device Switcher
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Device Profile Switcher
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/
#ifndef SPI_TCLDEVICESWITCHER_H_
#define SPI_TCLDEVICESWITCHER_H_


#include "Lock.h"
#include "spi_tclDeviceSwitcherIntf.h"
#include "spi_tclDeviceSwitchBase.h"
#include "spi_tclDiscoveryDataIntf.h"
#include "spi_tclUSBResetIntf.h"
#include "spi_tclDiscovererSettingsIntf.h"

/**
 * \class spi_tclDeviceSwitcher
 * \brief Device switcher is responsible for switching devices to different connection modes
 *          For example in case of USB, Device switcher is responsible to switch device to AOAP mode,
 *          Mirrorlink mode, Role switch for Carplay, USB reset etc.
 *          Class needs to be extended for Wi-Fi in future
 */

class spi_tclDeviceSwitcher: public spi_tclDeviceSwitcherIntf
{

   public:
      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::spi_tclDeviceSwitcher
       ***************************************************************************/
      /*!
       * \brief   Default Constructor
       **************************************************************************/
      spi_tclDeviceSwitcher(spi_tclDiscoveryDataIntf *poDiscoverIntf,
               std::map<tenDeviceCategory, spi_tclDeviceSwitchBase*> poDeviceSwitchers,
               spi_tclUSBResetIntf *poUSBReset);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::~spi_tclDeviceSwitcher()
       ***************************************************************************/
      /*!
       * \brief   Destructor
       **************************************************************************/
      ~spi_tclDeviceSwitcher();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \brief   Register discoverer callbacks
       * \brief   corfrDisccbs: Structure containing the function pointers to be invoked
       **************************************************************************/
      t_Void vRegisterCallbacks(const trDeviceSwitcherCbs corfrDisccbs);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vPrepareDeviceSwitch
       ***************************************************************************/
      /*!
       * \brief   Initialize and prepare device switch information
       * \param   enDeviceCat: SPI technology to which the device has to be switched to
       * \param   cou32DeviceHandle: Device Handle
       * \param   enDeviceType: Device Type : Android/ Apple etc
	   * \param   enDeviceConnType : Device connection transport
       **************************************************************************/
      t_Void vPrepareDeviceSwitch(tenDeviceCategory enDeviceCat, const t_U32 cou32DeviceHandle, tenDeviceConnectionType enDeviceConnType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::bSwitchDeviceMode
       ***************************************************************************/
      /*!
       * \brief   Switch the connection profile of the device
       * \param   enDeviceCat: SPI technology to which the device has to be switched to
       * \param   cou32DeviceHandle: Device Handle
       * \param   enDeviceType: Device Type : Android/ Apple etc
       **************************************************************************/
      t_Bool bSwitchDeviceMode(tenDeviceCategory enDeviceCat, const t_U32 cou32DeviceHandle,
               tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceSwitcher::vSetHeadUnitInfo
       ***************************************************************************/
      /*!
       * \brief   Sets the Headunit information such as model name, vehicle ID etc.
       *          This function has to be called on load settings so that the underlying discoverers
       *          are aware of this info before performing the switch operations
       * \param   corfrHeadUnitInfo: Reference to the structure containing head unit info
       **************************************************************************/
      t_Void vSetHeadUnitInfo(const trHeadUnitInfo &corfrHeadUnitInfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::bIsSwitchInProgress
       ***************************************************************************/
      /*!
       * \brief   Checks if Device switch is in progress for the specified devicehandle
       * \param   cou32DeviceHandle: Device Handle
       **************************************************************************/
      t_Bool bIsSwitchInProgress(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vResetDeviceSwitchInfo
       ***************************************************************************/
      /*!
       * \brief   Reset device switch information. This method should be invoked once the device switch is complete
       *          i.e when the device reappears in projection mode
       * \param   cou32DeviceHandle: Device Handle
       **************************************************************************/
      t_Void vResetDeviceSwitchInfo(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vGetDeviceSwitchInfo
       ***************************************************************************/
      /*!
       * \brief   Provides device switch information for the device specified by the device handle
       * \param   cou32DeviceHandle: Device Handle
       * \param   [OUT]reference to the device switch information structure
       **************************************************************************/
      t_Void vGetDeviceSwitchInfo(const t_U32 cou32Devicehandle, trDeviceSwitchInfo &rfrDeviceSwitchInfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::bIsDeviceSwitcherBusy
       ***************************************************************************/
      /*!
       * \fn     bIsDeviceSwitcherBusy
       * \brief  Returns true if the Device switcher is busy in a profile switch
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       **************************************************************************/
      t_Bool bIsDeviceSwitcherBusy(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vSetProjectionState
       ***************************************************************************/
      /*!
       * \fn     vSetProjectionState
       * \brief  Set the projection status for a particular device category
       * \param  enProjType : Projection type
       **************************************************************************/
      t_Void vSetProjectionState(tenDeviceCategory enProjType, tenEnabledInfo enEnabledInfo);

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclDeviceSwitcher::vSetEarlyDeviceSwitchInfo
      ***************************************************************************/
      /*!
      * \brief  Interface to set the early device switch info. Indicates if
      *         early switch is requested or not
      * \param  [IN] cou32DeviceID : Device Id to be checked
      * \retval true if the device is same as last connected device otherwise false
      **************************************************************************/
      t_Void vSetEarlyDeviceSwitchInfo(const t_U32 cou32DeviceID, t_Bool bEarlySwitchRequired);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::vOnSwitchStatusFailed
       ***************************************************************************/
      /*!
       * \brief   This method should be invoked once the device switch is failed, i.e, on the RoleSwitchReq from MP
       * \param   cou32DeviceHandle: Device Handle
       **************************************************************************/
      t_Void vOnSwitchStatusFailed(const t_U32 cou32DeviceHandle, t_Bool bSwitchStatus);


   private:

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclDeviceSwitcher::bSwitchtoDefaultMode
       ***************************************************************************/
      /*!
       * \fn      t_U32 bSwitchtoDefaultMode()
       * \brief   Trigger a USB profile switch to re-enumerate the device in default USB Mode
       * \param   rfrDeviceInfo : Contains information of the USB device
       **************************************************************************/
      t_Bool bSwitchtoDefaultMode(const trUSBDeviceInfo &rfrDeviceInfo, const t_U32 cou32DeviceHandle,
               tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vOnDeviceSwitchCompleteCb
       ***************************************************************************/
      /*!
       * \fn     vOnDeviceSwitchCompleteCb
       * \brief  Called when device switch is complete
       * \param  cou32DeviceHandle : Device handle of the disconnected device
       **************************************************************************/
      t_Void vOnDeviceSwitchCompleteCb(const t_U32 cou32DeviceHandle, t_Bool  bSwitchStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::bIsDeviceWhitelisted
       ***************************************************************************/
      /*!
       * \fn     bIsDeviceWhitelisted
       * \brief  Called to check if device with device id is whitelisted
       * \param  cou32DeviceHandle : Device handle of the device
       **************************************************************************/
      t_Bool bIsDeviceWhitelisted(const t_U32 cou32DeviceID);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::bDeviceSwitchTimerCb
       ***************************************************************************/
      /*!
       * \fn     bDeviceSwitchTimerCb
       * \brief  called on expiry of USB Device switch timer
       * \param  rTimerID: ID of the timer which has expired
       * \param  pvObject: pointer to object passed while starting the timer
       * \param  pvUserData: data passed during start of the timer
       **************************************************************************/
      static t_Bool bDeviceSwitchTimerCb(timer_t rTimerID, t_Void *pvObject,
               const t_Void *pvUserData);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vRetryDeviceSwitch
       ***************************************************************************/
      /*!
       * \fn     vRetryDeviceSwitch
       * \brief  Called to retry mirrorlink switch after USB reset has completed
       * \param  cou32DeviceHandle : Device handle of the disconnected device
       **************************************************************************/
      t_Void vRetryDeviceSwitch(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCat);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vDoUSBResetOnSwitchFailure
       ***************************************************************************/
      /*!
       * \fn     vDoUSBResetOnSwitchFailure
       * \brief  Reset USB device if switch fails for whitelisted device
       * \param  cou32DeviceHandle : Device handle of the disconnected device
       **************************************************************************/
      t_Void vDoUSBResetOnSwitchFailure(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vClearSwitchFailureCount
       ***************************************************************************/
      /*!
       * \fn     vClearSwitchFailureCount
       * \brief  Clear switch failure count
       * \param  cou32DeviceHandle : Device handle of the disconnected device
       **************************************************************************/
      t_Void vClearSwitchFailureCount(const t_U32 cou32DeviceHandle);


      /***************************************************************************
       ** FUNCTION:  t_Void  spi_tclDeviceSwitcher::vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
       ***************************************************************************/
       /*!
       * \fn      t_Void vSetDiscoverersettingsInstance()
       * \brief   Method to set discoverer settings instance.
       * \param   poDiscovererSettingsIntf  : pointer to DiscoverersettingsIntf.
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf) override;

      //! Stores the function pointer for device switcher callbacks
      trDeviceSwitcherCbs m_rDeviceSwitcherCbs;

      //! Map to store device switchers
      std::map<tenDeviceCategory, spi_tclDeviceSwitchBase*> m_mapDeviceSwitchers;

      //map to store device IDs against timer IDs.This is to fix the valgrind issue in vPrepareDeviceSwitch
      //Map is used since it is possible that deviceswitcher may be called from two different threads and device switch can be in progress for more than one device.
      std::map<timer_t, t_U32> m_mapDeviceIDForSwitch;

      //to protect m_mapDeviceIDForSwitch.
      Lock m_oDeviceIDForSwitchLock;

      spi_tclDiscoveryDataIntf *m_pDiscovereryDataIntf;

      spi_tclUSBResetIntf *m_pUSBReset;
	  
      //! Lock to handle critical section for device switcher map
      Lock m_oDeviceSwitchersMapLock;

      spi_tclDiscovererSettingsIntf* m_poDiscovererSettings;
};
/*! } */

#endif /* SPI_TCLDEVICESWITCHER_H_ */
