  /*!
 *******************************************************************************
 * \file         spi_tclBluetooth.cpp
 * \brief        Bluetooth Manager class that provides interface to delegate 
                 the execution of command and handle response
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Bluetooth Manager class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 21.02.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 03.03.2014 |  Ramya Murthy (RBEI/ECP2)         | Included BT DeviceConnection and
                                                  SPI Device Deselection implementation.
 08.03.2014 |  Ramya Murthy (RBEI/ECP2)         | Included call handling.
 08.04.2014 |  Ramya Murthy (RBEI/ECP2)         | Removed project variant filters.
 06.04.2014 |  Ramya Murthy (RBEI/ECP2)         | Initialisation sequence implementation
 04.06.2014 |  Ramya Murthy (RBEI/ECP2)         | Device switch cancellation handling changes.
 31.07.2014 |  Ramya Murthy (RBEI/ECP2)         | SPI feature configuration via LoadSettings()
 29.09.2014 |  Ramya Murthy (RBEI/ECP2)         | Fix for GMMY16-16068, to prevent disconnection
                                                  of BT devices when ML device BT Address is empty.
 01.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Removed Telephone client handler
 07.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Implementation for BTPairingRequired property
 08.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Impl. of CarPlay DisableBluetooth msg handling
                                                  and SelectedDevice info access protection
 24.11.2014 |  Ramya Murthy (RBEI/ECP2)         | Implemented BT block/unblock for GM
 28.05.2015 |  Tejaswini H B(RBEI/ECP2)         | Added Lint comments to suppress C++11 Errors
 19.05.2015 |  Ramya Murthy (RBEI/ECP2)         | Added vSendBTDeviceName()
 25.06.2015 |  Tejaswini HB (RBEI/ECP2)         |Featuring out Android Auto
 01.03.2017 |  Shiva Kumar G                    | Changes to enable/disable CarPlay feature   
 11.04.2018 |  Dundamma S B                     | Added vPostTelephoneCallActivityMsg() function
 16.07.2018	|  Roveena Francy Lobo (RBEI/ECO2)  | Added vSetGeneralRestrictions()
 01.12.2018 |  Dundamma S B  ((RBEI/ECO2)       | Added function vRegisterVehicleBTMacAddressCallback() to bind callback trVehicleBTAddressCallback
                                                 

 04.12.2018 |  Dundamma S B (RBEI/ECO2)         | Removing implemetaion of Registering BT callbacks to ExtCmdBluetooth
                                                  as BT clients(Technology specific) are registering callbacks to themselves.

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclFactory.h"
#include "spi_tclMediator.h"
#include "spi_ConnMngrTypeDefines.h"
#include "spi_tclDeviceIDDataIntf.h"
#include "spi_tclBluetoothRespIntf.h"
#include "spi_tclBluetoothDevBase.h"

#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
#include "spi_tclDiPoBluetooth.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
#include "spi_tclAAPBluetooth.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
#include "spi_tclMySPINBluetooth.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
#include "spi_tclBDCLBluetooth.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
#include "spi_tclOnCarBluetooth.h"
#endif

#include "spi_tclBluetooth.h"
#include "spi_tclExtCmdBluetooth.h"
#include "spi_tclExtCmdTelephone.h"
#include "spi_tclExtCompManager.h"

#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BLUETOOTH
      #include "trcGenProj/Header/spi_tclBluetooth.cpp.trc.h"
   #endif
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
std::map<tenBTHandlerIndex, GetBluetooth_t*> Bluetooth;
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

static const t_U16    scu16AAPBTLimModeEnabledLevelInfoBit = 3;
/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
*********************************PUBLIC*************************************
***************************************************************************/

/***************************************************************************
** FUNCTION:  spi_tclBluetooth::spi_tclBluetooth();
***************************************************************************/
spi_tclBluetooth::spi_tclBluetooth(spi_tclBluetoothRespIntf* poResponseIntf)
      :  spi_tclSelectionIntf(e32COMPID_BLUETOOTH),
        m_poBTRespIntf(poResponseIntf)
{
   ETG_TRACE_USR1(("spi_tclBluetooth() entered "));
   SPI_NORMAL_ASSERT(NULL == m_poBTRespIntf);
   //! Initialize the BT handlers
   for (t_U8 u8Index = 0; u8Index < cou8NUM_BT_HANDLERS; ++u8Index)
   {
      m_poBTHandlersLst[u8Index] = NULL;
   } //for (t_U8 u8Index = 0;...)
} //!end of spi_tclBluetooth()

/***************************************************************************
** FUNCTION:  spi_tclBluetooth::~spi_tclBluetooth();
***************************************************************************/
spi_tclBluetooth::~spi_tclBluetooth()
{
   ETG_TRACE_USR1(("~spi_tclBluetooth() entered "));
   for (t_U8 u8Index = 0; u8Index < cou8NUM_BT_HANDLERS; ++u8Index)
   {
      m_poBTHandlersLst[u8Index] = NULL;
   } //for (t_U8 u8Index = 0;...)
   m_poBTRespIntf = NULL;
} //!end of ~spi_tclBluetooth()

/***************************************************************************
** FUNCTION:  t_Bool spi_tclBluetooth::bInitialize()
***************************************************************************/
t_Bool spi_tclBluetooth::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclBluetooth::bInitialize() entered "));

   t_Bool bInit = false;


#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
      ETG_TRACE_USR1(("spi_tclBluetooth::bInitialize Bluetooth[e8_BT_ML_INDEX]=%p",Bluetooth[e8_BT_ML_INDEX]));
      if ((NULL != Bluetooth[e8_BT_ML_INDEX]))
      {
         ETG_TRACE_USR1(("spi_tclAudio::bInitialize before obj create m_poBTHandlersLst[e8_BT_ML_INDEX] = %p ", m_poBTHandlersLst[e8_BT_ML_INDEX]));
         m_poBTHandlersLst[e8_BT_ML_INDEX] = Bluetooth[e8_BT_ML_INDEX](this);
         SPI_NORMAL_ASSERT(NULL == m_poBTHandlersLst[e8_BT_ML_INDEX]);
      }
      ETG_TRACE_USR1(("spi_tclAudio::bInitialize obj success m_poBTHandlersLst[e8_BT_ML_INDEX] = %p ", m_poBTHandlersLst[e8_BT_ML_INDEX]));
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
         m_poBTHandlersLst[e8_BT_DIPO_INDEX] = new spi_tclDiPoBluetooth(this);
         SPI_NORMAL_ASSERT(NULL == m_poBTHandlersLst[e8_BT_DIPO_INDEX]);
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
         m_poBTHandlersLst[e8_BT_AAP_INDEX] = new spi_tclAAPBluetooth(this);
         SPI_NORMAL_ASSERT(NULL == m_poBTHandlersLst[e8_BT_AAP_INDEX]);

#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
         m_poBTHandlersLst[e8_BT_MYSPIN_INDEX] = new (std::nothrow) spi_tclMySPINBluetooth(this);
         SPI_NORMAL_ASSERT(NULL == m_poBTHandlersLst[e8_BT_MYSPIN_INDEX]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
         m_poBTHandlersLst[e8_BT_CARLIFE_INDEX] = new (std::nothrow) spi_tclBDCLBluetooth(this);
         SPI_NORMAL_ASSERT(NULL == m_poBTHandlersLst[e8_BT_CARLIFE_INDEX]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
         m_poBTHandlersLst[e8_BT_ONCAR_INDEX] = new (std::nothrow) spi_tclOnCarBluetooth(this);
         SPI_NORMAL_ASSERT(NULL == m_poBTHandlersLst[e8_BT_ONCAR_INDEX]);
#endif

         //! Initialize the BT handlers
         for (t_U8 u8Index = 0; u8Index < cou8NUM_BT_HANDLERS; ++u8Index)
         {
            if (NULL != m_poBTHandlersLst[u8Index])
            {
               bInit = (m_poBTHandlersLst[u8Index]->bInitialise()) || bInit;
            }
         } //for (t_U8 u8Index = 0;...)

   //@todo - Return value to be revisited along with startup sequence.
   return true;
} //!end of bInitialize()

t_Void spi_tclBluetooth::vRegisterVehicleBTMacAddressCallback()
{
	ETG_TRACE_USR1(("spi_tclBluetooth::vRegisterVehicleBTMacAddressCallback() entered "));

    trVehicleBTAddressCallback rVehicleBTAddressCallback;

    rVehicleBTAddressCallback.fvOnVehicleBTAddress =
             std::bind(&spi_tclBluetooth::vOnVehicleBTAddressCb,
                      this,
                      SPI_FUNC_PLACEHOLDERS_1);

    spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = poFetchExtCmdBluetoothIntf();
    if (NULL != poExtCmdBluetoothIntf)
    {
   	 poExtCmdBluetoothIntf->vRegisterBTMacAddressCallbacks(rVehicleBTAddressCallback);
    }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclBluetooth::bUnInitialize()
***************************************************************************/
t_Bool spi_tclBluetooth::bUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclBluetooth::bUnInitialize() entered "));

   t_Bool bUninitSuccess = true;

   //! Uninitialize & delete the BT handlers
   for (t_U8 u8Index = 0; u8Index < cou8NUM_BT_HANDLERS; ++u8Index)
   {
      if (NULL != m_poBTHandlersLst[u8Index])
      {
         t_Bool bIsSuccess = m_poBTHandlersLst[u8Index]->bUninitialise();
         bUninitSuccess = bUninitSuccess && bIsSuccess;
      }
      RELEASE_MEM(m_poBTHandlersLst[u8Index]);
   } //for (t_U8 u8Index = 0;...)

   ETG_TRACE_USR2(("spi_tclBluetooth::bUnInitialize() left with result = %u ", bUninitSuccess));
   return bUninitSuccess;
} //!end of bUnInitialize()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vLoadSettings()
***************************************************************************/
t_Void spi_tclBluetooth::vLoadSettings()
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vLoadSettings() entered "));
   if(NULL != m_poBluetoothSettings)
   {
      for(t_U8 u8Index = 0; u8Index < cou8NUM_BT_HANDLERS; u8Index++)
      {
         if(NULL != m_poBTHandlersLst[u8Index])
         {
        	 m_poBTHandlersLst[u8Index]->vSetBluetoothSettingsInstance(m_poBluetoothSettings);
         }//if((NULL != m_poBTHandlersLst[
      }//for(t_U8 u8Index=0;u8Index < cou8NUM_BT_HANDLERS; u8Index++)
   }

   vRegisterVehicleBTMacAddressCallback();
   //Add code
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vSaveSettings()
***************************************************************************/
t_Void spi_tclBluetooth::vSaveSettings()
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vSaveSettings() entered "));
   //Add code
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vOnSPISelectDeviceRequest(t_U32...)
***************************************************************************/
t_Void spi_tclBluetooth::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vOnSPISelectDeviceRequest(): "
         "DeviceHandle = 0x%x, DeviceConnectionReq = %d ",
         corfrSelectReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ, corfrSelectReq.m_enDevConnReq)));

   t_Bool bIsReqProcessed = false;
   if (IS_VALID_DEVHANDLE(corfrSelectReq.m_u32DeviceHandle))
   {
      if (e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq)
      {
         //! Get selected device's details
         vClearSelectedDevInfo();
         vFetchSelectedDeviceDetails(corfrSelectReq.m_u32DeviceHandle);
      }

      //! Forward Select/Deselect request to respective BT handler
      tenBTHandlerIndex enBTHndlrIndex = enGetSelDevBTHandlerIndex();
      if ((e8_BT_INDEX_UNKNOWN != enBTHndlrIndex) && (NULL != m_poBTHandlersLst[enBTHndlrIndex]))
      {
         bIsReqProcessed = true;
         if (e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq)
         {
            m_poBTHandlersLst[enBTHndlrIndex]->vOnSPISelectDeviceRequest(corfrSelectReq);
         }
         else
         {
            m_poBTHandlersLst[enBTHndlrIndex]->vOnSPIDeselectDeviceRequest(corfrSelectReq);
         }
      }//if ((e8_BT_INDEX_UNKNOWN != enBTHndlrIndex)&&...)
   } //if (IS_VALID_DEVHANDLE(u32ProjectionDevHandle))
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetooth::vOnSPISelectDeviceRequest: Invalid input! Nothing to be done. "));
   }

   if (false == bIsReqProcessed)
   {
      vSendSelectDeviceResult(true);
   }
} //!end of vOnSPISelectDeviceRequest()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vOnSPISelectDeviceResponse(t_U32...)
***************************************************************************/
t_Void spi_tclBluetooth::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vOnSPISelectDeviceResponse() entered : bIsDeviceSwitch = %d ", ETG_ENUM(BOOL, corfrSelectReq.m_bisSwitchRequired)));
   ETG_TRACE_USR3((" vOnSPISelectDeviceResponse: Received DeviceHandle = 0x%x ", corfrSelectReq.m_u32DeviceHandle));
   ETG_TRACE_USR3((" vOnSPISelectDeviceResponse: Received DevConnectionRequest = %d ",
         ETG_ENUM(CONNECTION_REQ, corfrSelectReq.m_enDevConnReq)));
   ETG_TRACE_USR3((" vOnSPISelectDeviceResponse: Received ErrorCode = %d ", ETG_ENUM(ERROR_CODE, enErrorCode)));

   tenResponseCode enRespCode =(e8NO_ERRORS == enErrorCode)?e8SUCCESS: e8FAILURE ;
   if (
      (IS_VALID_DEVHANDLE(corfrSelectReq.m_u32DeviceHandle))
      &&
      (corfrSelectReq.m_u32DeviceHandle == u32GetSelectedDevHandle()) /*Check if response is for active device*/
      )
   {
      tenBTHandlerIndex enBTHndlrIndex = enGetSelDevBTHandlerIndex();
      if ((e8_BT_INDEX_UNKNOWN != enBTHndlrIndex) && (NULL != m_poBTHandlersLst[enBTHndlrIndex]))
      {
         //! Forward Select/Deselect response to respective BT handler
         if (e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq)
         {
            m_poBTHandlersLst[enBTHndlrIndex]->vOnSPISelectDeviceResponse(corfrSelectReq, enRespCode);
         }
         else
         {
            m_poBTHandlersLst[enBTHndlrIndex]->vOnSPIDeselectDeviceResponse(corfrSelectReq, enRespCode);
         }
      }//if ((e8_BT_INDEX_UNKNOWN != enBTHndlrIndex) &&...)

      //! Clear stored SelectedDevice's details if no device is active.
      if (
         ((e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq) && (e8FAILURE == enRespCode))
         ||
         (e8DEVCONNREQ_DESELECT == corfrSelectReq.m_enDevConnReq)
         )
      {
         vClearSelectedDevInfo();
      }
   }//if ((IS_VALID_DEVHANDLE(u32ProjectionDevHandle))&&...)
   else
   {
      ETG_TRACE_ERR((" vOnSPISelectDeviceResponse: Invalid input/No device active! Nothing to be done. "));
	  //@TODO - to be evaluated again. SelectDeviceResponse failure without receiving SelectDevice request in BT Submodule to be handled.
	  //When the change is made, please revisit the change in DiPOBluetooth.ccp ->vOnSPISelectDeviceResponse methods also
	  if((e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq) && (e8FAILURE == enRespCode) && (e8DEV_TYPE_DIPO == corfrSelectReq.m_enDevCategory))
	  {
		  m_poBTHandlersLst[e8_BT_DIPO_INDEX]->vOnSPISelectDeviceResponse(corfrSelectReq, enRespCode);
	  }
   }
} //!end of vOnSPISelectDeviceResponse()


/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vGetVehicleBTAddress(t_String& ...)
***************************************************************************/
t_Void spi_tclBluetooth::vGetVehicleBTAddress(t_String& rfszVehicleBTAddress)
{
   if (NULL != m_poBluetoothSettings)
   {
	   m_poBluetoothSettings->vGetVehicleBTAddress(rfszVehicleBTAddress);
   }
   ETG_TRACE_USR4(("spi_tclBluetooth::vGetVehicleBTAddress() left with : %s ",
         rfszVehicleBTAddress.c_str()));
} //!end of vGetVehicleBTAddress()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vOnVehicleBTAddressCb(t_String& ...)
***************************************************************************/
t_Void spi_tclBluetooth::vOnVehicleBTAddressCb(t_String rfszVehicleBTAddress)
{
   if (NULL != m_poBluetoothSettings)
   {
	   m_poBluetoothSettings->vOnVehicleBTAddress(rfszVehicleBTAddress);
   }
   ETG_TRACE_USR4(("spi_tclBluetooth::vOnVehicleBTAddressCb() left with : %s ",
         rfszVehicleBTAddress.c_str()));
} //!end of vGetVehicleBTAddress()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vSendSelectDeviceResult(t_Bool...)
***************************************************************************/
t_Void spi_tclBluetooth::vSendSelectDeviceResult(t_Bool bSelectionSuccess)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vSendSelectDeviceResult() entered: SelectionSuccess = %u ",
         ETG_ENUM(BOOL, bSelectionSuccess)));

   if (false == bSelectionSuccess)
   {
      //! Clear all selected device data since Projection device selection is canceled/failed.
      vClearSelectedDevInfo();
   }
   tenErrorCode enErrorCode = (true==bSelectionSuccess)? e8NO_ERRORS : e8SELECTION_FAILED;
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   SPI_NORMAL_ASSERT(NULL == poMediator);
   if (NULL != poMediator)
   {
      poMediator->vPostSelectDeviceRes(e32COMPID_BLUETOOTH, enErrorCode);
   }
} //!end of vSendSelectDeviceResult()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vSendDeselectDeviceRequest(const t_Bool)
***************************************************************************/
t_Void spi_tclBluetooth::vSendDeselectDeviceRequest(const t_Bool &corfbIsUserTrigger)
{
   SPI_INTENTIONALLY_UNUSED(corfbIsUserTrigger);
   ETG_TRACE_USR1(("spi_tclBluetooth::vSendDeselectDeviceRequest() entered "));

   //! Trigger Deselection of Projection device.
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   SPI_NORMAL_ASSERT(NULL == poMediator);
   if (NULL != poMediator)
   {
      t_U32 u32SelDevHandle = u32GetSelectedDevHandle();
      if (IS_VALID_DEVHANDLE(u32SelDevHandle))
      {
         poMediator->vPostAutoDeviceSelection(u32SelDevHandle, e8DEVCONNREQ_DESELECT);
         //! Set the user deselect flag in Conn Manager if it is triggered by user.
         //@TODO: temporary workaround since internal deselection results in cyclic select
         //with automatic selection strategy in GM.
         //if (true == corfbIsUserTrigger)
         //{
        	 //poMediator->vPostSetUserDeselect(u32SelDevHandle);
         //}
      }
      else
      {
         ETG_TRACE_ERR(("spi_tclBluetooth::vSendDeselectDeviceRequest: "
               "No action taken since no Projection device is active! "));
      }
   }//if (NULL != poMediator)
} //!end of vSendDeselectDeviceRequest()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vSendDeviceSwitchEvent(t_U32 ...)
***************************************************************************/
t_Void spi_tclBluetooth::vSendDeviceSwitchEvent(t_U32 u32BluetoothDevHandle,
      t_U32 u32ProjectionDevHandle,
      tenBTChangeInfo enBTChange,
      t_U8 u8Profiles,
      t_Bool bSameDevice,
      t_Bool bCallActive)
{
   ETG_TRACE_USR4(("spi_tclBluetooth::vSendDeviceSwitchEvent() entered: "
         "BluetoothDevHandle = 0x%x, ProjectionDevHandle = 0x%x, "
         "BTChange = %u, SameDevice = %u, CallActive = %u , Active Profiles : [%d] ",
         u32BluetoothDevHandle, u32ProjectionDevHandle,
         ETG_ENUM(BT_CHANGE_INFO, enBTChange),
         ETG_ENUM(BOOL, bSameDevice),
         ETG_ENUM(BOOL, bCallActive),u8Profiles));

   if (NULL != m_poBTRespIntf)
   {

      //! Notify clients about switch between Projection & BT device.
      m_poBTRespIntf->vPostBluetoothDeviceStatus(u32BluetoothDevHandle,
            u32ProjectionDevHandle, enBTChange, u8Profiles, bSameDevice, bCallActive);
      ETG_TRACE_USR4(("spi_tclBluetooth::Informed user about device switch. Awaiting user action. "));
   }
} //!end of vSendDeviceSwitchEvent()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vSendBTDeviceInfo(...)
***************************************************************************/
t_Void spi_tclBluetooth::vSendBTDeviceInfo(t_U32 u32ProjectionDevHandle,
         t_String szBTDeviceName, t_String szBTAddress)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vSendBTDeviceInfo() entered "
            "ProjectionDevHandle = %d, BTDeviceName = %s",
            u32ProjectionDevHandle, szBTDeviceName.c_str()));

   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   SPI_NORMAL_ASSERT(NULL == poMediator);
   if ((NULL != poMediator) && (IS_VALID_DEVHANDLE(u32ProjectionDevHandle)))
   {
      poMediator->vPostBTDeviceInfo(u32ProjectionDevHandle, szBTDeviceName, szBTAddress);
   }
} //!end of vSendBTDeviceInfo()

/***************************************************************************
** FUNCTION:  t_U32 spi_tclBluetooth::u32GetSelectedDevHandle()
***************************************************************************/
t_U32 spi_tclBluetooth::u32GetSelectedDevHandle()
{
   //Return DeviceHandle of selected projection device
   m_SelDevDetailsLock.s16Lock();
   t_U32 u32SelDevHandle = m_rSelDeviceDetails.u32DeviceHandle;
   m_SelDevDetailsLock.vUnlock();
   return u32SelDevHandle;
} //!end of u32GetSelectedDevHandle()

/***************************************************************************
** FUNCTION:  t_String spi_tclBluetooth::szGetSelectedDevBTAddress()
***************************************************************************/
t_String spi_tclBluetooth::szGetSelectedDevBTAddress()
{
   //Return BT Address of selected projection device
   m_SelDevDetailsLock.s16Lock();
   t_String szSelDevBTAddr = m_rSelDeviceDetails.szBTAddress;
   m_SelDevDetailsLock.vUnlock();

   ETG_TRACE_USR4(("spi_tclBluetooth::szGetSelectedDevBTAddress: %s ", szSelDevBTAddr.c_str()));
   return szSelDevBTAddr;
} //!end of szGetSelectedDevBTAddress()

/***************************************************************************
** FUNCTION:  tenDeviceStatus spi_tclBluetooth::enGetSelectedDevStatus()
***************************************************************************/
tenDeviceStatus spi_tclBluetooth::enGetSelectedDevStatus()
{
   //Return DeviceStatus of selected projection device
   m_SelDevDetailsLock.s16Lock();
   tenDeviceStatus enSelDevStatus = m_rSelDeviceDetails.enDeviceStatus;
   m_SelDevDetailsLock.vUnlock();
   return enSelDevStatus;
} //!end of enGetSelectedDevStatus()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vSetSelectedDevStatus()
***************************************************************************/
t_Void spi_tclBluetooth::vSetSelectedDevStatus(tenDeviceStatus enDevStatus)
{
   //Store the DeviceStatus of selected projection device
   m_SelDevDetailsLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBluetooth::vSetSelectedDevStatus(): "
         "current DeviceStatus = %u, new DeviceStatus = %u ",
         ETG_ENUM(DEVICE_STATUS, m_rSelDeviceDetails.enDeviceStatus),
         ETG_ENUM(DEVICE_STATUS, enDevStatus)));
   m_rSelDeviceDetails.enDeviceStatus = enDevStatus;
   m_SelDevDetailsLock.vUnlock();
} //!end of vSetSelectedDevStatus()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vSetSelectedDevBTAddress()
***************************************************************************/
t_Void spi_tclBluetooth::vSetSelectedDevBTAddress(t_String szBTAddress)
{
   //Store the BT Address of selected projection device
   m_SelDevDetailsLock.s16Lock();
   m_rSelDeviceDetails.szBTAddress = szBTAddress;
   m_SelDevDetailsLock.vUnlock();
} //!end of vSetSelectedDevBTAddress()

/***************************************************************************
** FUNCTION:  tenDeviceCategory spi_tclBluetooth::enGetSelectedDeviceCategory()
***************************************************************************/
tenDeviceCategory spi_tclBluetooth::enGetSelectedDeviceCategory()
{
   //Return the selected device's index
   m_SelDevDetailsLock.s16Lock();
   tenDeviceCategory enSelDevCat = m_rSelDeviceDetails.enDeviceCategory;
   m_SelDevDetailsLock.vUnlock();

   return enSelDevCat;
} //!end of enGetSelectedDeviceCategory()

/***************************************************************************
*********************************PRIVATE************************************
***************************************************************************/

/***************************************************************************
** FUNCTION:  tenBTHandlerIndex spi_tclBluetooth::enGetSelDevBTHandlerIndex()
***************************************************************************/
tenBTHandlerIndex spi_tclBluetooth::enGetSelDevBTHandlerIndex()
{
   //Return the selected device's index
   m_SelDevDetailsLock.s16Lock();
   tenDeviceCategory enSelDevCat = m_rSelDeviceDetails.enDeviceCategory;
   m_SelDevDetailsLock.vUnlock();

   tenBTHandlerIndex enBTIndex = e8_BT_INDEX_UNKNOWN;
   switch (enSelDevCat)
   {
      case e8DEV_TYPE_MIRRORLINK:
         enBTIndex = e8_BT_ML_INDEX;
         break;
      case e8DEV_TYPE_DIPO:
         enBTIndex = e8_BT_DIPO_INDEX;
         break;
      case e8DEV_TYPE_ANDROIDAUTO:
         enBTIndex = e8_BT_AAP_INDEX;
         break;
      case e8DEV_TYPE_MYSPIN:
         enBTIndex = e8_BT_MYSPIN_INDEX;
         break;
      case e8DEV_TYPE_CARLIFE:
         enBTIndex = e8_BT_CARLIFE_INDEX;
         break;
      case e8DEV_TYPE_ONCAR:
         enBTIndex = e8_BT_ONCAR_INDEX;
         break;
      default:
         break;
   }//switch (enSelDevCat)
   return enBTIndex;
} //!end of enGetSelDevBTHandlerIndex()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vFetchSelectedDeviceDetails(t_U32...)
***************************************************************************/
t_Void spi_tclBluetooth::vFetchSelectedDeviceDetails(t_U32 u32SelectedDevHandle)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vFetchSelectedDeviceDetails() entered: DeviceHandle = 0x%x ",
         u32SelectedDevHandle));

   m_SelDevDetailsLock.s16Lock();

   //Store device handle
   m_rSelDeviceDetails.u32DeviceHandle = u32SelectedDevHandle;

   //! Get Selected ML/DiPO device's BTAddress from ConnectionManager
   spi_tclFactory* poSPIFactory = spi_tclFactory::getInstance();
   spi_tclConnMngr* poConnMngr = (NULL != poSPIFactory)?
           (poSPIFactory->poGetConnMngrInstance()) : (NULL);
   if (
      (NULL != poConnMngr)
      &&
      (cou32InvalidDeviceHandle != u32SelectedDevHandle)
      )
   {
      //Fetch Device category of projection device
      m_rSelDeviceDetails.enDeviceCategory = poConnMngr->enGetDeviceCategory(u32SelectedDevHandle);

      //Fetch BT address of ML device (AA and CP BT Address is received from phone during session)
      if (e8DEV_TYPE_MIRRORLINK == m_rSelDeviceDetails.enDeviceCategory)
      {
         poConnMngr->vGetBTAddress(u32SelectedDevHandle, m_rSelDeviceDetails.szBTAddress);
      } 

      ETG_TRACE_USR4(("spi_tclBluetooth::vFetchSelectedDeviceDetails: Retrieved DeviceCategory = %d, BTAddress = %s ",
            ETG_ENUM(DEVICE_CATEGORY, m_rSelDeviceDetails.enDeviceCategory),
            m_rSelDeviceDetails.szBTAddress.c_str()));
   } //if (NULL != poSPIFactory)

   m_SelDevDetailsLock.vUnlock();

} //!end of vFetchSelectedDeviceDetails()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vClearSelectedDevInfo()
***************************************************************************/
t_Void spi_tclBluetooth::vClearSelectedDevInfo()
{
   //Set selected device details to defaults
   m_SelDevDetailsLock.s16Lock();
   m_rSelDeviceDetails.u32DeviceHandle  = cou32InvalidDeviceHandle;
   m_rSelDeviceDetails.szBTAddress      = "";
   m_rSelDeviceDetails.enDeviceCategory = e8DEV_TYPE_UNKNOWN;
   m_rSelDeviceDetails.enDeviceStatus   = e8DEVICE_CHANGE_COMPLETE;
   m_SelDevDetailsLock.vUnlock();
} //!end of vClearSelectedDevInfo()


/***************************************************************************
** FUNCTION:  t_Bool spi_tclBluetooth::bSetBTLimitationMode()
***************************************************************************/
t_Bool spi_tclBluetooth::bSetBTLimitationMode(const t_String& rfcszBTAddress, const t_String& rfcszBTDeviceName,tenBTTechnology enTechnology,
                                      tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::bSetBTLimitationMode() entered"));
   t_Bool bRetval = false;
   spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = poFetchExtCmdBluetoothIntf();
   if(NULL != poExtCmdBluetoothIntf)
   {
      bRetval = poExtCmdBluetoothIntf->bSetBTLimitationMode(rfcszBTAddress,rfcszBTDeviceName,enTechnology,enCommChannel,enAction);
      if(false == bRetval)
      {
           vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
           m_rPendingBTLimModeInfo.szBTAddress = rfcszBTAddress;
           m_rPendingBTLimModeInfo.szBTDeviceName = rfcszBTDeviceName;
           m_rPendingBTLimModeInfo.enTechnology = enTechnology;
           m_rPendingBTLimModeInfo.enCommChannel = enCommChannel;
           m_rPendingBTLimModeInfo.enAction = enAction;
      }
   }
   return bRetval;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vTriggerWiFiCredentialExchange(
***************************************************************************/
t_Void spi_tclBluetooth::vTriggerWiFiCredentialExchange(t_U32 u32DeviceID)
{
	ETG_TRACE_USR1(("spi_tclBluetooth::vTriggerWiFiCredentialExchange:DeviceID:%d",u32DeviceID));
	
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
	   //get the BT MAC Address of the device.
	   spi_tclDeviceIDDataIntf oDataIntf;
	   t_String szBTMACAddress;
	   oDataIntf.vGetBTAddress(szBTMACAddress,u32DeviceID);
	   
       poMediator->vPostWiFiCredentials(szBTMACAddress); 
   } //if(NULL != poMediator)
	
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBluetooth::vSetGeneralRestrictions(...)
 ***************************************************************************/
t_Void spi_tclBluetooth::vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory,t_U16 u16GeneralRestrInfo)
{
    //Evaluate the fourth bit to check if BT limitation mode is enabled
    ETG_TRACE_USR1((" spi_tclBluetooth::vSetGeneralRestriction: enDeviceCategory - %d,"
       "General Restriction info = %d",ETG_ENUM(DEVICE_CATEGORY,enDeviceCategory), u16GeneralRestrInfo));
    //Evaluate the 4th bit to check if BT limitation mode is enabled.
    t_Bool bDelayBTPairing = (u16GeneralRestrInfo >> scu16AAPBTLimModeEnabledLevelInfoBit) & 1 ;
    ETG_TRACE_USR1(("spi_tclBluetooth::vSetGeneralRestriction bBTLimModeEnabled - %d  ",
                ETG_ENUM(BOOL, bDelayBTPairing)));
    if ((e8DEV_TYPE_ANDROIDAUTO == enDeviceCategory) && (NULL != m_poBTHandlersLst[e8_BT_AAP_INDEX]))
    {
         m_poBTHandlersLst[e8_BT_AAP_INDEX]->vDelayBTPairing(bDelayBTPairing);
    }
}

/***************************************************************************
 ** FUNCTION:  spi_tclExtCmdBluetoothIntf* spi_tclBluetooth::poFetchExtCmdBluetoothIntf()
 ***************************************************************************/
spi_tclExtCmdBluetoothIntf* spi_tclBluetooth::poFetchExtCmdBluetoothIntf()
{
    spi_tclExtCmdBluetoothIntf* poExtCmdBluetoothIntf = NULL;
    spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
    if(NULL != poExtCompMgr)
    {
        poExtCmdBluetoothIntf = poExtCompMgr->poGetCmdBluetoothIntfInst();
    }
    return poExtCmdBluetoothIntf;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vOnDisableBluetoothTrigger(
***************************************************************************/
t_Void spi_tclBluetooth::vOnDisableBluetoothTrigger(t_String szBTAddress)
{
	ETG_TRACE_USR1(("spi_tclBluetooth:vOnDisableBluetoothTrigger():BTAddress-%s",szBTAddress.c_str()));
	spi_tclMediator* poMediator = spi_tclMediator::getInstance();
	if (NULL != poMediator)
	{
		poMediator->vOnDisableBluetoothTrigger(szBTAddress);
	}
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBluetooth::vSetBluetoothSettingsInstance()
***************************************************************************/
t_Void spi_tclBluetooth::vSetBluetoothSettingsInstance(spi_tclBluetoothSettingsIntf* poBluetoothSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vSetBluetoothSettingsInstance entered"));
   if( NULL != poBluetoothSettingsIntf)
   {
	   ETG_TRACE_USR1(("spi_tclBluetooth::vSetBluetoothSettingsInstance: Setting m_poBluetoothSettings val "));
       m_poBluetoothSettings = poBluetoothSettingsIntf;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBluetooth::vGetBluetoothConfig()
***************************************************************************/
t_Void spi_tclBluetooth::vGetBluetoothConfig()
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vGetBluetoothConfig entered"));
   if( NULL != m_poBluetoothSettings)
   {
      t_String szVehicleBTAddress;
      m_poBluetoothSettings->vGetVehicleBTAddress(szVehicleBTAddress);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclBluetooth::vUpdateSelectionState
 ***************************************************************************/
t_Void spi_tclBluetooth::vUpdateSelectionState(const t_U32 cou32DeviceHandle, tenDeviceSelectionState enDevSelectionState,
 tenDeviceConnectionType enDeviceConnType)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vUpdateSelectionState () Device Handle = 0x%x with selection state =%d having Device connection type is %d",
            cou32DeviceHandle, ETG_ENUM(SELECTION_STATE, enDevSelectionState), ETG_ENUM(CONNECTION_TYPE, enDeviceConnType)));

   trDeviceSelectionStateDetails rDeviceSelectionDetails;
   //! Get the Selection State from the DeviceSelector
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if(NULL != poMediator)
   {
      poMediator->vGetDeviceSelectorCurrentSelectionState(rDeviceSelectionDetails);
   }

   if((e8WIRELESS_CONNECTED == enDeviceConnType) && (0 != cou32DeviceHandle) && (e8_SELECTION_STATE_DEVICE_NOT_SELECTED == enDevSelectionState) &&
            ((e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS != rDeviceSelectionDetails.enDeviceSelectionState) ||
                     (e8_SELECTION_STATE_DEVICE_SELECTED != rDeviceSelectionDetails.enDeviceSelectionState)))
   {
      //get the BT MAC Address of the device.
      spi_tclDeviceIDDataIntf oDataIntf;
      t_String szBTMACAddress;
      oDataIntf.vGetBTAddress(szBTMACAddress,cou32DeviceHandle);
      if (NULL != m_poBTHandlersLst[e8_BT_DIPO_INDEX])
      {
         m_poBTHandlersLst[e8_BT_DIPO_INDEX]->vOnAutoSelectionFailureForWireless(szBTMACAddress, e8BT_TECH_CARPLAY,
                  e8BT_COMM_CHN_WIFI, e8BT_LIM_ACTION_DEACTIVATE);;
      }

   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetooth::vSetBTLimitationModeResult()
***************************************************************************/
t_Void spi_tclBluetooth::vSetBTLimitationModeResult(const trBTLimitationModeInfo corBTLimitationModeInfo)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vSetBTLimitationModeResult entered"));
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vPostBTLimitationModeResult(corBTLimitationModeInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBluetooth::vPostBTServiceAvailabilityMsg()
 ***************************************************************************/
t_Void spi_tclBluetooth::vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable)
{
   ETG_TRACE_USR1(("spi_tclBluetooth::vPostBTServiceAvailabilityMsg entered bServiceAvailable = %d",ETG_ENUM(BOOL,bServiceAvailable)));
   if((true == bServiceAvailable) &&
       (e8DEVICE_CHANGE_DELAYED == enGetSelectedDevStatus()))
   {
       spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = poFetchExtCmdBluetoothIntf();
       if(NULL != poExtCmdBluetoothIntf)
       {
          t_Bool bBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
              m_rPendingBTLimModeInfo.szBTAddress,
              m_rPendingBTLimModeInfo.szBTDeviceName,
              m_rPendingBTLimModeInfo.enTechnology,
              m_rPendingBTLimModeInfo.enCommChannel,
              m_rPendingBTLimModeInfo.enAction);
          if (true == bBTLimModeRequested)
          {
             vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
          }
          else
          {
             vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
          }
          ETG_TRACE_USR4(("[PARAM]::spi_tclBluetooth::vPostBTServiceAvailabilityMsg() - SetLimitationMode result - %d", ETG_ENUM(BOOL, bBTLimModeRequested)));
       }
   }
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclBluetooth::szGetBTDeviceName()
 ***************************************************************************/
t_String spi_tclBluetooth::szGetBTDeviceName()
{
   ETG_TRACE_USR1(("spi_tclBluetooth::szGetBTDeviceName entered"));

   if (0 != m_rSelDeviceDetails.u32DeviceHandle)
   {
      spi_tclDeviceIDDataIntf oDeviceDataIntf;
      oDeviceDataIntf.vGetBTDeviceName(m_rSelDeviceDetails.u32DeviceHandle,m_szFriendlyName);
      ETG_TRACE_USR2(("spi_tclBluetooth::szGetBTDeviceName() szFriendlyName [%s]",m_szFriendlyName.c_str()));
   }
   return m_szFriendlyName;
}
//lint -restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
