/*!
 *******************************************************************************
 * \file             spi_tclBluetoothDevBase.h
 * \brief            Abstract class that specifies the Bluetooth interface which
 *                   must be implemented by device class (Mirror Link/Digital iPod out)
 *                   for handling Bluetooth connections
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Base class for ML and DiPO Interfaces
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 10.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 19.05.2015 |  Ramya Murthy (RBEI/ECP2)         | Added vOnBTDeviceNameUpdate()
 16.07.2018	|  Roveena Francy Lobo (RBEI/ECO2)  | Added bDelayBTPairing()

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#ifndef _SPI_TCLBLUETOOTHDEVBASE_H_
#define _SPI_TCLBLUETOOTHDEVBASE_H_

#include "BaseTypes.h"
#include "spi_BluetoothTypedefs.h"
#include "spi_tclBluetoothIntf.h"
#include "spi_ConnMngrTypeDefines.h"
#include <map>

#include "spi_tclBluetoothSettingsIntf.h"

/*
 * Abstract class that specifies the Bluetooth interface which
 * must be implemented by device class (Mirror Link/Digital iPod out)
 * for handling Bluetooth connections
 */

class spi_tclBluetoothDevBase
{
public:

	/***************************************************************************
	*********************************PUBLIC*************************************
	***************************************************************************/

	/***************************************************************************
    ** FUNCTION:  spi_tclBluetoothDevBase::spi_tclBluetoothDevBase();
    ***************************************************************************/
   /*!
    * \fn      spi_tclBluetoothDevBase()
    * \brief   Default Constructor
    ***************************************************************************/
   spi_tclBluetoothDevBase():m_poBluetoothSettings(NULL)
   {
   }

   /***************************************************************************
    ** FUNCTION:  spi_tclBluetoothDevBase::~spi_tclBluetoothDevBase();
    ***************************************************************************/
   /*!
    * \fn      ~spi_tclBluetoothDevBase()
    * \brief   Virtual Destructor
    ***************************************************************************/
   virtual ~spi_tclBluetoothDevBase()
   {
      m_poBluetoothSettings = NULL;
   }

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclBluetoothDevBase::bInitialise();
    ***************************************************************************/
   /*!
    * \fn      bInitialise()
    * \brief   Method to initialises the service handler. (Performs initialisations which
    *          are not device specific.)
    *          Optional interface to be implemented.
    * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
    * \sa      bUninitialise()
    ***************************************************************************/
   virtual t_Bool bInitialise()
   {
      return true;
   }

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclBluetoothDevBase::bUninitialise();
    ***************************************************************************/
   /*!
    * \fn      bUninitialise()
    * \brief   Method to uninitialise the service handler.
    *          Optional interface to be implemented.
    * \retval  t_Bool: TRUE - If ServiceHandler is uninitialised successfully, else FALSE.
    * \sa      bInitialise()
    ***************************************************************************/
   virtual t_Bool bUninitialise()
   {
      return true;
   }

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothDevBase::vOnSPISelectDeviceRequest(
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
   * \brief   Called when SelectDevice request is received by SPI.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \retval  None
   **************************************************************************/
   virtual t_Void vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothDevBase::vOnSPISelectDeviceResponse(
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
   *             tenResponseCode enRespCode)
   * \brief   Called when SelectDevice operation is complete, with the result
   *          of the operation. 
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \param   [IN] enRespCode: Response code enumeration
   * \retval  None
   **************************************************************************/
   virtual t_Void vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothDevBase::vOnSPISelectDeviceRequest(
   ***************************************************************************/
   /*!
   * \fn      vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
   * \brief   Called when DeselectDevice request is received by SPI.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \retval  None
   **************************************************************************/
   virtual t_Void vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothDevBase::vOnSPIDeselectDeviceResponse(
   ***************************************************************************/
   /*!
   * \fn      vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
   *             tenResponseCode enRespCode)
   * \brief   Called when DeselectDevice operation is complete, with the result
   *          of the operation.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \param   [IN] enRespCode: Response code enumeration
   * \retval  None
   **************************************************************************/
   virtual  t_Void vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode) = 0;

   /***************************************************************************
    ** FUNCTION:  spi_tclBluetoothDevBase::vDelayBTPairing
    ***************************************************************************/
    /*!
    * \fn     vDelayBTPairing
    * \brief  This method checks for if BT pairing should be delayed or not
    *  depending on the FI general restriction bit.
    * \param  void
    **************************************************************************/
   virtual t_Void vDelayBTPairing(t_Bool /*bDelayBTPairing*/)
   {

   }

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothDevBase::vOnBTProfileStatus(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to receive notification when there is a update of BT profile status.
   * \retval  None
   **************************************************************************/
   virtual t_Void vOnBTProfileStatus()
   {
   }
   
   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBluetoothDevBase::vSetBluetoothSettingsInstance(spi_tclBluetoothSettingsIntf* poBluetoothSettingsIntf)
    ***************************************************************************/
    /*!
    * \fn      t_Void vSetBluetoothSettingsInstance()
    * \brief   Method to set Bluetooth settings instance.
    * \param   poBluetoothSettingsIntf  : pointer to BluetoothSettingsIntf.
    * \retval  t_Void
    **************************************************************************/
   t_Void vSetBluetoothSettingsInstance(spi_tclBluetoothSettingsIntf* poBluetoothSettingsIntf)
   {
      if( NULL != poBluetoothSettingsIntf)
      {
         m_poBluetoothSettings = poBluetoothSettingsIntf;
      }
   }
   
   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBluetoothDevBase::vOnAutoSelectionFailureForWireless()
   ***************************************************************************/
   /*!
   * \fn      vOnAutoSelectionFailureForWireless()
   * \brief   Interface to set a Bluetooth limitation in HU.
   * \param   [IN] rfcszBTAddress: BT address of device
   * \param   [IN] enTechnology: Technology for which BT limitation mode is required
   * \param   [IN] enCommChannel: Communication channel of the device
   * \param   [IN] enAction: Type of action for which BT limitation mode is required
   * \retval  t_Bool: True - if request to set limitation mode is sent successfully, false
   **************************************************************************/
   virtual t_Void vOnAutoSelectionFailureForWireless(const t_String& /*rfcszBTAddress*/, tenBTTechnology /*enTechnology*/,
                                tenBTCommunicationChannel /*enCommChannel*/, tenBTLimitationAction /*enAction*/){}
   
   /**************************************************************************
   ****************************END OF PUBLIC**********************************
   **************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/

   spi_tclBluetoothSettingsIntf* m_poBluetoothSettings;

   /***************************************************************************
   ******************************END OF PROTECTED******************************
   ***************************************************************************/


private:
   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ******************************END OF PRIVATE********************************
   ***************************************************************************/

};

typedef spi_tclBluetoothDevBase* GetBluetooth_t(spi_tclBluetoothIntf*);
extern std::map<tenBTHandlerIndex,GetBluetooth_t*> Bluetooth;

#endif // _SPI_TCLBLUETOOTHDEVBASE_H_

