/*!
 *******************************************************************************
 * \file             spi_tclBDCLVideo.cpp
 * \brief            Baidu Carlife Video class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Baidu Carlife Video class
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 21.04.2017  | Ramya Murthy          | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "BDCLTypes.h"
#include "spi_tclVideoTypedefs.h"
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLVideo.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_VIDEO
      #include "trcGenProj/Header/spi_tclBDCLVideo.cpp.trc.h"
   #endif
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
static timer_t srSDKVersionCheckTimerId = 0;
static t_Bool bSDKVersionCheckTimerRunning = false;
static const t_U32 cou32SDKVersionCheckTimerVal = 5000;
static const t_U16 scou16minAndroidSdkVersionforBDCL = 21;
/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLVideo::spi_tclBDCLVideo()
 ***************************************************************************/
spi_tclBDCLVideo::spi_tclBDCLVideo():m_u32SelectedDeviceID(0),
m_bMDInfoMessageReceived(false),
m_u32AndroidSdkVersion(0),
m_SelectionInProgress(false),
m_u32SelectedDevId(0)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo() entered"));

   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   if (spoBDCLManager)
   {
      m_spoCmdVideo = spoBDCLManager->spoGetCmdVideoInstance();

      //! Register with BDCL manager for Video callbacks
      spoBDCLManager->bRegisterObject((spi_tclBDCLRespVideo*)this);
      spoBDCLManager->bRegisterObject((spi_tclBDCLRespSession*) this);
   }
   SPI_NORMAL_ASSERT((!spoBDCLManager) || (!m_spoCmdVideo));
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLVideo::~spi_tclBDCLVideo()
 ***************************************************************************/
spi_tclBDCLVideo::~spi_tclBDCLVideo()
{
   ETG_TRACE_USR1(("~spi_tclBDCLVideo() entered"));
   m_u32SelectedDeviceID = 0;
   m_bMDInfoMessageReceived = false;
   m_SelectionInProgress = false;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclBDCLVideo::bInitialize()
 ***************************************************************************/
t_Bool spi_tclBDCLVideo::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::bInitialize() entered"));
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideo::vUninitialize()
 ***************************************************************************/
t_Void spi_tclBDCLVideo::vUninitialize()
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::vUninitialize() entered"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideo::vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLVideo::vRegisterCallbacks(
         const trVideoCallbacks& corfrVideoCallbacks)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::vRegisterCallbacks() entered"));
   m_rVideoCallbacks = corfrVideoCallbacks;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideo::vCarlifeSupportCheck()
 ***************************************************************************/
t_Void spi_tclBDCLVideo::vCarlifeSupportCheck(const t_U32 & cou32DevId)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::vCarlifeSupportCheck() entered"));
   //scou16minAndroidSdkVersionforBDCL <= m_u32AndroidSdkVersion is for android devices with OS > 5.0 and 0 == m_u32AndroidSdkVersion is for iphones
   if( (scou16minAndroidSdkVersionforBDCL <= m_u32AndroidSdkVersion) || (0 == m_u32AndroidSdkVersion))
   {
      ETG_TRACE_USR2(("spi_tclBDCLVideo::vCarlifeSupportCheck() Carlife is supported"));
      vProceedWithSelection(cou32DevId);
   }else
   {
      ETG_TRACE_USR2(("spi_tclBDCLVideo::vCarlifeSupportCheck() Carlife is not supported"));
      if(NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
      {         
         (m_rVideoCallbacks.fpvSelectDeviceCb)(cou32DevId, e8FATAL_ERROR);
      } //if(NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideo::vSelectDevice()
 ***************************************************************************/
t_Void spi_tclBDCLVideo::vSelectDevice(const t_U32 cou32DevId,
      const tenDeviceConnectionReq coenConnReq)
{
   /*lint -esym(40,fpvSelectDeviceCb) fpvSelectDeviceCb is not declared */
   ETG_TRACE_USR1(("spi_tclBDCLVideo:vSelectDevice: Device ID = 0x%x Selection Type = %d ",
            cou32DevId, ETG_ENUM(CONNECTION_REQ, coenConnReq)));

   if ((e8DEVCONNREQ_SELECT == coenConnReq)&&(m_spoCmdVideo))
   {
      m_u32SelectedDevId = cou32DevId;
      m_SelectionInProgress = true;
      if(true == m_bMDInfoMessageReceived)
      {
          vCarlifeSupportCheck(cou32DevId);
      }else
      {
         vStartSDKVersionCheckTimer();
      }
   }//if( coenConnReq == e8DEVCONNREQ_SELECT )
   else
   {
      m_u32SelectedDevId = 0;
      m_SelectionInProgress = false;
      m_bMDInfoMessageReceived = false;
      vUninitializeVideo(cou32DevId);
      if (NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
      {
         (m_rVideoCallbacks.fpvSelectDeviceCb)(cou32DevId, e8NO_ERRORS);
      }//if(NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
   }//else
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideo::vOnSelectDeviceResult()
***************************************************************************/
t_Void spi_tclBDCLVideo::vOnSelectDeviceResult(const t_U32 cou32DevId,
         const tenDeviceConnectionReq coenConnReq, const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo:vOnSelectDeviceResult: "
         "Device ID = 0x%x, Device Selection Type = %d, Response Code = %d",
         cou32DevId, ETG_ENUM(CONNECTION_REQ, coenConnReq), ETG_ENUM(RESPONSE_CODE,coenRespCode)));
   m_SelectionInProgress = false;
   m_bMDInfoMessageReceived = false;
   //Device selection is failed. Clear the video resources,if already allocated
   if ((e8DEVCONNREQ_SELECT == coenConnReq)&&(e8FAILURE == coenRespCode))
   {
      vUninitializeVideo(m_u32SelectedDeviceID);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclBDCLVideo::bLaunchVideo()
***************************************************************************/
t_Bool spi_tclBDCLVideo::bLaunchVideo(const t_U32 cou32DevId,
                                     const t_U32 cou32AppId,
                                     const tenEnabledInfo coenSelection)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::bLaunchVideo:Device ID = 0x%x AppID = 0x%x",
      cou32DevId,cou32AppId));
   SPI_INTENTIONALLY_UNUSED(coenSelection);

   return true;
}

/***************************************************************************
 ** FUNCTION:  t_U32  spi_tclBDCLVideo::vStartVideoRendering()
 ***************************************************************************/
t_Void spi_tclBDCLVideo::vStartVideoRendering(t_Bool bStartVideoRendering)
{
	/*lint -esym(40,fpvVideoRenderStatusCb) fpvVideoRenderStatusCb identifier*/

   // Send response to the set request
   if (NULL != m_rVideoCallbacks.fpvVideoRenderStatusCb)
   {
      (m_rVideoCallbacks.fpvVideoRenderStatusCb)(bStartVideoRendering, e8DEV_TYPE_CARLIFE);
   }//if(NULL != m_rVideoCallbacks.fpvVideoRenderStatusCb )
}

/***************************************************************************
** FUNCTION: t_Void spi_tclBDCLVideo::vGetVideoSettings()
***************************************************************************/
t_Void spi_tclBDCLVideo::vGetVideoSettings(const t_U32 cou32DevId,
         trVideoAttributes& rfrVideoAttributes)
{
   SPI_INTENTIONALLY_UNUSED(cou32DevId);
   SPI_INTENTIONALLY_UNUSED(rfrVideoAttributes);
   //unused
}

/***************************************************************************
** FUNCTION: t_Void spi_tclBDCLVideo::vSetOrientationMode()
***************************************************************************/
t_Void spi_tclBDCLVideo::vSetOrientationMode(const t_U32 cou32DevId,
                                            const tenOrientationMode coenOrientationMode,
                                            const trUserContext& corfrUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::vSetOrientationMode"));
   SPI_INTENTIONALLY_UNUSED(coenOrientationMode);

   /*lint -esym(40,fpvSetOrientationModeCb) fpvSetOrientationModeCb Undeclared identifier */
   if(NULL != m_rVideoCallbacks.fpvSetOrientationModeCb)
   {
      (m_rVideoCallbacks.fpvSetOrientationModeCb)(cou32DevId,
            e8UNSUPPORTED_OPERATION, corfrUsrCntxt, e8DEV_TYPE_CARLIFE);
   }// if(NULL != m_rVideoCallbacks.fpvCbSetVideoBlockingMode)
}

/***************************************************************************
 ** FUNCTION:  t_Void vOnVideoEncoderInitDone()
 ***************************************************************************/
t_Void spi_tclBDCLVideo::vOnVideoEncoderInitDone(t_U32 u32Width,
         t_U32 u32Height, t_U32 u32Framerate)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo:vOnVideoEncoderInitDone entered: Width=%d, Height=%d, Framerate=%d",
            u32Width, u32Height, u32Framerate));
   //Note @ Select device response is sent inside VOnSelectDevice and not inside vOnVideoEncoderInitDone.
   //If we send select device response inside vOnVideoEncoderInitDone, Selection thread will be
   //blocked. If we receive NO_DEVICE error during this time, deselection cannot be triggered
   //Which will lead to many other issues. Refer ticket NCG3D-105792.
   /*if ( (NULL != m_rVideoCallbacks.fpvSelectDeviceCb) && (0 != m_u32SelectedDeviceID))
   {
      (m_rVideoCallbacks.fpvSelectDeviceCb)(m_u32SelectedDeviceID, e8NO_ERRORS);
   }*/

}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclBDCLVideo::bInitializeVideo()
***************************************************************************/
t_Bool spi_tclBDCLVideo::bInitializeVideo(t_U32 u32DevID)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::bInitVideoSession: Device ID = 0x%x", u32DevID));
 
   t_Bool bResult = false;
 
   if (0 != u32DevID)
   {
      //Uninitialize video session for the earlier connected device, in case if the clean up was not performed.
      if (0 != m_u32SelectedDeviceID)
      {
         vUninitializeVideo(m_u32SelectedDeviceID);
      }

      m_u32SelectedDeviceID = u32DevID;

      if ((NULL != m_poVideoSettings) && (NULL != m_spoCmdVideo))
      {
         trVideoConfigData rVideoConfigData;
         tvecVideoConfigList vecVideoConfigList;
         m_poVideoSettings->vGetVideoConfigData(e8DEV_TYPE_CARLIFE, vecVideoConfigList);
         m_poVideoSettings->vGetPrimaryDisplayConfiguration(vecVideoConfigList,rVideoConfigData);

         trBdclVideoConfig rVideoConfig;
         rVideoConfig.u32ScreenWidth = rVideoConfigData.u32Screen_Width;
         rVideoConfig.u32ScreenHeight = rVideoConfigData.u32Screen_Height;
         rVideoConfig.u16LayerID = rVideoConfigData.u32LayerId;
         rVideoConfig.u16SurfaceID = rVideoConfigData.u32SurfaceId;
         rVideoConfig.u8Fps = m_poVideoSettings->u8GetFramesPerSec();
         rVideoConfig.u16ProjScreenWidth = (t_U16) rVideoConfigData.u32ProjScreen_Width;
         rVideoConfig.u16ProjScreenHeight = (t_U16) rVideoConfigData.u32ProjScreen_Height;

         ETG_TRACE_USR4(("bInitVideoSession: u32ScreenWidth = %d, u32ScreenHeight = %d, "
            "u16LayerID = %d, u16SurfaceID = %d, u8Fps = %d, "
            "u16ProjScreenWidth = %d, u16ProjScreenHeight = %d, ", rVideoConfig.u32ScreenWidth,
            rVideoConfig.u32ScreenHeight, rVideoConfig.u16LayerID, rVideoConfig.u16SurfaceID,
            rVideoConfig.u8Fps, rVideoConfig.u16ProjScreenWidth, rVideoConfig.u16ProjScreenHeight));

         //set app settings pointer
         m_spoCmdVideo->vSetVideoSettingsInstance(m_poVideoSettings);
         bResult = m_spoCmdVideo->bInitialize(rVideoConfig);
      }//if ((NULL != m_poVideoSettings) &&...)
   }//if (0 != u32DevID)

   ETG_TRACE_USR4(("spi_tclBDCLVideo::bInitVideoSession left with %d", ETG_ENUM(BOOL, bResult)));

   return bResult;
}


/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideo::vUninitializeVideo()
***************************************************************************/
t_Void spi_tclBDCLVideo::vUninitializeVideo(t_U32 u32DevID)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::vUnInitVideoSession:Device ID = 0x%x", u32DevID));

   m_u32SelectedDeviceID = 0;
   if (m_spoCmdVideo)
   {
      m_spoCmdVideo->vUninitialize();
   }
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideo::vOnMDInfo()
 ***************************************************************************/
t_Void spi_tclBDCLVideo::vOnMDInfo(const trBdclMDInfo& crfrMDInfo)
{
   ETG_TRACE_USR2(("spi_tclBDCLVideo::vOnMDInfo Android SDK version is = %d ", crfrMDInfo.u32Sdk));
   m_bMDInfoMessageReceived = true;
   m_u32AndroidSdkVersion = crfrMDInfo.u32Sdk;
   if( true == bSDKVersionCheckTimerRunning)
   {
      ETG_TRACE_USR1(("spi_tclBDCLVideo::vOnMDInfo SDKVersionCheckTimer is running "));
      vStopSDKVersionCheckTimer();
      vCarlifeSupportCheck(m_u32SelectedDevId);
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideo::vOnCoreError()
 ***************************************************************************/
t_Void spi_tclBDCLVideo::vOnCoreError(tenBdclCarLifeError enCarlifeError)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::vOnCoreError entered: %d", ETG_ENUM(BDCL_ERROR, enCarlifeError)));
   if ((e8CL_AOAP_NO_DEVICE_ERROR == enCarlifeError) && (true == m_SelectionInProgress))
   {
      ETG_TRACE_USR2(("[DESC]:: AOAP_NO_DEVICE_ERROR: Sending select device response with no device error "));
      vStopSDKVersionCheckTimer();
      if (NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
      {
         (m_rVideoCallbacks.fpvSelectDeviceCb)(m_u32SelectedDevId, e8DEVICE_DISCONNECTED_DURING_SELECTION);
      }
   }// if (e8CL_AOAP_NO_DEVICE_ERROR == enCarlifeError)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideo::vProceedWithSelection
 ***************************************************************************/
t_Void spi_tclBDCLVideo::vProceedWithSelection(const t_U32 cou32DevId)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideo::vProceedWithSelection entered "));
   if (NULL != m_rVideoCallbacks.fpvUpdateSelectionProgressState)
   {
      (m_rVideoCallbacks.fpvUpdateSelectionProgressState)(cou32DevId,e8_SELECTION_PROGRESS_STATE_RESOURCE_INITIALIZATION);
   }//if(NULL != m_rVideoCallbacks.fpvUpdateSelectionProgressState)

   //Initialize Video session
   tenErrorCode enErrCode = (bInitializeVideo(cou32DevId)) ? e8NO_ERRORS : e8UNKNOWN_ERROR;
   if ((e8UNKNOWN_ERROR != enErrCode)&& (NULL != m_spoCmdVideo) )
   {
      m_spoCmdVideo->vPrepareVideo();
   }
   if (NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
   {
      (m_rVideoCallbacks.fpvSelectDeviceCb)(cou32DevId, enErrCode);
   }//if(NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideo::vStartSDKVersionMatchTimer()
***************************************************************************/
t_Void spi_tclBDCLVideo::vStartSDKVersionCheckTimer ()
{
   m_oSDKVersionCheckLock.s16Lock();
   ETG_TRACE_USR4(("spi_tclBDCLVideo::vStartSDKVersionMatchTimer entered: Timer running status : %d",
            ETG_ENUM(BOOL, bSDKVersionCheckTimerRunning)));

   Timer* poTimer = Timer::getInstance();
   if((NULL != poTimer) && (false == bSDKVersionCheckTimerRunning))
   {
      poTimer->StartTimer(srSDKVersionCheckTimerId, cou32SDKVersionCheckTimerVal, 0, this,
         &spi_tclBDCLVideo::bSDKVersionCheckTimerCb, NULL);

      ETG_TRACE_USR4(("[DESC]:vStartSDKVersionMatch Timer started"));

      bSDKVersionCheckTimerRunning = true;
   }//End of if(NULL != poTimer)

   ETG_TRACE_USR1(("spi_tclBDCLVideo::vStartSDKVersionMatchTimer left"));
   m_oSDKVersionCheckLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideo::vStopSDKVersionMatchTimer ()
***************************************************************************/
t_Void spi_tclBDCLVideo::vStopSDKVersionCheckTimer ()
{
   m_oSDKVersionCheckLock.s16Lock();
   ETG_TRACE_USR4(("spi_tclBDCLVideo::vStopSDKVersionMatchTimer  entered: Timer running status : %d",
            ETG_ENUM(BOOL, bSDKVersionCheckTimerRunning)));

   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (true == bSDKVersionCheckTimerRunning))
   {
      poTimer->CancelTimer(srSDKVersionCheckTimerId);
      ETG_TRACE_USR4(("[DESC]:SDK version check Timer Stopped"));

      bSDKVersionCheckTimerRunning = false;
   }//End of if (NULL != poTimer)...

   ETG_TRACE_USR1(("spi_tclBDCLVideo::vStopSDKVersionMatchTimer  left"));
   m_oSDKVersionCheckLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclBDCLVideo::bSDKVersionCheckTimerCb()
***************************************************************************/
t_Bool spi_tclBDCLVideo::bSDKVersionCheckTimerCb (timer_t rTimerID,
         t_Void *pvObject, const t_Void *pvUserData)
{
   ETG_TRACE_USR4(("spi_tclBDCLVideo::bSDKVersionCheckTimerCb entered: Timer running status : %d",
            ETG_ENUM(BOOL, bSDKVersionCheckTimerRunning)));
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);

   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      poTimer->CancelTimer(srSDKVersionCheckTimerId);
      ETG_TRACE_USR4(("SDKVersionCheck Timer expired"));

      bSDKVersionCheckTimerRunning = false;
   }//if (NULL != poTimer)

   spi_tclBDCLVideo* poBDCLVideo = static_cast<spi_tclBDCLVideo*>(pvObject);
   if (NULL != poBDCLVideo)
   {
      ETG_TRACE_USR4(("SDKVersionCheck Timer expired. Hence sending select devcie response as unknown error."));
      if ((NULL != poBDCLVideo->m_rVideoCallbacks.fpvSelectDeviceCb) && ( true == poBDCLVideo->m_SelectionInProgress) && ( 0 != poBDCLVideo->m_u32SelectedDevId ))
      {
         (poBDCLVideo->m_rVideoCallbacks.fpvSelectDeviceCb)(poBDCLVideo->m_u32SelectedDevId, e8UNKNOWN_ERROR);
      }//if(NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
   }
   return true;
}
//lint restore

///////////////////////////////////////////////////////////////////////////////
// <EOF>
