/*!
 *******************************************************************************
 * \file             spi_tclBDCLResourceMngr.cpp
 * \brief            Resource Manager for Baidu Carlife
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Resource Manager for Baidu Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author              | Modifications
 04.05.2017 |  Ramya Murthy        | Initial Version
 06.03.2018 |  Ramya Murthy        | Added notification of Session error on transport error

 \endverbatim
 ******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "spi_tclResorceMngrDefines.h"
#include "spi_tclBDCLAudioResourceMngr.h"
#include "spi_tclBDCLVideoResourceMngr.h"
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLResourceMngr.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
      #include "trcGenProj/Header/spi_tclBDCLResourceMngr.cpp.trc.h"
   #endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	
/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
static t_U32 su32CurSelectedDevId = 0;
static tenNavAppState senCurAccNavAppState = e8SPI_NAV_NOT_ACTIVE;

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
** FUNCTION:  spi_tclBDCLResourceMngr::spi_tclBDCLResourceMngr()
***************************************************************************/
spi_tclBDCLResourceMngr::spi_tclBDCLResourceMngr() :
         m_poVideoRsrcMngr(NULL), m_poAudioRsrcMngr(NULL)
{
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr entered "));

   m_poVideoRsrcMngr = new spi_tclBDCLVideoResourceMngr();
   m_poAudioRsrcMngr = new spi_tclBDCLAudioResourceMngr();

   //! Register with BDCL manager for Session & Audio callbacks
   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   SPI_NORMAL_ASSERT(!spoBDCLManager);
   if (spoBDCLManager)
   {
      spoBDCLManager->bRegisterObject((spi_tclBDCLRespSession*)this);
      m_spoCmdSession = spoBDCLManager->spoGetCmdSessionInstance();
   }

   m_rDeviceAppStates.enSpeechAppState = e8SPI_SPEECH_UNKNOWN;
   m_rDeviceAppStates.enPhoneAppState  = e8SPI_PHONE_UNKNOWN;
   m_rDeviceAppStates.enNavAppState    = e8SPI_NAV_UNKNOWN;
}

/***************************************************************************
** FUNCTION:  spi_tclBDCLResourceMngr::~spi_tclBDCLResourceMngr()
***************************************************************************/
spi_tclBDCLResourceMngr::~spi_tclBDCLResourceMngr()
{
   ETG_TRACE_USR1(("~spi_tclBDCLResourceMngr entered"));

   RELEASE_MEM(m_poVideoRsrcMngr);
   RELEASE_MEM(m_poAudioRsrcMngr);

   m_rDeviceAppStates.enSpeechAppState = e8SPI_SPEECH_UNKNOWN;
   m_rDeviceAppStates.enPhoneAppState  = e8SPI_PHONE_UNKNOWN;
   m_rDeviceAppStates.enNavAppState    = e8SPI_NAV_UNKNOWN;
}

/***************************************************************************
** FUNCTION:  t_Void  spi_tclBDCLResourceMngr::vRegRsrcMngrCallBack()
***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vRegRsrcMngrCallBack(
         trRsrcMngrCallback rRsrcMngrCallback)
{
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::vRegRsrcMngrCallBack entered"));
   m_rRsrcMngrCb = rRsrcMngrCallback;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclResourceMngrBase::bInitialize()
***************************************************************************/
t_Bool spi_tclBDCLResourceMngr::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::bInitialize entered"));

   t_Bool bRet = false;
   if (NULL != m_poVideoRsrcMngr)
   {
      bRet = m_poVideoRsrcMngr->bInitialize();
   }//if(NULL != m_poVideoRsrcMngr)

   if (NULL != m_poAudioRsrcMngr)
   {
      trBdclResMngrCbs rAudioResMngrCbs;
      rAudioResMngrCbs.fvBdclVRState =
      std::bind(&spi_tclBDCLResourceMngr::vOnVRStateUpdate,
                this,SPI_FUNC_PLACEHOLDERS_1);
      m_poAudioRsrcMngr->vRegisterCallbacks(rAudioResMngrCbs);
   }
   if (NULL != m_poVideoRsrcMngr)
   {
      trBdclResMngrCbs rVideoResMngrCbs;
      rVideoResMngrCbs.fvBdclSessionState =
      std::bind(&spi_tclBDCLResourceMngr::vOnSessionState,
                this,SPI_FUNC_PLACEHOLDERS_1);
      rVideoResMngrCbs.fvUpdateDeviceDisplayCntxt =
      std::bind(&spi_tclBDCLResourceMngr::vUpdateDeviceDisplayCntxt,
                this,SPI_FUNC_PLACEHOLDERS_2);
      m_poVideoRsrcMngr->vRegisterCallbacks(rVideoResMngrCbs);
   }
   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vUninitialize()
***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vUnInitialize()
{
   if (NULL != m_poVideoRsrcMngr)
   {
      m_poVideoRsrcMngr->vUnInitialize();
   }//if(NULL != m_poVideoRsrcMngr)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vSelectDevice()
***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vSelectDevice(const t_U32 cou32DevId,
         tenDeviceConnectionReq enConnReq, tenDeviceCategory enDevCat)
{
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::vSelectDevice Device id- %d, Connection request - %d, Device"
      "Category - %d ", cou32DevId, ETG_ENUM(CONNECTION_REQ, enConnReq), ETG_ENUM(DEVICE_CATEGORY, enDevCat)));

   t_Bool bResult = true;

   if (e8DEVCONNREQ_SELECT == enConnReq)
   {
      su32CurSelectedDevId = cou32DevId;
      ETG_TRACE_USR4(("[PARM]::spi_tclBDCLResourceMngr::vSelectDevice: Device ID = %d ", su32CurSelectedDevId));
   }
   if (e8DEVCONNREQ_DESELECT == enConnReq)
   {
      if (NULL != m_poAudioRsrcMngr)
      {
         m_poAudioRsrcMngr->vOnDeviceDeselection();
      }
      if (NULL != m_poVideoRsrcMngr)
      {
         m_poVideoRsrcMngr->vOnDeviceDeselection();
      }

      vOnSessionState(e8_SESSION_INACTIVE);

      if ((NULL != m_rRsrcMngrCb.fvPostDeviceAppState) && (NULL != m_rRsrcMngrCb.fvSetDeviceAppState))
      {
         m_rDeviceAppStates.enSpeechAppState = e8SPI_SPEECH_END;
         m_rDeviceAppStates.enPhoneAppState = e8SPI_PHONE_NOT_ACTIVE;
         m_rDeviceAppStates.enNavAppState = e8SPI_NAV_NOT_ACTIVE;
         vAcquireDevAppStateLock();
         m_rRsrcMngrCb.fvSetDeviceAppState(m_rDeviceAppStates.enSpeechAppState,
                  m_rDeviceAppStates.enPhoneAppState,
                  m_rDeviceAppStates.enNavAppState);
         vReleaseDevAppStateLock();
         m_rRsrcMngrCb.fvPostDeviceAppState(m_rDeviceAppStates.enSpeechAppState,
                  m_rDeviceAppStates.enPhoneAppState,
                  m_rDeviceAppStates.enNavAppState,
                  corEmptyUsrContext);
      }
   }
   if (NULL != m_rRsrcMngrCb.fvSelectDeviceResult)
   {
      m_rRsrcMngrCb.fvSelectDeviceResult(bResult);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclBDCLResourceMngr::vOnSPISelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vOnSPISelectDeviceResult(t_U32 u32DevID,
         tenDeviceConnectionReq enDeviceConnReq, tenResponseCode enRespCode,
         tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::vOnSPISelectDeviceResult entered"));

   su32CurSelectedDevId = ((e8DEVCONNREQ_SELECT == enDeviceConnReq)
            && (e8SUCCESS == enRespCode)) ? u32DevID : 0;

   if (NULL != m_poVideoRsrcMngr)
   {
      m_poVideoRsrcMngr->vOnSPISelectDeviceResult(u32DevID,
            enDeviceConnReq, enRespCode, enErrorCode);
   }//if(NULL != poVideoRsrcMngr)

   /*if (NULL != m_poAudioRsrcMngr)
   {
      m_poAudioRsrcMngr->vOnSPISelectDeviceResult(u32DevID,
            enDeviceConnReq, enRespCode, enErrorCode);
   }*/


}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vSetAccessoryDisplayContext()
***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vSetAccessoryDisplayContext(
         const t_U32 cou32DevId, t_Bool bDisplayFlag,
         tenDisplayContext enDisplayContext, const trUserContext& rfrcUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::vSetAccessoryDisplayContext entered"));
   SPI_INTENTIONALLY_UNUSED(rfrcUsrCntxt);

   if (NULL != m_poVideoRsrcMngr)
   {
      m_poVideoRsrcMngr->vSetAccessoryDisplayContext(cou32DevId,
               bDisplayFlag, enDisplayContext);
   }//if(NULL != poVideoRsrcMngr)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclBDCLResourceMngr::vSetAccessoryDisplayMode(t_U32...
***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vSetAccessoryDisplayMode(const t_U32 cou32DeviceHandle,
        const trDisplayContext corDisplayContext,
        const trDisplayConstraint corDisplayConstraint,
        const tenDisplayInfo coenDisplayInfo)
{
   if (NULL != m_poVideoRsrcMngr)
   {
      m_poVideoRsrcMngr->vSetAccessoryDisplayMode(cou32DeviceHandle,
         corDisplayContext, corDisplayConstraint, coenDisplayInfo);
   }//if(NULL != poVideoRsrcMngr)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vSetAccessoryAudioContext()
***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vSetAccessoryAudioContext(
         const t_U32 cou32DevId, const tenAudioContext coenAudioCntxt,
         t_Bool bReqFlag, const trUserContext& rfrcUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::vSetAccessoryAudioContext entered"));
   SPI_INTENTIONALLY_UNUSED(rfrcUsrCntxt);

   if (NULL != m_poAudioRsrcMngr)
   {
      m_poAudioRsrcMngr->vSetAccessoryAudioContext(cou32DevId,
               coenAudioCntxt, bReqFlag);
   }//if (NULL != m_poAudioRsrcMngr)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vSetAccessoryAppState()
***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vSetAccessoryAppState(
         const tenSpeechAppState enAccSpeechAppState,
         const tenPhoneAppState enAccPhoneAppState,
         const tenNavAppState enAccNavAppState,
         const trUserContext& rfrcUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::vSetAccessoryAppState: NavAppState - Current %d, Updated %d ",
            ETG_ENUM(NAV_APP_STATE, senCurAccNavAppState), ETG_ENUM(NAV_APP_STATE, enAccNavAppState)));
   SPI_INTENTIONALLY_UNUSED(enAccSpeechAppState);
   SPI_INTENTIONALLY_UNUSED(enAccPhoneAppState);
   SPI_INTENTIONALLY_UNUSED(rfrcUsrCntxt);

   if ((0 != su32CurSelectedDevId) &&
            ((enAccNavAppState != senCurAccNavAppState) && (e8SPI_NAV_ACTIVE == enAccNavAppState)))
   {
      if ((NULL != m_spoCmdSession) && (NULL != m_rRsrcMngrCb.fvPostDeviceAppState) &&
               (NULL != m_rRsrcMngrCb.fvSetDeviceAppState))
      {
         ETG_TRACE_USR2(("spi_tclBDCLResourceMngr::vSetAccessoryAppState: NavAppState - changed"));

         tenBdclNaviState enNaviModuleStatus = (e8SPI_NAV_ACTIVE == enAccNavAppState)
                  ? (e8CL_NAVI_STATUS_IDLE) : (e8CL_NAVI_STATUS_START);

         m_spoCmdSession->vSetNavigationState(enNaviModuleStatus);

         //!Use Case:Device Navi Active. User initiates Carlife Navigation.
         //!Update the new APP state to HMI
         m_rDeviceAppStates.enNavAppState = (e8CL_NAVI_STATUS_IDLE == enNaviModuleStatus) ? e8SPI_NAV_NOT_ACTIVE:e8SPI_NAV_ACTIVE;
         vAcquireDevAppStateLock();
         m_rRsrcMngrCb.fvSetDeviceAppState(m_rDeviceAppStates.enSpeechAppState,
                  m_rDeviceAppStates.enPhoneAppState, m_rDeviceAppStates.enNavAppState);
         vReleaseDevAppStateLock();

         m_rRsrcMngrCb.fvPostDeviceAppState(m_rDeviceAppStates.enSpeechAppState,
                  m_rDeviceAppStates.enPhoneAppState, m_rDeviceAppStates.enNavAppState, corEmptyUsrContext);
      }//End of if (NULL != poCmdSession)
      }//End of if if ((NULL != poAAPManager) && (0 != su32CurSelectedDevId)...


   //!Store current accessory Navigation app state
   senCurAccNavAppState = enAccNavAppState;
 }

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vUpdateDeviceDisplayCntxt()
***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vUpdateDeviceDisplayCntxt(t_Bool bDisplayFlag,
         tenDisplayContextReason enDisplayContextReason)
{
   /*lint -esym(40,fvPostDeviceDisplayContext) fvPostDeviceDisplayContext is not declared */
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::vUpdateDeviceDisplayCntxt: bDisplayFlag-%d",
            ETG_ENUM(BOOL, bDisplayFlag)));

   if (NULL != m_rRsrcMngrCb.fvPostDeviceDisplayContext)
   {
      (m_rRsrcMngrCb.fvPostDeviceDisplayContext)(bDisplayFlag, enDisplayContextReason,
               tenDiPOEntity::e8DIPO_ENTITY_NA, corEmptyUsrContext);
   }//if(NULL != m_rRsrcMngrCb.fvPostDeviceDisplayContext)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vRequestProjUI()
***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vRequestProjUI(t_U32 u32DevId,t_U32 u32AppId)
{
   if (NULL != m_poVideoRsrcMngr)
   {
      m_poVideoRsrcMngr->vRequestProjUI(u32DevId,u32AppId);
   }//if(NULL != poVideoRsrcMngr)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vOnModuleStatusInfo()
 ***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vOnModuleStatusInfo(
         const tBdclModuleStatusList& corfStatusList)
{
	ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::vOnModuleStatusInfo entered"));

	//! Module status should first be sent to audio manager to handle audio playback since
	//! notifying app states to HMI introduces delay
	if (NULL != m_poAudioRsrcMngr)
	{
	   m_poAudioRsrcMngr->vOnModuleStatusInfo(corfStatusList);
	}

	t_Bool bPhoneAppStateChanged = false;
	t_Bool bNaviAppStateChanged = false;
	for (auto itr = corfStatusList.begin(); itr != corfStatusList.end(); ++itr)
	{
		tenBdclCmdModuleID enModuleID = static_cast<tenBdclCmdModuleID>(itr->u32ModuleID);
		switch (enModuleID)
		{
		   case e8CL_PHONE_MODULE_ID:
		   {
		      tenBdclPhoneState enPhoneModuleStatus = static_cast<tenBdclPhoneState>(itr->u32StatusID);
		      ETG_TRACE_USR4(("[PARAM]::Phone module state = %d", ETG_ENUM(BDCL_PHONE_STATE, enPhoneModuleStatus)));
		      //!Phone App state has to be updated to HMI.
		      tenPhoneAppState enNewPhoneAppState =
		               (e8CL_PHONE_STATUS_IDLE == enPhoneModuleStatus)? (e8SPI_PHONE_NOT_ACTIVE) : (e8SPI_PHONE_ACTIVE);
		      bPhoneAppStateChanged = (enNewPhoneAppState != m_rDeviceAppStates.enPhoneAppState);
		      m_rDeviceAppStates.enPhoneAppState = enNewPhoneAppState;
		   }
		      break;
		   case e8CL_NAVI_MODULE_ID:
		   {
		      tenBdclNaviState enNaviModuleStatus = static_cast<tenBdclNaviState>(itr->u32StatusID);
		      ETG_TRACE_USR4(("[PARAM]::Navigation module state = %d", ETG_ENUM(BDCL_NAV_STATE, enNaviModuleStatus)));
        	   //!Navigation App state has to be updated to HMI.
		      tenNavAppState enNewNaviAppState =
		               (e8CL_NAVI_STATUS_IDLE == enNaviModuleStatus)?e8SPI_NAV_NOT_ACTIVE:e8SPI_NAV_ACTIVE;
		      bNaviAppStateChanged = (enNewNaviAppState!= m_rDeviceAppStates.enNavAppState);
		      m_rDeviceAppStates.enNavAppState = enNewNaviAppState;
		   }
		      break;
		} //switch (enModuleID)
   } //for (auto itr = corfStatusList.begin();...)

	if((true == bPhoneAppStateChanged) || (true == bNaviAppStateChanged ))
	{
	   vAcquireDevAppStateLock();
	   m_rRsrcMngrCb.fvSetDeviceAppState(m_rDeviceAppStates.enSpeechAppState,
	            m_rDeviceAppStates.enPhoneAppState, m_rDeviceAppStates.enNavAppState);
	   vReleaseDevAppStateLock();
	   m_rRsrcMngrCb.fvPostDeviceAppState(m_rDeviceAppStates.enSpeechAppState,
	            m_rDeviceAppStates.enPhoneAppState, m_rDeviceAppStates.enNavAppState, corEmptyUsrContext);
	}
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vOnVRStateUpdate()
 ***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vOnVRStateUpdate(tenBdclVRState enBdclVRState)
{
   ETG_TRACE_USR2(("spi_tclBDCLResourceMngr::vOnVRStateUpdate entered: %d ", enBdclVRState));
   if (e8CL_VR_STATUS_NOT_SUPPORT != enBdclVRState )
   {
      tenSpeechAppState enSpeechAppState =(e8CL_VR_STATUS_RECORD_IDLE == enBdclVRState)?
               e8SPI_SPEECH_END:e8SPI_SPEECH_SPEAKING;
      //fix for NCG3D-90185. Workaround is done to overcome the limitation of BDCL
      //When VR is activated BDCL does not requests video focus
      if((m_rDeviceAppStates.enSpeechAppState != e8SPI_SPEECH_SPEAKING) && (enSpeechAppState == e8SPI_SPEECH_SPEAKING))
      {
         vRequestProjUI(su32CurSelectedDevId,0);
      }
      m_rDeviceAppStates.enSpeechAppState = enSpeechAppState;
      vAcquireDevAppStateLock();
      m_rRsrcMngrCb.fvSetDeviceAppState(m_rDeviceAppStates.enSpeechAppState,
                     m_rDeviceAppStates.enPhoneAppState, m_rDeviceAppStates.enNavAppState);
      vReleaseDevAppStateLock();
       m_rRsrcMngrCb.fvPostDeviceAppState(m_rDeviceAppStates.enSpeechAppState,
                     m_rDeviceAppStates.enPhoneAppState, m_rDeviceAppStates.enNavAppState, corEmptyUsrContext);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLResourceMngr::vOnSessionState()
 ***************************************************************************/
t_Void spi_tclBDCLResourceMngr::vOnSessionState(tenSessionStatus enSessionStatus)
{
   ETG_TRACE_USR1(("spi_tclBDCLResourceMngr::vOnSessionState entered: %d ", ETG_ENUM(SESSION_STATUS, enSessionStatus)));

   /*lint -esym(40,fvUpdateSessionStatus)fvUpdateSessionStatus Undeclared identifier */
   if (NULL != m_rRsrcMngrCb.fvUpdateSessionStatus)
   {
      (m_rRsrcMngrCb.fvUpdateSessionStatus)(su32CurSelectedDevId, e8DEV_TYPE_CARLIFE, enSessionStatus);
   }
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
