/*!
 *******************************************************************************
 * \file             spi_tclBDCLAudioResourceMngr.h
 * \brief            Audio Resource Manager for Baidu Carlife
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Resource Manager for Baidu Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author              | Modifications
 04.05.2017 |  Ramya Murthy        | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLBDCLAUDIORESOURCEMNGR_H_
#define _SPI_TCLBDCLAUDIORESOURCEMNGR_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "BDCLTypes.h"
#include "SPITypes.h"
#include "spi_tclBDCLRespAudio.h"
#include "Lock.h"

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

class spi_tclBDCLAudioResourceMngr:
      public spi_tclBDCLRespAudio
{

public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLAudioResourceMngr()
   ***************************************************************************/
   /*!
   * \fn      spi_tclBDCLAudioResourceMngr()
   * \brief   Default Constructor
   * \param   t_Void
   * \sa      ~spi_tclBDCLAudioResourceMngr()
   **************************************************************************/
   spi_tclBDCLAudioResourceMngr();

   /***************************************************************************
   ** FUNCTION:  ~spi_tclBDCLAudioResourceMngr()
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclBDCLAudioResourceMngr()
   * \brief   Destructor
   * \param   t_Void
   * \sa      spi_tclBDCLAudioResourceMngr()
   **************************************************************************/
   ~spi_tclBDCLAudioResourceMngr();

   /***************************************************************************
   ** FUNCTION:  t_Void vSetAccessoryAudioContext(...)
   ***************************************************************************/
   /*!
   * \fn     vSetAccessoryAudioContext(const t_U32 cou32DevId, const t_U8 cu8AudioCntxt
   *           t_Bool bDisplayFlag)
   * \brief   To send accessory audio context related info .
   * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
   * \param   cu8AudioCntxt : [IN] Audio Context
   * \pram    bReqFlag : [IN] Request/ Release flag
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetAccessoryAudioContext(const t_U32 cou32DevId,
         const tenAudioContext coenAudioCntxt,
         t_Bool bReqFlag);

   /***************************************************************************
    ** FUNCTION:  t_Void vOnModuleStatusInfo()
    ***************************************************************************/
   /*!
    * \fn      vOnModuleStatusInfo
    * \brief   Triggered when BDCL device updates status of one or modules on the device
    * \param   corfStatusList [IN]: Structure containing status information of
    *             various logical modules in BDCL device
    * \retval  t_Void
    **************************************************************************/
   t_Void vOnModuleStatusInfo(const tBdclModuleStatusList& corfStatusList);

   /***************************************************************************
    ** FUNCTION:  t_Void vOnVRModuleStatus(...)
    ***************************************************************************/
   /*!
    * \fn      vOnVRModuleStatus
    * \brief   Triggered when MD VR state changes
    * \param   enVRModuleState: [IN] VR state enumeration
    * \retval  t_Void
    **************************************************************************/
   t_Void vOnVRModuleStatus(tenBdclVRState enVRModuleState);

   /***************************************************************************
    ** FUNCTION:  t_Void vOnAudioRequest(...)
    ***************************************************************************/
   /*!
    * \fn      vOnAudioRequest
    * \brief   Triggered when BDCL device starts audio streaming
    * \param   crfrAudioRequest: [IN] Structure containing audio request info
    * \retval  t_Void
    **************************************************************************/
   t_Void vOnAudioRequest(const trBdclAudioRequest& crfrAudioRequest);

   /***************************************************************************
    ** FUNCTION:  t_Void vRegisterCallbacks(...)
    ***************************************************************************/
   /*!
    * \fn     vRegisterCallbacks()
    * \brief  interface for the creator class to register for the required
    *         callbacks.
    * \param  corfrResMngrCbs : reference to the callback structure
    *         populated by the caller
    **************************************************************************/
   t_Void vRegisterCallbacks(const trBdclResMngrCbs& corfrResMngrCbs);

   /***************************************************************************
    ** FUNCTION:  t_Void vOnDeviceDeselection()
    ***************************************************************************/
   /*!
    * \fn      vOnDeviceDeselection()
    * \brief   Triggered on start of Device deselection
    * \retval  t_Void
    **************************************************************************/
   t_Void vOnDeviceDeselection();

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLAudioResourceMngr(const spi_tclBDCLAudioResourceMngr...
   ***************************************************************************/
   /*!
   * \fn      spi_tclBDCLAudioResourceMngr(const spi_tclBDCLAudioResourceMngr& corfoSrc)
   * \brief   Copy constructor - Do not allow the creation of copy constructor
   * \param   corfoSrc : [IN] reference to source data interface object
   * \retval
   * \sa      spi_tclBDCLAudioResourceMngr()
   ***************************************************************************/
   spi_tclBDCLAudioResourceMngr(const spi_tclBDCLAudioResourceMngr& corfoSrc);

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLAudioResourceMngr& operator=( const spi_tclBDCL...
   ***************************************************************************/
   /*!
   * \fn      spi_tclBDCLAudioResourceMngr& operator=(
   *                          const spi_tclBDCLAudioResourceMngr& corfoSrc))
   * \brief   Assignment operator
   * \param   corfoSrc : [IN] reference to source data interface object
   * \retval
   * \sa      spi_tclBDCLAudioResourceMngr(const spi_tclBDCLAudioResourceMngr& otrSrc)
   ***************************************************************************/
   spi_tclBDCLAudioResourceMngr& operator=(const spi_tclBDCLAudioResourceMngr& corfoSrc);


   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  t_Void vValidateModuleState()
   ***************************************************************************/
   /*!
    * \fn      vValidateModuleState
    * \brief   Determines whether audio playback is feasible based on current audio
    *            states and prepares audio channel if required
    * \param   enAudStream: [IN] Audio stream type to be prepared
    * \retval  t_Void
    **************************************************************************/
   t_Void vValidateModuleState(tenBdclAudStreamType enAudStream, t_Bool bModuleActive);

   /***************************************************************************
   ** FUNCTION:  t_Void vProcessAudioStartRequest()
   ***************************************************************************/
   /*!
    * \fn      vProcessAudioStartRequest
    * \brief   Determines whether audio playback request can be accepted and
    *             notifies CL device to stop audio if the request cannot be processed.
    * \param   crfrAudioRequest: [IN] Structure containing audio request info
    * \retval  t_Void
    **************************************************************************/
   t_Void vProcessAudioStartRequest(const trBdclAudioRequest& crfrAudioRequest);

   /***************************************************************************
   ** FUNCTION:  t_Void vProcessAudioStopRequest()
   ***************************************************************************/
   /*!
    * \fn      vProcessAudioStopRequest
    * \brief   Handles audio playback stop request
    * \param   crfrAudioRequest: [IN] Structure containing audio request info
    * \retval  t_Void
    **************************************************************************/
   t_Void vProcessAudioStopRequest(const trBdclAudioRequest& crfrAudioRequest);

   /***************************************************************************
   ** FUNCTION:  t_Void vProcessAudioInterruptRequest()
   ***************************************************************************/
   /*!
    * \fn      vProcessAudioInterruptRequest
    * \brief   Handles audio interrupt request (currently only for VR)
    * \param   crfrAudioRequest: [IN] Structure containing audio request info
    * \retval  t_Void
    **************************************************************************/
   t_Void vProcessAudioInterruptRequest(const trBdclAudioRequest& crfrAudioRequest);

   /***************************************************************************
   ** FUNCTION:  t_Void vDenyAudioRequest()
   ***************************************************************************/
   /*!
    * \fn      vDenyAudioRequest
    * \brief   Informs CL device to stop the requested audio playback
    * \param   crfrAudioRequest: [IN] Structure containing audio request info
    * \retval  t_Void
    **************************************************************************/
   t_Void vDenyAudioRequest(const trBdclAudioRequest& crfrAudioRequest);

   /***************************************************************************
   ** FUNCTION:  trBdclAudioStates enGetCurAudioStates()
   ***************************************************************************/
   /*!
   * \fn      enGetCurAudioStates()
   * \brief   Function to read the current Audio state of different streams
   * \param   None
   * \retval  trBdclAudioStates: Current audio states
   **************************************************************************/
   trBdclAudioStates enGetCurAudioStates() const;

   /***************************************************************************
   ** FUNCTION:  t_Void vSetCurAudioStates()
   ***************************************************************************/
   /*!
   * \fn      vSetCurAudioStates()
   * \brief   Function to set the current Audio state of different streams
   * \param   rAudStates : [IN] New audio states
   * \retval  None
   **************************************************************************/
   t_Void vSetCurAudioStates(trBdclAudioStates rAudStates,
         t_Bool bSetByAccessory);

   /***************************************************************************
   ** FUNCTION:  t_Void vSetCurAudioStates()
   ***************************************************************************/
   /*!
   * \fn      vSetCurAudioStates()
   * \brief   Function to set the current Audio state of different streams
   * \param   rAudStates : [IN] New audio states
   * \retval  None
   **************************************************************************/
   t_Void vSetCurAudioStates(trBdclAudioStates rAudStates,
         t_Bool bSetByAccessory, tenBdclAudStreamType enChangedAudStream);

   /***************************************************************************
   ** FUNCTION:  tenBdclAudioState enGetCurAudioState()
   ***************************************************************************/
   tenBdclAudioState enGetCurAudioState(tenBdclAudStreamType enAudStream) const;

   /***************************************************************************
   ** FUNCTION:  t_Void vSetCurAudioStates()
   ***************************************************************************/
   t_Void vSetCurAudioState(tenBdclAudStreamType enAudStream,
         tenBdclAudioState enAudState,
         t_Bool bSetByAccessory);

   /***************************************************************************
   ** FUNCTION:  tenAudioContextType enGetCurMuteState()
   ***************************************************************************/
   /*!
   * \fn      enGetCurMuteState()
   * \brief   Function to read current Mute state
   * \param   None
   * \retval  tenAudioMuteState: Mute state
   **************************************************************************/
   tenAudioMuteState enGetCurMuteState() const;

   /***************************************************************************
   ** FUNCTION:  t_Void vSetCurMuteState()
   ***************************************************************************/
   /*!
   * \fn      vSetCurMuteState()
   * \brief   Function to store current Mute state
   * \param   enMuteState : [IN] New Mute state
   * \retval  None
   **************************************************************************/
   t_Void vSetCurMuteState(tenAudioMuteState enMuteState);

   /***************************************************************************
   ** FUNCTION:  tenAudioContextType enGetCurTakeConstraint()
   ***************************************************************************/
   /*!
   * \fn      enGetCurTakeConstraint()
   * \brief   Function to read current Take constraint
   * \param   None
   * \retval  tenBdclAudioConstraint: Take constraint
   **************************************************************************/
   tenBdclAudioConstraint enGetCurTakeConstraint() const;

   /***************************************************************************
   ** FUNCTION:  t_Void vSetCurTakeConstraint()
   ***************************************************************************/
   /*!
   * \fn      vSetCurTakeConstraint()
   * \brief   Function to store current Take constraint
   * \param   enAudConstraint : [IN] New Take constraint
   * \retval  None
   **************************************************************************/
   t_Void vSetCurTakeConstraint(tenBdclAudioConstraint enAudConstraint);

   /***************************************************************************
   ** FUNCTION:  tenAudioContextType enGetCurBorrowConstraint()
   ***************************************************************************/
   /*!
   * \fn      enGetCurBorrowConstraint()
   * \brief   Function to read current Borrow constraint
   * \param   None
   * \retval  tenBdclAudioConstraint: Borrow constraint
   **************************************************************************/
   tenBdclAudioConstraint enGetCurBorrowConstraint() const;

   /***************************************************************************
   ** FUNCTION:  t_Void vSetCurBorrowConstraint()
   ***************************************************************************/
   /*!
   * \fn      vSetCurBorrowConstraint()
   * \brief   Function to store current Borrow constraint
   * \param   enAudConstraint : [IN] New Borrow constraint
   * \retval  None
   **************************************************************************/
   t_Void vSetCurBorrowConstraint(tenBdclAudioConstraint enAudConstraint);

   /***************************************************************************
   ** FUNCTION:  tenAudioContextType enGetCurMixConstraint()
   ***************************************************************************/
   /*!
   * \fn      enGetCurMixConstraint()
   * \brief   Function to read current Mix constraint
   * \param   None
   * \retval  tenBdclAudioConstraint: Mix constraint
   **************************************************************************/
   tenBdclAudioConstraint enGetCurMixConstraint() const;

   /***************************************************************************
   ** FUNCTION:  t_Void vSetCurMixConstraint()
   ***************************************************************************/
   /*!
   * \fn      vSetCurMixConstraint()
   * \brief   Function to store current Mix constraint
   * \param   enAudConstraint : [IN] New Mix constraint
   * \retval  None
   **************************************************************************/
   t_Void vSetCurMixConstraint(tenBdclAudioConstraint enAudConstraint);

   /***************************************************************************
    ** FUNCTION:  t_Void vPostVRStatus()
    ***************************************************************************/
   /*!
    * \fn      vPostVRStatus
    * \brief   Interface to notify different VR  state changes
    * \param   enBdclVRState: [IN] VR states.
    * \retval  t_Void
    **************************************************************************/
    t_Void vPostVRStatus(tenBdclVRState enBdclVRState);

    /***************************************************************************
     ** FUNCTION:  t_Void vSendAudioRequest(...)
     ***************************************************************************/
    /*!
     * \fn      vSendAudioRequest
     * \brief   Sends audio request to Audio handler
     * \param   crfrAudioRequest: [IN] Structure containing audio request info
     * \retval  t_Void
     **************************************************************************/
    t_Void vSendAudioRequest(const trBdclAudioRequest& crfrAudioRequest);

   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/

   Lock  m_oLock;

   //! Stores current evaluated audio states
   std::map<tenBdclAudStreamType, tenBdclAudioState>  m_DevAudStates;

   tenAudioMuteState       m_enMuteState;
   tenBdclAudioConstraint  m_enTakeConstraint;
   tenBdclAudioConstraint  m_enBorrowConstraint;
   tenBdclAudioConstraint  m_enMixConstraint;

   //! Stores latest audio context and flag received in vSetAccessoryAudioContext()
   tenAudioContext         m_enCurAudioCntxt;
   t_Bool                  m_bCurCtxtFlag;

   //! Stores latest module statuses
   tenBdclMusicState       m_enMusicModuleStatus;
   tenBdclVRState          m_enVRModuleStatus;
   tenBdclPhoneState       m_enPhoneModuleStatus;

   Lock  m_oModuleStatesLock;

   //! Stores latest device audio stream states
   std::map <tenBdclAudStreamType, tenBdclStreamState>   m_mapAudStreamState;

   trBdclResMngrCbs        m_rResMngrCbs;

}; //class spi_tclBDCLAudioResourceMngr

#endif //_SPI_TCLBDCLAUDIORESOURCEMNGR_H_

///////////////////////////////////////////////////////////////////////////////
// <EOF>
