/*!
 *******************************************************************************
 * \file             spi_tclBDCLDataService.h
 * \brief            Implements the data service functionality for Baidu Carlife
                     using interface to CL Wrapper.
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Data service implementation for Baidu Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author              | Modifications
 04.05.2017 |  Ramya Murthy        | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLBDCLDATASERVICE_H_
#define SPI_TCLBDCLDATASERVICE_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include<memory>
#include <algorithm>

#include "SPITypes.h"
#include "spi_tclDataServiceDevBase.h"
#include "spi_tclDataServiceTypes.h"
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLRespCarData.h"

#include "spi_tclBDCLRespMediaData.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

struct trBDCLSelectedDeviceInfo
{
   t_U32 u32DeviceHandle;

   trBDCLSelectedDeviceInfo():
      u32DeviceHandle(0)
   {
   }
};

/* Forward declarations */


/*!
 * \class spi_tclBDCLDataService
 * \brief BDCL Connection class implements Data Service Info Management for
 *        BDCL capable devices. This class must be derived from base
 *        Data Service class.
 */

class spi_tclBDCLDataService : public spi_tclDataServiceDevBase,public spi_tclBDCLRespCarData
{
public:
   /***************************************************************************
    *********************************PUBLIC*************************************
    ***************************************************************************/

   /***************************************************************************
    ** FUNCTION:  spi_tclBDCLDataService::spi_tclBDCLDataService(const trDataServiceCb...))
    ***************************************************************************/
   /*!
    * \fn     spi_tclBDCLDataService(const trDataServiceCb& rfcorDataServiceCb)
    * \brief  Parameterised Constructor
    * \param  rfcorDataServiceCb: [IN] Structure containing callbacks to
    *            DataService Manager.
    * \sa     ~spi_tclBDCLDataService()
    **************************************************************************/
   spi_tclBDCLDataService(const trDataServiceCb& rfcorDataServiceCb);

   /***************************************************************************
    ** FUNCTION:  spi_tclBDCLDataService::~spi_tclBDCLDataService
    ***************************************************************************/
   /*!
    * \fn     ~spi_tclBDCLDataService()
    * \brief  Destructor
    * \sa     spi_tclBDCLDataService()
    **************************************************************************/
   virtual ~spi_tclBDCLDataService();


   /***** Start of Methods overridden from spi_tclDataServiceDevBase *********/

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBDCLDataService::bInitialise();
   ***************************************************************************/
   /*!
   * \fn      bInitialise()
   * \brief   Method to initialise the service handler. (Performs initialisations which
   *          are not device specific.)
   *          Mandatory interface to be implemented.
   * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
   * \sa      bUninitialise()
   ***************************************************************************/
   virtual t_Bool bInitialise();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBDCLDataService::bUninitialise();
   ***************************************************************************/
   /*!
   * \fn      bUninitialise()
   * \brief   Method to uninitialise the service handler.
   *          Mandatory interface to be implemented.
   * \retval  t_Bool: TRUE - If ServiceHandler is uninitialised successfully, else FALSE.
   * \sa      bInitialise()
   ***************************************************************************/
   virtual t_Bool bUninitialise();

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBDCLDataService::vSelectDevice(t_U32...)
    ***************************************************************************/
   /*!
    * \fn      vSelectDevice
    * \brief   Called when a device is selection request is received
    * \param   [IN] cou32DevId: Unique handle of selected device
    * \param   [IN] enConnReq: Indicated the category of the device
    * \retval  None
    **************************************************************************/
   virtual t_Void vSelectDevice(const t_U32 cou32DevId,
         tenDeviceConnectionReq enConnReq);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDataServiceDevBase::vOnSelectDeviceResult(t_U32...)
    ***************************************************************************/
   /*!
    * \fn      vOnSelectDeviceResult(t_U32 u32DeviceHandle)
    * \brief   Called when a device is selected.
    *          Mandatory interface to be implemented.
    * \param   [IN] u32DeviceHandle: Unique handle of selected device
    * \param   coenConnReq : [IN] Identifies the Connection Request.
    * \param   coenRespCode: [IN] Response code. Success/Failure
    * \retval  None
    **************************************************************************/
   virtual t_Void vOnSelectDeviceResult(t_U32 u32DeviceHandle,
            tenDeviceConnectionReq coenConnReq,
            tenResponseCode coenRespCode);


   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnDeselectDeviceResult(t_U32...)
    ***************************************************************************/
   /*!
    * \fn      vOnDeselectDeviceResult(t_U32 u32DeviceHandle)
    * \brief   Called when currently selected device is de-selected.
    *          Mandatory interface to be implemented.
    * \param   [IN] u32DeviceHandle: Unique handle of selected device
    * \retval  None
    * \sa      vOnSelectDevice()
    **************************************************************************/
   virtual t_Void vOnDeselectDeviceResult(t_U32 u32DeviceHandle);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnData(const trVehicleData& rfrcVehicleData)
   ***************************************************************************/
   /*!
   * \fn      vOnData(const trVehicleData rfrcVehicleData)
   * \brief   Method to receive Vehicle (gear) data.
   *          Optional interface to be implemented.
   * \param   rfrcVehicleData: [IN] Vehicle data
   * \param   bSolicited: [IN] True if the data update is for changed vehicle data, else False
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnData(const trVehicleData& rfrcVehicleData, t_Bool bSolicited);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnData(const trSensorData& rfcorSensorData)
   ***************************************************************************/
   /*!
   * \fn      vOnData(const trSensorData& rfcorSensorData)
   * \brief   Method to receive Sensor data.
   *          Optional interface to be implemented.
   * \param   rfcorSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnData(const trSensorData& rfcorSensorData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnAccSensorData
   ** (const std::vector<trAccSensorData>& rfcoSensorData)
   ***************************************************************************/
   /*!
   * \fn      vOnAccSensorData(const std::vector<trAccSensorData>& rfcoSensorData)
   * \brief   Method to receive Sensor data.
   *          Optional interface to be implemented.
   * \param   rfcorSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnAccSensorData(const std::vector<trAccSensorData>& rfcoSensorData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnGyroSensorData
   ** (const std::vector<trGyroSensorData>& rfcoSensorData)
   ***************************************************************************/
   /*!
   * \fn      vOnGyroSensorData(const std::vector<trGyroSensorData>& rfcoSensorData)
   * \brief   Method to receive Sensor data.
   *          Optional interface to be implemented.
   * \param   rfcorSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   virtual t_Void vOnGyroSensorData(const std::vector<trGyroSensorData>& rfcoSensorData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLDataService::vSetLocDataAvailablility(t_Bool...)
   ***************************************************************************/
   /*!
   * \fn      vSetLocDataAvailablility(t_Bool bLocDataAvailable)
   * \brief   Interface to set the availability of LocationData
   * \param   rfrDataServiceConfigInfo: the structure consists of values TRUE
   *           for location data, dead reckoning data, environment data,
   *           gear status, accelerometer data and gyroscope data if available,
   *           FALSE if not.
   * \retval  None
   **************************************************************************/
   virtual t_Void vSetSensorDataAvailablility(
         const trDataServiceConfigData& rfrDataServiceConfigInfo);

   /***************************************************************************
   ** FUNCTION:  t_Void vReportGear()
   ***************************************************************************/
   /*!
   * \fn      vReportGear()
   * \brief   Method to receive Vehicle (gear) data.
   * \param
   * \retval  None
   ***************************************************************************/
   t_Void vReportGear();

   /******* End of Methods overridden from spi_tclDataServiceDevBase *********/

   /***** Start of Methods overridden from spi_tclBDCLRespCarData *********/
   /***************************************************************************
    ** FUNCTION:  t_Void vCarDataCallback()
    ***************************************************************************/
   /*!
    * \fn      t_Void vCarDataCallback()
    * \brief   Called on receiving CarDataSubscription  Msg.
    * \retval  t_Void
    **************************************************************************/
    t_Void vCarDataSubscriptionCallback(const tBdclCarDataInfoList& rCarDataSubscription);

    /*********************************************************************************
     ** FUNCTION:  t_Void vCarDataSubscriptionStartCallback()
     *********************************************************************************/
    /*!
     * \fn      t_Void vCarDataSubscriptionStartCallback()
     * \brief   Called on receiving CarDataSubscription Start Msg.
     * \retval  t_Void
     *********************************************************************************/
     t_Void vCarDataSubscriptionStartCallback(const tBdclCarDataInfoList& rCarDataSubscriptionStart);

     /********************************************************************************
      ** FUNCTION:  t_Void vCarDataSubscriptionStopCallback()
      ********************************************************************************/
     /*!
      * \fn      t_Void vCarDataSubscriptionStopCallback()
      * \brief   Called on receiving CarDataSubscription Stop Msg.
      * \retval  t_Void
      ********************************************************************************/
      t_Void vCarDataSubscriptionStopCallback(const tBdclCarDataInfoList& rCarDataSubscriptionStop);

      /******* End of Methods overridden from spi_tclBDCLRespCarData *********/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/
   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLDataService::bVehicleVelocityTimerCb
   ***************************************************************************/
   /*!
   * \fn     bVehicleVelocityTimerCb
   * \brief  called on expiry of VehicleSpeed timer
   * \param  rTimerID: ID of the timer which has expired
   * \param  pvObject: pointer to object passed while starting the timer
   * \param  pvUserData: data passed during start of the timer
   **************************************************************************/
   static t_Bool bVehicleVelocityTimerCb(timer_t rTimerID, t_Void *pvObject,
           const t_Void *pvUserData);

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLDataService::bVehicleVelocityTimerCb
   ***************************************************************************/
   /*!
   * \fn     bVehicleVelocityTimerCb
   * \brief  called on expiry of VehicleSpeed timer
   * \param  rTimerID: ID of the timer which has expired
   * \param  pvObject: pointer to object passed while starting the timer
   * \param  pvUserData: data passed during start of the timer
   **************************************************************************/
   static t_Bool bVehicleGPSTimerCb(timer_t rTimerID, t_Void *pvObject,
           const t_Void *pvUserData);

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLDataService::bVehicleVelocityTimerCb
   ***************************************************************************/
   /*!
   * \fn     bVehicleVelocityTimerCb
   * \brief  called on expiry of VehicleSpeed timer
   * \param  rTimerID: ID of the timer which has expired
   * \param  pvObject: pointer to object passed while starting the timer
   * \param  pvUserData: data passed during start of the timer
   **************************************************************************/
   static t_Bool bVehicleGyroscopeTimerCb(timer_t rTimerID, t_Void *pvObject,
           const t_Void *pvUserData);

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLDataService::bVehicleVelocityTimerCb
   ***************************************************************************/
   /*!
   * \fn     bVehicleVelocityTimerCb
   * \brief  called on expiry of VehicleSpeed timer
   * \param  rTimerID: ID of the timer which has expired
   * \param  pvObject: pointer to object passed while starting the timer
   * \param  pvUserData: data passed during start of the timer
   **************************************************************************/
   static t_Bool bVehicleAccelerationTimerCb(timer_t rTimerID, t_Void *pvObject,
           const t_Void *pvUserData);

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLDataService::bVehicleVelocityTimerCb
   ***************************************************************************/
   /*!
   * \fn     bVehicleVelocityTimerCb
   * \brief  called on expiry of VehicleSpeed timer
   * \param  rTimerID: ID of the timer which has expired
   * \param  pvObject: pointer to object passed while starting the timer
   * \param  pvUserData: data passed during start of the timer
   **************************************************************************/
   static t_Bool bVehicleGearTimerCb(timer_t rTimerID, t_Void *pvObject,
           const t_Void *pvUserData);

    /***************************************************************************
    ** FUNCTION:  spi_tclBDCLDataService::vStartVehicleSpeedUpdate
    ***************************************************************************/
   /*!
   * \fn      vStartVehicleSpeedUpdate()
   * \brief   Method to start timer to send Vehicle Speed info
   * \retval  t_Void
   ***************************************************************************/
   t_Void vStartVehicleUpdate(tenBdclCarDataModuleID enModuleID);

   /***************************************************************************
    ** FUNCTION:  spi_tclBDCLDataService::vStopVehicleSpeedUpdate
    ***************************************************************************/
   /*!
   * \fn      vStopVehicleSpeedUpdate()
   * \brief   Method to stop timer to send Vehicle Speed info
   * \retval  t_Void
   ***************************************************************************/
   t_Void vStopVehicleUpdate(tenBdclCarDataModuleID enModuleID);

   /***************************************************************************
    ** FUNCTION:  spi_tclBDCLDataService::vReportVehicleGPSInfo
    ***************************************************************************/
   /*!
   * \fn      vReportVehicleVelocityInfo()
   * \brief   Method to send Vehicle GPS Info.
   * \retval  t_Void
   ***************************************************************************/
   t_Void vReportVehicleGPSInfo();
   
   /***************************************************************************
    ** FUNCTION:  spi_tclBDCLDataService::vReportVehicleVelocityInfo
    ***************************************************************************/
   /*!
   * \fn      vReportVehicleVelocityInfo()
   * \brief   Method to send Vehicle Speed Info.
   * \retval  t_Void
   ***************************************************************************/
   t_Void vReportVehicleVelocityInfo();

   /***************************************************************************
    ** FUNCTION:  spi_tclBDCLDataService::vReportVehicleGyroscopeInfo
    ***************************************************************************/
   /*!
   * \fn      vReportVehicleGyroscopeInfo()
   * \brief   Method to send Vehicle Gyroscope Info.
   * \retval  t_Void
   ***************************************************************************/
   t_Void vReportVehicleGyroscopeInfo();

   /***************************************************************************
    ** FUNCTION:  spi_tclBDCLDataService::vReportVehicleAccInfo
    ***************************************************************************/
   /*!
   * \fn      vReportVehicleAccInfo()
   * \brief   Method to send Vehicle Acceleration Info.
   * \retval  t_Void
   ***************************************************************************/
   t_Void vReportVehicleAccInfo();

   /***************************************************************************
    ** FUNCTION:t_Void spi_tclBDCLDataService::vSendSubscriptionDoneMsg(...)
    ***************************************************************************/
   /*!
    * \fn      vSendSubscriptionDoneMsg()
    * \brief   Used to send Subscription Done message
    * \param   [IN] rCarDataSubscriptionDoneList: List of modules supported
    * \retval  None
    **************************************************************************/
   t_Void vSendSubscriptionDoneMsg(const tBdclCarDataInfoList& rCarDataSubscriptionDoneList);

   /***************************************************************************
    ** FUNCTION:t_Void spi_tclBDCLDataService::vDataSubscribe(...)
    ***************************************************************************/
   /*!
    * \fn      vDataSubscribe
    * \brief   Used to subscribe for sensors data
    * \param   [IN] bSubscriptionState: Indicates if data should be subscribed/unsubscribed.
    * \retval  None
    **************************************************************************/
   t_Void vDataSubscribe(t_Bool bSubscriptionState);

   /***************************************************************************
   ** Data Members
   ***************************************************************************/

   /***************************************************************************
    ** Selected device's information
    ***************************************************************************/
   trBDCLSelectedDeviceInfo     m_rSelDevInfo;
   
   /***************************************************************************
   ** Vehicle Speed info
   ***************************************************************************/
   t_S16    m_s16VehicleSpeed;
   
   /***************************************************************************
   ** Vehicle GPS info
   ***************************************************************************/
   trBdclCarGPS m_rGPS;

   /***************************************************************************
   ** Vehicle Gyroscope info
   ***************************************************************************/
   std::vector<trGyroSensorData> m_vecGyroscopeData;

   /***************************************************************************
   ** Vehicle Acceleration info
   ***************************************************************************/
   std::vector<trAccSensorData> m_vecAccelerationData;

   /***************************************************************************
   ** Vehicle Gear info
   ***************************************************************************/
   tenGearPosition m_enGearPosition;
   
   /***************************************************************************
    ** DataService callbacks structure
    ***************************************************************************/
   trDataServiceCb            m_rDataServiceCb;
   
   /***************************************************************************
   **Pointer to spi_tclBDCLCmdCarData class
   ***************************************************************************/
   t_SptrBDCLCmdCarData m_spoCmdCarData;

   /****************************************************************************
   ** Config info
   *****************************************************************************/
   trDataServiceConfigData m_rBDCLDataServiceConfigData;

};
#endif // SPI_TCLBDCLDATASERVICE_H_
