/*!
 *******************************************************************************
 * \file             spi_tclBDCLConnection.cpp
 * \brief            Baidu Carlife Connection class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Baidu Carlife Connection class to handle devices capable of Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 21.04.2017  | Ramya Murthy          | Initial Version

 \endverbatim
 ******************************************************************************/
#include <bdcl/BdclLogging.h>
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLConnection.h"
#include "spi_tclMediator.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdNativeTransport.h"
#include "spi_tclExtCmdBluetooth.h"
#include "spi_tclExtCompManager.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclBDCLConnection.cpp.trc.h"
#endif
#endif

static const t_Char* sczApplicationVersion = "1.0.0";
static const t_U32 su32ApplicationVersion = 1;
static const t_U16 scu16ProtocolMatchTimeoutInSecForAndroid = 8;
//sets number of trials for retry i.e 4 trials of 2( (8/4) = 2 ) seconds
static const t_U16 scu16ProtocolMatchTimeoutInSecForIphone = 4;
const static t_U32 INVALID_DEVICE_HANDLE = 0;


static const trBdclFeatureConfigInfo scrAndroidDevFeatConfigInfo[ ] = {
         {"VOICE_MIC",0},
         {"VOICE_WAKEUP",0},
         {"BLUETOOTH_AUTO_PAIR",1},
         {"BLUETOOTH_INTERNAL_UI",0},
         {"FOCUS_UI",0},
         {"MEDIA_SAMPLE_RATE",1},
         {"AUDIO_TRANSMISSION_MODE",0}
};
static const trBdclFeatureConfigInfo scrAppleDevFeatConfigInfo[ ] = {
         {"VOICE_MIC",0},
         {"VOICE_WAKEUP",0},
         {"BLUETOOTH_AUTO_PAIR",1},
         {"BLUETOOTH_INTERNAL_UI",0},
         {"FOCUS_UI",0},
         {"MEDIA_SAMPLE_RATE",0},
         {"AUDIO_TRANSMISSION_MODE",0}
};

static tBdclFeatureConfigList sAndroidDevFeatConfigList;
static tBdclFeatureConfigList sAppleDevFeatConfigList;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::spi_tclBDCLConnection
 ***************************************************************************/
spi_tclBDCLConnection::spi_tclBDCLConnection() :
   m_bStartTransportAfterSwitch(false), m_u32SelectedDevice(INVALID_DEVICE_HANDLE), m_enDeviceType(e8_UNKNOWN_DEVICE),
   m_u32ConnectCount(0), m_u32ConnectSuccessCount(0), m_u32ConnectTime(0),
   m_enDeviceErrorDuringSelection(e8CL_AOAP_NO_ERROR),m_bWaitingForFeatureConfigRequest(false),m_bIsPreparesessioncalled(false),
   m_enDeviceSelectionState(e8_SELECTION_STATE_DEVICE_NOT_SELECTED)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::spi_tclBDCLConnection entered"));

   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   if (spoBDCLManager)
   {
      //! Register with BDCL manager.
      spoBDCLManager->bRegisterObject((spi_tclBDCLRespSession*) this);
      spoBDCLManager->bRegisterObject((spi_tclBDCLRespVideo*) this);
      m_spoCmdSession = spoBDCLManager->spoGetCmdSessionInstance();
   }
   SPI_NORMAL_ASSERT((!spoBDCLManager) || (!m_spoCmdSession));

   //! Initialize feature configuration list
   const t_U8 cu8NumFeatures = sizeof(scrAndroidDevFeatConfigInfo) / sizeof(scrAndroidDevFeatConfigInfo[0]);
   for (t_U8 u8FeatureIndex = 0; u8FeatureIndex < cu8NumFeatures ; ++u8FeatureIndex)
   {
      sAndroidDevFeatConfigList.push_back(scrAndroidDevFeatConfigInfo[u8FeatureIndex]);
      sAppleDevFeatConfigList.push_back(scrAppleDevFeatConfigInfo[u8FeatureIndex]);
   }

   ETG_TRACE_USR1(("spi_tclBDCLConnection::spi_tclBDCLConnection left"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::bInitializeConnection
 ***************************************************************************/
t_Bool spi_tclBDCLConnection::bInitializeConnection()
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::bInitializeConnection entered "));

   //! Register all BDCL DLT contexts
   bdclRegisterCtxtWithDLT();

   ETG_TRACE_USR1(("spi_tclBDCLConnection::bInitializeConnection left "));
   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::~spi_tclBDCLConnection
 ***************************************************************************/
spi_tclBDCLConnection::~spi_tclBDCLConnection()
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::~spi_tclBDCLConnection entered "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vOnLoadSettings
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vOnLoadSettings(trHeadUnitInfo &rfrheadUnitInfo, tenCertificateType enCertificateType)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vOnLoadSettings entered "));

   m_rHeadUnitInfo = rfrheadUnitInfo;
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vRegisterCallbacks(trConnCallbacks &rfrConnCallbacks)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vRegisterCallbacks entered "));
   m_rConnCallbacks = rfrConnCallbacks;
   //register for mediaplayer callback
   RespRegister *pRespRegister = RespRegister::getInstance();
   if(NULL!= pRespRegister)
   {
      pRespRegister->bRegisterObject((spi_tclExtRespNativeTransport*)this);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vSetDeviceSwitchInfo
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vSetDeviceSwitchInfo(trDeviceSwitchInfo &rfrDeviceSwitchInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vSetDeviceSwitchInfo entered Headunit ID = 0x%x, Device ID = 0x%x, SwitchInProgress = %d",
            rfrDeviceSwitchInfo.s32HeadUnitID,
            rfrDeviceSwitchInfo.s32SwitchedDeviceID,
            ETG_ENUM(BOOL, rfrDeviceSwitchInfo.bSwitchInProgress)));
   m_rDeviceSwitchInfo = rfrDeviceSwitchInfo;
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vOnSelectDevice()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vOnSelectDevice(t_U32 u32DeviceHandle,
         tenDeviceConnectionType enConnType, tenDeviceConnectionReq enConnReq,
         tenEnabledInfo enDAPUsage, tenEnabledInfo enCDBUsage,
         tenSelectReason enSelectionReason, trUserContext corUsrCntxt,
         tenDeviceType enDeviceType)
{
   m_oSessionPrepareLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLConnection::vOnSelectDevice entered for Device ID = 0x%x, "
            "ConnType = %d, enConnReq = %d, enDeviceType = %d",
            u32DeviceHandle, ETG_ENUM(CONNECTION_TYPE, enConnType),
            ETG_ENUM(CONNECTION_REQ, enConnReq), ETG_ENUM(DEVICE_TYPE, enDeviceType)));
   SPI_INTENTIONALLY_UNUSED(enConnType);
   SPI_INTENTIONALLY_UNUSED(enDAPUsage);
   SPI_INTENTIONALLY_UNUSED(enCDBUsage);
   SPI_INTENTIONALLY_UNUSED(enSelectionReason);
   SPI_INTENTIONALLY_UNUSED(corUsrCntxt);

   m_enDeviceType = enDeviceType;
   m_enDeviceErrorDuringSelection = e8CL_AOAP_NO_ERROR;


   if (e8DEVCONNREQ_SELECT == enConnReq)
   {
      vSetSelectedDevice(u32DeviceHandle);
      ETG_TRACE_USR1(("spi_tclBDCLConnection::vOnSelectDevice value of m_bIsPreparesessioncalled = %d", ETG_ENUM(BOOL, m_bIsPreparesessioncalled)));
      if (enDeviceType == e8_ANDROID_DEVICE)
      {
         if(false == m_bIsPreparesessioncalled)
         {
            enPrepareSession(m_enDeviceType);
         }
         else
         {
            ETG_TRACE_USR2(("[DESC]::vOnSelectDevice: Already enPrepareSession() is called ....."));
         }
      }
      else
      {
         if (m_bStartTransportAfterSwitch)
         {
            ETG_TRACE_USR2(("[DESC]::vOnSelectDevice: Native transport start is already received. Calling PrepareSession"));
            if(false == m_bIsPreparesessioncalled)
            {
               enPrepareSession(m_enDeviceType);
            }
            else
            {
               ETG_TRACE_USR2(("[DESC]::vOnSelectDevice: Already enPrepareSession() is called ....."));
            }
		    m_bStartTransportAfterSwitch = false;
         }
         else
         {
            ETG_TRACE_USR2(("[DESC]::vOnSelectDevice: Native transport start is not yet received. Waiting ....."));
         }
      }
   }
   else if (e8DEVCONNREQ_DESELECT == enConnReq)
   {
      vSetSelectedDevice(INVALID_DEVICE_HANDLE);
      tenErrorCode enErrorType = (bTeardownSession()) ? (e8NO_ERRORS) : e8SELECTION_FAILED;
      vSendSelectDeviceResult(enErrorType);
   }

   m_oSessionPrepareLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vOnSelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vOnSelectDeviceResult(t_U32 u32DeviceHandle,
         tenDeviceConnectionReq enConnReq, tenResponseCode enRespCode,
         tenDeviceCategory enDevCat, tenSelectReason enSelectionReason,
         tenDeviceType enDeviceType)
{
   m_oSessionPrepareLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLConnection::vOnSelectDeviceResult entered for Device ID = 0x%x, "
            "ConnReq = %d, enDeviceType = %d, enRespCode = %d ",
            u32DeviceHandle, ETG_ENUM(CONNECTION_REQ, enConnReq),
            ETG_ENUM(DEVICE_TYPE, enDeviceType), ETG_ENUM(RESPONSE_CODE, enRespCode)));
   SPI_INTENTIONALLY_UNUSED(enDevCat);
   SPI_INTENTIONALLY_UNUSED(enSelectionReason);
   m_bWaitingForFeatureConfigRequest = false;
   if (e8DEVCONNREQ_SELECT == enConnReq)
   {
      if (e8SUCCESS == enRespCode)
      {
         vSendHUInfo();
      }
      else
      {
		 vSetSelectedDevice(INVALID_DEVICE_HANDLE);
         bTeardownSession();
      }
   }//if (e8DEVCONNREQ_SELECT == enConnReq)
   else
   {
      m_bIsPreparesessioncalled = false;
      m_bStartTransportAfterSwitch = false;
      ETG_TRACE_USR2(("[DESC]::vOnSelectDeviceResult: Cleaned m_bIsPreparesessioncalled and m_bStartTransportAfterSwitch values"));
   }

   m_oSessionPrepareLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLConnection::vSendHUInfo()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vSendHUInfo()
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vSendHUInfo entered"));

   if (m_spoCmdSession)
   {
      ETG_TRACE_USR4(("[PARAM]::vSendHUInfo Manufacturer = %s",m_rHeadUnitInfo.szHUManufacturer.c_str() ));
      ETG_TRACE_USR4(("[PARAM]::vSendHUInfo Model = %s",m_rHeadUnitInfo.szHUModelName.c_str() ));
      ETG_TRACE_USR4(("[PARAM]::vSendHUInfo Release = %s",m_rHeadUnitInfo.szSoftwareVersion.c_str() ));
      ETG_TRACE_USR4(("[PARAM]::vSendHUInfo Serial = %s",m_rHeadUnitInfo.szVehicleID.c_str() ));
      //TODO
      trBdclHUInfo rHUInfo;
      rHUInfo.szOS = "Linux";
      rHUInfo.szManufacturer = m_rHeadUnitInfo.szHUManufacturer; //or m_rHeadUnitInfo.szVehicleManufacturer; ?
      rHUInfo.szModel = m_rHeadUnitInfo.szHUModelName; // or m_rHeadUnitInfo.szVehicleModelName;?
      rHUInfo.szRelease = m_rHeadUnitInfo.szSoftwareVersion;
      rHUInfo.szSerial = m_rHeadUnitInfo.szVehicleID;

      /* not mandatory fields in specification so commented
      rHUInfo.szBoard = "xxx";
      rHUInfo.szBootloader = "xxx";
      rHUInfo.szBrand = "xxx";
      rHUInfo.szCpu_abi = "xxx";
      rHUInfo.szCpu_abi2 = "xxx";
      rHUInfo.szDevice = "xxx";
      rHUInfo.szDisplay = "xxx";
      rHUInfo.szFingerprint = "xxx";
      rHUInfo.szHardware = "xxx";
      rHUInfo.szHost = "xxx";
      rHUInfo.szCid = "xxx";
      rHUInfo.szProduct = "xxx";
      rHUInfo.szCodename = "xxx";
      rHUInfo.szIncremental = "xxx";
      rHUInfo.szSdk = "xxx";
      rHUInfo.u32Sdk = 1;
      rHUInfo.szToken = "xxx";*/

      m_spoCmdSession->vSetHeadUnitInfo(rHUInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLConnection::vGetStatisticsInfo()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vGetStatisticsInfo(trBdclStatisticsInfo& rfrStatisticsInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vGetStatisticsInfo entered"));

   t_String szChannelNumber;
   
   t_String szVehicleBTMacAddr;

   if(NULL != m_poConnSettings)
   {
      szChannelNumber = m_poConnSettings->szGetChannelID();
	  m_poConnSettings->vGetVehicleBTAddress(szVehicleBTMacAddr);
   }

   rfrStatisticsInfo.szCuid = szVehicleBTMacAddr;
   ETG_TRACE_USR4(("[PARAM]::vGetStatisticsInfo cuid = %s ",szVehicleBTMacAddr.c_str() ));
   rfrStatisticsInfo.szVersionName = sczApplicationVersion;
   rfrStatisticsInfo.u32VersionCode = su32ApplicationVersion;
   rfrStatisticsInfo.szChannel = szChannelNumber;
   rfrStatisticsInfo.u32ConnectCount = m_u32ConnectCount;
   rfrStatisticsInfo.u32ConnectSuccessCount = m_u32ConnectSuccessCount;
   rfrStatisticsInfo.u32ConnectTime = m_u32ConnectTime;
   rfrStatisticsInfo.szCrashLog = "";
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vPostSelctDeviceResult()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vSendSelectDeviceResult(tenErrorCode enErrorType)
{
   /*lint -esym(40,fvSelectDeviceResult)fvSelectDeviceResult Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vPostSelectDeviceResult: Sending device selection result "));

   if (NULL != m_rConnCallbacks.fvSelectDeviceResult)
   {
      (m_rConnCallbacks.fvSelectDeviceResult)(enErrorType);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::enPrepareSession
 ***************************************************************************/
tenErrorCode spi_tclBDCLConnection::enPrepareSession(tenDeviceType enDeviceType)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::enPrepareSession entered"));
   m_bIsPreparesessioncalled = true;
   tenErrorCode enErrorType = e8SELECTION_FAILED;

   if ((m_spoCmdSession) && (e8_UNKNOWN_DEVICE != enDeviceType))
   {
      (e8_ANDROID_DEVICE == enDeviceType)? (m_spoCmdSession->vSetProtocolStatusTimeOut(scu16ProtocolMatchTimeoutInSecForAndroid))
                                           : (m_spoCmdSession->vSetProtocolStatusTimeOut(scu16ProtocolMatchTimeoutInSecForIphone));
  
      trBdclInitInfo rInitInfo;
      vGetInitInfo(enDeviceType, rInitInfo);

      trBdclStatisticsInfo rStatisticsInfo;
      vGetStatisticsInfo(rStatisticsInfo);

      vUpdateSelectionProgressState(u32GetSelectedDevice(), e8_SELECTION_PROGRESS_STATE_SERVICE_DISCOVERY);

      //@Note: enInitialize is a blocking call (10s)!
      tenBdclInitStatus enInitStatus = m_spoCmdSession->enInitialize(rInitInfo, rStatisticsInfo);
      ETG_TRACE_USR2(("[DESC]::enPrepareSession: Init result = %d", ETG_ENUM(BDCL_INIT_STATUS, enInitStatus)));

      if (m_enDeviceErrorDuringSelection == e8CL_AOAP_NO_DEVICE_ERROR)
      {
         enErrorType = e8DEVICE_DISCONNECTED_DURING_SELECTION;
      }
      else
      {
         switch (enInitStatus)
         {
            case e8CL_INIT_NO_ERR:
            {
               enErrorType = e8NO_ERRORS;
            }
            break;
            case e8CL_INIT_TIMEOUT_ERR:
            {
               enErrorType = e8CONNECT_TIME_OUT;
            }
            break;
            default:
            {
               enErrorType = e8FATAL_ERROR;
            }
            break;
         }
      }

      if(enErrorType != e8NO_ERRORS)
      {
         vSendSelectDeviceResult(enErrorType);
      }else
      {
         m_bWaitingForFeatureConfigRequest = true;
      }
   }else//if ((m_spoCmdSession) && (e8_UNKNOWN_DEVICE != enDeviceType))
   {
      ETG_TRACE_ERR(("spi_tclBDCLConnection::enPrepareSession could not start. Error occured"));
      vSendSelectDeviceResult(e8UNKNOWN_ERROR);
   }

   ETG_TRACE_USR2(("[DESC]::spi_tclBDCLConnection::enPrepareSession left with error = %d", ETG_ENUM(ERROR_CODE, enErrorType)));
   return enErrorType;
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::bTeardownSession
 ***************************************************************************/
t_Bool spi_tclBDCLConnection::bTeardownSession()
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::bTeardownSession entered"));
   //if session error occures Clear the flag of preparesession called
   m_bIsPreparesessioncalled = false;
   m_bStartTransportAfterSwitch = false;
   
   t_Bool bResult = false;

   if (m_spoCmdSession)
   {
      bResult = true;
      m_spoCmdSession->vUninitialize();
   }//if (m_spoCmdSession)

   ETG_TRACE_USR2(("[DESC]::spi_tclBDCLConnection::bTeardownSession left with result = %d", ETG_ENUM(BOOL, bResult)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vGetInitInfo()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vGetInitInfo(tenDeviceType enDeviceType, trBdclInitInfo& rfrInitInfo)
{
   SPI_NORMAL_ASSERT(e8_UNKNOWN_DEVICE == enDeviceType);

   if (e8_ANDROID_DEVICE == enDeviceType)
   {
      rfrInitInfo.bIsAoapDevice       = true;
      rfrInitInfo.s32AoapAccessoryId  = m_rDeviceSwitchInfo.s32HeadUnitID;
      rfrInitInfo.s32AoapDeviceId     = m_rDeviceSwitchInfo.s32SwitchedDeviceID;
   }
   else if (e8_APPLE_DEVICE == enDeviceType)
   {
      //TODO - move device name to policy configuration
      rfrInitInfo.bIsAoapDevice       = false;
      rfrInitInfo.szReadDevice        = "/dev/ffs/ep4";
      rfrInitInfo.szWriteDevice       = "/dev/ffs/ep3";
      ETG_TRACE_USR4(("[DESC]::vGetInitInfo: Initialising for Apple device"));
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vOnExitCb
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vOnExitInfo()
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vOnExitInfo entered"));

   vSendDeviceDeselect();

   ETG_TRACE_USR2(("spi_tclBDCLConnection::vOnExitInfo left "));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLConnection::vOnCoreError()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vOnCoreError(tenBdclCarLifeError enCarlifeError)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vOnCoreError entered: %d", ETG_ENUM(BDCL_ERROR, enCarlifeError)));

   //TODO - handle other AOAP errors to terminate session?
   if (e8CL_AOAP_NO_DEVICE_ERROR == enCarlifeError)
   {
      ETG_TRACE_USR2(("[DESC]:: AOAP_NO_DEVICE_ERROR: Requesting device deselection"));

      m_enDeviceErrorDuringSelection = e8CL_AOAP_NO_DEVICE_ERROR;

      // Here the Device is physically Disconnedted.
      vSendDeviceDeselect(e8_REASON_NO_DEVICE_ERROR);
   }// if (e8CL_AOAP_NO_DEVICE_ERROR == enCarlifeError)
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclBDCLConnection::vPostNativeTransportStartResult()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vPostNativeTransportStartResult(const t_U32 cou32DevID)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vPostNativeTransportStartResult from MediaPlayer entered cou32DevID = 0x%x", cou32DevID));
   vOnNativeTransportStart(cou32DevID);
}

/***************************************************************************
 ** FUNCTION: spi_tclBDCLConnection::vPostNativeTransportStopResult()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vPostNativeTransportStopResult(t_U32 u32DeviceId)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vPostNativeTransportStopResult from MediaPlayer entered cou32DevID = 0x%x. Triggering deselection",u32DeviceId));
   if( (INVALID_DEVICE_HANDLE != u32GetSelectedDevice())&&(u32GetSelectedDevice()== u32DeviceId))
   {
	  //Work around for NCG3D-155243 Calling session error when Nativetrasportstop message received
	  vReportSessionErrorStatus(e8DEV_TYPE_CARLIFE,e8_SESSION_ERROR);
      vSendDeviceDeselect();
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclBDCLConnection::vOnNativeTransportStart()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vOnNativeTransportStart(const t_U32 cou32DevID)
{
   m_oSessionPrepareLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vOnNativeTransportStart entered cou32DevID = 0x%x", cou32DevID));

   if (INVALID_DEVICE_HANDLE != cou32DevID)
   {
      if (cou32DevID == u32GetSelectedDevice())
      {
         ETG_TRACE_USR2(("[DESC]::vOnNativeTransportStart received after selection sequence started"));
         if(false == m_bIsPreparesessioncalled)
         {
            enPrepareSession(m_enDeviceType);
         }
         else
         {
            ETG_TRACE_USR2(("[DESC]::vOnNativeTransportStart: Already enPrepareSession() is called ....."));
         }
      }
      else
      {
         // Since selection is in progress, if native transport has started,
         // start bdcl session immediately after switch is complete.
         ETG_TRACE_USR2(("[DESC]::vOnNativeTransportStart received before starting selection sequence"));
         m_bStartTransportAfterSwitch = true;
      }
   }
   else if(e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS == m_enDeviceSelectionState)
   {
      // Since selection is in progress, if native transport has started,
      // start bdcl session immediately after switch is complete.
      ETG_TRACE_USR2(("[DESC]::vOnNativeTransportStart received during Roleswitch is in progress"));
      m_bStartTransportAfterSwitch = true;
   }
   m_oSessionPrepareLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclBDCLConnection::vSetSelectedDevice(
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vSetSelectedDevice(const t_U32 cou32DevID)
{
   m_u32SelectedDevice = cou32DevID;
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vSetSelectedDevice m_u32SelectedDevice = 0x%x",m_u32SelectedDevice));
}

/***************************************************************************
 ** FUNCTION: t_U32 spi_tclBDCLConnection::u32GetSelectedDevice()
 ***************************************************************************/
t_U32 spi_tclBDCLConnection::u32GetSelectedDevice()
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::u32GetSelectedDevice m_u32SelectedDevice = 0x%x",m_u32SelectedDevice));
   return m_u32SelectedDevice;
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vUpdateSelectionProgressState
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vUpdateSelectionProgressState(const t_U32 cou32DeviceHandle,
         tenDeviceSelectionProgressState enDeviceSelectionProgressState)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vUpdateSelectionProgressState entered "));
   //! Set selection progress state
   if ((NULL != m_rConnCallbacks.fvSelectionProgressState) && (INVALID_DEVICE_HANDLE != cou32DeviceHandle))
   {
      (m_rConnCallbacks.fvSelectionProgressState)(cou32DeviceHandle, enDeviceSelectionProgressState);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLConnection::vOnProtocolVersionMatchStatus()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vOnProtocolVersionMatchStatus(t_Bool bProtocolMatchStatus)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vOnProtocolVersionMatchStatus entered "));

   if (bProtocolMatchStatus == true)
   {
      vUpdateSelectionProgressState(u32GetSelectedDevice(), e8_SELECTION_PROGRESS_STATE_AUTHENTICATING);
   }
}

/***************************************************************************
 ** FUNCTION: spi_tclBDCLConnection::vSendDeviceDeselect()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vSendDeviceDeselect(tenSelectReason enSelectReason)
{
   ETG_TRACE_USR4(("[DESC]::vSendDeviceDeselect: Requesting deselection of device "));

   //! Post internal deselection
   //! Send Device deselection to SPI components
   spi_tclMediator *poMediator = spi_tclMediator::getInstance();
   t_U32 u32SelDevHandle = u32GetSelectedDevice();

   if ((NULL != poMediator) && (INVALID_DEVICE_HANDLE != u32SelDevHandle))
   {
      poMediator->vPostAutoDeviceSelection(u32SelDevHandle, e8DEVCONNREQ_DESELECT, enSelectReason);
   } // if (NULL != poMediator)
}

/***************************************************************************
 ** FUNCTION: spi_tclBDCLConnection::vUpdateDeviceConnectionCountInfo()
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vUpdateDeviceConnectionCountInfo(trDeviceConnectionCountInfo &rfrDeviceConnectionCountInfo)
{
	ETG_TRACE_USR4(("spi_tclBDCLConnection::vUpdateDeviceConnectionCountInfo: Entered "));
	//! -1 is done because spi_tclDeviceDiscoverer already increments the
	//!value when it gets select device and we are sending information about last connection.
	//!TODO may be change the logic to send information about current connection
	m_u32ConnectCount = rfrDeviceConnectionCountInfo.rDeviceStatisticInfo[e8DEV_TYPE_CARLIFE].u32ConnectCount - 1;
	m_u32ConnectSuccessCount = rfrDeviceConnectionCountInfo.rDeviceStatisticInfo[e8DEV_TYPE_CARLIFE].u32ConnectSuccessCount;
	m_u32ConnectTime = rfrDeviceConnectionCountInfo.rDeviceStatisticInfo[e8DEV_TYPE_CARLIFE].u32ConnectTime;

	ETG_TRACE_USR4(("[PARAM]:: spi_tclBDCLConnection::vUpdateDeviceConnectionCountInfo::CarlifeConnectCount = %d",m_u32ConnectCount));
	ETG_TRACE_USR4(("[PARAM]:: spi_tclBDCLConnection::vUpdateDeviceConnectionCountInfo::CarlifeConnectSuccessCount = %d",m_u32ConnectSuccessCount));
	ETG_TRACE_USR4(("[PARAM]:: spi_tclBDCLConnection::vUpdateDeviceConnectionCountInfo::CarlifeConnectTime = %d ms",m_u32ConnectTime));

}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vOnFeatureConfigRequest
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vOnFeatureConfigRequest()
{
   //Acquiring the lock to synchronize between enInitialize and feature config request.
   //In case of iphone feature config request sometimes comes before we exit prepare session
   m_oSessionPrepareLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vOnFeatureConfigRequest entered"));

   if ((m_spoCmdSession) && (true == m_bWaitingForFeatureConfigRequest) )
   {
      const tBdclFeatureConfigList& corfFeatConfigList =
               (e8_ANDROID_DEVICE == m_enDeviceType)? (sAndroidDevFeatConfigList) : (sAppleDevFeatConfigList);
      m_spoCmdSession->vSetFeatureConfigInfo(corfFeatConfigList);
      vSendSelectDeviceResult(e8NO_ERRORS);
   }
   ETG_TRACE_USR2(("spi_tclBDCLConnection::vOnFeatureConfigRequest left "));
   m_oSessionPrepareLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vOnSetSelectionProgressState(
 ***************************************************************************/
t_Void spi_tclBDCLConnection::vOnSetSelectionProgressState(tenDeviceSelectionState enDeviceSelectionState)
{
   ETG_TRACE_USR2(("spi_tclBDCLConnection::vOnSetSelectionProgressState entered with result = %d", ETG_ENUM(tenDeviceSelectionState, enDeviceSelectionState)));
   m_enDeviceSelectionState = enDeviceSelectionState;
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLConnection::vReportSessionErrorStatus()
 **************************************************************************/
t_Void spi_tclBDCLConnection::vReportSessionErrorStatus(tenDeviceCategory coenDevCat,tenSessionStatus enSessionStatus)
{
   ETG_TRACE_USR1(("spi_tclBDCLConnection::vReportSessionErrorStatus: Sending Session error state to resource manager"));

   //! Post Session error to HMI
   //! Send  Session error  to Resource manager (SPI components)
   spi_tclMediator *poMediator = spi_tclMediator::getInstance();

   if( (NULL != poMediator) && (INVALID_DEVICE_HANDLE != m_u32SelectedDevice))
   {
      poMediator->vReportSessionErrorStatus(m_u32SelectedDevice, coenDevCat, enSessionStatus);
   }
}
//lint -restore

///////////////////////////////////////////////////////////////////////////////
// <EOF>
