/*!
 *******************************************************************************
 * \file         spi_tclBDCLBluetooth.cpp
 * \brief        BDCL Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    BDCL Bluetooth handler class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 22.06.2017 |  Pooja Krishnan (RBEI/ECO2)       | Initial Version
 23.10.2017 |  Kavya Mogeri  (RBEI/ECO2)		| Modified Autopair sequence, added vOnMDInfo()
 04.12.2018 |  Dundamma S B (RBEI/ECO2)         | Registering BT callbacks to ExtCmdBluetooth directly without the interface of common Bluetooth class

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <string>
 
#include "spi_tclBluetoothIntf.h"
#include "spi_tclBDCLCmdBluetooth.h"
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLBluetooth.h"
#include "spi_tclExtCmdBluetooth.h"

#include "Trace.h"
#ifdef TARGET_BUILD
 #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BLUETOOTH
    #include "trcGenProj/Header/spi_tclBDCLBluetooth.cpp.trc.h"
 #endif
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/
static const tenBTSetPairingMethod scenBTSetPreferredBDCLPairingMethod =
         e8PAIRING_TYPE_SSP_NUMERIC_COMPARISON;

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
static t_Bool sbBTLimModeRequested = false;
static t_String szPairingPin = "";

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLBluetooth::spi_tclBDCLBluetooth();
 ***************************************************************************/
spi_tclBDCLBluetooth::spi_tclBDCLBluetooth(spi_tclBluetoothIntf* poBTInterface)
        : m_cpoBTInterface(poBTInterface),
          m_enPairingStatus(e8DEVICE_PAIRING_NOT_STARTED),
          m_bIdentificationRequestSent(false),
          m_spoCmdBluetooth(NULL),
		  m_poExtCmdBluetoothIntf(NULL),
          m_enPendingBTLimAction(e8BT_LIM_ACTION_UNKNOWN),
          m_enPendingBTLimTechnology(e8BT_TECH_CARLIFE)
{

   ETG_TRACE_USR1(("spi_tclBDCLBluetooth() entered "));

   SPI_NORMAL_ASSERT(NULL == m_cpoBTInterface);

   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   if (spoBDCLManager)
   {
      // Initializations to be done which are not specific to device connected/selected
      m_spoCmdBluetooth = spoBDCLManager->spoGetCmdBluetoothInstance();
   }
   SPI_NORMAL_ASSERT((!spoBDCLManager) || (!m_spoCmdBluetooth));
   if (NULL != spoBDCLManager)
   {
      //! Register with BDCL manager for BT callbacks
      spoBDCLManager->bRegisterObject((spi_tclBDCLRespBluetooth*)this);
      spoBDCLManager->bRegisterObject((spi_tclBDCLRespSession*) this);
   }//if(NULL != poBDCLManager)

} //!end of spi_tclBDCLBluetooth()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLBluetooth::~spi_tclBDCLBluetooth();
 ***************************************************************************/
spi_tclBDCLBluetooth::~spi_tclBDCLBluetooth()
{
   ETG_TRACE_USR1(("~spi_tclBDCLBluetooth() entered "));
} //!end of ~spi_tclBDCLBluetooth()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclBDCLBluetooth::bInitialise();
 ***************************************************************************/
t_Bool spi_tclBDCLBluetooth::bInitialise()
{
   t_Bool bInit = false;
   RespRegister *pRespRegister = RespRegister::getInstance();
   if(NULL!= pRespRegister)
   {
	   bInit = pRespRegister->bRegisterObject((spi_tclExtRespBluetooth*)this);
   }
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::bInitialise() left with Result %u",bInit));
   return bInit;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnSPISelectDeviceRequest(...)
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnSPISelectDeviceRequest() entered "));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   if (NULL != m_cpoBTInterface)
   {
      //! Set Device status - this is in order to prevent SwitchDevice during ongoing selection
      m_poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();

      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_SELECTION_IN_PROGRESS);

      //@Note: BT connection is handled after BT PairingRequest is received from BDCL
      t_Bool bInitSuccess = false;
      if ((NULL != m_spoCmdBluetooth))
      {
         bInitSuccess = m_spoCmdBluetooth->bInitialize();
      }
      ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnSPISelectDeviceRequest() Result:%d ",bInitSuccess));
      m_cpoBTInterface->vSendSelectDeviceResult(bInitSuccess);
   }
} //!end of vOnSPISelectDeviceRequest()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnSPISelectDeviceResponse(..)
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
      tenResponseCode enRespCode)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnSPISelectDeviceResponse() entered "));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);

   sbBTLimModeRequested = false;
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName ;
   //! For Carlfie iPhones BTAddress is not known so this parameter can be an empty string.
   t_String szSelectDeviceBTAddress;
   if ((NULL != m_cpoBTInterface)&& (e8SUCCESS == enRespCode) && (NULL != m_poExtCmdBluetoothIntf))
   {
      if ((e8_ANDROID_DEVICE == corfrSelectReq.m_enDeviceType) && (!(m_cpoBTInterface->szGetSelectedDevBTAddress().empty())))
      {
           ETG_TRACE_USR1(("[PARAM]::spi_tclBDCLBluetooth::vOnSPISelectDeviceResponse() Selected Device BT address Result: %s", 
           m_cpoBTInterface->szGetSelectedDevBTAddress().c_str()));
           sbBTLimModeRequested = m_poExtCmdBluetoothIntf->bSetBTLimitationMode(
                  m_cpoBTInterface->szGetSelectedDevBTAddress(),szBTDeviceName,
                  e8BT_TECH_CARLIFE,
                  e8BT_COMM_CHN_USB,
                  e8BT_LIM_ACTION_ACTIVATE);
  
         if (false == sbBTLimModeRequested)
         {
            m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
            m_enPendingBTLimTechnology = e8BT_TECH_CARLIFE;
            m_enPendingBTLimAction = e8BT_LIM_ACTION_ACTIVATE;
         }
      }//if (NULL != m_cpoBTInterface)
      else if(e8_APPLE_DEVICE == corfrSelectReq.m_enDeviceType)
      {
         sbBTLimModeRequested = m_poExtCmdBluetoothIntf->bSetBTLimitationMode(
                  szSelectDeviceBTAddress,
                  m_cpoBTInterface->szGetBTDeviceName(),
                  e8BT_TECH_CARLIFE_IOS,
                  e8BT_COMM_CHN_USB,
                  e8BT_LIM_ACTION_ACTIVATE);
         if (false == sbBTLimModeRequested)
         {
            m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
            m_enPendingBTLimTechnology = e8BT_TECH_CARLIFE_IOS;
            m_enPendingBTLimAction = e8BT_LIM_ACTION_ACTIVATE;
         }
      }
   }
   else
   {
      //! On selection failure, perform cleanup.
      if (NULL != m_spoCmdBluetooth)
      {
         m_spoCmdBluetooth->vUninitialize();
      }
   }//else if (e8FAILURE == enRespCode)
} //!end of vOnSPISelectDeviceResponse()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnSPIDeselectDeviceRequest(...)
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnSPIDeselectDeviceRequest() entered "));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   if (NULL != m_cpoBTInterface)
   {
      // Set Device status - this is in order to prevent SwitchDevice during ongoing deselection
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_DESELECTION_IN_PROGRESS);
      // Nothing else to be done. Simply send success result.
      m_cpoBTInterface->vSendSelectDeviceResult(true);
   }

   //! Perform cleanup
   if (NULL != m_spoCmdBluetooth)
   {
      m_spoCmdBluetooth->vUninitialize();
   }
} //!end of vOnSPIDeselectDeviceRequest()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnSPIDeselectDeviceResponse(...)
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnSPIDeselectDeviceResponse() entered "));
   SPI_INTENTIONALLY_UNUSED(enRespCode);
 
   // Clear device status
   if (NULL != m_cpoBTInterface)
   {
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
   }
   vSetPairingStatus(e8DEVICE_PAIRING_NOT_STARTED);
   m_bIdentificationRequestSent = false;

   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;
   //! For Carlfie iPhones BTAddress is not known so this parameter can be an empty string.
   t_String szBTAddress;

   if ((sbBTLimModeRequested) && (NULL != m_cpoBTInterface) && (NULL != m_poExtCmdBluetoothIntf))
   {
      if(corfrSelectReq.m_enSelectionReason != e8_REASON_HMI_SELECT_SPI_AUTO_DESELECT_TRIGGER)
      {
         if(e8_ANDROID_DEVICE == corfrSelectReq.m_enDeviceType)
         {
             if (false == m_poExtCmdBluetoothIntf->bSetBTLimitationMode(
                      m_cpoBTInterface->szGetSelectedDevBTAddress(),szBTDeviceName,
                      e8BT_TECH_CARLIFE,
                      e8BT_COMM_CHN_USB,
                      e8BT_LIM_ACTION_DEACTIVATE))
             {
                m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
                m_enPendingBTLimTechnology = e8BT_TECH_CARLIFE;
                m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
             }
         }
         else if(e8_APPLE_DEVICE == corfrSelectReq.m_enDeviceType)
         {
            sbBTLimModeRequested = m_poExtCmdBluetoothIntf->bSetBTLimitationMode(
                     szBTAddress,m_cpoBTInterface->szGetBTDeviceName(),
                     e8BT_TECH_CARLIFE_IOS,
                     e8BT_COMM_CHN_USB,
                     e8BT_LIM_ACTION_DEACTIVATE);
            if (false == sbBTLimModeRequested)
            {
               m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
               m_enPendingBTLimTechnology = e8BT_TECH_CARLIFE_IOS;
               m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
            }
         }     
      }
      sbBTLimModeRequested = false;
   }
} //!end of vOnSPIDeselectDeviceResponse()



/***************************************************************************
 *********************************PROTECTED**********************************
 ***************************************************************************/

/***************************************************************************
 *********************************PRIVATE************************************
 ***************************************************************************/
/***************************************************************************
 ** FUNCTION:   t_Void spi_tclBDCLBluetooth::vDeselectBDCLDevice()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vDeselectBDCLDevice()
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vDeselectBDCLDevice() entered "));
   if (NULL != m_cpoBTInterface)
   {
      //Clear device status
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
      m_cpoBTInterface->vSendDeselectDeviceRequest();
   }
} //!end of vDeselectBDCLDevice()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::enGetPairingStatus()
 ***************************************************************************/
tenDevicePairingStatus spi_tclBDCLBluetooth::enGetPairingStatus()
{
   m_oPairingStateLock.s16Lock();
   tenDevicePairingStatus enPairingStatus = m_enPairingStatus;
   m_oPairingStateLock.vUnlock();

   ETG_TRACE_USR4(("[PARAM]:spi_tclBDCLBluetooth::enGetPairingStatus() left with enPairingStatus %d ",
            ETG_ENUM(SPI_BT_PAIRING_STATE, enPairingStatus)));
   return enPairingStatus;
} //!end of enGetPairingStatus()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vSetPairingStatus()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vSetPairingStatus(tenDevicePairingStatus enPairingStatus)
{
   ETG_TRACE_USR2(("[DESC]spi_tclBDCLBluetooth::vSetPairingStatus() enPairingStatus : %d ",
            ETG_ENUM(SPI_BT_PAIRING_STATE, enPairingStatus)));

   m_oPairingStateLock.s16Lock();
   m_enPairingStatus = enPairingStatus;
   m_oPairingStateLock.vUnlock();
} //!end of vSetPairingStatus()

/***************************************************************************
 ** FUNCTION:   t_Void spi_tclBDCLBluetooth::vSendDevicePairingResponse()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vSendDevicePairingResponse(tenBdclHUBTStatus enBTStatus)
{
   ETG_TRACE_USR2(("[DESC]spi_tclBDCLBluetooth::vSendDevicePairingResponse: Sending enBTStatus:%d ",ETG_ENUM(BDCL_HU_BT_STATUS,enBTStatus)));
   //! Send BTDevicePairingResponse to device
   if ( (NULL != m_cpoBTInterface) &&(NULL != m_poBluetoothSettings) && (NULL != m_spoCmdBluetooth))
   {
      trBdclBluetoothOOBInfo rOOBInfo;
      m_poBluetoothSettings->vGetVehicleBTAddress(rOOBInfo.szBTAddress);
      ETG_TRACE_USR2(("[PARAM]spi_tclBDCLBluetooth::vSendDevicePairingResponse: BT Address %s", rOOBInfo.szBTAddress.c_str()));
      bFormatBTToMacAddress(rOOBInfo.szBTAddress);
      rOOBInfo.szBTAddress =  rOOBInfo.szBTAddress;
      rOOBInfo.szPassKey = szPairingPin;
      rOOBInfo.szName = m_poExtCmdBluetoothIntf->szGetVehicleBTFriendlyName();
      rOOBInfo.enBTStatus = enBTStatus;
      rOOBInfo.bIsPaired =  m_poExtCmdBluetoothIntf->bGetPairingStatus(m_cpoBTInterface->szGetSelectedDevBTAddress());
      m_spoCmdBluetooth->bSendHUBluetoothOOBInfo(&rOOBInfo);
   }
} //!end of vSendDevicePairingResponse()

/***************************************************************************
 ** FUNCTION:   t_Bool spi_tclBDCLBluetooth::vOnBluetoothIdentifyResultIndMsg()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vOnBluetoothIdentifyResultIndMsg(const trBdclBluetoothIdentifyResultInd* cprBTIdentityResultIndInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnBluetoothIdentifyResultIndMsg() entered "));

   if(NULL != cprBTIdentityResultIndInfo)
   {
      if(cprBTIdentityResultIndInfo->enStatus == e8CL_BT_IDENTIFY_SUCCESS)
      {
         ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnBluetoothIdentifyResultIndMsg(): HFP Connection with MD successful"));
      }
      else
      {
         ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnBluetoothIdentifyResultIndMsg(): HFP Connection with MD failed"));
      }
   }
}

/***************************************************************************
 ** FUNCTION:   t_Bool spi_tclBDCLBluetooth::vOnStartBluetoothAutoPairMsg()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vOnStartBluetoothAutoPairMsg(const trBdclStartBluetoothAutopair* cprStartBTAutopairInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnStartBluetoothAutoPairMsg() entered"));

   if(cprStartBTAutopairInfo != NULL)
   {
      if(cprStartBTAutopairInfo->enOSType == e8CL_OSTYPE_ANDROID)
      {
         ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnStartBluetoothAutoPairMsg() received for Android phone. "));
		 vSetPairingStatus(e8DEVICE_PAIRING_REQUESTED);
		 //Sending HU is in Idle state
         vSendDevicePairingResponse(e8CL_BT_STATUS_IDLE);
      }
      else
      {
         ETG_TRACE_ERR(("[ERR]::spi_tclBDCLBluetooth::vOnStartBluetoothAutopairMsg() received for iPhone - no action taken "));
      }

   }

}

/***************************************************************************
 ** FUNCTION:   t_Bool spi_tclBDCLBluetooth::vSendIdentificationRequest()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vSendIdentificationRequest()
{
   ETG_TRACE_USR2(("[DESC]:vSendIdentificationRequest() entered"));
   t_Bool bRet = false;
   if ((NULL != m_cpoBTInterface)&& (NULL != m_poBluetoothSettings))
   {
         t_String szVehicleBTAddress;
         m_poBluetoothSettings->vGetVehicleBTAddress(szVehicleBTAddress);
         ETG_TRACE_USR2(("[DESC]:vSendIdentificationRequest: BT Address:%s ",szVehicleBTAddress.c_str()));
         bFormatBTToMacAddress(szVehicleBTAddress);
         bRet =  m_spoCmdBluetooth->bSendBluetoothStartIndentifyReq(szVehicleBTAddress);
   }
   ETG_TRACE_USR2(("spi_tclBDCLBluetooth::vSendIdentificationRequest() Result:%d ",bRet));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnMDInfo()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vOnMDInfo(const trBdclMDInfo& crfrMDInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnMDInfo entered "));
   ETG_TRACE_USR4(("[PARAM]::spi_tclBDCLBluetooth :: vOnMDInfo: MDBTaddress = %s", crfrMDInfo.szBTAddress.c_str()));
   m_szMDBTAddress = crfrMDInfo.szBTAddress;
   //Calls the Function bIsValidBTAddress() to check BT_MAC_address validation
   //returns bValidBTStatus true if the BT_MAC_address is valid
   t_Bool bValidBTStatus = bIsValidBTAddress(m_szMDBTAddress);

   if(bValidBTStatus != true)
   {
      ETG_TRACE_ERR(("[ERROR]::spi_tclBDCLBluetooth :: vOnMDInfo: MDBTaddress = %s is invalid ", crfrMDInfo.szBTAddress.c_str()));
   }
   //Calls the function to format MAC to BT address because we want to store the invalid address also into trBdclMDInfo DB(example: unknown)
   bFormatMacToBTAddress(m_szMDBTAddress);
   if( (NULL != m_cpoBTInterface ) && (bValidBTStatus == true) )
   {
      m_cpoBTInterface->vSetSelectedDevBTAddress(m_szMDBTAddress);
      //Sending BT MAC update (device name NULL as it is not updated by phone at this point). Mainly added for already connected BT devices
      m_cpoBTInterface->vSendBTDeviceInfo(
            m_cpoBTInterface->u32GetSelectedDevHandle(), "", m_szMDBTAddress);
   }
   //vRegisterBTDeviceNameCallback(m_szMDBTAddress);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnBluetoothOOBInfoMsg()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vOnBluetoothOOBInfoMsg(const trBdclBluetoothOOBInfo* cprInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnBluetoothOOBInfoMsg() entered "));

   if ((NULL != m_cpoBTInterface))
   {
	   //Sending BT name update
	   m_cpoBTInterface->vSendBTDeviceInfo(
			   m_cpoBTInterface->u32GetSelectedDevHandle(), cprInfo->szName, m_szMDBTAddress);
   }

   if (cprInfo->enBTStatus == e8CL_BT_STATUS_READY)
   {
      ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vOnBluetoothOOBInfoMsg() Sending HU ready status to MD "));
      vSetPairingStatus(e8DEVICE_PAIRING_RESPONSE_SENT);
      vSendDevicePairingResponse(e8CL_BT_STATUS_READY);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclBDCLBluetooth::bFormatBTToMacAddress()
 ***************************************************************************/
t_Bool spi_tclBDCLBluetooth::bFormatBTToMacAddress(t_String &rfszMACAddress)
{
   // Insert the ":" charecter in the address string begin from second position and after the two charecters.
   const t_U8 u8InitPos = 2;
   const t_U8 u8Offset = 3;
   t_Bool bStatus = false;

   //! If string is empty return false.
   if(! rfszMACAddress.empty())
   {
      bStatus = true;
   }
   
  std::transform(rfszMACAddress.begin(), rfszMACAddress.end(), rfszMACAddress.begin(), ::toupper);
   
   for(t_U8 u8Index = u8InitPos; u8Index < rfszMACAddress.length(); u8Index+=u8Offset)
   {
      rfszMACAddress.insert(u8Index,":");
   }

   ETG_TRACE_USR4(("bFormatBTToMacAddress::Formatted MAC address : %s", rfszMACAddress.c_str()));
   return bStatus;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclBDCLBluetooth::bFormatMacToBTAddress()
 ***************************************************************************/
t_Bool spi_tclBDCLBluetooth::bFormatMacToBTAddress(t_String &rfszMACAddress)
{
   t_Bool bStatus = false;
   //! If string is empty return false.
   if(! rfszMACAddress.empty())
   {
      bStatus = true;
   }
   rfszMACAddress.erase(std::remove(rfszMACAddress.begin(), rfszMACAddress.end(), ':'), rfszMACAddress.end());
   std::transform(rfszMACAddress.begin(), rfszMACAddress.end(), rfszMACAddress.begin(), ::toupper);

   ETG_TRACE_USR4(("bFormatMacToBTAddress():Formatted MAC address : %s", rfszMACAddress.c_str()));
   return bStatus;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclBDCLBluetooth::bIsValidBTAddress()
 ***************************************************************************/
t_Bool spi_tclBDCLBluetooth::bIsValidBTAddress(const t_String &rfszMACAddress)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::bIsValidBTAddress entered: check for BT address Validation "));
   const t_U8 u8Sample_MACAddressSize = 17;
   t_U8 u8Colon_Counter = 0;
   t_Bool bBTValidStatus = false;

   if( rfszMACAddress.empty())
   {
      return bBTValidStatus;
   }

   if ((u8Sample_MACAddressSize) != rfszMACAddress.size())
   {
      return bBTValidStatus;
   }

   for (t_U16 i = 0; i < rfszMACAddress.size(); i++)
   {
      if (('0' <= rfszMACAddress[i]) && (rfszMACAddress[i] <= '9'))
      {
         // numeric character
      }
      else if (('a' <= rfszMACAddress[i]) && (rfszMACAddress[i] <= 'f'))
      {
         // alphabetic character
      }
      else if (('A' <= rfszMACAddress[i]) && (rfszMACAddress[i] <= 'F'))
      {
        // alphabetic character
      }
      else if ((':' == rfszMACAddress[i]))
      {
         // Colon Character
         u8Colon_Counter++;
      }
      else
      {
         // invalid character
         return bBTValidStatus;
      }
   }

   if(u8Colon_Counter == 5)
   {
      bBTValidStatus = true;
   }

   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::bIsValidBTAddress left"));
   return bBTValidStatus;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vPostBTConnectionChangedMsg()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vPostBTConnectionChangedMsg(t_String szBTDeviceAddress,tenBTConnectionResult enBTConnResult)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vPostBTConnectionChangedMsg() entered: BTConnectionResult = %d, BTAddress = %s ",
         ETG_ENUM(BT_CONNECTION_RESULT, enBTConnResult), szBTDeviceAddress.c_str()));

   tenDevicePairingStatus enPairingStatus = enGetPairingStatus();
   //! If a MD has connected via BT after AutoPair sequence is completed then send HUOOBInfo(Connected) message to MD.
   if(( m_cpoBTInterface != NULL) && (e8BT_RESULT_CONNECTED == enBTConnResult)
		   && (szBTDeviceAddress == m_cpoBTInterface->szGetSelectedDevBTAddress())
		   && (enPairingStatus == e8DEVICE_PAIRING_RESPONSE_SENT) && (e8DEV_TYPE_CARLIFE == m_cpoBTInterface->enGetSelectedDeviceCategory()) )
   {
	   ETG_TRACE_USR2(("[DESC]::spi_tclBDCLBluetooth::vPostBTConnectionChangedMsg() Sending HU Connected status to MD "));
       vSetPairingStatus(e8DEVICE_PAIRING_NOT_STARTED);
         // HU and MD are successfully connected
       vSendDevicePairingResponse(e8CL_BT_STATUS_CONNECTED);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vPostBTPairingInfoMsg()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vPostBTPairingInfoMsg(trBTPairingRequestInfo rBTPairingReqInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vPostBTPairingInfoMsg() entered: BTAddress = %s ",
            rBTPairingReqInfo.szRemoteDevBTAddr.c_str()));
   ETG_TRACE_USR4(("[PARAM]:vPostBTPairingInfoMsg: PairingMethod = %d, PairingPin = %s ",
            ETG_ENUM(BTSET_PAIRING_METHOD, rBTPairingReqInfo.enPairingMethod),
            rBTPairingReqInfo.szPairingPin.c_str()));
   if ((NULL != m_cpoBTInterface)
      &&(e8DEV_TYPE_CARLIFE == m_cpoBTInterface->enGetSelectedDeviceCategory()) /*If BDCL device is active*/
      )
   {
      if ((rBTPairingReqInfo.szRemoteDevBTAddr == m_cpoBTInterface->szGetSelectedDevBTAddress())
          &&
          (scenBTSetPreferredBDCLPairingMethod == rBTPairingReqInfo.enPairingMethod)
          )
      {
         ETG_TRACE_USR2(("[DESC]:vPostBTPairingInfoMsg: Pairing request from BDCL device received. "));
         szPairingPin = ((false == (rBTPairingReqInfo.szPairingPin).empty())?rBTPairingReqInfo.szPairingPin : "");
      }
   }//if ((NULL != m_cpoBTInterface) && ...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vPostBTLimitationModeMsg()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo)
{
	ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vPostBTLimitationModeMsg() entered "));
	ETG_TRACE_USR2(("[DESC]::vPostBTLimitationModeMsg: "
			"Technology = %d, CommChannel = %d, ActionState = %d, BTAddress = %s ",
	ETG_ENUM(BTSET_TECHNOLOGY, rBTLimitationModeInfo.enTechnology),
	ETG_ENUM(BTSET_COMM_CHN, rBTLimitationModeInfo.enCommChannel),
	ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rBTLimitationModeInfo.enActionState),
	        rBTLimitationModeInfo.szBTAddress.c_str()));

	if ((NULL != m_cpoBTInterface)
	       &&
	       (e8BT_TECH_CARLIFE == rBTLimitationModeInfo.enTechnology)
	       &&
	       (e8BT_COMM_CHN_USB == rBTLimitationModeInfo.enCommChannel)
	       &&
	       (rBTLimitationModeInfo.szBTAddress == m_cpoBTInterface->szGetSelectedDevBTAddress())
	       )
	{
		switch (rBTLimitationModeInfo.enActionState)
	    {

		case e8BT_LIM_ACTION_STATE_ACTIVE:
		{
			ETG_TRACE_USR2(("[DESC]::vPostBTLimitationModeMsg: Carlife BTLIM mode activation is successful"));
	        //sending IdentificationRequest message to phone to start Autopair sequence
	        if(false == m_bIdentificationRequestSent)
	        {
	           	vSendIdentificationRequest();
	           	m_bIdentificationRequestSent = true;
	        }
	    }
	        break;
	    case e8BT_LIM_ACTION_STATE_ERROR:
	    {
	    	ETG_TRACE_USR2(("[DESC]::vPostBTLimitationModeMsg: Carlife BT mode Error"));
	    }
	        break;
	    default:
	        //! Add code
	        break;
	      }//switch (rBTLimitationMode.enActionState)
	}
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vPostBTServiceAvailabilityMsg()
 ***************************************************************************/
t_Void spi_tclBDCLBluetooth::vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable)
{
   ETG_TRACE_USR1(("spi_tclBDCLBluetooth::vPostBTServiceAvailabilityMsg entered bServiceAvailable = %d",ETG_ENUM(BOOL,bServiceAvailable)));
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;
   
   if( (true == bServiceAvailable) &&
       (NULL != m_cpoBTInterface) &&
       (e8DEVICE_CHANGE_DELAYED == m_cpoBTInterface->enGetSelectedDevStatus()) &&
       (e8DEV_TYPE_CARLIFE == m_cpoBTInterface->enGetSelectedDeviceCategory()) &&
       (e8BT_LIM_ACTION_UNKNOWN != m_enPendingBTLimAction))
   {
       spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
       if(NULL != poExtCmdBluetoothIntf)
       {
          sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
              m_cpoBTInterface->szGetSelectedDevBTAddress(), szBTDeviceName,
              m_enPendingBTLimTechnology,
              e8BT_COMM_CHN_USB,
              m_enPendingBTLimAction);
          if (true == sbBTLimModeRequested)
          {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
          }
          else
          {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
          }
          ETG_TRACE_USR4(("[PARAM]::spi_tclBDCLBluetooth::vPostBTServiceAvailabilityMsg() - SetLimitationMode result - %d", ETG_ENUM(BOOL, sbBTLimModeRequested)));
       }
   }
}
//lint �restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
