/***********************************************************************/
/*!
* \file  spi_tclBDCLCmdVideo.cpp
* \brief  Implementation of the Class spi_tclBDCLCmdVideo
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         rur1kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
21.04.2017  | Ramya Murthy          | Initial Version
13.03.2018  |  Ramya Murthy         | Added video thread prio configuration

\endverbatim
*************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <cstdio>
#include <bdcl/GstreamerVideoSink.h>
using namespace adit::bdcl;

#include "SPITypes.h"
#include "spi_tclBDCLMsgQInterface.h"
#include "spi_tclBDCLVideoDispatcher.h"
#include "spi_tclBDCLCmdVideo.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BDCLWRAPPER
      #include "trcGenProj/Header/spi_tclBDCLCmdVideo.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
static const t_Char* sczPipelineTemplate =
         "vpudec low-latency=true frame-plus=2 framedrop=false framerate-nu=%d dis-reorder=true ! gst_apx_sink display-width=%d display-height=%d layer-id=%d surface-id=%d sync=false qos=false max-lateness=3000000000";

static trBdclVideoConfig srVideoConfig;

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/


/***************************************************************************
 ** FUNCTION:   spi_tclBDCLCmdVideo()
 ***************************************************************************/
spi_tclBDCLCmdVideo::spi_tclBDCLCmdVideo():
         m_enVidState(e8CL_VID_UNINITIALISED),
         m_bIsVideoPauseEnabled(false),
         m_poVideoSettings(NULL)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::spi_tclBDCLCmdVideo entered "));
}

/***************************************************************************
 ** FUNCTION:   ~spi_tclBDCLCmdVideo()
 ***************************************************************************/
spi_tclBDCLCmdVideo::~spi_tclBDCLCmdVideo()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::~spi_tclBDCLCmdVideo entered "));
   m_poVideoSettings = NULL;
}

/***************************************************************************
 ** FUNCTION:   t_Bool bInitialize()
 ***************************************************************************/
t_Bool spi_tclBDCLCmdVideo::bInitialize(const trBdclVideoConfig& corfrVideoConfig)
{
   m_oEndpointLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::bInitialize entered "));

   t_Bool bResult = false;

   t_SptrBDCLProxyManager spoProxyManager = spi_tclBDCLProxyManager::getInstance();
   t_SptrBDCLCoreDataIntf spoCoreDataIntf = (spoProxyManager)?
            (spoProxyManager->spoGetCoreDataIntf()) : (nullptr);
   t_CoreCallbackDealer* poCallbackDealer = (spoCoreDataIntf)?
            (spoCoreDataIntf->poGetCoreCallbackDealer()) : (NULL);
   SPI_NORMAL_ASSERT(NULL == poCallbackDealer);

   m_spoVideoProxy = (spoProxyManager)? (spoProxyManager->spoGetVideoProxyInstance()) : (nullptr);
   SPI_NORMAL_ASSERT(!m_spoVideoProxy);

   if ((m_spoVideoProxy) && (NULL != poCallbackDealer))
   {
      srVideoConfig = corfrVideoConfig;

      m_spoVideoSink = std::move(t_SptrBDCLVideoSink(new adit::bdcl::GstreamerVideoSink(this, poCallbackDealer)));
      SPI_NORMAL_ASSERT(!m_spoVideoSink);

      if (m_spoVideoSink)
      {
         vRegisterCallbacks();

         vSetVideoSinkConfig(corfrVideoConfig);

         ETG_TRACE_USR2(("[DESC] Initializing video endpoint "));
         bResult = m_spoVideoSink->initialize();

         m_enVidState = (bResult)? (e8CL_VID_INITIALISED) : (e8CL_VID_UNINITIALISED);
      }
   }

   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[DESC]::spi_tclBDCLCmdVideo::bInitialize left with result = %d ", ETG_ENUM(BOOL, bResult)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:   t_Void vUninitialize()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vUninitialize()
{
   m_oEndpointLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vUninitialize entered "));

   if ((m_spoVideoSink) && (e8CL_VID_UNINITIALISED != m_enVidState))
   {
      m_spoVideoSink->teardown();
      ETG_TRACE_USR2(("[DESC] Video endpoint teardown complete "));
   }
   else
   {
      ETG_TRACE_USR2(("[DESC] Video endpoint was not initialised. Hence teardown is not called. "));
   }
   m_enVidState = e8CL_VID_UNINITIALISED;
   m_spoVideoSink = nullptr;
   m_spoVideoProxy = nullptr;

   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vUninitialize left "));
}

/***************************************************************************
 ** FUNCTION:   t_Void vPrepareVideo()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vPrepareVideo()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vPrepareVideo entered "));

   if (m_spoVideoSink)
   {
      ETG_TRACE_USR2(("[DESC] Configuring video stream "));
      m_spoVideoSink->configureStream();
   }

   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vPrepareVideo left "));
}

/***************************************************************************
 ** FUNCTION:   t_Void vStartVideo()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vStartVideo()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vStartVideo entered: CurVidState = %d ", m_enVidState));

   if (m_spoVideoSink)
   {
      switch (m_enVidState)
      {
         case e8CL_VID_INITIALISED:
         {
            ETG_TRACE_USR4(("[DESC]: Video is not yet started. Requesting video playback start"));
            m_enVidState = e8CL_VID_REQUESTED;
            m_spoVideoSink->playbackStart();
         }
            break;
         case e8CL_VID_REQUESTED:
         {
            ETG_TRACE_USR4(("[DESC]: Video is already requested. Ignoring this request"));
         }
            break;
         case e8CL_VID_STREAMING:
         {
            ETG_TRACE_USR4(("[DESC]: Video is already started. Mocking video playback start"));
            onFirstFrameRendered();
         }
            break;
         case e8CL_VID_PAUSED:
         {
            m_enVidState = e8CL_VID_STREAMING;
            m_spoVideoSink->playbackStart();
            ETG_TRACE_USR4(("[DESC]: Video playback requested. Mocking video playback start"));
            onFirstFrameRendered();
         }
            break;
         default:
            ETG_TRACE_ERR(("[ERR]: Cannot process request when video state is %d", m_enVidState));
            break;
      }//switch (m_enVidState)
   }//if (m_spoVideoSink)
}

/***************************************************************************
 ** FUNCTION:   t_Void vPauseVideo()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vPauseVideo()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vPauseVideo entered "));

   if (m_bIsVideoPauseEnabled)
   {
      if (m_spoVideoSink)
      {
         m_spoVideoSink->playbackPause();
         m_enVidState = e8CL_VID_PAUSED;
      }
   }
}

/***************************************************************************
 ** FUNCTION:   void onFirstFrameRendered()
 ***************************************************************************/
void spi_tclBDCLCmdVideo::onFirstFrameRendered()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::onFirstFrameRendered entered "));

   m_enVidState = e8CL_VID_STREAMING;

   spi_tclBDCLMsgQInterface *poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLVideoPlaybackStartMsg oMsg;
      poMsgQInterface->bWriteMsgToQ(&oMsg, sizeof(oMsg));
   }//if (NULL != poMsgQInterface)
}

/***************************************************************************
 ** FUNCTION:   void onVideoEncoderInitDone()
 ***************************************************************************/
void spi_tclBDCLCmdVideo::onVideoEncoderInitDone(unsigned int inWidth,
         unsigned int inHeight, unsigned int inFramerate)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::onVideoEncoderInitDone entered "));

   if (m_spoVideoSink)
   {
      //TODO - Remove below configuration once ADIT implements storing the config
      ETG_TRACE_USR2(("[DESC] Setting received configuration "));

      trBdclVideoConfig rNewVideoConfig;
      rNewVideoConfig = srVideoConfig;
      //TODO - Currently disabled setting the video configuration received from MD
      //To be enabled once video resolution handling is confirmed
      //rNewVideoConfig.u32ScreenWidth = inWidth;
      //rNewVideoConfig.u32ScreenHeight = inHeight;
      rNewVideoConfig.u8Fps = inFramerate;

      t_String szVideoPipeline = szPrepareVideoPipeline(srVideoConfig);
      m_spoVideoSink->setConfigItem("gstreamer-video-pipeline", szVideoPipeline);

      m_spoVideoSink->setConfigItem("video-width", std::to_string(rNewVideoConfig.u32ScreenWidth));
      m_spoVideoSink->setConfigItem("video-height", std::to_string(rNewVideoConfig.u32ScreenHeight));
      m_spoVideoSink->setConfigItem("video-fps", std::to_string(rNewVideoConfig.u8Fps));
   }

   spi_tclBDCLMsgQInterface *poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLVideoEncoderInitDoneMsg oMsg;
      oMsg.m_u32Width = static_cast<t_U32>(inWidth);
      oMsg.m_u32Height = static_cast<t_U32>(inHeight);
      oMsg.m_u32Framerate = static_cast<t_U32>(inFramerate);

      poMsgQInterface->bWriteMsgToQ(&oMsg, sizeof(oMsg));
   }//if (NULL != poMsgQInterface)
}

/***************************************************************************
 ** FUNCTION:   void onConnectException()
 ***************************************************************************/
void spi_tclBDCLCmdVideo::onConnectException(tBdclConnectException inConnectException)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::onConnectException entered: %d ", inConnectException));

   //TODO
}

/***************************************************************************
 ** FUNCTION:   void onError()
 ***************************************************************************/
void spi_tclBDCLCmdVideo::onError(tBdclErrorCodes inErrorCode)
{
   tenBdclCarLifeError enError = static_cast<tenBdclCarLifeError>(inErrorCode);
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::onError entered: %d", ETG_ENUM(BDCL_ERROR, enError)));

   //TODO
}

/***************************************************************************
 ** FUNCTION:   t_Void vHeartBeatMsg()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vOnHeartBeatCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vOnHeartBeatCb entered "));

   spi_tclBDCLMsgQInterface *poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLVideoHeartBeatMsg oMsg;
      poMsgQInterface->bWriteMsgToQ(&oMsg, sizeof(oMsg));
   }//if (NULL != poMsgQInterface)
}

/***************************************************************************
 ** FUNCTION:   t_Void vOnForegroundCb()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vOnForegroundCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vOnForegroundCb entered"));

   spi_tclBDCLMsgQInterface *poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLVideoForegroundMsg oForegroundMsg;
      oForegroundMsg.m_bForeground = true;
      poMsgQInterface->bWriteMsgToQ(&oForegroundMsg, sizeof(oForegroundMsg));
   }//if (NULL != poMsgQInterface)
}

/***************************************************************************
 ** FUNCTION:   t_Void vOnBackgroundCb()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vOnBackgroundCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vOnBackgroundCb entered"));

   spi_tclBDCLMsgQInterface *poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLVideoForegroundMsg oForegroundMsg;
      oForegroundMsg.m_bForeground = false;
      poMsgQInterface->bWriteMsgToQ(&oForegroundMsg, sizeof(oForegroundMsg));
   }//if (NULL != poMsgQInterface)
}

/***************************************************************************
 ** FUNCTION:   t_Void vOnGotoForegroundCb()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vOnGotoForegroundCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vOnGotoForegroundCb "));

   spi_tclBDCLMsgQInterface *poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLVideoGotoForegroundMsg oGotoForegroundMsg;
      poMsgQInterface->bWriteMsgToQ(&oGotoForegroundMsg, sizeof(oGotoForegroundMsg));
   }//if (NULL != poMsgQInterface)
}

/***************************************************************************
 ** FUNCTION:   t_Void vOnScreenOnCb()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vOnScreenOnCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vOnScreenOnCb "));
   //TODO - send via dispatcher if msg is used
}

/***************************************************************************
 ** FUNCTION:   t_Void vOnScreenOffCb()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vOnScreenOffCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vOnScreenOffCb "));
   //TODO - send via dispatcher if msg is used
}

/***************************************************************************
 ** FUNCTION:   t_Void vOnScreenUserPresentCb()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vOnScreenUserPresentCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vOnScreenUserPresentCb "));
   //TODO - send via dispatcher if msg is used
}

/***************************************************************************
 ** FUNCTION:   t_Void vOnGotoDeskTopCb()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vOnGotoDeskTopCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vOnGotoDeskTopCb "));

   spi_tclBDCLMsgQInterface *poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLVideoGotoDesktopMsg oGotoDesktopMsg;
      poMsgQInterface->bWriteMsgToQ(&oGotoDesktopMsg, sizeof(oGotoDesktopMsg));
   }//if (NULL != poMsgQInterface)
}

/***************************************************************************
 ** FUNCTION:  t_Bool vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vRegisterCallbacks()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vRegisterCallbacks entered"));

   if (m_spoVideoProxy)
   {
      trBdclVideoCbs rVideoCbs;
      rVideoCbs.fvOnHeartBeat = vOnHeartBeatCb;
      rVideoCbs.fvOnGotoDeskTop = vOnGotoDeskTopCb;
      rVideoCbs.fvOnForeground = vOnForegroundCb;
      rVideoCbs.fvOnBackground = vOnBackgroundCb;

      //TODO - check if below callbacks are required
      /*rVideoCbs.fvOnGotoForeground = vOnGotoForegroundCb;
      VideoCbs.fvOnScreenOn = vOnScreenOnCb;
      rVideoCbs.fvOnScreenOff = vOnScreenOffCb;
      rVideoCbs.fvOnScreenUserPresent = vOnScreenUserPresentCb;*/

      m_spoVideoProxy->vRegisterVideoCallbacks(rVideoCbs);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool szPrepareVideoPipeline()
 ***************************************************************************/
t_String spi_tclBDCLCmdVideo::szPrepareVideoPipeline(const trBdclVideoConfig& corfrVideoConfig)
{

   //! Prepare GstreamerPipeline element value
   t_Char szVideoPipeline[MAX_KEYSIZE]={0};
   snprintf(szVideoPipeline,
            MAX_KEYSIZE,
            sczPipelineTemplate,
            corfrVideoConfig.u8Fps,
            corfrVideoConfig.u32ScreenWidth,
            corfrVideoConfig.u32ScreenHeight,
            corfrVideoConfig.u16LayerID,
            corfrVideoConfig.u16SurfaceID);

   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::szPrepareVideoPipeline left with %s", szVideoPipeline));
   return szVideoPipeline;
}

/***************************************************************************
 ** FUNCTION:  t_Void vSetVideoSinkConfig()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vSetVideoSinkConfig(const trBdclVideoConfig& corfrVideoConfig)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vSetVideoSinkConfig entered "));

   t_String szVideoPipeline = szPrepareVideoPipeline(corfrVideoConfig);

   if ((m_spoVideoSink) && (NULL != m_poVideoSettings) && (!szVideoPipeline.empty()))
   {
      ETG_TRACE_USR2(("[DESC] Setting video pipeline configuration "));

      m_spoVideoSink->setConfigItem("gstreamer-video-pipeline", szVideoPipeline);
      m_spoVideoSink->setConfigItem("video-width", std::to_string(corfrVideoConfig.u32ScreenWidth));
      m_spoVideoSink->setConfigItem("video-height", std::to_string(corfrVideoConfig.u32ScreenHeight));
      m_spoVideoSink->setConfigItem("video-fps", std::to_string(corfrVideoConfig.u8Fps));

      std::map< t_String, t_String > mapVideoSinkConfig;
      m_poVideoSettings->vGetBdclVideoSinkConfig(mapVideoSinkConfig);

      if (!mapVideoSinkConfig.empty())
      {
         for (auto itr = mapVideoSinkConfig.begin(); itr != mapVideoSinkConfig.end(); ++itr)
         {
            if ((false == itr->first.empty()) && (false == itr->second.empty()))
            {
               m_spoVideoSink->setConfigItem(itr->first, itr->second);
            }
            else
            {
               ETG_TRACE_ERR(("[ERR]::vSetVideoSinkConfig: Skipping empty entry in map"));
            }
         }
      }

      ETG_TRACE_USR2(("[DESC] Video sink configuration complete"));
   } //if ((m_spoVideoSink) && ...)
}

/***************************************************************************
 ** FUNCTION:  t_Void vSetVideoSettingsInstance()
 ***************************************************************************/
t_Void spi_tclBDCLCmdVideo::vSetVideoSettingsInstance(spi_tclVideoSettingsIntf* poVideoSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vSetVideoSettingsInstance entered "));
   if( NULL != poVideoSettingsIntf)
   {
	   ETG_TRACE_USR1(("spi_tclBDCLCmdVideo::vSetVideoSettingsInstance: setting m_poVideoSettings value "));
       m_poVideoSettings = poVideoSettingsIntf;
   }
}
///////////////////////////////////////////////////////////////////////////////
// <EOF>
