/***********************************************************************/
/*!
* \file  spi_tclBDCLLibProxy.cpp
* \brief  Implementation of the proxy library to Carlife stack
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         rur1kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
21.04.2017  | Ramya Murthy          | Initial Version

\endverbatim
*************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "spi_tclBDCLLibProxy.h"

using namespace adit::bdcl;

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BDCLWRAPPER
      #include "trcGenProj/Header/spi_tclBDCLLibProxy.cpp.trc.h"
   #endif
#endif


/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
 ** FUNCTION:   spi_tclBDCLLibProxy()
 ***************************************************************************/
spi_tclBDCLLibProxy::spi_tclBDCLLibProxy() :
   m_poCCarLifeLib(NULL), m_poCallbackDealer(NULL)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::spi_tclBDCLLibProxy entered "));

   m_poCCarLifeLib = CCarLifeLib::getInstance();
   SPI_NORMAL_ASSERT(NULL == m_poCCarLifeLib);

   m_poCallbackDealer = new t_CoreCallbackDealer();
   SPI_NORMAL_ASSERT(NULL == m_poCallbackDealer);

   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::spi_tclBDCLLibProxy left "));
}

/***************************************************************************
 ** FUNCTION:   ~spi_tclBDCLLibProxy()
 ***************************************************************************/
spi_tclBDCLLibProxy::~spi_tclBDCLLibProxy()
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::~spi_tclBDCLLibProxy entered "));

   RELEASE_MEM(m_poCallbackDealer);

   m_oLock.s16Lock();
   ETG_TRACE_USR2(("[DESC] calling carLifeLibDestory "));
   m_poCCarLifeLib->carLifeLibDestory();
   ETG_TRACE_USR2(("[DESC] carLifeLibDestory complete "));
   m_oLock.vUnlock();

   m_poCCarLifeLib = NULL;

   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::~spi_tclBDCLLibProxy left "));
}

/***************************************************************************
 ** FUNCTION:   t_Void vRegisterVRCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vRegisterVRCallbacks(trBdclVoiceRecognitionCbs rVoiceRecCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vRegisterAudioSinkCallbacks entered "));

   if (NULL != m_poCCarLifeLib)
   {
      m_poCCarLifeLib->vrRegisterStatusModule(rVoiceRecCbs.fvOnVRModuleStatus);
   }
}

/***************************************************************************
 ** FUNCTION:   t_Void vRegisterSessionCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vRegisterSessionCallbacks(trBdclSessionCbs rSessionCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vRegisterSessionCallbacks entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      m_poCCarLifeLib->cmdRegisterMDInfro(rSessionCbs.fvOnMDInfo);
      m_poCCarLifeLib->cmdRegisterModuleStatus(rSessionCbs.fvOnModuleStatus);
      m_poCCarLifeLib->cmdRegisterMdExit(rSessionCbs.fvOnExit);
      m_poCCarLifeLib->cmdRegisterFeatureConfigRequest (rSessionCbs.fvOnFeatureConfigRequest);
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendHUInfo()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendHUInfo(const trBdclHUInfo& corfrHUInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendHUInfo entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_HU_INFO rInfo;
      rInfo.os = corfrHUInfo.szOS;
      rInfo.manufacturer = corfrHUInfo.szManufacturer;
      rInfo.model = corfrHUInfo.szModel;
      rInfo.release = corfrHUInfo.szRelease;
      rInfo.board = corfrHUInfo.szBoard;
      rInfo.bootloader = corfrHUInfo.szBootloader;
      rInfo.brand = corfrHUInfo.szBrand;
      rInfo.cpu_abi = corfrHUInfo.szCpu_abi;
      rInfo.cpu_abi2 = corfrHUInfo.szCpu_abi2;
      rInfo.device = corfrHUInfo.szDevice;
      rInfo.display = corfrHUInfo.szDisplay;
      rInfo.fingerprint = corfrHUInfo.szFingerprint;
      rInfo.hardware = corfrHUInfo.szHardware;
      rInfo.host = corfrHUInfo.szHost;
      rInfo.cid = corfrHUInfo.szCid;
      rInfo.product = corfrHUInfo.szProduct;
      rInfo.serial = corfrHUInfo.szSerial;
      rInfo.codename = corfrHUInfo.szCodename;
      rInfo.incremental = corfrHUInfo.szIncremental;
      rInfo.sdk = corfrHUInfo.szSdk;
      rInfo.sdk_int = corfrHUInfo.u32Sdk;
      rInfo.token = corfrHUInfo.szToken;
      rInfo.btaddress = corfrHUInfo.szBTAddress;

      t_S32 s32Result = m_poCCarLifeLib->cmdHUInfro(&rInfo);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendHUInfo left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCmdModuleControl()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCmdModuleControl(t_U32 u32ModuleID,
      t_U32 u32StatusID)
{
   S_MODULE_STATUS_CONTROL* prCmdModuleCtrl = new S_MODULE_STATUS_CONTROL;
   if ((NULL != prCmdModuleCtrl) && (NULL != m_poCCarLifeLib))
   {
      prCmdModuleCtrl->moduleID = u32ModuleID;
      prCmdModuleCtrl->statusID = u32StatusID;
      if (0 != m_poCCarLifeLib->cmdModuleControl(prCmdModuleCtrl))
      {
         ETG_TRACE_ERR(("[ERR]::spi_tclBDCLLibProxy::vSendCmdModuleControl: "
               "Setting module %d to state %d failed", u32ModuleID, u32StatusID));
      }
   }
   RELEASE_MEM(prCmdModuleCtrl);
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendFeatureConfigList()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendFeatureConfigList(const tBdclFeatureConfigList& corfrFeatConfigList)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendFeatureConfigList entered "));
   if ((NULL != m_poCCarLifeLib))
   {
      S_FEATURE_CONFIG_LIST* prFeatConfigList = new(std::nothrow) S_FEATURE_CONFIG_LIST;
      if( NULL != prFeatConfigList)
      {
         prFeatConfigList->pFeatureConfig = NULL;
         prFeatConfigList->cnt = corfrFeatConfigList.size();

         for (t_U8 u8Index = 0; u8Index < corfrFeatConfigList.size(); ++u8Index)
         {
            S_FEATURE_CONFIG* prFeatureConfig = new(std::nothrow) S_FEATURE_CONFIG;
            if( NULL != prFeatureConfig)
            {
               prFeatureConfig->key = corfrFeatConfigList[u8Index].szKey;
               prFeatureConfig->value = corfrFeatConfigList[u8Index].u32Value;
               prFeatureConfig->pNext = prFeatConfigList->pFeatureConfig;
               prFeatConfigList->pFeatureConfig = prFeatureConfig;
            }
         }
         m_poCCarLifeLib->cmdFeatureConfigResponse(prFeatConfigList);
         while (NULL != prFeatConfigList->pFeatureConfig)
         {
            S_FEATURE_CONFIG* prNextFeatConfig = prFeatConfigList->pFeatureConfig->pNext;
            delete (prFeatConfigList->pFeatureConfig);
            prFeatConfigList->pFeatureConfig = prNextFeatConfig;
         }
         delete prFeatConfigList;
      }
   }
}

/***************************************************************************
 ** FUNCTION:   t_Void vRegisterVideoCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vRegisterVideoCallbacks(trBdclVideoCbs rVideoCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vRegisterVideoCallbacks entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      m_poCCarLifeLib->videoRegisterHeartBeat(rVideoCbs.fvOnHeartBeat);
      m_poCCarLifeLib->cmdRegisterForeground(rVideoCbs.fvOnForeground);
      m_poCCarLifeLib->cmdRegisterBackground(rVideoCbs.fvOnBackground);
      m_poCCarLifeLib->cmdRegisterRequestGoToForeground(rVideoCbs.fvOnGotoForeground);

      //TODO - check if below callbacks are required
      m_poCCarLifeLib->cmdRegisterScreenOn(rVideoCbs.fvOnScreenOn);
      m_poCCarLifeLib->cmdRegisterScreenOff(rVideoCbs.fvOnScreenOff);
      m_poCCarLifeLib->cmdRegisterScreenUserPresent(rVideoCbs.fvOnScreenUserPresent);
      m_poCCarLifeLib->cmdRegisterGoToDeskTop(rVideoCbs.fvOnGotoDeskTop);
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendKeyCode()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendKeyCode(t_U32 u32KeyCode)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendKeyCode entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_TOUCH_CAR_HARD_KEY_CODE rKeyCode;
      rKeyCode.keycode = u32KeyCode;
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendKeyCode Keycode sent to phone is = %d", u32KeyCode));

      t_S32 s32Result = m_poCCarLifeLib->ctrlTouchCarHardKeyCode(&rKeyCode);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendKeyCode left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vRegisterBTCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vRegisterBTCallbacks(trBdclBluetoothCbs rBluetoothCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vRegisterBTCallbacks entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      m_poCCarLifeLib->cmdRegisterMDBTOobInfo(rBluetoothCbs.fvOnBluetoothOOBInfo);
      m_poCCarLifeLib->cmdRegisterStartBtAutoPairRequest(rBluetoothCbs.fvOnBluetoothStartAutoPairRequest);
      m_poCCarLifeLib->cmdRegisterBTHfpStatusRequest(rBluetoothCbs.fvOnBluetoothHFPStatusRequest);
      m_poCCarLifeLib->cmdRegisterBTIdentifyResultInd(rBluetoothCbs.fvOnBluetoothIdentifyResultInd);
	  m_poCCarLifeLib->cmdRegisterBtHfpRequest(rBluetoothCbs.fvOnBluetoothHFPRequest);

   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendBTOOBInfo()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendHUBTOOBInfo(trBdclBluetoothOOBInfo* prOOBInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendHUBTOOBInfo entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_BT_OOB_INFO rBTInfo;
      rBTInfo.address = prOOBInfo->szBTAddress;//HU BT adress
      rBTInfo.passKey = prOOBInfo->szPassKey;
      rBTInfo.name = prOOBInfo->szName; //HU BT name
      rBTInfo.status = static_cast<tenBdclHUBTStatus>(prOOBInfo->enBTStatus);
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy:: vSendHUBTOOBInfo BT address = %s", prOOBInfo->szBTAddress.c_str()));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendHUBTOOBInfo Pass Key = %s", prOOBInfo->szPassKey.c_str()));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendHUBTOOBInfo Name = %s", prOOBInfo->szName.c_str()));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendHUBTOOBInfo BT status = %d", ETG_ENUM(BDCL_HU_BT_STATUS, prOOBInfo->enBTStatus)));
      t_S32 s32Result = m_poCCarLifeLib->cmdHUBTOobInfo(&rBTInfo);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendHUBTOOBInfo left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendBTHfpIndication()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendBTHfpIndication(trBdclBluetoothHFPIndication* prHfpIndicationInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendBTHfpIndication entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_BT_HFP_INDICATION rHFPIndication;
      rHFPIndication.state = static_cast<carlifeBTHfpIndicationIndicationType>(prHfpIndicationInfo->enIndicationType);
      t_S32 s32Result = m_poCCarLifeLib->cmdBtHfpIndication(&rHFPIndication);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendBTHfpIndication left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendBTHfpConnection()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendBTHfpConnection(trBdclBluetoothHFPConn* prHfpConnInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendBTHfpConnection entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      //TODO
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendBTHfpResponse()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendBTHfpResponse(trBdclBluetoothHFPResponse* prHfpRespInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendBTHfpResponse entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      //TODO
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendBTHfpStatusResponse()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendBTHfpStatusResponse(trBdclBluetoothHFPStatusResponse* prHfpStatusRespInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendBTHfpStatusResponse entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      //TODO
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendBTStartIdentifyRequest()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendBTStartIdentifyRequest(t_String szAddress)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendBTStartIdentifyRequest entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_BT_START_IDENTIFY_REQ rBTIdentifyRequest;
      rBTIdentifyRequest.address = szAddress;
 	   ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendBTStartIdentifyRequest BT address = %s", szAddress.c_str()));
      t_S32 s32Result = m_poCCarLifeLib->cmdBTStartIdentifyReq(&rBTIdentifyRequest);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendBTStartIdentifyRequest left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vRegisterCarDataCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vRegisterCarDataCallbacks(trBdclCarDataCbs rCarDataCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vRegisterCarDataCallbacks entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      m_poCCarLifeLib->cmdRegisterCarDataSubscribe(rCarDataCbs.fvOnCarDataSubscribe);
      m_poCCarLifeLib->cmdRegisterCarDataSubscribeStart(rCarDataCbs.fvOnCarDataSubcribeStart);
      m_poCCarLifeLib->cmdRegisterCarDataSubscribeStop(rCarDataCbs.fvOnCarDataSubscribeStop);
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCarGPS()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCarGPS(trBdclCarGPS rCarGpsInfo)
{
   //ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendCarGPS entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_CAR_GPS rGPS;
      rGPS.antennaState = rCarGpsInfo.s32AntennaState;
      rGPS.signalQuality = rCarGpsInfo.s32SignalQuality;
      rGPS.latitude = rCarGpsInfo.s32Latitude;
      rGPS.longitude = rCarGpsInfo.s32Longitude;
      rGPS.height = rCarGpsInfo.s32Height;
      rGPS.speed = rCarGpsInfo.u32Speed;
      rGPS.heading = rCarGpsInfo.u32Heading;
      rGPS.year = rCarGpsInfo.u32Year;
      rGPS.month = rCarGpsInfo.u32Month;
      rGPS.day = rCarGpsInfo.u32Day;
      rGPS.hrs = rCarGpsInfo.u32Hrs;
      rGPS.min = rCarGpsInfo.u32Min;
      rGPS.sec = rCarGpsInfo.u32Sec;
      rGPS.fix = rCarGpsInfo.u32Fix;
      rGPS.hdop = rCarGpsInfo.u32hdop;
      rGPS.pdop = rCarGpsInfo.u32Pdop;
      rGPS.vdop = rCarGpsInfo.u32Vdop;
      rGPS.satsUsed = rCarGpsInfo.u32SatsUsed;
      rGPS.satsVisible = rCarGpsInfo.u32SatsVisible;
      rGPS.horPosError = rCarGpsInfo.u32HorPosError;
      rGPS.vertPosError = rCarGpsInfo.u32VertPosError;
      rGPS.northSpeed = rCarGpsInfo.s32NorthSpeed;
      rGPS.eastSpeed = rCarGpsInfo.s32EastSpeed ;
      rGPS.vertSpeed = rCarGpsInfo.s32VertSpeed;
      rGPS.timeStamp = rCarGpsInfo.u64TimeStamp;
      t_S32 s32Result = m_poCCarLifeLib->cmdCarGPS(&rGPS);
      //ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendCarGPS left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCarAcceleration()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCarAcceleration(trAccSensorData rCarAccelerationInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendCarAcceleration entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_CAR_ACCELERATION rAccSensorData;
      rAccSensorData.accZ= rCarAccelerationInfo.fAccZ;
      rAccSensorData.accY = rCarAccelerationInfo.fAccY;
      rAccSensorData.accX = rCarAccelerationInfo.fAccX;
      rAccSensorData.timeStamp = rCarAccelerationInfo.u64AccTimeStamp;
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarAcceleration AccSensorData.accZ = %f", rCarAccelerationInfo.fAccZ));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarAcceleration AccSensorData.accY = %f", rCarAccelerationInfo.fAccY));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarAcceleration AccSensorData.accX = %f", rCarAccelerationInfo.fAccX));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarAcceleration timeStamp = %s", std::to_string(rCarAccelerationInfo.u64AccTimeStamp).c_str()));
      t_S32 s32Result = m_poCCarLifeLib->cmdCarAcceleration(&rAccSensorData);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendCarAcceleration left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCarGyro()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCarGyro(trGyroSensorData rCarGyroInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendCarGyro entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_CAR_GYROSCOPE rCarGyroscope;
     // rCarGyroscope.gyroType = rCarGyroInfo.GyroType;
      rCarGyroscope.gyroZ = rCarGyroInfo.fGyroZ;
      rCarGyroscope.gyroY = rCarGyroInfo.fGyroY;
      rCarGyroscope.gyroX = rCarGyroInfo.fGyroX;
      rCarGyroscope.timeStamp = rCarGyroInfo.u64GyroTimeStamp;
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarGyro CarGyroscope.gyroZ = %f", rCarGyroInfo.fGyroZ));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarGyro CarGyroscope.gyroY = %f", rCarGyroInfo.fGyroY));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarGyro CarGyroscope.gyroX = %f", rCarGyroInfo.fGyroX));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarGyro CarGyroscope.gyroZ = %s", std::to_string(rCarGyroInfo.u64GyroTimeStamp).c_str()));
      t_S32 s32Result = m_poCCarLifeLib->cmdCarGyroscope(&rCarGyroscope);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendCarGyro left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCarOil()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCarOil(trBdclCarOil rCarOilInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendCarOil entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_CAR_OIL rCarOil;
      rCarOil.level = rCarOilInfo.s32level;
      rCarOil.range = rCarOilInfo.s32range;
      rCarOil.lowFullWarning = rCarOilInfo.bLowFuelWarning;
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarOil Oil level = %d", rCarOilInfo.s32level));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarOil Oil range = %d", rCarOilInfo.s32range));
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarOil Low Fuel Warning = %d", rCarOilInfo.bLowFuelWarning));
      t_S32 s32Result = m_poCCarLifeLib->cmdCarOil(&rCarOil);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendCarOil left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCarVelocity()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCarVelocity(trBdclCarVelocity rCarVelocityInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendCarVelocity entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_CAR_VELOCITY rCarVelocity;
      rCarVelocity.speed = rCarVelocityInfo.u32speed;
      //rCarVelocity.timeStamp = rCarVelocityInfo.u64timestamp;
	  ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarVelocity Car velocity speed = %d", rCarVelocityInfo.u32speed));
      t_S32 s32Result = m_poCCarLifeLib->cmdCarVelocity(&rCarVelocity);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendCarVelocity left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCarGearState()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCarGearState(tenBdclGearState rGearStateInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendCarGearState entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_GEAR_INFO rGearInfo;
      rGearInfo.gear = rGearStateInfo;
 	   ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLLibProxy::vSendCarGearState Car Gear state info = %d", rGearStateInfo));
      t_S32 s32Result = m_poCCarLifeLib->cmdCarDataGear(&rGearInfo);
      ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendCarGearState left with result = %d", s32Result));
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendDataSubscribeDone()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendDataSubscribeDone(const tBdclCarDataInfoList rCarDataInfoList)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendDataSubscribeDone entered: Number of modules = %d", rCarDataInfoList.size()));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      S_VEHICLE_INFO_LIST* prVehicleInfoList = new(std::nothrow) S_VEHICLE_INFO_LIST;

      if (NULL != prVehicleInfoList)
      {
         prVehicleInfoList->pVehicleInfo = NULL;
         prVehicleInfoList->cnt = rCarDataInfoList.size();

         for (t_U8 u8Index = 0; u8Index < rCarDataInfoList.size(); ++u8Index)
         {
            ETG_TRACE_USR2(("[PARAM]:spi_tclBDCLLibProxy::vSendDataSubscribeDone() Sent Module ID = %d, "
                     "Support Flag = %d, Frequency = %d", rCarDataInfoList[u8Index].enCarDataModuleID,
                     rCarDataInfoList[u8Index].s32SupportFlag, rCarDataInfoList[u8Index].u32Frequency));

            S_VEHICLE_INFO* prVehicleInfo = new(std::nothrow) S_VEHICLE_INFO;
            if (NULL != prVehicleInfo)
            {
               prVehicleInfo->moduleID = rCarDataInfoList[u8Index].enCarDataModuleID;
               prVehicleInfo->supportFlag = rCarDataInfoList[u8Index].s32SupportFlag;
               prVehicleInfo->frequency = rCarDataInfoList[u8Index].u32Frequency;
               prVehicleInfo->pNext = prVehicleInfoList->pVehicleInfo;
               prVehicleInfoList->pVehicleInfo = prVehicleInfo;
            }
         }
		
         t_S32 s32Result = m_poCCarLifeLib->cmdCarDataSubscribeDone(prVehicleInfoList);

         //ETG_TRACE_USR4(("[DESC]::vSendDataSubscribeDone: Doing cleanup "));
         while (NULL != prVehicleInfoList->pVehicleInfo)
         {
            S_VEHICLE_INFO* prNextVehicleModule = prVehicleInfoList->pVehicleInfo->pNext;
            delete (prVehicleInfoList->pVehicleInfo);
            prVehicleInfoList->pVehicleInfo = prNextVehicleModule;
         }
         delete prVehicleInfoList;
        
         ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendDataSubscribeDone left with result = %d", s32Result));
      }
      //ETG_TRACE_USR4(("[DESC]::vSendFeatureConfigInfo: Cleanup complete "));

   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vRegisterMediaDataCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vRegisterMediaDataCallbacks(trBdclMediaDataCbs rMediaDataCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vRegisterMediaDataCallbacks entered "));

   m_oLock.s16Lock();
   if (NULL != m_poCCarLifeLib)
   {
      m_poCCarLifeLib->cmdRegisterMediaInfo(rMediaDataCbs.fvOnMediaInfo);
      m_poCCarLifeLib->cmdRegisterMediaProgressBar(rMediaDataCbs.fvOnProgressBarInfo);
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCarLifeDataSubscribe()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCarLifeDataSubscribe(const tBdclCarLifeInfoList& corfrCarLifeInfoList)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendCarLifeDataSubscribe entered: Number of modules = %d", corfrCarLifeInfoList.size()));

   m_oLock.s16Lock();
   if (m_poCCarLifeLib)
   {
      S_SUBSCRIBE_MOBILE_CARLIFE_INFO_LIST* prCarLifeInfoList = new(std::nothrow)S_SUBSCRIBE_MOBILE_CARLIFE_INFO_LIST;

      if(NULL != prCarLifeInfoList)
      {
          prCarLifeInfoList->pMobileCarLifeInfo = NULL;
          prCarLifeInfoList->cnt = corfrCarLifeInfoList.size();

          for (t_U8 u8Index = 0; u8Index < corfrCarLifeInfoList.size(); ++u8Index)
          {
             S_MOBILE_CARLIFE_INFO* prCarLifeInfo = new(std::nothrow)S_MOBILE_CARLIFE_INFO;
             if(NULL != prCarLifeInfo)
             {
                 prCarLifeInfo->moduleID = static_cast<carlifeSubscribeMobileCarLifeInfoModuleID>(
                          corfrCarLifeInfoList[u8Index].enCarlifeDataModuleID);
                 prCarLifeInfo->supportFlag = corfrCarLifeInfoList[u8Index].bSupportFlag;
                 prCarLifeInfo->frequency = corfrCarLifeInfoList[u8Index].u32Frequency;
                 prCarLifeInfo->pNext = prCarLifeInfoList->pMobileCarLifeInfo;
                 prCarLifeInfoList->pMobileCarLifeInfo = prCarLifeInfo;
             }
          }

          t_S32 s32Result = m_poCCarLifeLib->cmdCarLifeDataSubscribe(prCarLifeInfoList);

          //ETG_TRACE_USR4(("[DESC]::vSendCarLifeDataSubscribe: Doing cleanup "));
          while (NULL != prCarLifeInfoList->pMobileCarLifeInfo)
          {
             S_MOBILE_CARLIFE_INFO* prNextCarLifeModule = prCarLifeInfoList->pMobileCarLifeInfo->pNext;
             delete (prCarLifeInfoList->pMobileCarLifeInfo);
             prCarLifeInfoList->pMobileCarLifeInfo = prNextCarLifeModule;
          }
          //ETG_TRACE_USR4(("[DESC]::vSendCarLifeDataSubscribe: Cleanup complete "));
          delete prCarLifeInfoList;
          ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendCarLifeDataSubscribe left with result = %d", s32Result));
      }
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCarLifeDataSubscribeStart()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCarLifeDataSubscribeStart(const tBdclCarLifeInfoList& corfrCarLifeInfoStartList)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendCarLifeDataSubscribeStart entered: Number of modules = %d ", corfrCarLifeInfoStartList.size()));

   m_oLock.s16Lock();
   if (m_poCCarLifeLib)
   {
      S_SUBSCRIBE_MOBILE_CARLIFE_INFO_LIST* prCarLifeInfoList = new(std::nothrow)S_SUBSCRIBE_MOBILE_CARLIFE_INFO_LIST;

      if(NULL != prCarLifeInfoList)
      {
          prCarLifeInfoList->pMobileCarLifeInfo = NULL;
          prCarLifeInfoList->cnt = corfrCarLifeInfoStartList.size();

          for (t_U8 u8Index = 0; u8Index < corfrCarLifeInfoStartList.size(); ++u8Index)
          {
             S_MOBILE_CARLIFE_INFO* prCarLifeInfo = new(std::nothrow)S_MOBILE_CARLIFE_INFO;
             if(NULL != prCarLifeInfo)
             {
                 prCarLifeInfo->moduleID = static_cast<carlifeSubscribeMobileCarLifeInfoModuleID>(
                          corfrCarLifeInfoStartList[u8Index].enCarlifeDataModuleID);
                 prCarLifeInfo->supportFlag = corfrCarLifeInfoStartList[u8Index].bSupportFlag;
                 prCarLifeInfo->frequency = corfrCarLifeInfoStartList[u8Index].u32Frequency;
                 prCarLifeInfo->pNext = prCarLifeInfoList->pMobileCarLifeInfo;
                 prCarLifeInfoList->pMobileCarLifeInfo = prCarLifeInfo;
             }
          }

          t_S32 s32Result = m_poCCarLifeLib->cmdCarLifeDataSubscribeStart(prCarLifeInfoList);

          //ETG_TRACE_USR4(("[DESC]::vSendCarLifeDataSubscribe: Doing cleanup "));
          while (NULL != prCarLifeInfoList->pMobileCarLifeInfo)
          {
             S_MOBILE_CARLIFE_INFO* prNextCarLifeModule = prCarLifeInfoList->pMobileCarLifeInfo->pNext;
             delete (prCarLifeInfoList->pMobileCarLifeInfo);
             prCarLifeInfoList->pMobileCarLifeInfo = prNextCarLifeModule;
          }
          //ETG_TRACE_USR4(("[DESC]::vSendCarLifeDataSubscribeStart: Cleanup complete "));
          delete prCarLifeInfoList;
          ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendCarLifeDataSubscribeStart left with result = %d", s32Result));
      }
   }
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:   t_Void vSendCarLifeDataSubscribeStop()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vSendCarLifeDataSubscribeStop(const tBdclCarLifeInfoList& corfrCarLifeInfoStopList)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vSendCarLifeDataSubscribeStop entered: Number of modules = %d ", corfrCarLifeInfoStopList.size()));

   m_oLock.s16Lock();
   if (m_poCCarLifeLib)
   {
      S_SUBSCRIBE_MOBILE_CARLIFE_INFO_LIST* prCarLifeInfoList = new(std::nothrow)S_SUBSCRIBE_MOBILE_CARLIFE_INFO_LIST;

      if(NULL != prCarLifeInfoList)
      {
          prCarLifeInfoList->pMobileCarLifeInfo = NULL;
          prCarLifeInfoList->cnt = corfrCarLifeInfoStopList.size();

          for (t_U8 u8Index = 0; u8Index < corfrCarLifeInfoStopList.size(); ++u8Index)
          {
             S_MOBILE_CARLIFE_INFO* prCarLifeInfo = new(std::nothrow)S_MOBILE_CARLIFE_INFO;
             if(NULL != prCarLifeInfo)
             {
                 prCarLifeInfo->moduleID = static_cast<carlifeSubscribeMobileCarLifeInfoModuleID>(
                          corfrCarLifeInfoStopList[u8Index].enCarlifeDataModuleID);
                 prCarLifeInfo->supportFlag = corfrCarLifeInfoStopList[u8Index].bSupportFlag;
                 prCarLifeInfo->frequency = corfrCarLifeInfoStopList[u8Index].u32Frequency;
                 prCarLifeInfo->pNext = prCarLifeInfoList->pMobileCarLifeInfo;
                 prCarLifeInfoList->pMobileCarLifeInfo = prCarLifeInfo;
             }
          }

          t_S32 s32Result = m_poCCarLifeLib->cmdCarLifeDataSubscribeStop(prCarLifeInfoList);

          //ETG_TRACE_USR4(("[DESC]::vSendCarLifeDataSubscribe: Doing cleanup "));
          while (NULL != prCarLifeInfoList->pMobileCarLifeInfo)
          {
             S_MOBILE_CARLIFE_INFO* prNextCarLifeModule = prCarLifeInfoList->pMobileCarLifeInfo->pNext;
             delete (prCarLifeInfoList->pMobileCarLifeInfo);
             prCarLifeInfoList->pMobileCarLifeInfo = prNextCarLifeModule;
          }
          //ETG_TRACE_USR4(("[DESC]::vSendCarLifeDataSubscribeStop: Cleanup complete "));

          delete prCarLifeInfoList;
          ETG_TRACE_USR2(("[DESC]::spi_tclBDCLLibProxy::vSendCarLifeDataSubscribeStop left with result = %d", s32Result));
      }
   }
   m_oLock.vUnlock();
}
/***************************************************************************
 ** FUNCTION:   t_Void vRegisterNavigationCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLLibProxy::vRegisterNavigationCallbacks(trBdclNavigationCbs rNavigationCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLLibProxy::vRegisterNavigationCallbacks entered "));

   m_oLock.s16Lock();
   if (m_poCCarLifeLib)
   {
      m_poCCarLifeLib->cmdRegisterCarLifeDataSubscribeDone(rNavigationCbs.fvOnCarlifeSubscriptionDone);
      m_poCCarLifeLib->cmdRegisterNaviNextTurnInfo(rNavigationCbs.fvOnNaviNextTurnInfo);
      m_poCCarLifeLib->cmdRegisterNaviAssistantGuideInfo(rNavigationCbs.fvOnNaviAssitantGuideInfo);
   }
   m_oLock.vUnlock();
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
