/***********************************************************************/
/*!
* \file  spi_tclBDCLAudioSourceAdapt.h
* \brief  Implementation of the Class spi_tclBDCLAudioSourceAdapt
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         pok6kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
21.04.2017  | Ramya Murthy          | Initial Version
06.03.2018  | Ramya Murthy          | Added flag to track mic status, to avoid notiying mic close 
                                      when mic was not opened.

\endverbatim
*************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <bdcl/AilAudioSource.h>
using namespace adit::bdcl;

#include "spi_tclBDCLAudioSourceAdapt.h"
#include "spi_tclBDCLMsgQInterface.h"
#include "spi_tclBDCLAudioDispatcher.h"
#include "spi_tclBDCLProxyManager.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BDCLWRAPPER
      #include "trcGenProj/Header/spi_tclBDCLAudioSourceAdapt.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
#define DISPATCH_AUD_REQ(REQ_TYPE) \
   spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance(); \
   if (NULL != poMsgQInterface)  \
   {  \
      BDCLAudioRequestMsg oAudioMsg;   \
      oAudioMsg.m_rAudioRequest.enAudStream = e8CL_AUDTYPE_MIC; \
      oAudioMsg.m_rAudioRequest.enAudRequest = REQ_TYPE; \
      poMsgQInterface->bWriteMsgToQ(&oAudioMsg, sizeof(oAudioMsg)); \
   }//if (NULL != poMsgQinterface)

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static const t_String sczAudSourceDeviceNameKey="audio-capture-device";

static const t_String sczAudSourceChnKey="audio-source-channels";
static const t_String sczAudSourceSampleRateKey="audio-source-sampling-rate";
static const t_String sczAudSourceBitsPerSampleKey="audio-source-bits-per-sample";


/***************************************************************************
 ** FUNCTION:   spi_tclBDCLAudioSourceAdapt()
 ***************************************************************************/
spi_tclBDCLAudioSourceAdapt::spi_tclBDCLAudioSourceAdapt():m_bIsMicOpened(false),
         m_poAudioSettings(NULL)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::spi_tclBDCLAudioSourceAdapt entered "));
}

/***************************************************************************
 ** FUNCTION:  ~spi_tclBDCLAudioSourceAdapt()
 ***************************************************************************/
spi_tclBDCLAudioSourceAdapt::~spi_tclBDCLAudioSourceAdapt()
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::~spi_tclBDCLAudioSourceAdapt entered "));
   m_poAudioSettings = NULL;
   vUninitialize();
}

/***************************************************************************
 ** FUNCTION:  t_Bool bInitialize()
 ***************************************************************************/
t_Bool spi_tclBDCLAudioSourceAdapt::bInitialize(const t_String& crfszAudioPipeConfig)
{
   m_oAudSourceLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt:bInitialize entered "));
   t_Bool bResult = false;

   t_SptrBDCLProxyManager spoProxyManager = spi_tclBDCLProxyManager::getInstance();
   t_SptrBDCLCoreDataIntf spoCoreDataIntf = (spoProxyManager)?
            (spoProxyManager->spoGetCoreDataIntf()) : (nullptr);
   t_CoreCallbackDealer* poCallbackDealer = (spoCoreDataIntf)?
            (spoCoreDataIntf->poGetCoreCallbackDealer()) : (NULL);
   SPI_NORMAL_ASSERT(NULL == poCallbackDealer);

   if (NULL != poCallbackDealer)
   {
      m_spoAudioSource = std::move(tSptrAditAudioSource(new adit::bdcl::AilAudioSource(this, poCallbackDealer)));
      SPI_NORMAL_ASSERT(!m_spoAudioSource);

      vSetAudioStreamConfig(crfszAudioPipeConfig);
      bResult = m_spoAudioSource->initialize();

      //! Register for callbacks
      t_SptrBDCLAudioProxy spoAudioProxy = (spoProxyManager)?
               (spoProxyManager->spoGetAudioProxyInstance()) : (nullptr);
      if ((bResult) && (spoAudioProxy))
      {
         trBdclVoiceRecognitionCbs rVoiceRecCbs;
         rVoiceRecCbs.fvOnVRModuleStatus = vOnVRModuleStatusCb;
         spoAudioProxy->vRegisterVRCallbacks(rVoiceRecCbs);
      }
   }

   m_oAudSourceLock.vUnlock();

   ETG_TRACE_USR1(("[PARAM]::spi_tclBDCLAudioSourceAdapt::bInitialize left with result = %d", ETG_ENUM(BOOL, bResult)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_Void vUninitialize()
 ***************************************************************************/
t_Void spi_tclBDCLAudioSourceAdapt::vUninitialize()
{
   m_oAudSourceLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::vUninitialize entered "));

   m_bIsMicOpened = false;
   if (m_spoAudioSource)
   {
      m_spoAudioSource->captureStop();
      ETG_TRACE_USR1(("[DESC] Audio source captureStop complete"));

      m_spoAudioSource->teardown();
      ETG_TRACE_USR1(("[DESC] Audio source teardown complete"));
   }
   m_spoAudioSource = nullptr;

   m_oAudSourceLock.vUnlock();

   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::vUninitialize left"));
}

/***************************************************************************
 ** FUNCTION:  t_Bool bMicrophoneRequestCompleted( )
 ***************************************************************************/
t_Bool spi_tclBDCLAudioSourceAdapt::bMicrophoneRequestCompleted(t_Bool bMicOpen)
{
   m_oAudSourceLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::bMicrophoneRequestCompleted entered: MicOpen = %d ",
            ETG_ENUM(BOOL, bMicOpen)));
   t_Bool bResult = false;

   if (m_spoAudioSource)
   {
      if (bMicOpen)
      {
         ETG_TRACE_USR2(("[DESC] Before captureStart "));
         bResult = m_spoAudioSource->captureStart();

         ETG_TRACE_USR2(("[DESC] captureStart completed with result %d", ETG_ENUM(BOOL, bResult)));
      }
      else
      {
         ETG_TRACE_USR2(("[DESC] Before captureStop "));
         bResult = m_spoAudioSource->captureStop();

         ETG_TRACE_USR2(("[DESC] captureStop completed with result %d ", ETG_ENUM(BOOL, bResult)));
      }
   } //if (m_spoAudioSource)
   m_oAudSourceLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclBDCLAudioSourceAdapt::bMicrophoneRequestCompleted left with result %d", ETG_ENUM(BOOL, bResult)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  void onMicRecordWakeupStart()
 ***************************************************************************/
void spi_tclBDCLAudioSourceAdapt::onMicRecordWakeupStart()
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::onMicRecordWakeupStart entered - not supported"));
}

/***************************************************************************
 ** FUNCTION:  void onMicRecordRecognitionStart()
 ***************************************************************************/
void spi_tclBDCLAudioSourceAdapt::onMicRecordRecognitionStart()
{
   m_oAudSourceLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::onMicRecordRecognitionStart entered "));

   if (m_spoAudioSource)
   {
      m_bIsMicOpened = true;

      DISPATCH_AUD_REQ(e8CL_AUD_REQ_START);
   } //if (m_spoAudioSource)
   m_oAudSourceLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclBDCLAudioSourceAdapt::onMicRecordRecognitionStart left"));
}

/***************************************************************************
 ** FUNCTION:  void onMicRecordEnd()
 ***************************************************************************/
void spi_tclBDCLAudioSourceAdapt::onMicRecordEnd()
{
   m_oAudSourceLock.s16Lock();
   t_Bool bIsMicOpened = m_bIsMicOpened;
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::onMicRecordEnd entered: IsMicOpened = %d", ETG_ENUM(BOOL, bIsMicOpened)));

   if (m_spoAudioSource)
   {
      m_bIsMicOpened = false;

      //! @Note: Check whether mic was opened before dispatching msg, because phone sometimes
      //! sends MicRecordEnd without a MicRecordRecognitionStart. Will avoid redundant msg processing.
      if (bIsMicOpened)
      {
         DISPATCH_AUD_REQ(e8CL_AUD_REQ_STOP);
      }
   } //if (m_spoAudioSource)

   m_oAudSourceLock.vUnlock();
   ETG_TRACE_USR2(("spi_tclBDCLAudioSourceAdapt::onMicRecordEnd left"));
}

/***************************************************************************
 ** FUNCTION:  void onError()
 ***************************************************************************/
void spi_tclBDCLAudioSourceAdapt::onError(tBdclErrorCodes inErrorCode)
{
   tenBdclCarLifeError enError = static_cast<tenBdclCarLifeError>(inErrorCode);
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::onError entered: %d", ETG_ENUM(BDCL_ERROR, enError)));

}

/***************************************************************************
 ** FUNCTION:  t_Void vSetAudioStreamConfig()
 ***************************************************************************/
t_Void spi_tclBDCLAudioSourceAdapt::vSetAudioStreamConfig(const t_String& crfszAudioPipeConfig)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::vSetAudioStreamConfig Entered: AudioPipeConfig = %s ", crfszAudioPipeConfig.c_str()));

   if ((m_spoAudioSource) && (NULL != m_poAudioSettings))
   {
      std::map< t_String, t_String > mapAudSourceConfig;
      m_poAudioSettings->vGetBdclVRSourceConfig(mapAudSourceConfig);

      if (!mapAudSourceConfig.empty())
      {
         m_spoAudioSource->setConfigItem(sczAudSourceDeviceNameKey, crfszAudioPipeConfig.c_str());

         for (auto itr = mapAudSourceConfig.begin(); itr != mapAudSourceConfig.end(); ++itr)
         {
            if ((false == itr->first.empty()) && (false == itr->second.empty()))
            {
               m_spoAudioSource->setConfigItem(itr->first, itr->second);
            }
            else
            {
               ETG_TRACE_ERR(("[ERR]::vSetAudioStreamConfig: Skipping empty entry in map"));
            }
         }
         ETG_TRACE_USR2(("[DESC] Audio source configuration complete"));
      }
   } //if ((m_spoAudioSource) && (NULL != m_poAudioSettings) ))
}

/***************************************************************************
 ** FUNCTION:  void vOnVRModuleStatusCb()
 ***************************************************************************/
t_Void spi_tclBDCLAudioSourceAdapt::vOnVRModuleStatusCb(S_VR_STATUS_LIST* prModuleStatusList)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::vOnVRModuleStatusCb entered "));

   if (NULL != prModuleStatusList)
   {
      ETG_TRACE_USR2(("[PARAM]::vOnVRModuleStatusCb: Number of modules = %d", prModuleStatusList->cnt));

      t_Bool bIsVRStateValid = false;
      tenBdclVRState enVRModuleState = e8CL_VR_STATUS_NOT_SUPPORT;

      S_VR_STATUS* prModuleStatus = prModuleStatusList->moduleStatus;
      for (t_U32 u32ListIndex = 0; u32ListIndex < (prModuleStatusList->cnt); ++u32ListIndex)
      {
         if (NULL != prModuleStatus)
         {
            tenBdclCmdModuleID enModuleID = static_cast<tenBdclCmdModuleID>(prModuleStatus->moduleID);
            ETG_TRACE_USR4(("[PARAM]::vOnVRModuleStatusCb: moduleID = %d, statusID = %d",
                     enModuleID, prModuleStatus->statusID));

            if (e8CL_VR_MODULE_ID == enModuleID)
            {
               enVRModuleState = static_cast<tenBdclVRState>(prModuleStatus->statusID);
               bIsVRStateValid = true;
               break;
            }
            prModuleStatus = prModuleStatus->next;
         }
      }//for (t_U32 u32ListIndex = 0; ...)

      spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
      if ((NULL != poMsgQInterface) && (bIsVRStateValid))
      {
         BDCLVRModuleStatusMsg oStatusMsg;
         oStatusMsg.m_enVRState = enVRModuleState;
         poMsgQInterface->bWriteMsgToQ(&oStatusMsg, sizeof(oStatusMsg));
      }
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudioSourceAdapt::vSetAudioSettingsInstance()
 ***************************************************************************/
t_Void spi_tclBDCLAudioSourceAdapt::vSetAudioSettingsInstance (spi_tclAudioSettingsIntf* poAudioSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::vSetAudioSettingsInstance Entered:"));
   if( NULL != poAudioSettingsIntf)
   {
	   ETG_TRACE_USR1(("spi_tclBDCLAudioSourceAdapt::vSetAudioSettingsInstance: Setting m_poAudioSettings value"));
       m_poAudioSettings = poAudioSettingsIntf;
   }
}
