/*!
 *******************************************************************************
 * \file             spi_tclBDCLAudio.cpp
 * \brief            Implements the Audio functionality for Baidu Carlife using
                     interface to CL Wrapper.
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Implementation for Baidu Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author              | Modifications
 18.04.2017 |  Ramya Murthy        | Initial Version
 06.03.2018 |  Ramya Murthy        | Impemented fix for NCG3D-81731 and usage of VR interrupt msg for cleanup
 07.06.2016 |  Ramya Murthy        | Last mode audio CRQ (RTC CRQ 268222)
 
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLAudio.h"

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclBDCLAudio.cpp.trc.h"
#endif

static t_U32 su32CurSelectedDevID             = 0;
static t_Bool sbVoiceSessionActive            = false;

static timer_t srStopAudioTimerID;
static t_Bool sbStopAudTimerRunning = false;

static timer_t srSpeechRecRelTimerID;
static t_Bool bSpeechRecRelTimerRunning = false;

static const t_U8 scou8AudioUnduck      = 1;
static const t_U8 scou8AudioDuckDefdB   = 41;
static const t_U16 scou16AudioRampDefDuration  = 1000;  //In msec

#define IS_SPEECH_SESSION_ACTIVE()        ((true == sbVoiceSessionActive)||(e8CL_AUD_STREAM_CLOSED != m_aenStreamState[e8CL_AUDTYPE_MIC]))

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::spi_tclBDCLAudio()
 ***************************************************************************/
spi_tclBDCLAudio::spi_tclBDCLAudio()
{
   ETG_TRACE_USR1(("spi_tclBDCLAudio entered"));

   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   if (spoBDCLManager)
   {
      // Initializations to be done which are not specific to device connected/selected
      m_spoCmdAudio = spoBDCLManager->spoGetCmdAudioInstance();
   }
   SPI_NORMAL_ASSERT((!spoBDCLManager) || (!m_spoCmdAudio));

   //! Initialize audio stream states.
   for(t_U8 u8Cnt = 0; u8Cnt < e8CL_MAX_AUDIO_STREAMS; u8Cnt++)
   {
       m_aenStreamState[u8Cnt] = e8CL_AUD_STREAM_CLOSED;
   }//for(t_U8 u8Cnt = 0; u8Cnt < e8CL_AUD_STREAM_CLOSED; u8Cnt++)

   //! Initialize audio channel states.
   for(t_U8 u8Cnt = 0; u8Cnt < e8AUD_INVALID; u8Cnt++)
   {
       m_aenChannelStatus[u8Cnt] = e8_AUD_NOT_ACTIVE;
   }//for(t_U8 u8Cnt = 0; u8Cnt < e8AUD_INVALID; u8Cnt++)

}//spi_tclBDCLAudio::spi_tclBDCLAudio()


/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::~spi_tclBDCLAudio()
 ***************************************************************************/
spi_tclBDCLAudio::~spi_tclBDCLAudio()
{
   ETG_TRACE_USR1(("~spi_tclBDCLAudio entered "));

   su32CurSelectedDevID = 0;
}//spi_tclBDCLAudio::~spi_tclBDCLAudio()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vRegisterCallbacks(trAudioCallbacks &rfrAudCallbacks)
{
   ETG_TRACE_USR1((" spi_tclBDCLAudio::vRegisterCallbacks entered "));
   m_rAudCallbacks = rfrAudCallbacks;
}//t_Void spi_tclBDCLAudio::vRegisterCallbacks()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::bInitializeAudioPlayback(t_U32 u32DeviceId)
 ***************************************************************************/
t_Bool spi_tclBDCLAudio::bInitializeAudioPlayback(t_U32 u32DeviceId,
      tenAudioDir enAudDir, tenAudioSamplingRate enSamplingRate,
      tenAudioSamplingRate enNativeSamplingRate)
{
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);
   SPI_INTENTIONALLY_UNUSED(enSamplingRate);

   ETG_TRACE_USR1(("spi_tclBDCLAudio::bInitializeAudioPlayback entered: AudDir %d", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   //! Initialize ECNR
   if (((e8AUD_PHONE_IN == enAudDir) || (e8AUD_VR_IN == enAudDir))
         && (NULL != m_rAudCallbacks.fvInitializeAudioIn))
   {
      tenAudioInDataSet enAudioDataSet = (enAudDir == e8AUD_PHONE_IN)?
         (e32_AUDIOIN_DATASET_PHONE_WB) : (e32_AUDIOIN_DATASET_VR_MIC_ONLY);

      (m_rAudCallbacks.fvInitializeAudioIn)(enAudDir, enAudioDataSet);
      (m_rAudCallbacks.fvSetAlsaDevice)(enAudDir);
   }

   // Set the audio last mode as native in datapool if last mode is disabled, so that CP last mode is cleared on starting BDCL music
   //TODO - handling of last mode for all technologies to be implemented
   if ((e8AUD_MAIN_OUT == enAudDir) && (NULL != m_poAudioSettings))
   {
      t_Bool bRetValue = false;
      bRetValue = m_poAudioSettings->bWriteAudioLastMode(m_bIsLastModeSupported);
      ETG_TRACE_USR2(("[DESC]::bInitializeAudioPlayback - Audio last mode is written to datapool: %d",
               ETG_ENUM(BOOL, bRetValue)));
   }

   return true;

}//t_Bool spi_tclBDCLAudio::bInitializeAudioPlayback()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::bFinalizeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir)
 ***************************************************************************/
t_Bool spi_tclBDCLAudio::bFinalizeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir)
{
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);

   /*lint -esym(40,tfbSetAudioDucking)fbSetAudioDucking Undeclared identifier */
   /*lint -esym(746,fbSetAudioDucking)call to function fbSetAudioDucking() not made in the presence of a prototype */

   m_oAudioStreamLock.s16Lock();

   ETG_TRACE_USR1((" spi_tclBDCLAudio::bFinalizeAudioPlayback entered: AudDir %d", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   //! Audio channel is deactivated. Update the audio channel state.
   m_aenChannelStatus[enAudDir] = e8_AUD_NOT_ACTIVE;

   //! Media Source is deactivated.
   if(e8AUD_MAIN_OUT == enAudDir)
   {
       ETG_TRACE_USR4(("Media audio channel is deactivated."));

       //! MD had setup a new Media Audio stream during the deactivation of Media Audio channel
       if (e8CL_AUD_STREAM_OPEN == m_aenStreamState[e8CL_AUDTYPE_MEDIA])
       {
           ETG_TRACE_USR4(("Media Audio Stream was opened by MD. Requesting Media audio channel Activation"));
           vActivateChannel(enAudDir);
       }//if(e8CL_AUD_STREAM_OPEN == m_aenStreamState[e8CL_AUDTYPE_MEDIA])

   }//if(e8AUD_MAIN_OUT == enAudDir)

   //! Mix Audio Source (Guidance) is deactivated.
   else if(e8AUD_DUCK == enAudDir)
   {
      ETG_TRACE_USR4(("Mix audio channel is deactivated."));

      if (e8CL_AUD_STREAM_OPEN == m_aenStreamState[e8CL_AUDTYPE_TTS])
      {
         ETG_TRACE_USR4(("TTS Stream is open. Hence requesting for re-activation of Mix audio channel "));
         vActivateChannel(enAudDir);
      }
      else if (NULL != m_rAudCallbacks.fbSetAudioDucking)
      {
         m_rAudCallbacks.fbSetAudioDucking(scou16AudioRampDefDuration, scou8AudioUnduck,
               e8_DUCKINGTYPE_UNDUCK, e8DEV_TYPE_CARLIFE);
      }
   }//else if(e8AUD_DUCK == enAudDir)

   //! Speech audio channel is deactivated.
   else if (e8AUD_VR_IN == enAudDir)
   {
      ETG_TRACE_USR4(("Speech audio channel is deactivated."));

      if ((e8CL_AUD_STREAM_OPEN == m_aenStreamState[e8CL_AUDTYPE_MIC]) || (e8CL_AUD_STREAM_OPEN == m_aenStreamState[e8CL_AUDTYPE_VOICE]))
      {
         ETG_TRACE_USR4(("Speech Session is active. Mic is open. Requesting for activation of Speech Audio Channel"));
         vActivateChannel(enAudDir);
      }
   }//else if (e8AUD_VR_IN == enAudDir)

   m_oAudioStreamLock.vUnlock();

   //! Uninitialize ECNR
   if (((e8AUD_PHONE_IN == enAudDir) || (e8AUD_VR_IN == enAudDir))
         && (NULL != m_rAudCallbacks.fvUninitializeAudioIn))
   {
      (m_rAudCallbacks.fvUninitializeAudioIn)(enAudDir);
   }

   ETG_TRACE_USR1((" spi_tclBDCLAudio::bFinalizeAudioPlayback left: AudDir %d", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   return true;

}//t_Bool spi_tclBDCLAudio::bFinalizeAudioPlayback()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::vSetAudioPipeConfig
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vSetAudioPipeConfig(const tmapAudioPipeConfig& crfmapAudioPipeConfig)
{
   if ((NULL != m_spoCmdAudio))
   {
      m_spoCmdAudio->vSetAudioSettingsInstance(m_poAudioSettings);
      m_spoCmdAudio->vSetAudioPipeConfig(crfmapAudioPipeConfig);
   }
}//t_Void spi_tclBDCLAudio::vSetAudioPipeConfig()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::bStartAudio(t_U32,t_String,tenAudioLink)
 ***************************************************************************/
t_Bool spi_tclBDCLAudio::bStartAudio(t_U32 u32DeviceId, t_String szAudioDev,
      tenAudioDir enAudDir)
{
   /*lint -esym(40,fvStartAudioResp)fvStartAudioResp Undeclared identifier */
   /*lint -esym(40,tfbSetAudioDucking)fbSetAudioDucking Undeclared identifier */

   SPI_INTENTIONALLY_UNUSED(u32DeviceId);

   m_oAudioStreamLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudio::bStartAudio entered for Audio Source %d with Audio Device - %s ",
            ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir), szAudioDev.c_str()));

   tenBdclAudStreamType enAudStream = e8CL_AUDTYPE_MEDIA;
   t_Bool bRetVal = bGetBdclAudioStreamStype(enAudDir, enAudStream);

   if ((NULL != m_spoCmdAudio) && (false == szAudioDev.empty()) && (true == bRetVal))
   {
      ETG_TRACE_USR4(("Current Stream State for Activated audio channel is %d",
               ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[enAudStream])));
      m_aenChannelStatus[enAudDir] = e8_AUD_ACT_GRANTED;

      if (0 != su32CurSelectedDevID)
      {
         //! Start playback only if the Audio Stream from MD is blocked for Activation of Audio channel
         //! Do nothing for cases like restoration of Audio channel by Audio manager where the MD may not have activated stream
         //! When playback start is received, request for Audio channel and SPI Audio Manager will mock the necessary response to complete sequence
         if (e8CL_AUD_STREAM_OPEN == m_aenStreamState[enAudStream])
         {
            ETG_TRACE_USR4(("Starting Audio playback for CL Stream %d", ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
            m_aenStreamState[enAudStream] = e8CL_AUD_STREAMING;
            bRetVal = m_spoCmdAudio->bStartAudioPlayback(enAudStream);
         }

         if((NULL != m_rAudCallbacks.fbSetAudioDucking) && (e8AUD_DUCK == enAudDir))
         {
            (m_rAudCallbacks.fbSetAudioDucking)(scou16AudioRampDefDuration, scou8AudioDuckDefdB,
                     e8_DUCKINGTYPE_DUCK, e8DEV_TYPE_CARLIFE);
         }
      }
   }//if ((NULL != m_spoCmdAudio) && ...)

   if (NULL != m_rAudCallbacks.fvStartAudioResp)
   {
      (m_rAudCallbacks.fvStartAudioResp)(enAudDir, bRetVal);
   }

   //! If the device is already deselected, request for de-activation of Audio channel
   //! This has to be initiated only after sending an acknowledgment for Source Activity (On) to Audio Mananger
   if (0 == su32CurSelectedDevID)
   {
      if (m_bIsLastModeSupported)
      {
         (e8AUD_MAIN_OUT == enAudDir) ? (vMuteAudio(true)) : (vDeactivateChannel(enAudDir));
      }
      else
      {
         vDeactivateChannel(enAudDir);
      }
   }
   else if ((e8AUD_DUCK == enAudDir) && (e8CL_AUD_STREAM_CLOSED == m_aenStreamState[e8CL_AUDTYPE_TTS]))
   {
      ETG_TRACE_USR4(("TTS stream is closed. Deactivate TTS channel "));
      vDeactivateChannel(enAudDir);
   }

   m_oAudioStreamLock.vUnlock();
   ETG_TRACE_USR1((" spi_tclBDCLAudio::bStartAudio() left: AudDir %d", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   return bRetVal;
}//t_Bool spi_tclBDCLAudio::bStartAudio()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::bStartAudio(t_U32,t_String,tenAudioLink)
 ***************************************************************************/
t_Bool spi_tclBDCLAudio::bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
         t_String szInputAudioDev, tenAudioDir enAudDir)
{
   /*lint -esym(40,fvStartAudioResp)fvStartAudioResp Undeclared identifier */
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);
   t_Bool bRetVal = true;

   m_oAudioStreamLock.s16Lock();
   ETG_TRACE_USR1((" spi_tclBDCLAudio::bStartAudio() entered for Audio Source %d",
            ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));
   ETG_TRACE_USR4(("Audio Output Device szOutputAudioDev =%s ", szOutputAudioDev.c_str()));
   ETG_TRACE_USR4(("Audio Input Device szInputAudioDev =%s ", szInputAudioDev.c_str()));

   if ((NULL != m_spoCmdAudio) &&
            (false == szOutputAudioDev.empty()) && (false == szInputAudioDev.empty()))
   {
      m_aenChannelStatus[enAudDir] = e8_AUD_ACT_GRANTED;

      if ((0 != su32CurSelectedDevID) && (e8AUD_VR_IN == enAudDir))
      {
         ETG_TRACE_USR4(("Speech Audio Channel Activated. Speech Session state - %d, Voice Stream State - %d, Mic Stream State - %d",
               ETG_ENUM(BOOL, sbVoiceSessionActive),
               ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_VOICE]),
               ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_MIC])));

         //! Start Microphone if it was opened and blocked for Speech Audio channel activation to be completed.
         //! Start ECNR for Microphone Input as channel is now allocated.
         //! If Microphone is still closed, do nothing. Audio Device can be configured when Microphone is opened later.

         if ((e8CL_AUD_STREAM_OPEN == m_aenStreamState[e8CL_AUDTYPE_MIC]) && (NULL != m_rAudCallbacks.fvStartAudioIn))
         {
            m_aenStreamState[e8CL_AUDTYPE_MIC] = e8CL_AUD_STREAMING;
            (m_rAudCallbacks.fvStartAudioIn)(e8AUD_VR_IN);
            ETG_TRACE_USR4(("ECNR Started. Starting Microphone Input"));
            bRetVal = (m_spoCmdAudio->bStartAudioPlayback(e8CL_AUDTYPE_MIC));
         }//if((e8CL_AUD_STREAM_OPEN ==....) && (NULL != ...))

         //! Start Voice Stream if it was opened and blocked for Speech Audio channel activation to be completed.
         //! If Voice Stream is still closed, do nothing. Audio Device can be configured if stream is opened later.

         if (e8CL_AUD_STREAM_OPEN == m_aenStreamState[e8CL_AUDTYPE_VOICE])
         {
            ETG_TRACE_USR4(("Audio Device Configured. Starting Voice Stream"));
            m_aenStreamState[e8CL_AUDTYPE_VOICE] = e8CL_AUD_STREAMING;
            bRetVal = (m_spoCmdAudio->bStartAudioPlayback(e8CL_AUDTYPE_VOICE));

         }//End of if(e8CL_AUD_STREAM_OPEN == m_aenStreamState[e8CL_AUDTYPE_VOICE])

      }//End of if ((0 != su32CurSelectedDevID) && (e8AUD_VR_IN == enAudDir))
   }//End of if if ((NULL != m_spoCmdAudio) &&...))

   //! Call the registered callbacks
   if (NULL != m_rAudCallbacks.fvStartAudioResp)
   {
      (m_rAudCallbacks.fvStartAudioResp)(enAudDir, bRetVal);
   }//End of if (NULL != m_rAudCallbacks.fvStartAudioResp)

   //! If the device is already deselected, request for de-activation of Audio channel
   //! This has to be initiated only after sending an acknowledgment for Source Activity (On) to Audio Mananger
   if (0 == su32CurSelectedDevID)
   {
       ETG_TRACE_USR4(("Audio channel activated but Device is already deselected. Deactivate audio channel"));
       vDeactivateChannel(enAudDir);
   }
   //! Request for deactivation of Speech channel only after the current activation is acknowledged.
   else if ((e8AUD_VR_IN == enAudDir) && (false == sbVoiceSessionActive) &&
            (e8CL_AUD_STREAM_CLOSED == m_aenStreamState[e8CL_AUDTYPE_MIC]) &&
            (e8CL_AUD_STREAM_CLOSED == m_aenStreamState[e8CL_AUDTYPE_VOICE]))
   {
      ETG_TRACE_USR4(("Speech session is inactive. Deactivate speech channel "));
      vDeactivateChannel(enAudDir);
   }

   m_oAudioStreamLock.vUnlock();
   ETG_TRACE_USR1((" spi_tclBDCLAudio::bStartAudio() left: AudDir %d", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   return bRetVal;
}//t_Bool spi_tclBDCLAudio::bStartAudio()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::vStopAudio(t_U32)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir, t_Bool bIsPaused)
{
   m_oAudioStreamLock.s16Lock();
   ETG_TRACE_USR1((" spi_tclBDCLAudio::vStopAudio() entered: u32DeviceId = %d enAudDir = %d  ",
            u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   ETG_TRACE_USR4(("Speech Session state - %d, Voice Stream State - %d, "
         "Microphone Stream State - %d, Mix Stream state - %d, "
         "Speech Channel Status - %d, Mix Channel Status - %d",
         ETG_ENUM(BOOL, sbVoiceSessionActive),
         ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_VOICE]),
         ETG_ENUM(BDCL_AUDSTREAM_STATE,m_aenStreamState[e8CL_AUDTYPE_MIC]),
         ETG_ENUM(BDCL_AUDSTREAM_STATE,m_aenStreamState[e8CL_AUDTYPE_TTS]),
         ETG_ENUM(AUD_CHANNEL_STATE,m_aenChannelStatus[e8AUD_VR_IN]),
         ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_DUCK])));

   ETG_TRACE_USR4(("Media Stream State - %d, Media Channel Status - %d",
         ETG_ENUM(BDCL_AUDSTREAM_STATE,m_aenStreamState[e8CL_AUDTYPE_MEDIA]),
         ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8CL_AUDTYPE_MEDIA])));

   t_Bool bIsSpeechActive = ((IS_SPEECH_SESSION_ACTIVE()) ||
            (e8_AUD_ACT_REQUESTED == m_aenChannelStatus[e8AUD_VR_IN]));

   //! Main audio channel is being deallocated. Media stream is open and MD is streaming.
   //! Wait for MD to stop streaming and then send StopAudio response for Main audio.
   //! Do not wait if Main audio channel deallocation is due to VR speech session initiation.
   if ((0 != su32CurSelectedDevID) && (e8AUD_MAIN_OUT == enAudDir) && (e8CL_AUD_STREAMING == m_aenStreamState[e8CL_AUDTYPE_MEDIA]))
   {
      if ((true == bIsSpeechActive) || (e8_AUD_DEACT_REQUESTED == m_aenChannelStatus[enAudDir]))
      {
         ETG_TRACE_USR2(("Media is streaming and Speech session is active/Media channel deactivation is requested. "
               "Closing Media stream forcefully and sending StopAudio response "));
         vProcessPlaybackStop(e8CL_AUDTYPE_MEDIA);
         vSendStopAudioResponse(enAudDir);
      }
      else
      {
         ETG_TRACE_USR2(("Media is streaming and Speech session is not active. "
               "StopAudio response will be sent after phone stops streaming media "));
         vStartAudioRespTimer();
      }
   }
   else
   {
      tenBdclAudStreamType enAudStream = e8CL_AUDTYPE_MEDIA;
      t_Bool bRetVal = bGetBdclAudioStreamStype(enAudDir, enAudStream);

      if ((bRetVal) && (m_spoCmdAudio) && (e8CL_AUD_STREAMING == m_aenStreamState[enAudStream]))
      {
         ETG_TRACE_USR2(("%d is streaming and channel is being deallocated. "
               "Closing stream forcefully and sending StopAudio response ",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));

         m_aenStreamState[enAudStream] = e8CL_AUD_STREAM_CLOSED;
         ETG_TRACE_USR4(("[PARAM]::vStopAudio: Updated %d Stream State to %d",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream),
                  ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[enAudStream])));

         m_spoCmdAudio->bStopAudioPlayback(enAudStream, true);
      }
      /*if ((bRetVal) && (m_spoCmdAudio) && (e8AUD_VR_IN == enAudDir) &&
               (e8CL_AUD_STREAM_CLOSED != m_aenStreamState[e8CL_AUDTYPE_MIC]) && (NULL != m_rAudCallbacks.fvStopAudioIn))
      {
         ETG_TRACE_USR2(("Mic is open/streaming and Speech channel is being deallocated. Closing Mic forcefully and stopping ECNR"));
         m_spoCmdAudio->bStopAudioPlayback(e8CL_AUDTYPE_MIC, true);

         (m_rAudCallbacks.fvStopAudioIn)(e8AUD_VR_IN);
      }*/

      vSendStopAudioResponse(enAudDir);
   }//End of if-else

   if ((0 == su32CurSelectedDevID) && (false == bIsPaused) && (e8AUD_MAIN_OUT == enAudDir))
   {
      vDeactivateChannel(enAudDir);
   }

   m_oAudioStreamLock.vUnlock();
   ETG_TRACE_USR1((" spi_tclBDCLAudio::vStopAudio() left: AudDir %d", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

}//t_Void spi_tclBDCLAudio::vStopAudio()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::bSelectAudioDevice(t_U32)
 ***************************************************************************/
t_Bool spi_tclBDCLAudio::bSelectAudioDevice(t_U32 u32DeviceId)
{
   /*lint -esym(40,fvSelectDeviceResp)fvSelectDeviceResp Undeclared identifier */
   ETG_TRACE_USR1((" spi_tclBDCLAudio::bSelectAudioDevice entered Device ID 0x%x", u32DeviceId));

   m_oAudioStreamLock.s16Lock();
   //! Clear all the stream states.
   for (t_U8 u8Cnt = 0; u8Cnt < e8CL_MAX_AUDIO_STREAMS; u8Cnt++)
   {
      ETG_TRACE_USR4(("Stream State for %d stream is %d", ETG_ENUM(BDCL_AUDSTREAM_TYPE, u8Cnt),
               ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[u8Cnt])));
      m_aenStreamState[u8Cnt] = e8CL_AUD_STREAM_CLOSED;
   }//for(t_U8 u8Cnt = 0; u8Cnt < e8_BDCL_MAX_AUDIO_STREAMS; u8Cnt++)

   //! Clear all the channel states.
   for (t_U8 u8Cnt = 0; u8Cnt < e8AUD_INVALID; u8Cnt++)
   {
      ETG_TRACE_USR4(("Channel State for %d stream is %d", ETG_ENUM(SPI_AUDIO_DIRECTION, u8Cnt),
               ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[u8Cnt])));
      m_aenChannelStatus[u8Cnt] = e8_AUD_NOT_ACTIVE;
   }//for(t_U8 u8Cnt = 0; u8Cnt < e8AUD_INVALID; u8Cnt++)

   su32CurSelectedDevID = u32DeviceId;
   sbVoiceSessionActive = false;

   t_Bool bRetVal = (m_spoCmdAudio) ? (m_spoCmdAudio->bInitialize()):(false);

   if (m_spoCmdAudio)
   {
      trBdclAudioCbs rAudioCbs;
      rAudioCbs.fvAudioRequestCb = std::bind(
            &spi_tclBDCLAudio::vProcessAudioRequest,
            this,
            std::placeholders::_1);
      rAudioCbs.fvAudioModStateCb = std::bind(
            &spi_tclBDCLAudio::vOnModuleState,
            this,
            std::placeholders::_1,
            std::placeholders::_2);
      m_spoCmdAudio->vRegisterAudioCbs(rAudioCbs);
   }

   m_oAudioStreamLock.vUnlock();

   //! Acknowledge Device Selector with response on creation of Audio Endpoints
   if (NULL != m_rAudCallbacks.fvSelectDeviceResp)
   {
      (m_rAudCallbacks.fvSelectDeviceResp)(u32DeviceId, e8DEVCONNREQ_SELECT, bRetVal);
   }

   return bRetVal;
}//t_Bool spi_tclBDCLAudio::bSelectAudioDevice()

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::vDeselectAudioDevice(t_U32)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vDeselectAudioDevice(t_U32 u32DeviceId)
{
   /*lint -esym(40,fvSelectDeviceResp)fvSelectDeviceResp Undeclared identifier */

   m_oAudioStreamLock.s16Lock();
   ETG_TRACE_USR1((" spi_tclBDCLAudio::vDeselectAudioDevice entered Device ID 0x%x", u32DeviceId));

   //vStopSpeechRecRelTimer();

   //! Destroy endpoints first to ensure ALSA device is released before deactivating audio channel
   if (m_spoCmdAudio)
   {
       m_spoCmdAudio->vUninitialize();
   }

   //! Deactivate audio channels if any active. If the channel is requested to be deactivated, no action required.
   //! If the Audio activation request was made, check for the selected device after the channel is activated.
   if(e8_AUD_ACT_GRANTED == m_aenChannelStatus[e8AUD_DUCK])
   {
       ETG_TRACE_USR1(("Device Deselected. Requesting Mix audio channel Deactivation"));
       vDeactivateChannel(e8AUD_DUCK);
   }
   if(e8_AUD_ACT_GRANTED == m_aenChannelStatus[e8AUD_VR_IN])
   {
       ETG_TRACE_USR1(("Device Deselected. Requesting Speech audio channel Deactivation"));
       vDeactivateChannel(e8AUD_VR_IN);
   }

   //! @Note: Deactivate main audio channel only if it is used by BDCL device (if channel is in paused state,
   //! deactivation is taken care once channel is restored)
   if (e8_AUD_ACT_GRANTED == m_aenChannelStatus[e8AUD_MAIN_OUT])
   {
      ETG_TRACE_USR1(("Device Deselected. Requesting Main Audio Channel Deactivation/Mute"));
      (m_bIsLastModeSupported)? (vMuteAudio(true)) : (vDeactivateChannel(e8AUD_MAIN_OUT));
   }

   //! @Note: Clear device ID after requesting for channels deactivation
   su32CurSelectedDevID = 0;

   m_oAudioStreamLock.vUnlock();

   if (NULL != m_rAudCallbacks.fvSelectDeviceResp)
   {
      (m_rAudCallbacks.fvSelectDeviceResp)(u32DeviceId, e8DEVCONNREQ_DESELECT, true);
   }

   ETG_TRACE_USR1((" spi_tclBDCLAudio::vDeselectAudioDevice() left"));

}//t_Void spi_tclBDCLAudio::bDeselectAudioDevice()

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclBDCLAudio::vUpdateDeviceSelection()
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vUpdateDeviceSelection(t_U32 u32DevID,
                                      tenDeviceCategory enDevCat,
                                      tenDeviceConnectionReq enDeviceConnReq,
                                      tenResponseCode enRespCode,
                                      tenErrorCode enErrorCode)
{
   SPI_INTENTIONALLY_UNUSED(enDevCat);
   SPI_INTENTIONALLY_UNUSED(enErrorCode);

   ETG_TRACE_USR4(("spi_tclBDCLAudio::vUpdateDeviceSelection entered: Device ID = 0x%x", u32DevID));

   if (e8DEVCONNREQ_SELECT == enDeviceConnReq)
   {
      m_oAudioStreamLock.s16Lock();

      su32CurSelectedDevID = (e8SUCCESS == enRespCode) ? (u32DevID) : (0);
      ETG_TRACE_USR4(("[DESC]::vUpdateDeviceSelection: Stored Selected Device ID 0x%x", su32CurSelectedDevID));

      m_oAudioStreamLock.vUnlock();

      if ((e8FAILURE == enRespCode) && (m_spoCmdAudio))
      {
         m_spoCmdAudio->vUninitialize();
      }//if ((e8FAILURE == enRespCode) && (m_spoCmdAudio))

   }//if(e8DEVCONNREQ_SELECT == enDeviceConnReq)

   ETG_TRACE_USR4(("spi_tclBDCLAudio:vUpdateDeviceSelection() left"));

}//t_Void spi_tclBDCLAudio::vUpdateDeviceSelection()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vOnAudioError(tenAudioDir, tenAudioError)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vOnAudioError(tenAudioDir enAudDir, tenAudioError enAudioError)
{
   SPI_INTENTIONALLY_UNUSED(enAudioError);

   m_oAudioStreamLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vOnAudioError() entered"));

   tenBdclAudStreamType enAudStream = e8CL_AUDTYPE_MEDIA;
   t_Bool bResult = bGetBdclAudioStreamStype(enAudDir, enAudStream);

   ETG_TRACE_USR1(("spi_tclBDCLAudio::vOnAudioError entered for Audio Source - %d and CL Stream - %d",
            ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir), ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));

   //! Failure to activate channel could be due to Audio Manager processing request for higher prio source.
   //! Could occur in case of timing issues in Audio activation requests from different components.

   //TODO - currently ARL does not notify audio errors

   m_oAudioStreamLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vOnAudioError() left"));

}//t_Void spi_tclBDCLAudio::vOnAudioError()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessAudioRequest(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vProcessAudioRequest(const trBdclAudioRequest& crfrRequest)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessAudioRequest() entered for Stream=%d, Request Type=%d",
            ETG_ENUM(BDCL_AUDSTREAM_TYPE, crfrRequest.enAudStream),
            ETG_ENUM(BDCL_AUD_REQUEST, crfrRequest.enAudRequest)));

   switch (crfrRequest.enAudRequest)
   {
      case e8CL_AUD_REQ_START:
      {
         vProcessPlaybackStart(crfrRequest.enAudStream);
      }
         break;
      case e8CL_AUD_REQ_STOP:
      {
         vProcessPlaybackStop(crfrRequest.enAudStream);
      }
         break;
      case e8CL_AUD_REQ_INTERRUPT:
      {
         vProcessInterrupt(crfrRequest.enAudStream);
      }
         break;
      default:
      {
         ETG_TRACE_ERR(("[ERR]::vProcessAudioRequest: Invalid request type"));
      }
         break;
   }//switch (crfrAudioRequest.enAudRequest)

   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessAudioRequest() left"));

}//t_Void spi_tclBDCLAudio::vProcessAudioRequest()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vEvaluateVRStates(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vEvaluateVRStates()
{
   //m_oAudioStreamLock.s16Lock(); -> Calling function has already acquired lock.
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vEvaluateVRStates entered"));

   if ((false == sbVoiceSessionActive) &&
            (e8CL_AUD_STREAM_CLOSED == m_aenStreamState[e8CL_AUDTYPE_MIC]) &&
            (e8CL_AUD_STREAM_CLOSED == m_aenStreamState[e8CL_AUDTYPE_VOICE]) &&
            (e8_AUD_ACT_GRANTED == m_aenChannelStatus[e8AUD_VR_IN]))
   {
      ETG_TRACE_USR4(("Speech Session Inactive. VR and Mic are closed. Speech channel is active. Deactivate speech channel"));

      vDeactivateChannel(e8AUD_VR_IN);
      //vStartSpeechRecRelTimer();

      //! NCG3D-100990: Phone opens Media stream before VR stream is closed (Borrow to take case)
      if (e8CL_AUD_STREAM_OPEN == m_aenStreamState[e8CL_AUDTYPE_MEDIA])
      {
         ETG_TRACE_USR4(("Speech Session ended. Media stream open. Activate media audio channel"));
         vActivateChannel(e8AUD_MAIN_OUT);
      }
   }
   else
   {
      ETG_TRACE_USR4(("[DESC]::vEvaluateVRStates: Nothing to do"));
   }

   //TODO - start speech rec timer if only VR module state is active?
   //m_oAudioStreamLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vOnModuleState(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vOnModuleState(tenBdclAudStreamType enAudStream, t_Bool bModuleActive)
{
   m_oAudioStreamLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vOnModuleState() entered for Module=%d, IsActive=%d",
            ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream), ETG_ENUM(BOOL, bModuleActive)));

   tenAudioDir enAudDir = e8AUD_INVALID;
   t_Bool bRetVal = bGetAudioDir(enAudStream, enAudDir);

   if (bRetVal)
   {
      sbVoiceSessionActive = (e8CL_AUDTYPE_VOICE == enAudStream) ? (bModuleActive) : (sbVoiceSessionActive);

      if (bModuleActive)
      {
         vActivateChannel(enAudDir);
      }
      else if (e8CL_AUDTYPE_VOICE == enAudStream)
      {
         vEvaluateVRStates();
      }
   }//if (bRetVal)

   m_oAudioStreamLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vOnModuleState() left"));

}//t_Void spi_tclBDCLAudio::vOnModuleState()

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclBDCLAudio::bGetAudioDir()
 ***************************************************************************/
t_Bool spi_tclBDCLAudio::bGetAudioDir(tenBdclAudStreamType enAudStream,
      tenAudioDir& rfenAudioDir)
{
   t_Bool bValidAudDir = true;

   switch (enAudStream)
   {
      case e8CL_AUDTYPE_MEDIA:
      {
         rfenAudioDir = e8AUD_MAIN_OUT;
      }
         break;
      case e8CL_AUDTYPE_TTS:
      {
         rfenAudioDir = e8AUD_DUCK;
      }
         break;
      case e8CL_AUDTYPE_VOICE:
      case e8CL_AUDTYPE_MIC:
      {
         rfenAudioDir = e8AUD_VR_IN;
      }
         break;
      default:
      {
         ETG_TRACE_ERR(("[ERR]::bGetAudioDir: Unsupported Audio direction! "));
         bValidAudDir = false;
      }
      break;
   } //switch(enAudStream)

   ETG_TRACE_USR1((" spi_tclBDCLAudio::bGetAudioDir: left with bValidAudDir = %d, Aud Dir %d ",
         ETG_ENUM(BOOL, bValidAudDir), ETG_ENUM(SPI_AUDIO_DIRECTION, rfenAudioDir)));
   return bValidAudDir;

}//t_Bool spi_tclBDCLAudio::bGetAudioDir(tenBdclAudStreamType enAudStream,..)

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclBDCLAudio::bGetBdclAudioStreamStype(...)
 ***************************************************************************/
t_Bool spi_tclBDCLAudio::bGetBdclAudioStreamStype(tenAudioDir enAudDir,
      tenBdclAudStreamType& rfenAudioStream)
{
   t_Bool bValidAudDir = true;
   switch (enAudDir)
   {
      case e8AUD_MAIN_OUT:
         rfenAudioStream = e8CL_AUDTYPE_MEDIA;
      break;
      case e8AUD_MIX_OUT:
      case e8AUD_DUCK:
         rfenAudioStream = e8CL_AUDTYPE_TTS;
      break;
      case e8AUD_VR_IN:
         rfenAudioStream = e8CL_AUDTYPE_VOICE; //TODO
      break;
      case e8AUD_INVALID:
      default:
      {
         ETG_TRACE_USR1((" bGetBdclAudioStreamStype: Unsupported Audio direction! "));
         bValidAudDir = false;
      }
      break;
   }

   ETG_TRACE_USR1((" spi_tclBDCLAudio::bGetBdclAudioStreamStype: left with bValidAudDir = %d, StreamType %d ",
         ETG_ENUM(BOOL, bValidAudDir), ETG_ENUM(BDCL_AUDSTREAM_TYPE, rfenAudioStream)));
   return bValidAudDir;
}//t_Bool spi_tclBDCLAudio::bGetBdclAudioStreamStype(tenAudioDir enAudDir,...)

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vActivateChannel(tenAudioDir enAudDir)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vActivateChannel(tenAudioDir enAudDir)
{
   /*lint -esym(40,fvLaunchAudioReq)fvLaunchAudioReq Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vActivateChannel() entered for Audio direction %d, Current DevID 0x%x ",
         ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir), su32CurSelectedDevID));

   tenBdclAudStreamType enAudStream = e8CL_AUDTYPE_MEDIA;
   t_Bool bRet = bGetBdclAudioStreamStype(enAudDir, enAudStream);

   if ((NULL != m_spoCmdAudio) && (NULL != m_rAudCallbacks.fvLaunchAudioReq) && (bRet) && (0 != su32CurSelectedDevID))
   {
      ETG_TRACE_USR4(("[PARAM]::spi_tclBDCLAudio::vActivateChannel: Current channel status = %d, Stream state = %d",
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[enAudDir]),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[enAudStream])));

      /*if (e8AUD_VR_IN == enAudDir)
      {
         vStopSpeechRecRelTimer();
      }*/

      if (e8_AUD_NOT_ACTIVE == m_aenChannelStatus[enAudDir]) //|| (e8_AUD_DEACT_REQUESTED == m_aenChannelStatus[enAudDir]))
      {
         trAudSampleRate rAudioSampleRate;
         switch (enAudDir)
         {
            case e8AUD_MAIN_OUT:
            {			   
               m_aenChannelStatus[enAudDir] = e8_AUD_ACT_REQUESTED;
               rAudioSampleRate.enSampleRate = e8AUD_SAMPLERATE_DEFAULT;
               (m_rAudCallbacks.fvLaunchAudioReq)(
                     su32CurSelectedDevID, e8DEV_TYPE_CARLIFE, e8AUD_MAIN_OUT, rAudioSampleRate);
            }//case e8AUD_MAIN_OUT:
            break;

            case e8AUD_VR_IN:
            {
               m_aenChannelStatus[enAudDir] = e8_AUD_ACT_REQUESTED;
               rAudioSampleRate.enSampleRate = e8AUD_SAMPLERATE_16KHZ;
               (m_rAudCallbacks.fvLaunchAudioReq)(
                     su32CurSelectedDevID, e8DEV_TYPE_CARLIFE, e8AUD_VR_IN, rAudioSampleRate);
            }//case e8AUD_VR_IN:
            break;

            case e8AUD_DUCK:
            {
               m_aenChannelStatus[enAudDir] = e8_AUD_ACT_REQUESTED;
			   rAudioSampleRate.enSampleRate = e8AUD_SAMPLERATE_DEFAULT;
               (m_rAudCallbacks.fvLaunchAudioReq)(
                        su32CurSelectedDevID, e8DEV_TYPE_CARLIFE, e8AUD_DUCK, rAudioSampleRate);
            }//case e8AUD_DUCK:
               break;
            default:
            {
               ETG_TRACE_ERR(("No action taken for invalid audio direction %d ", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));
            }//default:
               break;
         }//switch (enAudDir)
      }
      else if (e8_AUD_ACT_GRANTED == m_aenChannelStatus[enAudDir])
      {
         ETG_TRACE_USR4(("Audio channel is already active."));

         //@TODO: If already in streaming state, should we stop and start stream?
         if (e8CL_AUD_STREAM_OPEN == m_aenStreamState[enAudStream])
         {
            ETG_TRACE_USR4(("Starting Audio playback for CL Stream %d", ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
            m_aenStreamState[enAudStream] = e8CL_AUD_STREAMING;
            m_spoCmdAudio->bStartAudioPlayback(enAudStream);
         }
         else if (e8CL_AUD_STREAM_CLOSED == m_aenStreamState[enAudStream])
         {
            ETG_TRACE_USR4(("Audio stream is not yet opened by device "));
         }
         else
         {
            ETG_TRACE_USR4(("Audio stream is already streaming for CL Stream %d", ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
            m_spoCmdAudio->bStartAudioPlayback(enAudStream);
         }
      }
      //! If audio channel is being activated/deactivated, wait for it to complete & reevaluate.
      else
      {
         ETG_TRACE_USR4(("Audio channel activation/deactivation is in progress. "));
         ETG_TRACE_USR4(("Evaluate the Audio stream condition again after channel activation/deactivation is completed "));
      }
   }//if ((NULL != m_spoCmdAudio) && (NULL != m_rAudCallbacks.fvLaunchAudioReq))
}//t_Void spi_tclBDCLAudio::vActivateChannel()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vDeactivateChannel(tenAudioDir enAudDir)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vDeactivateChannel(tenAudioDir enAudDir)
{
   /*lint -esym(40,fvTerminateAudioReq)fvTerminateAudioReq Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vDeactivateChannel() entered for Audio direction %d ",
         ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   /*if (e8AUD_VR_IN == enAudDir)
   {
      vStopSpeechRecRelTimer();
   }*/

   if (NULL != m_rAudCallbacks.fvTerminateAudioReq)
   {
      m_aenChannelStatus[enAudDir] = e8_AUD_DEACT_REQUESTED;
      (m_rAudCallbacks.fvTerminateAudioReq)(su32CurSelectedDevID, enAudDir, e8REASON_NOMEDIA);
   }
}//t_Void spi_tclBDCLAudio::vDeactivateChannel()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vMuteAudio(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vMuteAudio(t_Bool bMute)
{
   /*lint -esym(40,fbSetSourceMute)fbSetSourceMute Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vMuteAudio: %d ", ETG_ENUM(BOOL, bMute)));

   if (NULL != m_rAudCallbacks.fbSetSourceMute)
   {
      (m_rAudCallbacks.fbSetSourceMute)(e8AUD_MAIN_OUT, bMute);
   }
}//t_Void spi_tclBDCLAudio::vMuteAudio()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessPlaybackStart(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vProcessPlaybackStart(tenBdclAudStreamType enAudStream)
{
   m_oAudioStreamLock.s16Lock();
   //ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessPlaybackStart entered"));

   tenAudioDir enAudDir = e8AUD_INVALID;
   t_Bool bRetVal = bGetAudioDir(enAudStream, enAudDir);

   if (bRetVal)
   {
      if (e8CL_AUD_STREAM_OPEN == m_aenStreamState[enAudStream])
      {
         ETG_TRACE_ERR(("[ERR]::vProcessPlaybackStart: Stream %d is already in %d state. No action taken. ",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream),
                  ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[enAudStream])));
      }
      else
      {
         m_aenStreamState[enAudStream] = e8CL_AUD_STREAM_OPEN;
         ETG_TRACE_USR4(("[PARAM]::spi_tclBDCLAudio::vProcessPlaybackStart: Updated %d Stream State to %d",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream),
                  ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[enAudStream])));

         switch (enAudStream)
         {
            case e8CL_AUDTYPE_MEDIA:
               vProcessMediaStart(enAudDir);
               break;
            case e8CL_AUDTYPE_TTS:
               vProcessTTSStart(enAudDir);
               break;
            case e8CL_AUDTYPE_VOICE:
               vProcessVoiceStart(enAudDir);
               break;
            case e8CL_AUDTYPE_MIC:
               vProcessMicStart(enAudDir);
               break;
            default:
               ETG_TRACE_USR4(("[ERR]::vProcessPlaybackStart: Not implemented for audio stream %d",
                     ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
               break;
         } //switch (enAudStream)
      } //if (bRetVal)
   }

   m_oAudioStreamLock.vUnlock();
   //ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessPlaybackStart() left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessPlaybackStop(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vProcessPlaybackStop(tenBdclAudStreamType enAudStream)
{
   m_oAudioStreamLock.s16Lock();
   //ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessPlaybackStop entered"));

   ETG_TRACE_USR4(("Speech session state - %d, VR Stream State - %d, "
            "Mic Stream State - %d, TTS Stream State - %d, Media Stream State - %d, "
            "Speech Channel Status - %d, TTS Channel State - %d, Media Channel State- %d",
            ETG_ENUM(BOOL, sbVoiceSessionActive),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_VOICE]),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_MIC]),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_TTS]),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_MEDIA]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_VR_IN]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_DUCK]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_MAIN_OUT])));

   tenAudioDir enAudDir = e8AUD_INVALID;
   t_Bool bRetVal = bGetAudioDir(enAudStream, enAudDir);

   if ((bRetVal) && (m_spoCmdAudio))
   {
      tenBdclStreamState enPrevStreamState = m_aenStreamState[enAudStream];
      m_aenStreamState[enAudStream] = e8CL_AUD_STREAM_CLOSED;
      ETG_TRACE_USR4(("[PARAM]::vProcessPlaybackStop: Updated %d Stream State to %d",
               ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream),
               ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[enAudStream])));

      switch (enAudStream)
      {
         case e8CL_AUDTYPE_MEDIA:
         {
            ETG_TRACE_USR4(("Media Stream playback stopped. Audio channel is not required to be deactivated"));

            t_Bool bIsSpeechActive = ((IS_SPEECH_SESSION_ACTIVE()) ||
                       (e8_AUD_ACT_REQUESTED == m_aenChannelStatus[e8AUD_VR_IN]));
            t_Bool bFlushBuffer = bIsSpeechActive || (e8CL_AUD_STREAMING != enPrevStreamState) ||
                     (sbStopAudTimerRunning) || (e8_AUD_NOT_ACTIVE == m_aenChannelStatus[enAudDir]);

            m_spoCmdAudio->bStopAudioPlayback(enAudStream, bFlushBuffer);

            //! Send StopAudio response if timer was started to wait for Media PlaybackStop
            vStopAudioRespTimer();
         }
            break;
         case e8CL_AUDTYPE_TTS:
         {
/*            t_Bool bIsSpeechActive = ((IS_SPEECH_SESSION_ACTIVE()) ||
                       (e8_AUD_ACT_REQUESTED == m_aenChannelStatus[e8AUD_VR_IN]));
            t_Bool bFlushBuffer = (bIsSpeechActive || (e8CL_AUD_STREAMING != enPrevStreamState)
                     || (e8_AUD_NOT_ACTIVE == m_aenChannelStatus[enAudDir]) || (e8CL_PHONE_STATUS_IDLE != m_enPhoneModuleStatus));*/
            //This work around is to fix the ticket NCG3D-144625 Tested by China team
            t_Bool bFlushBuffer = true;
            m_spoCmdAudio->bStopAudioPlayback(enAudStream, bFlushBuffer);
            //@Note: Clearing TTS buffer if Speech is active because playing buffered data can block
            //BDCL dispatcher thread for long time depending on length of TTS prompt

            if (e8_AUD_ACT_GRANTED == m_aenChannelStatus[enAudDir])
            {
               ETG_TRACE_USR4(("TTS playback stopped. Deactivate audio channel"));
               vDeactivateChannel(enAudDir);
            }
         }
            break;
         case e8CL_AUDTYPE_VOICE:
         {
            t_Bool bFlushBuffer = (e8CL_AUD_STREAMING != enPrevStreamState)
                     || (e8_AUD_NOT_ACTIVE == m_aenChannelStatus[enAudDir]);
            m_spoCmdAudio->bStopAudioPlayback(enAudStream, bFlushBuffer);

            vEvaluateVRStates();
         }
            break;
         case e8CL_AUDTYPE_MIC:
         {
            m_spoCmdAudio->bStopAudioPlayback(enAudStream, false);

            if ((e8_AUD_ACT_GRANTED == m_aenChannelStatus[enAudDir]) && (NULL != m_rAudCallbacks.fvStopAudioIn))
            {
               ETG_TRACE_USR4(("Microphone Closed. Stop ECNR"));
               (m_rAudCallbacks.fvStopAudioIn)(e8AUD_VR_IN);
            }

            vEvaluateVRStates();
         }
            break;
         default:
            ETG_TRACE_USR4(("[ERR]::vProcessPlaybackStop: Not implemented for audio stream %d",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
            break;
      }//switch (enAudStream)
   }//if (bRetVal)

   m_oAudioStreamLock.vUnlock();
   //ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessPlaybackStop() left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessInterrupt(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vProcessInterrupt(tenBdclAudStreamType enAudStream)
{
   m_oAudioStreamLock.s16Lock();
   //ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessInterrupt entered"));

   if (e8CL_AUDTYPE_VOICE == enAudStream)
   {
      if (e8CL_AUD_STREAM_CLOSED != m_aenStreamState[enAudStream])
      {
         m_aenStreamState[enAudStream] = e8CL_AUD_STREAM_CLOSED;
         ETG_TRACE_USR4(("[PARAM]::vProcessInterrupt: Updated %d Stream State to %d",
                 ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream),
                 ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[enAudStream])));

         //@Note: Ideally not required to close VR stream since it will be taken care in ADIT
         m_spoCmdAudio->bStopAudioPlayback(enAudStream, true);
      }
      else
      {
         ETG_TRACE_USR4(("[PARAM]::vProcessInterrupt: VR stream is already closed"));
      }

      vEvaluateVRStates();
   }
   else if (e8CL_AUDTYPE_TTS == enAudStream)
   {
      if ((e8CL_AUD_STREAM_CLOSED != m_aenStreamState[enAudStream]) && (m_spoCmdAudio))
      {
         m_aenStreamState[enAudStream] = e8CL_AUD_STREAM_CLOSED;
         ETG_TRACE_USR4(("[PARAM]::vProcessInterrupt: Updated %d Stream State to %d",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream),
                  ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[enAudStream])));

         m_spoCmdAudio->bStopAudioPlayback(enAudStream, true);
      }
      else
      {
         ETG_TRACE_USR4(("[PARAM]::vProcessInterrupt: TTS stream is already closed. Nothing to do."));
      }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]::vProcessInterrupt: Not handled for stream %d",
               ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
   }

   m_oAudioStreamLock.vUnlock();
   //ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessInterrupt() left"));
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessMediaStart(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vProcessMediaStart(tenAudioDir enAudDir)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessMediaStart entered"));

   //! Speech Channel is not active or deactivation has already been requested.
   //! Request for activation of Media channel.
   ETG_TRACE_USR4(("Speech session state - %d, VR Stream State - %d, Mic Stream State - %d, "
            "Speech Channel Status - %d, Media Channel State- %d",
            ETG_ENUM(BOOL, sbVoiceSessionActive),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_VOICE]),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_MIC]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_VR_IN]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[enAudDir])));

   if((e8_AUD_NOT_ACTIVE == m_aenChannelStatus[e8AUD_VR_IN])||(e8_AUD_DEACT_REQUESTED == m_aenChannelStatus[e8AUD_VR_IN]))
   {
      //! Activate Media channel
      ETG_TRACE_USR4(("Speech audio channel not active. Can activate Media audio channel"));
      vActivateChannel(enAudDir);
   }//if((e8_AUD_NOT_ACTIVE == m_aenChannelStatus[e8AUD_VR_IN])||...)
   else
   {
      ETG_TRACE_USR4(("Speech channel is active/activation in progress. Cannot activate Media audio channel"));
   }

   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessMediaStart() left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessTTSStart(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vProcessTTSStart(tenAudioDir enAudDir)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessTTSStart entered"));

   //! Speech Channel is not active or deactivation has already been requested.
   //! Request for activation of Media channel.
   ETG_TRACE_USR4(("Speech session state - %d, VR Stream State - %d, Mic Stream State - %d, "
            "Speech Channel Status - %d, TTS Channel State- %d",
            ETG_ENUM(BOOL, sbVoiceSessionActive),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_VOICE]),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_MIC]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_VR_IN]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[enAudDir])));

   if((e8_AUD_NOT_ACTIVE == m_aenChannelStatus[e8AUD_VR_IN])||(e8_AUD_DEACT_REQUESTED == m_aenChannelStatus[e8AUD_VR_IN]))
   {
      //! Activate Media channel
      ETG_TRACE_USR4(("Speech audio channel not active. Can activate Mix audio channel"));
      vActivateChannel(enAudDir);
   }//if((e8_AUD_NOT_ACTIVE == m_aenChannelStatus[e8AUD_VR_IN])||...)
   else
   {
      ETG_TRACE_USR4(("Speech channel is active/activation in progress. Cannot activate Mix audio channel"));
   }

   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessTTSStart() left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessVoiceStart(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vProcessVoiceStart(tenAudioDir enAudDir)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessVoiceStart entered"));

   ETG_TRACE_USR4(("Speech session state - %d, VR Stream State - %d, Mic Stream State - %d, "
            "Speech Channel Status - %d, TTS Channel State- %d",
            ETG_ENUM(BOOL, sbVoiceSessionActive),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_VOICE]),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_MIC]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_VR_IN]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_DUCK])));

   vActivateChannel(enAudDir);
   //TODO - Process VR open if VR module state is inactive?

   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessVoiceStart() left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessMicStart(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vProcessMicStart(tenAudioDir enAudDir)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessMicStart entered"));

   ETG_TRACE_USR4(("Speech session state - %d, VR Stream State - %d, Mic Stream State - %d, "
            "Speech Channel Status - %d, TTS Channel State- %d",
            ETG_ENUM(BOOL, sbVoiceSessionActive),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_VOICE]),
            ETG_ENUM(BDCL_AUDSTREAM_STATE, m_aenStreamState[e8CL_AUDTYPE_MIC]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_VR_IN]),
            ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[e8AUD_DUCK])));

   if (e8_AUD_ACT_GRANTED == m_aenChannelStatus[enAudDir])
   {
      if ((m_spoCmdAudio) && (NULL != m_rAudCallbacks.fvStartAudioIn))
      {
         ETG_TRACE_USR2(("Mic is opened. Speech channel active. Start ECNR and Microphone Input "));

         m_aenStreamState[e8CL_AUDTYPE_MIC] = e8CL_AUD_STREAMING;
         (m_rAudCallbacks.fvStartAudioIn)(e8AUD_VR_IN);
         m_spoCmdAudio->bStartAudioPlayback(e8CL_AUDTYPE_MIC);
      }
   }
   else
   {
      vActivateChannel(enAudDir);
   }
   //TODO - Process VR open if VR module state is inactive?

   ETG_TRACE_USR1(("spi_tclBDCLAudio::vProcessMicStart() left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vStartAudioRespTimer()
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vStartAudioRespTimer()
{
   //! Start timer
   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (false == sbStopAudTimerRunning))
   {
      poTimer->StartTimer(srStopAudioTimerID, 2000, 0, this,
            &spi_tclBDCLAudio::bStopAudioTimerCb, NULL);
      sbStopAudTimerRunning = true;

      ETG_TRACE_USR4(("Stop Audio Timer started"));
   }//if ((NULL != poTimer) && (false == bStopAudTimerRunning))
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vStopAudioRespTimer()
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vStopAudioRespTimer()
{
   //! Stop timer and send StopAudio response
   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (true == sbStopAudTimerRunning))
   {
      poTimer->CancelTimer(srStopAudioTimerID);
      sbStopAudTimerRunning = false;
      ETG_TRACE_USR4(("Stop Audio Timer Stopped"));

      vSendStopAudioResponse(e8AUD_MAIN_OUT);
   }//if ((NULL != poTimer) && (true == bStopAudTimerRunning))
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vSendStopAudioResponse(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vSendStopAudioResponse(tenAudioDir enAudDir)
{
   /*lint -esym(40,fvStopAudioResp) fvStopAudioResp is not referenced */
   /*lint -esym(40,fvStopAudioResp) fvStopAudioResp Undeclared identifier */

   ETG_TRACE_USR4(("spi_tclBDCLAudio:vSendStopAudioResponse entered"));
   if (NULL != m_rAudCallbacks.fvStopAudioResp)
   {
      (m_rAudCallbacks.fvStopAudioResp)(enAudDir, true);
   }//if (NULL != m_rAudCallbacks.fvStopAudioResp)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vStartSpeechRecRelTimer(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vStartSpeechRecRelTimer()
{
   //m_oAudioStreamLock.s16Lock(); -> Calling function has already acquired lock.

   ETG_TRACE_USR4(("spi_tclBDCLAudio:vStartSpeechRecRelTimer entered, Timer Running = %d",
           ETG_ENUM(BOOL, bSpeechRecRelTimerRunning)));

   Timer* poTimer = Timer::getInstance();

   if ((NULL != poTimer) && (false == bSpeechRecRelTimerRunning) &&
       (e8_AUD_ACT_GRANTED == m_aenChannelStatus[e8AUD_VR_IN]))
   {
      t_U16 u16Timeout = ((true == sbVoiceSessionActive) &&
               (e8CL_AUD_STREAM_CLOSED == m_aenStreamState[e8CL_AUDTYPE_MIC]) &&
               (e8CL_AUD_STREAM_CLOSED == m_aenStreamState[e8CL_AUDTYPE_VOICE]))? (5000) : (1000);

      poTimer->StartTimer(srSpeechRecRelTimerID, u16Timeout, 0, this,
            &spi_tclBDCLAudio::bSpeechRecRelTimerCb, NULL);

      ETG_TRACE_USR4(("Speech Rec Release Timer started"));

      bSpeechRecRelTimerRunning = true;
   }//if ((false == bSpeechRecRelTimerRunning) && ...

   //m_oAudioStreamLock.vUnlock();

   ETG_TRACE_USR4(("spi_tclBDCLAudio:vStartSpeechRecRelTimer() left"));

}//t_Void spi_tclBDCLAudio::vStartSpeechRecRelTimer()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudio::vStopSpeechRecRelTimer(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudio::vStopSpeechRecRelTimer()
{
   // m_oAudioStreamLock.s16Lock(); -> Calling function has already acquired lock
   ETG_TRACE_USR4(("spi_tclBDCLAudio:vStopSpeechRecRelTimer entered, Timer Running = %d",
           ETG_ENUM(BOOL, bSpeechRecRelTimerRunning)));

   Timer* poTimer = Timer::getInstance();
   if ((true == bSpeechRecRelTimerRunning) && (NULL != poTimer))
   {
      poTimer->CancelTimer(srSpeechRecRelTimerID);
      ETG_TRACE_USR4(("Speech Rec Release Timer Stopped"));
      bSpeechRecRelTimerRunning = false;
   }//End of if ((true == bSpeechRecRelTimerRunning) && (NULL != poTimer))

   //m_oAudioStreamLock.vUnlock();
   //ETG_TRACE_USR4(("spi_tclBDCLAudio:vStopSpeechRecRelTimer() left"));

}//t_Void spi_tclBDCLAudio::vStopSpeechRecRelTimer()

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::bStopAudioTimerCb
 ***************************************************************************/
t_Bool spi_tclBDCLAudio::bStopAudioTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   ETG_TRACE_USR4(("spi_tclBDCLAudio:bStopAudioTimerCb entered"));

   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);

   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      poTimer->CancelTimer(srStopAudioTimerID);
      sbStopAudTimerRunning = false;
      ETG_TRACE_USR4(("Stop Audio Timer Stopped"));
   }//if (NULL != poTimer)

   spi_tclBDCLAudio* poBDCLAudio = static_cast<spi_tclBDCLAudio*> (pvObject);
   if((NULL != poBDCLAudio))
   {
      //Calling playback stop
      poBDCLAudio->vProcessPlaybackStop(e8CL_AUDTYPE_MEDIA);
      //work around for ticket NCG3D-154642 When baidu music is playing Audio context MAIN FALSE is not received from HMI
      //So setting Muusic state IDLE to Phone
	  t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
	  t_SptrBDCLCmdSession spoCmdSession = (spoBDCLManager) ? (spoBDCLManager->spoGetCmdSessionInstance()) : (nullptr);
	  spoCmdSession->vSetMusicState(e8CL_MUSIC_STATUS_IDLE);

      poBDCLAudio->vSendStopAudioResponse(e8AUD_MAIN_OUT);
      ETG_TRACE_USR2(("Sent StopAudio response since media playback stop was not received before timer expiration "));
   }//if(NULL != poBDCLAudio)

   ETG_TRACE_USR1(("spi_tclBDCLAudio::bStopAudioTimerCb() left"));
   return true;

}//spi_tclBDCLAudio::bSpeechRecRelTimerCb(..)

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclBDCLAudio::bSpeechRecRelTimerCb
 ***************************************************************************/
t_Bool spi_tclBDCLAudio::bSpeechRecRelTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   ETG_TRACE_USR4(("spi_tclBDCLAudio:bSpeechRecRelTimerCb entered"));

   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);

   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      poTimer->CancelTimer(srSpeechRecRelTimerID);
      ETG_TRACE_USR4(("Speech Rec Release Timer Stopped"));

      bSpeechRecRelTimerRunning = false;
   }//if (NULL != poTimer)

   spi_tclBDCLAudio* poBDCLAudio = static_cast<spi_tclBDCLAudio*>(pvObject);
   if (NULL != poBDCLAudio)
   {
      poBDCLAudio->m_oAudioStreamLock.s16Lock();
      ETG_TRACE_USR4(("Speech Session State - %d, Speech Channel Status - %d, VR Stream State - %d, Microphone Stream State - %d",
               ETG_ENUM(BOOL, sbVoiceSessionActive),
               ETG_ENUM(AUD_CHANNEL_STATE, poBDCLAudio->m_aenChannelStatus[e8AUD_VR_IN]),
               ETG_ENUM(BDCL_AUDSTREAM_STATE, poBDCLAudio->m_aenStreamState[e8CL_AUDTYPE_VOICE]),
               ETG_ENUM(BDCL_AUDSTREAM_STATE, poBDCLAudio->m_aenStreamState[e8CL_AUDTYPE_MIC])));

      //!De-activate Speech channel
      if (e8_AUD_ACT_GRANTED == poBDCLAudio->m_aenChannelStatus[e8AUD_VR_IN])
      {
         poBDCLAudio->vDeactivateChannel(e8AUD_VR_IN);
      }//End of if (e8_AUD_ACT_GRANTED == poBDCLAudio->m_aenChannelStatus[e8AUD_VR_IN])
      else
      {
         ETG_TRACE_USR4(("Speech channel not active. No action taken."));
      }

      /*//! Case 1: Media Stream is open, request for activation of Media Audio channel
      //! Case 2: Media stream is already in streaming state (due to timeout condition when source activation was not requested due to active voice session),
      //! request for activation of Media audio channel.
      if ((e8CL_AUD_STREAM_CLOSED != poBDCLAudio->m_aenStreamState[e8CL_AUDTYPE_MEDIA]) &&
               (e8_AUD_NOT_ACTIVE == poBDCLAudio->m_aenChannelStatus[e8AUD_MAIN_OUT]))
      {
         //! Request for Activation of Media Audio channel
         poBDCLAudio->vActivateChannel(e8AUD_MAIN_OUT);
      }//End of if
      */

      poBDCLAudio->m_oAudioStreamLock.vUnlock();
   }//if(NULL != poBDCLAudio)

   ETG_TRACE_USR1(("spi_tclBDCLAudio::bSpeechRecRelTimerCb() left"));
   return true;

}//spi_tclBDCLAudio::bSpeechRecRelTimerCb(..)

//lint restore

///////////////////////////////////////////////////////////////////////////////
// <EOF>
