/*!
 *******************************************************************************
 * \file              spi_tclAAPInputHandler.cpp
 * \brief             SPI input handler for AAP devices
 *******************************************************************************
 \verbatim
 PROJECT:       Gen3
 SW-COMPONENT:  Smart Phone Integration
 DESCRIPTION:   Input handler class to send input events from Head Unit to
                AAP mobile device
 AUTHOR:        Sameer Chandra (RBEI/ECP2)
 COPYRIGHT:     &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 06.03.2015 |  Sameer Chandra              | Initial Version
 07.07.2015 |  Shiva Kumar G               | Dynamic display configuration
 17.07.2015	|  Sameer Chandra              | Knob Encoder Implementation
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclAAPCmdInputIntf.h"
#include "spi_tclAAPManager.h"
#include "spi_tclAAPInputHandler.h"


#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_INPUTHANDLER
#include "trcGenProj/Header/spi_tclAAPInputHandler.cpp.trc.h"
#endif
#endif

static const t_U8 cou8EnableVerbose   = 0;
static const t_U8 cou8TouchMaximum    = 2;
static const t_U8 cou8TriggerInterval = 50;


//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
** FUNCTION:  spi_tclAAPInputHandler::spi_tclAAPInputHandler()
***************************************************************************/
spi_tclAAPInputHandler::spi_tclAAPInputHandler(): m_poAAPCmdInput(NULL)
{
  ETG_TRACE_USR1(("spi_tclAAPInputHandler::spi_tclAAPInputHandler entered "));

  spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
  m_poAAPCmdInput = (NULL != poAAPManager) ? (poAAPManager->poGetInputInstance()) : (NULL);

  if (NULL != poAAPManager)
  {
      //! Register with AAP manager for Video callbacks
      poAAPManager->bRegisterObject((spi_tclAAPRespVideo*)this);
  }

  //! Android Auto Supported Keys
  m_setenAASuppKeys =
  {
    e32DEV_BACKWARD,
    e32DEV_MENU,
    e32MULTIMEDIA_NEXT,
    e32MULTIMEDIA_PREVIOUS,
    e32DEV_SEARCH,
    e32APP_KEYCODE_MEDIA,
    e32APP_KEYCODE_TELEPHONY,
    e32DEV_NAV,
    e32MULTIMEDIA_PLAY,
    e32MULTIMEDIA_PAUSE,
    e32DEV_PHONE_CALL,
    e32DEV_PHONE_END,
    e32MULTIMEDIA_FORWARD,
    e32MULTIMEDIA_REWIND,
    e32MULTIMEDIA_STOP,
    e32TWODKNOB0_SHIFT_RIGHT,
    e32TWODKNOB0_SHIFT_LEFT,
    e32TWODKNOB0_SHIFT_UP,
    e32TWODKNOB0_SHIFT_DOWN,
    e32TWODKNOB0_SHIFT_UP_RIGHT,
    e32TWODKNOB0_SHIFT_UP_LEFT,
    e32TWODKNOB0_SHIFT_DOWN_RIGHT,
    e32TWODKNOB0_SHIFT_DOWN_LEFT,
    e32MULTIMEDIA_PLAY_PAUSE,
    e32ROTARY_KNOB,
    e32INVALID_KEY
  };
}

/***************************************************************************
** FUNCTION:  spi_tclAAPInputHandler::~spi_tclAAPInputHandler()
***************************************************************************/
spi_tclAAPInputHandler::~spi_tclAAPInputHandler()
{
   ETG_TRACE_USR1(("spi_tclAAPInputHandler::~spi_tclAAPInputHandler entered"));
   m_poAAPCmdInput = NULL;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAAPInputHandler::vProcessTouchEvent()
***************************************************************************/
t_Void spi_tclAAPInputHandler::vProcessTouchEvent(t_U32 u32DeviceHandle,trTouchData &rfrTouchData)const
{
   ETG_TRACE_USR1(("spi_tclAAPInputHandler::vProcessTouchEvent entered "));

   if (NULL != m_poAAPCmdInput)
   {
      m_poAAPCmdInput->vReportTouch(u32DeviceHandle, rfrTouchData, m_rScalingAttributes);
   }
}
/***************************************************************************
** FUNCTION: t_Void spi_tclAAPInputHandler::vProcessKeyEvents()
***************************************************************************/
t_Void spi_tclAAPInputHandler::vProcessKeyEvents(t_U32 u32DeviceHandle,tenKeyMode enKeyMode,
                        tenKeyCode enKeyCode) const
{
   ETG_TRACE_USR1(("spi_tclAAPInputHandler::vProcessKeyEvents entered "));
   ETG_TRACE_USR2(("[DESC]:vProcessKeyEvents: Key Mode : %d Key Code : %d",
      ETG_ENUM(KEY_MODE,enKeyMode), ETG_CENUM(tenKeyCode,enKeyCode)));

   if (NULL != m_poAAPCmdInput)
   {
      //forward the received info to the Input Command using pointer obtained.
      m_poAAPCmdInput->vReportkey(u32DeviceHandle, enKeyMode, enKeyCode);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAAPInputHandler::vSelectDevice()
***************************************************************************/
t_Void spi_tclAAPInputHandler::vSelectDevice(const t_U32 cou32DevId,
               const tenDeviceConnectionReq coenConnReq)
{
	/*lint -esym(40,fvSelectDeviceResp) fvSelectDeviceResp is not declared */
   ETG_TRACE_USR1(("spi_tclAAPInputHandler:vSelectDevice Device Id - 0x%x Connection Request - %d",
            cou32DevId, ETG_ENUM(CONNECTION_REQ, coenConnReq)));

   tenErrorCode enErrCode = e8INVALID_DEV_HANDLE;
   if (e8DEVCONNREQ_SELECT == coenConnReq)
   {
      //! Initialize the InputSource/WaylandInputSource endpoint.
      enErrCode = (bInitInputSession()) ? e8NO_ERRORS : e8SELECTION_FAILED;
   }
   else if (NULL != m_poAAPCmdInput)
   {
      enErrCode = e8NO_ERRORS;
      //! Un-initialize the InputSource endpoint.
      m_poAAPCmdInput->bUnInitializeInput();
   }

   ETG_TRACE_USR2(("[DESC]:vSelectDevice: Error code - %d",ETG_ENUM(ERROR_CODE,enErrCode)));

   if (NULL != m_rInputCallbacks.fvSelectDeviceResp)
   {
      (m_rInputCallbacks.fvSelectDeviceResp)(cou32DevId, enErrCode);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAAPInputHandler::vRegisterInputCallbacks()
***************************************************************************/
t_Void spi_tclAAPInputHandler::vRegisterInputCallbacks(const trInputCallbacks& corfrInputCallbacks)
{
   ETG_TRACE_USR1(("spi_tclAAPInputHandler::vRegisterVideoCallbacks entered "));
   m_rInputCallbacks = corfrInputCallbacks;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAAPInputHandler::vOnSelectDeviceResult()
***************************************************************************/
t_Void spi_tclAAPInputHandler::vOnSelectDeviceResult(const t_U32 cou32DevId,
            const tenDeviceConnectionReq coenConnReq,
            const tenResponseCode coenRespCode)
{

   ETG_TRACE_USR1(("spi_tclAAPInputHandler::vOnSelectDeviceResult:Device handle-0x%x Connection Request-%d, Response Code-%d",
            cou32DevId, ETG_ENUM(CONNECTION_REQ,coenConnReq), ETG_ENUM(RESPONSE_CODE,coenRespCode)));

   if ((e8FAILURE == coenRespCode) && (e8DEVCONNREQ_SELECT == coenConnReq)
            && (NULL != m_poAAPCmdInput))
   {
      //! Un-initialize the InputSource endpoint.
      m_poAAPCmdInput->bUnInitializeInput();
   }

}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAAPInputHandler::bInitInputSession()
***************************************************************************/
t_Bool spi_tclAAPInputHandler::bInitInputSession()
{
   //! Get Video settings instance for screen properties
	ETG_TRACE_USR1(("spi_tclAAPInputHandler::bInitInputSession entered "));
   t_Bool bInitStatus = false;

   if ((NULL != m_poAAPCmdInput) && (NULL != m_poInputHandlerSettings))
   {
      //! Get Screen dimensions
      trAAPInputConfig rAAPInputConfig;
      tvecVideoConfigList vecVideoConfigList;
      trVideoConfigData rVideoConfigData;
      std::set<tenKeyCode> rsetAASuppKeys;
      m_poInputHandlerSettings->vGetVideoConfigData(e8DEV_TYPE_ANDROIDAUTO, vecVideoConfigList);
      m_poInputHandlerSettings->vGetPrimaryDisplayConfiguration(vecVideoConfigList,rVideoConfigData);
      trScreenSize rScreenSize;
      rAAPInputConfig.u32DisplayHeight = rVideoConfigData.u32ProjScreen_Height;
      rAAPInputConfig.u32DisplayWidth = rVideoConfigData.u32ProjScreen_Width;
      rAAPInputConfig.u16LayerID = static_cast<t_U16> (rVideoConfigData.u32TouchLayerId);
      rAAPInputConfig.u16SurfaceID = static_cast<t_U16> (rVideoConfigData.u32TouchSurfaceId);
      rAAPInputConfig.u16TouchHeight = static_cast<t_U16> (rVideoConfigData.u32ProjScreen_Height);
      rAAPInputConfig.u16TouchWidth = static_cast<t_U16> (rVideoConfigData.u32ProjScreen_Width);
      rAAPInputConfig.enAAPTouchEventType = (e8MULTI_TOUCH == rVideoConfigData.enTouchEventType)
                                                                                                 ? e8AAP_MULTI_TOUCH
                                                                                                 : e8AAP_SINGLE_TOUCH;
      rAAPInputConfig.u8EnableVerbose = cou8EnableVerbose;
      rAAPInputConfig.u8TouchMaximum = cou8TouchMaximum;
      rAAPInputConfig.u8TriggerInterval = cou8TriggerInterval;
      rAAPInputConfig.bIsRotaryCtrl = m_poInputHandlerSettings->bGetRotaryCtrlSupport();
      rAAPInputConfig.enAAPTouchScreenType
               = (e8RESISTIVE_SCREEN == rVideoConfigData.enDisplayType)
                                                                        ? e8_TOUCHSCREEN_RESISTIVE
                                                                        : e8_TOUCHSCREEN_CAPACITIVE;
      rAAPInputConfig.bIsITCmdrCtrl = m_poInputHandlerSettings->bGetITCommanderSupport();
      m_poInputHandlerSettings->bGetKeyInfo(rsetAASuppKeys,e8DEV_TYPE_ANDROIDAUTO);
      bInitStatus = m_poAAPCmdInput->bInitializeInput(rAAPInputConfig,rsetAASuppKeys);
   }

   return bInitStatus;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAAPInputHandler::vProcessKnobKeyEvents()
***************************************************************************/
t_Void spi_tclAAPInputHandler::vProcessKnobKeyEvents(t_U32 u32DeviceHandle,t_S8 s8EncoderDeltaCnt) const
{
   ETG_TRACE_USR1(("spi_tclAAPInputHandler::vProcessKnobKeyEvents entered "));
   ETG_TRACE_USR2(("[DESC]:vProcessKnobKeyEvents: Device Id-0x%x Encoder Delta Count-%d",
             u32DeviceHandle,s8EncoderDeltaCnt));

   if (NULL != m_poAAPCmdInput)
   {
      //forward the received info to the Input Command using pointer obtained.
      m_poAAPCmdInput->vReportKnobkey(u32DeviceHandle, s8EncoderDeltaCnt);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAAPInputHandler::vVideoConfigCallback()
 ***************************************************************************/
t_Void spi_tclAAPInputHandler::vVideoConfigCallback(t_S32 s32LogicalUIWidth, t_S32 s32LogicalUIHeight)
{
   ETG_TRACE_USR1(("spi_tclAAPInputHandler::vVideoConfigCallback entered"));

   // Fetch video configurations from config reader.
   m_rScalingAttributes.s32YStartCoordinate = 0;
   m_rScalingAttributes.s32XStartCoordinate = 0;

   trVideoConfigData rVideoConfigData;

   if (NULL != m_poInputHandlerSettings)
   {
      tvecVideoConfigList vecVideoConfigList;
      m_poInputHandlerSettings->vGetVideoConfigData(e8DEV_TYPE_ANDROIDAUTO, vecVideoConfigList);
      m_poInputHandlerSettings->vGetPrimaryDisplayConfiguration(vecVideoConfigList,rVideoConfigData);

      //Calculate Phone Aspect ratio & Display Aspect Ratio
      //Dividing integer expressions and then converting the integer quotient to type float. 
	  //Any remainder, or fractional part of the quotient, is ignored.
	  //hence the changes are made to cast numerator to float before division operation is done
	  
      t_Float fPhoneAspectRatio = static_cast <t_Float>(s32LogicalUIWidth/s32LogicalUIHeight);

      t_Float fDispAspectRatio = static_cast <t_Float>(rVideoConfigData.u32ProjScreen_Width/
                                          rVideoConfigData.u32ProjScreen_Height);

      ETG_TRACE_USR4(("[PARAM]::vVideoConfigCallback - Phone Aspect ratio: %f Display Aspect ratio: %f",
         fPhoneAspectRatio, fDispAspectRatio));

      if (fPhoneAspectRatio < fDispAspectRatio)
      {
         // Calculate the new width
         m_rScalingAttributes.u32ScreenWidth = static_cast<t_U32>(static_cast<t_Float>(rVideoConfigData.u32ProjScreen_Height) * fPhoneAspectRatio);
         m_rScalingAttributes.u32ScreenHeight = rVideoConfigData.u32ProjScreen_Height;
         //Calculate new X axis
         m_rScalingAttributes.s32XStartCoordinate = (rVideoConfigData.u32ProjScreen_Width - m_rScalingAttributes.u32ScreenWidth) / 2;
      }
      else
      {
         m_rScalingAttributes.u32ScreenHeight =  static_cast<t_U32>(static_cast<t_Float>(rVideoConfigData.u32ProjScreen_Width) / fDispAspectRatio);
         m_rScalingAttributes.u32ScreenWidth  =   rVideoConfigData.u32ProjScreen_Width;

         // Calculate new Y axis
         m_rScalingAttributes.s32YStartCoordinate = (rVideoConfigData.u32ProjScreen_Height - m_rScalingAttributes.u32ScreenHeight) / 2 ;
      }

      m_rScalingAttributes.fHeightScalingValue = static_cast <t_Float>(static_cast<t_U32>((s32LogicalUIHeight))
                                                  /(m_rScalingAttributes.u32ScreenHeight));

      m_rScalingAttributes.fWidthScaleValue = static_cast <t_Float>((static_cast<t_U32>(s32LogicalUIWidth))
                                                  /(m_rScalingAttributes.u32ScreenWidth));

      ETG_TRACE_USR4(("[PARAM]::vVideoConfigCallback - Height scaling factor is %f and Width scaling factor is %f",
          m_rScalingAttributes.fHeightScalingValue, m_rScalingAttributes.fWidthScaleValue));

      ETG_TRACE_USR4(("[PARAM]::vVideoConfigCallback - New start coordinate for X axis : %d"
          "New start coordinate for Y axis : %d",m_rScalingAttributes.s32XStartCoordinate,m_rScalingAttributes.s32YStartCoordinate));
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAAPInputHandler::bCheckHardKeyValidity()
 ***************************************************************************/
t_Bool spi_tclAAPInputHandler::bCheckHardKeyValidity(const std::set<tenKeyCode>& rfrsetSuppkeys,
                                                 std::set<tenKeyCode>& rfrsetUnSuppkeys) const
{
   ETG_TRACE_USR1(("spi_tclAAPInputHandler::bCheckHardKeyValidity entered"));
   t_Bool bretVal = true;
   // Check keys received from client with Android Auto supported keys if there is
   // any mismatch fill it in a set
   std::set_difference(rfrsetSuppkeys.begin(),rfrsetSuppkeys.end(),
                       m_setenAASuppKeys.begin(),m_setenAASuppKeys.end(),
                       std::inserter(rfrsetUnSuppkeys,rfrsetUnSuppkeys.end()));

   if(scou8MaxKeys < rfrsetSuppkeys.size() || 0 < rfrsetUnSuppkeys.size())
   {
      bretVal = false;
   }
   return bretVal;
}
