/*!
 *******************************************************************************
 * \file             spi_tclAautoConnection.h
 * \brief            Android auto Connection class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Android auto Connection class to handle android devices capable of android auto
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 18.12.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 25.5.2015  |  Vinoop U      			   | Extended for Media metadata handling
 25.01.2016 |  Rachana L Achar             | Logiscope improvements
 08.11.2016 |  Noopur R K                  | Session Configuration for AAP
 18.07.2017 |  Noopur R K                  | Added vSetGeneralRestrictions method
 \endverbatim
 ******************************************************************************/
#include "time.h"


#include "spi_tclAautoConnection.h"
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
#include "spi_tcl_MockAAPManager.h"
#else
#include "spi_tclAAPManager.h"
#endif
#include "spi_tclAAPCmdSession.h"
#include "spi_tclMediator.h"
#if !(defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
#include "spi_tclAAPSession.h"
#endif
#include "spi_tclAAPMsgQInterface.h"
#include "spi_tclAAPSessionDispatcher.h"
#include "Timer.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclAautoConnection.cpp.trc.h"
#endif
#endif


static const t_Char* sczHeadUnitDescription = "Android Auto";
static const t_Char* sczApplicationURI = "https://www.android.com/auto/";
static const t_Char* sczApplicationVersion = "1.0";
static const t_U8    scu8MaxSessionErrCnt = 4;
static const t_U32   scu32HideClock = 0x01;
static const t_U32   scu32HidePhoneSignal = 0x02;
static const t_U32   scu32HideBatteryLevel = 0x04;

static const t_U16    scu16AAPHideBatteryLevelInfoBit = 0;
static const t_U16    scu16AAPHideClockInfoBit = 1;
static const t_U16    scu16AAPHidePhoneSignal = 2;

//! Ping weak counter
static const t_U16   scu16PingTimerDuration = 1000; //1 second
static const t_U8    scu8MaxUnAckPingCount = 3;
static const t_U8    scu8MaxWeakConnCount = 5;
static const t_U8    scu8MaxRTTAllowed = 200;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::spi_tclAautoConnection
 ***************************************************************************/
spi_tclAautoConnection::spi_tclAautoConnection() :
   m_u32CurrSelectedDevice(0), m_bSwitchInProgress(false),
   m_enCertificateType(e8_CERTIFICATETYPE_DEVELOPER),
   m_rPingTimerId(0),
   m_u8PingRespCounter(0),
   m_u8PingWeakConnCounter(0),
   m_enDevConnType(e8UNKNOWN_CONNECTION),
   m_enDevConnReq(e8DEVCONNREQ_UNKNOWN)
{
#if !(defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   ETG_TRACE_USR1(("spi_tclAautoConnection:: spi_tclAautoConnection0x%X | %d\n", GOOGLE_PROTOBUF_VERSION, GOOGLE_PROTOBUF_VERSION));
#endif
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
   if (NULL != poAAPManager)
   {
      //! Register with AAP manager.
      poAAPManager->bRegisterObject((spi_tclAAPRespSession*)this);
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::~spi_tclAautoConnection
 ***************************************************************************/
spi_tclAautoConnection::~spi_tclAautoConnection()
{
   ETG_TRACE_USR1((" spi_tclAautoConnection::~spi_tclAautoConnection() entered \n"));
   m_rPingTimerId = 0;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vOnLoadSettings
 ***************************************************************************/
t_Void spi_tclAautoConnection::vOnLoadSettings(trHeadUnitInfo &rfrheadUnitInfo, tenCertificateType enCertificateType)
{
   ETG_TRACE_USR1((" spi_tclAautoConnection::vOnLoadSettings entered \n"));
   m_rHeadUnitInfo = rfrheadUnitInfo;
   m_rCertsConfigData.szKeyId = rfrheadUnitInfo.oAAAutoCertConfigData.szKeyId;
   m_rCertsConfigData.szCertificateFolderPath = rfrheadUnitInfo.oAAAutoCertConfigData.szCertificateFolderPath;
   m_enCertificateType = enCertificateType;
   m_rSessionHistory.u32DeviceID = 0;
   m_rSessionHistory.u8ErrorCnt = 0;
   m_rSessionHistory.enSessionStatus = e8_SESSION_UNKNOWN;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::bPrepareSession
 ***************************************************************************/
t_Bool spi_tclAautoConnection::bPrepareSession()
{
   t_Bool bRetVal = false;
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
   if (NULL != poAAPManager)
   {
      spi_tclAAPCmdSessionIntf *poCmdSession = poAAPManager->poGetSessionInstance();
      if (NULL != poCmdSession)
      {
         bRetVal = poCmdSession->bInitializeSession();
         time_t rTime = time(NULL);
         struct tm rTimestamp = *localtime(&rTime);
         //! create and initialize gal receiver session
         if (true == bRetVal)
         {
            ETG_TRACE_USR1(("Before SetCertificate: %d-%d-%d %d:%d:%d", rTimestamp.tm_year + 1900, rTimestamp.tm_mon + 1, rTimestamp.tm_mday, rTimestamp.tm_hour, rTimestamp.tm_min, rTimestamp.tm_sec));
            //! TODO Add certificate path in connection settings
            ETG_TRACE_USR1((" Key Id is %s",m_rCertsConfigData.szKeyId.c_str()));
            ETG_TRACE_USR1((" Certificate Path  is %s",m_rCertsConfigData.szCertificateFolderPath.c_str()));

            bRetVal = poCmdSession->bSetCertificates(m_rCertsConfigData, m_enCertificateType);
            trAAPHeadUnitInfo rHeadUnitInfo;
            rHeadUnitInfo.szDescription = sczHeadUnitDescription;
            rHeadUnitInfo.szManufacturer = m_rHeadUnitInfo.szHUManufacturer;
            rHeadUnitInfo.szModelName = m_rHeadUnitInfo.szHUModelName;
            rHeadUnitInfo.szYear = m_rHeadUnitInfo.szModelYear;
            rHeadUnitInfo.szSerial = m_rHeadUnitInfo.szVehicleID;
            rHeadUnitInfo.szURI = sczApplicationURI;
            rHeadUnitInfo.szVersion = sczApplicationVersion;
            rHeadUnitInfo.szSoftwareVersion = m_rHeadUnitInfo.szSoftwareVersion;
            rHeadUnitInfo.szVehicleManufacturer = m_rHeadUnitInfo.szVehicleManufacturer;
            rHeadUnitInfo.szVehicleModel = m_rHeadUnitInfo.szVehicleModelName;
            rHeadUnitInfo.enDriverPos = (e8RIGHT_HAND_DRIVE == m_rHeadUnitInfo.enDrivePosition) ?
                              e8DRIVERPOSITION_RIGHT : e8DRIVERPOSITION_LEFT;
            rHeadUnitInfo.szSoftwareBuild = m_rHeadUnitInfo.szSoftwareBuild;

            poCmdSession->vSetHeadUnitInfo(rHeadUnitInfo);
         }
         ETG_TRACE_USR1(("After SetCertificate: %d-%d-%d %d:%d:%d", rTimestamp.tm_year + 1900, rTimestamp.tm_mon + 1, rTimestamp.tm_mday, rTimestamp.tm_hour, rTimestamp.tm_min, rTimestamp.tm_sec));
      }
   }
   ETG_TRACE_USR1((" spi_tclAautoConnection::bPrepareSession: Prepare information needed for AOAP Session bRetVal = %d \n",
         ETG_ENUM(BOOL, bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vOnAddDevicetoList
 ***************************************************************************/
t_Void spi_tclAautoConnection::vOnAddDevicetoList(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclAautoConnection::vOnAddDevicetoList not used \n"));
   SPI_INTENTIONALLY_UNUSED(cou32DeviceHandle);
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclAautoConnection::vRegisterCallbacks(trConnCallbacks &rfrConnCallbacks)
{
   ETG_TRACE_USR1((" spi_tclAautoConnection::vRegisterCallbacks entered \n"));
   m_rAautoConnCallbacks = rfrConnCallbacks;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vSetDeviceSwitchInfo
 ***************************************************************************/
t_Void spi_tclAautoConnection::vSetDeviceSwitchInfo(trDeviceSwitchInfo &rfrDeviceSwitchInfo)
{
   ETG_TRACE_USR1((" spi_tclAautoConnection::vSetDeviceSwitchInfo entered Headunit ID =0x%x Device iD =0x%x\n",
            rfrDeviceSwitchInfo.s32HeadUnitID, rfrDeviceSwitchInfo.s32SwitchedDeviceID));
   m_rDeviceSwitchInfo = rfrDeviceSwitchInfo;
}


/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vOnSelectDevice()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vOnSelectDevice(const t_U32 cou32DeviceHandle, tenDeviceConnectionType enDevConnType,
         tenDeviceConnectionReq enDevConnReq, tenEnabledInfo enDAPUsage, tenEnabledInfo enCDBUsage,
         tenSelectReason enSelectionReason, const trUserContext corUsrCntxt, tenDeviceType enDeviceType)
{
   SPI_INTENTIONALLY_UNUSED(enDAPUsage);
   SPI_INTENTIONALLY_UNUSED(enCDBUsage);
   SPI_INTENTIONALLY_UNUSED(corUsrCntxt);
   SPI_INTENTIONALLY_UNUSED(enDevConnType);
   SPI_INTENTIONALLY_UNUSED(enDeviceType);//For fixing lint warnings
   m_enDevConnType = enDevConnType ;
   m_enDevConnReq = enDevConnReq ;
   ETG_TRACE_USR1((" spi_tclAautoConnection::vOnSelectDevice entered"));
   switch(enDevConnReq)
   {
      case e8DEVCONNREQ_SELECT:
      {
         // Trigger AAP switch and post SelectDevice result
         vProcessDeviceSelection(cou32DeviceHandle);
         break;
      }
      case e8DEVCONNREQ_DESELECT:
      {
         //! Wait for bye bye response
         //! Send bye bye message only if the device is deselected from HMI
         //!  wait till ByeByeResponse from phone before stopping transport
         vProcessDeviceDeselection(cou32DeviceHandle, enSelectionReason);
         break;
      }
      default: break;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vProcessDeviceSelection()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vProcessDeviceSelection(const t_U32& corfu32DeviceHandle)
{
   SPI_INTENTIONALLY_UNUSED(corfu32DeviceHandle);
   ETG_TRACE_USR1(("spi_tclAautoConnection::vProcessDeviceSelection entered"));
   t_Bool bResult = bPrepareSession();
   //! Post select device result for device selection request
   tenErrorCode enErrorType = (true == bResult) ? e8NO_ERRORS : e8SELECTION_FAILED;
   tenResponseCode enRespCode = (true == bResult) ? e8SUCCESS : e8FAILURE;
   vPostSelectDeviceResult(e8DEVCONNREQ_SELECT, enRespCode, enErrorType);
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vSetGeneralRestrictions()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vSetGeneralRestrictions(t_U16 u16GeneralRestrictionInfo)
{
   ETG_TRACE_USR1(("spi_tclAautoConnection::vSetGeneralRestrictions entered"));
   //calculate session config info
   //store the value?
   trSessionConfigurationInfo rSessionConfigurationInfo;
   rSessionConfigurationInfo.bHideBatteryLevel = (u16GeneralRestrictionInfo >> scu16AAPHideBatteryLevelInfoBit) & 1;
   rSessionConfigurationInfo.bHideClock = (u16GeneralRestrictionInfo >> scu16AAPHideClockInfoBit) & 1;
   rSessionConfigurationInfo.bHidePhoneSignal = (u16GeneralRestrictionInfo >> scu16AAPHidePhoneSignal) & 1;

   ETG_TRACE_USR1(("spi_tclAautoConnection::vSetGeneralRestrictions - The session configuration values are"
            "Hide Battery level = %d, Hide Clock = %d , Hide Phone Signal = %d",ETG_ENUM(BOOL,rSessionConfigurationInfo.bHideBatteryLevel)
            ,ETG_ENUM(BOOL,rSessionConfigurationInfo.bHideClock) ,ETG_ENUM(BOOL,rSessionConfigurationInfo.bHidePhoneSignal)));
   vSetSessionConfigurationInfo(rSessionConfigurationInfo);
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vProcessDeviceDeselection()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vProcessDeviceDeselection(const t_U32& corfu32DeviceHandle,
       tenSelectReason enSelectionReason)
{
    SPI_INTENTIONALLY_UNUSED(corfu32DeviceHandle);
   ETG_TRACE_USR1(("spi_tclAautoConnection::vProcessDeviceDeselection entered"));
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
   
    if (NULL != poAAPManager)
    {
       spi_tclAAPCmdSessionIntf *poCmdSession = poAAPManager->poGetSessionInstance();
       if (NULL != poCmdSession)
       {
          // Check if device is deselected from HMI
          if((e8_REASON_USER_TRIGGER == enSelectionReason) || (e8_FACTORY_RESET == enSelectionReason))
          {
             poCmdSession->vSendByeByeMessage();
          }
          //Stop transport and post selectDevice result for device deselection request
          else
          {
             poCmdSession->vStopTransport();
             vPostSelectDeviceResult(e8DEVCONNREQ_DESELECT, e8SUCCESS, e8NO_ERRORS);
          }
       } //if (NULL != poCmdSession)
    } //if (NULL != poAAPManager)
    //! Reset the device connection type
    m_enDevConnType = e8UNKNOWN_CONNECTION ;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vPostSelctDeviceResult()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vPostSelectDeviceResult(tenDeviceConnectionReq coenConnReq,
         tenResponseCode enResponse, tenErrorCode enErrorType)
{
	/*lint -esym(40,fvSelectDeviceResult)fvSelectDeviceResult Undeclared identifier */
   SPI_INTENTIONALLY_UNUSED(enResponse);
   SPI_INTENTIONALLY_UNUSED(coenConnReq);
   ETG_TRACE_USR1(("  spi_tclAautoConnection::vPostSelectDeviceResult: Sending device selection result \n"));
   if(NULL != m_rAautoConnCallbacks.fvSelectDeviceResult)
   {
      (m_rAautoConnCallbacks.fvSelectDeviceResult)(enErrorType);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vOnSelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vOnSelectDeviceResult(const t_U32 cou32DeviceHandle,
         const tenDeviceConnectionReq coenConnReq, const tenResponseCode coenRespCode,
         tenDeviceCategory enDevCat, tenSelectReason enSelectionReason, tenDeviceType enDeviceType)
{
   SPI_INTENTIONALLY_UNUSED(enSelectionReason);
   SPI_INTENTIONALLY_UNUSED(enDevCat);
   SPI_INTENTIONALLY_UNUSED(enDeviceType);
   ETG_TRACE_USR1((" spi_tclAautoConnection::vOnSelectDeviceResult entered \n"));

   //! Start transport to establish communication between AAP Device and HU
   //! once selection is complete
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
   if ((e8SUCCESS == coenRespCode) && (NULL != poAAPManager))
   {
      spi_tclAAPCmdSessionIntf *poCmdSession = poAAPManager->poGetSessionInstance();
      if (NULL != poCmdSession)
      {
         //! Get the session info stored during AOAP switch
         trAAPSessionInfo rSessionInfo;
         //poCmdDisc->vGetSessionInfo(cou32DeviceHandle, rSessionInfo);
         rSessionInfo.u32HeadUnitID = m_rDeviceSwitchInfo.s32HeadUnitID;
         rSessionInfo.u32DeviceID = m_rDeviceSwitchInfo.s32SwitchedDeviceID;
         ETG_TRACE_USR4((" spi_tclAautoConnection::vOnSelectDeviceResult Session Info Headunit ID = %d, DeviceID =0x%x  \n",
                  m_rDeviceSwitchInfo.s32HeadUnitID,m_rDeviceSwitchInfo.s32SwitchedDeviceID));
         if (e8DEVCONNREQ_SELECT == coenConnReq)
         {
            m_u32CurrSelectedDevice = cou32DeviceHandle;
            //! Start transport on successful selection
            poCmdSession->bStartTransport(rSessionInfo);
            //! Check if the device selected is same as that was selected in last selection cycle.
            //! In case the device is different, clear the last session history.
            time_t rTime = time(NULL);
            struct tm rTimestamp = *localtime(&rTime);
            ETG_TRACE_USR1(("After StartTransport: %d-%d-%d %d:%d:%d", rTimestamp.tm_year + 1900, rTimestamp.tm_mon + 1, rTimestamp.tm_mday, rTimestamp.tm_hour, rTimestamp.tm_min, rTimestamp.tm_sec));
            if (cou32DeviceHandle != m_rSessionHistory.u32DeviceID)
            {
               ETG_TRACE_USR4((" Different device selected. Clearing Session History\n"));
               m_rSessionHistory.u32DeviceID = cou32DeviceHandle;
               m_rSessionHistory.u8ErrorCnt = 0;
               //! If Session has become active even before selection result is received, keep the status unchanged.
               m_rSessionHistory.enSessionStatus = (e8_SESSION_ACTIVE != m_rSessionHistory.enSessionStatus)
                     ? (e8_SESSION_INACTIVE) : (m_rSessionHistory.enSessionStatus);
            }// if( cou32DeviceHandle == m_rSessionHistory.u32DeviceID)
         }
         else if (e8DEVCONNREQ_DESELECT == coenConnReq)
         {
            m_u32CurrSelectedDevice = 0;
            ETG_TRACE_USR1(("Stopping the ping timer because device disconnected."));
            //! Stop the periodic ping timer
            vStopPingTimer();
            poCmdSession->vUnInitializeSession();
         }
      }
   }
   m_enDevConnReq = e8DEVCONNREQ_UNKNOWN ;
   // If the response is a failure, assume that any of the internal
   // component is not ready to accept the latest state chanege and handle it accordingly.
   //! TODO Reset USB Connection on failure???

}

//! Methods overwriteen from Andorid auto discoverer response class
/***************************************************************************
** FUNCTION:  t_Void spi_tclAautoConnection::vPostDeviceInfo
***************************************************************************/
t_Void spi_tclAautoConnection::vPostDeviceInfo(
    const t_U32 cou32DeviceHandle,
    const trDeviceInfo &corfrDeviceInfo)
{
   ETG_TRACE_USR1((" spi_tclAautoConnection::vPostDeviceInfo entered \n"));
   vOnDeviceConnection(cou32DeviceHandle, corfrDeviceInfo);
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPRespDiscoverer::vPostDeviceDisconnected
***************************************************************************/
t_Void spi_tclAautoConnection::vPostDeviceDisconnected(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("  spi_tclAautoConnection::vPostDeviceDisconnected entered \n"));
   vOnDeviceDisconnection(cou32DeviceHandle);
}


//! Methods overwriteen from Andorid auto session response class
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAautoConnection::vUnrecoverableErrorCallback()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vUnrecoverableErrorCallback()
{
   ETG_TRACE_USR1(("  spi_tclAautoConnection::vUnrecoverableErrorCallback entered \n"));
   //! No action needed as the reset is triggered during deselection
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAautoConnection::vByeByeRequestCb()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vByeByeRequestCb(tenAAPByeByeReason enReason)
{
   ETG_TRACE_USR1(("  spi_tclAautoConnection::vByeByeRequestCb enReason = %d \n", enReason));
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
   if (NULL != poAAPManager)
   {
      spi_tclAAPCmdSessionIntf *poCmdSession = poAAPManager->poGetSessionInstance();
      if(NULL != poCmdSession)
      {
         //! Set the select error flag as device has sent a ByeBye message
         if (NULL != (m_rAautoConnCallbacks.fvSelectDeviceError))
         {
            (m_rAautoConnCallbacks.fvSelectDeviceError)(m_u32CurrSelectedDevice, true);
         }
         //! TODO : do cleanup on byebye message check if deselection is sufficient or USB reset needed
         //! Ideally post internal deselection
         //! Send Device deselection to SPI components
         spi_tclMediator *poMediator = spi_tclMediator::getInstance();
         if (NULL != poMediator)
         {
            poMediator->vPostAutoDeviceSelection(m_u32CurrSelectedDevice, e8DEVCONNREQ_DESELECT);
         } // if (NULL != poMediator)
         poCmdSession->vSendByeByeResponse();
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAautoConnection::vByeByeResponseCb()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vByeByeResponseCb()
{
   ETG_TRACE_USR1(("  spi_tclAautoConnection::vByeByeResponseCb: Stopping transport and deselecting the device \n"));
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
   if (NULL != poAAPManager)
   {
      spi_tclAAPCmdSessionIntf *poCmdSession = poAAPManager->poGetSessionInstance();
      if (NULL != poCmdSession)
      {
         poCmdSession->vStopTransport();
      }
   }
   //! Post select device result after receiving bye bye response from phone
   vPostSelectDeviceResult(e8DEVCONNREQ_DESELECT,e8SUCCESS, e8NO_ERRORS);
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vOnDeviceConnection
 ***************************************************************************/
t_Void spi_tclAautoConnection::vOnDeviceConnection(const t_U32 cou32DeviceHandle, const trDeviceInfo &corfrDeviceInfo)
{
   SPI_INTENTIONALLY_UNUSED(cou32DeviceHandle);
   SPI_INTENTIONALLY_UNUSED(corfrDeviceInfo);
   ETG_TRACE_USR1((" spi_tclAautoConnection::vOnDeviceConnection entered \n"));
   if (NULL != (m_rAautoConnCallbacks.fvDeviceConnection))
   {
      //(m_rAautoConnCallbacks.fvDeviceConnection)(cou32DeviceHandle, corfrDeviceInfo, e8DEV_TYPE_ANDROIDAUTO);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAautoConnection::vOnDeviceDisconnection
 ***************************************************************************/
t_Void spi_tclAautoConnection::vOnDeviceDisconnection(const t_U32 cou32DeviceHandle)
{
   SPI_INTENTIONALLY_UNUSED(cou32DeviceHandle);
   ETG_TRACE_USR1(("spi_tclAautoConnection::vOnDeviceDisconnection entered \n"));

   if (NULL != (m_rAautoConnCallbacks.fvDeviceDisconnection))
   {
     // (m_rAautoConnCallbacks.fvDeviceDisconnection)(cou32DeviceHandle, e8DEV_TYPE_ANDROIDAUTO);
   }//if(NULL != (m_rDiPoConnCallbacks.fvDeviceDisconnection))
}


/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAautoConnection::vSessionStatusInfo()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vSessionStatusInfo(tenSessionStatus enSessionStatus, t_Bool bSessionTimedOut)
{
   /*lint -esym(40,fvUpdateSessionStatus)fvUpdateSessionStatus Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclAautoConnection::vSessionStatusInfo enSessionStatus = %d\n", ETG_ENUM(SESSION_STATUS,
         enSessionStatus)));

   if ((e8_SESSION_ACTIVE == enSessionStatus) || (e8_SESSION_INACTIVE == enSessionStatus))
   {
      //! Check if the session became INACTIVE after an Error Notification.
      if ((e8_SESSION_ERROR == m_rSessionHistory.enSessionStatus) && (e8_SESSION_INACTIVE == enSessionStatus))
      {
         ETG_TRACE_USR4(("Session Inactive is received after Session Error was notified. Session Error Count is kept unchanged"));
      }//if((e8_SESSION_ERROR == m_rSessionHistory.enSessionStatus) && (e8_SESSION_INACTIVE == enSessionStatus))
      else
      {
         ETG_TRACE_USR4(("Session is activated successfully or gracefully ended. Clearing Session Error Count"));
         //! No update required for device handle. If the device handle has changed, this will be updated in SelectDeviceResult.
         //! In case the device handle is same, it is just sufficient to reset the error count and update the session status.
         //! Update the session status independent of Device handle when Session Setup/end is successful/graceful.
         m_rSessionHistory.enSessionStatus = enSessionStatus;
         //! Clear the error count as the session could be activated/ended successfully now.
         m_rSessionHistory.u8ErrorCnt = 0;
      }
   }//if((e8_SESSION_ACTIVE == enSessionStatus) || (e8_SESSION_INACTIVE == enSessionStatus))

   if ((e8_SESSION_ERROR == enSessionStatus) || (e8_SESSION_PING_FAILURE == enSessionStatus))
   {
      ETG_TRACE_USR4(("Session Error Count is %d", m_rSessionHistory.u8ErrorCnt));
      //! Check if the last state of the session was error. If the session was previously ended gracefully or activated successfully,
      //! it is not required to check for maximum session failures.
      if (e8_SESSION_ERROR == m_rSessionHistory.enSessionStatus)
      {
         ETG_TRACE_USR4(
               ("Session error notified by MD. Previous session activation was failed due to error.Updating Session Error Count"));
         m_rSessionHistory.u8ErrorCnt++;
      }      //if(e8_SESSION_ERROR == m_rSessionHistory.enSessionStatus)
      else
      {
         //! Update the Session state to error. Start counting session errors.
         ETG_TRACE_USR4(("Session error notified by MD. No error from previous session.Updating Session State to Error and Initialize Count"));
         m_rSessionHistory.enSessionStatus = e8_SESSION_ERROR;
         m_rSessionHistory.u8ErrorCnt = 1;
      }      // end of if-else; if(e8_SESSION_ERROR == m_rSessionHistory.enSessionStatus)

         //! Check if the maximum number of consecutive session errors has been reached or if the error is triggerred due to timeout
      if (((scu8MaxSessionErrCnt == m_rSessionHistory.u8ErrorCnt) || (true == bSessionTimedOut))
               && (NULL != (m_rAautoConnCallbacks.fvSelectDeviceError)))
         {
            ETG_TRACE_USR4(("Maximum number of attempts to activate session without error reached. Set Error flag for Device"));
            //! Set the Error Flag for the device to prevent subsequent automatic selection.
            (m_rAautoConnCallbacks.fvSelectDeviceError)(m_rSessionHistory.u32DeviceID, true);
            //! Clear the Session History as error has been handled for the device
            m_rSessionHistory.u8ErrorCnt = 0;
            m_rSessionHistory.enSessionStatus = e8_SESSION_UNKNOWN;
            m_rSessionHistory.u32DeviceID = 0;
      }      //! if( == m_rSessionHistory.u8ErrorCnt)

      //! Send Device deselection to SPI components
      spi_tclMediator *poMediator = spi_tclMediator::getInstance();
      //! TO BE CONSIDERED: Handling of Session error notified from even before Device selection process is completed from SPI.
      //! Currently, not handled as m_u32CurrSelectedDevice will not be updated and attempt to deselect will results in BUSY status.
      if (NULL != poMediator)
      {
         poMediator->vPostAutoDeviceSelection(m_u32CurrSelectedDevice, e8DEVCONNREQ_DESELECT);
      } // if (NULL != poMediator)
   }//! if(e8_SESSION_ERROR == m_rSessionHistory.enSessionStatus)
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAautoConnection::vServiceDiscoveryRequestCb()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vServiceDiscoveryRequestCb(t_String szSmallIcon, t_String szMediumIcon,
         t_String szLargeICon, t_String szLabel, t_String szDeviceName)
{
   SPI_INTENTIONALLY_UNUSED(szSmallIcon);
   SPI_INTENTIONALLY_UNUSED(szMediumIcon);
   SPI_INTENTIONALLY_UNUSED(szLargeICon);
   SPI_INTENTIONALLY_UNUSED(szLabel);

   ETG_TRACE_USR1(("spi_tclAautoConnection::vServiceDiscoveryRequestCb Device Name = %s \n", szDeviceName.c_str()));

   if (NULL != (m_rAautoConnCallbacks.fvSetDeviceName))
   {
      (m_rAautoConnCallbacks.fvSetDeviceName)(m_rSessionHistory.u32DeviceID, szDeviceName);
   }

   Timer* poAAPPingTimer = Timer::getInstance();
   
   if(NULL != poAAPPingTimer)
   {
      timer_t rTimerID;
      poAAPPingTimer->StartTimer(rTimerID,
                                 scu16PingTimerDuration,
                                 scu16PingTimerDuration,
                                 this,
                                 &spi_tclAautoConnection::bAAPPingTimerCb,
                                 NULL);
      m_rPingTimerId = rTimerID;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAautoConnection::bAAPPingTimerCb()
 ***************************************************************************/
t_Bool spi_tclAautoConnection::bAAPPingTimerCb(timer_t rTimerID,
      t_Void *pvObject, const t_Void *pvUserData)
{
   ETG_TRACE_USR1(("spi_tclPing::bPingTimerCb entered rTimerID = %p", rTimerID));
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   spi_tclAautoConnection* poSelf = static_cast<spi_tclAautoConnection*> (pvObject);
   if(NULL != poSelf)
   {
      poSelf->vTriggerPingRequest();
   }
   return true ;
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAautoConnection::vStopPingTimer()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vStopPingTimer()
{
   ETG_TRACE_USR1(("spi_tclAautoConnection::vStopPingTimer entered"));
   Timer* poAAPPingTimer = Timer::getInstance();
   if((NULL != poAAPPingTimer) && (0 != m_rPingTimerId))
   {
      //! Stop the timer if session message is received
      poAAPPingTimer->CancelTimer(m_rPingTimerId);
   }
   m_u8PingRespCounter = 0;
   m_u8PingWeakConnCounter = 0;
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAautoConnection::vTriggerPingRequest()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vTriggerPingRequest()
{
   ETG_TRACE_USR1(("  spi_tclAautoConnection::vsendPingRequestCb entered \n"));
   if (m_u8PingRespCounter < scu8MaxUnAckPingCount)
   {
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
      if ((NULL != poAAPManager) && (m_enDevConnReq != e8DEVCONNREQ_DESELECT))
      {
         spi_tclAAPCmdSessionIntf *poCmdSession = poAAPManager->poGetSessionInstance();
         if (NULL != poCmdSession)
         {
            poCmdSession->vSendPingRequest();
            m_u8PingRespCounter++;
         }
      }
    }
     else
      {
         //! Send session status as inactive if no message is received from phone(timer expiry)
         AAPSessionStatusMsg oSessionStatus;
         oSessionStatus.m_enSessionstatus = e8_SESSION_PING_FAILURE;
         spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
         if (NULL != poMsgQinterface)
         {
            poMsgQinterface->bWriteMsgToQ(&oSessionStatus, sizeof(oSessionStatus));
         }//if (NULL != poMsgQinterface)
      }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAautoConnection::vServiceDiscoveryRequestCb()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vPingResponseCb(t_S64 s64timestamp)
{
   ETG_TRACE_USR1(("spi_tclAautoConnection::vPingResponseCb entered \n"));
   if(m_u8PingRespCounter != 0)
   {
      m_u8PingRespCounter--;
      struct timespec start;
      if (clock_gettime(CLOCK_REALTIME, &start) != -1)
      {
         t_S64 s64RTT = start.tv_sec - s64timestamp;
         ETG_TRACE_USR1((" Ping RTT = %s \n", std::to_string(s64RTT).c_str()));
         if (s64RTT > scu8MaxUnAckPingCount * scu16PingTimerDuration)
         {
            //! Stop the periodic ping timer
            vStopPingTimer();
            AAPSessionStatusMsg oSessionStatus;
            spi_tclAAPMsgQInterface *poMsgQinterface =
                  spi_tclAAPMsgQInterface::getInstance();
            oSessionStatus.m_enSessionstatus = e8_SESSION_PING_FAILURE;
            if (NULL != poMsgQinterface)
            {
               poMsgQinterface->bWriteMsgToQ(&oSessionStatus,sizeof(oSessionStatus));
            }//if (NULL != poMsgQinterface)
            return;
         }
         if(e8WIRELESS_CONNECTED == m_enDevConnType)
         {
            (s64RTT < scu8MaxRTTAllowed)?(m_u8PingWeakConnCounter = 0):(m_u8PingWeakConnCounter++);
            if(m_u8PingWeakConnCounter >= scu8MaxWeakConnCount)
            {
               AAPSessionStatusMsg oSessionStatus;
               spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
               oSessionStatus.m_enSessionstatus = e8_SESSION_PING_FAILURE;
               if (NULL != poMsgQinterface)
               {
                  poMsgQinterface->bWriteMsgToQ(&oSessionStatus,sizeof(oSessionStatus));
               }//if(NULL != poMsgQinterface)
            }//if(m_u8PingWeakConnCounter >= 5)
         }//if(e8WIRELESS_CONNECTED == m_enDevConnType)
      }
      else
      {
         ETG_TRACE_ERR(("[ERR]:spi_tclAautoConnection::generateTimeStamp "
               "Failed with error:%d ", errno));
      }
   }//end if(m_u8PingRespCounter != 0)
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAautoConnection::vSetSessionConfiguration()
 ***************************************************************************/
t_Void spi_tclAautoConnection::vSetSessionConfigurationInfo(trSessionConfigurationInfo &rfrSessionConfigurationInfo)
{
   //logic to calculate the session configuration value
   //the following values correspond to clock = 1,phone signal = 2, battery level = 4.
   t_U32 u32SessionConfiguration = 0x00;
   if (true == rfrSessionConfigurationInfo.bHideClock)
   {
      u32SessionConfiguration = u32SessionConfiguration | scu32HideClock;
   }
   if (true == rfrSessionConfigurationInfo.bHidePhoneSignal)
   {
      u32SessionConfiguration = u32SessionConfiguration | scu32HidePhoneSignal;
   }
   if (true == rfrSessionConfigurationInfo.bHideBatteryLevel)
   {
      u32SessionConfiguration = u32SessionConfiguration | scu32HideBatteryLevel;
   }
   ETG_TRACE_USR1(("spi_tclAautoConnection :: vSetSessionConfigurationInfo SessionConfiguration Value is : %d\n", u32SessionConfiguration));

   //assign this value to the session thru cmd session
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
   if (NULL != poAAPManager)
   {
      spi_tclAAPCmdSessionIntf *poCmdSession = poAAPManager->poGetSessionInstance();
      if (NULL != poCmdSession)
      {
         poCmdSession->vSendSessionConfiguration(u32SessionConfiguration);
      }
   }
}
//lint -restore
