  /*!
 *******************************************************************************
 * \file         spi_tclAAPBluetooth.cpp
 * \brief        AAP Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    AAP Bluetooth handler class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 04.03.2015 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 26.05.2015 |  Tejaswini H B(RBEI/ECP2)         | Added Lint comments to suppress C++11 Errors
 18.08.2015 |  Ramya Murthy (RBEI/ECP2)         | Revised BT logic based on new BTSettings interfaces
 16.02.2016 |  Ramya Murthy (RBEI/ECP2)         | Changes to switch ON BT when AA device is activated (Fix for NCG3D-5752)
 23.05.2016 |  Ramya Murthy (RBEI/ECP2)         | Removed validation of Pairable, Connectable flags for default mode (Fix for NCG3D-9178)
 16.07.2018 |  Roveena Francy Lobo (RBEI/ECO2)  | Added bDelayBTPairing()
 04.12.2018 |  Dundamma S B (RBEI/ECO2)         | Registering BT callbacks to ExtCmdBluetooth directly without the interface of common Bluetooth class
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclBluetoothIntf.h"
#include "spi_tclAAPCmdBluetoothIntf.h"
#include "spi_tclAAPManager.h"
#include "spi_tclAAPBluetooth.h"

#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BLUETOOTH
      #include "trcGenProj/Header/spi_tclAAPBluetooth.cpp.trc.h"
   #endif
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/
static const tenBTSetPairingMethod scenBTSetPreferredAAPPairingMethod =
      e8PAIRING_TYPE_SSP_NUMERIC_COMPARISON;
static const t_U32 scou32BTAuthResultTimeout_ms = 30000;

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
*********************************PUBLIC*************************************
***************************************************************************/


static t_Bool sbBTLimModeRequested = false;

/***************************************************************************
** FUNCTION:  spi_tclAAPBluetooth::spi_tclAAPBluetooth();
***************************************************************************/
spi_tclAAPBluetooth::spi_tclAAPBluetooth(spi_tclBluetoothIntf* poBTInterface)
      : m_cpoBTInterface(poBTInterface),
        m_enPairingStatus(e8DEVICE_PAIRING_NOT_STARTED),
        m_rBTAuthResultTimerID(0),
        m_enPendingBTLimAction(e8BT_LIM_ACTION_UNKNOWN)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth() entered "));
   SPI_NORMAL_ASSERT(NULL == m_cpoBTInterface);

   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
   if (NULL != poAAPManager)
   {
      //! Register with AAP manager for BT callbacks
      poAAPManager->bRegisterObject((spi_tclAAPRespBluetooth*)this);
   }//if(NULL != poAAPManager)
   m_bDelayBTPairing = false ;
} //!end of spi_tclAAPBluetooth()

/***************************************************************************
** FUNCTION:  spi_tclAAPBluetooth::~spi_tclAAPBluetooth();
***************************************************************************/
spi_tclAAPBluetooth::~spi_tclAAPBluetooth()
{
   ETG_TRACE_USR1(("~spi_tclAAPBluetooth() entered "));
} //!end of ~spi_tclAAPBluetooth()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPBluetooth::bInitialise();
 ***************************************************************************/
t_Bool spi_tclAAPBluetooth::bInitialise()
{
   //! Register to BT client
   //vRegisterBTPairInfoCallbacks();
   t_Bool bInit = false;
   RespRegister *pRespRegister = RespRegister::getInstance();
   if(NULL!= pRespRegister)
   {
      bInit = pRespRegister->bRegisterObject((spi_tclExtRespBluetooth*)this);
   }
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::bInitialise() left with Result %u",bInit));
   return bInit;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPBluetooth::vOnSPISelectDeviceRequest(
***************************************************************************/
t_Void spi_tclAAPBluetooth::vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vOnSPISelectDeviceRequest() entered "));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);

   if (NULL != m_cpoBTInterface)
   {
      //! Set Device status - this is in order to prevent SwitchDevice during ongoing selection
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_SELECTION_IN_PROGRESS);
      //! Initialize BT Endpoint
      spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
      spi_tclAAPCmdBluetoothIntf* poCmdBluetooth = (NULL != poAAPManager)?
            (poAAPManager->poGetBluetoothInstance()) : (NULL);
      t_Bool bInitSuccess = false;

      if ((NULL != poCmdBluetooth) && (NULL != m_poBluetoothSettings))
      {
         t_String szVehicleBTAddress;
         m_poBluetoothSettings->vGetVehicleBTAddress(szVehicleBTAddress);
         bInitSuccess = poCmdBluetooth->bInitialiseBTEndpoint(
               szVehicleBTAddress, scenPreferredAAPBTPairingMethod);
      }

      //! send success result for SelectDevice
      m_cpoBTInterface->vSendSelectDeviceResult(bInitSuccess);
   }//if (NULL != m_cpoBTInterface)
} //!end of vOnSPISelectDeviceRequest()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPBluetooth::vOnSPISelectDeviceResponse(
***************************************************************************/
t_Void spi_tclAAPBluetooth::vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
      tenResponseCode enRespCode)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vOnSPISelectDeviceResponse() entered "));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);

   sbBTLimModeRequested = false;

   if (NULL != m_cpoBTInterface)
   {
      //! Clear device status
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
   }//if (NULL != m_cpoBTInterface)

   //! On selection failure, perform cleanup.
   if (e8FAILURE == enRespCode)
   {
      spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
      spi_tclAAPCmdBluetoothIntf* poCmdBluetooth = (NULL != poAAPManager)?
            (poAAPManager->poGetBluetoothInstance()) : (NULL);
      if (NULL != poCmdBluetooth)
      {
         poCmdBluetooth->vUninitialiseBTEndpoint();
      }
   }//else if (e8FAILURE == enRespCode)
} //!end of vOnSPISelectDeviceResponse()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPBluetooth::vOnSPIDeselectDeviceRequest(
***************************************************************************/
t_Void spi_tclAAPBluetooth::vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vOnSPIDeselectDeviceRequest() entered "));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   if (NULL != m_cpoBTInterface)
   {
      // Set Device status - this is in order to prevent SwitchDevice during ongoing deselection
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_DESELECTION_IN_PROGRESS);
      // Nothing else to be done. Simply send success result.
      m_cpoBTInterface->vSendSelectDeviceResult(true);
   }
} //!end of vOnSPIDeselectDeviceRequest()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPBluetooth::vOnSPIDeselectDeviceResponse(
***************************************************************************/
t_Void spi_tclAAPBluetooth::vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vOnSPIDeselectDeviceResponse() entered "));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   SPI_INTENTIONALLY_UNUSED(enRespCode);

    // Clear device status
   if (NULL != m_cpoBTInterface)
   {
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
   }

   //! Perform cleanup
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
   spi_tclAAPCmdBluetoothIntf* poCmdBluetooth = (NULL != poAAPManager)?
         (poAAPManager->poGetBluetoothInstance()) : (NULL);
   if (NULL != poCmdBluetooth)
   {
      poCmdBluetooth->vUninitialiseBTEndpoint();
   }

   vSetPairingStatus(e8DEVICE_PAIRING_NOT_STARTED);

   vStopAAPBTAuthResultStatusTimer();

   //! Need not trigger DEACTIVATE when the  Select Reason is e8_REASON_HMI_SELECT_SPI_AUTO_DESELECT_TRIGGER, as ReplaceBTLimitation with PREPARE would take care of it
   //! set the sbBTLimModeRequested flag accordingly
   if ((sbBTLimModeRequested) && (NULL != m_cpoBTInterface))
   {
      //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
      //! In cases (name not known, BT address is available) this parameter should be an empty string
      t_String szBTDeviceName;
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      if((NULL != poExtCmdBluetoothIntf) && (corfrSelectReq.m_enSelectionReason != e8_REASON_HMI_SELECT_SPI_AUTO_DESELECT_TRIGGER))
      {
         if (false == poExtCmdBluetoothIntf->bSetBTLimitationMode(
                m_cpoBTInterface->szGetSelectedDevBTAddress(),szBTDeviceName,
                e8BT_TECH_ANDROIDAUTO,
                e8BT_COMM_CHN_USB,
                e8BT_LIM_ACTION_DEACTIVATE))
         {
            m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
            m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
         }
      }
      sbBTLimModeRequested = false;
   }

} //!end of vOnSPIDeselectDeviceResponse()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTPairingRequest(...)
***************************************************************************/
t_Void spi_tclAAPBluetooth::vPostBTPairingRequest(t_String szAAPBTAddress,
      tenBTPairingMethod enPairingMethod)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vPostBTPairingRequest() entered: "
         "PairingMethod = %d, AAPBTAddress = %s ",
         ETG_ENUM(AAP_BTPAIRING_METHOD, enPairingMethod), szAAPBTAddress.c_str()));

   //! Store BT address of AAP device
   //! IMPORTANT! This must be done first, since in following logic, other functions will use the stored BT address.
   spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = NULL;
   if (NULL != m_cpoBTInterface)
   {
      m_cpoBTInterface->vSetSelectedDevBTAddress(szAAPBTAddress);
      poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
   }

   //! Send initial response as not ready (response will be sent later based on success of pairing initiation)
   //! @Note: Pairing state of device is not relevant when sending delayed status.
   vSendDevicePairingResponse(false);
   ETG_TRACE_USR4(("[PARAM]::spi_tclAAPBluetooth::vPostBTPairingRequest() - The delay pairing status- %d ", ETG_ENUM(BOOL, m_bDelayBTPairing)));
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;
   if (false == m_bDelayBTPairing)
   {
       vSetPairingStatus(e8DEVICE_PAIRING_REQUESTED);
       if((NULL != poExtCmdBluetoothIntf) && (NULL != m_cpoBTInterface) && (e8DEV_TYPE_ANDROIDAUTO == m_cpoBTInterface->enGetSelectedDeviceCategory()))
       {
           ETG_TRACE_USR4(("[PARAM]::spi_tclAAPBluetooth::vPostBTPairingRequest() - BT service is available"));
           sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
                szAAPBTAddress,szBTDeviceName,
                e8BT_TECH_ANDROIDAUTO,
                e8BT_COMM_CHN_USB,
                e8BT_LIM_ACTION_ACTIVATE);

           ETG_TRACE_USR4(("[PARAM]::spi_tclAAPBluetooth::vPostBTPairingRequest() - SetLimitationMode result - %d", ETG_ENUM(BOOL, sbBTLimModeRequested)));
           if (false == sbBTLimModeRequested)
           {
              ETG_TRACE_USR2(("vPostBTPairingRequest: BT service unavailable. Will request BT Lim mode later"));
              m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
              m_enPendingBTLimAction = e8BT_LIM_ACTION_ACTIVATE;
              vSendDevicePairingResponse(false);
           }
           else
           {
              m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
           }
       }//if (poExtCmdBluetoothIntf)
   }
   else
   {
      ETG_TRACE_USR1(("Delay BT pairing is set hence not requesting for BT limitation mode"));
      m_szPairingPending = szAAPBTAddress ;

   }
   //! Register callback for BT Device name
   if(NULL != poExtCmdBluetoothIntf)
   {
      poExtCmdBluetoothIntf->vRegisterDeviceNameCallback(szAAPBTAddress);
      //Latest BT device name available with BTClient will be retrieved
      poExtCmdBluetoothIntf->vTriggerBTNameUpdate(szAAPBTAddress);
   }
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vPostBTPairingRequest() left "));
} //!end of vPostBTPairingRequest()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTAuthenticationResult(...)
***************************************************************************/
t_Void spi_tclAAPBluetooth::vPostBTAuthenticationResult(t_S32 s32AuthResult)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vPostBTAuthenticationResult() entered with s32AuthResult = [%d]",s32AuthResult));

   vStopAAPBTAuthResultStatusTimer();

   spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = (NULL != m_cpoBTInterface) ?
         (m_cpoBTInterface->poFetchExtCmdBluetoothIntf()):NULL;

   if((NULL != poExtCmdBluetoothIntf) && (e8DEVICE_PAIRING_NOT_STARTED != enGetPairingStatus()))
   {
      t_Bool bPairingResponse = (0 == s32AuthResult)?
            poExtCmdBluetoothIntf->bSetBTPairingResponse(e8BT_PAIRING_RESPONSE_YES):
            poExtCmdBluetoothIntf->bSetBTPairingResponse(e8BT_PAIRING_RESPONSE_NO);
      if(true == bPairingResponse)
      {
         vSetPairingStatus(e8DEVICE_PAIRING_RESPONSE_SENT);
         //! Moved here since onPairingRequest() BT DeviceList doesnt contain device details yet
         if(NULL != m_cpoBTInterface)
         {
            t_String szAAPBTAddress = m_cpoBTInterface->szGetSelectedDevBTAddress();
            poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
            //! Notify AA BT device info
            t_String szBTDeviceName = (NULL != poExtCmdBluetoothIntf) ?
               (poExtCmdBluetoothIntf->szGetBTDeviceName(szAAPBTAddress)):("");
            m_cpoBTInterface->vSendBTDeviceInfo(
               m_cpoBTInterface->u32GetSelectedDevHandle(), szBTDeviceName, szAAPBTAddress);
         }
      }
   }
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vPostBTAuthenticationResult() left "));
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclAAPBluetooth::vDelayBTPairing(..)
 ***************************************************************************/
t_Void spi_tclAAPBluetooth::vDelayBTPairing(t_Bool bDelayBTPairing)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vDelayBTPairing entered"));
   ETG_TRACE_USR2(("spi_tclAAPBluetooth:vDelayBTPairing():BTAddress-%s",m_szPairingPending.c_str()));
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::bDelayBTPairing entered with bBTLimModeEnabled - %d  ",
            ETG_ENUM(BOOL, bDelayBTPairing)));
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;
   if ((false == bDelayBTPairing) && (false == m_szPairingPending.empty()))
   {
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = NULL;
      if (NULL != m_cpoBTInterface)
      {
         poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      }
      vSetPairingStatus(e8DEVICE_PAIRING_REQUESTED);
      if (NULL != poExtCmdBluetoothIntf)
      {
         sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
                  m_szPairingPending,szBTDeviceName,
                  e8BT_TECH_ANDROIDAUTO,
                  e8BT_COMM_CHN_USB,
                  e8BT_LIM_ACTION_ACTIVATE);

         if (false == sbBTLimModeRequested)
         {
            m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
            m_enPendingBTLimAction = e8BT_LIM_ACTION_ACTIVATE;
         }
      }
   }
   m_bDelayBTPairing = bDelayBTPairing;
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::bDelayBTPairing left with bBTLimModeEnabled - %d  ",
            ETG_ENUM(BOOL, bDelayBTPairing)));
}

/***************************************************************************
*********************************PRIVATE************************************
***************************************************************************/

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPBluetooth::enGetPairingStatus()
***************************************************************************/
tenDevicePairingStatus spi_tclAAPBluetooth::enGetPairingStatus()
{
   m_oPairingStateLock.s16Lock();
   tenDevicePairingStatus enPairingStatus = m_enPairingStatus;
   m_oPairingStateLock.vUnlock();

   ETG_TRACE_USR4(("[PARAM]:spi_tclAAPBluetooth::enGetPairingStatus() left with %d ",
         ETG_ENUM(SPI_BT_PAIRING_STATE, enPairingStatus)));
   return enPairingStatus;
} //!end of enGetPairingStatus()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPBluetooth::vSetPairingStatus()
***************************************************************************/
t_Void spi_tclAAPBluetooth::vSetPairingStatus(tenDevicePairingStatus enPairingStatus)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vSetPairingStatus() entered: %d ",
         ETG_ENUM(SPI_BT_PAIRING_STATE, enPairingStatus)));

   m_oPairingStateLock.s16Lock();
   m_enPairingStatus = enPairingStatus;
   m_oPairingStateLock.vUnlock();
} //!end of vSetPairingStatus()

/***************************************************************************
** FUNCTION:   t_Void spi_tclAAPBluetooth::vDeselectAAPDevice()
***************************************************************************/
t_Void spi_tclAAPBluetooth::vDeselectAAPDevice()
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vDeselectAAPDevice() entered "));
   if (NULL != m_cpoBTInterface)
   {
      //Clear device status
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
      m_cpoBTInterface->vSendDeselectDeviceRequest();
   }
} //!end of vDeselectAAPDevice()

/***************************************************************************
** FUNCTION:   t_Void spi_tclAAPBluetooth::vSendDevicePairingResponse()
***************************************************************************/
t_Void spi_tclAAPBluetooth::vSendDevicePairingResponse(t_Bool bReadyToPair)
{
   if (NULL != m_cpoBTInterface)
   {
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      t_String szAAPBTAddress = m_cpoBTInterface->szGetSelectedDevBTAddress();

      spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
      spi_tclAAPCmdBluetoothIntf* poCmdBluetooth = (NULL != poAAPManager)?
            (poAAPManager->poGetBluetoothInstance()) : (NULL);

      //! Send BTDevicePairingResponse to device
      if ((true == IS_VALID_BT_ADDRESS(szAAPBTAddress)) &&
            (NULL != poExtCmdBluetoothIntf) && (NULL != poCmdBluetooth))
      {
         //@Note: If HU is not ready to pair, send AAP device paired info as unpaired.
         t_Bool bAAPDevicePaired = (true == bReadyToPair) ?
               (poExtCmdBluetoothIntf->bGetPairingStatus(szAAPBTAddress)) :
               (false);
         poCmdBluetooth->vSendBTPairingResponse(bReadyToPair, bAAPDevicePaired);
         ETG_TRACE_USR2(("[DESC]spi_tclAAPBluetooth::vSendDevicePairingResponse: Sending BTPairingResponse "));
      }
   }//if (NULL != m_cpoBTInterface)
} //!end of vSendDevicePairingResponse()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPBluetooth::bAAPBTAuthResultTimeoutCb()
 ***************************************************************************/
t_Bool spi_tclAAPBluetooth::bAAPBTAuthResultTimeoutCb(timer_t rTimerID,
      t_Void *pvObject, const t_Void *pvUserData)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::bAAPBTAuthResultTimeoutCb entered"));
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   SPI_INTENTIONALLY_UNUSED(pvUserData);

   t_Bool bRetVal = false;

   spi_tclAAPBluetooth* poAAPBluetooth = static_cast<spi_tclAAPBluetooth*>(pvObject);
   if (NULL != poAAPBluetooth)
   {
      poAAPBluetooth->vStopAAPBTAuthResultStatusTimer();
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = (NULL != poAAPBluetooth->m_cpoBTInterface)?
              (poAAPBluetooth->m_cpoBTInterface->poFetchExtCmdBluetoothIntf()) : NULL;

      if (NULL != poExtCmdBluetoothIntf)
      {
         bRetVal = true;
         if(e8DEVICE_PAIRING_COMPLETE != poAAPBluetooth->enGetPairingStatus())
         {
            poExtCmdBluetoothIntf->bSetBTPairingResponse(e8BT_PAIRING_RESPONSE_CANCEL);
         }
      }
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPBluetooth::vStartAAPBTAuthResultStatusTimer()
 ***************************************************************************/
t_Void spi_tclAAPBluetooth::vStartAAPBTAuthResultStatusTimer()
{
   ETG_TRACE_USR2(("spi_tclAAPBluetooth::vStartAAPBTAuthResultStatusTimer entered"));
   Timer* poTimer = Timer::getInstance();

   if(NULL != poTimer)
   {
      timer_t rTimerID;
      poTimer->StartTimer(rTimerID,
                          scou32BTAuthResultTimeout_ms,
                          0,
                          this,
                          &spi_tclAAPBluetooth::bAAPBTAuthResultTimeoutCb,
                          NULL);

      m_rBTAuthResultTimerID = rTimerID;
   }
   ETG_TRACE_USR2(("spi_tclAAPBluetooth::vStartAAPBTAuthResultStatusTimer left "));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPBluetooth::vStopAAPBTAuthResultStatusTimer()
 ***************************************************************************/
t_Void spi_tclAAPBluetooth::vStopAAPBTAuthResultStatusTimer()
{
   ETG_TRACE_USR2(("spi_tclAAPBluetooth::vStopAAPBTAuthResultStatusTimer entered "));
   Timer* poTimer = Timer::getInstance();

   if((m_rBTAuthResultTimerID != 0) && (NULL != poTimer))
   {
      poTimer->CancelTimer(m_rBTAuthResultTimerID);

      m_rBTAuthResultTimerID = 0;
   }
   ETG_TRACE_USR2(("spi_tclAAPBluetooth::vStopAAPBTAuthResultStatusTimer left "));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTPairingStatusMsg()
 ***************************************************************************/
t_Void spi_tclAAPBluetooth::vPostBTPairingStatusMsg(tenBTSetPairingStatus enBTSetPairingStatus, t_String szBTDeviceAddress)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vPostBTPairingStatusMsg() : entered with BTPairingStatus = %d, BT Address = %s",
         ETG_ENUM(BTSET_PAIRING_STATUS, enBTSetPairingStatus),
         szBTDeviceAddress.c_str()));

   if((e8PAIRING_UNSUCCESSFUL == enBTSetPairingStatus) || (e8PAIRING_CANCELLED == enBTSetPairingStatus) ||
          ((e8PAIRING_SUCCESSFUL == enBTSetPairingStatus) && (szBTDeviceAddress == m_cpoBTInterface->szGetSelectedDevBTAddress())))
   {
      vSetPairingStatus(e8DEVICE_PAIRING_COMPLETE);
   }
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTPairingInfoMsg()
 ***************************************************************************/
t_Void spi_tclAAPBluetooth::vPostBTPairingInfoMsg(trBTPairingRequestInfo rBTPairingReqInfo)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vPostBTPairingInfoMsg() entered: BTAddress = %s ",
        rBTPairingReqInfo.szRemoteDevBTAddr.c_str()));
   ETG_TRACE_USR4(("[PARAM]:vPostBTPairingInfoMsg: PairingMethod = %d, PairingPin = %s ",
   ETG_ENUM(BTSET_PAIRING_METHOD, rBTPairingReqInfo.enPairingMethod),
        rBTPairingReqInfo.szPairingPin.c_str()));

   if (
       (NULL != m_cpoBTInterface)
       &&
       (e8DEV_TYPE_ANDROIDAUTO == m_cpoBTInterface->enGetSelectedDeviceCategory()) /*If AAP device is active*/
      )
   {
      //! If pairing is started for AAP device, accept the pairing and send PIN number to MD.
      if (
          (rBTPairingReqInfo.szRemoteDevBTAddr == m_cpoBTInterface->szGetSelectedDevBTAddress())
          &&
          (scenBTSetPreferredAAPPairingMethod == rBTPairingReqInfo.enPairingMethod)
         )
      {
          ETG_TRACE_USR2(("[DESC]:vOnBTPairingInfo: Pairing request from AA device received. "));
          spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
          spi_tclAAPCmdBluetoothIntf* poCmdBluetooth = (NULL != poAAPManager)?
               (poAAPManager->poGetBluetoothInstance()) : (NULL);
          if (NULL != poCmdBluetooth)
          {
             //TO DO: Hardcoding as first step , to reflect value in tenBTSetPairingMethod
             tenBTPairingMethod enPairingMethod = e8BT_PAIRING_NUMERIC_COMPARISON;
             poCmdBluetooth->vSendAuthenticationData(rBTPairingReqInfo.szPairingPin ,enPairingMethod);

             vStopAAPBTAuthResultStatusTimer();  /* Clear the previous timer if any. */
             vStartAAPBTAuthResultStatusTimer();
          }
      }
   }//if ((NULL != m_cpoBTInterface) && ...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTLimitationModeMsg()
 ***************************************************************************/
t_Void spi_tclAAPBluetooth::vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vPostBTLimitationModeMsg() entered "));
   ETG_TRACE_USR4(("[PARAM]::vPostBTLimitationModeMsg: "
         "Technology = %d, CommChannel = %d, ActionState = %d, BTAddress = %s ",
        ETG_ENUM(BTSET_TECHNOLOGY, rBTLimitationModeInfo.enTechnology),
        ETG_ENUM(BTSET_COMM_CHN, rBTLimitationModeInfo.enCommChannel),
        ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rBTLimitationModeInfo.enActionState),
        rBTLimitationModeInfo.szBTAddress.c_str()));
   if (
       (NULL != m_cpoBTInterface)
       &&
       (e8DEV_TYPE_ANDROIDAUTO == m_cpoBTInterface->enGetSelectedDeviceCategory()) /*If AA device is active*/
       &&
       (e8BT_TECH_ANDROIDAUTO == rBTLimitationModeInfo.enTechnology)
       &&
       (e8BT_COMM_CHN_USB == rBTLimitationModeInfo.enCommChannel)
       &&
       (rBTLimitationModeInfo.szBTAddress == m_cpoBTInterface->szGetSelectedDevBTAddress())
      )
   {
      switch (rBTLimitationModeInfo.enActionState)
      {
         case e8BT_LIM_ACTION_STATE_ACTIVE:
         {
            ETG_TRACE_USR2(("[DESC]::vPostBTLimitationModeMsg: AA BT mode activation is successful. "));
            if (e8DEVICE_PAIRING_REQUESTED == enGetPairingStatus())
            {
               ETG_TRACE_USR2(("[DESC]::vPostBTLimitationModeMsg: Sending pairing response to MD. "));
               vSendDevicePairingResponse(true);
            }
         }
         break;
         case e8BT_LIM_ACTION_STATE_ERROR:
         {
            //! TODO - Deselection is currently disabled. To be enabled again after BT limitation
            //! mode interface is extended to get more information on the error.
            //ETG_TRACE_USR2(("[DESC]::vOnBTLimitationMode: "
            //"AA BT mode activation is unsuccessful. Deactivating AA device. "));
            //vDeselectAAPDevice();
         }
         break;
         default:
           //! Add code
         break;
      }//switch (rBTLimitationMode.enActionState)
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTDeviceNameUpdateMsg()
 ***************************************************************************/
t_Void spi_tclAAPBluetooth::vPostBTDeviceNameUpdateMsg(t_String szBTDeviceAddress,t_String szBTDeviceName)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vPostBTDeviceNameUpdateMsg() entered: BTDeviceAddress = %s ", szBTDeviceAddress.c_str()));

   if ((false == szBTDeviceName.empty()) && (NULL != m_cpoBTInterface) &&
       (e8DEV_TYPE_ANDROIDAUTO == m_cpoBTInterface->enGetSelectedDeviceCategory()))
   {
      t_String szSelDevBTAddr = m_cpoBTInterface->szGetSelectedDevBTAddress();
      if (
          (IS_VALID_BT_ADDRESS(szBTDeviceAddress))
          &&
          (IS_VALID_BT_ADDRESS(szSelDevBTAddr)) /*If BT Address of active projection device is available*/
          &&
          (szBTDeviceAddress == szSelDevBTAddr) /*If BT device is active projection device*/
         )
      {
         t_U32 u32SelProjDevHandle = m_cpoBTInterface->u32GetSelectedDevHandle();
         m_cpoBTInterface->vSendBTDeviceInfo(u32SelProjDevHandle, szBTDeviceName, szSelDevBTAddr);
      }
   }//if ((false == szBTDeviceName.empty()) && (NULL != m_cpoBTInterface))
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTServiceAvailabilityMsg()
 ***************************************************************************/
t_Void spi_tclAAPBluetooth::vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable)
{
   ETG_TRACE_USR1(("spi_tclAAPBluetooth::vPostBTServiceAvailabilityMsg entered bServiceAvailable = %d",ETG_ENUM(BOOL,bServiceAvailable)));
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;
   if( (true == bServiceAvailable) &&
       (NULL != m_cpoBTInterface) &&
       (e8DEVICE_CHANGE_DELAYED == m_cpoBTInterface->enGetSelectedDevStatus()) &&
       (e8DEV_TYPE_ANDROIDAUTO == m_cpoBTInterface->enGetSelectedDeviceCategory()) &&
       (e8BT_LIM_ACTION_UNKNOWN != m_enPendingBTLimAction))
   {
       spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
       if(NULL != poExtCmdBluetoothIntf)
       {
          sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
              m_cpoBTInterface->szGetSelectedDevBTAddress(),szBTDeviceName,
              e8BT_TECH_ANDROIDAUTO,
              e8BT_COMM_CHN_USB,
              m_enPendingBTLimAction);
          if (true == sbBTLimModeRequested)
          {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
          }
          else
          {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
          }
          ETG_TRACE_USR4(("[PARAM]::spi_tclAAPBluetooth::vPostBTServiceAvailabilityMsg() - SetLimitationMode result - %d", ETG_ENUM(BOOL, sbBTLimModeRequested)));
       }
   }
}
//lint –restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
