  /*!
 *******************************************************************************
 * \file         spi_tclAAPBluetooth.h
 * \brief        AAP Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    AAP Bluetooth handler class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 04.03.2015 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 19.05.2015 |  Ramya Murthy (RBEI/ECP2)         | Added vOnBTDeviceNameUpdate()
 18.08.2015 |  Ramya Murthy (RBEI/ECP2)         | Revised BT logic based on new BTSettings interfaces
 16.07.2018 |  Roveena Francy Lobo (RBEI/ECO2)  | Added vDelayBTPairing()
 
 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLAAPBLUETOOTH_H_
#define _SPI_TCLAAPBLUETOOTH_H_

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include "AAPTypes.h"
#include "spi_BluetoothTypedefs.h"
#include "spi_tclBluetoothDevBase.h"
#include "spi_tclAAPRespBluetooth.h"
#include "spi_tclExtRespBluetooth.h"
#include "Lock.h"
#include "Timer.h"

class spi_tclExtCmdBluetoothIntf;

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/


/* Forward Declarations. */
class spi_tclBluetoothIntf;
class spi_tclAAPCmdBluetoothIntf;


/*!
* \class spi_tclAAPBluetooth
* \brief This is the AAP Bluetooth class that handles the Bluetooth
*        connection logic during a AAP device session
*/
class spi_tclAAPBluetooth : public spi_tclBluetoothDevBase, public spi_tclAAPRespBluetooth, public spi_tclExtRespBluetooth
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclAAPBluetooth::spi_tclAAPBluetooth(...)
   ***************************************************************************/
   /*!
   * \fn      spi_tclAAPBluetooth(spi_tclBluetoothIntf* poBTInterface,
   *             spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf)
   * \brief   Parameterized Constructor
   * \param   [IN] poBTInterface: Pointer to Bluetooth manager interface
   ***************************************************************************/
   spi_tclAAPBluetooth(spi_tclBluetoothIntf* poBTInterface);

   /***************************************************************************
   ** FUNCTION:  spi_tclAAPBluetooth::~spi_tclAAPBluetooth();
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclAAPBluetooth()
   * \brief   Virtual Destructor
   ***************************************************************************/
   virtual ~spi_tclAAPBluetooth();

   /*********Start of functions overridden from spi_tclBluetoothDevBase*******/

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclAAPBluetooth::bInitialise();
    ***************************************************************************/
   /*!
    * \fn      bInitialise()
    * \brief   Method to initialises the service handler. (Performs initialisations which
    *          are not device specific.)
    *          Optional interface to be implemented.
    * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
    * \sa      bUninitialise()
    ***************************************************************************/
   t_Bool bInitialise();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclAAPBluetooth::vOnSPISelectDeviceRequest(
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
   * \brief   Called when SelectDevice request is received by SPI.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \retval  None
   **************************************************************************/
   t_Void vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclAAPBluetooth::vOnSPISelectDeviceResponse(
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
   *             tenResponseCode enRespCode)
   * \brief   Called when SelectDevice operation is complete, with the result
   *          of the operation.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \param   [IN] enRespCode: Response code enumeration
   * \retval  None
   **************************************************************************/
   t_Void vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclAAPBluetooth::vOnSPISelectDeviceRequest(
   ***************************************************************************/
   /*!
   * \fn      vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
   * \brief   Called when DeselectDevice request is received by SPI.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \retval  None
   **************************************************************************/
   t_Void vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclAAPBluetooth::vOnSPIDeselectDeviceResponse(
   ***************************************************************************/
   /*!
   * \fn      vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
   *             tenResponseCode enRespCode)
   * \brief   Called when DeselectDevice operation is complete, with the result
   *          of the operation.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \param   [IN] enRespCode: Response code enumeration
   * \retval  None
   **************************************************************************/
   t_Void vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode);

   /***************************************************************************
    ** FUNCTION:  spi_tclAAPBluetooth::vDelayBTPairing()
    ***************************************************************************/
    /*!
    * \fn     bDelayBTPairing
    * \brief  This method checks for if BT pairing should be delayed or not
    *  depending on the FI general restriction bit.
    * \param  void
    **************************************************************************/
    t_Void vDelayBTPairing(t_Bool bDelayBTPairing);

   /*********End of functions overridden from spi_tclBluetoothDevBase*********/

   /********Start of functions overridden from spi_tclAAPRespBluetooth********/

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTPairingRequest()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTPairingRequest()
    * \brief   Called when Pairing request is sent by AAP device.
    *          Mandatory interface to be implemented.
    * \param   [IN] szAAPBTAddress: BT address of AAP device
    * \param   [IN] enPairingMethod: Pairing method selected by AAP device
    * \sa      None
    **************************************************************************/
   t_Void vPostBTPairingRequest(t_String szAAPBTAddress,
         tenBTPairingMethod enPairingMethod);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTAuthenticationResult()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTAuthenticationResult()
    * \brief   Called when BT Authentication Result is sent by AAP device.
    *          Mandatory interface to be implemented.
    * \param   [IN] s32AuthResult: BT Authentication Result from AAP device
    * \sa      None
    **************************************************************************/
   t_Void vPostBTAuthenticationResult(t_S32 s32AuthResult);

   /**********End of functions overridden from spi_tclAAPRespBluetooth********/

   /**************************************************************************
   ****************************END OF PUBLIC**********************************
   **************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
    ** FUNCTION: spi_tclAAPBluetooth(const spi_tclAAPBluetooth &rfcoBluetooth)
    ***************************************************************************/
   /*!
    * \fn      spi_tclAAPBluetooth(const spi_tclAAPBluetooth &rfcoBluetooth)
    * \brief   Copy constructor not implemented hence made private
    **************************************************************************/
   spi_tclAAPBluetooth(const spi_tclAAPBluetooth &rfcoBluetooth);

   /***************************************************************************
    ** FUNCTION: const spi_tclAAPBluetooth & operator=(
    **                                 const spi_tclAAPBluetooth &rfcoBluetooth);
    ***************************************************************************/
   /*!
    * \fn      const spi_tclAAPBluetooth & operator=(const spi_tclAAPBluetooth &rfcoBluetooth);
    * \brief   assignment operator not implemented hence made private
    **************************************************************************/
   const spi_tclAAPBluetooth & operator=(
            const spi_tclAAPBluetooth &rfcoBluetooth);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclAAPBluetooth::vStopBTPairing()
   ***************************************************************************/
   /*!
   * \fn      enGetPairingStatus()
   * \brief   Interface to read current pairing status of AAP device
   * \retval  tenDevicePairingStatus
   **************************************************************************/
   tenDevicePairingStatus enGetPairingStatus();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclAAPBluetooth::vSetPairingStatus()
   ***************************************************************************/
   /*!
   * \fn      vSetPairingStatus()
   * \brief   Interface to set pairing status of AAP device
   * \param   enPairingStatus : [IN] Pairing state enum
   **************************************************************************/
   t_Void vSetPairingStatus(tenDevicePairingStatus enPairingStatus);

   /***************************************************************************
   ** FUNCTION:   t_Void spi_tclDiPoBluetooth::vDeselectAAPDevice()
   ***************************************************************************/
   /*!
   * \fn      vDeselectAAPDevice()
   * \brief   Triggers deselection request for AAP device
   * \retval  None
   **************************************************************************/
   t_Void vDeselectAAPDevice();

   /***************************************************************************
   ** FUNCTION:   t_Void spi_tclAAPBluetooth::vSendDevicePairingResponse()
   ***************************************************************************/
   /*!
   * \fn      vSendDevicePairingResponse(t_Bool bReadyToPair)
   * \brief   Sends a BTPairingResponse to AAP device
   * \param   bReadyToPair [IN]: True - if HU is ready to pair, else False
   * \retval  None
   **************************************************************************/
   t_Void vSendDevicePairingResponse(t_Bool bReadyToPair);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTPairingInfoMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTPairingInfoMsg()
    * \brief   Called when BT Pairing message is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTPairingInfoMsg(trBTPairingRequestInfo rBTPairingReqInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTPairingStatusMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTPairingStatusMsg()
    * \brief   Called when BT Pairing Status is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTPairingStatusMsg(tenBTSetPairingStatus enBTSetPairingStatus, t_String szBTDeviceAddress);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTLimitationModeMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTLimitationModeMsg()
    * \brief   Called when BT limitation mode message is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTDeviceNameUpdateMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTDeviceNameUpdateMsg()
    * \brief   Called when BT Device Name update message is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTDeviceNameUpdateMsg(t_String szBTDeviceAddress,t_String szBTDeviceName);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclAAPBluetooth::vStartAAPBTAuthResultStatusTimer()
    ***************************************************************************/
   /*!
    * \fn      t_Void vStartAAPBTAuthResultStatusTimer()
    * \brief   Called to start the timer just after when Authorization Data is send to the MD,
    *          monitoring for the Authentication Result from MD.
    * \sa      None
    **************************************************************************/
   t_Void vStartAAPBTAuthResultStatusTimer();

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclAAPBluetooth::vStopAAPBTAuthResultStatusTimer()
    ***************************************************************************/
   /*!
    * \fn      t_Void vStopAAPBTAuthResultStatusTimer()
    * \brief   Called to stop the timer on cases such as deselection of MD with Timer still running, 
    *           or when Authentication Result from MD is received.
    * \sa      None
    **************************************************************************/
   t_Void vStopAAPBTAuthResultStatusTimer();

   /***************************************************************************
    ** FUNCTION:  spi_tclAAPBluetooth::bAAPBTAuthResultTimeoutCb
    ***************************************************************************/
    /*!
    * \fn     bAAPBTAuthResultTimeoutCb
    * \brief  called on expiry of component response timer
    * \param  rTimerID: ID of the timer which has expired
    * \param  pvObject: pointer to object passed while starting the timer
    * \param  pvUserData: data passed during start of the timer
    **************************************************************************/
   static t_Bool bAAPBTAuthResultTimeoutCb(timer_t rTimerID, t_Void *pvObject,
                     const t_Void *pvUserData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTServiceAvailabilityMsg()
   ***************************************************************************/
   /*!
   * \fn      t_Void vPostBTServiceAvailabilityMsg()
   * \brief   Called when BT On Service Availability is  sent by BT Client Handler.
   * \sa      None
   **************************************************************************/
   t_Void vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable);
   
   /***************************************************************************
   * ! Data members
   ***************************************************************************/

   /***************************************************************************
   ** BT Manager interface pointer
   ***************************************************************************/
   spi_tclBluetoothIntf* const   m_cpoBTInterface;

   /***************************************************************************
   ** Pairing status of AAP device
   ***************************************************************************/
   tenDevicePairingStatus  m_enPairingStatus;

   /***************************************************************************
   ** Lock object for device pairing status
   ***************************************************************************/
   Lock  m_oPairingStateLock;

   /***************************************************************************
   ** Pairing pending status
   ***************************************************************************/
   t_String m_szPairingPending ;

   /***************************************************************************
   ** Delay pairing bit
   ***************************************************************************/
   t_Bool m_bDelayBTPairing ;

   /***************************************************************************
   ** Timer ID
   ***************************************************************************/
   timer_t m_rBTAuthResultTimerID;
   
   /***************************************************************************
   ** Action of pending SetBTLimitationMode request
   ***************************************************************************/
   tenBTLimitationAction m_enPendingBTLimAction;

   
   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

};
#endif // _SPI_TCLAAPBLUETOOTH_H_

