/***********************************************************************/
/*!
* \file   spi_tclAAPCmdVideo.cpp
* \brief  Wrapper class for GalReceiver Video Sink
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Wrapper class for GalReceiver Video Sink
AUTHOR:         Shiva Kumar Gurija
COPYRIGHT:      &copy; RBEI

HISTORY:
Date         | Author                        | Modification
20.03.2015   | Shiva Kumar Gurija            | Initial Version
28.05.2015   | Tejaswini H B(RBEI/ECP2)      | Added Lint comments to suppress C++11 Errors
11.07.2015   | Ramya Murthy                  | Fix for same session ID being used for multiple Endpoints
25.11.2016   | Noopur R K                    | HMI Synchronization

\endverbatim
*************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <time.h>

//#include "AautoLogging.h"
#include "StringHandler.h"
#include "spi_tclAAPMsgQInterface.h"
#include "spi_tclAAPSessionDataIntf.h"
#include "spi_tclAAPVideoDispatcher.h"
#include "spi_tclAAPCmdVideo.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AAPWRAPPER
#include "trcGenProj/Header/spi_tclAAPCmdVideo.cpp.trc.h"
#endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
#define MAX_KEYSIZE 256
#define SEM_ERROR -1
#define SEM_TIMEOUT_IN_SEC 1

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static const t_String cszVideoCodecKey = "video-codec";
static const t_String cszMaxUnAckedFramesKey = "video-maxUnackedFrames";
static const t_String cszVideoPipelineKey = "gstreamer-video-pipeline";
static const t_String cszVideoCodecResolutionFourEighty = "video-codec-resolution-800x480";
static const t_String cszVideoCodecResolutionSevenTwenty = "video-codec-resolution-1280x720";
static const t_String cszVideoCodecResolutionTenEighty = "video-codec-resolution-1920x1080";
static const t_U32 scu32ViewingDistance = 950; //Default viewing distance in mm

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
** FUNCTION:  spi_tclAAPCmdVideo::spi_tclAAPCmdVideo()
***************************************************************************/
spi_tclAAPCmdVideo::spi_tclAAPCmdVideo() : m_poVideoSink(NULL)
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::spi_tclAAPCmdVideo entered "));
   //! load the library: enables ADIT logging for video
   adit::aauto::GstreamerEntryPoint(NULL);
}

/***************************************************************************
** FUNCTION:  spi_tclAAPCmdVideo::~spi_tclAAPCmdVideo()
***************************************************************************/
spi_tclAAPCmdVideo::~spi_tclAAPCmdVideo()
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::~spi_tclAAPCmdVideo() entered "));
   //! Unload the library : disables ADIT logging for video
   adit::aauto::GstreamerExitPoint();
   m_oEndpointLock.s16Lock();
   RELEASE_MEM(m_poVideoSink); 
   m_oEndpointLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclAAPCmdVideo::bInitialize()
***************************************************************************/
t_Bool spi_tclAAPCmdVideo::bInitialize(trAAPVideoConfig& rfrAAPVideoConfig,
         tvecAAPVideoPipeLineConfigList& rfvecAAPVideoPipeLineConfigList)
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::bInitialize entered"));
   /*lint -esym(40,nullptr) nullptr is not declared */

   t_Bool bRet = false;

   //Get the GalReceiver using session data intf
   spi_tclAAPSessionDataIntf oSessionDataIntf;
   shared_ptr<GalReceiver> spGalRcvr = oSessionDataIntf.poGetGalReceiver();

   vUninitialize();
   m_oEndpointLock.s16Lock();
   if ((spGalRcvr != nullptr) && (NULL == m_poVideoSink))
   {
      //Create GstreamerVideoSink object for Video Session
      m_poVideoSink = new (std::nothrow)adit::aauto::GstreamerVideoSink(e32SESSIONID_AAPVIDEO,
         spGalRcvr->messageRouter(),rfrAAPVideoConfig.bAutoStartProjection);

      if(NULL !=m_poVideoSink)
      {
         m_poVideoSink->setViewingDistance(scu32ViewingDistance);
         vSetVideoConfigData(rfrAAPVideoConfig);
         vSetVideoPipeLineConfig(rfvecAAPVideoPipeLineConfigList);

         m_poVideoSink->registerCallbacks(this);

         bRet = m_poVideoSink->init();

         // register for Video Sink Service with GalReciever
         if (true == bRet)
         {
            if(true == spGalRcvr->registerService(m_poVideoSink))
            {
               bRet = true;
            }//if(true == spGalRcvr->registerServic
            else
            {
               ETG_TRACE_ERR(("[ERR]:spi_tclAAPCmdVideo::bInitialize - Error in registering for Service"));
            }
         }//if(true == bRet)
         else
         {
            ETG_TRACE_ERR(("[ERR]:spi_tclAAPCmdVideo::bInitialize - error in initializing Video Sink "));
         }
      }//if(NULL != m_poVideoSink)
      else
      {
         ETG_TRACE_ERR(("spi_tclAAPCmdVideo::bInitialize - m_poVideoSink is NULL "));
      }
   } //if ((spGalRcvr != nullptr) && (NULL == m_poVideoSink))
   m_oEndpointLock.vUnlock();

   ETG_TRACE_USR4(("spi_tclAAPCmdVideo::bInitialize left with %d ", ETG_ENUM(BOOL, bRet)));

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPCmdVideo::vUninitialize()
 ***************************************************************************/
t_Void spi_tclAAPCmdVideo::vSetVideoConfigData(const trAAPVideoConfig& rfrAAPVideoConfig)
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vSetVideoConfigData() entered "));
   StringHandler oStrUtil;
   if (NULL != m_poVideoSink)
   {
      //Sets media codec type which needs to be set.In Galreceiver1.0 only "MEDIA_CODEC_VIDEO_H264_BP" is available
      ETG_TRACE_USR4(("[PARAM]:vSetVideoConfig - VideoCodec:%s", rfrAAPVideoConfig.szVideoCodec.c_str()));
      m_poVideoSink->setConfigItem(cszVideoCodecKey.c_str(), rfrAAPVideoConfig.szVideoCodec.c_str());

      //Sets the maximum unacked frames for the session.
      t_String szMaxUnAckedFrames;
      oStrUtil.vConvertIntToStr(rfrAAPVideoConfig.u8MaxUnAckedFrames, szMaxUnAckedFrames, DECIMAL_STRING);
      ETG_TRACE_USR4(("PARAM]:vSetVideoConfig - MaxUnAckedFrames:%s", szMaxUnAckedFrames.c_str()));
      m_poVideoSink->setConfigItem(cszMaxUnAckedFramesKey.c_str(), szMaxUnAckedFrames.c_str());

      //Set the config item for enabling the verbose logging.
      m_poVideoSink->setConfigItem("enable-verbose-logging", "1");

      //prepare GstreamerPipeline appsrc element value
      t_Char szVideoPipeline[MAX_KEYSIZE * 3] = { 0 };
#ifdef VARIANT_S_FTR_ENABLE_G4G
         snprintf(szVideoPipeline,MAX_KEYSIZE,
               "h264parse ! omxh264dec ! vspfilter input-color-range=full ! video/x-raw, format=BGRA ! waylandsink sync=false  qos=false surface-id=59");
#else
         sprintf(szVideoPipeline,
            "vpudec low-latency=true frame-plus=2 framedrop=false framerate-nu=30 dis-reorder=true ! "
            "gst_apx_sink display-width=%d display-height=%d sync=false qos=false "
            "max-lateness=3000000000 layer-id=%d surface-id=%d",
            rfrAAPVideoConfig.u32ScreenWidth,
            rfrAAPVideoConfig.u32ScreenHeight,
            rfrAAPVideoConfig.u16LayerID,
            rfrAAPVideoConfig.u16SurfaceID);



#endif
      ETG_TRACE_USR4(("PARAM]:vSetVideoConfig - VideoPipeline:%s", szVideoPipeline));

      //Sets the GstreamerPipeline appsrc element
      m_poVideoSink->setConfigItem(cszVideoPipelineKey.c_str(), szVideoPipeline);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPCmdVideo::vUninitialize()
***************************************************************************/
t_Void spi_tclAAPCmdVideo::vSetVideoPipeLineConfig(const tvecAAPVideoPipeLineConfigList& rfvecAAPVideoPipeLineConfigList)
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vSetVideoPipeLineConfig() entered "));

   size_t siConfigListSize = 0;
   //The value of the additional depth is set to 0 by default
   t_U16 u16AdditionalDepth = 0;
   siConfigListSize = rfvecAAPVideoPipeLineConfigList.size();
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vSetVideoPipeLineConfig() Size of the vector is : %d",siConfigListSize));
   if (NULL != m_poVideoSink)
   {
      for (size_t siIndex = 0; siIndex < siConfigListSize; siIndex++)
      {
         t_Char szVideoPipelineValue[MAX_KEYSIZE] = { 0 };
         ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vSetVideoPipeLineConfig() I am in the loop here!!! "));
         t_Double dPixelAspectRatio = static_cast<t_Double>(std::stod(rfvecAAPVideoPipeLineConfigList[siIndex].szPixelAspectRatio));

         snprintf(szVideoPipelineValue,
                  MAX_KEYSIZE,
                  "width=%d height=%d density=%d realdensity=%d pixelAspectRatio=%lf additional-depth=%d fps=%d",
                  rfvecAAPVideoPipeLineConfigList[siIndex].u16ProjScreenWidth,
                  rfvecAAPVideoPipeLineConfigList[siIndex].u16ProjScreenHeight,
                  rfvecAAPVideoPipeLineConfigList[siIndex].u16DpiDensity,
                  rfvecAAPVideoPipeLineConfigList[siIndex].u16RealDensity,
                  dPixelAspectRatio,
                  u16AdditionalDepth,
                  rfvecAAPVideoPipeLineConfigList[siIndex].u8Fps);

         t_String szVideoPipelineKey;

         ETG_TRACE_USR4(("The height to be evaluated is - %d",rfvecAAPVideoPipeLineConfigList[siIndex].u16ProjScreenHeight));
         if (rfvecAAPVideoPipeLineConfigList[siIndex].u16ProjScreenHeight <= 480)
         {
            szVideoPipelineKey = cszVideoCodecResolutionFourEighty;
         }
         else if ((480 < rfvecAAPVideoPipeLineConfigList[siIndex].u16ProjScreenHeight) && ( rfvecAAPVideoPipeLineConfigList[siIndex].u16ProjScreenHeight <= 720))
         {
            szVideoPipelineKey = cszVideoCodecResolutionSevenTwenty;
         }
         else
         {
            szVideoPipelineKey = cszVideoCodecResolutionTenEighty;
         }

         ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vSetVideoPipeLineConfig() The key is : %s ",szVideoPipelineKey.c_str()));
         ETG_TRACE_USR4(("PARAM]:vSetVideoConfig - VideoPipeline value :%s", szVideoPipelineValue));
         m_poVideoSink->setConfigItem(szVideoPipelineKey.c_str(), szVideoPipelineValue);

      }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclAAPCmdVideo::vSetVideoConfig- Video Sink is NULL"));
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPCmdVideo::vUninitialize()
***************************************************************************/
t_Void spi_tclAAPCmdVideo::vUninitialize()
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vUninitialize() entered "));
   //add code
   m_oEndpointLock.s16Lock();
   if(NULL != m_poVideoSink)
   {
      m_poVideoSink->shutdown();
   }
   RELEASE_MEM(m_poVideoSink); 
   m_oEndpointLock.vUnlock();

   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vUninitialize() left "));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPCmdVideo::vSetVideoFocus()
***************************************************************************/
t_Void spi_tclAAPCmdVideo::vSetVideoFocus(tenVideoFocus enVideoFocus,
                                          t_Bool bUnsolicited)
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vSetVideoFocus:VideoFocusMode-%d Unsolicited-%d ",
		   ETG_ENUM(VIDEOFOCUS_MODE,enVideoFocus),ETG_ENUM(BOOL,bUnsolicited)));

   m_oEndpointLock.s16Lock();
   if (NULL != m_poVideoSink)
   {
      t_S32 s32VideoFocus = static_cast<t_S32> (enVideoFocus);
      m_poVideoSink->setVideoFocus(s32VideoFocus,bUnsolicited);
   }//if (NULL != m_poVideoSink)
   else
   {
      ETG_TRACE_ERR(("spi_tclAAPCmdVideo::vSetVideoFocus:VideoSink is NULL"));
   }
   m_oEndpointLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPCmdVideo::playbackStartCallback()
***************************************************************************/
t_Void spi_tclAAPCmdVideo::playbackStartCallback()
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::playbackStartCallback() entered "));

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPCmdVideo::playbackStopCallback()
 ***************************************************************************/
t_Void spi_tclAAPCmdVideo::playbackStopCallback()
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::playbackStopCallback() entered "));

   PlaybackStopMsg oPlaybackStopMsg;
   VideoRenderingStatus oVideoRenderingStatus;
   oVideoRenderingStatus.enProjVideoRenderingStatus = e8_PROJECTION_VIDEO_RENDERING_STOPPED;
   spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      ETG_TRACE_USR1(("spi_tclAAPCmdVideo::playbackStopCallback() :: sending playback stop message"));
      poMsgQinterface->bWriteMsgToQ(&oPlaybackStopMsg, sizeof(oPlaybackStopMsg));
      ETG_TRACE_USR1(("spi_tclAAPCmdVideo::playbackStopCallback() :: sending video rendering message"));
      poMsgQinterface->bWriteMsgToQ(&oVideoRenderingStatus, sizeof(oVideoRenderingStatus));
   }//if (NULL != poMsgQinterface)
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::playbackStopCallback() left "));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPCmdVideo::setupCallback(t_S32 s32MediaCodecType)
***************************************************************************/
t_Void spi_tclAAPCmdVideo::setupCallback(t_S32 s32MediaCodecType)
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::setupCallback() entered MediaCodecType - %d",
            ETG_ENUM(MEDIACODEC_TYPE, s32MediaCodecType)));

   VideoSetupMsg oVideoSetupMsg;
   oVideoSetupMsg.m_enMediaCodecType = static_cast<tenMediaCodecTypes>(s32MediaCodecType);

   spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oVideoSetupMsg, sizeof(oVideoSetupMsg));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPCmdVideo::videoFocusCallback()
***************************************************************************/
t_Void spi_tclAAPCmdVideo::videoFocusCallback(t_S32 s32Focus, t_S32 s32Reason)
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::videoFocusCallback: FocusMode-%d, FocusReason-%d ",
      ETG_ENUM(VIDEOFOCUS_MODE,s32Focus),ETG_ENUM(VIDEOFOCUS_REASON,s32Reason)));

   VideoFocusMsg oVideoFocusMsg;
   oVideoFocusMsg.m_enVideoFocus = static_cast<tenVideoFocus>(s32Focus);
   oVideoFocusMsg.m_enVideoFocusReason = static_cast<tenVideoFocusReason>(s32Reason);

   spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oVideoFocusMsg, sizeof(oVideoFocusMsg));
   }//if (NULL != poMsgQinterface)  
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPCmdVideo::sourceVideoConfigCallback()
***************************************************************************/
t_Void spi_tclAAPCmdVideo::sourceVideoConfigCallback(t_S32 s32VideoWidth, t_S32 s32VideoHeight,
         t_S32 s32UIResWidth, t_S32 s32UIResHeight)
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::sourceVideoConfigCallback: Video Width-%d, Video Height - %d , "
            "UI resolution width - %d, UI resolution height - %d",
            s32VideoWidth, s32VideoHeight, s32UIResWidth, s32UIResHeight));

   VideoConfigMsg oVideoConfigMsg;
   oVideoConfigMsg.m_s32LogicalUIHeight = s32UIResHeight;
   oVideoConfigMsg.m_s32LogicalUIWidth  = s32UIResWidth;

   spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oVideoConfigMsg, sizeof(oVideoConfigMsg));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPCmdVideo::playbackFirstFrameRenderedCallback()
***************************************************************************/
t_Void spi_tclAAPCmdVideo::playbackFirstFrameRenderedCallback()
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::playbackFirstFrameRenderedCallback() entered "));
   PlaybackStartMsg oPlaybackStartMsg;
   spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oPlaybackStartMsg, sizeof(oPlaybackStartMsg));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPCmdVideo::vNativeVideoRenderingStatus()
 ***************************************************************************/
t_Void spi_tclAAPCmdVideo::vNativeVideoRenderingStatus(tenNativeVideoRenderingStatus enNativeVideoRenderingStatus)
{
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vNativeVideoRenderingStatus() entered "));
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vNativeVideoRenderingStatus() Display Status - %d entered ", ETG_ENUM(NATIVE_VIDEO_RENDERING_STATUS,
            enNativeVideoRenderingStatus)));
   ETG_TRACE_USR1(("spi_tclAAPCmdVideo::vNativeVideoRenderingStatus() exited"));
}
///////////////////////////////////////////////////////////////////////////////
// <EOF>
