/*!
 * \file vds_tclSimpleThread.h
 */


// ***************** C L A S S  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Interface for a sensor data source.
//!
//!   This interface is used to encapsulate all accesses to an OSAL
//!   device in a separate class. A vds_tclSimpleThread uses a data
//!   source object to read data.  The data source may access the OSAL
//!   device, or for debugging purposes deliver data from a file or dummy
//!   data.
//!
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
template <class T_osal>
class vds_tclDataSourceIf
{
   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Default Destructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual ~vds_tclDataSourceIf( ) {}

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Pure Virtual Function. Used for initializing/opening the device
      //!
      //! \return
      //!   Success code in case of success is:
      //!   - \c  true : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c false : Error in intializing the device
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual tBool bInit() = 0;

      // ***************** F U N C T I O N  H E A D E R ***************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Pure Virtual Function. Used for closing the device
      //!
      //! \return
      //!   Success code in case of success is:
      //!   - \c  true : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c false : Error in closing the device
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // --------------------------------------------------------------------------
      //***************************************************************************
      virtual tBool bShutdown() = 0;

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Pure Virtual Function. Used for Reading data from the device.
      //!
      //! \return
      //!   Number of elements read.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual tS32 s32GetData
      (
         //! (O) Buffer to store the read data
         T_osal *parData,
         //! (I) Maximum Elements to read
         tU32 u32MaxElems
      ) = 0;

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Pure Virtual Function. Returns the Cycle time
      //!
      //! \return
      //!   Returns a single constant, which is the cycle time
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual tU32 u32GetCycleTime() = 0;
};


// ***************** C L A S S  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Convert an OSAL data type to an internal VD Sensor datatype.
//!
//!   A vds_tclSimpleThread uses a data converter to convert the sensor
//!   data from the OSAL device to the corresponding VD Sensor internal
//!   data type.
//!
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
template <class T_vds, class T_osal>
class vds_tclDataConverterIf
{
   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Virtual Function. Default destructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual ~vds_tclDataConverterIf( ) {}

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Pure Virtual Function. Converts the OSAL Data Type to the VDS data type
      //!
      //! \return
      //!   Success code in case of success is:
      //!   - \c  true : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c false : Error in conversion.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual tBool bConvert
      (
         //! (O) Destination Buffer
         T_vds *poTo,
         //! (I) Source Buffer
         T_osal *poFrom
      ) = 0;
};


// ***************** C L A S S  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Factory for OSAL threads.
//!
//!   Instances of this interface can create an OSAL thread with the
//!   given entry function.  The function oCreateThread will only be
//!   called once.  The instance must provide all other thread parameters
//!   (inlcuding the thread name).
//!
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
class vds_tclThreadFactory
{
   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Virtual Function. Default destructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual ~vds_tclThreadFactory( ) {}

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Pure Virtual Function. Creates thread for the object
      //!
      //! \return
      //!   Thread ID of the new thread created.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual OSAL_tThreadID oCreateThread
      (
         //! (I) Entry function for the thread
         OSAL_tpfThreadEntry pfEntryFunction,
         //! (I) Arguments for the Entry function
         tPVoid pvArgument
      ) = 0;
};


// ***************** C L A S S  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   A generic sensor worker thread.
//!
//!   This class implements a sensor worker thread template.  All sensor
//!   type specific operations are abstracted through the
//!   vds_tclDataSourceIf and vds_tclDataConverterIf interfaces.
//!
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
template <class T_vds, class T_osal>
class vds_tclSimpleThread : public tclSensorThread
{
   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Default constructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      vds_tclSimpleThread() :
            pfs32AddList( OSAL_NULL ),
            parData( OSAL_NULL ),
            parStorage( OSAL_NULL ),
            u32NumOfData( 0 ),
            poDataSource( OSAL_NULL ),
            poDataConverter( OSAL_NULL )
      {
         pszThreadName = OSAL_NULL;
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Virtual Function. Default destructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual ~vds_tclSimpleThread()
      {
         vds_tclSimpleThread<T_vds, T_osal>::vThreadClose();
         
         parData = OSAL_NULL;
         parStorage = OSAL_NULL;
         poDataSource = OSAL_NULL;
         poDataConverter = OSAL_NULL;
         pszThreadName = OSAL_NULL;
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Pure Virtual Function. Initialises the Odometer thread
      //!
      //! \return
      //!   Success code in case of success is:
      //!   - \c  VDS_E_NO_ERROR : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c VDS_E_INVALID_PARAMETER : Invalid Parameter.
      //!   - \c VDS_E_ODOMTHREAD_ALLOC_ERROR : Allocation Error
      //!   - \c VDS_E_ODOMTHREAD_THREADCREATE_FAILED : Error in creating thread
      //!   - \c VDS_E_ODOMTHREAD_NOT_UNINITIALIZED : Thread uninitialized error
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32ThreadInit
      (
         //! (I) Name of the Thread
         tString pszThreadNameParam,
         //! (I) Entry function for the thread.
         tS32( *pfs32AddListParam )( T_vds*, tU32 ),
         //! (I) Arguments to the Source Parameter for the Conversion
         vds_tclDataSourceIf<T_osal> *poDataSourceParam,
         //! (I) Arguments to the Destination Parameter for the conversion
         vds_tclDataConverterIf<T_vds, T_osal> *poDataConverterParam
      )
      {
         tS32 s32RetVal = VDS_E_NO_ERROR;

         if ( VDS_C_S32_THREAD_UNINITIALIZED == s32GetInternalState() )
         {
            pszThreadName = pszThreadNameParam;
            pfs32AddList = pfs32AddListParam;

            if ( NULL == pfs32AddListParam )
               s32RetVal = VDS_E_INVALID_PARAMETER;

            poDataConverter = poDataConverterParam;
            poDataSource = poDataSourceParam;

            if ( poDataSource != OSAL_NULL )
            {
               if ( !poDataSource->bInit() )
               {
                  vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                             "%s: unable to init data source", pszThreadName );
                  s32RetVal = VDS_E_INVALID_PARAMETER;
               }
               else
               {
                  u32SensorCycleTime = poDataSource->u32GetCycleTime( );
               }
            }
            else
            {
               s32RetVal = VDS_E_INVALID_PARAMETER;
            }

            u32NumOfData = 2;
            parData = new T_osal[ u32NumOfData ];
            parStorage = new T_vds[ u32NumOfData ];

            // if error occured...
            if (     NULL == parData
                     || NULL == parStorage )
               s32RetVal = VDS_E_ODOMTHREAD_ALLOC_ERROR;

            // *** create thread-attribute ***
            OSAL_trThreadAttribute rThreadAttribute = {0};
            rThreadAttribute.szName       = pszThreadName;
            rThreadAttribute.pfEntry      = vds_tclSimpleThread<T_vds, T_osal>::vThreadMainWrapper;
            rThreadAttribute.pvArg        = ( tPVoid ) this;      // Argument der Startroutine
            // get thread-configuration
            if ( OSAL_ERROR == scd_s32GetThreadConfiguration( CCA_C_U16_APP_SENSOR,
                  pszThreadName,
                  &rThreadAttribute.u32Priority,
                  &rThreadAttribute.s32StackSize ) )
            {
               vTraceMsg( VDS_C_TRACELEVEL_ERROR, "Error: can't get ThreadConfiguration for %s", pszThreadName );
               s32RetVal = VDS_E_ODOMTHREAD_THREADCREATE_FAILED;
            }

            // *** create sensor-thread ***
            if ( s32RetVal == VDS_E_NO_ERROR )
            {
               if ( OSAL_ERROR == ( SensorThreadId = OSAL_ThreadCreate( &rThreadAttribute ) ) )
               {
                  vTraceMsg( VDS_C_TRACELEVEL_ERROR, "Error Creating %s.", pszThreadName );
                  s32RetVal = VDS_E_ODOMTHREAD_THREADCREATE_FAILED;
               }
            }

            // does error occured?
            if ( VDS_E_NO_ERROR == s32RetVal )
               vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
            else
               vThreadClose();
         }
         else
            s32RetVal = VDS_E_ODOMTHREAD_NOT_UNINITIALIZED;

         return s32RetVal;
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Main function of the Thread handling code. It will continually read
      //!   from the device until the thread is stopped
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tVoid vThreadMain ( tPVoid )
      {
         if ( s32GetInternalState() == VDS_C_S32_THREAD_INITIALIZED
               && 0 < u32NumOfData
               && NULL != parData )
         {
            // thread is activated
            vSetInternalState( VDS_C_S32_THREAD_RUNNING );
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "%s: started running.", pszThreadName );

            do
            {
               s32ReadData( parData, u32NumOfData );
            }
            while ( VDS_C_S32_THREAD_RUNNING == s32GetInternalState() );

            // set new state
            if ( s32GetInternalState() == VDS_C_S32_THREAD_STOP )
            {
               vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "%s: shutdown.", pszThreadName );
               vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
            }
            else
            {
               vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "%s: shutdown.", pszThreadName );
               // close handles, ...
               vThreadClose();
               vSetInternalState( VDS_C_S32_THREAD_UNINITIALIZED );
            }
         }

         OSAL_vThreadExit();
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Virtual Function. Closes the thread and releases allocated resources.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual tVoid vThreadClose ()
      {
         if ( NULL != parData )
         {
            delete [] parData;
            parData = NULL;
         }

         if ( NULL != parStorage )
         {
            delete [] parStorage;
            parStorage = NULL;
         }

         u32NumOfData = 0;
         pfs32AddList = NULL;
         tclSensorThread::vThreadClose();
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Wrapper to the Main thread running function.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      static tVoid vThreadMainWrapper
      (
         //! (I) Thread arguments
         tPVoid pvThreadArg
      )
      {
         if ( OSAL_NULL != pvThreadArg )
         {
            vds_tclSimpleThread<T_vds, T_osal> *poThread =
               static_cast<vds_tclSimpleThread<T_vds, T_osal> *>( pvThreadArg );
            poThread->vThreadMain( NULL );
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR, "SimpleThread: ThreadArg NULL" );
         }
      }

   private:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   This interface reads from the data source and stores the result
      //!   in the buffer.
      //!
      //! \return
      //!   Returns the actual number of elements read.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32ReadData
      (
         //! (O) Buffer to store the read result
         T_osal *parmData,
         //! (I) Num of elements to read
         tU32 u32NumOfElem
      )
      {
         tS32 s32NumOfElemRead = 0;

         if ( NULL       != parmData
               && OSAL_NULL != poDataSource
               && 0          <  u32NumOfElem
               && 0x80000000 >  u32NumOfElem )
         {
            s32NumOfElemRead = poDataSource->s32GetData( parmData, u32NumOfData );

            if ( ( s32NumOfElemRead > 0 ) && ( ( tU32 )s32NumOfElemRead <= u32NumOfElem ) )
            {
               vTraceMsg( VDS_C_TRACELEVEL_DATA, "%s: read %d elements",
                          pszThreadName, s32NumOfElemRead );

               if ( VDS_E_NO_ERROR != s32StoreData( parmData, ( tU32 )s32NumOfElemRead ) )
                  s32NumOfElemRead = VDS_E_ODOMTHREAD_STORAGE_ERROR;
            }
            else
            {
               s32NumOfElemRead = VDS_E_ODOMTHREAD_IOREAD_ERROR;
            }
         }
         else
            s32NumOfElemRead = VDS_E_INVALID_PARAMETER;

         return s32NumOfElemRead;
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Writes into the device, the contents of the buffer
      //!
      //! \return
      //!   Returns actual number of bytes of written into device.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32StoreData
      (
         //!  (I) Buffer containing the data to be written
         T_osal *parmData,
         //!  (I) Number of elements to write
         tU32 u32NumOfElem
      )
      {
         tS32 s32RetVal = VDS_E_NO_ERROR;

         if ( NULL != parmData
               && u32NumOfElem   > 0 )
         {
            if (     NULL != parStorage
                     && NULL != pfs32AddList
                     && poDataConverter != OSAL_NULL )
            {
               for ( tU32 u32Ndx = 0; u32Ndx < u32NumOfElem; ++u32Ndx )
               {
                  poDataConverter->bConvert( &parStorage[u32Ndx], &parmData[u32Ndx] );
               }

               // send odometer-data
               if ( VDS_E_NO_ERROR != pfs32AddList( parStorage, u32NumOfElem ) )
                  s32RetVal = VDS_E_ODOMTHREAD_STORAGE_ERROR;
            }
            else
            {
               s32RetVal = VDS_E_ODOMTHREAD_NOT_INITIALIZED;
            }
         }
         else
         {
            s32RetVal = VDS_E_INVALID_PARAMETER;
         }

         return s32RetVal;
      }

      tS32( *pfs32AddList )( T_vds *, tU32 );
      T_osal *parData;
      T_vds *parStorage;
      tU32 u32NumOfData;
      vds_tclDataSourceIf<T_osal> *poDataSource;
      vds_tclDataConverterIf<T_vds, T_osal> *poDataConverter;
      tString pszThreadName;
};


