/*!
 * \file vds_tclScratchBuffer.h
 */

#if !defined(vds_tclScratchBuffer_h)
#define vds_tclScratchBuffer_h 1

/*!
 * \brief A scratch buffer for items of type T.
 *
 * The purpose of this class is to provide a scratch buffer for short
 * time usage.  If the requested buffer size is smaller than some
 * fixed amount, a preallocated buffer is used.  Else, a new buffer of
 * the requested size is allocated.  This way, the total number of
 * allocations is kept down for the common (at least in VD Sensor) use
 * case were normally a buffer of limited size is needed and only
 * occasionally a larger buffer is used.
 *
 * The functions of this class are not thread safe.  It is the
 * responsibility of the caller to ensure that an instance of
 * vds_tclScratchBuffer is used only from one thread or within
 * mutually exclusive sections in more than one thread.
 */
template <class T>
class vds_tclScratchBuffer
{
   private:
      T *parFixedData;
      tU32 u32FixedSize;
      T *parTempData;
      tU32 u32TempSize;
      tBool bFixedUsed;
      tBool bTempUsed;

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Deallocate the preallocated fixed size buffer.
      //!
      //!   This function should only be called once.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      void vDeallocateFixed( )
      {
         if ( parFixedData != OSAL_NULL )
         {
            delete[] parFixedData;
            parFixedData = OSAL_NULL;
            u32FixedSize = 0;
         }
      }

      /*!
       * \brief Deallocate the variable size temporary buffer.
       *
       * This function should be called when the user is done with the
       * buffer.
       */
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Deallocate the variable size temporary buffer.
      //!
      //!   This function should be called when the user is done with the buffer
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      void vDeallocateTemp( )
      {
         if ( parTempData != OSAL_NULL )
         {
            delete[] parTempData;
            parTempData = OSAL_NULL;
            u32TempSize = 0;
         }
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Allocate the fixed size buffer.
      //!
      //!   This function should only be called once.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      void vAllocateFixed
      (
         //!  (I) Size of the Buffer to be allocated
         tU32 u32FixedSizeParam
      )
      {
         vDeallocateFixed();
         parFixedData = new T[u32FixedSizeParam];

         if ( parFixedData != OSAL_NULL )
         {
            u32FixedSize = u32FixedSizeParam;
         }
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Allocate a temporary buffer.
      //!
      //!   This function should be called when the user requests a buffer
      //!   larger than the preallocated fixed size buffer.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      void vAllocateTemp
      (
         //!  (I) Size of the Temporary buffer to be allocated
         tU32 u32TempSizeParam
      )
      {
         vDeallocateTemp();
         parTempData = new T[u32TempSizeParam];

         if ( parTempData != OSAL_NULL )
         {
            u32TempSize = u32TempSizeParam;
         }
      }

   public:

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Default Constructor.\n
      //!   Set up the object and allocate the fixed size buffer.
      //!
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      vds_tclScratchBuffer
      (
         //!  (I) Size of the buffer to be allocated
         tU32 u32FixedSizeParam
      ) :
            parFixedData( OSAL_NULL ),
            u32FixedSize( 0 ),
            parTempData( OSAL_NULL ),
            u32TempSize( 0 ),
            bFixedUsed( false ),
            bTempUsed( false )
      {
         vAllocateFixed( u32FixedSizeParam );
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Default Destructor.\n
      //!   Deallocate all buffers.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      ~vds_tclScratchBuffer( )
      {
         vDeallocateFixed();
         vDeallocateTemp();
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Allocate a buffer of size u32Size.
      //!
      //!   Every call to this function must be followed by a call to
      //!   vDeallocate( ) before it can be called again.
      //!
      //! \return
      //!   Pointer to an array of at least u32Size elements or OSAL_NULL.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      T *poAllocate
      (
         //! (I) Number of elements in the buffer.
         tU32 u32Size
      )
      {
         if ( bFixedUsed || bTempUsed )
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "Scratch buffer already in use" );
            return OSAL_NULL;
         }

         if ( u32Size <= u32FixedSize )
         {
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
                       "Scratch buffer: %d requested, fixed size buffer (%d) used",
                       u32Size, u32FixedSize );
            bFixedUsed = true;
            return parFixedData;
         }
         else
         {
            bTempUsed = true;
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
                       "Scratch buffer: %d requested, temporary buffer used",
                       u32Size );
            vAllocateTemp( u32Size );
            return parTempData;
         }
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Deallocate buffer previously allocated by a call to poAllocate( ).
      //!
      //!   When this function has returned, poAllocate() may be called again.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      void vDeallocate( )
      {
         if ( bFixedUsed )
         {
            bFixedUsed = false;
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
                       "Scratch buffer: fixed size buffer released" );
         }
         else if ( bTempUsed )
         {
            bTempUsed = false;
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
                       "Scratch buffer: temporary buffer released" );
            vDeallocateTemp( );
         }
      }
};

#endif // #if !defined(vds_tclScratchBuffer_h)

