/*!
 * \file vds_tclHistoryProperty.h
 */

#if !defined(vds_tclHistoryProperty_h)
#define vds_tclHistoryProperty_h 1

/*!
 * \brief Common functionality for all history properties.
 *
 * A history property is a property whose value is a list of some type
 * T.  The elements of the list represent the values of the property
 * at different times.  Each element should contain a time stamp
 * field.  For each client, VD Sensor keeps track of which elements it
 * has already seen and at each update, it sends only those elements,
 * that the client has not yet seen.
 *
 * History properties are used for the sensor data for all sensors.
 *
 * This class is abstract and is meant to be the base class for type
 * specific derived classes.  A derived class has to implement a
 * number of abstract methods to handle some aspects of sensor data
 * handling that differ between the sensor data types.  These are:
 *
 * - vDataAdded()
 * - vMarkBufferOverflow()
 * - vMarkAsInvalid()
 * - poCreateMessage()
 * - bMaySend()
 */
template <class T>
class vds_tclHistoryProperty
{
   protected:
      tU16 u16FunctionId;

      vds_tclRingBuffer<T> oRingBuffer;
      vds_tclScratchBuffer<T> oScratchBuffer;

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Perform actions when data are added to the ring buffer.
      //!
      //!    Derived classes may use this function to implement sensor data
      //!    synchronisation actions, such as triggering a synch event or
      //!    checking for a synch event time out.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual void vDataAdded( ) = 0;

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Mark data to signal that a buffer overflow occured.
      //!
      //!   The usual way to do this is to set an INTERNAL_ERROR status flag
      //!   or similar on the first data element.
      //!   When this function is called, it is guaranteed that the first
      //!   data element of prData is the oldest remaining element in the
      //!   ring buffer.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual void vMarkBufferOverflow
      (
         //!  (I) Data in RingBuffer
         T* prData,
         //!  (I) Number of Elements in the buffer
         tU32 u32Elems
      ) = 0;

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Mark data as invalid.
      //!
      //!   The usual way to do this is to set an INVALID status flag on all
      //!   elements.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual void vMarkAsInvalid
      (
         //!  (I) Data in Ring Buffer
         T* prData,
         //!  (I) Number of elements
         tU32 u32Elems
      ) = 0;

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Create a CCA status message of the proper type.
      //!
      //!    The message contents should be filled with prData.  The target
      //!    application ID should be set to u16TargetAppId.  The sender
      //!    application ID should be set to CCA_C_U16_APP_SENSOR.
      //!
      //! \return
      //!   Pointer to an object containing the CCA Status Message
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual amt_tclServiceData* poCreateMessage
      (
         //! (I) Data in RingBuffer
         T* prData,
         //! (I) Number of elements
         tU32 u32Elems,
         //! (I) Target App Id
         tU16 u16TargetAppId
      ) = 0;

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    May CCA messages be sent?
      //!
      //!    This function may e.g. look at the internal state of an
      //!    associated message interface class.
      //!
      //! \return
      //!   True if permission granted, else False.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual tBool bMaySend() = 0;

      vds_tclCriticalSection& rfoCS;
      tS32 ( *pfs32SendSubscriberMessage )(
         amt_tclServiceData *,
         tclSubscriberManager * );

   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Base set up.
      //!
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      vds_tclHistoryProperty<T>
      (
         //! (I) CCA function ID of the property.
         tU16 u16FunctionIDParam,
         //! (I) reference to the critical section to be used
         //! to protect the ring buffer.  The critical section must have
         //! constructed, but it is not necessary for it to have been
         //! initialized yet.
         vds_tclCriticalSection& rfoCSParam
      ) :  u16FunctionId( u16FunctionIDParam ),
            oScratchBuffer( 20 ),
            rfoCS( rfoCSParam ),
            pfs32SendSubscriberMessage( OSAL_NULL )
      {
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Default Destructor
      //!
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual ~vds_tclHistoryProperty<T>( )
      {
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Set up things that are unknown when the constructor is called.
      //!
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32Init
      (
         //! (I) number of elements in the ring buffer.
         tU32 u32RingBufferSize,
         //! (I) pointer to the function used to send CCA messages.
         tS32 ( *pfs32SendSubscriberMessageParam )
         ( amt_tclServiceData *, tclSubscriberManager * )
      )
      {
         if ( pfs32SendSubscriberMessageParam != OSAL_NULL )
         {
            pfs32SendSubscriberMessage = pfs32SendSubscriberMessageParam;
         }
         else
         {
            return VDS_E_INVALID_PARAMETER;
         }

         oRingBuffer.vAllocate( u32RingBufferSize );

         if ( oRingBuffer.bIsValid() )
         {
            return VDS_E_NO_ERROR;
         }
         else
         {
            return VDS_E_ALLOC_ERROR;
         }
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Add data to the ring buffer.
      //!
      //!
      //! \return
      //!   -  \c  VDS_E_NO_ERROR          : Success. No Error.\n
      //!   -  \c  VDS_E_ALLOC_ERROR       : Error in allocating Memory.\n
      //!   -  \c  VDS_E_INVALID_PARAMETER : Invalid Parameter.\n
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32AddList
      (
         //!  (I) RingBuffer Data
         const T *prData,
         //!  (I) Number of elements
         tU32 u32NumOfData
      )
      {
         tS32 s32RetVal = VDS_E_NO_ERROR;

         if ( NULL != prData && 0 != u32NumOfData )
         {
            // consistence check
            if ( oRingBuffer.bIsValid() && rfoCS.bEnter() )
            {
               oRingBuffer.u32AddList( prData, u32NumOfData );
               rfoCS.vLeave();
               vDataAdded();
            }
            else
            {
               vTraceMsg(
                  VDS_C_TRACELEVEL_ERROR,
                  "vds_tclHistoryProperty: s32AddList: not init." );
               s32RetVal = VDS_E_ALLOC_ERROR;
            }
         }
         else
         {
            vTraceMsg(
               VDS_C_TRACELEVEL_ERROR,
               "vds_tclHistoryProperty: s32AddList: invalid param" );
            s32RetVal = VDS_E_INVALID_PARAMETER;
         }

         return s32RetVal;
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Get a specified numeber of data elements.
      //!
      //!
      //! \return
      //!   The number of elements copied (a positive number), 0 if
      //!   no new elements were found in the ring buffer or
      //!   VDS_E_INVALID_PARAMETER if an error occured.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32GetData
      (
         //! (I) pointer to buffer to fill with data.
         T *prData,
         //! (I) index in the ring buffer to start
         //! from.  After getting data, this value will be updated to the
         //! index of the next data element in the ring buffer (which may or
         //! may not be present yet).
         tU32 &rfu32NextElemToRead,
         //! (I) maximum number of data elements to get.
         tU32 u32ElemsToRead
      )
      {
         tS32 s32ElemsCopied = 0;

         if ( NULL != prData && 0 != u32ElemsToRead )
         {
            if ( oRingBuffer.bIsValid() && rfoCS.bLocked() )
            {
               tU32 u32Elems;
               u32Elems = oRingBuffer.u32GetList(
                             prData,
                             u32ElemsToRead,
                             rfu32NextElemToRead );

               if ( u32Elems != u32ElemsToRead )
                  vMarkBufferOverflow( prData, u32Elems );

               s32ElemsCopied = ( tS32 )u32Elems;
            }
            else
            {
               vTraceMsg(
                  VDS_C_TRACELEVEL_ERROR,
                  "vds_tclHistoryProperty: s32GetData: not init." );
               s32ElemsCopied = VDS_E_INVALID_PARAMETER;
            }
         }
         else
         {
            vTraceMsg(
               VDS_C_TRACELEVEL_ERROR,
               "vds_tclHistoryProperty: s32GetData: invalid param" );
            s32ElemsCopied = VDS_E_INVALID_PARAMETER;
         }

         return s32ElemsCopied;
      }
      
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Get Number of elements in the ring buffer.
      //! \return
      //!    The number of elements in the buffer.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************      
      tU32 u32NumOfElmInBuf(tVoid)
      {
         return (oRingBuffer.u32GetNumElems());
      }
	  
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Get Number of elements in the ring buffer from the specific element.
      //! \return
      //!    The number of elements in the buffer from "u32FromElem" element.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tU32 u32GetNumOfElemsFrom(tU32 u32FromElm)
      {
         return (oRingBuffer.u32GetElemsFrom(u32FromElm));
      }
      
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Create a CCA message and fill it with data from the ring buffer.
      //!
      //!    This function reads all available data elements from the ring
      //!    buffer and creates and fills a CCA message by calling
      //!    poCreateMessage with the data.
      //!
      //! \return
      //!   The number of data elements in the CCA message (a
      //!   positive value or zero) or an error code (a negative value).
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32GetMessage
      (
         //! (I) Pointer to a variable where the CCA message object is stored.
         amt_tclServiceData **ppoServiceData,
         //! (I) Index of the next element to read.
         tU32& rfu32NextElemToRead,
         //! (I) Target ID for the CCA message.
         tU16 u16TargetAppId
      )
      {
         tS32 s32RetVal = 0;
         T *parTempGyroData = NULL;
         tU32 u32NumOfElemToSend = 0;
         tU32 u32TempNumOfData = 0;

         // param-check
         if ( NULL != ppoServiceData )
         {
            *ppoServiceData = NULL;

            if ( rfoCS.bLocked() )
            {
               u32NumOfElemToSend = oRingBuffer.u32GetElemsFrom( rfu32NextElemToRead );

               // elems available ?
               if ( u32NumOfElemToSend > 0 )
               {
                  u32TempNumOfData = u32NumOfElemToSend;
                  parTempGyroData = oScratchBuffer.poAllocate( u32TempNumOfData );

                  if ( NULL != parTempGyroData )
                  {
                     if ( ( tS32 )u32NumOfElemToSend != s32GetData(
                              parTempGyroData, rfu32NextElemToRead, u32NumOfElemToSend ) )
                     {
                        vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                                   "HP: s32GetMessage: alloc-error" );
                        s32RetVal = VDS_E_GYROIF_ALLOC_ERROR;
                     }

                     amt_tclServiceData *poStatus = poCreateMessage(
                                                       parTempGyroData, u32TempNumOfData, u16TargetAppId );

                     if ( poStatus != OSAL_NULL )
                     {
                        *ppoServiceData = poStatus;
                        s32RetVal = ( tS32 )u32TempNumOfData;
                     }
                  }

                  oScratchBuffer.vDeallocate( );
               } // end if elems available
               else
               {
                  // no elem to send
                  amt_tclServiceData *poStatus = poCreateMessage(
                                                    OSAL_NULL, 0, u16TargetAppId );

                  if ( poStatus != OSAL_NULL )
                  {
                     *ppoServiceData = poStatus;
                     s32RetVal = 0;
                  }
               }
            } // end if oCS.bLocked
            else
               s32RetVal = VDS_E_GYROIF_NOT_INITIALIZED;
         }
         else
            s32RetVal = VDS_E_INVALID_PARAMETER;

         return s32RetVal;
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Send update messages to all registered clients.
      //!
      //! \return
      //!   VDS_E_NO_ERROR or error code.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32CheckForSubscriber ( )
      {
         tS32 s32RetVal = VDS_E_NO_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_DATA, "HP: s32CheckForSubscriber" );

         if ( bMaySend() )
         {
            // get subscriber
            if ( vds_goMasterLock.bEnter() )
            {
               tclSubscriberManager *poSubscriber
                      = tclSubscriberManager::poGetSubscriberWithFId( NULL,
                              CCA_C_U16_SRV_SENSOR_LOCATION,
                              u16FunctionId );

               // while subscribers are available...
               while ( NULL != poSubscriber )
               {
                  tU32 u32NextElemToRead = poSubscriber->u32GetNextElemToRead();
                  tS32 s32TempErrorValue = VDS_E_NO_ERROR;
                  tS32 s32ElemsCreated = 0;

                  do
                  {
                     amt_tclServiceData *poServiceData = NULL;

                     if ( rfoCS.bEnter() )
                     {
                        s32ElemsCreated = s32GetMessage( &poServiceData,
                                                         u32NextElemToRead,
                                                         poSubscriber->u16GetTargetAppId() );
                        vTraceMsg( VDS_C_TRACELEVEL_DATA, "HP: s32GetMessage -> %d", s32ElemsCreated );
                        rfoCS.vLeave();
                     }
                     else
                     {
                        s32RetVal = VDS_E_INVALID_PARAMETER;
                     }

                     if ( ( s32ElemsCreated > 0 )&&
                          ( NULL != poServiceData )&&
                          ( OSAL_NULL != pfs32SendSubscriberMessage ) )
                     {
                        //Send message
                        if ( VDS_E_NO_ERROR == pfs32SendSubscriberMessage( poServiceData,poSubscriber ) )
                        {
                           // set next elem to read
                           poSubscriber->vSetNextElemToRead( u32NextElemToRead );
                        }
                        else
                        {
                           s32TempErrorValue = VDS_E_POST_MESSAGE_FAILED;
                        }

                        delete poServiceData;
                     }
                     else if ( s32ElemsCreated < 0 )
                     {
                        s32TempErrorValue = s32ElemsCreated;
                     }
                  }while ( ( s32ElemsCreated > 0 )&&
                           ( VDS_E_NO_ERROR == s32TempErrorValue ) );

                  //if error occurs in loop and no error is set, take over temp error-value
                  if ( ( VDS_E_NO_ERROR == s32RetVal )&&
                       ( VDS_E_NO_ERROR != s32TempErrorValue ) )
                  {
                     s32RetVal = s32TempErrorValue;
                  }

                  //Get next subscriber with this function-id
                  poSubscriber = tclSubscriberManager::poGetSubscriberWithFId( poSubscriber,
                                                                               CCA_C_U16_SRV_SENSOR_LOCATION,
                                                                               u16FunctionId );
               }// end while (subscribers available)

               // leave critical section
               vds_goMasterLock.vLeave();
            } // end if lock subscriber successful

            // else no subscriber exists, return VDS_E_NO_ERROR
         }

         return s32RetVal;
      }
};

#endif // #if !defined(vds_tclHistoryProperty_h)
