/*!
 *  \file
 *           tclSubscriberManager.h
 *  \par SW_COMPONENT
 *   VD-Sensor
 *  \brief
 *    class-declaration: adds and deletes subscribers
 *                 to a list, stores response-information (TargetAppId, ...)
 *  \par AUTHOR
 *         CM-DI/ESA1-Fischer
 *  \par COPYRIGHT
 *      (c) 2002 Blaupunkt GmbH
 *
 *  \date  20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
 *            Initial Revision.
 */

#ifndef tclSubscriberManager_h
#define tclSubscriberManager_h 1

/*!
 * \brief Abstract interface for synchronisation.
 *
 * Implementors of this interface may be registered at a
 * vds_tclSynchManager.  They will then be notified of synchronisation
 * events.
 *
 * Synchronisation events are used to implement synchronous sending of
 * sensor data messages.  Gyro and odometer data are sampled
 * continously but not sent until a syncronisation event occurs, which
 * is when a new GPS position has been calculated.
 */
class vds_tclSynchListenerIf
{
   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Virtual Function. Default destructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual ~vds_tclSynchListenerIf()
      {
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Notify that a synchronisation event has occurred.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual void vSynch( void ) = 0;
};

/*!
 * \brief Debugging implementation of vds_tclSynchListenerIf.
 */
class vds_tclTraceSynchListener : public vds_tclSynchListenerIf
{
   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Output trace message.
      //
      //   return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual void vSynch( void );
};

/*!
 * \brief Manages synchronisation listeners.
 *
 * A synchronisation manager keeps track of synchronisation listeners
 * and notifies them when synchronisation events occur.
 *
 * The synchronisation manager also has a primitive time out
 * mechanism.  It relies on the function bCheckTimeOut() is called
 * regularly.  If the function determines that a configurable maximum
 * time delay has elapsed since the last synchronisation event, a
 * synchronisation event is generated and the registered listeners are
 * called anyway.
 */
class vds_tclSynchManager
{
   private:
      /*!
       * \brief Maximum number of child listeners.
       */
      static const int iMaxChildren = 10;

      /*!
       * \brief An array of pointers to the child listeners.
       */
      vds_tclSynchListenerIf *arpoChildren[iMaxChildren];

      /*!
       * \brief Current number of child listeners.
       */
      int iChildren;

      /*!
       * \brief The value of the OSAL millisecond counter at the last
       * synchronisation event.
       */
      tU32 u32LastSynchEventMs;

      /*!
       * \brief Flag for time out control.
       *
       * This flag is TRUE if the last synchronisation event was a normal
       * event caused by a client calling vSynch().  It is FALSE if it
       * was generated by the timeout mechanism.
       */
      tBool bLastSynchEventNormal;

      /*!
       * \brief The number of milliseconds after a synchronisation event
       * when a time out occurs for the first time.
       */
      tU32 u32TimeOutFirstMs;

      /*!
       * \brief The number of milliseconds between subsequent time outs.
       */
      tU32 u32TimeOutSecondMs;


      void vSynchInternal( tBool bNormal );

   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Default Constructor
      //
      //   return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      vds_tclSynchManager();

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Default Destructor
      //
      //   return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      ~vds_tclSynchManager();

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   Add a listener to the list of children.
      //
      //  return
      //   TRUE if successful; FALSE if poListener could not be added.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tBool bAddListener( vds_tclSynchListenerIf *poListener );

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   Set time out period.
      //
      //  return
      //   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      void vSetTimeOut( tU32 u32FirstMs, tU32 u32SecondMs );

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   Check if a time out has occurred.
      //
      //  return
      //   TRUE if timeout has occured, else FALSE
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tBool bCheckTimeOut( void );

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   Notifies all registered listeners that a synchronisation
      //   event has occurred.
      //
      //   When a client wishes to synchronise with the registered clients
      //   (e.g. when a GPS position has been calculated and all
      //   accumulated data from the other sensors should be sent to the
      //   registered CCA clients), it should call this function. The
      //   vSynch member functions of all registered listeners are then
      //   called.
      //
      //  return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      void vSynch( );
};

/*!
 * \brief The synchronisation manager for CCA sensor properties.
 *
 * This global synchronisation manager is used by the message
 * interface classes to synchronise the CCA messages that are sent to
 * clients. The GPS interface generates synchronisation events and the
 * gyro and odometer interfaces listen for them.
 */
extern vds_tclSynchManager vds_goSensorPropertySynchManager;

#define VDS_C_S32_SUBSCRIBER_UNINITIALIZED   (tS32)0
#define VDS_C_S32_SUBSCRIBER_INITIALIZED  (tS32)1

// errors
#define VDS_E_SUBSCRIBER_SEM_ERROR       -1
#define VDS_E_SUBSCRIBER_ALREADY_INITIALIZED -2
#define VDS_E_SUBSCRIBER_UNINITIALIZED      -3

typedef struct
{
    tU32 u32NextGnssElementToRead;
    tU32 u32NextGyroElementToRead;
    tU32 u32NextOdoElementToRead;
    tU32 u32NextAccElementToRead;
    tU32 u32NextAbsElementToRead;

}Vds_AllSenDataNextElemToSend;

//! \brief Subscriber Manager Class.
class tclSubscriberManager
{
   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   Default constructor
      //
      //
      //  return
      //   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
      //**************************************************************************
      tclSubscriberManager();

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   Constructor
      //
      //   adds subscriber to double-linked list.  if first subscriber,
      //   store pointer in poFirstSubscriber
      //
      //  return
      //    None
      //  HISTORY:
      //       Date      |        Author           |       MODIFICATION
      // -------------------------------------------------------------------------
      //     04.04.02    |  CM-DI/ESA1-Fischer     |   Initial Revision.
      //     23.04.02    |  CM-DI/ESA1-Fischer     | sub-id and command-cntr added
      //**************************************************************************
      tclSubscriberManager (
         tU16 u16RegisterId,
         tU16 u16ServiceId,
         tU16 u16FunctionId,
         tU16 u16TargetAppId,
         tU16 u16MessageSubId,
         tU16 u16CommandCntr,
         tU8 u8ACT );

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   Destructor
      //
      //
      //  return
      //   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
      //**************************************************************************
      ~tclSubscriberManager();


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   initializes subscriber-list, creates semaphore
      //
      //  return
      //    ok    - VDS_E_NO_ERROR;
      //    error - VDS_E_SUBSCRIBER_ALREADY_INITIALIZED or
      //            VDS_E_SUBSCRIBER_SEM_ERROR
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
      //**************************************************************************
      static tS32 s32Init ();

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   deletes subscribers from list, deletes semaphore
      //
      //  return
      //    ok    - VDS_E_NO_ERROR;
      //    error - VDS_E_SUBSCRIBER_UNINITIALIZED or VDS_E_SUBSCRIBER_SEM_ERROR
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
      //**************************************************************************
      static tS32 s32Close ();

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //     adds subscriber to linked list
      //
      //  return
      //    ok    - VDS_E_NO_ERROR;
      //    error - VDS_E_SUBSCRIBER_UNINITIALIZED
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
      //**************************************************************************
      tS32 s32AddSubscriber ();

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //     deletes subscriber from linked list
      //
      //  return
      //    ok    - VDS_E_NO_ERROR;
      //    error - VDS_E_SUBSCRIBER_UNINITIALIZED
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
      //**************************************************************************
      tS32 s32DeleteSubscriber ();

      // *****************************************************************************
      // * METHOD:      tclSubscriberManager::poGetSubscriberWithFId
      // * DESCRIPTION: returns pointer to subscriber with given service-id/function-id.
      // *
      // * PARAMETER:
      // *              tclSubscriberManager *poSubscriberManager: (->I)
      // *                 return successor to this subscriber
      // *              tU16 u16ServiceId: (I)
      // *                 return successor with this service-id
      // *              tU16 u16FunctionId: (I)
      // *                 return successor with this function-id
      // *
      // * RETURNVALUE: tclSubscriberManager *poSubscriberManager:
      // *                 if poSubscriberManager is NULL, the first subscriber with
      // *                    u16ServiceId/u16FunctionId will be returned.
      // *                 if poSubscriberManager is valid, the first successor with
      // *                    u16ServiceId/u16FunctionId will be returned.
      // *                 if no (successor)-subscriber with u16ServiceId/u16FunctionId exists,
      // *                    NULL will be returned
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // * 27.06.03 R. Schedel, 3SOFT
      // *          Added service id.
      // *****************************************************************************
      static tclSubscriberManager* poGetSubscriberWithFId ( const tclSubscriberManager* poSubscriberManager,
            tU16 u16ServiceId,
            tU16 u16FunctionId );

      // *****************************************************************************
      // * METHOD:      tclSubscriberManager::poGetSubscriberWithRegId
      // * DESCRIPTION: returns pointer to subscriber with given RegisterId.
      // *
      // * PARAMETER:
      // *              tclSubscriberManager *poSubscriberManager: (->I)
      // *                 return successor to this subscriber
      // *              tU16 u16RegisterId: (I)
      // *                 return successor with this RegisterId
      // *
      // * RETURNVALUE: tclSubscriberManager *poSubscriberManager:
      // *                 if poSubscriberManager is NULL, the first subscriber with
      // *                    u16RegisterId will be returned.
      // *                 if poSubscriberManager is valid, the first successor with
      // *                    u16RegisterId will be returned.
      // *                 if no (successor)-subscriber with u16RegisterId exists,
      // *                    NULL will be returned
      // *
      // * HISTORY:
      // * 28.05.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      static tclSubscriberManager* poGetSubscriberWithRegId ( const tclSubscriberManager* poSubscriberManager, tU16 u16RegisterId );

      // *****************************************************************************
      // * METHOD:      tclSubscriberManager::vResetSubscribers
      // * DESCRIPTION: sets all subscribers to 'no init-msg sent'
      // *
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tVoid
      // *
      // * HISTORY:
      // * 28.05.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      static tVoid vResetSubscribers ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set Asynchronous Control Token
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tVoid vSetACT 
      (
         //!  (I) : set Asynchronous Control Token
         tU8 u8ACT
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   get Asynchronous Control Token
      //! \return
      //!   tU8 - Asynchronous Control Token
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tU8 u8GetACT 
      (
         //! None
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set register-id
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tVoid vSetRegisterId 
      (
         //!  (I) : register-id 
         tU16 u16RegisterId 
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   get register-Id
      //! \return
      //!   tU16 - register-id
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tU16 u16GetRegisterId 
      (
         //! None
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set service-id
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tVoid vSetServiceId 
      (
         //!  (I) : service-id 
         tU16 u16ServiceId 
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   get service-Id
      //! \return
      //!   tU16 - service-id
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tU16 u16GetServiceId
      (
         //! None
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set function-id
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tVoid vSetFunctionId 
      (
         //!  (I) : function-id 
         tU16 u16FunctionId 
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   get function-id
      //! \return
      //!   tU16 - function-id
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tU16 u16GetFunctionId 
      (
         //! None
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set target-application-id
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tVoid vSetTargetAppId 
      (
         //!  (I) : target-application-id 
         tU16 u16TargetAppId 
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   get target-application-id
      //! \return
      //!   tU16 - target-application-id
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tU16 u16GetTargetAppId 
      (
         //! None
      );
    
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set sub-id of message
      //!   source-sub-id of UpReg-message will become
      //!   target-sub-id of status-message
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 23.04.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tVoid vSetMessageSubId 
      (
         //!  (I) : sub-id
         tU16 u16MessageSubId
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   get message-sub-id
      //! \return
      //!   tU16 - message-sub-id
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 23.04.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************  
      tU16 u16GetMessageSubId 
      (
         //! None
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set command-counter of message
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 23.04.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tVoid vSetCommandCntr 
      (
         //!  (I) : command-counter
         tU16 u16CommandCntr
      );
     
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   get command-counter
      //! \return
      //!   tU16 - command-counter
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 23.04.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //****************************************************************************** 
      tU16 u16GetCommandCntr 
      (
         //! None
      );
     
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set next element to read
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tVoid vSetNextElemToRead 
      (
         //!  (I) : next element to read
         tU32 u32NextElemToRead 
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   get next element to read
      //! \return
      //!   tU32 - next element to read
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tU32 u32GetNextElemToRead 
      (
         //! None
      );
      
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set next elements to read of AllSensorData property.
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 13.08.2014  | Sanjay                 |Function to update next elements to send
      //             | Gurugubelli(RBEI/ECF5) |to AllSensorData property subscribers
      //******************************************************************************
      tVoid vSetAllSen_NxtElmToRead
      (
         //! (I) : structure which holds next elements to read
         Vds_AllSenDataNextElemToSend Str
      );
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   get next elements to read of AllSensorData property.
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 13.08.2014  | Sanjay                 |Function to get next elements to send
      //             | Gurugubelli(RBEI/ECF5) |to AllSensorData property subscribers
      //******************************************************************************
      tVoid vGetAllSen_NxtElmToRead
      (
         //! (O) : structure which holds next elements to read
         Vds_AllSenDataNextElemToSend &Str
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set flag 'init message-sent'.
      //!   after this flag was set, only update-messages will be sent
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.            
      // 22.05.02  | CM-DI/ESA1-Fischer    | Parameter bInitMessageSent added.
      //******************************************************************************
      tVoid vSetInitMessageSent 
      (
         //! (I) : TRUE : init-message sent\n
         //!       FALSE: send init-message needed
         tBool bInitMessageSent
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   returns state of flag 'init message sent'
      //! \return
      //!   - \c TRUE : send update-messages
      //!   - \c FALSE : init-message has to be sent
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tBool bGetInitMessageSent 
      (
         //! None
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set flag 'message-lock needed'.
      //!   tclAilVDSensor will be locked, when this flag is set
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 22.05.02  | CM-DI/ESA1-Fischer    | Initial Revision.            
      //******************************************************************************
      tVoid vSetMessageLockNeeded 
      (
         //! (I) : TRUE - Lock needed\n
         //!       FALSE - Not needed
         tBool bMessageLockNeeded 
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   returns state of flag 'message-lock needed'
      //! \return
      //!   - \c TRUE : message-lock needed
      //!   - \c FALSE : Not needed
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 22.05.02  | CM-DI/ESA1-Fischer    | Initial Revision.                                            
      //******************************************************************************
      tBool bGetMessageLockNeeded 
      (
         //! None
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   set value 'msg-send-interval'.
      //!   msgs will be send in this intervall periodically
      //! \return
      //!   None
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 26.11.02  | CM-DI/ESA1-Fischer    | Initial Revision.            
      //******************************************************************************
      tVoid vSetMsgSendInterval
      (
         //! (I) : value-range: 0x1-0xFFFFFFFF,
         //!       unit depends on message
         tU32 u32NewMsgSendInterval
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   returns msg-send-interval
      //! \return
      //!   tU32 - msg-send-interval
      //  HISTORY:
      // Date      |  Author               | MODIFICATION
      // ----------------------------------------------------------------------------
      // 26.11.02  | CM-DI/ESA1-Fischer    | Initial Revision.            
      //******************************************************************************
      tU32 u32GetMsgSendInterval 
      (
         //! None
      );

   private:
      static OSAL_tSemHandle hSubscriberSemaphore;
      static tCString coszSubscriberSemName;
      // pointer to first subscriber in list.
      // if no one is subscribed -> NULL
      static tclSubscriberManager* poFirstSubscriber;
      static tS32 s32InternalState;
      tU16 u16RegisterId;
      tU16 u16ServiceId;
      tU16 u16FunctionId;
      tU16 u16TargetAppId;
      tU16 u16MessageSubId;
      tU16 u16CommandCntr;
      // Asynchronous Completion Token
      tU8 u8ACT;
      tclSubscriberManager* poPrevSubscriber;
      tclSubscriberManager* poNextSubscriber;
      // this attribut is used to manage which data-elements from tclMsg*If have to
      // be sent. Elements in these Objects are numbered [0x1..0xFFFFFFFF].
      tU32 u32NextElemToRead;
      // this attribut is used to manage how often msgs have to be send. Elements in
      // these Objects are numbered [0x1..0xFFFFFFFF], unit depends on the message,
      // for example seconds. Zero means no periodically update-messages.
      tU32 u32MsgSendInterval;
      tBool bInitMessageSent;
      tBool bMessageLockNeeded;
      Vds_AllSenDataNextElemToSend rAllSenDataNextElemToSend;
};

inline tVoid tclSubscriberManager::vSetACT ( tU8 u8ACTParam )
{
   this->u8ACT = u8ACTParam;
}

inline tU8 tclSubscriberManager::u8GetACT ()
{
   return ( u8ACT );
}

inline tVoid tclSubscriberManager::vSetRegisterId ( tU16 u16RegisterIdParam )
{
   this->u16RegisterId = u16RegisterIdParam;
}

inline tU16 tclSubscriberManager::u16GetRegisterId ()
{
   return ( u16RegisterId );
}

inline tVoid tclSubscriberManager::vSetServiceId ( tU16 u16ServiceIdParam )
{
   this->u16ServiceId = u16ServiceIdParam;
}

inline tU16 tclSubscriberManager::u16GetServiceId ()
{
   return ( u16ServiceId );
}

inline tVoid tclSubscriberManager::vSetFunctionId ( tU16 u16FunctionIdParam )
{
   this->u16FunctionId = u16FunctionIdParam;
}

inline tU16 tclSubscriberManager::u16GetFunctionId ()
{
   return ( u16FunctionId );
}

inline tVoid tclSubscriberManager::vSetTargetAppId ( tU16 u16TargetAppIdParam )
{
   this->u16TargetAppId = u16TargetAppIdParam;
}

inline tU16 tclSubscriberManager::u16GetTargetAppId ()
{
   return ( u16TargetAppId );
}

inline tVoid tclSubscriberManager::vSetMessageSubId ( tU16 u16MessageSubIdParam )
{
   this->u16MessageSubId = u16MessageSubIdParam;
}

inline tU16 tclSubscriberManager::u16GetMessageSubId ()
{
   return ( u16MessageSubId );
}

inline tVoid tclSubscriberManager::vSetCommandCntr ( tU16 u16CommandCntrParam )
{
   this->u16CommandCntr = u16CommandCntrParam;
}

inline tU16 tclSubscriberManager::u16GetCommandCntr ()
{
   return ( u16CommandCntr );
}

inline tVoid tclSubscriberManager::vSetNextElemToRead ( tU32 u32NextElemToReadParam )
{
   this->u32NextElemToRead = u32NextElemToReadParam;
}

inline tU32 tclSubscriberManager::u32GetNextElemToRead ()
{
   return ( u32NextElemToRead );
}

inline tVoid  tclSubscriberManager::vSetAllSen_NxtElmToRead(Vds_AllSenDataNextElemToSend Str)
{
   this->rAllSenDataNextElemToSend.u32NextGnssElementToRead = Str.u32NextGnssElementToRead;
   this->rAllSenDataNextElemToSend.u32NextGyroElementToRead = Str.u32NextGyroElementToRead;
   this->rAllSenDataNextElemToSend.u32NextOdoElementToRead = Str.u32NextOdoElementToRead;
   this->rAllSenDataNextElemToSend.u32NextAccElementToRead = Str.u32NextAccElementToRead;
   this->rAllSenDataNextElemToSend.u32NextAbsElementToRead = Str.u32NextAbsElementToRead;
}

inline tVoid  tclSubscriberManager::vGetAllSen_NxtElmToRead(Vds_AllSenDataNextElemToSend &Str)
{
   Str.u32NextGnssElementToRead = this->rAllSenDataNextElemToSend.u32NextGnssElementToRead;
   Str.u32NextGyroElementToRead = this->rAllSenDataNextElemToSend.u32NextGyroElementToRead;
   Str.u32NextOdoElementToRead = this->rAllSenDataNextElemToSend.u32NextOdoElementToRead;
   Str.u32NextAccElementToRead = this->rAllSenDataNextElemToSend.u32NextAccElementToRead;
   Str.u32NextAbsElementToRead = this->rAllSenDataNextElemToSend.u32NextAbsElementToRead;
}

inline tVoid tclSubscriberManager::vSetInitMessageSent ( tBool bInitMessageSentParam )
{
   this->bInitMessageSent = bInitMessageSentParam;
}

inline tBool tclSubscriberManager::bGetInitMessageSent ()
{
   return ( bInitMessageSent );
}

inline tVoid tclSubscriberManager::vSetMessageLockNeeded ( tBool bMessageLockNeededParam )
{
   this->bMessageLockNeeded = bMessageLockNeededParam;
}

inline tBool tclSubscriberManager::bGetMessageLockNeeded ()
{
   return ( bMessageLockNeeded );
}

inline tVoid tclSubscriberManager::vSetMsgSendInterval ( tU32 u32NewMsgSendIntervalParam )
{
   this->u32MsgSendInterval = u32NewMsgSendIntervalParam;
}

inline tU32 tclSubscriberManager::u32GetMsgSendInterval ()
{
   return ( u32MsgSendInterval );
}
#endif
