// *****************************************************************************
// * FILE:         tclSubscriberManager.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition: adds and deletes subscribers
// *               to a list, stores response-information (TargetAppId, ...)
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
// * 28.08.09 RBEI/ECF1 - sak9kor - Trace Output related modifications bave been done
// * 08.09.09 RBEI/ECF1 - sak9kor - Trace levels have been modified
// * 30.09.09 RBEI/ECF1 - sak9kor - Added doxygen headers for all classes/funcs
// *****************************************************************************

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#include "sensor_fi_if.h"

#define VDS_S_IMPORT_INTERFACE_MESSAGE_INTERFACES
#define VDS_S_IMPORT_INTERFACE_SUBSCRIBER_MANAGER
#define VDS_S_IMPORT_INTERFACE_SENSOR_DEFINES
#define VDS_S_IMPORT_INTERFACE_TRACE
#include "vds_internal_if.h"

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   Output trace message.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
void vds_tclTraceSynchListener::vSynch( void )
{
   vTraceMsg( VDS_C_TRACELEVEL_EVENT, "Synch" );
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Constructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
vds_tclSynchManager::vds_tclSynchManager() :
      iChildren( 0 ),
      u32LastSynchEventMs( 0 ),
      bLastSynchEventNormal( TRUE ),
      u32TimeOutFirstMs( 1150 ),
      u32TimeOutSecondMs( 1000 )
{
   bAddListener( new vds_tclTraceSynchListener );
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  Default Destructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
vds_tclSynchManager::~vds_tclSynchManager()
{
   for ( int i = 0; i < iChildren; ++i )
   {
      if ( arpoChildren[i] != OSAL_NULL )
      {
         delete arpoChildren[i];
      }
   }
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  Add a listener to the list of children.
//!
//! \return
//!   TRUE if successful; FALSE if poListener could not be added.
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
tBool vds_tclSynchManager::bAddListener
(
   //! (I) The listener to be added.
   vds_tclSynchListenerIf *poListener
)
{
   if ( iChildren  < iMaxChildren )
   {
      arpoChildren[iChildren++] = poListener;
      return TRUE;
   }
   else
   {
      delete poListener;
      return FALSE;
   }
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  Set time out period.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
void vds_tclSynchManager::vSetTimeOut
(
   //! (I) Number of ms until a synchronisation event is
   //! generated artificially if none occurs.
   tU32 u32FirstMs,
   //! (I) Number of ms between all subsequent time outs.
   tU32 u32SecondMs
)
{
   u32TimeOutFirstMs = u32FirstMs;
   u32TimeOutSecondMs = u32SecondMs;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  Check if a time out has occurred.
//!
//! \return
//!   TRUE if timeout has occured, else FALSE
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
tBool vds_tclSynchManager::bCheckTimeOut( void )
{
   tU32 u32CurrentTimeMs = OSAL_ClockGetElapsedTime( );
   tBool bTimeOut = FALSE;

   if ( bLastSynchEventNormal )
   {
      if ( u32CurrentTimeMs > u32LastSynchEventMs + u32TimeOutFirstMs )
      {
         bTimeOut = TRUE;
      }
   }
   else
   {
      if ( u32CurrentTimeMs > u32LastSynchEventMs + u32TimeOutSecondMs )
      {
         bTimeOut = TRUE;
      }
   }

   if ( bTimeOut )
   {
      vTraceMsg( VDS_C_TRACELEVEL_DATA, "Synchronisation time out" );
      vSynchInternal( FALSE );
      return TRUE;
   }

   return FALSE;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  Initiates and internal synch, and outputs traces.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
void vds_tclSynchManager::vSynchInternal
(
   //! (I) To set the last synch event
   tBool bNormal
)
{
   u32LastSynchEventMs = OSAL_ClockGetElapsedTime( );
   bLastSynchEventNormal = bNormal;

   for ( int i = 0; i < iChildren; ++i )
   {
      if ( arpoChildren[i] != OSAL_NULL )
      {
         arpoChildren[i]->vSynch( );
      }
   }
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  Notifies all registered listeners that a synchronisation
//!  event has occurred.
//!
//!  When a client wishes to synchronise with the registered clients
//!  (e.g. when a GPS position has been calculated and all
//!  accumulated data from the other sensors should be sent to the
//!  registered CCA clients), it should call this function. The
//!  vSynch member functions of all registered listeners are then
//!  called.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
void vds_tclSynchManager::vSynch( )
{
   vSynchInternal( TRUE );
}



vds_tclSynchManager vds_goSensorPropertySynchManager;

// Class tclSubscriberManager
OSAL_tSemHandle tclSubscriberManager::hSubscriberSemaphore = OSAL_C_INVALID_HANDLE;
tCString tclSubscriberManager::coszSubscriberSemName = "VDS_SUBSCRIBER_SEM";
tclSubscriberManager* tclSubscriberManager::poFirstSubscriber = OSAL_NULL;
tS32 tclSubscriberManager::s32InternalState = VDS_C_S32_SUBSCRIBER_UNINITIALIZED;

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  Default constructor
//!
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tclSubscriberManager::tclSubscriberManager() :
      u16RegisterId( 0 ),
      u16ServiceId( 0 ),
      u16FunctionId( 0 ),
      u16TargetAppId( 0 ),
      u16MessageSubId( 0 ),
      u16CommandCntr( 0 ),
      u8ACT( 0 ),
      poPrevSubscriber( OSAL_NULL ),
      poNextSubscriber( OSAL_NULL ),
      u32NextElemToRead( 0 ),
      u32MsgSendInterval( 0 ),
      bInitMessageSent( FALSE ),
      bMessageLockNeeded( FALSE )
{
    memset(&rAllSenDataNextElemToSend,0,sizeof(Vds_AllSenDataNextElemToSend));
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  Constructor
//!
//!  adds subscriber to double-linked list.  if first subscriber,
//!  store pointer in poFirstSubscriber
//!
//! \return
//!   None
//  HISTORY:
//       Date      |        Author           |       MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer     |   Initial Revision.
//     23.04.02    |  CM-DI/ESA1-Fischer     | sub-id and command-cntr added
//**************************************************************************
tclSubscriberManager::tclSubscriberManager
(
   //! (I) needed for the answer-message
   tU16 u16RegisterIdParam,
   //! (I) needed for the answer-message
   tU16 u16ServiceIdParam,
   //! (I) needed for the answer-message
   tU16 u16FunctionIdParam,
   //! (I) needed for the answer-message
   tU16 u16TargetAppIdParam,
   //! (I) needed for the answer-message
   tU16 u16MessageSubIdParam,
   //! (I) needed for the answer-message
   tU16 u16CommandCntrParam,
   //! (I) needed for the answer-message
   tU8 u8ACTParam
)  : u16RegisterId( 0 ),
      u16ServiceId( 0 ),
      u16FunctionId( 0 ),
      u16TargetAppId( 0 ),
      u16MessageSubId( 0 ),
      u16CommandCntr( 0 ),
      u8ACT( 0 ),
      poPrevSubscriber( OSAL_NULL ),
      poNextSubscriber( OSAL_NULL ),
      u32NextElemToRead( 0 ),
      u32MsgSendInterval( 0 ),
      bInitMessageSent( FALSE ),
      bMessageLockNeeded( FALSE )
{
   this->u16RegisterId     = u16RegisterIdParam;
   this->u16ServiceId      = u16ServiceIdParam;
   this->u16FunctionId     = u16FunctionIdParam;
   this->u16TargetAppId = u16TargetAppIdParam;
   this->u16MessageSubId   = u16MessageSubIdParam;
   this->u16CommandCntr = u16CommandCntrParam;
   this->u8ACT          = u8ACTParam;
   memset(&rAllSenDataNextElemToSend,0,sizeof(Vds_AllSenDataNextElemToSend));
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  Destructor
//!
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tclSubscriberManager::~tclSubscriberManager()
{
   // is object in list?
   if (     NULL != poPrevSubscriber
            || NULL != poNextSubscriber
            || poFirstSubscriber == this )
      s32DeleteSubscriber();
      
   poPrevSubscriber = OSAL_NULL;
   poNextSubscriber = OSAL_NULL;
   
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  initializes subscriber-list, creates semaphore
//!
//! \return
//!   ok    - VDS_E_NO_ERROR;
//!   error - VDS_E_SUBSCRIBER_ALREADY_INITIALIZED or
//!           VDS_E_SUBSCRIBER_SEM_ERROR
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tS32 tclSubscriberManager::s32Init ()
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( VDS_C_S32_SUBSCRIBER_UNINITIALIZED == s32InternalState )
   {
      // create semaphore
      if ( OSAL_ERROR != OSAL_s32SemaphoreCreate(
               coszSubscriberSemName,
               &hSubscriberSemaphore,
               1 ) )
      {
         if ( NULL != poFirstSubscriber )
         {
#ifdef VDS_S_TRACE_COMPONENT
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                       "Subscriber: s32Init: first subscriber exists, will be deleted" );
#endif
            poFirstSubscriber = NULL;
         }

         s32InternalState = VDS_C_S32_SUBSCRIBER_INITIALIZED;
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "Subscriber: sem-create error" );
         s32RetVal = VDS_E_SUBSCRIBER_SEM_ERROR;
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR, "Subscriber: already initialized" );
      s32RetVal = VDS_E_SUBSCRIBER_ALREADY_INITIALIZED;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!  deletes subscribers from list, deletes semaphore
//!
//! \return
//!   ok    - VDS_E_NO_ERROR;
//!   error - VDS_E_SUBSCRIBER_UNINITIALIZED or VDS_E_SUBSCRIBER_SEM_ERROR
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tS32 tclSubscriberManager::s32Close ()
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if (     VDS_C_S32_SUBSCRIBER_INITIALIZED == s32InternalState
            && OSAL_C_INVALID_HANDLE         != hSubscriberSemaphore )
   {
      // set state to uninitialized, so noone tries to lock semaphore
      s32InternalState = VDS_C_S32_SUBSCRIBER_UNINITIALIZED;

      // lock semaphore
      if ( OSAL_ERROR == OSAL_s32SemaphoreWait(
               hSubscriberSemaphore,
               OSAL_C_U32_INFINITE ) )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "Subscriber: sem-wait error" );
         s32RetVal = VDS_E_SUBSCRIBER_SEM_ERROR;
      }

      // - delete subscribers from linked list
      // get first subscriber
      tclSubscriberManager *poSubscriber = poFirstSubscriber;
      // helper for deleting list-members
      tclSubscriberManager *poSubscriberHelper = poSubscriber;
      // reset first-subscriber, else warning is thrown in
      // destructor when first object will be deleted
      poFirstSubscriber = NULL;

      // while subscribers exist
      while ( NULL != poSubscriber )
      {
         // get next subscriber
         poSubscriberHelper = poSubscriber->poNextSubscriber;
         // reset links
         poSubscriber->poPrevSubscriber = NULL;
         poSubscriber->poNextSubscriber = NULL;
         // delete subscriber-object
         delete poSubscriber;
         // move subscriber for next pass
         poSubscriber = poSubscriberHelper;
      }

      // unlock semaphore (doesn't matter, if lock above
      // failed, sem will be deleted)
      OSAL_s32SemaphorePost( hSubscriberSemaphore );
      // delete semaphore
      OSAL_s32SemaphoreClose( hSubscriberSemaphore );

      if ( OSAL_ERROR == OSAL_s32SemaphoreDelete( coszSubscriberSemName ) )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "Subscriber: sem-delete error" );
         s32RetVal = VDS_E_SUBSCRIBER_SEM_ERROR;
      }

      hSubscriberSemaphore = OSAL_C_INVALID_HANDLE;
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "Subscriber: s32Close: uninitialized" );
      s32RetVal = VDS_E_SUBSCRIBER_UNINITIALIZED;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    adds subscriber to linked list
//!
//! \return
//!   ok    - VDS_E_NO_ERROR;
//!   error - VDS_E_SUBSCRIBER_UNINITIALIZED
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tS32 tclSubscriberManager::s32AddSubscriber ()
{
   // does first subscriber exist?
   if ( NULL == poFirstSubscriber )
   {
      poPrevSubscriber = NULL;
      poNextSubscriber = NULL;
      poFirstSubscriber = this;
   }
   else
   {
      // add object in linked list
      poPrevSubscriber = NULL;
      poNextSubscriber = poFirstSubscriber;
      poNextSubscriber->poPrevSubscriber = this;
      poFirstSubscriber = this;
   }

   return VDS_E_NO_ERROR;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    deletes subscriber from linked list
//!
//! \return
//!   ok    - VDS_E_NO_ERROR;
//!   error - VDS_E_SUBSCRIBER_UNINITIALIZED
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tS32 tclSubscriberManager::s32DeleteSubscriber ()
{
   // does predecessor exist?
   if ( NULL != poPrevSubscriber )
      poPrevSubscriber->poNextSubscriber = poNextSubscriber;
   else if ( poFirstSubscriber == this )
      // this object is first subscriber in list, move pointer
      poFirstSubscriber = poNextSubscriber;

   // else object is not in the list
   // does successor exist?
   if ( NULL != poNextSubscriber )
      poNextSubscriber->poPrevSubscriber = poPrevSubscriber;

   // else object is last object or not in the list
   poPrevSubscriber = NULL;
   poNextSubscriber = NULL;
   return VDS_E_NO_ERROR;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    returns pointer to subscriber with given service-id/function-id.
//!
//! \return
//!   if poSubscriberManager is NULL, the first subscriber with
//!       u16ServiceId/u16FunctionId will be returned.
//!   if poSubscriberManager is valid, the first successor with
//!       u16ServiceId/u16FunctionId will be returned.
//!   if no (successor)-subscriber with u16ServiceId/u16FunctionId exists,
//!       NULL will be returned
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     21.03.02    |     CM-DI/ESA1-Fischer        |  Initial Revision.
//     27.06.03    |     R. Schedel, 3SOFT         |  Added service id.
//**************************************************************************
tclSubscriberManager* tclSubscriberManager::poGetSubscriberWithFId
(
   //! (I) return successor to this subscriber
   const tclSubscriberManager* poSubscriberManagerParam,
   //! (I) return successor with this service-id
   tU16 u16ServiceIdParam,
   //! (I) return successor with this function-id
   tU16 u16FunctionIdParam
)
{
   tclSubscriberManager *poSubscriber = ( NULL != poSubscriberManagerParam )
                                        ? poSubscriberManagerParam->poNextSubscriber
                                        : poFirstSubscriber;

   while (     poSubscriber != NULL
               && (
                  poSubscriber->u16ServiceId != u16ServiceIdParam ||
                  poSubscriber->u16FunctionId != u16FunctionIdParam ) )
      poSubscriber = poSubscriber->poNextSubscriber;

   return poSubscriber;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    returns pointer to subscriber with given RegisterId.
//!
//! \return
//!   if poSubscriberManager is NULL, the first subscriber with
//!       u16RegisterId will be returned.
//!   if poSubscriberManager is valid, the first successor with
//!       u16RegisterId will be returned.
//!   if no (successor)-subscriber with u16RegisterId exists,
//!       NULL will be returned
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     28.05.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tclSubscriberManager* tclSubscriberManager::poGetSubscriberWithRegId
(
   //!  (I)  return successor to this subscriber
   const tclSubscriberManager* poSubscriberManagerParam,
   //!  (I)  return successor with this RegisterId
   tU16 u16RegisterIdParam
)
{
   tclSubscriberManager *poSubscriber = ( NULL != poSubscriberManagerParam )
                                        ? poSubscriberManagerParam->poNextSubscriber
                                        : poFirstSubscriber;

   while (     poSubscriber != NULL
               && poSubscriber->u16RegisterId != u16RegisterIdParam )
      poSubscriber = poSubscriber->poNextSubscriber;

   return poSubscriber;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    sets all subscribers to 'no init-msg sent'
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     28.05.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tVoid tclSubscriberManager::vResetSubscribers ()
{
   if ( vds_goMasterLock.bEnter() )
   {
      tclSubscriberManager *poSubscriber = poFirstSubscriber;

      while ( poSubscriber != NULL )
      {
         poSubscriber->vSetInitMessageSent( FALSE );
         poSubscriber = poSubscriber->poNextSubscriber;
      }

      vds_goMasterLock.vLeave();
   }
}

