// *****************************************************************************
// * FILE:         tclOdometerThread.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition: inherits from tclSensorThread,
// *               thread-main-loop, reads data from odometer-device
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
//  * 28.08.09 RBEI/ECF1 - sak9kor - Trace Output related modifications bave been done
//  * 08.09.09 RBEI/ECF1 - sak9kor - Trace levels have been modified
//  * 30.09.09 RBEI/ECF1 - sak9kor - Added doxygen headers for all classes/funcs
// * 29.08.11 RBEI CM-AI/PJ-CF31 - kas4hi - Solution for GMNGA-14867 is provided
// * 06.01.17 RBEI/ECF12 - amg6kor - Port to Gen4
// *****************************************************************************
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#include "sensor_fi_if.h"

#define VDS_S_IMPORT_INTERFACE_SUBSCRIBER_MANAGER
#define VDS_S_IMPORT_INTERFACE_SENSOR_THREADS
#define VDS_S_IMPORT_INTERFACE_SENSOR_DEFINES
#define VDS_S_IMPORT_INTERFACE_TRACE
#define VDS_S_IMPORT_INTERFACE_PERSISTENT_DATA
#define VDS_S_IMPORT_INTERFACE_MESSAGE_INTERFACES
#include "vds_internal_if.h"

#if OSAL_OS==OSAL_NUCLEUS || OSAL_OS==OSAL_TENGINE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#endif

#define CCA_S_IMPORT_INTERFACE_GENERIC // CCA_C_U16_APP_SENSOR
#include "cca_if.h"
#define SCD_S_IMPORT_INTERFACE_GENERIC // scd_s32GetThreadConfiguration
#include "scd_if.h"
#include "regkeys.h"    // THREADNAME_C_STRING_SENSOR_VDS_ODO
#include "tclMsgOdometerIf.h"

#define MIN_CYCLE_TIME (10 * 1000000) // 10ms = 100Hz

// Class tclOdometerThread
// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Constructor
//!
//! \return
//!   None
//!
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
tclOdometerThread::tclOdometerThread()
      : pfs32AddOdometerList( OSAL_NULL ),
      parOdometerData( OSAL_NULL ),
      parOdometerStorage( OSAL_NULL ),
      u32NumOfOdometerData( 0 ),
      poMsgOdometerIf( OSAL_NULL ),
      u32OdometerCycleTimeInNS( 50000000 )
{
   /*coszSensorSemName = "VDS_ODOMETER_THREAD_SEM";*/
   coszSensorSemName = NULL;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Destructor
//!
//! \return
//!   None
//!
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************
tclOdometerThread::~tclOdometerThread()
{
   tclOdometerThread::vThreadClose();
   
   parOdometerData = OSAL_NULL;
   parOdometerStorage = OSAL_NULL;
   poMsgOdometerIf = OSAL_NULL;
}



// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   inherited from class tclSensorThread.
//!   open odometer-device, create semaphore, create
//!   odometer-thread, set function-pointer, set internal state
//!
//! \return
//!   ok   : VDS_E_NO_ERROR (thread and
//!          semaphore created successfully)
//!   error: VDS_E_THREAD_IOOPEN_ERROR, VDS_E_THREAD_SEM_ERROR,
//!          VDS_E_THREAD_CREATE_ERROR, VDS_E_INVALID_PARAMETER,
//!          VDS_E_THREAD_ALREADY_INITIALIZED
//!
//  HISTORY:
//      Date       |            Author             |    MODIFICATION
// -------------------------------------------------------------------------
//    21.03.02     |       CM-DI/ESA1-Fischer      |  Initial Revision.
//**************************************************************************
tS32 tclOdometerThread::s32ThreadInit
(
   //!  (I) Odometer Interface parameter
   tclMsgOdometerIf  *poMsgOdometerIfParam,
   //!  (I)  function pointer to store sensor-data. this function
   //!  has to be called, when sensor-data is read.
   tS32( *pfs32AddOdometerListParam )( vds_trOdometerData*, tU32 )
   //!  (I)  function pointer to recalibrate the Vehicle parameters to
   //!   the newer values
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( VDS_C_S32_THREAD_UNINITIALIZED == s32GetInternalState() )
   {
      this->poMsgOdometerIf = poMsgOdometerIfParam;

      // *** set function-pointer ***
      this->pfs32AddOdometerList = pfs32AddOdometerListParam;

      if ( NULL == pfs32AddOdometerListParam )
      {
         s32RetVal = VDS_E_INVALID_PARAMETER;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL, "OdoThread:Init:Invalid param");
      }

      if ( OSAL_ERROR == ( SensorFd = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ODOMETER, OSAL_EN_READONLY ) ) )
      {
         s32RetVal = VDS_E_ODOMTHREAD_DEVICE_NOT_FOUND;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL, "OdoThread:Init:Invalid param");
      }

      // configure odometer-device
      if ( VDS_E_NO_ERROR == s32RetVal )
      {
         tS32 s32OdometerCycleTime = 0;
         tS32 s32OdometerCycleRetVal =  OSAL_s32IOControl( SensorFd,
                                                           OSAL_C_S32_IOCTRL_ODOMETER_GETCYCLETIME,
                                                           (tLong)&s32OdometerCycleTime );
      
         //check return value of OSAL_s32IOControl
         //and cycle time value, must be != 0, because of division with it in tclMsgOdometerIf
         if ( ( OSAL_ERROR != s32OdometerCycleRetVal )&&
              ( s32OdometerCycleTime >= MIN_CYCLE_TIME ) ) //lint !e774 !e845 PQM_authorized_multi_527  PQM_authorized_multi_549*/
         {
            u32OdometerCycleTimeInNS = (tU32)s32OdometerCycleTime;
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
            "OdoThread: OSAL_C_S32_IOCTRL_ODOMETER_GETCYCLETIME SUCCESS s32OdometerCycleRetVal - %d s32OdometerCycleTime - %d"
            ,s32OdometerCycleRetVal ,s32OdometerCycleTime);
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                       "OdoThread: Invalid odometer cycletime %d , use init default  value %d",
                       s32OdometerCycleTime, u32OdometerCycleTimeInNS );
         }
      }

      // *** create semaphore ***
      /*if ( OSAL_ERROR == OSAL_s32SemaphoreCreate( coszSensorSemName, &hSensorSemaphore, 1 ) )
         s32RetVal = VDS_E_ODOMTHREAD_SEM_ERROR;*/

      // *** create odometer-storage ***
      // storage for reading from odometer-device
      u32NumOfOdometerData = VDS_C_U16_ODOMTHREAD_BLOCKREADSIZE;
      parOdometerData = new OSAL_trIOCtrlOdometerData[ u32NumOfOdometerData ];
      // storage for sending data to odom-if
      parOdometerStorage = new vds_trOdometerData[ u32NumOfOdometerData ];

      // if error occured...
      if( ( NULL == parOdometerData )|| ( NULL == parOdometerStorage ) )
      {
         s32RetVal = VDS_E_ODOMTHREAD_ALLOC_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL, "OdoThread::Init:Alloc error.");
      }

      // *** create thread-attribute ***
      OSAL_trThreadAttribute rThreadAttribute = {0};
      rThreadAttribute.szName       = const_cast<tString>(THREADNAME_C_STRING_SENSOR_VDS_ODO);  // Name
      rThreadAttribute.pfEntry      = ( OSAL_tpfThreadEntry ) tclOdometerThread::vThreadMainWrapper;
      rThreadAttribute.pvArg        = ( tPVoid ) this;      // Argument der Startroutine

      // get thread-configuration
      if ( OSAL_ERROR == scd_s32GetThreadConfiguration( CCA_C_U16_APP_SENSOR,
                                                        THREADNAME_C_STRING_SENSOR_VDS_ODO,
                                                        &rThreadAttribute.u32Priority,
                                                        &rThreadAttribute.s32StackSize ) )
      {
         vTraceMsg( VDS_C_TRACELEVEL_FATAL, "OdoThread:Init:Can't get thread configuration");
         s32RetVal = VDS_E_ODOMTHREAD_THREADCREATE_FAILED;
      }

      // *** create sensor-thread ***
      if ( s32RetVal == VDS_E_NO_ERROR )
      {
         if ( OSAL_ERROR == ( SensorThreadId = OSAL_ThreadCreate( &rThreadAttribute ) ) )
         {
            vTraceMsg( VDS_C_TRACELEVEL_FATAL, "OdoThread:Init:Error Creating Odom Thread");
            s32RetVal = VDS_E_ODOMTHREAD_THREADCREATE_FAILED;
         }
      }

      // does error occured?
      if ( VDS_E_NO_ERROR == s32RetVal )
      {
         vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
      }
      else
      {
         vThreadClose();
      }
   }
   else
   {
      s32RetVal = VDS_E_ODOMTHREAD_NOT_UNINITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_FATAL, "OdoThread:Init:Already Initialized");
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   inherited from class tclSensorThread.
//!   when thread is started, set state RUNNING.
//!   receive odometer-data from odometer-device
//!   when internal state is set to STOP, leave loop and
//!   set state INITIALIZED.
//!
//! \return
//!   None
//!
//  HISTORY:
//      Date       |            Author             |    MODIFICATION
// -------------------------------------------------------------------------
//    21.03.02     |       CM-DI/ESA1-Fischer      |  Initial Revision.
//**************************************************************************
tVoid tclOdometerThread::vThreadMain
(
   //!  (I) not used, argument declared in s32ThreadInit
   tPVoid
)
{
   tU32 u32TempNumOfOdometerData = 0;
   OSAL_trIOCtrlOdometerData *parTempOdometerData = NULL;

   if( ( s32GetInternalState() == VDS_C_S32_THREAD_INITIALIZED )&&
       ( OSAL_ERROR != SensorFd )&&
       ( 0 < u32NumOfOdometerData )&&
       ( NULL != parOdometerData ) )
   {
      //Thread is activated
      vSetInternalState( VDS_C_S32_THREAD_RUNNING );

      //Read available data from from odometer-device
      if ( OSAL_ERROR == OSAL_s32IOControl( SensorFd,
                                            OSAL_C_S32_IOCTRL_ODOMETER_GETCNT,
                                            (tLong) &u32TempNumOfOdometerData ) )
      {
         //Read u32NumOfOdometerData data (=VDS_C_U16_ODOMTHREAD_BLOCKREADSIZE)
         u32TempNumOfOdometerData = u32NumOfOdometerData;
      }

      // at the first time, there would be more odometer-data than u32NumOfOdometerData
      if ( u32TempNumOfOdometerData > u32NumOfOdometerData )
      {
         parTempOdometerData = new OSAL_trIOCtrlOdometerData[ u32TempNumOfOdometerData ];
         if ( NULL == parTempOdometerData )
         {
            // creation of temp buffer failed, so use parOdometerData-buffer
            parTempOdometerData = parOdometerData;
            u32TempNumOfOdometerData = u32NumOfOdometerData;
         }
      }
      else
      {
         // buffer is big enough, so use it
         parTempOdometerData = parOdometerData;
      }

      // read and store odometer data
      s32ReadOdometerData( parTempOdometerData, u32TempNumOfOdometerData );

      // delete temp-buffer
      if ( parTempOdometerData != parOdometerData )
      {
         delete [] parTempOdometerData;
         parTempOdometerData = NULL;
      }

      do
      {
         //Consistence check
         if ( ( OSAL_ERROR != SensorFd ) && ( NULL != parOdometerData ) )
         {
            //Read and store odometer-data (blocking device)
            s32ReadOdometerData( parOdometerData, u32NumOfOdometerData );
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR, "OdoThread: IO or DataMemory invalid" );
            vSetInternalState( VDS_C_S32_THREAD_ERROR );
         }

      } while ( VDS_C_S32_THREAD_RUNNING == s32GetInternalState() );
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "OdoThread:Exit main Loop!");

      // set new state
      if ( s32GetInternalState() == VDS_C_S32_THREAD_STOP )
      {
         vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
      }
      else
      {
         // close handles, ...
         vThreadClose();
         vSetInternalState( VDS_C_S32_THREAD_UNINITIALIZED );
      }
   }

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTempOdometerData);

   OSAL_vThreadExit();
   // else no error-handling needed, state is uninitialized
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   inherited from class tclSensorThread.
//!   closes handles, deletes semaphores
//!
//! \return
//!   None
//!
//  HISTORY:
//      Date       |            Author             |    MODIFICATION
// -------------------------------------------------------------------------
//    09.04.02     |       CM-DI/ESA1-Fischer      |  Initial Revision.
//**************************************************************************
tVoid tclOdometerThread::vThreadClose ()
{

   // if odometer-data exists, delete
   if ( NULL != parOdometerData )
   {
      delete [] parOdometerData;
      parOdometerData = NULL;
   }

   // if odometer-storage exists, delete
   if ( NULL != parOdometerStorage )
   {
      delete [] parOdometerStorage;
      parOdometerStorage = NULL;
   }

   u32NumOfOdometerData = 0;
   // semaphore and device will be deleted in tclSensorThread
   // reset function-pointer
   pfs32AddOdometerList = NULL;
   tclSensorThread::vThreadClose();
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   reads odometer-data from odometer-device
//!
//! \return
//!   [0..n]: odometer-data-elements read
//!   error : VDS_E_INVALID_PARAMETER,
//!           VDS_E_ODOMTHREAD_NOT_INITIALIZED,
//!           VDS_E_ODOMTHREAD_IOREAD_ERROR,
//!           VDS_E_ODOMTHREAD_STORAGE_ERROR
//!
//  HISTORY:
//      Date       |            Author             |    MODIFICATION
// -------------------------------------------------------------------------
//     21.03.02    |       CM-DI/ESA1-Fischer      |   Initial Revision.
//**************************************************************************
tS32 tclOdometerThread::s32ReadOdometerData
(
   //! (O) Buffer for received odometer-data
   OSAL_trIOCtrlOdometerData *parIOCtrlOdometerData,
   //! (I) max. number of odometer-data to be stored in buffer
   tU32 u32NumOfElem
)
{

   tS32 s32NumOfElemRead = 0;

   //Parameter check.
   if ( ( NULL != parIOCtrlOdometerData )&&
        ( 0 <  u32NumOfElem )&&
        ( 0x80000000 >  u32NumOfElem ) )
   {
      //Consistence check
      if ( OSAL_ERROR != SensorFd )
      {
         //Read odometer data.
         tS32 s32OdometerDataRead = 
            OSAL_s32IORead( SensorFd,
                            (tPS8)parIOCtrlOdometerData,
                            u32NumOfElem * (tU32) sizeof(OSAL_trIOCtrlOdometerData) );
         
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "OdoThread:Odometer read -> %d",s32OdometerDataRead );           

         s32NumOfElemRead = 
            s32OdometerDataRead 
            / 
            (tS32)sizeof(OSAL_trIOCtrlOdometerData);

         if( s32OdometerDataRead > 0 ) 
         {
            for( int i = 0 ; i < s32NumOfElemRead ; ++i )
            {
               vTraceMsg( VDS_C_TRACELEVEL_DATA,
                          "OdoThread: ts=%d, ct=%d, direction=%d \n"
                          "error counter=%d, odo status  = %d \n",
                          parIOCtrlOdometerData[i].u32TimeStamp,
                          parIOCtrlOdometerData[i].u32WheelCounter,
                          parIOCtrlOdometerData[i].enDirection,
                          parIOCtrlOdometerData[i].u16ErrorCounter,
                          parIOCtrlOdometerData[i].enOdometerStatus);
            }
         }
          
         if ((s32NumOfElemRead > 0) 
             &&
             ((tU32)s32NumOfElemRead <= u32NumOfElem))
         {
            if ( VDS_E_NO_ERROR 
                 !=
                 s32StoreOdometerData( parIOCtrlOdometerData,
                                       (tU32)s32NumOfElemRead ))
            {
               s32NumOfElemRead = VDS_E_ODOMTHREAD_STORAGE_ERROR;
            }
         }
         else
         {
            if( VDS_C_U32_SENSOR_ABS != 
                 ( tclSystemInformation::rInfo.rSystem.u32SensorTypes 
                   & VDS_C_U32_SENSOR_ABS ) )
            {
               vTraceMsg( VDS_C_TRACELEVEL_ERROR, "OdomThread:No ABS in project" );
            }
            else
            {
               vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, 
                          "OdomThread: IORead failed" );
            }

            OSAL_s32ThreadWait( 200 );
            s32NumOfElemRead = VDS_E_ODOMTHREAD_IOREAD_ERROR;            
         }
      }
      else
      {
         s32NumOfElemRead = VDS_E_ODOMTHREAD_NOT_INITIALIZED;
      }
   }
   else
   {
      s32NumOfElemRead = VDS_E_INVALID_PARAMETER;
   }

   return s32NumOfElemRead;

}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   sends received odometer-data to tclMsgOdometerIf
//!
//! \return
//!   ok    : VDS_E_NO_ERROR
//!   error : VDS_E_INVALID_PARAMETER,
//!           VDS_E_ODOMTHREAD_NOT_INITIALIZED,
//!           VDS_E_ODOMTHREAD_ALLOC_ERROR,
//!           VDS_E_ODOMTHREAD_STORAGE_ERROR
//!
//  HISTORY:
//      Date       |            Author             |    MODIFICATION
// -------------------------------------------------------------------------
//     21.03.02    |       CM-DI/ESA1-Fischer      |   Initial Revision.
//**************************************************************************
tS32 tclOdometerThread::s32StoreOdometerData
(
   //!  (I)  buffer with received odometer-data
   const OSAL_trIOCtrlOdometerData *parIOCtrlOdometerData,
   //!  (I)  number of odometer-data in buffer
   tU32 u32NumOfElem
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
   vds_trOdometerData *parTempOdometerStorage = NULL;
   tU32 u32OdometerIndex = 0;

   //Parameter check.
   if ( ( NULL != parIOCtrlOdometerData )&& ( u32NumOfElem   > 0 ) )
   {
      //Consistence check
      if ( ( NULL != parOdometerStorage )&& 
           ( NULL != pfs32AddOdometerList ) )
      {
         //if too much data, generate temp storage (expected for the first time)
         if ( u32NumOfElem > u32NumOfOdometerData )
         {
            parTempOdometerStorage = new vds_trOdometerData[ u32NumOfElem ];

            if ( NULL == parTempOdometerStorage )
            {
               vTraceMsg( VDS_C_TRACELEVEL_ERROR, 
                          "OdomThread:s32StoreOdometerData:alloc error" );

               s32RetVal = VDS_E_ODOMTHREAD_ALLOC_ERROR;
            }
         }
         else
         {
            parTempOdometerStorage = parOdometerStorage;
         }

         if ( NULL != parTempOdometerStorage )
         {
            // copy odometer-device-format to odometer-vdsensor-format
            for ( u32OdometerIndex = 0; 
                  u32OdometerIndex < u32NumOfElem; 
                  u32OdometerIndex++ )
            {
               parTempOdometerStorage[ u32OdometerIndex ].u32Timestamp = 
                  parIOCtrlOdometerData[ u32OdometerIndex ].u32TimeStamp;

               parTempOdometerStorage[ u32OdometerIndex ].u16OdometerCount  = 
                  (tU16)(parIOCtrlOdometerData[ u32OdometerIndex ].u32WheelCounter);
            
               // copy direction
               switch( parIOCtrlOdometerData[ u32OdometerIndex ].enDirection )
               {
                  case OSAL_EN_RFS_FORWARD:
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerCountDirection = 
                        VDS_C_U8_ODOMCOUNT_DIR_FORWARD;
                     break;
                  case OSAL_EN_RFS_REVERSE:
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerCountDirection = 
                        VDS_C_U8_ODOMCOUNT_DIR_REVERSE;
                     break;
                  case OSAL_EN_RFS_UNKNOWN:
                     // fall through
                  default:
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerCountDirection = 
                        VDS_C_U8_ODOMCOUNT_DIR_UNKNOWN;
                     break;
               }
               
               // copy status
               switch( parIOCtrlOdometerData[ u32OdometerIndex ].enOdometerStatus )
               {
                  case ODOMSTATE_CONNECTED_NORMAL :
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                        VDS_C_U8_ODOMSTATE_CONNECTED_NORMAL;
                     break;
                  case ODOMSTATE_CONNECTED_NO_CALIBRATION :
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                        VDS_C_U8_ODOMSTATE_NO_CALIBRATION;
                     break;
                  case ODOMSTATE_CONNECTED_DATA_INVALID :
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                        VDS_C_U8_ODOMSTATE_CONNECTED_DATA_INVALID;
                     break;
                  case ODOMSTATE_CONNECTED_INTERNALERROR :
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                        VDS_C_U8_ODOMSTATE_CONNECTED_INTERNALERROR;
                     break;
                  case ODOMSTATE_CONNECTED_NOINFO :
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                        VDS_C_U8_ODOMSTATE_CONNECTED_NOINFO;
                     break;
                  case ODOMSTATE_CONNECTED_NOMOVEMENT :
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                        VDS_C_U8_ODOMSTATE_CONNECTED_NOMOVEMENT;
                     break;
                  case ODOMSTATE_CONNECTED_ERROR :
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                        VDS_C_U8_ODOMSTATE_CONNECTED_ERROR;
                     break;
                  case ODOMSTATE_DISCONNECTED :
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                        VDS_C_U8_ODOMSTATE_DISCONNECTED;
                     break;
                  case ODOMSTATE_UNKNOWN :
                  default:
                     parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                        VDS_C_U8_ODOMSTATE_UNKNOWN;
               } 

               // check consistnecy of status and error counter
               if ( ( ODOMSTATE_CONNECTED_NORMAL
                      ==
                      parIOCtrlOdometerData[ u32OdometerIndex ].enOdometerStatus )
                    &&
                    ( 0
                      !=
                      parIOCtrlOdometerData[ u32OdometerIndex ].u16ErrorCounter))
               {
                  // this case should never occur since an error counter != 0
                  // means there is an error. If that hasn't been detected, we
                  // set the status here to unknown since it is definitly not
                  // normal.
                  parTempOdometerStorage[ u32OdometerIndex ].u8OdometerStatus =
                     VDS_C_U8_ODOMSTATE_UNKNOWN;
               }
            }

            // send odometer-data

            try
            {
               if ( VDS_E_NO_ERROR
                    !=
                    pfs32AddOdometerList( parTempOdometerStorage, u32NumOfElem ))
               {
                  s32RetVal = VDS_E_ODOMTHREAD_STORAGE_ERROR;
               }
            }
            catch(std::bad_alloc& ba)
            {
               vTraceMsg( VDS_C_TRACELEVEL_ERROR,"exception bad_alloc caught: %s",ba.what());
            }

            // delete temp buffer
            if ( parTempOdometerStorage != parOdometerStorage )
            {
               delete [] parTempOdometerStorage;
               parTempOdometerStorage = NULL;
            }
         }
      }
      else
      {
         s32RetVal = VDS_E_ODOMTHREAD_NOT_INITIALIZED;
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTempOdometerStorage);

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   static wrapper for main-loop of instance.
//!
//! \return
//!   None
//!
//  HISTORY:
//      Date       |            Author             |    MODIFICATION
// -------------------------------------------------------------------------
//     16.04.02    |       CM-DI/ESA1-Fischer      |   Initial Revision.
//**************************************************************************
tVoid tclOdometerThread::vThreadMainWrapper
(
   //!  (I) pointer to instance
   tPVoid pvThreadArg
)
{
   //Parameter check
   if ( OSAL_NULL != pvThreadArg )
   {
      tclOdometerThread *poOdometerThread = static_cast<tclOdometerThread *>( pvThreadArg );
      //Can not be NULL.
      poOdometerThread->vThreadMain( NULL );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdoThread:Thread Argument NULL" );
   }
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   returns gyro-sensor-cycle-time.
//!
//! \return
//!   gyro-sensor-cycle-time
//!
//  HISTORY:
//      Date       |            Author             |    MODIFICATION
// -------------------------------------------------------------------------
//     21.04.06    |        Grasser                |   Initial Revision.
//**************************************************************************
tU32 tclOdometerThread::u32GetOdometerCycleTime()
{
   return u32OdometerCycleTimeInNS;
}



