// *****************************************************************************
// * FILE:         tclMsgOdometerIf.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition: handles access to odometer-data,
// *               handles subscriber-messages
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
// * 19.06.09 - sak9kor - APIs "s32GetDistcalMessage","s32GetOdomMessage", and
// *                      "s32GetEmptyOdomMessage" have been modified to
// *                       to adopt the XML generated FI
// * 29.06.09 - sak9kor - Replacing the old nomenclature of the
// *                      objects (As per review comment from Mr.Peter Koenig)
// * 13.07.09 - sak9kor - Time stamp related traces have been removed
// * 27.07.09 - sak9kor - APIs "s32GetDriveDirectionMessage" and "s32GetGalaSpeedMessage"
// *                      have been modified to adapt the XML generated FI
// * 10.08.09 - sak9kor - Modifications have been done to avoid the usage
// *                      aditional memory pool, and new parameter poSubscriber has been added
// *                      in to 's32GetDistcalMessage" and "s32GetOdomMessage" functions
// * 12.08.09 - sak9kor - Modifications have been done to avoid the usage additional memory pool,
// *                      and new parameter poSubscriber has been added in to
// *                      's32GetDriveDirectionMessage' and 's32GetGalaSpeedMessage'
// * 28.08.09 RBEI/ECF1 - sak9kor - Trace Output related modifications bave been done
// * 08.09.09 RBEI/ECF1 - sak9kor - Trace levels have been modified
// * 30.09.09 RBEI/ECF1 - sak9kor - Added doxygen headers for all classes/funcs
// *****************************************************************************

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#if OSAL_OS==OSAL_TENGINE   
  #define SYSTEM_S_IMPORT_INTERFACE_EOLCALIB_DEF
  #include "system_pif.h"                                      
#endif //OSAL_OS==OSAL_NUCLEUS

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"

#define CCA_S_IMPORT_INTERFACE_GENERIC
#include "cca_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_ERRORCODES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_FUNCTIONIDS
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_SERVICEINFO
#include "sensor_fi_if.h"
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "fi_msgfw_if.h"
#define SENSOR_S_IMPORT_INTERFACE_SERVER
#include "vds2_if.h"

#define VDS_S_IMPORT_INTERFACE_SUBSCRIBER_MANAGER
#define VDS_S_IMPORT_INTERFACE_MESSAGE_INTERFACES
#define VDS_S_IMPORT_INTERFACE_PERSISTENT_DATA
#define VDS_S_IMPORT_INTERFACE_TRACE
#include "vds_internal_if.h"

#define VDSENSOR_MSG_ODOMETER_OK    (tU8) 0
#define VDSENSOR_MSG_ODOMETER_ERROR (tU8) 1

extern void vGetTimeStamp(tPU32 pu32TimeStamp);

/*!
 * \brief Synchronisation listener for the odometer interface.
 *
 * This listener triggers the odometer message interface object when a
 * synchronisation event occurs.
 */
class vds_tclOdomSynchListener : public vds_tclSynchListenerIf
{
   private:
      tclMsgOdometerIf *poOdomIf;
   public:

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Default Constructor
      //!
      //! \return
      //!   Current Speed
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      vds_tclOdomSynchListener
      (
         //!  (I)  pointer to the odometer message interface object.
         tclMsgOdometerIf *poOdomIfArg
      )
      {
         poOdomIf = poOdomIfArg;
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Notify interface object.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual void vSynch( void )
      {
         vTraceMsg( VDS_C_TRACELEVEL_EVENT, "Synch (odometer)" );

         if ( poOdomIf != OSAL_NULL )
         {
            poOdomIf->vSynch( );
         }
      }
};

// Class tclMsgOdometerIf 

tclMsgOdometerIf *tclMsgOdometerIf::poThisMsgOdometerIf;
// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    Default Constructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tclMsgOdometerIf::tclMsgOdometerIf()
      : oCS( "VDS_ODOMETER_IF_SEM" ),
        poOdometerThread( OSAL_NULL ),
        s32InternalState( VDS_C_S32_STATE_UNINITIALIZED ),
        u8DriveDirection( VDS_C_U8_ODOMCOUNT_DIR_UNKNOWN )
{
   poThisMsgOdometerIf = this;
   pfs32SendSubscriberMessage = NULL;
   vds_goSensorPropertySynchManager.bAddListener( 
      new vds_tclOdomSynchListener( this ));
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    Default Destructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tclMsgOdometerIf::~tclMsgOdometerIf()
{
   oRingBuffer.vDeallocate();
   oCS.bShutdown();

   // delete odometer-thread
   if ( NULL != poOdometerThread )
   {
      poOdometerThread->s32ThreadStop();
      delete poOdometerThread;
      poOdometerThread = NULL;
   }

   poThisMsgOdometerIf = NULL;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    initializes odometer-interface.
//!    create and initialize instance of odometer-thread, create
//!    semaphore, create ringbuffer for odometer-data,
//!    initialize member-attributes
//!
//! \return
//!   ok   : VDS_E_NO_ERROR
//!   error: VDS_E_ODOMIF_ALLOC_ERROR,
//!          VDS_E_INVALID_PARAMETER,
//!          VDS_E_ODOMIF_THREADINIT_ERROR,
//!          VDS_E_ODOMIF_SEM_ERROR
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//     17.04.02    |  CM-DI/ESA1-Fischer           |   parameter changed
//**************************************************************************
tS32 tclMsgOdometerIf::s32Init
(
   //!(I) pointer to function, which sends given answer-message
   //!    with given parameters to subscribers
   tS32( *pfs32SendSubscriberMessageParam )( amt_tclServiceData *, tclSubscriberManager * ))
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
   tU32 u32OdometerCycleTime = 0;

   //Check for valid parameter.
   if ( NULL != pfs32SendSubscriberMessageParam )
   {
      // save ail-sensor-interface
      this->pfs32SendSubscriberMessage = pfs32SendSubscriberMessageParam;

      // create odometer-thread
      poOdometerThread = new tclOdometerThread();
      if ( NULL != poOdometerThread )
      {
         if ( VDS_E_NO_ERROR 
              == 
              poOdometerThread->s32ThreadInit( poThisMsgOdometerIf,
                                               tclMsgOdometerIf::s32AddDataWrapper ) )
         {
            //Create semaphore.
            if ( oCS.bInit() )
            {
               tU32 u32RingBufferSize;

               u32OdometerCycleTime = poOdometerThread->u32GetOdometerCycleTime();
               if(u32OdometerCycleTime != 0)
               {
               // create odometer-storage (seconds*num_of_sensordata_per_second)
               u32RingBufferSize = (tU32)( VDS_C_U32_BUFFERSIZE_SECONDS *
                                         (1000000000U /u32OdometerCycleTime ) );

               vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                          "OdomIf:s32Init:Odometer u32RingBufferSize = %d ", u32RingBufferSize  );
               }
               else
               {
                  vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                             "OdomIf:s32Init:VDS_E_ALLOC_ERROR :Invalid Odo cycle time .");
                  return VDS_E_ALLOC_ERROR;
               }

               oRingBuffer.vAllocate( u32RingBufferSize );
               if ( !oRingBuffer.bIsValid() )
               {
                  oCS.bShutdown();
                  s32RetVal = VDS_E_ODOMIF_ALLOC_ERROR;
                  vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                             "OdomIf:s32Init:Unable to create Odometer ring buffer.");
               }
            }
            else
            {
               s32RetVal = VDS_E_ODOMIF_SEM_ERROR;
               vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                          "OdomIf:s32Init:Unable to create Odometer semaphore.");
            }

            if ( VDS_E_NO_ERROR != s32RetVal )
            {
               poOdometerThread->vThreadClose();
            }
         }
         else
         {
            s32RetVal = VDS_E_ODOMIF_THREADINIT_ERROR;
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                       "OdomIf:s32Init:Unable initialise Odometer Thread.");
         }

         if ( VDS_E_NO_ERROR != s32RetVal )
         {
            delete poOdometerThread;
            poOdometerThread = NULL;
         }
      }
      else
      {
         s32RetVal = VDS_E_ODOMIF_ALLOC_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "OdomIf:s32Init:Unable to create tclOdometerThread instance");
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,"OdomIf:s32Init:Invalid parameter" );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    start odometer-thread
//!
//! \return
//!   ok   : VDS_E_NO_ERROR
//!   error: VDS_E_ODOMIF_THREADSTART_ERROR,
//!          VDS_E_ODOMIF_NOT_INITIALIZED
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tS32 tclMsgOdometerIf::s32StartThread()
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Start Odometer Thread.
   if ( NULL != poOdometerThread )
   {
      if( VDS_C_S32_THREAD_RUNNING != poOdometerThread->s32ThreadStart() )
      {
         s32RetVal = VDS_E_ODOMIF_THREADSTART_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdomIf:Couldn't start OdomThread");
      }
   }
   else
   {
      s32RetVal = VDS_E_ODOMIF_NOT_INITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdomIf:poOdometerThread NOT initialized" );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    stop odometer-thread.
//! \return
//!   ok   : VDS_E_NO_ERROR
//!   error: VDS_E_ODOMIF_THREADSTOP_ERROR,
//!          VDS_E_ODOMIF_NOT_INITIALIZED
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tS32 tclMsgOdometerIf::s32StopThread()
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Stop Odometer Thread.
   if ( NULL != poOdometerThread )
   {
      if ( VDS_C_S32_THREAD_INITIALIZED != poOdometerThread->s32ThreadStop() )
      {
         s32RetVal = VDS_E_ODOMIF_THREADSTOP_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdomIf:Couldn't stop OdomThread" );
      }
   }
   else
   {
      s32RetVal = VDS_E_ODOMIF_NOT_INITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdomIf:poOdometerThread NOT initialized" );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    delete odometer-thread
//!
//! \return
//!   ok   : VDS_E_NO_ERROR
//!   error: VDS_E_ODOMIF_THREADSTOP_ERROR,
//!          VDS_E_ODOMIF_NOT_INITIALIZED
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tS32 tclMsgOdometerIf::s32DeleteThread()
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Delete Odometer Thread.
   if( NULL != poOdometerThread )
   {
      if( VDS_C_S32_THREAD_INITIALIZED != poOdometerThread->s32ThreadStop() )
      {
         s32RetVal = VDS_E_ODOMIF_THREADSTOP_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdomIf:Couldn't stop OdomThread");
      }

      delete poOdometerThread;
      poOdometerThread = NULL;
   }
   else
   {
      s32RetVal = VDS_E_ODOMIF_NOT_INITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdomIf:poOdometerThread NOT initialized" );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//  DESCRIPTION:
//
//! \brief
//!    set internal state (normal or pending)
//! \param
//!   tS32 s32NewState:new state ( VDS_C_S32_ODOMIF_NORMAL or
//!                                VDS_C_S32_ODOMIF_PENDING )
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tVoid tclMsgOdometerIf::vSetState
(
   //!(I)  new state
   tS32 s32NewState
)
{
   if ( ( VDS_C_S32_STATE_NORMAL == s32NewState ) ||
        ( VDS_C_S32_STATE_NORMAL_DIAG == s32NewState )
      )
   {
      if( VDS_C_S32_STATE_PENDING == s32InternalState )
      {
         s32CheckForOdomSubscriber( NULL,0 );
         /* s32CheckForDistcalSubscriber(); */
      }

      s32InternalState = s32NewState;
   }

   if ( VDS_C_S32_STATE_PENDING == s32NewState )
   {
      s32InternalState = VDS_C_S32_STATE_PENDING;
   }

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
              "OdomIf:Odometer New State = %d",s32InternalState );
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    stores odometer-data in ringbuffer, triggers check
//!    for subscribers (and send data)
//!
//! \return
//!   ok   : VDS_E_NO_ERROR
//!   error: VDS_E_INVALID_PARAMETER,
//!          VDS_E_ODOMIF_NOT_INITIALIZED
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tS32 tclMsgOdometerIf::s32AddOdometerData
(
   //! (I) odometer-data to be stored in ring buffer
   vds_trOdometerData* prOdometerData
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if( NULL != prOdometerData )
   {
      //Consistence check
      if( oRingBuffer.bIsValid() && oCS.bEnter() )
      {
         oRingBuffer.u32AddOne( *prOdometerData );
         oCS.vLeave();

         if( VDS_E_NO_ERROR != s32CheckForOdomSubscriber( prOdometerData, 1 ) )
         {
            s32RetVal = VDS_E_ODOMIF_SUBSCRIBER_ERROR;
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "OdomIf:s32AddOdometerData:Check for subscriber failed");
         }
      }
      else
      {
         s32RetVal = VDS_E_ODOMIF_NOT_INITIALIZED;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "OdomIf:s32AddOdometerData:Odometer ring buffer not initialized");
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "OdomIf:s32AddOdometerData:Invalid parameter" );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    Synchronises the Data. Updates Gala Speed, Checks for new Odometer
//!    and Gala Speed subscribers.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
void tclMsgOdometerIf::vSynch( void )
{
   s32CheckForOdomSubscriber( OSAL_NULL, 0 );
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    stores list of odometer-data in ringbuffer, triggers check
//!    for subscribers (and send data)
//!
//! \return
//!   ok   : VDS_E_NO_ERROR
//!   error: VDS_E_INVALID_PARAMETER,
//!          VDS_E_ODOMIF_NOT_INITIALIZED
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
//     21.03.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
//**************************************************************************
tS32 tclMsgOdometerIf::s32AddOdometerList
(
   //!  (I) list of odometer-data to be stored in ringbuffer
   const vds_trOdometerData *prOdometerData,
   //!  (I) num of datas in odometer-list
   tU32 u32NumOfData
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Parameter check
   if ( ( NULL != prOdometerData ) && ( 0 != u32NumOfData ) )
   {
      vOdoDataSanityCheck( 10, prOdometerData, u32NumOfData );
      
      //Consistence check
      if (oRingBuffer.bIsValid() && oCS.bEnter() )
      {
         oRingBuffer.u32AddList( prOdometerData, u32NumOfData );
         oCS.vLeave();

         //if ( VDS_E_NO_ERROR != s32CheckForOdomSubscriber( prOdometerData, u32NumOfData ) )
         //s32RetVal = VDS_E_ODOMIF_SUBSCRIBER_ERROR;
         
         // This has to be done after we have left the critical
         // section.  If it does cause a synchronisation event, we
         // will enter the critical section again when we try to send
         // off our data.
         vds_goSensorPropertySynchManager.bCheckTimeOut( );
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdomIf:Ring buffer not initialized" );
         s32RetVal = VDS_E_ODOMIF_NOT_INITIALIZED;
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdomIf:s32AddOdometerList:Invalid parameter" );
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    check for subscribers and send message.
//!    the actual list of odometer-data is given, so we normally
//!    don't need to create a temporary storage for the
//!    answer-message.
//!
//! \return
//!   ok   : VDS_E_NO_ERROR
//!   error: VDS_E_INVALID_PARAMETER,
//!          VDS_E_ODOMIF_ALLOC_ERROR,
//!          VDS_E_ODOMIF_GETODOMDATA_FAILED,
//!          VDS_E_ODOMIF_SETODOMMSG_FAILED,
//!          VDS_E_POST_MESSAGE_FAILED
//  HISTORY:
// Date            |  Author              | MODIFICATION
// -------------------------------------------------------------------------
//     21.03.02    |  CM-DI/ESA1-Fischer  |   Initial Revision.
//    25.06.2009   | sak9kor (RBEI/ECF1)  | Modified to adapt the XML generated
//                 |                      | FI
//    10.08.2009   | sak9kor (RBEI/ECF1)  | Modifications have been done to
//                 |                      | avoid the usage of additional memory pool
//**************************************************************************
tS32 tclMsgOdometerIf::s32CheckForOdomSubscriber
(
   //!  (I) received list of odometer-data
   vds_trOdometerData *prOdometerData,
   //!  (I) num of odometer-data in list
   tU32 u32NumOfData
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( ( VDS_C_S32_STATE_NORMAL == s32InternalState ) ||
        ( VDS_C_S32_STATE_NORMAL_DIAG == s32InternalState ) )
   {
      //Take Master Lock.
      if ( vds_goMasterLock.bEnter() )
      {
         tclSubscriberManager *poSubscriber =
                 tclSubscriberManager::poGetSubscriberWithFId( 
                    OSAL_NULL,
                    CCA_C_U16_SRV_SENSOR_LOCATION,
                    VDS_C_U16_FKTID_ODOMDATA_UPDATE );

         //While subscribers are available.
         while ( OSAL_NULL != poSubscriber )
         {
            //Get number of data to send
            tU32 u32NextElemToRead = poSubscriber->u32GetNextElemToRead();

            tS32 s32TempErrorValue = VDS_E_NO_ERROR;

            tS32 s32ElemsCreated = 0;

            do
            {
               amt_tclServiceData *poServiceData = OSAL_NULL;

               if ( oCS.bEnter() )
               {
                  //Get odometer-message
                  s32ElemsCreated = 
                     s32GetOdomMessage( OSAL_NULL,
                                        poSubscriber,
                                        &poServiceData,
                                        prOdometerData,
                                        u32NumOfData,
                                        u32NextElemToRead,
                                        !poSubscriber->bGetInitMessageSent() );
                  oCS.vLeave();
               }
               else
               {
                  s32ElemsCreated = VDS_E_ODOMIF_UNKNOWN_ERROR;
               }

               if( ( s32ElemsCreated > 0 ) && ( OSAL_NULL != poServiceData ) )
               {
                  //Send message
                  if ( VDS_E_NO_ERROR 
                       == 
                       pfs32SendSubscriberMessage( poServiceData, poSubscriber ) )
                  {
                     poSubscriber->vSetInitMessageSent( TRUE );
                     //Set next element to read
                     poSubscriber->vSetNextElemToRead( u32NextElemToRead );
                  }
                  else
                  {
                     s32TempErrorValue = VDS_E_POST_MESSAGE_FAILED;
                  }

                  OSAL_DELETE poServiceData;
               }
               else if ( s32ElemsCreated < 0 )
               {
                  s32TempErrorValue = s32ElemsCreated;
               }

            } while ( ( s32ElemsCreated > 0 ) 
                      && 
                      ( VDS_E_NO_ERROR == s32TempErrorValue ) );

            //If error occurs in loop and no error is set, take over temp error-value
            if ( ( VDS_E_NO_ERROR == s32RetVal ) && ( VDS_E_NO_ERROR != s32TempErrorValue ) )
            {
               s32RetVal = s32TempErrorValue;
            }

            //Get next subscriber with this function-id
            poSubscriber = 
               tclSubscriberManager::poGetSubscriberWithFId( poSubscriber,
                                                             CCA_C_U16_SRV_SENSOR_LOCATION,
                                                             VDS_C_U16_FKTID_ODOMDATA_UPDATE );

         }//End of while loop(subscribers available)

         //Leave critical section
         vds_goMasterLock.vLeave();

      } // end if lock subscriber successful
        // else no subscriber exists, return VDS_E_NO_ERROR
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    Deprecated.
//!    creates odometer-init-message.
//!
//! \return
//!   Deprecated values.
//!   [0..n] : num of data in message
//!   error: VDS_E_INVALID_PARAMETER,
//!          VDS_E_ODOMIF_ALLOC_ERROR,
//!          VDS_E_ODOMIF_GETODOMDATA_FAILED,
//!          VDS_E_ODOMIF_SETODOMMSG_FAILED
//!
//!   Current Value
//!   error: VDS_E_ODOMIF_UNKNOWN_ERROR
//!
//  HISTORY:
// Date            |  Author              | MODIFICATION
// -------------------------------------------------------------------------
//     03.03.03    |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
// Deprecated
tS32 tclMsgOdometerIf::s32GetOdomInitMessage
(
   //!  (I) pointer to created message
   amt_tclServiceData **ppoServiceData,
   //!  (I) target-application-id
   tU16 u16TargetAppId
)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( ppoServiceData );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u16TargetAppId );
   
   return VDS_E_ODOMIF_UNKNOWN_ERROR;
} // end of s32GetOdomInitMessage

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    creates odometer-status-message.
//!
//! \return
//!   [0..n] : num of data in message
//!   error: VDS_E_INVALID_PARAMETER,
//!          VDS_E_ODOMIF_ALLOC_ERROR,
//!          VDS_E_ODOMIF_GETODOMDATA_FAILED,
//!          VDS_E_ODOMIF_SETODOMMSG_FAILED
//!
//!
//  HISTORY:
// Date            |  Author              | MODIFICATION
// -------------------------------------------------------------------------
//     07.05.02    |   CM-DI/ESU1-Oester  | Initial Revision.
//     25.06.09    | sak9kor (RBEI/ECF1)  | XML generated FI has been adapted
//                 |                      | instead manual generated FI
//     29.06.09    | sak9kor (RBEI/ECF1)  | Replacing the old nomenclature of the
//                 |                      | objects (As per review comment
//                 |                      | from Mr.Peter Koenig)
//     13.07.2009  | sak9kor (RBEI/ECF1)  |  Time stamp related traces have been removed
//     10.08.2009  | sak9kor (RBEI/ECF1)  | New parameter poSubscriber has been added
//**************************************************************************
tS32 tclMsgOdometerIf::s32GetOdomMessage
(
   //!  (I) pointer to service data
   const amt_tclServiceData *poRequestMessage,
   //!  (I) pointer to subscriber data
   tclSubscriberManager *poSubscriber,
   //!  (O) pointer to created message
   amt_tclServiceData **ppoServiceData,
   //!  (I) list of odometer-data
   vds_trOdometerData *prOdometerData,
   //!  (I) num of odometer-data in list
   tU32 u32NumOfData,
   //!  (I) num of next elem to read
   tU32 &rfu32NextElemToRead,
   //!  (I) TRUE: init-message, FALSE: update-message
   tBool bSendInitMessage,
   //!  (I) TRUE: create message with state=invalid
   tBool bCreateInvalidMessage
)
{
   tS32 s32RetVal = OSAL_OK;
   vds_trOdometerData *parLocalOdometerData = OSAL_NULL; // points to temp buffer or param "prOdometerData"
   vds_trOdometerData *parTempOdometerData = OSAL_NULL;
   tU32 u32NumOfElemToSend = 0;
   tU32 u32TempNumOfData = 0;
   tU16 u16RegisterID = 0;
   tU16 u16CmdCounter = 0;
   tU16 u16TargetAppID = 0;
   tU16 u16ServiceID = 0;

   // param-check
   if(OSAL_NULL != poRequestMessage) 
   {
      u16TargetAppID = poRequestMessage->u16GetSourceAppID();
      u16RegisterID = poRequestMessage->u16GetRegisterID();
      u16CmdCounter = poRequestMessage->u16GetCmdCounter();
      u16ServiceID = poRequestMessage->u16GetServiceID();
   }
   else if(OSAL_NULL != poSubscriber) 
   {
      u16TargetAppID = poSubscriber->u16GetTargetAppId();
      u16RegisterID = poSubscriber->u16GetRegisterId();
      u16CmdCounter = poSubscriber->u16GetCommandCntr();
      u16ServiceID = poSubscriber->u16GetServiceId();
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }
   
   if ((OSAL_OK == s32RetVal)&&(OSAL_NULL != ppoServiceData))
   {
      *ppoServiceData = OSAL_NULL;

      if ( oCS.bLocked() )
      {
         u32NumOfElemToSend = oRingBuffer.u32GetElemsFrom( rfu32NextElemToRead );

         // elems available ?
         if ( u32NumOfElemToSend > 0 )
         {
            // *** get data to send ***
            
            // more data in buffer (to send) than in params ?
            if ( u32NumOfElemToSend > u32NumOfData )
            {
               // there are data in ringbuffer left, create temp.buffer to send data
               u32TempNumOfData = u32NumOfElemToSend;
               parTempOdometerData = OSAL_NEW vds_trOdometerData[ u32TempNumOfData ];
               if ( OSAL_NULL != parTempOdometerData )
               {
                  if ( (tS32)u32NumOfElemToSend == s32GetOdometerData(
                                                   parTempOdometerData,   
                                                   rfu32NextElemToRead,
                                                   u32NumOfElemToSend ) )
                  {
                     parLocalOdometerData = parTempOdometerData;
                     if ( bCreateInvalidMessage )
                     {
                        for ( tU32 ctu32OdomNumber = 0;
                              ctu32OdomNumber < u32NumOfElemToSend;
                              ctu32OdomNumber++ )
                        {
                           parLocalOdometerData[ctu32OdomNumber].u8OdometerStatus =
                              VDS_C_U8_ODOMSTATE_CONNECTED_DATA_INVALID;
                        }
                     }
                  }
                  else
                  {
                     vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                        "OdomIf: couldn't get requested (%d) num of data",
                        u32NumOfElemToSend );
                     s32RetVal = VDS_E_ODOMIF_GETODOMDATA_FAILED;
                  }
               }
               else
               {
                  vTraceMsg( VDS_C_TRACELEVEL_ERROR, "OdomIf: s32GetOdomMessage: alloc-error" );
                  s32RetVal = VDS_E_ODOMIF_ALLOC_ERROR;
               }
            }
            else
            {
               // no data in ringbuffer left, send only actual data
               parLocalOdometerData = prOdometerData;
               rfu32NextElemToRead += u32NumOfElemToSend;
            }

            // are there elems to send?
            if ( OSAL_NULL != parLocalOdometerData )
            {
                    
                              // *** create update-message ***
               /* Create the object for XML generated FI for Status OP type*/
               sensor_locationfi_tclMsgOdometerData_UpdateStatus poOdometerDataUpdateStatus;
               
               // populate the data structure in to this status FI object
               //oOdoDataUpdateStatus.OdometerData = std::vector<sensor_fi_tcl_OdometerData>(u32NumOfElemToSend);
               try
               {
                  poOdometerDataUpdateStatus.OdometerData.resize(u32NumOfElemToSend);
               }
               catch(std::bad_alloc& ba)
               {
                  vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdoIf:exception bad_alloc caught",ba.what());
               }

               for (tU32 u32Loop = 0; u32Loop < u32NumOfElemToSend; u32Loop++)
               {
                  poOdometerDataUpdateStatus.OdometerData[u32Loop].Timestamp = 
                     parLocalOdometerData[u32Loop].u32Timestamp;

                  poOdometerDataUpdateStatus.OdometerData[u32Loop].OdometerCount = 
                     parLocalOdometerData[u32Loop].u16OdometerCount;

                  poOdometerDataUpdateStatus.OdometerData[u32Loop].OdometerStatus.enType = 
                     (sensor_fi_tcl_e8_OdometerStatus::tenType)parLocalOdometerData[u32Loop].u8OdometerStatus;

                  poOdometerDataUpdateStatus.OdometerData[u32Loop].OdometerCount_Direction.enType = 
                     (sensor_fi_tcl_e8_OdometerDirection::tenType)parLocalOdometerData[u32Loop].u8OdometerCountDirection;
               }
             
               tU32 u32TS1 = parLocalOdometerData[0].u32Timestamp;
               tU32 u32TS2 = parLocalOdometerData[u32NumOfElemToSend-1].u32Timestamp;

               vTraceMsg( VDS_C_TRACELEVEL_EVENT,
                          "Odometer message ts = %d--%d", 
                           u32TS1, u32TS2);

               vOdoDataSanityCheck( 100, parLocalOdometerData, 
                                      u32NumOfElemToSend );
                        
               
               // populate the initial value in to this status FI object
               poOdometerDataUpdateStatus.FirstOdomValueIsInitial = bSendInitMessage;
               /* Create the object of visitor message and insert the FI object*/
               fi_tclVisitorMessage* poResultMessage = OSAL_NEW fi_tclVisitorMessage( poOdometerDataUpdateStatus, VDS_C_U16_SRV_SENSORS_MAJOR_VERSION );
               if ( OSAL_NULL != poResultMessage )
               {
                  // Prepare response message
                  if(poResultMessage->bIsValid())
                  {
                     tU32 u32TimeStamp = 0;
                     vGetTimeStamp( &u32TimeStamp );
                     poResultMessage->vInitServiceData
                     ( CCA_C_U16_APP_SENSOR, // Source
                       u16TargetAppID,       // Target
                       AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,     // StreamType
                       0,                            // StreamCounter
                       u16RegisterID,    // RegisterID
                       u16CmdCounter,    // nCmdCounter,
                       u16ServiceID,     // nServiceID,
                       SENSOR_LOCATIONFI_C_U16_ODOMETERDATA_UPDATE, // nFunctionID,
                       AMT_C_U8_CCAMSG_OPCODE_STATUS,         // OpCode
                       0,                            // Asynchronous Completion Token (ACT)
                       AMT_C_U16_DEFAULT_NULL,       // Source Sub
                       AMT_C_U16_DEFAULT_NULL,       // Target Sub
                       u32TimeStamp                  //TimeStamp
                     );
                     *ppoServiceData = poResultMessage;/*static_cast<amt_tclServiceData *>(poResultMessage);*/
                     s32RetVal = ( tS32 )u32NumOfElemToSend;
                  }
                  else
                  {
                     s32RetVal = VDS_E_ODOMIF_ALLOC_ERROR;;
                  }

                  if ( s32RetVal < ( tS32 )u32NumOfElemToSend )
                  {
                     OSAL_DELETE  poResultMessage;
                     poResultMessage = OSAL_NULL;
                  }
                  VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(poResultMessage);
               }
               else
               {
                  s32RetVal = VDS_E_ODOMIF_ALLOC_ERROR;
               }

               parLocalOdometerData = OSAL_NULL;
            }

            // else no data to send (nor data in ringbuffer neither in params)

            // (if exists) delete temp buffer
            if (     parTempOdometerData != OSAL_NULL
                  && parTempOdometerData != prOdometerData )
            {
               // delete temp.buffer
               OSAL_DELETE [] parTempOdometerData;
               parTempOdometerData = OSAL_NULL;
            }
         } // end if elems available
         else
         {
            // no elem to send
            s32RetVal = VDS_E_ODOMIF_DATA_UNAVAILABLE;
         }
      } // end if bCritSectionLocked
      else
      {
         s32RetVal = VDS_E_ODOMIF_NOT_INITIALIZED;
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }
   
   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTempOdometerData);
   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parLocalOdometerData);
   
   
   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    returns odometer-value of last read data-set
//!
//! \return
//!   ok   : VDSENSOR_MSG_ODOMETER_OK
//!   error: VDSENSOR_MSG_ODOMETER_ERROR
//!
//!
//  HISTORY:
// Date            |  Author              | MODIFICATION
// -------------------------------------------------------------------------
//     21.03.02    |   CM-DI/ESU1-Oester  | Initial Revision.
//**************************************************************************
tU8 tclMsgOdometerIf::getOdometerValue
(
   //! (O) storage for odometer-value
   tPU16 pu16OdometerValue
)
{
   tU8 u8RetVal = VDSENSOR_MSG_ODOMETER_ERROR;

   // param-check
   if ( NULL != pu16OdometerValue )
   {
      // consistence check
      if ( oRingBuffer.bIsValid() && oCS.bLocked() )
      {
         vds_trOdometerData rTemp = {0};
         // odometer-value available?
         if ( oRingBuffer.bGetLastElem( &rTemp ) )
         {
            *pu16OdometerValue = rTemp.u16OdometerCount;
            u8RetVal = VDSENSOR_MSG_ODOMETER_OK;
         }
      }
   }
   return u8RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    returns a list of odometer-datas from ringbuffer.
//!    the number of the element from where to be read could be
//!    determined from rfu32NextElemToRead. if the number is
//!    smaller than the ones in the list, the oldest datas will be
//!    returned, if the number is bigger than the ones in the list,
//!    no data is returned. rfu32NextElemToRead will be set to
//!    the number of the last returned data-set + 1.
//!
//! \return
//!   [0..n]   : num of elems copied
//!   error    : VDS_E_INVALID_PARAMETER,
//!              VDS_E_ODOMIF_NOT_INITIALIZED
//!
//  HISTORY:
// Date            |  Author              | MODIFICATION
// -------------------------------------------------------------------------
//     21.03.02    |   CM-DI/ESU1-Oester  | Initial Revision.
//**************************************************************************
tS32 tclMsgOdometerIf::s32GetOdometerData
(
   //!  (O) storage for odometer-list
   vds_trOdometerData *prOdometerData,
   //!  (IO) number of element that has to be read the next time
   tU32 &rfu32NextElemToRead,
   //!  (I) number of elements to be read
   tU32 u32ElemsToRead
)
{
   tS32 s32ElemsCopied = 0;
   vds_trOdometerData *prData = prOdometerData;
   
   // param-check
   if (     NULL != prData
         && 0    != u32ElemsToRead )
   {
      // consistence check
      if ( oRingBuffer.bIsValid() && oCS.bLocked() )
      {
         tU32 u32Elems;

         u32Elems = oRingBuffer.u32GetList( prData, u32ElemsToRead, rfu32NextElemToRead );
         
         // elems lost? -> set status in first copied odometer-data
         if ( u32Elems != u32ElemsToRead )
            if(prData->u8OdometerStatus != VDS_C_U8_ODOMSTATE_CONNECTED_DATA_INVALID)
               prData->u8OdometerStatus = VDS_C_U8_ODOMSTATE_CONNECTED_INTERNALERROR;
         
         s32ElemsCopied = (tS32)u32Elems;
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "OdomIf: s32GetOdometerData: not init." );
         s32ElemsCopied = VDS_E_ODOMIF_NOT_INITIALIZED;
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR, "OdomIf: s32GetOdometerData: invalid param" );
      s32ElemsCopied = VDS_E_INVALID_PARAMETER;
   }

   return s32ElemsCopied;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!    stores list of odometer-data in ringbuffer, triggers check
//!    for subscribers (and send data)
//!
//! \return
//!  ok   : VDS_E_NO_ERROR
//!  error: VDS_E_INVALID_PARAMETER,
//!         VDS_E_ODOMIF_NOT_INITIALIZED
//!
//  HISTORY:
// Date            |        Author        | MODIFICATION
// -------------------------------------------------------------------------
//    19.04.02     | CM-DI/ESA1-Fischer   | Initial Revision.
//**************************************************************************
tS32 tclMsgOdometerIf::s32AddDataWrapper
(
   //! (I) list of odometer-data to be stored in ringbuffer
   vds_trOdometerData *prOdometerData,
   //! (I)  num of datas in odometer-list
   tU32 u32NumOfData
)
{
   tS32 s32RetVal = VDS_E_ODOMIF_NOT_INITIALIZED;

   if ( NULL != poThisMsgOdometerIf )
   {
      s32RetVal = poThisMsgOdometerIf->s32AddOdometerList( prOdometerData, u32NumOfData );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!     Gets Odometer Range
//!
//! \return
//!   None
//!
//  HISTORY:
//      Date       |        Author        | MODIFICATION
// ---------------------------------------------------------------------
//*********************************************************************
void tclMsgOdometerIf::vGetOdometerRange
(
   //!  (O)  reference to Minimum Range
   tF32& rfrf32Min,
   //!  (O)  reference to Max Range
   tF32& rfrf32Max,
   //!  (O)  reference to Status
   tU8& rfru8Status )
{
   rfrf32Min = 0.0;
   rfrf32Max = 0.0;
   rfru8Status = VDS_C_U8_ODOMETERRANGE_UNKNOWN;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!     Updates Odo data in AllSensorData cmb fi object.
//!
//! \return
//!   Success  :  VDS_E_NO_ERROR
//!   Error    :  VDS_E_ODOMIF_NOT_INITIALIZED
//!               VDS_E_ODOMIF_GETODOMDATA_FAILED
//!               VDS_E_ODOMIF_DATA_UNAVAILABLE
//!               VDS_E_ODOMIF_ALLOC_ERROR
//!
//  HISTORY:
//      Date       |        Author        | MODIFICATION
// -------------------------------------------------------------------------
//**************************************************************************

tS32 tclMsgOdometerIf::s32GetOdoRecords
(
   sensor_fi_tcl_AllSensorData *pCmbFiGnssData,

   tU32 &NextelementToRead
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
   vds_trOdometerData *parTempOdometerData = OSAL_NULL;
   tU32 u32NumOfElemToSend = 0;


   if( oCS.bLocked() )
   {
      u32NumOfElemToSend = oRingBuffer.u32GetElemsFrom( NextelementToRead );

      //Elements Available?
      if ( u32NumOfElemToSend > 0 )
      {
         //There is still data in ring buffer left, create temporary buffer to send data
         parTempOdometerData = OSAL_NEW vds_trOdometerData[ u32NumOfElemToSend ];

         if( OSAL_NULL == parTempOdometerData )
         {
            s32RetVal = VDS_E_ODOMIF_ALLOC_ERROR;
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdoIf:OdoAllSen:Unable to allocate memory.");
         }
         else if ( (tS32)u32NumOfElemToSend != s32GetOdometerData( parTempOdometerData,
                                                                   NextelementToRead,
                                                                   u32NumOfElemToSend ) )
         {
            s32RetVal = VDS_E_ODOMIF_GETODOMDATA_FAILED;
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                          "OdoIf:OdoAllSen:couldn't get requested (%d) num of data",
                          u32NumOfElemToSend );
         }

         //Are there elements to send?
         if( ( OSAL_NULL != parTempOdometerData ) && ( s32RetVal == VDS_E_NO_ERROR ) )
         {
            //Populate the data structure in to this status FI object
            try
            {
               pCmbFiGnssData->OdometerUpdate.resize(u32NumOfElemToSend);
            }
            catch(std::bad_alloc& ba)
            {
               vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdoIf:exception bad_alloc caught",ba.what());
            }
            for (tU32 u32Loop = 0; u32Loop < u32NumOfElemToSend; u32Loop++)
            {
               pCmbFiGnssData->OdometerUpdate[u32Loop].Timestamp 
                        = parTempOdometerData[u32Loop].u32Timestamp;
               pCmbFiGnssData->OdometerUpdate[u32Loop].OdometerCount
                        = parTempOdometerData[u32Loop].u16OdometerCount;
               pCmbFiGnssData->OdometerUpdate[u32Loop].OdometerStatus.enType
                        = (sensor_fi_tcl_e8_OdometerStatus::tenType)parTempOdometerData[u32Loop].u8OdometerStatus;
               pCmbFiGnssData->OdometerUpdate[u32Loop].OdometerCount_Direction.enType
                        = (sensor_fi_tcl_e8_OdometerDirection::tenType)parTempOdometerData[u32Loop].u8OdometerCountDirection;
            }

            tU32 u32TS1 = parTempOdometerData[0].u32Timestamp;
            tU32 u32TS2 = parTempOdometerData[u32NumOfElemToSend-1].u32Timestamp;

            vTraceMsg( VDS_C_TRACELEVEL_EVENT,
                       "OdoIf:OdoAllSen:Odometer message ts = %d--%d",
                       u32TS1, u32TS2);

            vOdoDataSanityCheck( 100, parTempOdometerData,u32NumOfElemToSend );
         }

         //(If exists)Delete temp buffer
         if ( parTempOdometerData != OSAL_NULL )
         {
            OSAL_DELETE [] parTempOdometerData;
            parTempOdometerData = OSAL_NULL;
         }
      }
      else
      {
         //No elements to send
         s32RetVal = VDS_E_ODOMIF_DATA_UNAVAILABLE;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdoIf:OdoAllSen:Data Unavailable" );
      }
   }
   else
   {
      s32RetVal = VDS_E_ODOMIF_NOT_INITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"OdoIf:OdoAllSen:Not Initialized" );
   }

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTempOdometerData);
   return s32RetVal;
}

//EOF

