// *****************************************************************************
// * FILE:         tclMsgGyroIf.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition: handles access to gyro-data,
// *               handles subscriber-messages
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
// * 07.07.09  -  sak9kor - APIs "s32GetGyroScalefactorMessage","s32GetGyroErrorMessage", and
// *                          "s32GetGyroMessage" have been modified to
// *                            to adopt the XML generated FI
// * 10.08.09 - sak9kor - Modifications have been done to avoid the usage
// *                       aditional memory pool, and new parameter poSubscriber has been added
// *                       in to "s32GetGyroScalefactorMessage","s32GetGyroErrorMessage",
// *                       "s32GetGyroMessage" functions
// * 28.08.09 RBEI/ECF1 - sak9kor - Trace Output related modifications bave been done
// * 08.09.09 RBEI/ECF1 - sak9kor - Trace levels have been modified
// * 30.09.09 RBEI/ECF1 - sak9kor - Added doxygen headers for all classes/funcs
// * 27.08.10 RBEI/ECF1 - sak9kor - New function bGetDiag3DGyroRawData has been added
// * 23.09.10 RBEI/ECF1 - sak9kor - Two new macros related to Gyro Axx100AQ have been
// *                                added for ADIT Gen2 in functions "setGyroManufacturer"
// *                                and setGyroType functions
// *****************************************************************************

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SYSTEM_S_IMPORT_INTERFACE_ERRMEM_DEF
#include "system_pif.h"

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"

#define CCA_S_IMPORT_INTERFACE_GENERIC
#include "cca_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_ERRORCODES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_FUNCTIONIDS
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_SERVICEINFO
#include "sensor_fi_if.h"
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "fi_msgfw_if.h"
#define SENSOR_S_IMPORT_INTERFACE_SERVER
#include "vds2_if.h"

#define VDS_S_IMPORT_INTERFACE_SUBSCRIBER_MANAGER
#define VDS_S_IMPORT_INTERFACE_MESSAGE_INTERFACES
#define VDS_S_IMPORT_INTERFACE_PERSISTENT_DATA
#define VDS_S_IMPORT_INTERFACE_SENSOR_DATATYPES
#define VDS_S_IMPORT_INTERFACE_TRACE
#include "vds_internal_if.h"

extern void vGetTimeStamp( tPU32 pu32TimeStamp );

/*!
* \brief Defines the interface for Gyrosynchlistner.
*/

class vds_tclGyroSynchListener : public vds_tclSynchListenerIf
{
   private:
      tclMsgGyroIf *poGyroIf;
   public:
      // ***************** F U N C T I O N  H E A D E R ************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Default constructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -----------------------------------------------------------------------
      //************************************************************************
      vds_tclGyroSynchListener
      (
         //! pointer to gyro interface argument
         tclMsgGyroIf *poGyroIfArg
      )
      {
         poGyroIf = poGyroIfArg;
      }
      // ***************** F U N C T I O N  H E A D E R ************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Performs the synch operation
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -----------------------------------------------------------------------
      //************************************************************************
      virtual void vSynch
      (
         //! None
         void
      )
      {
         vTraceMsg( VDS_C_TRACELEVEL_EVENT, "GyroIf:Synch (gyro)" );

         if ( poGyroIf != OSAL_NULL )
         {
            poGyroIf->vSynch( );
         }
      }
};


// Class tclMsgGyroIf
tclMsgGyroIf *tclMsgGyroIf::poThisMsgGyroIf;
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default constructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclMsgGyroIf::tclMsgGyroIf()
      : oCS( "VDS_GYRO_IF_SEM" ),
      poGyroThread( OSAL_NULL ),
      s32InternalState( VDS_C_S32_STATE_UNINITIALIZED )
{
   poThisMsgGyroIf = this;

   vds_goSensorPropertySynchManager.bAddListener( 
      new vds_tclGyroSynchListener( this ) );
   pfs32SendSubscriberMessage = NULL;
   u32NextTimeToSend = 0;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Destructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclMsgGyroIf::~tclMsgGyroIf()
{

   oRingBuffer3d.vDeallocate();
  
   if ( !oCS.bShutdown() )
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "GyroIf:Error shutting down MsgGyroIf cs" );
   }

   //Delete gyro-thread
   if ( NULL != poGyroThread )
   {
      poGyroThread->s32ThreadStop();
      delete poGyroThread;
      poGyroThread = NULL;
   }
   poThisMsgGyroIf = NULL;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   initializes gyro-interface create and initialize instance of gyro-thread,
//!   create semaphore, create ringbuffer for gyro-data, initialize
//!   member-attributes
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_ALLOC_ERROR : Gyro interface allocation error
//!   - \c VDS_E_GYROIF_SEM_ERROR :  Gyro sempahore error
//!   - \c VDS_E_GYROIF_THREADINIT_ERROR : Gyro interface thread interface error
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGyroIf::s32Init
(
   //! (I) : pointer to function, which sends given answer message with given
   //! parameters to subscribers
   tS32( *pfs32SendSubscriberMessageParam )( amt_tclServiceData *, tclSubscriberManager * )
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
   tU32 u32GyroCycleTime = 0;

   //Parameter check
   if ( NULL != pfs32SendSubscriberMessageParam )
   {
      //Save ail-sensor-interface
      this->pfs32SendSubscriberMessage = pfs32SendSubscriberMessageParam;

      //Create GYRO thread
      poGyroThread = new tclGyroThread();

      if ( NULL != poGyroThread )
      {
         if ( VDS_E_NO_ERROR 
              == 
              poGyroThread->s32ThreadInit( this, tclMsgGyroIf::s32Add3dDataWrapper ) )
         {
            //Create semaphore
            if ( oCS.bInit() )
            {
               tU32 u32RingBufferSize;
               
               u32GyroCycleTime = poGyroThread->u32GetGyroCycleTime();
               if(u32GyroCycleTime != 0)
               {
               //Create GYRO storage (seconds*num_of_sensordata_per_second)
               u32RingBufferSize = ( tU32 )( VDS_C_U32_BUFFERSIZE_SECONDS *
                                              ( 1000000000U /  u32GyroCycleTime) );
               vTraceMsg( 
                  VDS_C_TRACELEVEL_COMPONENT,
                  "GyroIf:Gyro u32RingBufferSize = %d",u32RingBufferSize );
               }
               else
               {
                  vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                             "GyroIf:s32Init:VDS_E_ALLOC_ERROR :Invalid Gyro cycle time .");
                  return VDS_E_ALLOC_ERROR;
               }

               oRingBuffer3d.vAllocate( u32RingBufferSize );

              //Does 3dGyro ring buffer created successfully.
               if ( !oRingBuffer3d.bIsValid() )
               {
                  s32RetVal = VDS_E_GYROIF_ALLOC_ERROR;
                  oCS.bShutdown();
                  vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                             "GyroIf:s32Init:RingBuffer3d alloc error!" );
               }
            }
            else
            {
               vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                          "GyroIf:s32Init:semaphore create error" );

               s32RetVal = VDS_E_GYROIF_SEM_ERROR;
            }

            if ( VDS_E_NO_ERROR != s32RetVal )
            {
               poGyroThread->vThreadClose();
            }
         }
         else
         {
            s32RetVal = VDS_E_GYROIF_THREADINIT_ERROR;
            vTraceMsg( VDS_C_TRACELEVEL_FATAL, "GyroIf:s32Init:Alloc error" );
         }

         if ( VDS_E_NO_ERROR != s32RetVal )
         {
            delete poGyroThread;
            poGyroThread = NULL;
         }
      }
      else
      {
         s32RetVal = VDS_E_GYROIF_ALLOC_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "GyroIf:s32Init:s32ThreadInit failed" );
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GyroIf:s32Init:Invalid parameters");
   }

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   start gyro-thread
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_THREADSTART_ERROR : Thread start error
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGyroIf::s32StartThread
(
   //! None
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   // start gyro-thread
   if ( NULL != poGyroThread )
   {
      if ( VDS_C_S32_THREAD_RUNNING != poGyroThread->s32ThreadStart() )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, 
                    "GyroIf:Couldn't start Gyro thread" );

         s32RetVal = VDS_E_GYROIF_THREADSTART_ERROR;
      }
   }
   else
   {
      s32RetVal = VDS_E_GYROIF_NOT_INITIALIZED;
   }
   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   stop gyro-thread
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_THREADSTOP_ERROR : Thread stop error
//!   - \c VDS_E_GYROIF_NOT_INITIALIZED : Gyro interface not initialized
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGyroIf::s32StopThread
(
   //! None
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Stop GYRO thread
   if ( NULL != poGyroThread )
   {
      if ( VDS_C_S32_THREAD_INITIALIZED != poGyroThread->s32ThreadStop() )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GyroIf:Couldn't stop GyroThread" );
         s32RetVal = VDS_E_GYROIF_THREADSTOP_ERROR;
      }
   }
   else
   {
      s32RetVal = VDS_E_GYROIF_NOT_INITIALIZED;
   }

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   delete gyro-thread
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_THREADSTOP_ERROR : Thread stop error
//!   - \c VDS_E_GYROIF_NOT_INITIALIZED : Gyro interface not initialized
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGyroIf::s32DeleteThread
(
   //! None
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Delete gyro-thread
   if ( NULL != poGyroThread )
   {
      if ( VDS_C_S32_THREAD_INITIALIZED != poGyroThread->s32ThreadStop() )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "GyroIf: couldn't stop GyroThread" );
         s32RetVal = VDS_E_GYROIF_THREADSTOP_ERROR;
      }

      delete poGyroThread;
      poGyroThread = NULL;
   }
   else
   {
      s32RetVal = VDS_E_GYROIF_NOT_INITIALIZED;
   }

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   set internal state (normal or pending)
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclMsgGyroIf::vSetState
(
   //! (I) : new state
   //! (VDS_C_S32_STATE_NORMAL, VDS_C_S32_STATE_NORMAL_DIAG, VDS_C_S32_STATE_PENDING)
   tS32 s32NewState
)
{
   if ( ( VDS_C_S32_STATE_NORMAL == s32NewState )||
        ( VDS_C_S32_STATE_NORMAL_DIAG == s32NewState ) )
   {

      s32InternalState = s32NewState;
   }

   if ( VDS_C_S32_STATE_PENDING == s32NewState )
   {
      s32InternalState = VDS_C_S32_STATE_PENDING;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Performs the synch operation
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void tclMsgGyroIf::vSynch
(
   //! None
   void
)
{

   s32CheckFor3dGyroSubscriber( OSAL_NULL, 0 ) ;

   if( (u32NextTimeToSend != 0) &&
       (u32NextTimeToSend - (OSAL_ClockGetElapsedTime()/1000) > 0x7FFFFFFF) )
   {
           u32NextTimeToSend = u32CheckForGyroTemperature();
   }
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   stores list of 3D gyro-data in ringbuffer, triggers check for
//!   subscribers (and send data)
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_SUBSCRIBER_ERROR : Subscriber error
//!   - \c VDS_E_GYROIF_NOT_INITIALIZED : Gyro interface not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGyroIf::s32Add3dGyroList
(
   //! (I) : list of gyro-data to be stored in ringbuffer
   const sensor_fi_tcl_3dGyroData *pr3dGyroData,
   //! (I) : num of datas in gyro-list
   tU32 u32NumOfData
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Parameter check
   if ( ( NULL != pr3dGyroData ) && ( 0 != u32NumOfData ) )
   {
      //vGyroDataSanityCheck( 10, prGyroData, u32NumOfData );
      //consistence check
      if( oRingBuffer3d.bIsValid() && oCS.bEnter() )
      {
         oRingBuffer3d.u32AddList( pr3dGyroData, u32NumOfData );
         oCS.vLeave();
         // if ( VDS_E_NO_ERROR != s32CheckForGyroSubscriber( prGyroData, u32NumOfData ) )
         // s32RetVal = VDS_E_GYROIF_SUBSCRIBER_ERROR;
         // This has to be done after we have left the critical
         // section.  If it does cause a synchronisation event, we
         // will enter the critical section again when we try to send
         // off our data.
         vds_goSensorPropertySynchManager.bCheckTimeOut( );
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "GyroIf: s32Add3dGyroList: not init." );
         s32RetVal = VDS_E_GYROIF_NOT_INITIALIZED;
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR, "GyroIf: s32Add3dGyroList: invalid param" );
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   creates gyro-status-message.
//! \return
//!   Succes code in case of success is:
//!   - Number of elements created\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_GETGYRODATA_FAILED : Get gyro data failed
//!   - \c VDS_E_GYROIF_ALLOC_ERROR : Allocation error
//!   - \c VDS_E_GYROIF_DATA_UNAVAILABLE : Data unavailable
//!   - \c VDS_E_GYROIF_NOT_INITIALIZED : Data unavailable
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
// 06.07.2009   | sak9kor (RBEI/ECF1)    | XML generated FI has been adapted
//                                         instead manual generated FI
// 10.08.2009   | sak9kor (RBEI/ECF1)    | New parameter poSubscriber has been added
//******************************************************************************

tS32 tclMsgGyroIf::s32Get3dGyroMessage
(
   //! (I) : pointer to service data
   amt_tclServiceData const *poRequestMessage,
   //! (I) : pointer to subscriber data
   tclSubscriberManager *poSubscriber,
   //! (O) : pointer to a location where a pointer to the reply message is available
   amt_tclServiceData **ppoServiceData,
   //! (I) : pointer to Gyro data
   sensor_fi_tcl_3dGyroData *pr3dGyroData,
   //! (I) : Number of data
   tU32 u32NumOfData,
   //! (IO) : number of element that has to be read the next time
   tU32 &rfu32NextElemToRead,
   //! (I) :  TRUE: init-message, FALSE: update-message
   tBool /*bSendInitMessage*/,
   //! (I) : TRUE: create message with state=invalid
   tBool bCreateInvalidMessage
)
{
   tS32 s32RetVal = OSAL_OK;
   sensor_fi_tcl_3dGyroData *parLocal3dGyroData = OSAL_NULL; // points to temp buffer or param "prGyroData"
   sensor_fi_tcl_3dGyroData *parTemp3dGyroData = OSAL_NULL;
   tU32 u32NumOfElemToSend = 0;
   tU32 u32TempNumOfData = 0;
   tU16 u16RegisterID = 0;
   tU16 u16CmdCounter = 0;
   tU16 u16TargetAppID = 0;
   tU16 u16ServiceID = 0;



   if ( OSAL_NULL != poRequestMessage )
   {
      u16TargetAppID = poRequestMessage->u16GetSourceAppID();
      u16RegisterID  = poRequestMessage->u16GetRegisterID();
      u16CmdCounter  = poRequestMessage->u16GetCmdCounter();
      u16ServiceID   = poRequestMessage->u16GetServiceID();
   }
   else if ( OSAL_NULL != poSubscriber )
   {
      u16TargetAppID = poSubscriber->u16GetTargetAppId();
      u16RegisterID  = poSubscriber->u16GetRegisterId();
      u16CmdCounter  = poSubscriber->u16GetCommandCntr();
      u16ServiceID   = poSubscriber->u16GetServiceId();
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   //Parameter check
   if ( ( OSAL_OK == s32RetVal ) && ( OSAL_NULL != ppoServiceData ) )
   {
      *ppoServiceData = OSAL_NULL;

      if ( oCS.bLocked() )
      {
         u32NumOfElemToSend = oRingBuffer3d.u32GetElemsFrom( rfu32NextElemToRead );

         // elems available ?
         if ( u32NumOfElemToSend > 0 )
         {
            //////Get data to send

            // more data in buffer (to send) than in params ?
            if ( u32NumOfElemToSend > u32NumOfData )
            {
               // there are data in ring buffer left, create temp.buffer to send data
               u32TempNumOfData = u32NumOfElemToSend;

               parTemp3dGyroData = 
                  OSAL_NEW sensor_fi_tcl_3dGyroData[ u32TempNumOfData ];

               if ( OSAL_NULL != parTemp3dGyroData )
               {
                  if ( (tS32)u32NumOfElemToSend  == 
                                   s32Get3dGyroData( parTemp3dGyroData, 
                                                     rfu32NextElemToRead, 
                                                     u32NumOfElemToSend ) )
                  {
                     parLocal3dGyroData = parTemp3dGyroData;

                     if ( bCreateInvalidMessage )
                     {
                        for ( tU32 ctu32GyroNumber = 0;
                              ctu32GyroNumber < u32NumOfElemToSend;
                              ctu32GyroNumber++ )
                        {
                           parLocal3dGyroData[ctu32GyroNumber].RStatus.enType = 
                              sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR;

                           parLocal3dGyroData[ctu32GyroNumber].SStatus.enType = 
                              sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR;

                           parLocal3dGyroData[ctu32GyroNumber].TStatus.enType = 
                              sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR;
                        }
                     }
                  }
                  else
                  {
                     s32RetVal = VDS_E_GYROIF_GETGYRODATA_FAILED;
                     vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                                "GyroIf: s32GetGyroMessage:Couldn't get requested (%d) 3dgyro-data",
                                u32NumOfElemToSend );
                  }
               }
               else
               {
                  s32RetVal = VDS_E_GYROIF_ALLOC_ERROR;
                  vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GyroIf:s32Get3dGyroMessage:Allocation error" );
               }
            }
            else
            {
               // no data in ringbuffer left, send only actual data
               parLocal3dGyroData = pr3dGyroData;
               rfu32NextElemToRead += u32NumOfElemToSend;
            }

            //Are there any elements to send?
            if ( OSAL_NULL != parLocal3dGyroData )
            {
               //Create update message.
               sensor_locationfi_tclMsgGyro3dData_UpdateStatus oGyro3dDataUpdateStatus;

               //oGyro3dDataUpdateStatus.GyroData.resize( u32NumOfElemToSend );

               for ( tU32 u32Loop = 0; u32Loop < u32NumOfElemToSend; u32Loop++ )
               {
                  // copy values from array to list
                  oGyro3dDataUpdateStatus.Gyro3dData.push_back(parLocal3dGyroData[u32Loop]);
               }

               tU32 u32TS1 = parLocal3dGyroData[0].TimeStamp3dGyro;
               tU32 u32TS2 = parLocal3dGyroData[u32NumOfElemToSend-1].TimeStamp3dGyro;

               vTraceMsg( VDS_C_TRACELEVEL_EVENT,
                          "GyroIf:3D-Gyro message ts = %d--%d", u32TS1, u32TS2 );

               // vGyroDataSanityCheck( 100, parLocal3dGyroData, u32NumOfElemToSend );
               /* Create the object of visitor message and insert the FI object*/
               fi_tclVisitorMessage* poResultMessage = 
                               OSAL_NEW fi_tclVisitorMessage( oGyro3dDataUpdateStatus, VDS_C_U16_SRV_SENSORS_MAJOR_VERSION );

               if ( OSAL_NULL != poResultMessage )
               {
                  // Prepare response message
                  if ( poResultMessage->bIsValid() )
                  {
                     tU32 u32TimeStamp = 0;
                     vGetTimeStamp( &u32TimeStamp );

                     poResultMessage->vInitServiceData
                             (  CCA_C_U16_APP_SENSOR,                 // Source
                                u16TargetAppID,                       // Target
                                AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,    // StreamType
                                0,                                    // StreamCounter
                                u16RegisterID,                        // RegisterID
                                u16CmdCounter,                        // nCmdCounter
                                u16ServiceID,                         // nServiceID
                                SENSOR_LOCATIONFI_C_U16_GYRO3DDATA_UPDATE, // nFunctionID
                                AMT_C_U8_CCAMSG_OPCODE_STATUS,        // OpCode
                                0,                                    // Asynchronous Completion Token (ACT)
                                AMT_C_U16_DEFAULT_NULL,               // Source Sub
                                AMT_C_U16_DEFAULT_NULL,               // Target Sub
                                u32TimeStamp                          // TimeStamp
                              );
                     *ppoServiceData = poResultMessage;/*static_cast<amt_tclServiceData *>(poResultMessage);*/
                     s32RetVal = ( tS32 )u32NumOfElemToSend;
                  }
                  else
                  {
                     s32RetVal = VDS_E_ODOMIF_ALLOC_ERROR;
                  }

                  if ( s32RetVal < ( tS32 )u32NumOfElemToSend )
                  {
                     OSAL_DELETE poResultMessage;
                     poResultMessage = OSAL_NULL;
                  }
               }
               else
               {
                  s32RetVal = VDS_E_GYROIF_ALLOC_ERROR;
               }

               parLocal3dGyroData = OSAL_NULL;

               VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parLocal3dGyroData);
               VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(poResultMessage);
            }

            // (if exists) delete temp buffer
            if ( ( parTemp3dGyroData != OSAL_NULL ) &&
                 ( parTemp3dGyroData != pr3dGyroData ) )
            {
               //Delete temporary buffer
               OSAL_DELETE [] parTemp3dGyroData;
               parTemp3dGyroData = OSAL_NULL;
            }
         } // end if elems available
         else
         {
            //No elements to send
            s32RetVal = VDS_E_GYROIF_DATA_UNAVAILABLE;
         }
      } // end if oCS.bLocked
      else
      {
         s32RetVal = VDS_E_GYROIF_NOT_INITIALIZED;
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GyroIf:s32Get3dGyroMessage:Invalid parameters");
   }

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTemp3dGyroData);
   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   returns a list of gyro-datas from ringbuffer.the number of the element
//!   from where to be read could be determined from rfu32NextElemToRead.
//!   if the number is smaller than the ones in the list, the oldest datas
//!   will be returned, if the number is bigger than the ones in the list,
//!   no data is returned. rfu32NextElemToRead will be set to
//!   the number of the last returned data-set + 1.
//! \return
//!   Elements copied in case of success
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_NOT_INITIALIZED : Gyro interface not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGyroIf::s32Get3dGyroData
(
   //! (O) : storage for gyro-list
   sensor_fi_tcl_3dGyroData *prGyroData,
   //! (IO) : number of element that has to be read the next time
   tU32 &rfu32NextElemToRead,
   //! (I) : number of elements to be read
   tU32 u32ElemsToRead
)
{
   tS32 s32ElemsCopied = 0;
   sensor_fi_tcl_3dGyroData *prData = prGyroData;

   //Parameter check
   if ( ( NULL != prData ) && ( 0 != u32ElemsToRead ) )
   {
      //Consistence check
      if( oRingBuffer3d.bIsValid() && oCS.bLocked() )
      {
         tU32 u32Elems = oRingBuffer3d.u32GetList( prData, u32ElemsToRead, rfu32NextElemToRead );

         // elems lost? -> set status in first copied gyro-data
         if ( u32Elems != u32ElemsToRead )
         {
            if( prData->RStatus.enType != 
                    sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR )
            {
               prData->RStatus.enType = 
                  sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR;

               prData->SStatus.enType =
                     sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR;

              prData->TStatus.enType =
                     sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR;
            }
         }
         s32ElemsCopied = (tS32)u32Elems;
      }
      else
      {
         s32ElemsCopied = VDS_E_GYROIF_NOT_INITIALIZED;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GyroIf:s32Get3dGyroData:Not initialized" );
      }
   }
   else
   {
      s32ElemsCopied = VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GyroIf:s32Get3dGyroData:Invalid parameters" );
   }

   return s32ElemsCopied;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   check for subscribers and send message. the actual list of gyro-data is given,
//!   so we normally don't need to create a temporary storage for the answer-message.
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_UNKNOWN_ERROR :   Unknown error
//!   - \c VDS_E_POST_MESSAGE_FAILED :  Message posting failed
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
// 03.07.2009   | sak9kor (RBEI/ECF1)    | Modified to adapt the XML generated
//                                         FI
// 10.08.2009   | sak9kor (RBEI/ECF1)    | Modifications have been done to avoid the usage
//                                         of additional memory pool
//******************************************************************************

tS32 tclMsgGyroIf::s32CheckFor3dGyroSubscriber
(
   //! (I) : received list of gyro-data
   sensor_fi_tcl_3dGyroData *pr3dGyroData,
   //! (I) : num of gyro-data in list
   tU32 u32NumOfData
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( ( VDS_C_S32_STATE_NORMAL == s32InternalState ) ||
        ( VDS_C_S32_STATE_NORMAL_DIAG == s32InternalState ) )
   {

      if ( vds_goMasterLock.bEnter() )
      {
         tclSubscriberManager *poSubscriber =
               tclSubscriberManager::poGetSubscriberWithFId( 
                  OSAL_NULL,
                  CCA_C_U16_SRV_SENSOR_LOCATION,
                  VDS_C_U16_FKTID_GYRO3DDATAUPDATE );

         // while subscribers are available...
         while ( OSAL_NULL != poSubscriber )
         {
            // get num of data to send
            tU32 u32NextElemToRead = poSubscriber->u32GetNextElemToRead();
            tS32 s32TempErrorValue = VDS_E_NO_ERROR;
            tS32 s32ElemsCreated = 0;

            do
            {
               amt_tclServiceData *poServiceData = OSAL_NULL;

               if ( oCS.bEnter() )
               {
                  // get Gyro-message
                  s32ElemsCreated = s32Get3dGyroMessage(
                                       OSAL_NULL,
                                       poSubscriber,
                                       &poServiceData,
                                       pr3dGyroData,
                                       u32NumOfData,
                                       u32NextElemToRead,
                                       !poSubscriber->bGetInitMessageSent() );
                  oCS.vLeave();
               }
               else
               {
                  s32RetVal = VDS_E_GYROIF_UNKNOWN_ERROR;
               }

               if ( ( s32ElemsCreated > 0 )&&
                    ( OSAL_NULL != poServiceData )&&
                    ( OSAL_NULL != pfs32SendSubscriberMessage ) )
               {
                  // send message
                  if ( VDS_E_NO_ERROR 
                       == 
                       pfs32SendSubscriberMessage( poServiceData,poSubscriber ) )
                  {
                     poSubscriber->vSetInitMessageSent( TRUE );
                     // set next elem to read
                     poSubscriber->vSetNextElemToRead( u32NextElemToRead );
                  }
                  else
                  {
                     s32TempErrorValue = VDS_E_POST_MESSAGE_FAILED;
                  }

                  OSAL_DELETE poServiceData;
               }
               else if ( s32ElemsCreated < 0 )
               {
                  s32TempErrorValue = s32ElemsCreated;
               }
            }
            while ( ( s32ElemsCreated > 0 )&&
                    ( VDS_E_NO_ERROR == s32TempErrorValue ) );

            // if error occurs in loop and no error is set, take over temp
            // error-value
            if ( ( VDS_E_NO_ERROR == s32RetVal )&&
                 ( VDS_E_NO_ERROR != s32TempErrorValue ) )
            {
               s32RetVal = s32TempErrorValue;
            }
            // get next subscriber with this function-id
            poSubscriber = 
               tclSubscriberManager::poGetSubscriberWithFId( 
                  poSubscriber,
                  CCA_C_U16_SRV_SENSOR_LOCATION,
                  VDS_C_U16_FKTID_GYRO3DDATAUPDATE );

         } // end while (subscribers available)

         // leave critical section
         vds_goMasterLock.vLeave();
      } // end if lock subscriber successful

      // else no subscriber exists, return VDS_E_NO_ERROR
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   check for gyro temperature subscribers and send message. 
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_UNKNOWN_ERROR :   Unknown error
//!   - \c VDS_E_POST_MESSAGE_FAILED :  Message posting failed
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tU32 tclMsgGyroIf::u32CheckForGyroTemperature
(
   //! (I) : void
  tVoid
)
{
   tU32 u32NextTimestamp = 0;

   if ( ( VDS_C_S32_STATE_NORMAL == s32InternalState ) ||
        ( VDS_C_S32_STATE_NORMAL_DIAG == s32InternalState )
      )
   {
      //Get master lock
      if ( vds_goMasterLock.bEnter() )
      {
         tU32 u32NextTimeToRead;
         tU32 u32CurrentTimestamp;
         tS32 s32MessageCreated;
         tclSubscriberManager *poSubscriber =
               tclSubscriberManager::poGetSubscriberWithFId( 
                  OSAL_NULL,
                  CCA_C_U16_SRV_SENSOR_LOCATION,
                  VDS_C_U16_FKTID_GYROTEMPERATURE );

         //Get current time stamp
          u32CurrentTimestamp = (tU32)( OSAL_ClockGetElapsedTime() / 1000 );

         //While subscribers are available...
         while ( OSAL_NULL != poSubscriber )
         {
            amt_tclServiceData *poServiceData = OSAL_NULL;
            //Get time when next temperature has to be send
            u32NextTimeToRead = poSubscriber->u32GetNextElemToRead();
            //tS32 s32TempErrorValue = VDS_E_NO_ERROR;

            //Does temperature-msg has to be send?
            if( ( u32NextTimeToRead - u32CurrentTimestamp ) 
                > 
                0x7FFFFFFF  ) // <= 0 ?
            {
               s32MessageCreated = 0;
               if( oCS.bEnter() )
               {
                  //Get gyro-temperature-message
                  s32MessageCreated = 
                     s32GetGyroTemperatureMessage( OSAL_NULL,
                                                   poSubscriber,
                                                   &poServiceData );
                  oCS.vLeave();
               }

               if( ( s32MessageCreated > 0 ) &&
                   ( OSAL_NULL != poServiceData ) &&
                   ( OSAL_NULL != pfs32SendSubscriberMessage ) )
               {
                  //Send message
                  if( VDS_E_NO_ERROR 
                      == 
                      (*pfs32SendSubscriberMessage)(poServiceData, poSubscriber ) )
                  {
                                 poSubscriber->vSetInitMessageSent( TRUE );
                     //Set time when next temperature has to be sent
                     u32NextTimeToRead += poSubscriber->u32GetMsgSendInterval();
                     //Set next element to read.
                     poSubscriber->vSetNextElemToRead( u32NextTimeToRead );

                     //Get next timestamp to read temperature
                     if( ( u32NextTimestamp == 0 ) ||
                         ( (u32NextTimeToRead - u32CurrentTimestamp)
                             < (u32NextTimestamp - u32CurrentTimestamp) ) )
                     {
                        u32NextTimestamp = u32NextTimeToRead;
                     }
                  }
                  else
                  {
                     vTraceMsg( VDS_C_TRACELEVEL_MESSAGE,
                                "GyroIf:ChkGyroTemp:Message post failed" );
                  }

               }
               else if ( s32MessageCreated < 0 )
               {
                  //s32TempErrorValue = s32MessageCreated;
                  vTraceMsg( VDS_C_TRACELEVEL_MESSAGE, 
                             "GyroIf:ChkGyroTemp:Message create failed" );
               }                        
            }
            else // else if (send temperature-msg needed)
            {
               vTraceMsg( VDS_C_TRACELEVEL_MESSAGE,
                          "GyroIf:ChkGyroTemp:condition check false");
               //Get next time stamp to read temperature
               if ( ( u32NextTimestamp == 0 ) 
                    ||
                    ( (u32NextTimeToRead - u32CurrentTimestamp) 
                      <
                      (u32NextTimestamp - u32CurrentTimestamp) )  )
               {
                  u32NextTimestamp = u32NextTimeToRead;
               }

            }//End if (send temperature-msg needed)

            if( OSAL_NULL != poServiceData )
            {
               OSAL_DELETE poServiceData;
               poServiceData = OSAL_NULL;
            }

            //Get next subscriber with this function-id
            poSubscriber =
               tclSubscriberManager::poGetSubscriberWithFId(
                  poSubscriber,
                  CCA_C_U16_SRV_SENSOR_LOCATION,
                  VDS_C_U16_FKTID_GYROTEMPERATURE );

         }//end while (subscribers available)

         //Leave critical section
         vds_goMasterLock.vLeave();
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_MESSAGE,
                    "GyroIf:ChkGyroTemp:Unable to get master lock" );
      }
      // else no subscriber exists, return VDS_E_NO_ERROR
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_MESSAGE,"GyroIf:ChkGyroTemp:Invalid state" );
   }

   return u32NextTimestamp;
}


// tBool tclMsgGyroIf::bCritSectionBegin ()
// {
//    return oCS.bEnter();
// }

// tVoid tclMsgGyroIf::vCritSectionEnd ()
// {
//    oCS.vLeave();
// }

// tBool tclMsgGyroIf::bCritSectionLocked ()
// {
//    return oCS.bLocked();
// }


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   stores list of 3D gyro-data in ringbuffer, triggers check for
//!   subscribers (and send data)
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_SUBSCRIBER_ERROR : Subscriber error
//!   - \c VDS_E_GYROIF_NOT_INITIALIZED : Gyro interface not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGyroIf::s32Add3dDataWrapper
(
   //! (I) : list of gyro-data to be stored in ringbuffer
   const sensor_fi_tcl_3dGyroData *pr3dGyroData,
   //! (I) : num of datas in gyro-list
   tU32 u32NumOfData
)
{
   tS32 s32RetVal = VDS_E_GYROIF_NOT_INITIALIZED;

   if ( NULL != poThisMsgGyroIf )
   {
      s32RetVal = 
         poThisMsgGyroIf->s32Add3dGyroList( pr3dGyroData, u32NumOfData );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Prepares s32GetGyroTemperatureMessage-status-message.
//! \return
//!   Succes code in case of success is:
//!   - \c  OSAL_OK : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//!   - \c VDS_E_GYROIF_TEMPERATURE_UNAVAILABLE : Gyro temeperature not available
//!   - \c VDS_E_GYROIF_ALLOC_ERROR : Allocation error
//!   - \c VDS_E_GYROIF_NOT_INITIALIZED : Gyro interface not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGyroIf::s32GetGyroTemperatureMessage
(
   //!  (I) : pointer to income message
   amt_tclServiceData const *poRequestMessage,
   //!  (I) : pointer to Subscriber data
   tclSubscriberManager *poSubscriber,
   //!  (O) : pointer to a location where a pointer to the reply message is available
   amt_tclServiceData **ppoResponseMessage
)
{
   tS32 s32RetVal = OSAL_OK;
   tU16 u16RegisterID = 0;
   tU16 u16CmdCounter = 0;
   tU16 u16TargetAppID = 0;
   tU16 u16ServiceID = 0;
   tF32 f32CurrentTemperature = 0.0;

   if ( OSAL_NULL != poRequestMessage )
   {
      u16TargetAppID = poRequestMessage->u16GetSourceAppID();
      u16RegisterID = poRequestMessage->u16GetRegisterID();
      u16CmdCounter = poRequestMessage->u16GetCmdCounter();
      u16ServiceID = poRequestMessage->u16GetServiceID();
   }
   else if ( OSAL_NULL != poSubscriber )
   {
      u16TargetAppID = poSubscriber->u16GetTargetAppId();
      u16RegisterID = poSubscriber->u16GetRegisterId();
      u16CmdCounter = poSubscriber->u16GetCommandCntr();
      u16ServiceID = poSubscriber->u16GetServiceId();
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   //Parameter check
   if ( ( OSAL_OK == s32RetVal ) &&
        ( OSAL_NULL != ppoResponseMessage ) &&
        ( OSAL_NULL != poGyroThread ) )
   {
      *ppoResponseMessage = OSAL_NULL;

      if ( oCS.bLocked() )
      {
         u32NextTimeToSend = ( OSAL_ClockGetElapsedTime() / 1000 + 1);

         f32CurrentTemperature = poGyroThread->f32GetGyroTemperature();

         //Temperature available?
         if ( f32CurrentTemperature > 1.0 )
         {
            //Create the object for XML generated FI for Set OP type
            sensor_locationfi_tclMsgGyroTemperatureStatus oGyroTemperatureStatus;

            //Insert the data in to the created class object
            oGyroTemperatureStatus.GyroTemperature = f32CurrentTemperature;

            //Create the object of visitor message and insert the class object
            fi_tclVisitorMessage* poResultMessage =
                        OSAL_NEW  fi_tclVisitorMessage( oGyroTemperatureStatus, VDS_C_U16_SRV_SENSORS_MAJOR_VERSION );

            if ( OSAL_NULL != poResultMessage )
            {
               tU32 u32TimeStamp = 0;
               vGetTimeStamp( &u32TimeStamp );

               if ( poResultMessage->bIsValid() )
               {
                  //Create answer message
                  poResultMessage->vInitServiceData(  
                     CCA_C_U16_APP_SENSOR,                // Source
                     u16TargetAppID,                      // Target
                     AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,   // StreamType
                     0,                                   // StreamCounter
                     u16RegisterID,                       // RegisterID
                     u16CmdCounter,                       // nCmdCounter
                     u16ServiceID,                        // nServiceID
                     SENSOR_LOCATIONFI_C_U16_GYROTEMPERATURE,  // nFunctionID
                     AMT_C_U8_CCAMSG_OPCODE_STATUS,       // OpCode
                     0,                                   // Asynchronous Completion Token (ACT)
                     AMT_C_U16_DEFAULT_NULL,              // Source Sub
                     AMT_C_U16_DEFAULT_NULL,              // Target Sub
                     u32TimeStamp                         // TimeStamp
                     );

                  *ppoResponseMessage = 
                     poResultMessage; /*static_cast<amt_tclServiceData *>(poResultMessage);*/
                  s32RetVal = 1;
               }
               else
               {
                  s32RetVal = VDS_E_GYROIF_ALLOC_ERROR;
               }

               if ( s32RetVal < 1 )
               {
                  OSAL_DELETE poResultMessage;
                  poResultMessage = OSAL_NULL;
               }
            }
            else
            {
               s32RetVal = VDS_E_GYROIF_ALLOC_ERROR;
            }

            VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(poResultMessage);
         } // end if temperature available
         else
         {
            //No elem to send
            s32RetVal = VDS_E_GYROIF_TEMPERATURE_UNAVAILABLE;
            vTraceMsg( VDS_C_TRACELEVEL_MESSAGE, 
                       "GyroIf:Unable to get temperature" );
         }
      } // end if bCritSectionLocked
      else
      {
         s32RetVal = VDS_E_GYROIF_NOT_INITIALIZED;
         vTraceMsg( VDS_C_TRACELEVEL_MESSAGE, 
                    "GyroIf:GyroTemp:Lock not initialized");
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_MESSAGE, 
                 "GyroIf:GyroTemp:Invalid parameters" );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Set Application state
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tVoid tclMsgGyroIf::vSetAppState
(
   //! (I) : Application state to be set
   tU32 u32AppState
)
{
   (void)u32AppState;
   //if ( NULL != poGyroThread )
   //{
      // at the moment the gyro driver is not dependent of the application state
      // poGyroThread->vSetAppState( u32AppState );
   //}
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//!  \brief
//!   Gets the index of last element in the buffer.
//!  \return
//!    The index of last element in the buffer.
//!  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//****************************************************************************

tU32 tclMsgGyroIf::u32GetLastGyroElemIndex(tVoid)
{
   return( oRingBuffer3d.u32GetNumElems() - 1);
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Collects Gyro records for "AllSensorData" property and updates in 
//   sensor_location_fi object. 
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROIF_NOT_INITIALIZED :if vdsensor is not initialized.
//!   - \c VDS_E_GYROIF_ALLOC_ERROR :if failed to allocate memory.
//!   - \c VDS_E_GYROIF_GETGYRODATA_FAILED: if unable to get data.
//!   - \c VDS_E_GYROIF_DATA_UNAVAILABLE:if no Gyro data is available. 
//  HISTORY:
//  Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//***************************************************************************** 

tS32 tclMsgGyroIf::s32GetGyroRecords
(
   sensor_fi_tcl_AllSensorData *pAllSensorData,
   
   tU32 &u32NextelementToRead
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
   sensor_fi_tcl_3dGyroData *parTemp3dGyroData = OSAL_NULL;

   if( ( poGyroThread != OSAL_NULL ) && oCS.bLocked() )
   {
      //Get GYRO Temperature
      pAllSensorData->GyroTemperature = poGyroThread->f32GetGyroTemperature();

      //Get GYRO Data
      tU32 u32NumOfElemToSend = 
         oRingBuffer3d.u32GetElemsFrom( u32NextelementToRead );

      //Elements available?
      if ( u32NumOfElemToSend > 0 )
      {

         parTemp3dGyroData = 
            OSAL_NEW sensor_fi_tcl_3dGyroData[u32NumOfElemToSend];

         if( OSAL_NULL != parTemp3dGyroData )
         {
            if( (tS32)u32NumOfElemToSend 
                != 
                s32Get3dGyroData( parTemp3dGyroData,
                                  u32NextelementToRead,
                                  u32NumOfElemToSend ) )
            {
               s32RetVal = VDS_E_GYROIF_GETGYRODATA_FAILED;
               vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                          "GyroIf:GyroAllSen:Couldn't Get Requested (%d) Gyro data",
                          u32NumOfElemToSend );
            }
         }
         else
         {
            s32RetVal = VDS_E_GYROIF_ALLOC_ERROR;
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "GyroIf:GyroAllSen:Allocation Error" );
         }

         //Are there any elements to send?
         if( ( OSAL_NULL != parTemp3dGyroData ) 
             && 
             ( s32RetVal == VDS_E_NO_ERROR ) )
         {
            for ( tU32 u32Loop = 0; u32Loop < u32NumOfElemToSend; u32Loop++ )
            {
               //Copy values from array to list
               pAllSensorData->GyroUpdate.push_back(parTemp3dGyroData[u32Loop]);
            }

            tU32 u32TS1 = 
               parTemp3dGyroData[0].TimeStamp3dGyro;

            tU32 u32TS2 = 
               parTemp3dGyroData[u32NumOfElemToSend-1].TimeStamp3dGyro;

            vTraceMsg( VDS_C_TRACELEVEL_EVENT,
                       "GyroIf:GyroAllSen:3D-Gyro message ts = %d--%d",
                       u32TS1, 
                       u32TS2 );
         }

         //(if exists) delete temp buffer
         if ( parTemp3dGyroData != OSAL_NULL)
         {
            //Delete temp.buffer
            OSAL_DELETE [] parTemp3dGyroData;
            parTemp3dGyroData = OSAL_NULL;
         }
      }
      else
      {
         //No elements to send
         s32RetVal = VDS_E_GYROIF_DATA_UNAVAILABLE;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GyroIf:GyroAllSen:No Gyro data available");
      }
   }
   else
   {
      s32RetVal = VDS_E_GYROIF_NOT_INITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "GyroIf:GyroAllSen:Getting Lock failed");
   }

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTemp3dGyroData);

   return s32RetVal;
}


// ***************** F U N C T I O N  H E A D E R *****************************
//! \brief
//!       Start Gyro self test.
//! \return
//!   - \c TRUE  : Self test finished successfully.
//!   - \c FALSE : Unable to start self test.
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclMsgGyroIf::bStartSelftest
(
   tS32 *ps32SelfTestResult
)
{
   tBool bRetVal = FALSE;

   if( NULL != poGyroThread )
   {
      //Start self test.
      bRetVal = poGyroThread->bStartSelftest( ps32SelfTestResult );
   }

   return bRetVal;
}
//EOF
