// *****************************************************************************
// * FILE:         tclMsgGpsIf.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition: handles access to gps-data,
// *               handles subscriber-messages
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
// * 18.11.02 CM-CR/EES4-Bode
// *          Adaption to Tuareg.
//  * 14.08.09 -  sak9kor - AntennaCurrent related function
//                                  have been modified to adapt the XML generated FI
//  * 20.07.09  -  sak9kor - APIS have been modified to adapt the XML generated FI
//  * 11.08.09 - sak9kor - Usage of additional memory pool has been removed and corresponding
//                                 modifications have been done
//  * 28.08.09 RBEI/ECF1 - sak9kor - Trace Output related modifications bave been done
//  * 28.08.09 RBEI/ECF1 - sak9kor - Trace Output related modifications bave been done
//  * 08.09.09 RBEI/ECF1 - sak9kor - Trace levels have been modified
//  * 30.09.09 RBEI/ECF1 - sak9kor - Added doxygen headers for all classes/funcs
//  * 16.12.10 RBEI/ECF1 - sak9kor - Diaglog related modifications have been done
//  * 04.05.11 RBEI CM-AI/PJ-CF31 - Sainath Changes done to fix MMS 294524 
//  * 09.11.12 RBEI/ECF5 - sak9kor-  Fix for the issue NIKAI-231 is added 
//  * 19.09.13 RBEI/ECF5 - sga5kor - Modifications done to adopt VDSensor according to
//                                   the CmbFi version-5.0.0
// *  05.11.15 RBEI/ECF5 - rmm1kor- Added support for Diagnosis satellite system control
//  * 10.11.15 RBEI/ECF5 - amg6kor - Adding Get operation for GnssEpoch Property
//  * 06.01.17 RBEI/ECF12 - amg6kor - Port to Gen4
//*****************************************************************************

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

// for Parameter in s32Init
#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_ERRORCODES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_FUNCTIONIDS
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_SERVICEINFO
#include "sensor_fi_if.h"
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "fi_msgfw_if.h"
#define SENSOR_S_IMPORT_INTERFACE_SENSOR_MESSAGE
#define SENSOR_S_IMPORT_INTERFACE_SERVER
#ifdef SENSOR_S_IMPORT_INTERFACE_SENSOR_MESSAGE
#endif 

#include "vds2_if.h"


#define VDS_S_IMPORT_INTERFACE_SUBSCRIBER_MANAGER
#define VDS_S_IMPORT_INTERFACE_MESSAGE_INTERFACES
#define VDS_S_IMPORT_INTERFACE_PERSISTENT_DATA
#define VDS_S_IMPORT_INTERFACE_TRACE
#include "vds_internal_if.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

extern void vGetTimeStamp( tPU32 pu32TimeStamp );
extern tS32 s32GetAllSensorDataMsg
(
   //! (I) : Pointer to Service data
   amt_tclServiceData const *poServiceData,
   //! (I) : Pointer to Subscriber data
   tclSubscriberManager *poSubscriber,
   //! (O) : pointer to a location where a pointer to the reply message is available
   amt_tclServiceData **ppoResponseMessage,
   //! (I) :Next elements to read
   Vds_AllSenDataNextElemToSend &rNextElmToRead
);

/*!****************************************************************************
 *! \brief defines the interface for AllSensorData synch listener.
 *!***************************************************************************/
class vds_tclAllSensorDataSynchListener : public vds_tclSynchListenerIf
{
private:
   tclMsgGpsIf *poGpsIf;
public:
   //! ***************** F U N C T I O N  H E A D E R **********************
   //! \brief  : default constructor
   //! \param  : poGpsIfArg - pointer to Gps interface argument
   //! \return : void
   //!**********************************************************************
   vds_tclAllSensorDataSynchListener( tclMsgGpsIf *poGpsIfArg )
   {
      poGpsIf = poGpsIfArg;
   }

   //!***************** F U N C T I O N  H E A D E R ***********************
   //! \brief  : performs the synch operation of AllSensorData property
   //! \param  : void
   //! \return : void
   //!**********************************************************************
   virtual void vSynch( void )
   {
      vTraceMsg( VDS_C_TRACELEVEL_EVENT, "Synch (AllSensorData)" );
      if ( poGpsIf != OSAL_NULL )
      {
         poGpsIf->vSynch( );
      }
   }
};

// Class tclMsgGpsIf

tclMsgGpsIf *tclMsgGpsIf::poThisMsgGpsIf;
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Default Constructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclMsgGpsIf::tclMsgGpsIf()
      : poGpsThread( OSAL_NULL ),
        hGpsIfSemaphore( OSAL_C_INVALID_HANDLE ),
        coszGpsIfSemName( "VDS_GPS_IF_SEM" ),
        bCritSectionIsLocked( FALSE ),
        s32InternalState( VDS_C_S32_STATE_UNINITIALIZED )
{
   pfs32SendSubscriberMessage = NULL;
   bCritSectionIsLocked = FALSE;
   poThisMsgGpsIf = this;
   vds_goSensorPropertySynchManager.bAddListener( new vds_tclAllSensorDataSynchListener( this ) );
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Default Destructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclMsgGpsIf::~tclMsgGpsIf()
{

   oGnssDataBuffer.vDeallocate();

   // delete semaphore
   if ( OSAL_C_INVALID_HANDLE != hGpsIfSemaphore )
   {
      OSAL_s32SemaphoreClose( hGpsIfSemaphore );
      OSAL_s32SemaphoreDelete( coszGpsIfSemName );
      hGpsIfSemaphore = OSAL_C_INVALID_HANDLE;
   }

   // delete gps-thread
   if ( NULL != poGpsThread )
   {
      poGpsThread->s32ThreadStop();
      delete poGpsThread;
      poGpsThread = NULL;
   }

   poThisMsgGpsIf = NULL;
   coszGpsIfSemName = NULL;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   initializes gps-interface.
//!   create and initialize instance of gps-thread, create semaphore, create ringbuffer
//!  for gps-big-data, initialize member-attributes
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_INVALID_PARAMETER :   Invalid Parameter
//!   - \c VDS_E_GPSIF_ALLOC_ERROR :  GPS interface allocation error
//!   - \c VDS_E_GPSIF_SEM_ERROR : GPS interface Semaphore error
//!   - \c VDS_E_GPSIF_THREADINIT_ERROR : GPS interface thread init error
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGpsIf::s32Init
(
   //! (I) : pointer to function, which sends given answer-message with given parameters to subscribers
   tS32( *pfs32SendSubscriberMessageParam )( amt_tclServiceData *, tclSubscriberManager * )
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( NULL != pfs32SendSubscriberMessageParam )
   {

      this->pfs32SendSubscriberMessage = pfs32SendSubscriberMessageParam;

      //Create tclGpsThread instance.
      poGpsThread = new tclGpsThread();

      if ( NULL != poGpsThread )
      {
         if ( VDS_E_NO_ERROR 
              == 
              poGpsThread->s32ThreadInit( tclMsgGpsIf::s32AddGnssDataWrapper ))
         {
            // create semaphore
            if( OSAL_ERROR != OSAL_s32SemaphoreCreate( coszGpsIfSemName, &hGpsIfSemaphore, 1 ) )
            {
               bCritSectionIsLocked = FALSE;
               //Create a buffer to store GNSS data.
               oGnssDataBuffer.vAllocate(VDS_C_U32_GPSIF_RINGBUFFERSIZE);
               if ( !oGnssDataBuffer.bIsValid() )
               {
                  s32RetVal = VDS_E_GPSIF_ALLOC_ERROR;
                  vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                             "GpsIf:Buffer allocation failed" );
               }
            }
            else
            {
               s32RetVal = VDS_E_GPSIF_SEM_ERROR;
               vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                          "GpsIf::s32Init:Creating Semaphore failed");
            }

            if ( VDS_E_NO_ERROR != s32RetVal )
            {
               poGpsThread->vThreadClose();
            }
         }
         else
         {
            s32RetVal = VDS_E_GPSIF_THREADINIT_ERROR;
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                       "GpsIf::s32Init:thread init failed");
         }

         if ( VDS_E_NO_ERROR != s32RetVal )
         {
            delete poGpsThread;
            poGpsThread = NULL;
         }
      }
      else
      {
         s32RetVal = VDS_E_GPSIF_ALLOC_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "GpsIf::s32Init:Creating tclGpsThread instance failed");
      }
   }
   else
   {
      s32RetVal =  VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GpsIf:s32Init:Invalid argument" );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  start gps-thread
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GPSIF_THREADSTART_ERROR :   GPS interface thread start error
//!   - \c VDS_E_GPSIF_NOT_INITIALIZED :  GPS interface not initialized
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGpsIf::s32StartThread
(
   //! None
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   // start gps-thread
   if ( NULL != poGpsThread )
   {
      if ( VDS_C_S32_THREAD_RUNNING != poGpsThread->s32ThreadStart() )
      {
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GpsIf:Couldn't start GpsThread" );
         s32RetVal = VDS_E_GPSIF_THREADSTART_ERROR;
      }
   }
   else
   {
      s32RetVal = VDS_E_GPSIF_NOT_INITIALIZED;
   }

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  stop gps-thread
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GPSIF_THREADSTOP_ERROR :   GPS interface thread stop error
//!   - \c VDS_E_GPSIF_NOT_INITIALIZED :  GPS interface not initialized
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGpsIf::s32StopThread
(
   //! None
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,"GpsIf:s32StopThread()..." );

   // stop gps-thread
   if ( NULL != poGpsThread )
   {
      if ( VDS_C_S32_THREAD_INITIALIZED != poGpsThread->s32ThreadStop() )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"s32StopThread() GpsThreadStop failed (state=%d)", poGpsThread->s32GetInternalState());
         s32RetVal = VDS_E_GPSIF_THREADSTOP_ERROR;
      }
   }
   else
   {
      s32RetVal = VDS_E_GPSIF_NOT_INITIALIZED;
   }

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   delete gps-thread
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GPSIF_THREADSTOP_ERROR :   GPS interface thread stop error
//!   - \c VDS_E_GPSIF_NOT_INITIALIZED :  GPS interface not initialized
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGpsIf::s32DeleteThread
(
   //! None
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   // delete gps-thread
   if ( NULL != poGpsThread )
   {
      if ( VDS_C_S32_THREAD_INITIALIZED != poGpsThread->s32ThreadStop() )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "s32DeleteThread() gps->s32ThreadStop failed (state=%d)",
          poGpsThread->s32GetInternalState());
          
         s32RetVal = VDS_E_GPSIF_THREADSTOP_ERROR;
      }

      delete poGpsThread;
      poGpsThread = NULL;
   }
   else
   {
      s32RetVal = VDS_E_GPSIF_NOT_INITIALIZED;
   }

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   set internal state  (normal or pending)
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclMsgGpsIf::vSetState
(
   //! (I) : New state to be set
   //!  i.e. VDS_C_S32_STATE_NORMAL or VDS_C_S32_STATE_NORMAL_DIAG
   //! or VDS_C_S32_STATE_PENDING
   tS32 s32NewState
)
{
   if (  ( VDS_C_S32_STATE_NORMAL == s32NewState ) ||
         ( VDS_C_S32_STATE_NORMAL_DIAG == s32NewState )
      )
   {
      if ( VDS_C_S32_STATE_PENDING == s32InternalState )
      {
         // Reset NextElemToRead for all GPS subscribers.
         if ( vds_goMasterLock.bEnter() )
         {
            tclSubscriberManager *poSubscriber = NULL;
            poSubscriber = tclSubscriberManager::poGetSubscriberWithFId( NULL,
                                                                         CCA_C_U16_SRV_SENSOR_LOCATION,
                                                                         VDS_C_U16_FKTID_GNSSDATA );

            //while subscribers are available...
            while ( NULL != poSubscriber )
            {
               //set next elem to read to 0.
               poSubscriber->vSetNextElemToRead( 0 );
               //Get next subscriber.
               poSubscriber = tclSubscriberManager::poGetSubscriberWithFId( poSubscriber,
                                                                            CCA_C_U16_SRV_SENSOR_LOCATION,
                                                                            VDS_C_U16_FKTID_GNSSDATA );
            }

            //Unlock subscriber
            vds_goMasterLock.vLeave();
         }

         s32InternalState = s32NewState;
      }
      else
      {
         s32InternalState = s32NewState;
      }
   }

   if ( VDS_C_S32_STATE_PENDING == s32NewState )
   {
      s32InternalState = VDS_C_S32_STATE_PENDING;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   stores gnss data in ringbuffer, triggers check for subscribers
//!  (and send data)
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//!   - \c VDS_E_GPSIF_SUBSCRIBER_ERROR : GPS interface subscriber error
//!   - \c VDS_E_GPSIF_NOT_INITIALIZED :  GPS interface not initialized
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tS32 tclMsgGpsIf::s32AddGnssDataToBuffer( const OSAL_trGnssFullRecord *prGnssFullRecord )
{
   tS32 s32RetVal=VDS_E_NO_ERROR;
   
   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
              "GpsIf:Adding GNSS Data to buffer");
   // consistence check
   if (  NULL != prGnssFullRecord )
   {
      if(bCritSectionBegin())
      {
         oGnssDataBuffer.u32AddOne(*prGnssFullRecord);
         vCritSectionEnd();
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GpsIf:s32AddGnssDataToBuffer:Not Initialized");
         s32RetVal = VDS_E_GPSIF_NOT_INITIALIZED;
      }

      // This should cause the gyro and odometer message interface
      // classes to send off their buffered data.
      vds_goSensorPropertySynchManager.vSynch( );
   
      if(VDS_E_NO_ERROR != s32CheckForGNSSdataSubscribers())
      {
         s32RetVal = VDS_E_GPSIF_SUBSCRIBER_ERROR;
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "GpsIf:s32AddGnssDataToBuffer:Invalid parameter" );
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   check for AllSensorData subscribers and send message.
//!
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GPSIF_UNKNOWN_ERROR : GPS interface unknown error
//!   - \c VDS_E_POST_MESSAGE_FAILED : Message post failed
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGpsIf::s32CheckForAllSensorDataSubscribers()
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( (VDS_C_S32_STATE_NORMAL == s32InternalState) ||
        (VDS_C_S32_STATE_NORMAL_DIAG == s32InternalState)
      )
   {
      //Take Master Lock.
      if( vds_goMasterLock.bEnter() )
      {
         tclSubscriberManager *poSubscriber  
                = tclSubscriberManager::poGetSubscriberWithFId( OSAL_NULL,
                                                                CCA_C_U16_SRV_SENSOR_LOCATION,
                                                                VDS_C_U16_FKTID_ALLSENSORDATA );
         //while subscribers are available...
         while ( OSAL_NULL != poSubscriber )
         {
            tS32 s32TempErrorValue = VDS_E_NO_ERROR;
            Vds_AllSenDataNextElemToSend rNextElemToSend = {0,0,0,0,0};
            tS32 s32RVal = 0;

            poSubscriber->vGetAllSen_NxtElmToRead(rNextElemToSend);

            do
            {
               amt_tclServiceData *ppoResponseMessage = OSAL_NULL;

               s32RVal = s32GetAllSensorDataMsg( OSAL_NULL,poSubscriber,&ppoResponseMessage,rNextElemToSend);

               if( ( s32RVal == VDS_E_NO_ERROR ) &&
                   ( OSAL_NULL != ppoResponseMessage ) &&
                   ( OSAL_NULL != pfs32SendSubscriberMessage ) )
               {
                  //Send message to subscriber.
                  if( VDS_E_NO_ERROR == (*pfs32SendSubscriberMessage )( ppoResponseMessage,poSubscriber ) )
                  {
                     poSubscriber->vSetInitMessageSent( TRUE );
                     poSubscriber->vSetAllSen_NxtElmToRead(rNextElemToSend);
                  }
                  else
                  {
                     s32TempErrorValue = VDS_E_POST_MESSAGE_FAILED;
                  }
               }
               else if ( s32RVal != VDS_E_NO_ERROR )
               {
                  s32TempErrorValue = s32RVal;
               }

               if( OSAL_NULL != ppoResponseMessage )
               {
                  OSAL_DELETE ppoResponseMessage;
               }

            } while( ( s32RVal != VDS_E_NO_ERROR ) &&
                        ( VDS_E_NO_ERROR == s32TempErrorValue ) );

            //If error occurs in loop and no error is set, take over temp error-value
            if( ( VDS_E_NO_ERROR == s32RetVal ) &&
                    ( VDS_E_NO_ERROR != s32TempErrorValue ) )
            {
               s32RetVal = s32TempErrorValue;
            }

            //Get next subscriber.
            poSubscriber = tclSubscriberManager::poGetSubscriberWithFId( poSubscriber,
                                                                         CCA_C_U16_SRV_SENSOR_LOCATION,
                                                                         VDS_C_U16_FKTID_ALLSENSORDATA );
         }

         //Unlock master lock.
         vds_goMasterLock.vLeave();
      }
   }

   return s32RetVal;
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   check for Gnssdata subscribers and send message.
//!
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GPSIF_UNKNOWN_ERROR : GPS interface unknown error
//!   - \c VDS_E_POST_MESSAGE_FAILED : Message post failed
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tS32 tclMsgGpsIf::s32CheckForGNSSdataSubscribers
(
   //! None.
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
 
   if ( ( VDS_C_S32_STATE_NORMAL == s32InternalState ) 
    ||
        ( VDS_C_S32_STATE_NORMAL_DIAG == s32InternalState ) )
   {

      //Take Master Lock.
      if( vds_goMasterLock.bEnter() )
      {
         tclSubscriberManager *poSubscriber = 
        tclSubscriberManager::poGetSubscriberWithFId( OSAL_NULL,
                              CCA_C_U16_SRV_SENSOR_LOCATION,
                              VDS_C_U16_FKTID_GNSSDATA );
         //While subscribers are available.
         while ( OSAL_NULL != poSubscriber )
         {
            tU32 u32NextElemToRead = poSubscriber->u32GetNextElemToRead();
            tS32 s32TempErrorValue = VDS_E_NO_ERROR;
            tS32 s32ElemsCreated = 0;

            do
            {
               amt_tclServiceData *ppoResponseMessage = OSAL_NULL;

               if( bCritSectionBegin() )
               {
                  //Get GNSS message
                  s32ElemsCreated = s32GetGnssDataMsg( OSAL_NULL,
                                                       poSubscriber,
                                                       &ppoResponseMessage,
                                                       u32NextElemToRead );
                  vCritSectionEnd();
               }
               else
               {
                  s32RetVal = VDS_E_GPSIF_UNKNOWN_ERROR;
               }

               if ( ( s32ElemsCreated > 0 ) &&
                    ( OSAL_NULL != ppoResponseMessage ) &&
                    ( OSAL_NULL != pfs32SendSubscriberMessage ) )
               {
                  //Send message to subscriber.
                  if ( VDS_E_NO_ERROR == ( *pfs32SendSubscriberMessage )(ppoResponseMessage, poSubscriber ) )
                  {
                     poSubscriber->vSetInitMessageSent( TRUE );
                     poSubscriber->vSetNextElemToRead( u32NextElemToRead );
                  }
                  else
                  {
                     s32TempErrorValue = VDS_E_POST_MESSAGE_FAILED;
                  }

                  OSAL_DELETE ppoResponseMessage;
               }
               else if ( s32ElemsCreated < 0 )
               {
                  s32TempErrorValue = s32ElemsCreated;
               }

            }while( ( s32ElemsCreated > 0 ) &&
                    ( VDS_E_NO_ERROR == s32TempErrorValue ) );

            //If error occurs in loop and no error is set,take over temp error-value.
            if( ( VDS_E_NO_ERROR == s32RetVal ) &&
                 ( VDS_E_NO_ERROR != s32TempErrorValue ) )
            {
               s32RetVal = s32TempErrorValue;
            }
            //Get next subscriber.
            poSubscriber = tclSubscriberManager::poGetSubscriberWithFId( poSubscriber,
                                                                         CCA_C_U16_SRV_SENSOR_LOCATION,
                                                                         VDS_C_U16_FKTID_GNSSDATA );
         }

         //Unlock master lock
         vds_goMasterLock.vLeave();
      }
   }

   return s32RetVal;
}

//!***************** F U N C T I O N  H E A D E R *****************************
//! \brief  performs the synch operation of AllSensorData property
//! \param  void
//! \return void
//!****************************************************************************
void tclMsgGpsIf::vSynch( void )
{
   s32CheckForAllSensorDataSubscribers();
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   locks access to critical sections.only when TRUE is returned,
//!   access is allowed
//! \return
//!   - \c  TRUE : access allowed\n
//!   - \c FALSE : error, access is not allowed\n
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclMsgGpsIf::bCritSectionBegin
(
   //! None
)
{
   tBool bLocked = FALSE;

   if ( OSAL_C_INVALID_HANDLE != hGpsIfSemaphore )
   {
      if ( OSAL_ERROR != OSAL_s32SemaphoreWait( hGpsIfSemaphore, OSAL_C_U32_INFINITE ) )
      {
         bCritSectionIsLocked = TRUE;
         bLocked = TRUE;
      }
   }

   return bLocked;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   unlocks access to critical sections.
//! \return
//!   None
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclMsgGpsIf::vCritSectionEnd
(
   //! None
)
{
   if ( OSAL_C_INVALID_HANDLE != hGpsIfSemaphore )
   {
      bCritSectionIsLocked = FALSE;
      OSAL_s32SemaphorePost( hGpsIfSemaphore );
   }
   else
   {
      bCritSectionIsLocked = FALSE;
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   does caller locked GpsIf?
//!   only when TRUE is returned, access is alloweds
//! \return
//!   - \c  TRUE : access allowed\n
//!   - \c FALSE : error, access is not allowed\n
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclMsgGpsIf::bCritSectionLocked
(
   //! None
)
{
   return bCritSectionIsLocked;
}

// *****************************************************************************
// * METHOD:      tclMsgGpsIf::s32AddGnssDataWrapper
// * DESCRIPTION: wrapper function which calls actual function which
// *              stores gnss data in ringbuffer.
// *              
// * PARAMETER:
// *              OSAL_trGnssFullRecord* rGnssFullRecord: (->I)
// *                 gnss data to be stored in ringbuffer
// *
// * RETURNVALUE: tS32
// *                 ok   : VDS_E_NO_ERROR
// *                 error: VDS_E_INVALID_PARAMETER,
// *                        VDS_E_GPSIF_NOT_INITIALIZED
// *
// * HISTORY:
// ***************************************************************************** 

tS32 tclMsgGpsIf::s32AddGnssDataWrapper
(
   OSAL_trGnssFullRecord *rGnssFullRecord
)
{
   tS32 s32RetVal = VDS_E_GPSIF_NOT_INITIALIZED;

   if ( NULL != poThisMsgGpsIf )
   {
      s32RetVal = poThisMsgGpsIf->s32AddGnssDataToBuffer( rGnssFullRecord );
   }
   
   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  returns current time of day
//! \return
//!   - \c  TRUE : In case of Success\n
//!   - \c  FALSE : In case of failure\n
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclMsgGpsIf::bGetTimeOfDay
(
   //! (O) : Time of day
   tU16 &rfu16TimeOfDay
) const
{
   tBool bRetVal = FALSE;

   if ( NULL != poGpsThread )
   {
      // get time of day
      bRetVal = poGpsThread->bGetTimeOfDay ( rfu16TimeOfDay );
   }

   return bRetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  set time of day
//! \return
//!   - \c  TRUE :  time of day changed\n
//!   - \c  FALSE : error while setting time of day\n
//!                 or feature not available for this receiver
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclMsgGpsIf::bSetTimeOfDay
(
//! (I) : Time of day
   const tU16 &rfu16TimeOfDay
)
{
   tBool bRetVal = FALSE;

   if ( NULL != poGpsThread )
   {
      // set time of day
      bRetVal = poGpsThread->bSetTimeOfDay ( rfu16TimeOfDay );
   }

   return bRetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Stop /dev/gps
//! \return
//!   - \c  TRUE :  if stopped successfully or already stopped
//!   - \c  FALSE : in case of error
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclMsgGpsIf::bStopDevGps
(
   //! None
   tVoid
)
{
   tBool bRetVal = FALSE;

   if ( NULL != poGpsThread )
   {
      bRetVal = poGpsThread->bStopDevGnss();
   }

   return bRetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Send Evnt to GPS
//! \return
//!   None
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclMsgGpsIf::vSetGpsThreadEvent
(
   //! Event :
   //! - \c  VDS_GPS_EVENT_NOEVENT
   //! - \c  VDS_GPS_EVENT_RESTART
   //! - \c  VDS_GPS_EVENT_START
   //! - \c  VDS_GPS_EVENT_STOP
   //! - \c  VDS_GPS_EVENT_SEND_OSAL_TIME_UPDATE
   tU8 u8Event
)
{
   if ( NULL != poGpsThread )
   {
      poGpsThread->vSetGpsThreadEvent( u8Event );
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Sets App state
//! \return
//!   None
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclMsgGpsIf::vSetAppState
(
   //! (I) : Appstate to be set
   tU32 u32AppState
)
{
   if ( NULL != poGpsThread )
   {
      poGpsThread->vSetAppState( u32AppState );
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Static wrapper for trace commands.
//! \return
//!   None
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgGpsIf::bExecuteGpsTraceCommand
(
   //! (I) : puchData trace input buffer.
   const tUChar* puchData
)
{
   if ( NULL != poGpsThread )
   {
      return poGpsThread->bExecuteGpsTraceCommand( puchData );
   }
   else
   {
      return OSAL_ERROR;
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Reset factory settings
//!  All persistently saved sensor data should be deleted and
//!  replaced with standard values.  A GPS cold start is performed.
//! \return
//!   None
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void tclMsgGpsIf::vFactorySettings
(
   //! None
)
{
   if ( NULL != poGpsThread )
   {
      poGpsThread->vSetGpsThreadEvent( VDS_GPS_EVENT_FACTORY_SETTINGS );
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Static wrapper for enable/disable the china encoder.
//! \return
//!   None
//  HISTORY:
// Date        |  Author             | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void tclMsgGpsIf::vEnDisChinaEnc
(
//! (I) : bInhibit
   tBool bInhibit
)
{
   if ( NULL != poGpsThread )
   {
      poGpsThread->bEnDisChinaEnc( bInhibit );
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief get the GNSS satellite system used
//!   
//! \return value of type \c sensor_fi_tcl_b8_GnssSatSys. Method returns the actual
//  used satellite systems. If no satellite system could be determined it retruns 
//  0.
//******************************************************************************
tU8 tclMsgGpsIf::u8GetGnssSatSys ( tVoid )
{
   tU8 u8RetVal = 0;

   if( poGpsThread != OSAL_NULL )
   {
      u8RetVal = poGpsThread->u8GetGnssSatSys ();
   }

   return u8RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief set the GNSS satellite system used persistently
//!   
//! \return value of type \c sensor_fi_tcl_b8_GnssSatSys. Method returns the actual
//  used satellite systems. If no satellite system could be determined it retruns 
//  0.
// ******************************************************************************
tU8 tclMsgGpsIf::u8SetGnssSatSys ( tU8 u8TargetGnssSatSys )
{
   tU8 u8RetVal = 0;

   if( poGpsThread != OSAL_NULL )
   {
      u8RetVal = poGpsThread->u8SetGnssSatSys ( u8TargetGnssSatSys ) ;
   }

   return u8RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief set the GNSS satellite system used NON-persistently
//!   
//! \return value of type \c sensor_fi_tcl_b8_GnssSatSys. Method returns the actual
//  used satellite systems. If no satellite system could be determined it retruns 
//  0.
// ******************************************************************************
tU8 tclMsgGpsIf::u8SetDiagGnssSatSys ( tU8 u8TargetGnssSatSys )
{
   tU8 u8RetVal = 0;

   if( poGpsThread != OSAL_NULL )
   {
      u8RetVal = poGpsThread->u8SetDiagGnssSatSys ( u8TargetGnssSatSys ) ;
   }

   return u8RetVal;
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief set the GNSS epoch WN
//!   
//! \return TRUE if setting was successful otherwise FALSE
// ******************************************************************************
tBool 
tclMsgGpsIf::bSetWnEpoch ( const OSAL_trGnssTimeUTC &rTargetEpochUTC,
                           OSAL_trGnssTimeUTC &rReturnActualDateUTC )
{
   tBool  bRetVal = false;

   if( poGpsThread != OSAL_NULL )
   {   
      bRetVal = poGpsThread->bSetWnEpoch( rTargetEpochUTC , rReturnActualDateUTC );
   }

   return bRetVal;   
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief Get the GNSS epoch WN
//!   
//! \return TRUE if getting epoch was successful otherwise FALSE
// ******************************************************************************
tBool 
tclMsgGpsIf::bGetWnEpoch ( const OSAL_trGnssTimeUTC &rActualDateUTC )
{
   tBool  bRetVal = false;

   if( poGpsThread != OSAL_NULL )
   {   
      bRetVal = poGpsThread->bGetWnEpoch( rActualDateUTC );
   }

   return bRetVal;   
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief set the timemode (auto/manual) and in case of manual the time
//!   
//! \return 
// ******************************************************************************
tBool 
tclMsgGpsIf::bSetTimeMode ( const vds_tenTimeMode &enTargetTimeMode, 
                            const OSAL_trTimeDate &rfTargetTimeDate )
{
   tBool  bRetVal = false;
   
   if( poGpsThread != OSAL_NULL )
   {
      bRetVal = poGpsThread->bSetTimeMode( enTargetTimeMode, rfTargetTimeDate );
   }
   
   return bRetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief get actualTimeMode 
//!   
//! \return 
// ******************************************************************************
tBool 
tclMsgGpsIf::bGetTimeMode ( vds_tenTimeMode & rfenActualTimeMode )
{
   tBool  bRetVal = false;

   if( poGpsThread != OSAL_NULL )
   {
      bRetVal =  poGpsThread->bGetTimeMode( rfenActualTimeMode );
   }

   return bRetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief Sets different flags in tclGpsThread.
//!   
//! \return 
// ******************************************************************************

tVoid tclMsgGpsIf::vSetGpsThreadFlags( vds_tenGpsThreadFlags enFlag, tBool bFlagStatus )
{
   if(poGpsThread != OSAL_NULL)
   {
      poGpsThread->vSetGpsThreadFlags( enFlag, bFlagStatus);  
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Compose satus message for the property GnssConfigData.
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_INVALID_PARAMETER :if the received argument parameters are invalid.
//!   - \c VDS_E_GPSIF_NOT_INITIALIZED :if GPS interface not initialized.
//!   - \c VDS_E_INTERNAL_ERROR :if failed to get config data.
//!   - \c VDS_E_GPSIF_ALLOC_ERROR :if failed to create visitor mesage object.
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tS32 tclMsgGpsIf::s32GetGnssConfigDataMsg
(
   //! (I) : Pointer to service data.
   amt_tclServiceData const *poServiceData,
   //! (O) : pointer to a location where a pointer
   //!       to the reply message is available.
   amt_tclServiceData **ppoResponseMessage
)
{
   tS32 s32RetVal      = VDS_E_NO_ERROR;
   tU16 u16RegisterID  = 0;
   tU16 u16CmdCounter  = 0;
   tU16 u16TargetAppID = 0;
   tU16 u16ServiceID   = 0;
   OSAL_trGnssConfigData rGnssConfigData;

   //Check Received Arguments.
   if( OSAL_NULL != poServiceData )
   {
      u16TargetAppID = poServiceData->u16GetSourceAppID();
      u16RegisterID  = poServiceData->u16GetRegisterID();
      u16CmdCounter  = poServiceData->u16GetCmdCounter();
      u16ServiceID   = poServiceData->u16GetServiceID();
   }
   //Return Invalid Parameters Error.
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }


   if( (VDS_E_NO_ERROR != s32RetVal)   || 
       (OSAL_NULL == ppoResponseMessage) )
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }
   //Check the lock.
   else if( !bCritSectionLocked() )
   {
      s32RetVal = VDS_E_GPSIF_NOT_INITIALIZED;
   }
   //Get config data.
   else if(!poGpsThread->bGetGnssConfigData(&rGnssConfigData))
   {
      s32RetVal = VDS_E_INTERNAL_ERROR;
   }
   //Compose a response message.
   else
   {
      *ppoResponseMessage = OSAL_NULL;
      sensor_locationfi_tclMsgGnssConfigDataStatus oGnssConfigDataStatus;

      switch (rGnssConfigData.enGnssHwType)
      {
      case GNSS_HW_STA8088:
         oGnssConfigDataStatus.GnssConfigData.GnssHardwareType.enType =
            sensor_fi_tcl_e8_GnssHw::FI_EN_GNSS_HW_STA8088;
         break;
      case GNSS_HW_STA8089:
         oGnssConfigDataStatus.GnssConfigData.GnssHardwareType.enType =
            sensor_fi_tcl_e8_GnssHw::FI_EN_GNSS_HW_STA8089;
         break;
      default:
         oGnssConfigDataStatus.GnssConfigData.GnssHardwareType.enType =
            sensor_fi_tcl_e8_GnssHw::FI_EN_GNSS_HW_UNKNOWN;
      }
      // For GNSS only systems set it hardcoded to NMEA for better DR performance
      if ( 0 == (tclSystemInformation::rInfo.rSystem.u32SensorTypes
                 &
                 (~((tU32)(VDS_C_U32_SENSOR_GPS | VDS_C_U32_SENSOR_GNSS_FEATURE)))))
      {
         oGnssConfigDataStatus.GnssConfigData.GnssHardwareType.enType =
            sensor_fi_tcl_e8_GnssHw::FI_EN_GNSS_HW_NMEA_GPS;
      }
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                 "u32sensortypes = %d",tclSystemInformation::rInfo.rSystem.u32SensorTypes);
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                 "GNSS Hardware type =%d", oGnssConfigDataStatus.GnssConfigData.GnssHardwareType.enType);        
     
      oGnssConfigDataStatus.GnssConfigData.GnssReceiverFirmwareVersion =
         rGnssConfigData.u32GnssRecvBinVer;

      oGnssConfigDataStatus.GnssConfigData.NumberOfChannels =
         rGnssConfigData.u16NumOfChannels;

      oGnssConfigDataStatus.GnssConfigData.UsedSatStatusBits.u8Value =
         (tU8)rGnssConfigData.u16GnssSatStatusSupported;

      oGnssConfigDataStatus.GnssConfigData.UpdateFrequency =
         rGnssConfigData.u8UpdateFrequency;

      oGnssConfigDataStatus.GnssConfigData.GnssReceiverFirmwareCrc =
         rGnssConfigData.u32GnssRecvFwCrc;

      //Create the object of visitor message and insert the class object
      fi_tclVisitorMessage* poResultMessage
              = OSAL_NEW  fi_tclVisitorMessage( oGnssConfigDataStatus, VDS_C_U16_SRV_SENSORS_MAJOR_VERSION );

      if( (OSAL_NULL != poResultMessage )&&( poResultMessage->bIsValid() ))
      {
         tU32 u32TimeStamp = 0;
         vGetTimeStamp( &u32TimeStamp );

         //Create Answer Message
         poResultMessage->vInitServiceData( CCA_C_U16_APP_SENSOR,                 // Source
                                            u16TargetAppID,                       // Target
                                            AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,    // StreamType
                                            0,                                    // StreamCounter
                                            u16RegisterID,                        // RegisterID
                                            u16CmdCounter,                        // nCmdCounter
                                            u16ServiceID,                         // nServiceID
                                            SENSOR_LOCATIONFI_C_U16_GNSSCONFIGDATA,    // nFunctionID
                                            AMT_C_U8_CCAMSG_OPCODE_STATUS,        // OpCode
                                            0,                      // Asynchronous Completion Token(ACT)
                                            AMT_C_U16_DEFAULT_NULL, // Source Sub
                                            AMT_C_U16_DEFAULT_NULL, // Target Sub
                                            u32TimeStamp            // TimeStamp
                                          );
         *ppoResponseMessage = poResultMessage;
      }
      else
      {
         OSAL_DELETE poResultMessage;
         poResultMessage = OSAL_NULL;
         s32RetVal = VDS_E_GPSIF_ALLOC_ERROR;
      }
   }

   return s32RetVal;
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Compose satus message for the property GnssData.
//!  Only one Gnss element per call will be returned.
//! \return
//!   Succes code in case of success is:
//!   - \c  Number of elements copied : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_INVALID_PARAMETER :if the received argument parameters are invalid.
//!   - \c VDS_E_GPSIF_NOT_INITIALIZED :if GPS interface not initialized.
//!   - \c VDS_E_GPSIF_DATA_UNAVAILABLE :if no data is available.
//!   - \c VDS_E_GPSIF_ALLOC_ERROR :if failed to create visitor mesage object.
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tS32 tclMsgGpsIf::s32GetGnssDataMsg
(
   //! (I) : Pointer to Service data
   amt_tclServiceData const *poServiceData,
   //! (I) : Pointer to Subscriber data
   tclSubscriberManager *poSubscriber,
   //! (O) : pointer to a location where a pointer
   //!       to the reply message is available
   amt_tclServiceData **ppoResponseMessage,
   //! (O) : Next element to read.
   tU32 &rfu32NextElemToRead
)
{
   
   tS32 s32NumOfElmCopied  = 0;
   tU16 u16RegisterID      = 0;
   tU16 u16CmdCounter      = 0;
   tU16 u16TargetAppID     = 0;
   tU16 u16ServiceID       = 0;
   tS32 s32RetVal          = VDS_E_NO_ERROR;
   OSAL_trGnssFullRecord rGnssData;
   
   
   if( OSAL_NULL != poServiceData )
   {
      u16TargetAppID = poServiceData->u16GetSourceAppID();
      u16RegisterID  = poServiceData->u16GetRegisterID();
      u16CmdCounter  = poServiceData->u16GetCmdCounter();
      u16ServiceID   = poServiceData->u16GetServiceID();
   }
   else if( OSAL_NULL != poSubscriber )
   {
      u16TargetAppID = poSubscriber->u16GetTargetAppId();
      u16RegisterID  = poSubscriber->u16GetRegisterId();
      u16CmdCounter  = poSubscriber->u16GetCommandCntr();
      u16ServiceID   = poSubscriber->u16GetServiceId();
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }


   //Parameter Check.
   if( (VDS_E_NO_ERROR != s32RetVal) ||
       (OSAL_NULL == ppoResponseMessage) )
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }
   //Check Lock.
   else if( !bCritSectionLocked() )
   {
      s32RetVal = VDS_E_GPSIF_NOT_INITIALIZED;
   }
   else
   {
      *ppoResponseMessage = OSAL_NULL;
      sensor_locationfi_tclMsgGnssDataStatus oGnssDataStatus;

      /* removed hard coded value i.e u32NumOfElmToRead = 1 and used 
      oGnssDataBuffer.u32GetElemsFrom return value to calculate u32NumOfElmToRead
      This is implemented for CFG3-1622. if  u32NumOfElmToRead is more than 
      1 throws  an exception as we have allocated stack for only 1 element of gnss data 
      so earlier u32NumOfElmToRead = 1 was given as fix for GMMY16-14530*/

      tU32 u32NumOfElmToRead = oGnssDataBuffer.u32GetElemsFrom(rfu32NextElemToRead);
      if( u32NumOfElmToRead > 1 )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "tclMsgGpsIf: Found more than 1 GNSS data element in ring buffer, u32NumOfElmToRead :%d",
         u32NumOfElmToRead ); 
         
         u32NumOfElmToRead = 1;
      }

      //Get GNSS data from ring buffer.
      s32NumOfElmCopied = s32GetGnssDataFromBuffer( &rGnssData,
                                                    rfu32NextElemToRead,
                                                    u32NumOfElmToRead );

      if( s32NumOfElmCopied  <= 0 )
      {
         /*
          *Oops!data is not available.If this call is UPREG then respond with some
          *dummy data.Since according to CCA guidelines UPREG should be served
          *without fail.
          */
         if( (OSAL_NULL != poServiceData)&&
             (poServiceData->u8GetOpCode()== AMT_C_U8_CCAMSG_OPCODE_UPREG) )
         {
            OSAL_pvMemorySet ( &rGnssData, 0, sizeof( OSAL_trGnssFullRecord ) );
            vCopyGnssDataToCmbFiObject(&oGnssDataStatus.GnssData,&rGnssData);
            s32NumOfElmCopied = 1;
         }
         else
         {
            s32RetVal = VDS_E_GPSIF_DATA_UNAVAILABLE;
         }
      }
      else
      {
         vCopyGnssDataToCmbFiObject(&oGnssDataStatus.GnssData,&rGnssData);
      }

      if( VDS_E_NO_ERROR == s32RetVal )
      {
         //Create the object of visitor message and insert the class object.
         fi_tclVisitorMessage* poResultMessage = OSAL_NEW fi_tclVisitorMessage( oGnssDataStatus, VDS_C_U16_SRV_SENSORS_MAJOR_VERSION );

         if(( OSAL_NULL != poResultMessage ) && (poResultMessage->bIsValid()) )
         {
            tU32 u32TimeStamp = 0;
            vGetTimeStamp( &u32TimeStamp );

            poResultMessage->vInitServiceData( CCA_C_U16_APP_SENSOR,               // Source,
                                               u16TargetAppID,                     // Target
                                               AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
                                               0,                                  // StreamCounter
                                               u16RegisterID,                      // RegisterID
                                               u16CmdCounter,                      // nCmdCounter,
                                               u16ServiceID,                       // nServiceID,
                                               SENSOR_LOCATIONFI_C_U16_GNSSDATA,        // nFunctionID,
                                               AMT_C_U8_CCAMSG_OPCODE_STATUS,      // OpCode
                                               0,                             // Asynchronous Completion Token (ACT)
                                               AMT_C_U16_DEFAULT_NULL,        // Source Sub
                                               AMT_C_U16_DEFAULT_NULL,        // Target Sub
                                               u32TimeStamp                   // TimeStamp
                                             );
            *ppoResponseMessage = poResultMessage;
         }
         else
         {
            OSAL_DELETE poResultMessage;
            poResultMessage = OSAL_NULL;
            s32RetVal = VDS_E_GPSIF_ALLOC_ERROR;
         }
      }
   }
   
   if(VDS_E_NO_ERROR == s32RetVal)
   {
      s32RetVal = s32NumOfElmCopied;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Reads GNSS data from ring buffer.
//! \return
//!   Succes code in case of success is:
//!   - \c  Number of elements copied : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_INVALID_PARAMETER :if the received argument parameters are invalid.
//!   - \c VDS_E_GPSIF_NOT_INITIALIZED :if GPS interface not initialized.
//!
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tS32 tclMsgGpsIf::s32GetGnssDataFromBuffer
(
   //! (O) : Pointer to a location to
   //!       where the read data is copied
   OSAL_trGnssFullRecord *prGnssData,
   //! (I) : Next elements to read
   tU32 &rfu32NextElemToRead,
   //! (I) :Number of elements to be read starting
   //!      from the element "NumOfElmToRead".
   tU32 u32NumOfElmToRead
)
{
   tS32 s32NumOfElemsCopied = 0;

   // param-check
   if( NULL != prGnssData )
   {
      //Is everything OK.
      if ( oGnssDataBuffer.bIsValid() && bCritSectionLocked() )
      {
         tU32 u32Elems;
         
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,         
                    "tclMsgGpsIf::s32GetGnssDataFromBuffer start elm: %d  num elem: %d",
                    rfu32NextElemToRead,
                    u32NumOfElmToRead );         

         u32Elems = oGnssDataBuffer.u32GetList( prGnssData,
                                                u32NumOfElmToRead,
                                                rfu32NextElemToRead );

         s32NumOfElemsCopied = ( tS32 ) u32Elems;
      }
      else
      {
         s32NumOfElemsCopied = VDS_E_GPSIF_NOT_INITIALIZED;
      }
   }
   else
   {
      s32NumOfElemsCopied = VDS_E_INVALID_PARAMETER;
   }
   
   return s32NumOfElemsCopied;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Copy GNSS data to CMB FI object.
//! \return
//!     None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tVoid tclMsgGpsIf::vCopyGnssDataToCmbFiObject
(
   //! (O) : Pointer to cmb fi objectt to
   //!       where GNSS data is copied
   sensor_fi_tcl_GnssData *prCmbfiGnssData,
   //! (I) : Pointer which is holding GNSS data. 
   const OSAL_trGnssFullRecord *prGnssData
)
{
   //Copy TimeStamp
   prCmbfiGnssData->Timestamp = prGnssData->u32TimeStamp;

   //Copy UTC Time
   prCmbfiGnssData->GnssPvtData.UtcTimeDate.td_year   = prGnssData->rPVTData.rTimeUTC.u16Year;
   prCmbfiGnssData->GnssPvtData.UtcTimeDate.td_month  = prGnssData->rPVTData.rTimeUTC.u8Month;
   prCmbfiGnssData->GnssPvtData.UtcTimeDate.td_day    = prGnssData-> rPVTData.rTimeUTC.u8Day;
   prCmbfiGnssData->GnssPvtData.UtcTimeDate.td_hour   = prGnssData-> rPVTData.rTimeUTC.u8Hour;
   prCmbfiGnssData->GnssPvtData.UtcTimeDate.td_minute = prGnssData-> rPVTData.rTimeUTC.u8Minute;
   prCmbfiGnssData->GnssPvtData.UtcTimeDate.td_second = prGnssData-> rPVTData.rTimeUTC.u8Second;
   prCmbfiGnssData->GnssPvtData.UtcTimeDate.td_milliSeconds = prGnssData-> rPVTData.rTimeUTC.u16Millisecond;
   
   //Copy Position Info
   prCmbfiGnssData->GnssPvtData.Latitude  = prGnssData-> rPVTData.f64Latitude;
   prCmbfiGnssData->GnssPvtData.Longitude = prGnssData-> rPVTData.f64Longitude;
   prCmbfiGnssData->GnssPvtData.AltitudeWGS84     = prGnssData-> rPVTData.f32AltitudeWGS84;
   prCmbfiGnssData->GnssPvtData.GeoidalSeparation = prGnssData-> rPVTData.f32GeoidalSeparation;
   
   //Copy Velocity Info
   prCmbfiGnssData->GnssPvtData.VelocityNorth = prGnssData-> rPVTData.f32VelocityNorth;
   prCmbfiGnssData->GnssPvtData.VelocityEast  = prGnssData-> rPVTData.f32VelocityEast;
   prCmbfiGnssData->GnssPvtData.VelocityUp    = prGnssData-> rPVTData.f32VelocityUp;
   
   //Copy Position Covariance Matrix Data 
   prCmbfiGnssData->GnssPvtData.PositionCovarianceMatrix.Elem0 =
              prGnssData-> rPVTData.rPositionCovarianceMatrix.f32Elem0;
   prCmbfiGnssData->GnssPvtData.PositionCovarianceMatrix.Elem4 =
              prGnssData-> rPVTData.rPositionCovarianceMatrix.f32Elem4;
   prCmbfiGnssData->GnssPvtData.PositionCovarianceMatrix.Elem5 =
              prGnssData-> rPVTData.rPositionCovarianceMatrix.f32Elem5;
   prCmbfiGnssData->GnssPvtData.PositionCovarianceMatrix.Elem8 =
              prGnssData-> rPVTData.rPositionCovarianceMatrix.f32Elem8;
   prCmbfiGnssData->GnssPvtData.PositionCovarianceMatrix.Elem9 =
              prGnssData-> rPVTData.rPositionCovarianceMatrix.f32Elem9;
   prCmbfiGnssData->GnssPvtData.PositionCovarianceMatrix.Elem10 =
              prGnssData-> rPVTData.rPositionCovarianceMatrix.f32Elem10;
   
   //Copy Velocity Covariance Matrix
   prCmbfiGnssData->GnssPvtData.VelocityCovarianceMatrix.Elem0 = 
              prGnssData-> rPVTData.rVelocityCovarianceMatrix.f32Elem0;
   prCmbfiGnssData->GnssPvtData.VelocityCovarianceMatrix.Elem4 =
              prGnssData-> rPVTData.rVelocityCovarianceMatrix.f32Elem4;
   prCmbfiGnssData->GnssPvtData.VelocityCovarianceMatrix.Elem5 = 
              prGnssData-> rPVTData.rVelocityCovarianceMatrix.f32Elem5;
   prCmbfiGnssData->GnssPvtData.VelocityCovarianceMatrix.Elem8 =
              prGnssData-> rPVTData.rVelocityCovarianceMatrix.f32Elem8;
   prCmbfiGnssData->GnssPvtData.VelocityCovarianceMatrix.Elem9 = 
              prGnssData-> rPVTData.rVelocityCovarianceMatrix.f32Elem9;
   prCmbfiGnssData->GnssPvtData.VelocityCovarianceMatrix.Elem10 =
              prGnssData-> rPVTData.rVelocityCovarianceMatrix.f32Elem10;
   
   prCmbfiGnssData->GnssPvtData.GnssStatus.GnssQuality.enType = 
             (sensor_fi_tcl_e8_GnssQuality::tenType)prGnssData-> rPVTData.rFixStatus.enQuality;
   prCmbfiGnssData->GnssPvtData.GnssStatus.GnssMode.enType = 
             (sensor_fi_tcl_e8_GnssMode::tenType)prGnssData-> rPVTData.rFixStatus.enMode;
   
   //Copy DOPs Data
   prCmbfiGnssData->GnssPvtData.GDOP = prGnssData-> rPVTData.f32GDOP;
   prCmbfiGnssData->GnssPvtData.PDOP = prGnssData-> rPVTData.f32PDOP;
   prCmbfiGnssData->GnssPvtData.HDOP = prGnssData-> rPVTData.f32HDOP;
   prCmbfiGnssData->GnssPvtData.TDOP = prGnssData-> rPVTData.f32TDOP;
   prCmbfiGnssData->GnssPvtData.VDOP = prGnssData-> rPVTData.f32VDOP;
   
   //Copy Data About Satellites
   prCmbfiGnssData->GnssPvtData.SatSysUsed.u8Value = prGnssData-> rPVTData.u8SatSysUsed;
   prCmbfiGnssData->GnssPvtData.SatellitesVisible  = prGnssData-> rPVTData.u16SatsVisible;
   prCmbfiGnssData->GnssPvtData.SatellitesReceived = prGnssData-> rPVTData.u16Received;
   prCmbfiGnssData->GnssPvtData.SatellitesUsed     = prGnssData-> rPVTData.u16SatsUsed;
   
   //Resize to accomodate all the data.
   try
   {
      prCmbfiGnssData->GnssChanDataList.resize( ( tU32 )prGnssData-> rPVTData.u16SatsVisible );
   }
   catch(std::bad_alloc& ba)
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GpsIf:exception bad_alloc caught",ba.what());
   }

   for( tU32 u32Loop = 0;
        u32Loop < ( tU32 )prGnssData-> rPVTData.u16SatsVisible && u32Loop < OSAL_C_U8_GNSS_NO_CHANNELS ;
        u32Loop++ )
   {
      prCmbfiGnssData->GnssChanDataList[u32Loop].SvId =
                        prGnssData->rChannelStatus[u32Loop].u16SvID;
      prCmbfiGnssData->GnssChanDataList[u32Loop].SatStatus.u8Value =
                        (tU8)prGnssData->rChannelStatus[u32Loop].u16SatStatus;
      prCmbfiGnssData->GnssChanDataList[u32Loop].Azimuth =
                        prGnssData->rChannelStatus[u32Loop].f32Azimuthal;
      prCmbfiGnssData->GnssChanDataList[u32Loop].Elevation =
                        prGnssData->rChannelStatus[u32Loop].f32Elevation;
      prCmbfiGnssData->GnssChanDataList[u32Loop].CarrierToNoiseRatio =
                        prGnssData->rChannelStatus[u32Loop].u8CarrierToNoiseRatio;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Collects GNSS records for "AllSensorData" property. 
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_INVALID_PARAMETER :if the received argument parameters are invalid.
//!   - \c VDS_E_GPSIF_DATA_UNAVAILABLE :if no data is available.
//!   - \c VDS_E_GPSIF_ALLOC_ERROR :if failed to allocate memory.
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tS32 tclMsgGpsIf::s32GetGnssRecords
(
   //! (O) :cmb fi objectt to copy GNSS data.
   sensor_fi_tcl_AllSensorData *pCmbFiGnssData,
   //! (I) :Element from where to copy.
   tU32 &rfu32NextelementToRead
)
{
   tS32 s32RetVal=VDS_E_NO_ERROR;
   OSAL_trGnssFullRecord *prGnssFullRecords = OSAL_NULL;
   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(prGnssFullRecords);

   if( ( pCmbFiGnssData != OSAL_NULL ) && 
       ( poGpsThread != OSAL_NULL ) &&
         bCritSectionLocked() 
      )
   {
      //Get Number of GNSS Records Available.
      tU32 u32NumOfGnssRecordsToRead = oGnssDataBuffer.u32GetElemsFrom(rfu32NextelementToRead);

      //If Any Records Available.
      if(u32NumOfGnssRecordsToRead > 0)
      {

         prGnssFullRecords = (OSAL_trGnssFullRecord*)
                 OSAL_pvMemoryAllocate(sizeof(OSAL_trGnssFullRecord)*u32NumOfGnssRecordsToRead);

         if( prGnssFullRecords != OSAL_NULL )
         {

            //Get All The Available Records.
            tS32 s32NumOfElmcopied = s32GetGnssDataFromBuffer( prGnssFullRecords,
                                                               rfu32NextelementToRead,
                                                               u32NumOfGnssRecordsToRead);

            if(s32NumOfElmcopied == (tS32)u32NumOfGnssRecordsToRead )
            {
               //Resize memory to accomodate all the elements.
               try
               {
                  pCmbFiGnssData->GnssUpdate.resize(u32NumOfGnssRecordsToRead);
               }
               catch(std::bad_alloc& ba)
               {
                  vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GpsIf:exception bad_alloc caught",ba.what());
               }
               //Copy data to CMB FI object.
               for(tU8 u8Record=0;u8Record < u32NumOfGnssRecordsToRead ;u8Record++ )
               {
                  vCopyGnssDataToCmbFiObject(&pCmbFiGnssData->GnssUpdate[u8Record],
                                             &prGnssFullRecords[u8Record] );
               }
            }
            else
            {
               s32RetVal = VDS_E_GPSIF_DATA_UNAVAILABLE;
               vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                          "GpsIf:GnssAllSen:Failed to read requested no.of data");
            }

            //Free memory.
            OSAL_vMemoryFree(prGnssFullRecords);
            prGnssFullRecords = OSAL_NULL;
         }
         else
         {
            s32RetVal = VDS_E_GPSIF_ALLOC_ERROR;
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "GpsIf:GnssAllSen:Allocation Error");
         }
      }
      else
      {
         s32RetVal = VDS_E_GPSIF_DATA_UNAVAILABLE;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GpsIf:GnssAllSen:Data Unavailable");
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GpsIf:GnssAllSen:Not Initialized");
   }
   
   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(prGnssFullRecords);
   
   return s32RetVal;
}


//EOF
