//  *****************************************************************************
//  * FILE:         tclAccThread.h
//  * SW_COMPONENT: VD-Sensor
//  * DESCRIPTION:  Class declaration for tclMsgAccIf class, which handles
//  *               the accessing Accelorometer
//  *
//  * AUTHOR:       RBEI/ECF1-Sainath Kalpuri
//  * COPYRIGHT:    (c) 2011 Robect Bosch.
//  * HISTORY:
//  * 08.03.11 Rev. 1.0 RBEI/ECF1-Sainath Kalpuri
//  *          Initial Revision.
//  *****************************************************************************

#ifndef tclMsgAccIf_h
#define tclMsgAccIf_h 1

#define VDS_C_U32_ACCIF_RINGBUFFERSIZE     (tU32) 160

#define VDS_C_S32_ACCIF_UNINITIALIZED      (tS32)0
#define VDS_C_S32_ACCIF_NORMAL             (tS32)1
#define VDS_C_S32_ACCIF_PENDING            (tS32)2

// errors
#define VDS_E_ACCIF_ALLOC_ERROR            -1
#define VDS_E_ACCIF_THREADINIT_ERROR       -2
#define VDS_E_ACCIF_SEM_ERROR              -3

#define VDS_E_ACCIF_THREADSTART_ERROR      -4
#define VDS_E_ACCIF_NOT_INITIALIZED        -5
#define VDS_E_ACCIF_THREADSTOP_ERROR       -6
#define VDS_E_ACCIF_GETACCDATA_FAILED      -7

#define VDS_E_ACCIF_SETACCMSG_FAILED       -8
#define VDS_E_ACCIF_SUBSCRIBER_ERROR       -9
#define VDS_E_ACCIF_VALUE_UNSET            -10
#define VDS_E_ACCIF_GET_ACC_OFFSET_FAILED -11
#define VDS_E_ACCIF_GET_ACC_SCALE_FAILED  -12
#define VDS_E_ACCIF_DATA_UNAVAILABLE       -13

#define VDS_E_ACCIF_UNKNOWN_ERROR          -99

/*!
* \brief Defines the interface for Accelerometer Message.
*/
class tclMsgAccIf
{

   public:
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   Default constructor
      //  Parameters: None
      //  return
      //   None
      //  HISTORY:
      // Date            |  Author         | MODIFICATION
      // ----------------------------------------------------------------------------
      //  08.03.11         sak9kor(RBEI/ECF1)    Initial Revision.

      //******************************************************************************
      tclMsgAccIf();
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Default Destructor
      //  Parameters : None
      //  return
      //    None
      //  HISTORY:
      // Date            |  Author         | MODIFICATION
      // ----------------------------------------------------------------------------
      //  08.03.11         sak9kor(RBEI/ECF1)    Initial Revision.
      //******************************************************************************
      virtual ~tclMsgAccIf();
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    TBD
      //  PARAMETERS:
      //  return
      //    Succes code in case of success is:
      //      VDS_E_NO_ERROR : Success\n
      //    Error code in case of failure are:\n
      //      VDS_E_ACCIF_ALLOC_ERROR : Acc interface allocation error
      //      VDS_E_ACCIF_SEM_ERROR :  Acc sempahore error
      //      VDS_E_ACCIF_THREADINIT_ERROR : Acc interface thread interface error
      //      VDS_E_INVALID_PARAMETER : Invalid parameter
      //  return
      //    None
      //  HISTORY:
      // Date            |  Author         | MODIFICATION
      // ----------------------------------------------------------------------------
      //  08.03.11         sak9kor(RBEI/ECF1)    Initial Revision.
      //******************************************************************************
      tS32 s32Init (
         tS32( *pfs32SendSubscriberMessageParam )( amt_tclServiceData *, tclSubscriberManager * ));
                    
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    start Acc-thread
      //  PARAMETERS: None
      //  return
      //    Succes code in case of success is:
      //      VDS_E_NO_ERROR : Success\n
      //    Error code in case of failure are:\n
      //      VDS_E_ACCIF_THREADSTART_ERROR : Thread start error
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32StartThread ();
      
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   stop acc-thread
      //  Parameters: None
      //  return
      //    Succes code in case of success is:
      //      VDS_E_NO_ERROR : Success\n
      //   Error code in case of failure are:\n
      //      VDS_E_ACCIF_THREADSTOP_ERROR : Thread stop error
      //      VDS_E_ACCIF_NOT_INITIALIZED : Acc interface not initialized
      //  HISTORY:
      // Date            |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32StopThread (); 

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   delete acc-thread
      //  Parameters : None
      //  return
      //    Succes code in case of success is:
      //       VDS_E_NO_ERROR : Success\n
      //    Error code in case of failure are:\n
      //       VDS_E_ACCIF_THREADSTOP_ERROR : Thread stop error
      //       VDS_E_ACCIF_NOT_INITIALIZED : Acc interface not initialized
      //  HISTORY:
      // Date          |  Author                | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32DeleteThread ();      

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    set internal state (normal or pending)
      //  PARAMETER:
      //       tS32 s32NewState: (I)
      //            new state (VDS_C_S32_ACCIF_NORMAL
      //            or VDS_C_S32_ACCIF_PENDING
      //
      //  return
      //   None
      //  HISTORY:
      // Date            |  Author                    | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tVoid vSetState ( tS32 s32NewState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Set Application state
      //  Parameters : None
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tVoid vSetAppState( tU32 u32AppState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Performs the synch operation
      //  Parameters : None
      //  return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      void vSynch( void );       
      
      tS32 s32Add3dAccList ( const sensor_fi_tcl_3dAccData *pr3dAccData,
                             tU32 u32NumOfData );      
                    
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Performs the synch operation
      //  Parameters : None
      //  return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************      
      tS32 s32Get3dAccMessage ( amt_tclServiceData const *poRequestMessage, 
                                tclSubscriberManager *poSubscriber, 
                                amt_tclServiceData **ppoServiceData, 
                                sensor_fi_tcl_3dAccData *pr3dAccData, 
                                tU32 u32NumOfData, 
                                tU32 &rfu32NextElemToRead, 
                                tBool bSendInitMessage, 
                                tBool bCreateInvalidMessage = FALSE ); 
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    returns a list of acc-datas from ringbuffer.the number of the element
      //    from where to be read could be determined from rfu32NextElemToRead.
      //    if the number is smaller than the ones in the list, the oldest datas
      //    will be returned, if the number is bigger than the ones in the list,
      //    no data is returned. rfu32NextElemToRead will be set to
      //    the number of the last returned data-set + 1.
      //  PARAMETER:
      //     sensor_fi_tcl_3dAccData *prAccData: (->O)
      //       storage for acc-list
      //     tU32 &rfu32NextElemToRead: (->IO)
      //       number of element that has to be read the next time
      //     tU32 u32ElemsToRead: (I)
      //       number of elements to be read
      //  return
      //    Elements copied in case of success
      //    Error code in case of failure are:\n
      //      VDS_E_ACCIF_NOT_INITIALIZED : Acc interface not initialized
      //      VDS_E_INVALID_PARAMETER : Invalid parameter
      //  HISTORY:
      // Date            |  Author                 | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32Get3dAccData ( sensor_fi_tcl_3dAccData *prAccData, 
                             tU32 &rfu32NextElemToRead, 
                             tU32 u32ElemsToRead );  
      
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   check for subscribers and send message. the actual list of acc-data is given,
      //   so we normally don't need to create a temporary storage for the answer-message.
      //  Parameters : 
      //      sensor_fi_tcl_3dAccData *pr3dAccData - 3dAccData
      //      tU32 u32NumOfData - Num of Elements
      //  return
      //    VDS_E_NO_ERROR - In case of success and error code in case of failure
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32CheckFor3dAccSubscriber ( sensor_fi_tcl_3dAccData *pr3dAccData, 
                                        tU32 u32NumOfData );      

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   stores list of 3D acc-data in ringbuffer, triggers check for
      //   subscribers (and send data)
      //  Parameters : 
      //      sensor_fi_tcl_3dAccData *pr3dAccData - 3dAccData
      //      tU32 u32NumOfData - Num of Elements
      //  return
      //    VDS_E_NO_ERROR - In case of success and error code in case of failure
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************      
      static tS32 s32Add3dDataWrapper ( const sensor_fi_tcl_3dAccData *pr3dAccData, 
                                        tU32 u32NumOfData );                    
      
      vds_tclCriticalSection oCS;
      
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief get the last element index in the acc ring buffer. 
      //!   
      //! \return 
      // ******************************************************************************         
      tU32 u32GetLastAccElemIndex(tVoid)
      {
         return(oRingBuffer3d.u32GetNumElems() - 1);
      }
     // ***************** F U N C T I O N  H E A D E R *****************************
     //
     //  DESCRIPTION:
     //
     //! \brief
     //!  Collects Acc records for "AllSensorData" property and updates in cmb fi object. 
     //! \return
     //!   Succes code in case of success is:
     //!   - \c  VDS_E_NO_ERROR : Success\n
     //!   Error code in case of failure are:\n
     //!   - \c VDS_E_ACCIF_NOT_INITIALIZED :if vdsensor is not initialized.
     //!   - \c VDS_E_ACCIF_ALLOC_ERROR :if failed to allocate memory.
     //!   - \c VDS_E_ACCIF_GETACCDATA_FAILED: if unable to get data.
     //!   - \c VDS_E_ACCIF_DATA_UNAVAILABLE:if no acc data is available. 
     //  HISTORY:
     //  Date            |  Author                       | MODIFICATION
     // ----------------------------------------------------------------------------
     //******************************************************************************       
     tS32 s32GetAccRecords( sensor_fi_tcl_AllSensorData *pCmbFiAccData,
                            tU32 &NextelementToRead );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //! \brief
      //!       Start Acc self test.
      //! \return
      //!   - \c TRUE  : Self test finished successfully.
      //!   - \c FALSE : Unable to start self test.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tBool bStartSelftest(tS32 *ps32SelfTestResult);
      // Flag to indicate whether Self Test is running
      tBool bIsSelftestRunning;

   private:
      // Data Members for Class Attributes
      tclAccThread *poAccThread;
      tS32(*pfs32SendSubscriberMessage)( amt_tclServiceData *, tclSubscriberManager * );
      vds_tclRingBuffer<sensor_fi_tcl_3dAccData> oRingBuffer3d;      
      static tclMsgAccIf *poThisMsgAccIf;
      tS32 s32InternalState;
};

#endif
